<?php namespace App\Repositories;

/**
 * Created by PhpStorm.

 */

/**
 * The Abstract Repository provides default implementations of the methods defined
 * in the base repository interface. These simply delegate static function calls
 * to the right eloquent model based on the $modelClassName.
 */

use App\Models;
use App\Repositories\Interfaces\IRepository;
use Illuminate\Support\Facades\DB;

class Repository implements IRepository {

    protected $modelClassName;
    protected $modelVariableName;

    public function relationship($functionName){
        return $this->model->$functionName;
    }

    public function all($columns = array('*')) {
        return $this->model->get($columns);
    }


    public function insert(array $attributes)
    {
        return call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::insert", $attributes);
    }

    public function create(array $attributes)
    {
        return call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::create", $attributes);
    }

    public function firstOrCreate(array $attributes)
    {
        return call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::firstOrCreate", $attributes);
    }

    public function delete()
    {
        return call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::delete");
    }


    public function first($with = [], $orderBy = [], $columns = ['*'], $trashed = false)
    {
        return $this->get($with, $orderBy, $columns, $trashed)->first();
    }


    public function get($with = [], $orderBy = [], $columns = ['*'], $trashed = false)
    {
        if (count($with)){
            $result = $this->model->with($with);
        }
        else {
            $result = $this->model->select($columns);
        }

        if ($trashed){
            switch ($trashed){
                case WITH_TRASHED:
                    $result->withTrashed();
                    break;
                case ONLY_TRASHED:
                    $result->onlyTrashed();
                    break;
            }
        }

        if ($orderBy){
            foreach ($orderBy as $each) {
                if (is_array($each)) {
                    $result->orderBy($each[0], isset($each[1]) ? $each[1] : 'ASC');
                }
                else {
                    $result->orderBy($each);
                }
            }
        }

        return  $result->get();
    }

    public function getAttributeById($id, $attribute){
        return $this->find($id)->$attribute;
    }

    public function find($id, $trashed = false)
    {
        if ($trashed){
            switch ($trashed){
                case WITH_TRASHED:
                    $result = call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::withTrashed");
                    $result->where('id', $id);
                    return $result->first();
                    break;
                case ONLY_TRASHED:
                    $result = call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::onlyTrashed");
                    $result->where('id', $id);
                    return $result->first();
                    break;
            }
        }

        $result = call_user_func(__NAMESPACE__MODEL . "{$this->modelClassName}::find", $id);

        return $result;
    }

    public function destroy($ids)
    {
        return call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::destroy", array($ids));
    }

    /*public function __call($method, $args)
    {
        return call_user_func([__NAMESPACE__MODEL.$this->modelClassName, $method], $args);
    }*/

    public function getTableName(){
        return $this->getTable();
    }


    /**
     * Find By Attributes And Update Found Items
     * @param $findAttributes
     * @param $updateAttributes
     */
    public function updateByAttributes($findAttributes, $updateAttributes){
        $search = $this->findAllByAttributes($findAttributes);
        if ($search && count($search)){
            foreach ($search as $each) {
                foreach ($updateAttributes as $attr => $value) {
                    $each->$attr = $value;
                }
                $each->save();
            }

        }
    }


    public function updateById($id, $attributes){
        if ($id) {
            $object = $this->find($id);
            if ($object) {
                foreach ($attributes as $attr => $value) {
                    $object->$attr = $value;
                }
                $object->save();
                return true;
            }
        }
        return false;
    }

    public function firstOrCreateMany(array $find_attributes, $key, $iterative_attribute){
        if ($find_attributes) {
            $iterative_attribute = is_array($iterative_attribute) ? $iterative_attribute : [$iterative_attribute];
            $ids = [];
            foreach ($iterative_attribute as $each) {

                $attributes = $find_attributes + [$key => $each];
                $ids[] = $this->firstOrCreate($attributes);
            }
            return $ids;
        }
    }

    public function createByAttributes(array $attributes){
        if ($attributes && count($attributes)) {
            $model = __NAMESPACE__MODEL.$this->modelClassName;
            $object = new $model;
            foreach ($attributes as $attr => $value) {
                $object->{$attr} = $value;
            }
            $object->save();
            return $object;
        }
    }

    public function update($object, $attributes){
        if ($object && $attributes && count($attributes)) {
            foreach ($attributes as $attr => $value) {
                $object->$attr = $value;
            }
            $object->save();
            return true;
        }
        return false;
    }


    public function deleteByAttributes(array $attributes){
        $result = $this->findAllByAttributes($attributes);
        foreach ($result as &$each) {
//            $each->delete();

            // Audit Deleted
            auditDeleted($each);
        }
    }

    public function deleteNotIn(array $attributes, array $values, $field = 'id'){
        foreach ($attributes as $attr => $value) {
            if (!isset($result)){
                $result = $this->model->where($attr, $value);
            }
            else {
                $result->where($attr, $value);
            }
        }
        if (isset($result)) {
            $list = $result->whereNotIn($field, $values)
                            ->get();

            // Audit Deleted
            auditDeleted($list);
        }
    }

    public function deleteObject($object){
//        $object->delete();

        // Audit Deleted
        auditDeleted($object);
    }

    /**
     * Lists = [0 => Value, 1 -> Key, 2->Concat
     * @param array $searchAttr
     * @param array $lists
     * @param array $orderBy
     * @param bool $selectOptionValue
     * @param bool $limit
     * @return array
     */
    public function findAndListModelVariable($searchAttr = [], $lists =  [0 => 'name'], $selectOptionValue = false, $limit = false, $orderBy = []){
        $select = ['*'];
        // CONCAT Option
        if (isset($lists[2])){
            $select = [];
            $selected = 'selected';
            $select[] = DB::raw('CONCAT('.implode(",'".$lists[2]."',", $lists[0]).') AS '.$selected);
            $select[] = 'id';
        }
        else {
            $selected = $lists[0];
        }
        $result = call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::select",$select);

        $this->queryBuilder($result, $searchAttr, [], $orderBy);

        $result->orderBy($selected);

        if ($limit)
            $result->limit($limit);

        $result =  isset($lists[1]) ? $result->pluck($selected, $lists[1])->all() : $result->pluck($selected)->all();
        return $selectOptionValue !== FALSE ? ['' => "Select"] + $result : $result;
    }

    public function listModelVariable($lists = [0 => 'name'], $selectOptionValue = FALSE, $withoutOrder = FALSE){
        $select = ['*'];
        if (isset($lists[2])){
            $select = [];
            $selected = 'selected';
            $select[] = DB::raw('CONCAT('.implode(",'".$lists[2]."',", $lists[0]).') AS '.$selected);
            $select[] = 'id';
        }
        else {
            $selected = $lists[0];
        }
        $result = call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::select", $select);
        // Don't Order if Set
        if (!$withoutOrder) {
            $result->orderBy($selected);
        }
        $result =  isset($lists[1]) ? $result->pluck($selected, $lists[1])->all() : $result->pluck($selected)->all();
        return $selectOptionValue !== FALSE ? ['' => "Select"] + $result : $result;
    }

    /**
     * Query Builder
     * @param $query
     * @param array $attributes
     * @param array $join
     * @param array $orderBy
     * @param bool|FALSE $limit
     * @param bool|FALSE $groupBy
     */
    protected function queryBuilder(&$query, $attributes = [], $join = [], $orderBy = [], $limit = FALSE, $groupBy = FALSE){

        if (count($join)){
            if (is_array($join[0])){
                foreach ($join as $each) {
                    if (isset($each[4]))
                        $query->join($each[0], $each[1], $each[2], $each[3], $each[4]);
                    else
                        $query->join($each[0], $each[1], $each[2], $each[3]);
                }
            }
            else {
                if (isset($join[4]))
                    $query->join($join[0], $join[1], $join[2], $join[3], $join[4]);
                else
                    $query->join($join[0], $join[1], $join[2], $join[3]);
            }
        }

        if (count($attributes)) {
            foreach ($attributes as $attr => $value) {
                if (is_null($value))
                    $query->whereNull($attr);
                else if (is_array($value)) {
                    switch ($value[0]) {
                        case WHERE_IN:
                            $query->whereIn($attr, $value[1]);
                            break;
                        case WHERE_NOT_IN:
                            $query->whereNotIn($attr, $value[1]);
                            break;
                        case WHERE_NULL:
                            $query->whereNull($attr);
                            break;
                        case WHERE_NOT_NULL:
                            $query->whereNotNull($attr);
                            break;
                        case WHERE_BETWEEN:
                            $query->whereBetween($attr, $value[1]);
                            break;
                        case WHERE_RAW:
                            $query->whereRaw($value[1]);
                            break;
                        default:
                            $query->where($attr, $value[0], $value[1]);
                            break;
                    }
                } else
                    $query->where($attr, $value);
            }
        }

        if ($orderBy){
            foreach ($orderBy as $each) {
                if (is_array($each)) {
                    $query->orderBy($each[0], isset($each[1]) ? $each[1] : 'ASC');
                }
                else {
                    $query->orderBy($each);
                }
            }
        }

        if ($groupBy)
            $query->groupBy($groupBy);

        if ($limit)
            $query->limit($limit);

    }


    public function findByAttributes($attributes = [], $with = [], $select = ['*'], $join = [], $orderBy = []){
        if (!count($select))
            $select = ['*'];

        if (count($with)) {
            //$result = $this->model->with($with);
            $result = call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::with",$with);
            $result->select($select);
        } else {
            //$result = $this->model;
            $result = call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::select",$select);
        }

        $this->queryBuilder($result, $attributes, $join, $orderBy);

        return $result->first();
    }

    public function findAllByAttributes($attributes = [], $with = [], $select = ['*'], $join = [], $orderBy = [], $limit = false, $groupBy = false){
        if (!count($select))
            $select = ['*'];

        if (count($with)) {
            //$result = $this->model->with($with);
            $result = call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::with",$with);
            $result->select($select);
        } else {
            //$result = $this->model;
            $result = call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::select",$select);
        }

        $this->queryBuilder($result, $attributes, $join, $orderBy, $limit, $groupBy);

        return $result->get();
    }

    /**
     * Get By Given Conditions
     * Sample Each Condition
     * @param $orderBy
     * @param $condition
     * @return mixed
     */
    public function getByCondition(array $condition, $orderBy = FALSE){
        $result = call_user_func(__NAMESPACE__MODEL."{$this->modelClassName}::select",['*']);

        foreach ($condition as $attr => $each) {
            $result->where($attr, $each);
        }

        if ($orderBy) {
            if (is_array($orderBy)){
                foreach ($orderBy as $each) {
                    $result->orderBy($each);
                }

            }
            else {
                $result->orderBy($orderBy);
            }
        }

        return $result->get();
    }

}
