<?php namespace App\Models; use Illuminate\Database\Eloquent\Model as Eloquent;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 */

class UserHistory extends Eloquent implements AuditableContract{

    use Auditable;

    protected $auditExclude = [
        'updated_by',
        'created_by',
    ];

    protected $table = "users__history";

    public function condition(){
        return $this->belongsTo(Condition::class, "condition_id");
    }

    public function airline(){
        return $this->belongsTo(Airline::class, "airline_id");
    }

    public function user(){
        return $this->belongsTo(User::class, "user_id");
    }

    public function createdBy(){
        return $this->belongsTo(User::class, "created_by");
    }

    public static function getCurrentUserHistory($onlyOfficeDuties = true, $handlingAirports = null){
        $now = date("Y-m-d")." 00:00";

        $list = UserHistory::with(["user", "condition"])
                            ->join("users", "users.id", "=", "users__history.user_id");


        if ($onlyOfficeDuties){
            $list->join("conditions", "conditions.id", "=", "users__history.condition_id")
                    ->where("conditions.type", ROSTER_TYPE);
        }


        if ($handlingAirports){
            $handlingAirports = is_array($handlingAirports) ? $handlingAirports : [$handlingAirports];

            $list->join("locations", "locations.id", "=", "users.location_id")
                ->whereIn("locations.airport_id", array_keys($handlingAirports));
        }

        $list = $list->where("condition_to", ">=", $now)
                            ->whereNull("users__history.deleted_at")
                            ->whereNull("users.deleted_at")
                            ->whereNull("users.vs")
                            ->whereNull("users__history.deleted_at")
                            ->get([
                                "users__history.*"
                            ]);

        return $list;

    }

    public static function getUserHistory($fromDate, $toDate, $groupedByUserID = false, $onlyOfficeDuties = true,
                                          $report_type = null, $handlingAirports = null, $selectedAirports = null, $userIDs = [],
                                          $selectedAirlines = null)
    {
        $list = UserHistory::with([
            "user",
            "user.position",
            "user.location",
            "user.location.airport",
            "condition",
//            "createdBy",
        ])
            ->join("users", "users.id", "=", "users__history.user_id")
            ->join("conditions", "conditions.id", "=", "users__history.condition_id");

        if ($report_type && in_array($report_type, [COMBINED_ALL_STATIONS, EXCLUDING_OCC_ALL_STATIONS, OCC_REPORT_ALL_STATIONS])){
            $selectedAirports = array_keys($handlingAirports); // [];
        }
        else {
            if ($selectedAirports) {
                $selectedAirports = is_array($selectedAirports) ? $selectedAirports : [$selectedAirports];
            }
        }

        if ($selectedAirlines){
            if (is_array($selectedAirlines)){
                if (count($selectedAirlines)){
                 $list->whereIn("airline_id", $selectedAirlines);
                }
            }
            else {
                $list->where("airline_id", $selectedAirlines);
            }
        }

        if ($selectedAirports && count($selectedAirports)){
            $list->join("locations", "locations.id", "=", "users.location_id")
                ->whereIn("locations.airport_id", $selectedAirports);
        }

        if ($onlyOfficeDuties){
            $list->where("conditions.type", ROSTER_TYPE);
        }

//        if ($skipOffs){
//            $list->where(function ($sql){
//                $sql->whereNull("skip_total")
//                    ->orWhere("skip_total", 0);
//            });
//        }

        if ($report_type){
            $list->join("users__departments", "users__departments.user_id", "=", "users__history.user_Id")
                ->join("structure__positions", "structure__positions.id", "=", "users__departments.position_id");

            switch ($report_type){
                case EXCLUDING_OCC:
                case EXCLUDING_OCC_ALL_STATIONS:
                    $list->whereNotIn("structure__positions.name", [
                        HEAD_OF_OCC_POSITION,
                        OCC_CONTROLLER_POSITION,
                    ]);
                    break;

                case OCC_REPORT:
                case OCC_REPORT_ALL_STATIONS:
                    $list->whereIn("structure__positions.name", [
                        HEAD_OF_OCC_POSITION,
                        OCC_CONTROLLER_POSITION,
                    ]);
                    break;

                case COMBINED_REPORT:
                case COMBINED_ALL_STATIONS:
                default:
                    break;
            }
        }

        $fromDate = date("Y-m-d", strtotime($fromDate))." 00:00";
        $toDate = date("Y-m-d", strtotime($toDate))." 23:59";

        if (count($userIDs)){
            $list->whereIn("user_id", $userIDs);
        }

        $list = $list->where(function($sql) use ($fromDate, $toDate) {
                    $sql->whereBetween("condition_to", [$fromDate, $toDate])
                        ->orWhereBetween("condition_from", [$fromDate, $toDate]);
                })
                ->whereNull("users__history.deleted_at")
                ->whereNull("users.deleted_at")
                ->whereNull("users.vs")
                ->whereNull("users__history.deleted_at")
                ->orderBy("condition_from")
                ->get([
                    "users__history.*"
                ]);

        if ($groupedByUserID){
            $result = [
                "total"     => 0,
                "list"      => [],
            ];

            $total = 0;
            foreach ($list as $item) {
                if (!isset($result["list"][$item->user_id])){
                    $result["list"][$item->user_id] = [
                        "user"          => $item->user,
                        "total"         => 0,
                        "history"       => [],
                        "wHistory"      => [],
                    ];
                }

                $result["list"][$item->user_id]["history"][] = $item;

                if (!$item->condition->skip_total){
                    //self::setWorkingPeriods($result["list"][$item->user_id]["wHistory"], $item->condition_from, $item->condition_to);
//                    $secs = self::getTotalHours($result["list"][$item->user_id]["wHistory"]);

                    $result["list"][$item->user_id]["wHistory"][] = [$item->condition_from, $item->condition_to];

                    $secs = Calculate_Duration($item->condition_from, $item->condition_to, true) * 3600;
                    $result["list"][$item->user_id]["total"] += $secs;

                    $total += $secs;
                }
            }

            $result["total"] = $total;

            return $result;
        }
        else {
            return $list;
        }
    }

    static function setWorkingPeriods(&$wHistory, $reportTime, $releaseTime){
        $index = $hourType = $value = null;

        foreach ($wHistory as $i => $each) {
            $start = strtotime($each[0]);
            $end = strtotime($each[1]);

            if (strtotime($reportTime) >= $start && strtotime($reportTime) <= $end){

                $index = $i;
                $hourType = 1;
                $value = $each[0];
                // Pick the larger release time
                $releaseTime = strtotime($releaseTime) >= $end ? $releaseTime : $each[1];
                break;
            }
            else if (strtotime($releaseTime) >= $start && strtotime($releaseTime) <= $end){
                $index = $i;
                $hourType = 2;
                $value = $each[1];
                break;
            }
            else if (strtotime($reportTime) <= $start && strtotime($releaseTime) >= $end){

                $index = $i;
                $hourType = 3;
                break;
            }
        }

        if ($index || $index === 0) {
            if ($hourType == 1){
                $wHistory[$index] = [ $value, $releaseTime ];
            }
            else if ($hourType == 2){
                $wHistory[$index] = [ $reportTime, $value ];
            }
            else {
                $wHistory[$index] = [ $reportTime, $releaseTime ];
            }
        }
        else {
            $wHistory[] = [$reportTime, $releaseTime];
        }
    }

    static function getTotalHours($wHistory)
    {
        $totalHours = 0;

        foreach ($wHistory as $each) {
            $totalHours += CalculateDifference($each[0], $each[1], true);
        }

        return $totalHours;
    }

    public static function getStaffLeaveRequests($from, $to, $userIDs = [], $airportID = null, $returnObjects = false){
        $list = [];

        $requests = UserLeaveRequest::with(["leave"]);

        if ($airportID){
            $requests->join("users", "users.id", "=", "users__leave_requests.user_id")
                     ->join("locations", "users.location_id", "=", "locations.id")
                     ->where("airport_id", $airportID);
        }

        $userIDs = is_array($userIDs) ? $userIDs : [$userIDs];

        $requests = $requests->whereIn("user_id", $userIDs)
            ->whereNull("users__leave_requests.deleted_at")
            ->where("status_id", 2)
            ->where(function($sql) use($from, $to){
                $sql->whereBetween("date_to", [$from, $to])
                    ->orWhereBetween("date_from", [$from, $to] )
                    ->orWhere(function ($sql2) use($from, $to){
                        $sql2->where("date_from", "<=", $from)
                            ->where("date_to", ">=", $to);
                    });
            })
            ->get([
                "users__leave_requests.*"
            ]);

        $history = UserHistory::with(["condition", "airline"]);

        if ($airportID){
            $history->join("users", "users.id", "=", "users__history.user_id")
                    ->join("locations", "users.location_id", "=", "locations.id")
                    ->where("airport_id", $airportID);
        }

        $history = $history->whereIn("user_id", $userIDs)
            ->whereNull("users__history.deleted_at")
            ->where(function($sql) use($from, $to){
                $sql->whereBetween("condition_to", [$from, $to])
                    ->orWhereBetween("condition_from", [$from, $to] )
                    ->orWhere(function ($sql2) use($from, $to){
                        $sql2->where("condition_from", "<=", $from)
                            ->where("condition_to", ">=", $to);
                    });
            })
            ->get([
                "users__history.*"
            ]);

        if ($returnObjects){
            $result = [];
            foreach ($requests as $each){
                $result[] = $each;
            }
            foreach ($history as $each){
                $result[] = $each;
            }

            return $result;
        }

        /*
        $offDays = User::whereIn("id", $userIDs)
            ->whereNotNull("roster_off_days")
            ->get();

        foreach ($offDays as $each) {
            if (!isset($list[$each->id])){
                $list[$each->id] = [];
            }

            $days = explode(";", $each->roster_off_days);
            for($i = strtotime($from); $i < strtotime($to); $i += 24 * 60 * 60){
                $weekDay = date("w", $i);
                if (in_array($weekDay, $days)){
                    $list[$each->id][] = [date("Y-m-d", $i)." 00:00:00", date("Y-m-d", $i)." 23:59:59", "Day off"];
                }
            }
        }
        */

        foreach ($requests as $request) {
            if (!isset($list[$request->user_id])){
                $list[$request->user_id] = [];
            }

            $list[$request->user_id][] = [
                $request->date_from,
                date("Y-m-d", strtotime($request->date_to)). " 23:59:59",
                ($request->leave ? $request->leave->name : ""),
                1,
                "",
                "-",
                1,
                "",
                "",
            ];
        }

        foreach ($history as $request) {
            if (!isset($list[$request->user_id])){
                $list[$request->user_id] = [];
            }


            $list[$request->user_id][] = [
                $request->condition_from,
                $request->condition_to,
                ($request->condition ? $request->condition->name : ""),
                0,
                $request->notes,
                ($request->airline ? $request->airline->airline : "-"),
                $request->condition ? $request->condition->skip_total : null,
                $request->condition ? $request->condition->bg_class : "",
                $request->condition ? $request->condition->text_class : "",
            ];
        }

        return $list;
    }

}

