<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model as Eloquent;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\HasApiTokens;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;
use OwenIt\Auditing\Contracts\UserResolver;

class User extends Authenticatable implements AuthenticatableContract, CanResetPasswordContract,
    AuditableContract, UserResolver {

    use HasApiTokens, HasFactory, CanResetPassword, Auditable, Notifiable;

    protected $auditExclude = [
        'updated_by',
        'created_by',
        'remember_token',
        'last_login',
    ];

    public function notifications()
    {
        return $this->morphMany(NotificationTable::class, 'notifiable')->orderBy('created_at', 'desc');
    }

    protected $table = 'users';

    /**
     * {@inheritdoc}
     */
    public static function resolve()
    {
        return Auth::check() ? Auth::user()->getAuthIdentifier() : null;
    }

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'idcard_expiry',
        'first_name',
        'last_name',
        'email',
        'passport_doi',
        'passport_dex',
    ];

    public $timestamps = TRUE;

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    public static $rules = array(
        'first_name'            => 'required|alpha|min:2',
        'last_name'             => 'required|alpha|min:2',
        'email'                 => 'required|email|unique:users',
        'department_id'         => 'required|exists:structure__departments,id',
        'password'              => 'required|alpha_num|between:6,18|confirmed',
        'password_confirmation' => 'required|alpha_num|between:6,18',
    );




    public function getAuthIdentifier()
    {
        return $this->getKey();
    }

    public function getAuthPassword()
    {
        return $this->password;
    }

    public function getRememberToken()
    {
        return $this->remember_token;
    }

    public function setRememberToken($value)
    {
        $this->remember_token = $value;
    }

    public function getRememberTokenName()
    {
        return "remember_token";
    }

    public function getReminderEmail()
    {
        return $this->email;
    }

    public function license(){
        return $this->hasMany(StaffLicense::class, "user_id");
    }

    public function service(){
        return $this->hasMany(StaffService::class, 'user_id');
    }

    public function preferredOff(){
        return $this->hasOne(UserPreferredOff::class, 'user_id');
    }

    public function CabinCrew(){
        return $this->hasOne('App\\Models\\CabinCrew', 'user_id');
    }

    public function FlightCrew(){
        return $this->hasOne('App\\Models\\CaptainFirstOfficer', 'user_id');
    }

    public function updatedBy(){
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function reportsTo(){
        return $this->belongsTo('App\Models\User', 'reports_to_user_id');
    }

    public function department()
    {
        return $this->belongsToMany("App\\Models\\Department", "users__departments", "user_id");
    }

    public function entitlements(){
        return $this->hasMany('App\\Models\\UserEntitlement', 'user_id');
    }

    public function role(){
        return $this->belongsTo('App\Models\UserRole', 'user_role_id');
    }

    public function location(){
        return $this->belongsTo('App\Models\Location', 'location_id');
    }

    public static function getStation($user){
        return $user && $user->location && $user->location->airport ? $user->location->airport->iata : "Base N/A";
    }

    public function crewAircraftTypes(){
        return $this->hasMany('App\Models\CrewAircraftType', 'user_id');
    }

    public function crewRestrictionRoutes(){
        return $this->hasMany('App\Models\CrewRestrictionRoute', 'user_id');
    }

    public function crewRestrictionPairings(){
        return $this->hasMany('App\Models\CrewRestrictionPairing', 'user_id');
    }

    public function crewRestrictionAircraftTypes(){
        return $this->hasMany('App\Models\CrewRestrictionAircraftType', 'user_id');
    }

    public function chatStatus(){
        return $this->belongsTo('App\Models\ChatStatus', 'chat_status_id');
    }

    public function pob(){
        return $this->belongsTo('App\Models\Country', 'pob_country_id');
    }

    public function addressCity(){
        return $this->belongsTo('App\Models\City', 'address_city_id');
    }

    public function nationality(){
        return $this->belongsTo('App\Models\Nationality', 'nationality_id');
    }

    public function maritalStatus(){
        return $this->belongsTo('App\Models\MaritalStatus', 'marital_status_id');
    }

    /**
     * Return link to appropriate user position
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function position()
    {
        return $this->belongsToMany("App\\Models\\Position", "users__departments", "user_id");
    }

    public function systemLanguage(){
        return $this->belongsTo(SystemLanguage::class, "system_language_id");
    }

    public function languages()
    {
        return $this->belongsToMany("App\\Models\\Language", "users__languages", "user_id");
    }

    public function userModule(){
        return $this->hasMany(UserModule::class, 'user_id');
    }

    public function userFamily(){
        return $this->hasMany(UserFamily::class, 'user_id');
    }
    /**
     * Get all standby by user ID
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function standby()
    {
        return $this->hasMany("App\\Models\\StandbyCrew", "user_id");
    }

    /**
     * Get flight crew items by user ID
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function flightCrewItem()
    {
        return $this->hasMany("App\\Models\\FlightCrew", "user_id");
    }


    public function userHistory()
    {
        return $this->hasMany("App\\Models\\UserHistory", "user_id")
            ->whereNull("deleted_at")
            ->orderBy("condition_from");
    }

    public function history()
    {
        return $this->hasMany(UserHistory::class, "user_id");
    }


    /**
     * Get customer user property Full Name
     * @return string
     */
    public function getUserFullNameAttribute()
    {
        return $this->attributes["last_name"] . " " . $this->attributes["first_name"];
    }

    /**
     * Get all captains and first officers
     * @return array|static[]
     */
    public function getCaptainsAndFirstOfficers()
    {
        $users = User::join("crew__flight", $this->table . ".id", "=", "crew__flight.user_id")
            ->orderBy("first_name")
            ->get();
        return $users->pluck("UserFullName", "user_id")->all();
    }


    public static function CabinCrewWithPosition()
    {
        $crew = User::join('users__departments', 'users.id', '=', 'users__departments.user_id')
            ->join('crew__cabin', 'users.id','=','crew__cabin.user_id')
            ->where('department_id', 14)
            ->get(['users.id as id', 'position_id'])
            ->toArray();
        $ccm = [];
        foreach ($crew as $each){
            $ccm[$each['id']] = $each;
        }
        return $ccm;
    }

    public static function getUsers($onlyActive = false){

        $result = User::with(["userHistory"])
            ->whereNull("vs")
            ->orderBy("last_name")
            ->orderBy("first_name");

        if ($onlyActive){
            $result->where(function($sql){
                    $sql->whereNull("users.resigned_date")
                        ->orWhere("users.resigned_date", EMPTY_DATE)
                        ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                })
                ->whereNull('deleted_at');
        }

        return $result->get();
    }

    public static function listUsers($lastNameFirst = true, $onlyActive = false){

        $select = $lastNameFirst ? "CONCAT(last_name, ' ', first_name) AS full_name" : "CONCAT(first_name, ' ', last_name) AS full_name";

        $result = User::select([DB::raw($select), "id"])
            ->whereNull("vs");

        if ($lastNameFirst){
            $result->orderBy("last_name");
        }
        else {
            $result->orderBy("first_name");
        }

        if ($onlyActive){
            $result->where(function($sql){
                    $sql->whereNull("users.resigned_date")
                        ->orWhere("users.resigned_date", EMPTY_DATE)
                        ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                })
                ->whereNull('deleted_at');
        }

        $result = $result->pluck("full_name", "id")
            ->all();

        return ["" => "Select Staff"] + $result;
    }

    public static function FlightCrewWithPosition()
    {
        $crew = User::join('users__departments', 'users.id', '=', 'users__departments.user_id')
            ->join('crew__flight', 'users.id','=','crew__flight.user_id')
            ->where('department_id', 12)
            ->get(['users.id as id', 'position_id'])
            ->toArray();
        $fcm = [];
        foreach ($crew as $each){
            $fcm[$each['id']] = $each;
        }
        return $fcm;
    }

    /**
     * Get User Full Name And Id For Select Boxes
     * @param $userId
     * @return array|bool
     */
    public static function getUserFullNameAndId($userId){
        $user = User::find($userId);
        if ($user){
            return $user->id;
        }
        return false;
    }

    public static function getUsersListByID(){
        $users = User::whereNull("deleted_at")
            ->get([
                "id",
                "first_name",
                "last_name",
            ])
            ->keyBy("id");

        return $users;
    }

    public static function getListByID(){
        $users = User::whereNull("deleted_at")
            ->get([
                DB::raw("CONCAT(first_name, ' ', last_name) as name"),
                "id"
            ])
            ->pluck("name", "id")
            ->all();

        return $users;
    }

    public static function getUsersByIDs($ids){
        $users = User::with([
            "location",
            "location.airport",
            "department",
            "position",
            "updatedBy",
        ])
            ->whereIn("id", $ids)
            ->whereNull("deleted_at")
            ->get();

        return $users;
    }

    public static function getCrewAircraftTypes($crew){
        $types = "";
        if ($crew->crewAircraftTypes){
            foreach ($crew->crewAircraftTypes as $each) {
                $types .= $each->aircraftType ? $each->aircraftType->name.", " : "";
            }

            $types = rtrim($types, ", ");
        }

        return $types;
    }

    /**
     * Get all First Officers
     * @param $type
     * @return array
     */
    public static function getFlightCrew($type)
    {
        switch($type){
            case FCM_CREW:
                $positionsTypes = [
                    FCM_CPT_TYPE_ID,
                    FCM_FA_TYPE_ID
                ];
                break;

            case FCM_CP_CREW:
                $positionsTypes = [
                    FCM_CPT_TYPE_ID
                ];
                break;

            case FCM_FO_CREW:
                $positionsTypes = [
                    FCM_FA_TYPE_ID
                ];
                break;

            case CCM_CREW:
                $positionsTypes = [
                    CCM_PSR_TYPE_ID,
                    CCM_CC_TYPE_ID
                ];
                break;

            case CCM_PR_CREW:
                $positionsTypes = [
                    CCM_PSR_TYPE_ID
                ];
                break;

            case CCM_FA_CREW:
                $positionsTypes = [
                    CCM_CC_TYPE_ID
                ];
                break;

            default:
                $positionsTypes = [];
                break;
        }

        if (count($positionsTypes) == 0){
            return [];
        }

        $crew = User::select([
            "users.id",
            "users.first_name",
            "users.last_name"
        ])
            ->join("users__departments", "users__departments.user_id", "=", "users.id")
            ->join('structure__positions', 'structure__positions.id', '=', 'users__departments.position_id')
            ->whereIn('structure__positions.type', $positionsTypes)
            ->whereNull("users.deleted_at")
            ->where(function($sql){
                $sql->whereNull("users.resigned_date")
                    ->orWhere("users.resigned_date", EMPTY_DATE)
                    ->orWhere("users.resigned_date", ">", date("Y-m-d"));
            })
            ->orderBy("last_name")
            ->get();

        return $crew->pluck("UserFullName", "id")->all();
    }

    /**
     * Get all cabin crew
     * @return array
     */
    public function getCabinCrew()
    {
        $users = User::join("crew__cabin", $this->table . ".id", "=", "crew__cabin.user_id")
            ->orderBy("first_name")
            ->get();
        return $users->pluck("UserFullName", "user_id")->all();
    }


    public static function getCabinCrewList($type = false)
    {
        $users = User::join("crew__cabin", "users.id", "=", "crew__cabin.user_id");

        if ($type){
            switch ($type){
                case PURSER:
                    $users->where('is_purser', 1)
                        ->orWhere("structure__positions.type", CCM_PSR_TYPE_ID);
                    break;

                case CABIN_CREW:
                    $users->whereNull('is_purser')
                        ->orWhere("structure__positions.type", CCM_CC_TYPE_ID);
                    break;
            }
        }

        return $users->orderBy("first_name")
            ->get()
            ->pluck("UserFullName", "user_id")
            ->all();
    }

    public static function getFlightCrewList($type = false)
    {
        $users = User::join("crew__flight", "users.id", "=", "crew__flight.user_id")
            ->join("users__departments", "users.id", "=", "users__departments.user_id")
            ->join("structure__positions", "structure__positions.id", "=", "users__departments.position_id");

        if ($type){
            switch ($type){
                case CAPTAIN:
                    $users->where('is_captain', 1)
                        ->orWhere("structure__positions.type", FCM_CPT_TYPE_ID);
                    break;

                case FIRST_OFFICER:
                    $users->whereNull('is_captain')
                        ->orWhere("structure__positions.type", FCM_FA_TYPE_ID);
                    break;
            }
        }

        return $users->orderBy("first_name")
            ->get()
            ->pluck("UserFullName", "user_id")
            ->all();
    }

    /**
     * @return array
     */
    public function getCabinCrewDetails()
    {
        $users = User::join("crew__cabin", $this->table . ".id", "=", "crew__cabin.user_id")
            ->orderBy("first_name")
            ->get();
        return $users;
    }


    public static function getUserBirthDays($date, $range = false){
        $dates = [date('Y-m-d', strtotime($date))];

        if ($range){
            for ($i = 1; $i < $range; $i++){
                $dates[] = date('Y-m-d', strtotime("+ $i days", strtotime($date)));
            }
        }


        $birthdays = User::join('users__departments', 'users__departments.user_id', '=', 'users.id')
            ->join('structure__departments', 'users__departments.department_id', '=', 'structure__departments.id')
            ->whereNull('vs')
            ->where(function($sql) use ($dates){
                foreach ($dates as $each){
                    $sql->orWhere(function($query) use ($each) {
                        $query->where(DB::raw('MONTH(dob)'), date('m', strtotime($each)))
                            ->where(DB::raw('DAY(dob)'), date('d', strtotime($each)));
                    });
                }
            })
            ->orderBy(DB::raw('MONTH(dob)'))
            ->orderBy(DB::raw('DAY(dob)'))
            ->get(['users.id', 'dob', 'first_name', 'last_name', 'name']);

        return $birthdays;
    }

    /**
     * Get User By Department And Subdepartment
     * @param null $department_id
     * @param null $subdepartment_id
     * @param bool $queryContinue
     * @return mixed
     */
    public static function getUsersByDepartment($department_id = null, $subdepartment_id = null, $queryContinue = FALSE, $showDeleted = null){
        $users =  User::with(['userFamily', 'reportsTo', 'location'])->select([
            "users.id",
            "users.first_name",
            "users.last_name",
            "users.reports_to_user_id",
            "users.marital_status_id",
            "users.email",
            "users.dob",
            "users.staff_number",
            "users.resigned_date",
//            "users.languages",
            "users.location_id",
//            "users.phone",
            "users.doj",
            "users.passport_num",
            "users.passport_doi",
            "users.passport_dex",
            "users.pob_country_id",
            "users.nationality_id",
            "users.idcard_number",
            "users.idcard_expiry",
            "users.picture",
            "users.thumb",
            "users.activated",
            "users.user_role_id",
            "users.deleted_at",
            "users__departments.department_id",
            "users__departments.subdepartment_id",
            "users__departments.position_id",
            "structure__departments.name AS department_name",
            "structure__subdepartments.name AS subdepartment_name",
            "structure__positions.name AS position_title",
            "structure__departments.has_subdepartment"
        ])
            ->join("users__departments", "users.id", "=", "users__departments.user_id")
            ->leftJoin("structure__departments", "users__departments.department_id", "=", "structure__departments.id")
            ->leftJoin("structure__positions", "users__departments.position_id", "=", "structure__positions.id")
            ->leftJoin("structure__subdepartments", "users__departments.subdepartment_id", "=", "structure__subdepartments.id")
            ->whereNull('vs');

        if (!$showDeleted){
            $users->whereNull('users.deleted_at');
        }

        if (isset($department_id))
            $users->where('users__departments.department_id', $department_id);

        if (isset($subdepartment_id))
            $users->where('users__departments.subdepartment_id', $subdepartment_id);

        if ($queryContinue)
            return $users;
        else
            return $users->orderBy("id")->get();

    }

    /**
     * Get user full information
     * @param $userId
     * @return \Illuminate\Support\Collection|static
     */
    public static function getUserFullInformation($userId)
    {
        return User::with('userFamily')->select([
            "users.id",
            "users.first_name",
            "users.last_name",
            "users.is_contractor",
            "users.marital_status_id",
            "users.email",
            "users.dob",
            "users.staff_number",
            "users.resigned_date",
//            "users.languages",
            "users.location_id",
//            "users.phone",
            "users.doj",
            "users.passport_num",
            "users.passport_doi",
            "users.passport_dex",
            "users.pob_country_id",
            "users.nationality_id",
            "users.idcard_number",
            "users.idcard_expiry",
            "users.picture",
            "users.thumb",
            "users.activated",
            "users.user_role_id",
            "users__departments.department_id",
            "users__departments.subdepartment_id",
            "users__departments.position_id",
            "structure__departments.name AS department_name",
            "structure__subdepartments.name AS subdepartment_name",
            "structure__positions.name AS position_title",
            "structure__departments.has_subdepartment"
        ])
            ->join("users__departments", "users.id", "=", "users__departments.user_id")
            ->join("structure__departments", "users__departments.department_id", "=", "structure__departments.id", "left")
            ->join("structure__positions", "users__departments.position_id", "=", "structure__positions.id", "left")
            ->join("structure__subdepartments", "users__departments.subdepartment_id", "=", "structure__subdepartments.id", "left")
            ->findOrFail($userId);
    }


    /**
     * Get All Crew Or (Filter By Designation, UserId)
     * @param $crewType
     * @param bool|false $designation
     * @param bool|false $userId
     * @param bool|false $asArray
     * @param array $joinTables
     * @return mixed
     */
    public static function getCrew($crewType, $designation = false, $userId = false, $asArray = false, $joinTables = [
        'userHistory',
        'crewAircraftTypes',
        'location'
    ] ){
        $crew = null;

        //if !(in_array($crewType, []))
        $selectGeneral = [
            'users.id as id',
            'position_id',
            'nationality_id',
            'pob_country_id',
            "structure__positions.department_id",
            "structure__positions.subdepartment_id",
            "structure__positions.id as position_id",
            "structure__departments.name AS department_name",
            "structure__subdepartments.name AS subdepartment_name",
            "structure__positions.name AS position_title",
            "structure__positions.type AS position_type",
            "structure__departments.has_subdepartment",
            'marital_status_id',
            'idcard_number',
            'idcard_expiry',
            'first_name',
            'last_name',
            'passport_num',
            'passport_doi',
            'passport_dex',
            'staff_number',
            'email',
            'phone',
            'mobile',
            'dob',
            'doj',
            'picture',
            'thumb',
            'location_id',
            'reports_to_user_id',
            'is_contractor'
        ];

        switch ($crewType) {
            case FCM_CREW:
                $crew = User::with($joinTables)
                    ->select(array_merge($selectGeneral, [
                        'hour_restriction',
                        'english_level',
                        'is_captain',
                        'aruba_validation_from',
                        'aruba_validation_to',
                        'cmc_from',
                        'cmc_to',
                        'sep_12',
                        'sep_36',
                        'crm_from',
                        'crm_to',
                        'dgr_from',
                        'dgr_to',
                        'security',
                        'medical_from',
                        'medical_to',
                        'flying_license_from',
                        'flying_license_to',
                        'license_attachment_from',
                        'license_attachment_to',
                        'lpc_ir_from',
                        'lpc_ir_to',
                        'opc_from',
                        'opc_to',
                        'tri_sfi_from',
                        'tri_sfi_to',
                        'line_check_from',
                        'line_check_to'
                    ]))
                    ->join('users__departments', 'users.id', '=', 'users__departments.user_id')
                    ->join('crew__flight', 'users.id', '=', 'crew__flight.user_id')
                    ->leftJoin('structure__departments', 'structure__departments.id', '=', 'users__departments.department_id')
                    ->leftJoin('structure__subdepartments', 'structure__subdepartments.id', '=', 'users__departments.subdepartment_id')
                    ->leftJoin('structure__positions', 'structure__positions.id', '=', 'users__departments.position_id')
                    ->where(function($sql){
                        $sql->whereNull("users.resigned_date")
                            ->orWhere("users.resigned_date", EMPTY_DATE)
                            ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                    });

                if ($designation) {
                    if ($designation == 'capt'){
                        $crew->where(function($sql){
                            $sql->where('is_captain', 1)
                                ->orWhere("structure__positions.type", FCM_CPT_TYPE_ID);
                        });
                    }
                    elseif ($designation == 'fo'){
                        $crew->where(function($sql){
                            $sql->whereNull('is_captain')
                                ->orWhere("structure__positions.type", FCM_FA_TYPE_ID);
                        });
                    }
                }
                break;
            case CCM_CREW:
                $crew = User::with($joinTables)
                    ->select( array_merge($selectGeneral, [
                        'is_purser',
                        'is_business_crew',
                        'cmc_from',
                        'cmc_to',
                        'sep_from',
                        'sep_to',
                        'crm_from',
                        'crm_to',
                        'dgr_from',
                        'dgr_to',
                        'security_from',
                        'security_to',
                        'practical_from',
                        'practical_to',
                        'medical_next',
                        'license_validity',
                        'line_check_from',
                        'line_check_to'
                    ]))
                    ->join('users__departments', 'users.id', '=', 'users__departments.user_id')
                    ->join('crew__cabin', 'users.id', '=', 'crew__cabin.user_id')
                    ->leftJoin('structure__departments', 'structure__departments.id', '=', 'users__departments.department_id')
                    ->leftJoin('structure__subdepartments', 'structure__subdepartments.id', '=', 'users__departments.subdepartment_id')
                    ->leftJoin('structure__positions', 'structure__positions.id', '=', 'users__departments.position_id')
                    ->where(function($sql){
                        $sql->whereNull("users.resigned_date")
                            ->orWhere("users.resigned_date", EMPTY_DATE)
                            ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                    });

                if ($designation) {
                    if ($designation == 'psr'){
                        $crew->where(function($sql){
                            $sql->where('is_purser', 1)
                                ->orWhere("structure__positions.type", CCM_PSR_TYPE_ID);
                        });
                    }
                    elseif ($designation == 'cc'){
                        $crew->where(function($sql){
                            $sql->whereNull('is_purser')
                                ->orWhere("structure__positions.type", CCM_CC_TYPE_ID);
                        });
                    }
                }
                break;
        }

        if (!$crew){
            return null;
        }

        // Hide Deleted Crew Members
        $crew->whereNull('users.deleted_at');

        if ($userId)
            $crew = $crew->where('users.id', $userId)->first();
        else
            $crew = $crew->orderBy('id')->get();

        if ($asArray)
            $crew = $crew->toArray();

        return $crew;
    }

    /**
     * Crew List By Type
     * @param $crewType
     * @return array|bool
     */
    public static function getCrewDesignationArray($crewType){
        $listCrew = false;
        if ($crewType) {
            switch ($crewType) {
                case "fcm":
                    $listCrew = ['' => 'All', 'capt' => 'Captain', 'fo' => 'First Officer'];
                    break;
                case "ccm":
                    $listCrew = ['' => 'All', 'psr' => "Purser", 'cc' => "Cabin Crew"];
                    break;
            }
        }
        return $listCrew;

    }

    /**
     * Unserialize Passport Data and Return Back User Object Or Array Of Users
     * @param $user
     * @return mixed
     */
    public static function unserializeUserPassportData($user){
        if (isset($user) && $user) {
            if (count($user) > 1) {
                foreach ($user as $i => $each) {
                    if ($each->passport_num && $each->passport_num != '') {
                        $passport_num = unserialize($each->passport);
                        if (!isset($passport["p_number"])) {
                            $passport["p_number"] = "";
                        }
                        if (!isset($passport["p_issued_date"])) {
                            $passport["p_issued_date"] = "";
                        }
                        if (!isset($passport["p_exp_date"])) {
                            $passport["p_exp_date"] = "";
                        }
                        $user[$i]->passport = $passport;
                        $user[$i]->passport = $passport;
                        $user[$i]->passport = $passport;
                    }
                    else {
                        $user[$i]->passport = [
                            "p_number" => "",
                            "p_issued_date" => "",
                            "p_exp_date" => ""
                        ];
                    }
                }
            }
            else {
                if ($user->passport && $user->passport != '') {
                    $passport = unserialize($user->passport);
                    if (!isset($passport["p_exp_date"])) {
                        $passport["p_exp_date"] = "";
                    }
                    if (!isset($passport["p_issued_date"])) {
                        $passport["p_issued_date"] = "";
                    }
                    if (!isset($passport["p_number"])) {
                        $passport["p_number"] = "";
                    }
                    $user->passport = $passport;
                }
                else {
                    $user->passport = [
                        "p_number"      => "",
                        "p_issued_date" => "",
                        "p_exp_date"    => ""
                    ];
                }
            }
        }
        return $user;
    }

    /**
     * @param $user
     * @return bool|string
     */
    public static function getCrewType($user){
        $crewType = false;
        if ($user) {
            if ($user->position && count($user->position)) {
                $position = trim($user->position[0]->name);

                if (in_array($user->position[0]->type, [CCM_PSR_TYPE_ID, CCM_CC_TYPE_ID]) ) {
                    $crewType = CCM_CREW;
                }
                elseif (in_array($user->position[0]->type, [FCM_CPT_TYPE_ID, FCM_FA_TYPE_ID])) {
                    $crewType = FCM_CREW;
                }
            }
            else {
                if ($user->CabinCrew){
                    return CCM_CREW;
                }
                else if ($user->FlightCrew){
                    return FCM_CREW;
                }
            }
        }
        return $crewType;
    }


    /**
     * Get Crew Type (CAPTAIN, FIRST_OFFICER, PURSER, CABIN_CREW)
     * @param $userId
     * @return bool|string
     */
    public static function getSpecificCrewType($userId){
        $user = User::find($userId);
        $crewType = "";
        if ($user) {
            $positionType = $user->position[0] ? trim($user->position[0]->type) : null;

            if ($positionType) {
                if ($positionType == FCM_CPT_TYPE_ID){
                    $crewType = CAPTAIN;
                }
                elseif ($positionType == FCM_FA_TYPE_ID){
                    $crewType = FIRST_OFFICER;
                }
                elseif ($positionType == CCM_PSR_TYPE_ID){
                    $crewType = PURSER;
                }
                elseif ($positionType == CCM_CC_TYPE_ID){
                    $crewType = CABIN_CREW;
                }

            }
            return $crewType;
        }
        return false;
    }


    /**
     * If Department Given
     * @param $userId
     * @return array
     */
    public static function getAllReportToEmployees($userId){

        $organization = User::recursiveReportToEmployees($userId);

        // Head
        $total = [User::find($userId)];
        if ($organization && count($organization)) {
            array_walk_recursive($organization, function (&$each) use (&$total) {
                if ($each)
                    $total[] = $each;
            });
        }

        return $total;
    }


    public static function getDepartmentStructure($department = null){

        $position = Position::whereNull("reports_to_id")
            ->where(function($sql){
                $sql->where("name", "Managing Director")
                    ->orWhere("name", "CEO");
            })
            ->first();

        $user = findUserByPosition($position->id);

        return [
            "user"      => $user,
            "position"  => $position,
            "reportees" => findReportsToPositionsTopDown($position, true)
        ];

        $head = User::getDepartmentHead($department);

        if (!$head){
            return [];
        }

        $organization = User::recursiveReportToEmployees($head->id, 0);

        // Head
        $total = [ $head ];
        if ($organization && count($organization)) {
            array_walk_recursive($organization, function (&$each) use (&$total) {
                if ($each)
                    $total[] = $each;
            });
        }

        return $total;
    }



    public static function recursiveReportToEmployees($userId, $calls = 0){
        if (!$userId){
            return null;
        }

        $reportees = User::with(['position'])
            ->where('reports_to_user_id', $userId)
            ->get();


        foreach ($reportees as $user) {
            $total[] = clone $user;
            $reportees2level = User::where('reports_to_user_id', $user->id)->get();

            if ($calls > 2){
                debug($calls);
                return $total;
            }

            if ($reportees2level) {
                foreach ($reportees2level as $_user) {
                    $total[] = clone $_user;
                    $total[] = self::recursiveReportToEmployees($_user->id, ++$calls);
                }
            }
        }
        return isset($total) ? $total : null;

    }


    public static function getDepartmentHead($department){

        if ($department->name == CEO) {
            $position = CHIEF_EXECUTIVE;
        }
        else {
            $position = DIRECTOR;
        }

        $head = User::with(['position'])
            ->join("users__departments", "users__departments.user_id", "=", "users.id" )
            ->join("structure__positions", "structure__positions.id", "=", "users__departments.position_id")
            ->where("users__departments.department_id", $department->id)
            ->where("structure__positions.name", "LIKE", "%{$position}")
            ->first(["users.*"]);

        return $head;
    }

    public static function getUsersList($departmentId = null, $includeCEO = false){

        $users = User::join("users__departments", "users__departments.user_id", "=", "users.id")
            ->join("structure__positions", "structure__positions.id", "=", "users__departments.position_id");

        if ($departmentId) {
            if ($includeCEO){
                $users->where(function($sql) use ($departmentId) {
                    $sql->where("users__departments.department_id", $departmentId)
                        ->orWhere("structure__positions.name", "LIKE", "%".CHIEF_EXECUTIVE."%");
                });
            }
            else {
                $users->where("users__departments.department_id", $departmentId);
            }
        }

        $users->select(DB::raw("CONCAT(first_name,' ',last_name) AS full_name, s_users.id as id"))
            ->whereNull('vs');

        return $users->orderBy('first_name')
            ->pluck('full_name', 'id')
            ->all();
    }

    public static function getUsersByLocation($airports = [], $terminatedStaff = false, $deletedStaff = false, $excludeIDs = null, $onlyActiveUsers = false, $orderByStation = false){

        $users = User::with([
            "location",
            "location.airport",
            "department",
            "position",
            "updatedBy",
        ])
            ->leftJoin("locations", "locations.id", "=", "users.location_id")
            ->leftJoin("airports", "airports.id", "=", "locations.airport_id")
//                      ->whereNotNull("location_id")
            ->whereNull("users.vs");
//                      ->whereNotNull("idcard_number")

        if ($airports){
            $airports = is_array($airports) ? $airports : [$airports];
            $users->where(function($sql) use ($airports) {
                $sql->whereNull("location_id")
                    ->orWhere(function($sql2) use ($airports){
                        $sql2->whereNotNull("location_id")
                            ->whereIn("locations.airport_id", $airports);
                    });
            });
        }

        if ($deletedStaff){
            $users->whereNotNull("users.deleted_at");
        }
        elseif ($terminatedStaff){
            $users->where(function($sql){
                $sql->whereNotNull("users.resigned_date")
                    ->where("users.resigned_date", "<=", date("Y-m-d"));
            });
        }
        else {
            $users->whereNull("users.deleted_at")
                ->where(function($sql){
                    $sql->whereNull("users.resigned_date")
                        ->orWhere("users.resigned_date", EMPTY_DATE)
                        ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                });
        }

        if ($excludeIDs && count($excludeIDs)){
            $users->whereNotIn("users.id", $excludeIDs);
        }

        if ($orderByStation){
            $users->orderBy("users.location_id");
        }

        if ($onlyActiveUsers){
            $users->where("users.activated", true);
        }

        return $users->orderBy('users.first_name')
            ->get(['users.*']);
    }

    public static function getStaffWithService($occStaffOnly = null, $airportIDs = []){

        $staff = StaffService::join("services", "services.id", "=", "staff__services.service_id")
            ->join("users", "users.id", "=", "staff__services.user_id");

        if ($airportIDs){
            $airportIDs = is_array($airportIDs) ? $airportIDs : [$airportIDs];
            $staff->join("locations", "locations.id", "=", "users.location_id")
                ->whereIn("locations.airport_id", $airportIDs);
        }

        if ($occStaffOnly){
            $staff->join("users__departments", "users__departments.user_id", "=", "users.id")
                ->join("structure__departments", "structure__departments.id", "=", "users__departments.department_id")
                ->where("structure__departments.name", OCC_DEPARTMENT);
        }


        $staff = $staff->whereNotNull("airline_id")
            ->whereNotNull("service_id")
            ->whereNull("users.deleted_at")
            ->where(function($sql){
                $sql->whereNull("users.resigned_date")
                    ->orWhere("users.resigned_date", EMPTY_DATE)
                    ->orWhere("users.resigned_date", ">", date("Y-m-d"));
            })
            ->whereNull("users.vs")
            ->groupBy("staff__services.user_id")
            ->get([
                "staff__services.user_id",
            ])
            ->pluck("user_id")
            ->all();

        return $staff;

    }


    public static function getUsersByStation($airports = [], $orderByFirstName = false){

        $users = User::with([
            "location",
            "location.airport",
            "department",
            "position",
            "entitlements",
            "updatedBy",
        ])
            ->leftJoin("locations", "locations.id", "=", "users.location_id")
            ->leftJoin("airports", "airports.id", "=", "locations.airport_id")
//                      ->whereNotNull("location_id")
            ->whereNull("users.vs");
//                      ->whereNotNull("idcard_number")

        if ($airports){
            $airports = is_array($airports) ? $airports : [$airports];
            $users->where(function($sql) use ($airports) {
                $sql->whereNull("location_id")
                    ->orWhere(function($sql2) use ($airports){
                        $sql2->whereNotNull("location_id")
                            ->whereIn("locations.airport_id", $airports);
                    });
            });
        }

        $users->whereNull("users.deleted_at")
                ->where(function($sql){
                    $sql->whereNull("users.resigned_date")
                        ->orWhere("users.resigned_date", EMPTY_DATE)
                        ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                });

        if (!$orderByFirstName){
            $users->orderBy("users.location_id");
        }

        return $users->orderBy('users.first_name')
            ->get(['users.*']);
    }

}
