<?php namespace App\Models; use App\Classes\Staff\Roster\Models\StaffServices;
use App\Classes\Staff\Roster\Staff\Staff;
use Illuminate\Database\Eloquent\Model as Eloquent;
use Illuminate\Support\Facades\Auth;


/**
 * Created by PhpStorm.

 */
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class StaffService extends Eloquent implements AuditableContract{

    use Auditable;

    protected $auditExclude = [
        'updated_by',
        'created_by',
    ];

    protected $table = "staff__services";

    public $timestamps = true;

    protected $guarded = [
        "id",
        "created_at",
        "updated_at"
    ];

    public static function getOptionsArray(){
        return [
            'Supervision'       => 'supervision',
            'Check-in'          => 'check_in',
            'Boarding'          => 'boarding',
            'E.B. Collection'   => 'eb_collection',
            'Ticketing'         => 'ticketing',
            'Ramp'              => 'ramp',
            'Walk out'          => 'walk_out',
            'Load & Balance'    => 'load_and_balance',
            'Lost & Found'      => 'lost_and_found',
            'AC Search'         => 'ac_search',
            'AC Access'         => 'ac_access',
        ];
    }

    public function user(){
        return $this->belongsTo('App\Models\User', 'user_id');
    }

    public function airline(){
        return $this->belongsTo('App\Models\Airline', 'airline_id');
    }

    public function service(){
        return $this->belongsTo('App\Models\Service', 'service_id');
    }

    public static function getStaff($station_id, $airline_id, $airlineService){


        if ($airlineService->service_timings){
            // Default
            $includeAllStations = $airlineService->service->include_all_stations;
        }
        else {
            $includeAllStations = $airlineService->include_all_stations;
        }

        $result = StaffService::join("users", "staff__services.user_id", "=", "users.id");

        // Include All Stations
        if (!$includeAllStations) {
            $result->join("locations", "users.location_id", "=", "locations.id")
                   ->where("airport_id", $station_id);
        }

        return $result->where("airline_id", $airline_id)
                ->where("service_id", $airlineService->service_id)
                ->whereNull("users.vs")
                ->whereNull("users.deleted_at")
                ->where(function($sql){
                    $sql->whereNull("users.resigned_date")
                        ->orWhere("users.resigned_date", EMPTY_DATE)
                        ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                })
                ->orderBy("users.first_name")
                ->orderBy("users.last_name")
                ->get([
                    'user_id',
                    'users.first_name',
                    'users.last_name'
                ]);
    }

    public static function getStaffWithLocation($airline_id, $airlineService, $station_id = null){

        $includeAllStations = $airlineService->service_timings ?
            $airlineService->service->include_all_stations : $airlineService->include_all_stations;

        $result = StaffService::join("users", "staff__services.user_id", "=", "users.id")
                                ->join("locations", "users.location_id", "=", "locations.id")
                                ->join("airports", "airports.id", "=", "locations.airport_id");

        if ($station_id && !$includeAllStations){
            $result->where("airport_id", $station_id);
        }

        return $result->where("airline_id", $airline_id)
                ->where("service_id", $airlineService->service_id)
                ->whereNull("users.vs")
                ->whereNull("users.deleted_at")
                ->where(function($sql){
                    $sql->whereNull("users.resigned_date")
                        ->orWhere("users.resigned_date", EMPTY_DATE)
                        ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                })
                ->orderBy("airports.iata")
                ->orderBy("users.first_name")
                ->orderBy("users.last_name")
                ->get([
                    'user_id',
                    'users.first_name',
                    'users.last_name',
                    'airports.iata'
                ]);
    }

    public static function getAirlineApprovedStaff($airline_id, $airports = []){

        $userIDs = StaffService::join("users", "staff__services.user_id", "=", "users.id")
                            ->where("airline_id", $airline_id)
                            ->whereNull("users.vs")
                            ->whereNull("users.deleted_at")
                            ->where(function($sql){
                                $sql->whereNull("users.resigned_date")
                                    ->orWhere("users.resigned_date", EMPTY_DATE)
                                    ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                            })
                            ->groupBy("user_id")
                            ->pluck('user_id')
                            ->all();

        $userIDs = array_unique($userIDs);

        $withoutServiceIDs = StaffAirline::join("users", "staff__airlines.user_id", "=", "users.id")
                                        ->where("airline_id", $airline_id)
                                        ->whereNull("users.vs")
                                        ->whereNull("users.deleted_at")
                                        ->where(function($sql){
                                            $sql->whereNull("users.resigned_date")
                                                ->orWhere("users.resigned_date", EMPTY_DATE)
                                                ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                                        })
                                        ->groupBy("user_id")
                                        ->pluck('user_id')
                                        ->all();

        $withoutServiceIDs = array_unique($withoutServiceIDs);
        $withoutServiceIDs = array_diff($withoutServiceIDs, $userIDs);

        $users = User::with(['service', 'location', 'location.airport']);

        if (count($airports)){
            $users->join("locations", "users.location_id", "=", "locations.id")
                  ->whereIn("locations.airport_id", $airports);
        }

        $users = $users->whereIn('users.id', $userIDs)
                       ->get([
                           "users.*"
                       ]);

        $usersWithoutService = User::with(['service', 'location', 'location.airport']);

        if (count($airports)){
            $usersWithoutService->join("locations", "users.location_id", "=", "locations.id")
                                ->whereIn("locations.airport_id", $airports);
        }

        $usersWithoutService = $usersWithoutService->whereIn('users.id', $withoutServiceIDs)
                                                ->get([
                                                    "users.*"
                                                ]);


        foreach ($usersWithoutService as $each) {
            $users->push($each);
        }

        return $users;
    }

    public static function saveDetails($user, $services){
        if ($user && $services && count($services)) {
            $processedIds = [];
            foreach ($services as $airlineAndService) {
                list($airlineId, $serviceId) = self::getAirlineAndService($airlineAndService);
                $staff = self::where('user_id', $user->id)
                    ->where('airline_id', $airlineId)
                    ->where("service_id", $serviceId)
                    ->first();

                if (!$staff) {
                    $staff = new StaffService();
                    $staff->user_id = $user->id;
                    $staff->airline_id = $airlineId;
                    $staff->service_id = $serviceId;
                    $staff->created_by = Auth::user()->id;
                    $staff->save();
                }

                $processedIds[] = $staff->id;
            }

            $sServices = StaffService::where('user_id', $user->id)
                                        ->whereNotIn('id', $processedIds)
                                        ->get();

            // Audit Deleted
            auditDeleted($sServices);


        } else {
            $sServices = StaffService::where('user_id', $user->id)
                                        ->get();
            // Audit Deleted
            auditDeleted($sServices);
        }

    }

    public static function getAirlineAndService($airlineIdAndService){
        $result = explode("_", $airlineIdAndService);

        if (count($result) != 2 || !$airlineIdAndService){
            return [
                null,
                null
            ];
        }

        return [
            $result[0],
            $result[1],
        ];
    }
}
