<?php namespace App\Models; use Illuminate\Database\Eloquent\Model as Eloquent;
use Illuminate\Support\Facades\DB;
use stdClass;


const SHOW_ALL = 'SHOW_ALL';
const SHOW_TOTALS = 'SHOW_TOTALS';
/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 27.08.14

 */
class PassengerFlight extends Eloquent
{
    protected $table = "passengers__flights";

    protected $guarded = [
        "id",
        "created_at",
        "updated_at"
    ];

    public $timestamps = TRUE;

     public function fareCurrency()
    {
        return $this->belongsTo("App\\Models\\Currency", "fare_currency_id");
    }

    public function passenger()
    {
        return $this->belongsTo("App\\Models\\Passenger", "passenger_id");
    }

    public function flight()
    {
        return $this->belongsTo("App\\Models\\Flight", "flight_id");
    }

    public function flightRbd()
    {
        return $this->belongsTo("App\\Models\\FlightRbd", "flight_rbd_id");
    }

    public function flightCabin()
    {
        return $this->belongsTo("App\\Models\\FlightCabin", "flight_cabin_id");
    }

    public function flightOrderStatus()
    {
        return $this->belongsTo("App\\Models\\FlightOrderStatus", "flight_order_status_id");
    }

    public function flightFareBasis()
    {
        return $this->belongsTo("App\\Models\\FlightFareBasis", "flight_fare_basis_id");
    }

    public function flightNumber()
    {
        return $this->belongsTo("App\\Models\\FlightNumber", "flight_number_id");
    }

    public function agencyCode()
    {
        return $this->hasManyThrough("App\\Models\\AgencyCode", "App\\Models\\PassengerOrder", 'id', 'agency_code_id');
    }

    public function agencyCodeCustom()
    {
        return $this->belongsTo("App\\Models\\AgencyCode", 'agency_code_id');
    }

    public function passengerOrder()
    {
        return $this->belongsTo("App\\Models\\PassengerOrder", "passenger_order_id");
    }

    public function passengerFrequentFlyer()
    {
        return $this->belongsTo("App\\Models\\PassengerFrequentFlyer", "passenger_frequent_flyer_id");
    }

    public static function getFlightsPassengers($flightIDs, $firstName = null, $lastName = null){
        $pax = PassengerFlight::with([
                                    'passenger',
                                    'flightRbd',
                                    'flightCabin',
                                    'passengerOrder',
                                    'passengerFrequentFlyer',
                                    'fareCurrency',
                                    'flightFareBasis',
                                ]);

        if ($firstName || $lastName){
            $pax->join('passengers', 'passengers__flights.passenger_id', '=', 'passengers.id');

            if ($firstName){
                $pax->where("passengers.first_name", "LIKE", $firstName."%");
            }

            if ($lastName){
                $pax->where("passengers.last_name", "LIKE", $lastName."%");
            }
        }

        $pax->whereNull("passengers__flights.deleted_at");

        if (is_array($flightIDs)){
            $pax->whereIn("flight_id", $flightIDs);
        }
        else {
            $pax->where("flight_id", $flightIDs);
        }

        return $pax->get(["passengers__flights.*"]);

    }

    /**
     * Get PAX and Revenue By Class (Filter: Date Period, Period, Flight Number Ids)
     * @param $dateSearch
     * @param null $cabin
     * @param null $flightSearch
     * @param null $period
     * @param $airport
     * @return mixed
     */
    public static function getPaxAndRevenueByCabin($dateSearch, $cabin = null, $flightSearch = null, $period = null, $airport){

        $select = "DATE(departure)";

        $passengerFlights = PassengerFlight::join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
                                            ->join('passengers__orders', 'passengers__orders.id', '=', 'passengers__flights.passenger_order_id')
                                            ->leftJoin('flights', 'flights.id', '=', 'passengers__flights.flight_id')
                                            ->whereNull("flights.cancelled_at")
                                            ->whereNull("flights.deleted_at");

        if (isset($dateSearch['month'], $dateSearch['year'])) {
            $month = is_array($dateSearch['month']) ? $dateSearch['month'] : [ $dateSearch ['month'] ];
            $year = is_array($dateSearch['year']) ? $dateSearch['year'] : [ $dateSearch ['year'] ];

            $passengerFlights->whereIn(DB::raw("MONTH(departure)"), $month)
                             ->whereIn(DB::raw("YEAR(departure)"), $year);

            // If Period is Not Set, Then Make It -> Monthly
            if (!isset($period) || $period == ''){
                $period = 2;
            }
        }
        elseif (isset($dateSearch['from'], $dateSearch['to'])) {
            $passengerFlights->whereBetween(DB::raw("DATE(departure)"), [$dateSearch['from'], $dateSearch['to']]);
        }

        if ($flightSearch){
            if (isset($flightSearch['flightNumberIds'])){
                $flightNumbersId = is_array($flightSearch['flightNumberIds']) ? $flightSearch['flightNumberIds'] : [ $flightSearch['flightNumberIds'] ];
                $passengerFlights->whereIn('flights__numbers.id', $flightNumbersId);
            }

            if (isset($flightSearch['sectorIds'])){
                $sectorIds = is_array($flightSearch['sectorIds']) ? $flightSearch['sectorIds'] : [ $flightSearch['sectorIds'] ];
                $passengerFlights->whereIn('flights__numbers.flight_sector_id', $sectorIds);
            }

            // If Period is Not Set Make It By Flight Number
            if (!isset($period) || $period == ''){
                $period = 6;
            }
        }


        if($airport){
            $airport = is_array($airport) ? $airport : [$airport];
            $passengerFlights->where('arrival_airport', $airport);
        }

        // If Period IS Selected Group By Period
        if (isset($period) && $period != ''){
            $select = groupSelectionNonComparison($period, 'departure');
            $passengerFlights->groupBy('date');
        }

        $allSelects = [];
        $allSelects[] = DB::raw('COUNT(s_passengers__flights.id) AS pax');
        $allSelects[] = DB::raw($select." AS date");
        $allSelects[] = 'departure';
        $allSelects[] = DB::raw("SUM(s_passengers__flights.fare) AS revenue");
        if ($cabin){
            switch ($cabin){
                case "Business":
                    $allSelects[] = DB::raw("SUM(IF(cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_c");
                    $allSelects[] = DB::raw("SUM(IF(cabin = 'Business', 1, 0)) AS pax_c");
                    break;
                case "Economy":
                    $allSelects[] = DB::raw("SUM(IF(cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_y");
                    $allSelects[] = DB::raw("SUM(IF(cabin = 'Economy', 1, 0)) AS pax_y");
                    break;
                default:
                    $allSelects[] = DB::raw("SUM(IF(cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_c");
                    $allSelects[] = DB::raw("SUM(IF(cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_y");
                    $allSelects[] = DB::raw("SUM(IF(cabin = 'Business', 1, 0)) AS pax_c");
                    $allSelects[] = DB::raw("SUM(IF(cabin = 'Economy', 1, 0)) AS pax_y");
                    break;
            }
        }

        return $passengerFlights->get($allSelects);
    }




    public static function getPaxAndRevenueDaily($date_from = null, $date_to = null, $group_period = null){

        $select = "DATE(std)";
        $group="date";

        if (isset($group_period)){ // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily
            $select = Flight::groupSelection($group_period);
        }
        elseif (isset($groupbyflight)) {
            $group = "flight_number";
        }

        if (isset($date_from, $date_to))
            $sql = " DATE(std) BETWEEN '$date_from' AND '$date_to'";
        elseif (isset($date_from))
            $sql = " DATE(std) = '$date_from'";
        else
            $sql = "";

         $data = PassengerFlight::selectRaw("SUM(IF(bound = '0' AND cabin = 'Business',1,0)) AS pax_out_c, SUM(IF(bound = '0'  AND cabin = 'Economy',1,0)) AS pax_out_y,
                SUM(IF(bound = '1' AND cabin = 'Business',1,0)) AS pax_in_c, SUM(IF(bound = '1' AND cabin = 'Economy',1,0)) AS pax_in_y,
                SUM(IF(bound = '1' AND cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_in_c, SUM(IF(bound = '1' AND cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_in_y,
                SUM(IF(bound = '0' AND cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_out_c, SUM(IF(bound = '0' AND cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_out_y, $select AS date")
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->join('flights', 'flights.id', '=', 'passengers__flights.flight_id')
            ->join('aircraft', 'flights.aircraft_id', '=', 'aircraft.id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at")
            ->whereRaw("$sql")
            ->orderBy($group)
            ->groupBy($group)
            ->get();

/*        $data = PassengerFlight::selectRaw("SUM(IF(s_flightnumbers_aircrafts_capacity.id, capacity_c, config_c)) AS cap_c, SUM(IF(rbd IS NULL, s_passengers__flights.fare, 0)) AS revenue_inf, SUM(IF(cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_c, SUM(IF(cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_y, COUNT(DISTINCT flight_number) AS flight_count, SUM(IF(rbd IS NULL, 1, 0)) AS pax_inf, SUM(IF(cabin = 'Business', 1, 0)) AS pax_c, SUM(IF(cabin = 'Economy', 1, 0)) AS pax_y ,sum(s_passengers__flights.fare) AS revenue_total, SUM(IF(rbd IS NOT NULL, 1, 0)) AS pax_total, DATE(std) AS date")
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('flights', 'passengers__flights.flight_id', '=', 'flights.id')
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->join('aircrafts', 'flights.aircraft_id', '=', 'aircrafts.id')
            ->join('flightnumbers_aircrafts_capacity', function($join)
            {
                $join->on('passengers__flights.flight_number_id', '=', 'flightnumbers_aircrafts_capacity.flight_number_id');
                $join->on('flights.aircraft_id', '=', 'flightnumbers_aircrafts_capacity.aircraft_id');
            })
            ->groupBy(DB::raw('DATE(std)'))
            ->get();*/

        return $data;
    }



    public static function getPaxAndRevenue($flight_id, $group_period = null, $economy = null){
        $select = "DATE(std)";
        $economy = (isset($economy)) ? "NOT" : "";

        $flight_id = implode(',',$flight_id);

        if (isset($group_period)){ // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily
            $select = Flight::groupSelection($group_period);
        }

        $data = PassengerFlight::selectRaw("COUNT(s_passengers__flights.id) AS pax, SUM(s_passengers__flights.fare) AS revenue, $select AS date")
            ->whereRaw("rbd $economy IN ('C','J','I','D') AND s_flights.id IN ($flight_id)")
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->join('flights', 'flights.id', '=', 'passengers__flights.flight_id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at")
            ->orderBy('date')
            ->groupBy('date')
            ->get();
        return $data;
    }


    /**
     * Get Top Routes By Revenue
     * @param $period
     * @return mixed
     */
    public static function getTopSectors($period){
        $data = PassengerFlight::select([
                "departure_date",
                "sales_date",
                "flight_number",
                "departure_airport",
                "arrival_airport",
                DB::raw("SUM(s_passengers__flights.fare) AS revenue")
            ])
            ->join('passengers__orders', 'passengers__orders.id', '=', 'passengers__flights.passenger_order_id')
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('flights', 'flights.id', '=', 'passengers__flights.flight_id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at");

        if (array_key_exists('departure_from', $period)){
            $data->whereBetween('departure_date', $period);
        }
        else {
            $data->whereBetween('sales_date', $period);
        }

        $data->orderBy('revenue', 'DESC')
             ->groupBy('flight_number')
             ->limit(5);

        return $data->get();
    }


    public static function getFlightPaxRevenue($date_from = null, $date_to = null, $group_period = null, $inbound = null, $economy = null){

        $inbound = (isset($inbound)) ? "bound = '1'" : "bound = '0'";
        $economy = (isset($economy)) ? "cabin = 'Economy'" : "cabin = 'Business'";

        $select = "DATE(std)";
        $group="departure_date";

        if (isset($group_period)){ // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily
            $select = Flight::groupSelection($group_period);
        }
        elseif (isset($groupbyflight)) {
            $group = "flight_number";
        }

        if (isset($date_from, $date_to))
            $sql = " AND (DATE(std) BETWEEN '$date_from' AND '$date_to')";
        elseif (isset($date_from))
            $sql = " AND DATE(std) = '$date_from'";
        else
            $sql = "";

        $data = PassengerFlight::selectRaw("COUNT(s_passengers__flights.id) AS pax, SUM(s_passengers__flights.fare) AS revenue, $select AS date")
            ->whereRaw("$economy AND $inbound $sql")
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('flights', 'flights.id', '=', 'passengers__flights.flight_id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at")
            ->orderBy('date')
            ->groupBy('date')
            ->get();
        return $data;
    }

    /*public static function getFlightPaxRevenueTotal($date_from = null, $date_to = null, $group_period = null, $inbound = null,$groupbyflight = null){

        $inbound = (isset($inbound)) ? "NOT" : "";

        $select = "DATE(std)";

        $group="departure_date";

        if (isset($group_period)){ // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily
            $select = Flight::groupSelection($group_period);
        }
        elseif (isset($groupbyflight)) {
            $group = "flight_number";
        }

        if (isset($date_from, $date_to))
            $sql = " AND (DATE(std) BETWEEN '$date_from' AND '$date_to')";
        elseif (isset($date_from))
            $sql = " AND DATE(std) = '$date_from'";
        else
            $sql = "";

        $data = PassengerFlight::selectRaw("SUM(IF(cabin = 'Economy',1,0)) AS pax_y, SUM(IF(cabin = 'Economy',s_passengers__flights.fare,0)) AS revenue_y, $select AS date,
            SUM(IF(cabin = 'Business',1,0)) AS pax_c, SUM(IF(cabin = 'Business',s_passengers__flights.fare,0)) AS revenue_c")
            ->whereRaw("departure_airport $inbound IN ('DYU', 'LBD') $sql")
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->join('flights', 'passengers__flights.flight_id', '=', 'flights.id')
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->orderBy('date')
            ->groupBy('date')
            ->get();
        return $data;
    }*/


    public static function getFlightsCapacity($date_from=null, $date_to = null){
        $sql = "";
        if (isset($date_from, $date_to))
            $sql = "(DATE(std) BETWEEN '$date_from' AND '$date_to') AND ";
        elseif (isset($date_from))
            $sql = "DATE(std) = '$date_from' AND ";

        $flights = PassengerFlight::selectRaw("COUNT(DISTINCT CASE WHEN s_aircraft.id = 1 THEN 1 END) AS con,
        SUM(IF(bound = '0',config_c + config_y,0)) AS config_out, SUM(IF(bound = '1',config_c + config_y,0)) AS config_in,
                DATE(std) AS departure_date")
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('flights', 'flights.id', '=', 'passengers__flights.flight_id')
            ->join('aircraft', 'flights.aircraft_id', '=', 'aircraft.id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at")
            ->whereRaw("$sql bound = '0'")
            ->orderByRaw('DATE(std)')
            ->groupBy(DB::raw('DATE(std)'))
            ->get();

        return $flights;
    }

    public static function getPaxAndRevenueDateFlights($date, $flight_number = null){
        $where = "";

        if (isset($flight_number))
            $where = " AND flight_number = '$flight_number'";

        $data = PassengerFlight::selectRaw("departure_airport, arrival_airport, SUM(IF(rbd IS NULL, 1, 0)) AS pax_inf, SUM(IF(cabin = 'Business', 1, 0)) AS pax_c, SUM(IF(cabin = 'Economy', 1, 0)) AS pax_y ,
            SUM(IF(cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_c,
            SUM(IF(cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_y,
            s_flights__numbers.flight_number, DATE(std) AS date")
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('flights', 'flights.id', '=', 'passengers__flights.flight_id')
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at")
            ->whereRaw("DATE(std) = '$date' $where")
            ->orderBy('flight_number')
            ->groupBy('flight_number')
            ->get();

        return $data;
    }


    public static function getFlightPreviousMonthPaxRevenue($date_from, $date_to=null, $flight_number=null){
        $where = (is_numeric($flight_number)) ? " AND flight_number = '$flight_number'" : "";

        $min_date = date('Y-m-d', strtotime("$date_from - 1 months"));

        $data = PassengerFlight::selectRaw("CONCAT(s_depAirport.iata,'-',s_arrAirport.iata) AS date,
                    SUM(IF(s_flights__cabin.cabin = 'Business', 1, 0)) AS pax_c,
                    SUM(IF(s_flights__cabin.cabin = 'Economy', 1, 0)) AS pax_y ,
                    SUM(IF(s_flights__cabin.cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_c,
                    SUM(IF(s_flights__cabin.cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_y,
                    flight_number, sales_date")
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->leftjoin('flights__cabin', 'passengers__flights.flight_cabin_id', '=', 'flights__cabin.id')
            ->leftjoin('flights__order_status', 'passengers__flights.flight_order_status_id', '=', 'flights__order_status.id')
            ->leftjoin('airports AS depAirport', 'flights__numbers.departure_airport_id', '=', 'depAirport.id')
            ->leftjoin('airports AS arrAirport', 'flights__numbers.arrival_airport_id', '=', 'arrAirport.id')
            ->leftJoin('flights', 'passengers__flights.flight_id','=','flights.id')
            ->whereRaw("DATE(departure) BETWEEN '$date_from' AND '$date_to' $where") // AND DATE(sales_date) > '$min_date'
            ->whereNotIn('order_status',['R', 'V', 'E'])
            ->orderBy('sales_date','desc')
            ->groupBy('sales_date')
            ->get();

        return $data;
    }



    public static function getPassengerFlightsByTicketNumber($ticket_number){

        $data = PassengerFlight::select([
                'passengers__flights.id AS passenger_id',
                'sales_date',
                'flight_number'.
                'departure_airport',
                DB::raw('DATE(std) AS departure_date'),
                'arrival_airport',
                'code',
                'agency'.
                'passengers.id AS passenger_id',
                'first_name',
                'last_name',
                'passengers__flights.fare',
                'coupon',
                'order_status',
                'ticket_number',
                'passengers__orders.fare_basis'
        ])
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('flights', 'passengers__flights.flight_id', '=', 'flights.id')
            ->join('passengers', 'passengers__flights.passenger_id', '=', 'passengers.id')
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->join('agency__codes', 'agency__codes.id', '=', 'passengers__orders.agency_code_id')
            ->join('agency', 'agency__codes.agency_id', '=', 'agency.id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at")
            ->where("ticket_number",$ticket_number)
            ->orderBy("ticket_number")
            ->get();

        return $data;
    }



    public static function getFlightPaxRevenueComparison($date_from = null, $date_to = null, $group_period = null, $comparing_years, $country=null, $city=null, $agency = null, $report = null){

        $select = "DATE(departure)";
        $group="date";

        if (isset($date_from, $date_to))
            $sql = " DATE(departure) BETWEEN '$date_from' AND '$date_to'";
        elseif (isset($date_from))
            $sql = " DATE(departure) = '$date_from'";
        else
            $sql = " DATE(departure) IS NOT NULL";


        if (isset($group_period)){ // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily, 6-ByFlights
            $select = Flight::groupSelection($group_period,"departure");
            if ($group_period == 0)
                $select = "'Year'";
            elseif ($group_period == 5){
                $select = "DATE_FORMAT(departure,'%b %e')";
            }
            elseif($group_period == 6){
                $group = "flight_number";
            }
            elseif($group_period == 7){
            }

        }


        if (isset($comparing_years) && is_array($comparing_years)){
            $comparing_query = "";
            foreach ($comparing_years AS $each){
                $comparing_query .= "SUM(IF( YEAR(departure) = '$each' AND bound = '0' AND cabin = 'Business',1,0)) AS pax_out_c_{$each},
                                     SUM(IF( YEAR(departure) = '$each' AND bound = '0' AND cabin = 'Economy',1,0)) AS pax_out_y_{$each},
                                     SUM(IF( YEAR(departure) = '$each' AND bound = '1' AND cabin = 'Business',1,0)) AS pax_in_c_{$each},
                                     SUM(IF( YEAR(departure) = '$each' AND bound = '1' AND cabin = 'Economy',1,0)) AS pax_in_y_{$each},
                                     SUM(IF( YEAR(departure) = '$each' AND bound = '0' AND cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_out_c_{$each},
                                     SUM(IF( YEAR(departure) = '$each' AND bound = '0' AND cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_out_y_{$each},
                                     SUM(IF( YEAR(departure) = '$each' AND bound = '1' AND cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_in_c_{$each},
                                     SUM(IF( YEAR(departure) = '$each' AND bound = '1' AND cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_in_y_{$each},";
            }
            $comparing_query = rtrim($comparing_query, ',');
        }
        else{
            $comparing_query = "SUM(IF(bound = '0' AND cabin = 'Business',1,0)) AS pax_out_c,
                                SUM(IF(bound = '0' AND cabin = 'Economy',1,0)) AS pax_out_y,
                                SUM(IF(bound = '1' AND cabin = 'Business',1,0)) AS pax_in_c,
                                SUM(IF(bound = '1' AND cabin = 'Economy',1,0)) AS pax_in_y,
                                SUM(IF(bound = '1' AND cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_in_c,
                                SUM(IF(bound = '1' AND cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_in_y,
                                SUM(IF(bound = '0' AND cabin = 'Business', s_passengers__flights.fare, 0)) AS revenue_out_c,
                                SUM(IF(bound = '0' AND cabin = 'Economy', s_passengers__flights.fare, 0)) AS revenue_out_y";
        }

        $selects = "$select AS date, $comparing_query ";

        if (isset($report)){
            switch ($report){
                case "Route":
                    $selects .=', CONCAT(departure_airport,"-",arrival_airport,"-",departure_airport) AS report_category';
                    $sql .= " AND departure_airport IN ('DYU', 'LBD') AND NOT (departure_airport = 'LBD' AND arrival_airport = 'DYU')";
                    break;
                case "Country":
                    break;
                case "POS":
                    $selects .=", CONCAT(s_countries.abbr,'-',city,'-',agency) AS report_category";
                    break;
            }
        }


        $data = PassengerFlight::selectRaw($selects);

        if (isset($country) || (isset($report) && $report == 'Country')){
            $data->addSelect('country');
        }
        if (isset($city)){
            $data->addSelect('city');
        }
        if (isset($agency)){
            $data->addSelect('agency');
        }

        $data->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
             ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id');

        if (isset($country) || isset($city) || isset($agency) || (isset($report) && ($report == 'Country' or $report == "POS"))) {
             $data->join('agency__codes', 'agency__codes.id', '=', 'passengers__orders.agency_code_id')
                ->join('agency', 'agency__codes.agency_id', '=', 'agency.id')
                ->join('cities', 'agency__codes.city_id', '=', 'cities.id')
                ->join('countries', 'agency__codes.country_id', '=', 'countries.id');

            if (isset($country)){
                    if (is_array($country))
                        $sql .= " AND s_countries.id IN(" . implode(",", $country) . ")";
                    else
                        $sql .= " AND s_countries.id = '$country''";
            }
            if (isset($city)){
                if (is_array($city))
                    $sql .= " AND s_cities.id IN(".implode(",",$city).")";
                else
                    $sql .= " AND s_cities.id = '$city''";
            }
            if (isset($agency)){
                if (is_array($agency))
                    $sql .= " AND s_agency.id IN(".implode(",",$agency).")";
                else
                    $sql .= " AND s_agency.id = '$agency''";
            }
        }

        $data->join('flights', 'flights.id', '=', 'passengers__flights.flight_id')
             ->whereNull("flights.cancelled_at")
             ->whereNull("flights.deleted_at")
             ->whereRaw($sql)
             ->whereNotIn('order_status',['R', 'V', 'E'])
             ->orderBy($group)
             ->groupBy($group);


        if (isset($country)){
            $data->groupBy('country');
        }
        if (isset($city)){
            $data->groupBy('city');
        }
        if (isset($agency)){
            $data->groupBy('agency');
        }

        if (isset($report)) {
            switch ($report){
                case "Country":
                    $data->groupBy('country');
                    break;
                case "Route":
                    $data->groupBy('report_category');
                    break;
                case "POS":
                    $data->groupBy('report_category');
            }
            return $data->paginate(35);

        }
        else
            return $data->get();
    }

    public static function getFlightPaxRevenueWidget($date_from = null, $date_to = null, $group_period = null, $flight_number = null, $compare_dates = false, $revenue_type = null)
    {
        if (!isset($revenue_type))
            $revenue_type = 'departure_date';

        $select = "$revenue_type";
        $group  = "date";

        if ($group_period)  { // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily, 6-ByFlights
            $select = Flight::groupSelection($group_period, $revenue_type);
        }
        if ($compare_dates) {
            $sql = " $revenue_type IN ('$date_from', '$date_to')";
        }
        else {
            if (isset($date_from, $date_to))
                $sql = " $revenue_type BETWEEN '$date_from' AND '$date_to'";
            elseif (isset($date_from))
                $sql = " $revenue_type = '$date_from'";
            else
                $sql = " $revenue_type IS NOT NULL";
        }


        $data = PassengerFlight::selectRaw("$select AS date, COUNT(*) AS pax, SUM(s_passengers__flights.fare) AS revenue")
            /*SUM(IF(bound IN (0,1) AND cabin IN (0,1),1,0)) AS pax,
            SUM(IF(bound IN (0,1) AND cabin IN (0,1), s_passengers__flights.fare, 0)) AS revenue")*/
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->join('flights__cabin', 'passengers__flights.flight_cabin_id', '=', 'flights__cabin.id')
            ->join('flights__order_status', 'passengers__flights.flight_order_status_id', '=', 'flights__order_status.id')
            ->join('flights', 'flights.id', '=', 'passengers__flights.flight_id')
            ->leftJoin('flights', 'passengers__flights.flight_id', '=', 'flights.id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at")
            ->whereRaw($sql)
            ->whereNotNull('bound')
            ->whereNotNull('cabin')
            ->whereNotIn('order_status',['R', 'V', 'E']);

        if (isset($flight_number))
            $data->whereIn('flight_number', $flight_number);

        $data->orderBy($group)
            ->groupBy($group);

        $data = $data->get();
        return $data;
    }


    /**
     * @param $inputs
     * @param array $groupBy
     * @return array
     */
    public static function getRevenue($inputs, $groupBy = [
        'field'         => 'sales_date',
        'period'        => 6,  // Daily,
        'date_format'   => TRUE
    ], $orderBy = false)
    {
        $data = PassengerFlight::join('passengers__orders', 'passengers__orders.id', '=', 'passengers__flights.passenger_order_id')
            ->join('passengers',         'passengers.id',         '=', 'passengers__flights.passenger_id'      )
            ->join('flights__numbers',   'flights__numbers.id',   '=', 'passengers__flights.flight_number_id'  )
            ->join('flights', 'flights.id', '=', 'passengers__flights.flight_id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at");

        // Flight Type
        if (isset($inputs['flight_type']) && $inputs['flight_type']) {
            $data->where('flights__numbers.flight_type_id', $inputs['flight_type']);
        }
        // Flight Number
        if (isset($inputs['flight_number'])){
            if (is_array($inputs['flight_number'])) {
                $data->whereIn('passengers__flights.flight_number_id', $inputs['flight_number']);
            }
            else {
                $data->where('passengers__flights.flight_number_id', $inputs['flight_number']);
            }
        }
        // Flight Id
        if (isset($inputs['flight_id']) && $inputs['flight_id']) {
            $data->whereIn('passengers__flights.flight_id', $inputs['flight_id']);
        }
        // Sector
        if (isset($inputs['sector'])){
            $data->join('airports AS depAirport', 'depAirport.id',            '=',   'flights__numbers.departure_airport_id'     )
                ->join('airports AS arrAirport', 'arrAirport.id',            '=',   'flights__numbers.arrival_airport_id'       );

            $data->where(function($sql) use ($inputs){
                foreach ($inputs['sector'] AS $i => $sector) {
                    preg_match('/(\w{3})-(\w{3})/', $sector, $matches);
                    if (isset($matches[1]) && isset($matches[2])){
                        list($str, $departureAirport, $arrivalAirport) = $matches;
                        if ($i == 0){
                            $sql->where(function($query) use ($departureAirport, $arrivalAirport){
                                $query->where('depAirport.iata', $departureAirport)
                                    ->where('arrAirport.iata', $arrivalAirport);
                            });
                        }
                        else {
                            $sql->orWhere(function($query) use ($departureAirport, $arrivalAirport){
                                $query->where('depAirport.iata', $departureAirport)
                                    ->where('arrAirport.iata', $arrivalAirport);
                            });
                        }
                    }
                }
            });
        }
        // JOIN Agency Country
        if (contains($groupBy['field'], ['channel', 'country', 'city', 'agency']) || isset($inputs['country']) || isset($inputs['city']) || isset($inputs['agency']) || isset($inputs['sales_channel']) || isset($inputs['sales_channel_type'])){
            $data->join('agency__codes', 'agency__codes.id', '=', 'passengers__orders.agency_code_id');

            // Added on DEC13
            $data->leftJoin('agency', 'agency.id', '=', 'agency__codes.agency_id');
        }
        if (contains($groupBy['field'], 'channel') || isset($inputs['sales_channel']) || isset($inputs['sales_channel_type'])) {
            $data->join('agency', 'agency.id', '=', 'agency__codes.agency_id')
                 ->join('sales__channels', 'sales__channels.id', '=', 'agency.sales_channel_id');
        }
        if (contains($groupBy['field'], 'country')) {
            $data->join('countries', 'countries.id', '=', 'agency__codes.country_id');
        }
        if (contains($groupBy['field'], 'cabin')) {
            $data->join('flights__cabin', 'flights__cabin.id', '=', 'passengers__flights.flight_cabin_id');
        }
        if (contains($groupBy['field'], 'rbd')) {
            $data->join('flights__rbd', 'flights__rbd.id', '=', 'passengers__flights.flight_rbd_id');
        }
        // Sales Channels
        if (isset($inputs['sales_channel']) && $inputs['sales_channel']) {
            $data->whereIn('agency.sales_channel_id', $inputs['sales_channel']);
        }
        // Sales Channels Type
        if (isset($inputs['sales_channel_type']) && $inputs['sales_channel_type']) {
            $data->where('sales__channels.sales_channel_type_id', $inputs['sales_channel_type']);
        }
        // Country (Agency)
        if (isset($inputs['country'])){
            $data->where(function($sql) use ($inputs){
                $sql->whereIn('agency__codes.country_id', $inputs['country'])
                    ->orWhereIn('agency.country_id', $inputs['country']);
            });
        }
        // City (Agency)
        if (isset($inputs['city'])){
            $data->whereIn('agency__codes.city_id', $inputs['city']);
        }
        // Name (Agency)
        if (isset($inputs['agency'])){
            $data->whereIn('agency__codes.agency_id', $inputs['agency']);
        }
        // Ticket Number
        if (isset($inputs['ticket_number']) && $inputs['ticket_number']) {
            self::prepareQueryByOperator($data, 'ticket_number', $inputs['ticket_number']);
            //$data->where('ticket_number', 'LIKE', "{$inputs['ticket_number']}%");
        }
        // IATA Code
        if (isset($inputs['iata_code']) && $inputs['iata_code']) {
            $data->where('agency__codes.code', 'LIKE', "{$inputs['iata_code']}%");
        }
        // TCH Code(Agency)
        if (isset($inputs['tch_code'])){
            $data->whereIn('passengers__orders.agency_codes_id', $inputs['tch_code']);
        }
        // Coupon (Ticket)
        if (isset($inputs['coupon'])){
            $data->whereIn('coupon', $inputs['coupon']);
        }
        // Fare Basis (Ticket)
        if (isset($inputs['fare_basis'])){
            $data->whereIn('passengers__flights.flight_fare_basis_id', $inputs['fare_basis']);
        }
        // Payment Form
        if (isset($inputs['payment_form'])){
            $data->where('payment_form', 'REGEXP', implode('|', $inputs['payment_form']));
        }
        // Name (Passenger)
        if (isset($inputs['first_name']) || isset($inputs['last_name'])) {
            if ($inputs['first_name'] && $inputs['last_name']) {
                $data->where('first_name', 'LIKE', "{$inputs['first_name']}%");
                $data->where('last_name', 'LIKE', "{$inputs['last_name']}%");
            } else {
                if ($inputs['first_name'] || $inputs['last_name']) {
                    $searchValue = $inputs['first_name'] ? $inputs['first_name'] : $inputs['last_name'];
                    $data->where(function ($sql) use ($searchValue) {
                        $sql->where('first_name', 'LIKE', "{$searchValue}%");
                        $sql->orWhere('last_name', 'LIKE', "{$searchValue}%");
                    });
                }
            }
        }
        // Middle Name
        if (isset($inputs['middle_name']) && $inputs['middle_name']) {
            $data->where('middle_name', 'LIKE', "{$inputs['middle_name']}%");
        }
        // Cabin
        if (isset($inputs['cabin']) && $inputs['cabin']){
            $data->where('passengers__flights.flight_cabin_id', $inputs['cabin']);
        }
        // Order Status
        if (isset($inputs['status'])){
            $data->whereIn('passengers__flights.flight_order_status_id', $inputs['status']);
        }
        // RBD
        if (isset($inputs['rbd'])){
            $data->whereIn('passengers__flights.flight_rbd_id', $inputs['rbd']);
        }
        // Sales Date
        if (isset($inputs['sales_from']) || isset($inputs['sales_to'])) {
            if (isset($inputs['sales_from']) && isset($inputs['sales_to'])) {
                $data->whereBetween('sales_date', [$inputs['sales_from'], $inputs['sales_to']]);
            } elseif (isset($inputs['sales_from'])) {
                $data->where('sales_date', $inputs['sales_from']);
            }
        }
        // Departure Date
        if (isset($inputs['departure_from']) || isset($inputs['departure_to'])) {
            if ($inputs['departure_from'] && $inputs['departure_to']) {
                $data->whereBetween('departure_date', [$inputs['departure_from'], $inputs['departure_to']]);
            } elseif ($inputs['departure_from']) {
                $data->where('departure_date', $inputs['departure_from']);
            }
        }

        if (isset($groupBy['date_format']) && $groupBy['date_format']) {
            $groupBy['field'] = groupByReportType($groupBy['period'], $groupBy['field']);
        }

        // Exclude Statuses That Won't Generate Revenue
        $data->join('flights__order_status', 'flights__order_status.id', '=', 'passengers__flights.flight_order_status_id');
        $data->join('flights__fare_basis', 'flights__fare_basis.id', '=', 'passengers__flights.flight_fare_basis_id');
        $data->whereNotIn('order_status', FlightOrderStatus::getExcludedFlightStatuses());
        $data->where('fare_basis', 'NOT LIKE', '%IN%');

        // Group
        $data->groupBy('grouped_field');

        if ($orderBy){
            $data->orderBy($orderBy);
        }
        else {
            $data->orderBy('revenue', 'DESC');
        }
        $data = $data->get([
            DB::raw($groupBy['field']." AS grouped_field"),
            DB::raw('SUM(s_passengers__flights.fare) AS revenue'),
            DB::raw('COUNT(*) AS pax')
        ]);

        return $data;
    }




    /**
     * Search Passenger Flights By Many Given Input Fields
     * @param $inputs
     * @param bool $cacheQuery
     * @param bool $totals
     * @param bool $pagination
     * @return array
     */
    public static function searchPassengerFlights($inputs, $cacheQuery = FALSE, $totals = FALSE, $pagination = FALSE){
        // Show only TOTALS(WHEN TYPE -> Yearly, ... IS SELECTED)
        $totalMode = FALSE;

        $data = PassengerFlight::with([
            "flight",
            "flightNumber",
            "flightNumber.departureAirport",
            "flightNumber.arrivalAirport",
            "flightRbd",
            "flightFareBasis",
            "flightOrderStatus",
            "passengerOrder.agencyCode",
            "passengerOrder.agencyCode.agency",
            "passengerOrder.agencyCode.city",
            "passengerOrder.agencyCode.country",
        ])
                ->select([
                    'passengers__flights.flight_id',
                    'passengers.id AS passenger_id',
                    'passenger_order_id',
                    'sales_date',
                    'last_name',
                    'first_name',
                    'dob',
                    'ticket_number',
                    'departure_date',
                    'passengers__flights.flight_number_id',
    //                'passengers__orders.agency_code_id',
                    'passengers__orders.agency_code_id',
                    'passengers__flights.flight_order_status_id',
                    'passengers__flights.flight_rbd_id',
                    'passengers__flights.flight_fare_basis_id',
                    'currency.code as currency',
                    'passengers__flights.fare',
//                    'original_fare',
                    'passengers__orders.agency_code_id',
                    'coupon'
            ])
            ->join('flights__order_status',     'flights__order_status.id', '=', 'passengers__flights.flight_order_status_id' )
            ->join('currency',                  'currency.id',              '=', 'passengers__flights.fare_currency_id')
            ->join('passengers__orders',        'passengers__orders.id',    '=', 'passengers__flights.passenger_order_id')
            ->join('passengers',                'passengers.id',            '=', 'passengers__flights.passenger_id' )
            ->join('flights__numbers',          'flights__numbers.id',      '=', 'passengers__flights.flight_number_id' )
            ->leftJoin('flights',               'flights.id',               '=', 'passengers__flights.flight_id' )
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at");

        // Flight Type
        if (isset($inputs['flight_type']) && $inputs['flight_type'])
        {
            $data->where('flights__numbers.flight_type_id', $inputs['flight_type']);
        }
        // Flight Number
        if (isset($inputs['flight_number']))
        {
            $data->whereIn('passengers__flights.flight_number_id', is_array($inputs['flight_number']) ? $inputs['flight_number'] : [$inputs['flight_number']]);
        }
        // Flight Number
        if (isset($inputs['flight_id']))
        {
            $data->whereIn('passengers__flights.flight_id', is_array($inputs['flight_id']) ? $inputs['flight_id'] : [$inputs['flight_id']]);
        }

        if (isset($inputs['flight_id_is_null']))
        {
            $data->whereNull('passengers__flights.flight_id');
        }

        if ((isset($inputs['report_type']) && in_array($inputs['report_type'], [7, 9]))
            || isset($inputs['sector_group']) || isset($inputs['route_group']))
        {
            $data->join('airports AS depAirport', 'depAirport.id', '=', 'flights__numbers.departure_airport_id')
                 ->join('airports AS arrAirport', 'arrAirport.id', '=', 'flights__numbers.arrival_airport_id');
        }

        // Sector
        if (isset($inputs['sector']))
        {
            $data->whereIn('flights__numbers.flight_sector_id', $inputs['sector']);
        }

        /*
        if (isset($inputs['sector'])) {
            // Commented Query slower , i expected at least similar performance
            //$data->whereIn(DB::raw('CONCAT(s_depAirport.iata, "-", s_arrAirport.iata)'), $inputs['sector']);
            $data->where(function($sql) use ($inputs){
                foreach ($inputs['sector'] AS $i => $sector) {
                    preg_match('/(\w{3})-(\w{3})/', $sector, $matches);
                    if (isset($matches[1]) && isset($matches[2])){
                        list($str, $departureAirport, $arrivalAirport) = $matches;
                        if ($i == 0){
                             $sql->where(function($query) use ($departureAirport, $arrivalAirport){
                                  $query->where('depAirport.iata', $departureAirport)
                                         ->where('arrAirport.iata', $arrivalAirport);
                             });
                        }
                        else {
                            $sql->orWhere(function($query) use ($departureAirport, $arrivalAirport){
                                $query->where('depAirport.iata', $departureAirport)
                                      ->where('arrAirport.iata', $arrivalAirport);
                            });
                        }
                    }
                }
            });
        }
        */
        // JOIN Agency Country
        if (isset($inputs['region']) || isset($inputs['country']) || isset($inputs['city']) || isset($inputs['agency']) || isset($inputs['sales_channel']) || isset($inputs['sales_channel_group'])
            || isset($inputs['region_group']) || isset($inputs['country_group']) || isset($inputs['city_group']) || isset($inputs['agency_group']))
        {
            $data->join('agency__codes', 'agency__codes.id', '=', 'passengers__orders.agency_code_id');

            // Added on DEC13
            $data->leftJoin('agency', 'agency.id', '=', 'agency__codes.agency_id');
        }

        if (isset($inputs['country']) || isset($inputs['city']) || isset($inputs['agency'])
            || isset($inputs['country_group']) || isset($inputs['city_group']) || isset($inputs['agency_group']))
        {
            $data->leftJoin('countries', 'countries.id', '=', 'agency__codes.country_id');
//            $data->join('countries', 'countries.id', '=', 'agency__codes.country_id');
        }

        // Region (Agency)
        if (isset($inputs['region']))
        {
            $data->whereIn('agency__codes.region_id', $inputs['region']);
        }
        // Country (Agency)
        if (isset($inputs['country']))
        {
            $data->where(function($sql) use ($inputs){
                $sql->whereIn('agency__codes.country_id', $inputs['country'])
                    ->orWhereIn('agency.country_id', $inputs['country']);
            });
//            $data->whereIn('agency__codes.country_id', $inputs['country']);
        }
        // City (Agency)
        if (isset($inputs['city']))
        {
            $data->where(function($sql) use ($inputs){
                $sql->whereIn('agency__codes.city_id', $inputs['city'])
                    ->orWhereIn('agency.city_id', $inputs['city']);
            });
//            $data->whereIn('agency__codes.city_id', $inputs['city']);
        }
        // Name (Agency)
        if (isset($inputs['agency']))
        {
            $data->whereIn('agency__codes.agency_id', $inputs['agency']);
        }
        // Sales Channel (Agency)
        if (isset($inputs['sales_channel']))
        {
            $data->whereIn('agency.sales_channel_id', $inputs['sales_channel']);
        }

        // Join due to grouping by these columns
        if (isset($inputs['region_group']))
        {
            $data->leftJoin('regions', 'regions.id', '=', 'agency__codes.region_id');
//            $data->join('regions', 'regions.id', '=', 'agency__codes.region_id');
        }
        if (isset($inputs['city_group']) || isset($inputs['agency_group']))
        {
            $data->leftJoin('cities', 'cities.id', '=', 'agency__codes.city_id');
//            $data->join('cities', 'cities.id', '=', 'agency__codes.city_id');
        }
        if (isset($inputs['agency_group']))
        {
            $data->join('agency', 'agency.id', '=', 'agency__codes.agency_id');
        }
        if (isset($inputs['sales_channel']) || isset($inputs['sales_channel_group'])){
            $data->join('sales__channels', 'sales__channels.id', '=', 'agency.sales_channel_id');
        }
        if (isset($inputs['rbd_group']) || isset($inputs['rbd_group']))
        {
            $data->join('flights__rbd', 'flights__rbd.id', '=', 'passengers__flights.flight_rbd_id');
        }
        if (isset($inputs['cabin_group']) || isset($inputs['cabin_group']))
        {
            $data->join('flights__cabin', 'flights__cabin.id', '=', 'passengers__flights.flight_cabin_id');
        }
        if (isset($inputs['gds_group']) || isset($inputs['gds_group']))
        {
            $data->join('reservations__systems', 'reservations__systems.id', '=', 'passengers__orders.reservations_system_id');
        }

        // Ticket Number
        if (isset($inputs['ticket_number']) && $inputs['ticket_number'])
        {
            self::prepareQueryByOperator($data, 'ticket_number', $inputs['ticket_number']);
            //$data->where('ticket_number', 'LIKE', "{$inputs['ticket_number']}%");
        }
        // IATA Code
        if (isset($inputs['iata_code']) && $inputs['iata_code'])
        {
            $data->where('agency__codes.code', 'LIKE', "{$inputs['iata_code']}%");
        }
        // TCH Code(Agency)
        if (isset($inputs['tch_code']))
        {
            $data->whereIn('passengers__orders.agency_codes_id', $inputs['tch_code']);
        }
        // Coupon (Ticket)
        if (isset($inputs['coupon']))
        {
            $data->whereIn('coupon', $inputs['coupon']);
        }
        // Fare Basis (Ticket)
        if (isset($inputs['fare_basis']))
        {
            $data->whereIn('passengers__flights.flight_fare_basis_id', $inputs['fare_basis']);
        }
        // Payment Form
        if (isset($inputs['payment_form']))
        {
            $data->where('payment_form', 'REGEXP', implode('|', $inputs['payment_form']));
        }
        // Name (Passenger)
        if (isset($inputs['first_name']) || isset($inputs['last_name']))
        {
            if ($inputs['first_name'] && $inputs['last_name'])
            {
                $data->where('first_name', 'LIKE', "{$inputs['first_name']}%");
                $data->where('last_name', 'LIKE', "{$inputs['last_name']}%");
            }
            else
            {
                if ($inputs['first_name'] || $inputs['last_name'])
                {
                    $searchValue = $inputs['first_name'] ? $inputs['first_name'] : $inputs['last_name'];
                    $data->where(function ($sql) use ($searchValue) {
                        $sql->where('first_name', 'LIKE', "{$searchValue}%");
                        $sql->orWhere('last_name', 'LIKE', "{$searchValue}%");
                    });
                }
            }
        }
        // Middle Name
        if (isset($inputs['middle_name']) && $inputs['middle_name'])
        {
            $data->where('middle_name', 'LIKE', "{$inputs['middle_name']}%");
        }

        // Cabin
        if (isset($inputs['cabin']) && $inputs['cabin'])
        {
            $data->whereIn('passengers__flights.flight_cabin_id', $inputs['cabin']);
        }

        // Order Status
        if (isset($inputs['status']))
        {
            $data->whereIn('passengers__flights.flight_order_status_id', $inputs['status']);
        }

        // Order Status
        if (isset($inputs['status_not_in']))
        {
            $data->whereNotIn('order_status', $inputs['status_not_in']);
        }

        // RBD
        if (isset($inputs['rbd']))
        {
            $data->whereIn('passengers__flights.flight_rbd_id', $inputs['rbd']);
        }

        $dateType = "sales_date";

        // Departure Date
        if (isset($inputs['departure_from']) || isset($inputs['departure_to']))
        {
            if (isset($inputs['departure_from']) && isset($inputs['departure_to']))
            {
                $dateType = "departure_date";
                $data->whereBetween('departure_date', [$inputs['departure_from'], $inputs['departure_to']]);
            }
            elseif (isset($inputs['departure_from']))
            {
                $dateType = "departure_date";
                $data->where('departure_date', $inputs['departure_from']);
            }
        }

        // Sales Date
        if (isset($inputs['sales_from']) || isset($inputs['sales_to']))
        {
            if ($inputs['sales_from'] && $inputs['sales_to'])
            {
                $dateType = "sales_date";
                $data->whereBetween('sales_date', [$inputs['sales_from'], $inputs['sales_to']]);
            }
            elseif ($inputs['sales_from'])
            {
                $dateType = "sales_date";
                $data->where('sales_date', $inputs['sales_from']);
            }
        }

        $groupSelection = "";
        $orderByCity = false;
        $orderByCountry = false;
        $orderByPeriod = true;
        $orderByBound = false;

        // Group By Agency, City, or Country
        if (isset($inputs['sales_channel_group']))
        {
            $groupSelection = "channel AS pos";
        }
        if (isset($inputs['region_group']))
        {
            $groupSelection = "region AS pos";
        }
        if (isset($inputs['country_group']))
        {
            $groupSelection = "country AS pos, country, s_agency__codes.country_id ";
            $orderByCountry = true;
        }
        if (isset($inputs['city_group']))
        {
            $groupSelection = "city AS pos, s_agency__codes.city_id, s_agency__codes.country_id, country ";
            $orderByCity = true;
            $orderByCountry = true;
        }
        if (isset($inputs['agency_group']))
        {
            $groupSelection = "agency AS pos, s_agency__codes.agency_id, s_agency__codes.country_id, country, city ";
            $orderByCity = true;
            $orderByCountry = true;
        }
        if (isset($inputs['flight_number_group']))
        {
            $groupSelection = "flight_number AS pos, bound";
        }
        if (isset($inputs['sector_group']))
        {
            $groupSelection = "CONCAT(s_depAirport.iata, '-', s_arrAirport.iata) AS pos, bound";
        }
        if (isset($inputs['route_group']))
        {
            $groupSelection = "CONCAT(s_depAirport.iata, '-', s_arrAirport.iata) AS pos, bound, " .
                              "IF( bound = 0, CONCAT(s_depAirport.iata, '-',s_arrAirport.iata, '-', s_depAirport.iata), ".
                              "CONCAT(s_arrAirport.iata, '-',s_depAirport.iata, '-', s_arrAirport.iata) ) AS route";
            $orderByBound = true;
        }

        if (isset($inputs['rbd_group']))
        {
            $groupSelection = "rbd AS pos";
        }
        if (isset($inputs['cabin_group']))
        {
            $groupSelection = "cabin AS pos";
        }
        if (isset($inputs['gds_group']))
        {
            $groupSelection = "CONCAT(s_reservations__systems.abbr, '-', s_reservations__systems.name) AS pos";
        }

        // Report Type (IF SET MODE WILL BE TOTALS ONLY, THUS TOTAL DATA NOT NEEDED)
        if ( (isset($inputs['report_type']) && $inputs['report_type']) || $groupSelection)
        {
            // Pagination Disabled
            $pagination = FALSE;
            // Mode IS Totals
            $totalMode = TRUE;

            $select = groupByReportType($inputs['report_type'], $dateType);

            // Add Group Select Field
            $selections = [];

            if (is_array($select))
            {
                foreach($select as $i => $each)
                {
                    $selections[] = DB::raw($each.($i == 0 ? " AS period" : ""));
                }
            }
            else
            {
                $selections[] = DB::raw($select." AS period ".($groupSelection ? ",{$groupSelection}" : ""));
            }

            $selections[] = DB::raw("SUM(IF (s_flights__order_status.order_status = 'R', 0, 1)* s_passengers__flights.fare) AS revenue");
//            $selections[] = DB::raw("SUM(IF (s_flights__order_status.order_status = 'R', -1, 1)* s_passengers__flights.fare) AS revenue");
            $selections[] = DB::raw("SUM(IF (s_flights__order_status.order_status = 'R', 0, 1)) AS pax");
//        $selections[] = DB::raw('COUNT(*) AS pax');
            // END of Selections

            $data->select($selections)
                 ->groupBy('period');

            if ($groupSelection)
            {
                $data->groupBy('pos');

                if ($orderByBound)
                {
                    $data->orderBy('bound');
                }
                if ($orderByPeriod)
                {
                    $data->orderBy('period');
                }
                if ($orderByCountry)
                {
                    $data->orderBy('country_id');
                }
                if ($orderByCity)
                {
                    $data->orderBy('city');
                }
                $data->orderBy('pos');
            }
        }
        else
        {
            // Total Revenue Calculator
            if ($totals && !$cacheQuery)
            {
                $totalData = clone $data;
                $totalData = $totalData->select([
                    DB::raw("SUM(IF (s_flights__order_status.order_status = 'R', 0, 1)* s_passengers__flights.fare) AS revenue"),
//                    DB::raw("SUM(IF (s_flights__order_status.order_status = 'R', -1, 1)* s_passengers__flights.fare) AS revenue"),
                    DB::raw("SUM(IF (s_flights__order_status.order_status = 'R', 0, 1)) AS pax"),
//                    DB::raw("COUNT(*) AS pax")
                ])
                    ->first();
            }
        }

        return [
            // Check IF Pagination Is Set
            0 => $pagination ? $data->simplePaginate(isset($inputs['recordsPerPage']) ? $inputs['recordsPerPage'] : 1000) : $data->get(),
            // Pagination ON/OFF
            1 => $pagination,
            // Total Mode
            2 => $totalMode,
            // Total Revenue Field
            3 => isset($totalData) ? $totalData : []
        ];
    }


    public static function prepareQueryByOperator(&$query, $field, $values, $operator = 'LIKE'){

            // If Operator is Given
            if (is_array($values) && array_key_exists('operator', $values)) {
                $operator = $values['operator'];
                unset($values['operator']);
            }

            switch ($operator){
                case '=':
                    if (is_array($values))
                        $query->whereIn($field, $values);
                    else
                        $query->where($field, $values);
                    break;
                case 'LIKE':
                    if (is_array($values)) {
                        array_walk($values, function($i, $each){
                            $each = $each.'%';
                        });
                        $query->whereIn($field, $values);
                    }
                    else {
                        $query->where($field, 'LIKE', $values . '%');
                    }
                    break;

            }


    }




}
