<?php namespace App\Models;
use App\Mail\TemplateEmail;
use App\Notifications\ReadAndSignNotification;
use Illuminate\Database\Eloquent\Model as Eloquent;
/**
 * Created by PhpStorm.

 */

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class LibrarySign extends Eloquent implements AuditableContract{

    use Auditable;

//    protected $auditExclude = [
//        'updated_by',
//        'created_by',
//    ];

    protected $guarded = [
        "id"
    ];

    protected $table = "library_sign";

    public $timestamps = true;

    public function library(){
        return $this->belongsTo(Library::class, "library_id");
    }

    public function airlineLibrary(){
        return $this->belongsTo(AirlineLibrary::class, "airline_library_id");
    }

    public function user(){
        return $this->belongsTo(User::class, "user_id");
    }

    public function locations(){
        return $this->hasMany(LibrarySignLocation::class, "library_sign_id");
    }

    public function departments(){
        return $this->hasMany(LibrarySignDepartment::class, "library_sign_id");
    }

    public function positions(){
        return $this->hasMany(LibrarySignPosition::class, "library_sign_id");
    }

    public function users(){
        return $this->hasMany(LibrarySignUser::class, "library_sign_id");
    }

    public static function checkIfFilterIsSame($request, $prevSign = null){
        if (!$prevSign){
            return false;
        }

        $userIDs = $request->get('user_ids');

        if($userIDs && count($userIDs)){
            foreach ($prevSign->users as $each){
                if (!in_array($each->user_id, $userIDs)){
                    return false;
                }
            }
        }

        $locationIDs = $request->get('location_ids');
        if ($locationIDs && count($locationIDs)){
            foreach ($prevSign->locations as $each){
                if (!in_array($each->location_id, $locationIDs)){
                    return false;
                }
            }
        }

        $departmentIDs = $request->get('department_ids');
        if ($departmentIDs && count($departmentIDs)){
            foreach ($prevSign->departments as $each){
                if (!in_array($each->department_id, $departmentIDs)){
                    return false;
                }
            }
        }

        $positionIDs = $request->get('position_ids');
        if ($positionIDs && count($positionIDs)){
            foreach ($prevSign->positions as $each){
                if (!in_array($each->position_id, $positionIDs)){
                    return false;
                }
            }
        }

        return true;
    }

    public static function libraryReadAndSignCheck($library, $viewed = null, $downloaded = null, $type = LIBRARY){
        $auth = Auth::user();

        if (!$auth || !$library){
            return;
        }

        $column = $type == LIBRARY ? "library_id" : "airline_library_id";
        if ($library->read_and_sign) {

            $view = LibraryView::orderBy("library_sign_id", "DESC")
                ->where("{$column}", $library->id)
                ->where("user_id", $auth->id)
                ->first();

            if ($view){
                if ($viewed) {
                    $view->viewed_at = date("Y-m-d H:i:s");
                    $view->save();
                }

                if ($downloaded) {
                    $view->downloaded_at = date("Y-m-d H:i:s");
                    $view->save();
                }
            }
        }
    }

    public static function libraryCheckSignedDocuments($library, $request, $type = LIBRARY, $newFileUploaded = false){
        $readAndSign = $request->get("read_and_sign") ? $request->get("read_and_sign") : $library->read_and_sign;

        $column = $type == LIBRARY ? "library_id" : "airline_library_id";

        if ($readAndSign && $library->attachment_file){

            $library->read_and_sign = date("Y-m-d H:i:s");
            $library->save();

            $locationIDs = $request->get('location_ids');
            $departmentIDs = $request->get('department_ids');
            $positionIDs = $request->get('position_ids');
            $userIDs = $request->get('user_ids');
            $auth = Auth::user();

            if (!$newFileUploaded) {
                // Remove previous signs if exists
                $prevSign = LibrarySign::where("{$column}", $library->id)
                    ->orderBy("created_at", "DESC")
                    ->first();

                if (self::checkIfFilterIsSame($request, $prevSign)) {
                    return;
                }
            }

            // Create new sign
            $sign = LibrarySign::create([
                "user_id"       => $auth->id,
                "{$column}"     => $library->id
            ]);

            // Add Filters
            if ($locationIDs && count($locationIDs)){
                foreach ($locationIDs as $each) {
                    LibrarySignLocation::create([ "library_sign_id"     => $sign->id, "location_id"       => $each ]);
                }
            }

            if ($departmentIDs && count($departmentIDs)){
                foreach ($departmentIDs as $each) {
                    LibrarySignDepartment::create([ "library_sign_id"   => $sign->id, "department_id"     => $each ]);
                }
            }

            if ($positionIDs && count($positionIDs)) {
                foreach ($positionIDs as $each) {
                    LibrarySignPosition::create(["library_sign_id" => $sign->id, "position_id" => $each]);
                }
            }

            if ($userIDs && count($userIDs)){
                foreach ($userIDs as $each) {
                    LibrarySignUser::create([ "library_sign_id"   => $sign->id, "user_id"     => $each ]);
                }
            }

            // If Users specified
            if ($userIDs && count($userIDs)){
                $users = User::whereIn("id", $userIDs)->get();
            }
            else {
                // 1.Matching users
                $users = findUsersByDepartmentPositionLocation($locationIDs, $departmentIDs, $positionIDs);
            }

            foreach ($users as $user){
                // 2.Create Library Views record for each one
                $viewRecord = LibraryView::where("user_id", $user->id)
                    ->where("{$column}", $library->id)
                    ->first();

                if ($viewRecord){
                    $viewRecord->viewed_at = null;
                    $viewRecord->downloaded_at = null;
                    $viewRecord->signed_at = null;
                    $viewRecord->save();
                }
                else {
                    LibraryView::create([
                        "user_id"           => $user->id,
                        "library_sign_id"   => $sign->id,
                        "{$column}"        => $library->id,
                    ]);
                }

                // 3.Notify
                try {
                    $user->notify(new ReadAndSignNotification($library));
                }
                catch (\Exception $e){
                    if (app()->environment() == PRODUCTION) {
                        Mail::to("dilovar88@mail.ru")
                            ->queue(new TemplateEmail($e->getMessage(), "Library Sign Issue Line 232"));
                    }
                }
            }

            $now = date("Y-m-d H:i:s");
            if (count($users)){
                $library->read_and_sign_notified_at = $now;
            }
            $library->save();

            // Save sign
            $sign->notified_at = $now;
            $sign->save();

            // 3.Cron and check if new users are added and Notify them and create record for them as well
//            User::find(44)->notify(new ReadAndSignNotification($library));
        }
    }
}
