<?php namespace App\Models;

use App\Classes\Staff\Flight\StaffFlight;
use App\Classes\Staff\Report\AirlineHours;
use App\Classes\Staff\Report\FlightStaffHours;
use App\Classes\Staff\Report\FlightHours;
use App\Classes\Staff\Report\FlightNumberHours;
use App\Classes\Staff\Report\StaffHours;
use App\Classes\Staff\Report\StaffHoursByAirline;
use function foo\func;
use Illuminate\Database\Eloquent\Model as Eloquent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

/**
 * Created by PhpStorm.

 */

use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class FlightStaff extends Eloquent implements AuditableContract{

    use Auditable;

    protected $auditExclude = [
        'updated_by',
        'created_by',
    ];

    protected $table = "flights__staff";

    public $timestamps = true;

    protected $guarded = [
        "id",
    ];

    public function flight(){
        return $this->belongsTo("App\\Models\\Flight", "flight_id");
    }

    public function user(){
        return $this->belongsTo("App\\Models\\User", "user_id");
    }

    public function service(){
        return $this->belongsTo("App\\Models\\Service", "service_id");
    }

    public static function getAllUpdated($period = ['from' => null, 'to' => null], $user = null, $selectedAirports = null, $selectedAirlines = null,
                                  $flightNumbers = null, $report_type = null, $handlingAirports = null){

        $flightStaff = FlightStaff::with([
            'user',
            'user.position',
            'user.location',
            'user.location.airport',
            'service',
            'flight',
            'flight.aircraft',
            'flight.flightNumber',
            'flight.flightNumber.airline',
            'flight.flightNumber.departureAirport',
            'flight.flightNumber.arrivalAirport',
        ])
            ->join("flights", "flights.id", "=", "flights__staff.flight_id");

        if ($report_type){
            $flightStaff->join("services", "services.id", "=", "flights__staff.service_id");

            switch ($report_type){
                case EXCLUDING_OCC:
                case EXCLUDING_OCC_ALL_STATIONS:
                    $flightStaff->where("services.abbr", "!=" , "OCC");
                    break;

                case OCC_REPORT:
                case OCC_REPORT_ALL_STATIONS:
                    $flightStaff->where("services.abbr", "OCC");
                    break;

                case COMBINED_REPORT:
                case COMBINED_ALL_STATIONS:
                default:
                    break;

            }
        }


        $flightStaff->whereNull("flights.deleted_at")
            ->where(function($sql){
                $sql->whereNull("flights.cancelled_at")
                    ->orWhere(function ($sql2){
                        $sql2->whereNotNull("cancelled_at")
                            ->where("sod_completed", true);
                    });
            });
//                    ->whereNull("flights.cancelled_at");

        if ($user){
            $flightStaff->where("flights__staff.user_id", $user->id);
        }

        // This has to be for flights !!!
        if (isset($period['month'], $period['year'])) {
            $month = is_array($period['month']) ? $period['month'] : [ $period ['month'] ];
            $year = is_array($period['year']) ? $period['year'] : [ $period ['year'] ];

            $flightStaff->where(function($query) use ($month, $year) {
                $query->where(function ($sql) use ($month, $year) {
                    $sql->whereIn(DB::raw("MONTH(planned_report_time)"), $month)
                        ->whereIn(DB::raw("YEAR(planned_report_time)"), $year);
                })
                    ->orWhere(function ($sql) use ($month, $year) {
                        $sql->whereIn(DB::raw("MONTH(planned_release_time)"), $month)
                            ->whereIn(DB::raw("YEAR(planned_release_time)"), $year);
                    })
                    ->orWhere(function ($sql) use ($month, $year) {
                        $sql->whereIn(DB::raw("MONTH(report_time)"), $month)
                            ->whereIn(DB::raw("YEAR(report_time)"), $year);
                    })
                    ->orWhere(function ($sql) use ($month, $year) {
                        $sql->whereIn(DB::raw("MONTH(release_time)"), $month)
                            ->whereIn(DB::raw("YEAR(release_time)"), $year);
                    })
                ;
            });
        }
        elseif (isset($period['from'], $period['to'])) {
            $flightStaff->where(function($sql) use ($period){
                $sql->whereBetween  ("planned_report_time", [ $period['from'], $period['to']])
                    ->orWhereBetween("planned_release_time", [ $period['from'], $period['to']])
                    ->orWhereBetween("report_time", [ $period['from'], $period['to']])
                    ->orWhereBetween("release_time", [ $period['from'], $period['to']]);

            });
        }

        /*
         * Have disabled it and put search flights
        if (isset($period['month']) && isset($period['year'])) {
            $month = is_array($period['month']) ? $period['month'] : [ $period ['month'] ];
            $year = is_array($period['year']) ? $period['year'] : [ $period ['year'] ];

            $flightStaff->whereIn(DB::raw("MONTH(planned_report_time)"), $month)
                        ->whereIn(DB::raw("YEAR(planned_report_time)"), $year);

        }
        elseif (isset($period['from']) || isset($period['to'])) {
            if (isset($period['from']) && $period['from'] && $period['from'] != "____-__-__"){
                $flightStaff->where(function($sql) use ($period){
                    $sql->where('planned_report_time', '>=', $period['from']." 00:00:00")
                        ->orWhere('report_time', '>=', $period['from']." 00:00:00");
                });
            }

            if (isset($period['to']) && $period['to'] && $period['to'] != "____-__-__"){
                $flightStaff->where(function($sql) use ($period){
                    $sql->where('planned_report_time', '<=', $period['to']." 23:59:59")
                        ->orWhere('report_time', '<=', $period['to']." 23:59:59");
                });
            }
        }
        */

        $flightStaff = $flightStaff->whereNull("flights__staff.deleted_at")
                                ->orderBy("flights__staff.planned_report_time")
                                ->orderBy("flights__staff.report_time")
                                ->orderBy("flights__staff.user_id")
                                ->get([
                                    "flights__staff.*"
                                ]);

        $data = [];

        if (in_array($report_type, [COMBINED_ALL_STATIONS, EXCLUDING_OCC_ALL_STATIONS, OCC_REPORT_ALL_STATIONS])){
            $selectedAirports = array_keys($handlingAirports); // [];
        }
        else {
            if ($selectedAirports) {
                $selectedAirports = is_array($selectedAirports) ? $selectedAirports : [$selectedAirports];
            }
        }

        if ($selectedAirlines){
            $selectedAirlines = is_array($selectedAirlines) ? $selectedAirlines : [ $selectedAirlines ];
        }

        foreach ($flightStaff as $i => $each) {

            if (!$each->flight){
                continue;
            }

            $flightNumber = $each->flight->flightNumber;

            if ($selectedAirports || $selectedAirlines){
                // Filter by Airport
                if ($selectedAirports) {
                    $departureAirport = $flightNumber->departureAirport;
                    $arrivalAirport = $flightNumber->arrivalAirport;
                    // debug($flightNumber->airline->iata." ".$flightNumber->flight_number." | ".$departureAirport->iata. "-".$arrivalAirport->iata);
                    if (
                        ($departureAirport && !in_array($departureAirport->id, $selectedAirports))
                        && ($arrivalAirport && !in_array($arrivalAirport->id, $selectedAirports))) {
                        continue;
                    }
                }

                // Filter by Airline
                if ($selectedAirlines) {
                    if (!in_array($flightNumber->airline->id, $selectedAirlines)) {
                        continue;
                    }
                }

                if ($flightNumbers){
                    if (!in_array($flightNumber->id, $flightNumbers)){
                        continue;
                    }
                }
            }

            $staffFlight = new StaffFlight($each);

            $depTime = strtotime($staffFlight->pReportTime);
            if (!isset($data[$depTime])){
                $data[$depTime] = $staffFlight;
            }
            else {
                do {
                    $depTime++;
                }while(isset($data[$depTime]));

                $data[$depTime] = $staffFlight;
            }

            /*
            if (!isset($data[$flightNumber->airline->id])){
                $data[$flightNumber->airline->id] = [];
            }

            if (!isset($data[$flightNumber->airline->id][$staffFlight->departure])){
                $data[$flightNumber->airline->id][$staffFlight->departure] = [];
            }

            $data[$flightNumber->airline->id][$staffFlight->departure][] = $staffFlight;
            */
        }

        ksort($data);

        return array_values($data);
    }


    public static function getAll($period = ['from' => null, 'to' => null],
                                  $user = null, $selectedAirports = null, $selectedAirlines = null,
                                  $flightNumbers = null, $report_type = null, $handlingAirports = null, $serviceIDs = null){

        $flightStaff = FlightStaff::with([
                                        'user',
                                        'user.position',
                                        'user.location',
                                        'user.location.airport',
                                        'service',
                                        'flight',
                                        'flight.aircraft',
                                        'flight.flightNumber',
                                        'flight.flightNumber.airline',
                                        'flight.flightNumber.departureAirport',
                                        'flight.flightNumber.arrivalAirport',
                                    ])
            ->join("flights", "flights.id", "=", "flights__staff.flight_id");

        if ($report_type){
            $flightStaff->join("services", "services.id", "=", "flights__staff.service_id");

            switch ($report_type){
                case EXCLUDING_OCC:
                case EXCLUDING_OCC_ALL_STATIONS:
                    $flightStaff->where("services.abbr", "!=" , "OCC");
                    break;

                case OCC_REPORT:
                case OCC_REPORT_ALL_STATIONS:
                    $flightStaff->where("services.abbr", "OCC");
                    break;

                case COMBINED_REPORT:
                case COMBINED_ALL_STATIONS:
                default:
                    break;

            }
        }

        if ($serviceIDs) {
            $flightStaff->whereIn("service_id", $serviceIDs);
        }


        $flightStaff->whereNull("flights.deleted_at")
                    ->where(function($sql){
                        $sql->whereNull("flights.cancelled_at")
                            ->orWhere(function ($sql2){
                               $sql2->whereNotNull("cancelled_at")
                                    ->where("sod_completed", true);
                            });
                    });
//                    ->whereNull("flights.cancelled_at");

        if ($user){
            $flightStaff->where("flights__staff.user_id", $user->id);
        }

        // This has to be for flights !!!
        if (isset($period['month'], $period['year'])) {
            $month = is_array($period['month']) ? $period['month'] : [ $period ['month'] ];
            $year = is_array($period['year']) ? $period['year'] : [ $period ['year'] ];

            $flightStaff->where(function($query) use ($month, $year) {
                $query->where(function ($sql) use ($month, $year) {
                    $sql->whereIn(DB::raw("MONTH(std)"), $month)
                        ->whereIn(DB::raw("YEAR(std)"), $year);
                })
                    ->orWhere(function ($sql) use ($month, $year) {
                        $sql->whereIn(DB::raw("MONTH(ptd)"), $month)
                            ->whereIn(DB::raw("YEAR(ptd)"), $year);
                    })
                    ->orWhere(function ($sql) use ($month, $year) {
                        $sql->whereIn(DB::raw("MONTH(etd)"), $month)
                            ->whereIn(DB::raw("YEAR(etd)"), $year);
                    })
                    ->orWhere(function ($sql) use ($month, $year) {
                        $sql->whereIn(DB::raw("MONTH(atd)"), $month)
                            ->whereIn(DB::raw("YEAR(atd)"), $year);
                    })
                ;
            });
        }
        elseif (isset($period['from'], $period['to'])) {
            $flightStaff->where(function($sql) use ($period){
                $sql->whereBetween  ("std", [ $period['from']." 00:00:00", $period['to']." 23:59:59"])
                    ->orWhereBetween("ptd", [ $period['from']." 00:00:00", $period['to']." 23:59:59"])
                    ->orWhereBetween("etd", [ $period['from']." 00:00:00", $period['to']." 23:59:59"])
                    ->orWhereBetween("atd", [ $period['from']." 00:00:00", $period['to']." 23:59:59"]);

            });
        }

        /*
         *
         * Have disabled it and put search flights
        if (isset($period['month']) && isset($period['year'])) {
            $month = is_array($period['month']) ? $period['month'] : [ $period ['month'] ];
            $year = is_array($period['year']) ? $period['year'] : [ $period ['year'] ];

            $flightStaff->whereIn(DB::raw("MONTH(planned_report_time)"), $month)
                        ->whereIn(DB::raw("YEAR(planned_report_time)"), $year);

        }
        elseif (isset($period['from']) || isset($period['to'])) {
            if (isset($period['from']) && $period['from'] && $period['from'] != "____-__-__"){
                $flightStaff->where(function($sql) use ($period){
                    $sql->where('planned_report_time', '>=', $period['from']." 00:00:00")
                        ->orWhere('report_time', '>=', $period['from']." 00:00:00");
                });
            }

            if (isset($period['to']) && $period['to'] && $period['to'] != "____-__-__"){
                $flightStaff->where(function($sql) use ($period){
                    $sql->where('planned_report_time', '<=', $period['to']." 23:59:59")
                        ->orWhere('report_time', '<=', $period['to']." 23:59:59");
                });
            }
        }
        */

        $flightStaff = $flightStaff->whereNull("flights__staff.deleted_at")
//                                    ->orderBy("flights__staff.planned_report_time")
//                                    ->orderBy("flights__staff.report_time")
//                                    ->orderBy("flights__staff.user_id")
                                    ->get([
                                        "flights__staff.*"
                                    ]);

        $data = [];

        if (in_array($report_type, [COMBINED_ALL_STATIONS, EXCLUDING_OCC_ALL_STATIONS, OCC_REPORT_ALL_STATIONS])){
            $selectedAirports = array_keys($handlingAirports); // [];
        }
        else {
            if ($selectedAirports) {
                $selectedAirports = is_array($selectedAirports) ? $selectedAirports : [$selectedAirports];
            }
        }

        if ($selectedAirlines){
            $selectedAirlines = is_array($selectedAirlines) ? $selectedAirlines : [ $selectedAirlines ];
        }

        $ap = $selectedAirports && count($selectedAirports) == 1 ? $selectedAirports[0] : null;
//        debug("SEL AP:");
        debug("AP: ".$ap);

        $handlingStations = Airport::getHandlingStationsIdAndObject();

        foreach ($flightStaff as $i => $each) {

            if (!$each->flight){
                continue;
            }

            // IT's clear doesn't take much time
            if ($ap){
                list($serviceType) = FlightStaff::getFlightServiceType($each->flight, $ap, $handlingStations);

//                debug("Flight AP service Type: {$serviceType} - Flight Staff Type: {$each->service_type}");
                if ($each->service_type != TURNAROUND_SERVICE && $serviceType != $each->service_type){
                    continue;
                }
            }

            $flightNumber = $each->flight->flightNumber;

            if ($selectedAirports || $selectedAirlines){
                // Filter by Airport

                if ($selectedAirports) {
                    /*
                     * Previous Filter by Flights DEP/ARR
                     *
                    $departureAirport = $each->flight->departureAirport ? $each->flight->departureAirport : $flightNumber->departureAirport;
                    $arrivalAirport = $each->flight->arrivalAirport ? $each->flight->arrivalAirport : $flightNumber->arrivalAirport;
                    if (($departureAirport && !in_array($departureAirport->id, $selectedAirports))
                         && ($arrivalAirport && !in_array($arrivalAirport->id, $selectedAirports))) {
                        continue;
                    }
                    */

                    // By User Location
                    if (!in_array($each->user->location->airport_id, $selectedAirports)){
                        continue;
                    }
                }

                // Filter by Airline
                if ($selectedAirlines) {
                    if (!in_array($flightNumber->airline->id, $selectedAirlines)) {
                        continue;
                    }
                }

                if ($flightNumbers){
                    if (!in_array($flightNumber->id, $flightNumbers)){
                        continue;
                    }
                }
            }

            $data[] = new StaffFlight($each);

            /*
            $depTime = strtotime($staffFlight->pReportTime);
            if (!isset($data[$depTime])){
                $data[$depTime] = $staffFlight;
            }
            else {
                do {
                    $depTime++;
                }while(isset($data[$depTime]));

                $data[$depTime] = $staffFlight;
            }
            */

            /*
            if (!isset($data[$flightNumber->airline->id])){
                $data[$flightNumber->airline->id] = [];
            }

            if (!isset($data[$flightNumber->airline->id][$staffFlight->departure])){
                $data[$flightNumber->airline->id][$staffFlight->departure] = [];
            }

            $data[$flightNumber->airline->id][$staffFlight->departure][] = $staffFlight;
            */
        }

//        ksort($data);

        return $data; // array_values($data);
    }

    /**
     * @var StaffFlight[] $staffFlights
     * @param bool|false $timeFormat
     * @return array
     */
    public static function getHoursByUser($staffFlights, $timeFormat = false){

        $results = [
            'daily'         => [],
            'monthly'       => [],
            'yearly'        => []
        ];

        foreach ($staffFlights as $each) {

            // PLN
            if ($each->pReportDate){
                // Daily
                if (!isset($results['daily'][$each->pReportDate])){
                    $results['daily']["{$each->pReportDate}"] = [
                        "act"  => 0,
                        "pln"  => 0,
                    ];
                }
                $results['daily']["{$each->pReportDate}"]['pln'] += $each->pDurationHours;

                // Monthly
                $month = $each->getReportPeriod("Monthly", true);
                if (!isset($results['monthly'][$month])){
                    $results['monthly'][$month] = [
                        "act"  => 0,
                        "pln"  => 0,
                    ];
                }
                $results['monthly'][$month]['pln'] += $each->pDurationHours;

                // Yearly
                $year = $each->getReportYear(true);
                if (!isset($results['yearly'][$year])){
                    $results['yearly'][$year] = [
                        "act"  => 0,
                        "pln"  => 0,
                    ];
                }
                $results['yearly'][$year]['pln'] += $each->pDurationHours;
            }

            // ACT
            if ($each->reportDate){
                // Daily
                if (!isset($results['daily'][$each->reportDate])){
                    $results['daily']["{$each->reportDate}"] = [
                        "act"  => 0,
                        "pln"  => 0,
                    ];
                }
                $results['daily']["{$each->reportDate}"]['act'] += $each->durationHours;

                // Monthly
                $month = $each->getReportPeriod("Monthly");
                if (!isset($results['monthly'][$month])){
                    $results['monthly'][$month] = [
                        "act"  => 0,
                        "pln"  => 0,
                    ];
                }
                $results['monthly'][$month]['act'] += $each->durationHours;

                // Yearly
                $year = $each->getReportYear(['act']);
                if (!isset($results['yearly'][$year])){
                    $results['yearly'][$year] = [
                        "act"  => 0,
                        "pln"  => 0,
                    ];
                }
                $results['yearly'][$year]['act'] += $each->durationHours;
            }
        }

        if ($timeFormat){
            foreach ($results as $time => $data ) {
                foreach ($data as $i => $each) {
                    $results[$time][$i]['pln'] = hoursToTime($each['pln'], true);
                    $results[$time][$i]['act'] = hoursToTime($each['act'], true);
                }
            }
        }

        return $results;

    }

    /**
     * @param $staffFlights
     * @param null $period
     * @param null $periodOption
     * @return array
     */
    public static function getHours($staffFlights, $period = null, $periodOption = null){

        $results = [
            "data"          => [],
            "pTotalTime"    => 0,
            "totalTime"     => 0,

            "pTotalHours"   => 0,
            "totalHours"    => 0,

            "holidays"      => 0,
        ];

        $resultsByAirline = [
            "data"      => [],
            "total"     => new AirlineHours(),
        ];

        $resultsByFlightNumber = [
            "data"      => [],
            "total"     => new FlightNumberHours(),
        ];

        $resultsByFlight = [
            "data"      => [],
            "total"     => new FlightHours(),
        ];

        $resultsDaily = [
            "data"      => [],
            "total"     => new FlightHours(),
        ];

        $periodUnique = [];

        $handlingStations = Airport::getHandlingStationsIdAndObject();

        /** @var StaffFlight $each */
        foreach ($staffFlights as $each) {

            if (!isset($results["data"][$each->userId])){
                $results["data"][$each->userId] = new StaffHours($period, $periodOption, $each);
            }

            if (!isset($resultsByAirline["data"][$each->airline->id])){
                $resultsByAirline["data"][$each->airline->id] = new AirlineHours($each);
            }

            if (!isset($resultsByFlightNumber["data"][$each->flightNumber->id])){
                $resultsByFlightNumber["data"][$each->flightNumber->id] = new FlightNumberHours($each);
            }

            if (!isset($resultsByFlight["data"][$each->flightNumber->id])){
                $resultsByFlight["data"][$each->flightNumber->id] = [];
            }

            if (!isset($resultsByFlight["data"][$each->flightNumber->id][$each->flight_id])){
                $resultsByFlight["data"][$each->flightNumber->id][$each->flight_id] = new FlightHours($each);
            }

            if (!isset($resultsDaily["data"][$each->flight_id])){
                $resultsDaily["data"][$each->flight_id] = new FlightStaffHours($period, $periodOption, $each, $handlingStations);
            }


            /** @var StaffHours $staff */
            $staff = $results["data"][$each->userId];
//            /** @var StaffHours $staffTotal */
//            $staffTotal = $results["total"];

            /** @var AirlineHours $airline */
            $airline = $resultsByAirline["data"][$each->airline->id];
            /** @var AirlineHours $airlineTotal */
            $airlineTotal = $resultsByAirline["total"];

            /** @var FlightNumberHours $flightNumber */
            $flightNumber = $resultsByFlightNumber["data"][$each->flightNumber->id];
            /** @var FlightNumberHours $flightNumberTotal */
            $flightNumberTotal = $resultsByFlightNumber["total"];

            /** @var FlightHours $flight */
            $flight = $resultsByFlight["data"][$each->flightNumber->id][$each->flight_id];
            /** @var FlightHours $flightTotal */
            $flightTotal = $resultsByFlight["total"];

            /** @var FlightStaffHours $dailyReport */
            $dailyReport = $resultsDaily["data"][$each->flight_id];
//            /** @var FlightStaffHours $dailyReportTotal */
//            $dailyReportTotal = $resultsDaily["total"];
            $dailyStaff =  new StaffHours($period, $periodOption, $each);
            $dailyReport->staffList[] = $dailyStaff;

            $pDuration = $each->pDurationHours;
            if ($pDuration){
                // Monthly
                $selectedPeriod = $each->getReportPeriod($period, true);

                $staff->setPlannedHours($selectedPeriod, $pDuration, $each->pReportTime, $each->pReleaseTime);
//                $staffTotal->setPlannedHoursAndSalary($selectedPeriod, $pDuration, $staff->getSalaryRate(), $each->pReportTime, $each->pReleaseTime);

                $airline->setPlannedHours($selectedPeriod, $pDuration, $staff->getSalaryRate());
                $airlineTotal->setPlannedHours($selectedPeriod, $pDuration, $staff->getSalaryRate());

                $flightNumber->setPlannedHours($selectedPeriod, $pDuration, $staff->getSalaryRate());
                $flightNumberTotal->setPlannedHours($selectedPeriod, $pDuration, $staff->getSalaryRate());

                $flight->setPlannedHours($selectedPeriod, $pDuration, $staff->getSalaryRate());
                $flightTotal->setPlannedHours($selectedPeriod, $pDuration, $staff->getSalaryRate());

                $dailyReport->setPlannedHours($selectedPeriod, $pDuration);
                $dailyStaff->setPlannedHours($selectedPeriod, $pDuration, $each->pReportTime, $each->pReleaseTime);
//                $dailyReportTotal->setPlannedHoursAndSalary($selectedPeriod, $pDuration, $staff->getSalaryRate());

                if (!in_array($selectedPeriod, $periodUnique)){
                    $periodUnique[] = $selectedPeriod;
                }
            }

            $duration = $each->durationHours;
            if ($duration){

                // Monthly
                $selectedPeriod = $each->getReportPeriod($period);

                $staff->setActualHours($selectedPeriod, $duration, $each->reportTime, $each->releaseTime);
                //$staffTotal->setActualHoursAndSalary($selectedPeriod, $duration, $staff->getSalaryRate());

                $airline->setActualHours($selectedPeriod, $duration, $staff->getSalaryRate());
                $airlineTotal->setActualHours($selectedPeriod, $duration, $staff->getSalaryRate());

                $flightNumber->setActualHours($selectedPeriod, $duration, $staff->getSalaryRate());
                $flightNumberTotal->setActualHours($selectedPeriod, $duration, $staff->getSalaryRate());

                $flight->setActualHours($selectedPeriod, $duration, $staff->getSalaryRate());
                $flightTotal->setActualHours($selectedPeriod, $duration, $staff->getSalaryRate());

                $dailyReport->setActualHours($selectedPeriod, $duration);
                $dailyStaff->setActualHours($selectedPeriod, $duration, $each->reportTime, $each->releaseTime);

//                $dailyReportTotal->setActualHoursAndSalary($selectedPeriod, $duration, $staff->getSalaryRate());
            }

            $dailyStaff->setTotals();
        }

        /* @var StaffHours $staff */
        $totalContracted = 0;
        $shortHours = $overtimeHours = [];
        foreach ($results["data"] as &$staff) {
            $staff->setTotals();
            $staff->setPeriodTotals();
            $totalContracted += $staff->contracted;

            foreach ($staff->shortHours as $month => $short) {
                if (!isset($shortHours[$month])){
                    $shortHours[$month] = 0;
                }

                $shortHours[$month] += $short;
            }

            foreach ($staff->overtimeHours as $month => $overtime) {
                if (!isset($overtimeHours[$month])){
                    $overtimeHours[$month] = 0;
                }

                $overtimeHours[$month] += $overtime;
            }

            $results["pTotalHours"]  += $staff->pTotalHours;
            $results["totalHours"]   += $staff->totalHours;
        }

        $results["pTotalTime"] = hoursToTime($results["pTotalHours"], true, true);
        $results["totalTime"] = hoursToTime($results["totalHours"], true, true);

//        $staffTotal = $results["total"];
//        $staffTotal->contracted = $totalContracted;
//        $staffTotal->setTotals();
//        $staffTotal->setPeriodTotals();
//        $staffTotal->overtimeHours = $overtimeHours;
//        $staffTotal->shortHours = $shortHours;

        /* @var AirlineHours $airline */
        foreach ($resultsByAirline["data"] as &$airline) {
            $airline->setTotals();
            $airline->setMonthlyTotals();
        }
        $airlineTotal = $resultsByAirline["total"];
        $airlineTotal->setTotals();
        $airlineTotal->setMonthlyTotals();

        /* @var FlightNumberHours $flightNumber */
        foreach ($resultsByFlightNumber["data"] as &$flightNumber) {
            $flightNumber->setTotals();
            $flightNumber->setMonthlyTotals();
        }
        $flightNumberTotal = $resultsByFlightNumber["total"];
        $flightNumberTotal->setTotals();
        $flightNumberTotal->setMonthlyTotals();

        /* @var FlightHours $flight */
        foreach ($resultsByFlight["data"] as $flightNumberId => &$flights) {
            foreach ($flights as &$flight) {
                $flight->setTotals();
                $flight->setMonthlyTotals();
            }
        }
        $flightTotal = $resultsByFlight["total"];
        $flightTotal->setTotals();
        $flightTotal->setMonthlyTotals();


        return [$results, $resultsByAirline, $resultsByFlightNumber, $resultsByFlight, $resultsDaily, $periodUnique];
    }

    public static function getFlightServiceType($flight, $airportId = null, $handlingStations = null){
        if (!$flight || !$flight->flightNumber){
            return null;
        }

        $flightNumber = $flight->flightNumber;
        $serviceType = $station = $depArrTime = null;

        $departureAirportID = $flight->departure_airport_id ? $flight->departure_airport_id : $flightNumber->departure_airport_id;
        $arrivalAirportID = $flight->arrival_airport_id ? $flight->arrival_airport_id : $flightNumber->arrival_airport_id;

        if (!$handlingStations){
            $handlingStations = Airport::getHandlingStationsIdAndObject();
        }

        if ($airportId)
        {
            if ($airportId == $departureAirportID &&
                array_key_exists($departureAirportID, $handlingStations))
            {
                $station = $handlingStations[$airportId];
                $serviceType = DEPARTURE_SERVICE;
                $depArrTime = getFlightDepartureInitialDate($flight);
            }
            elseif ($airportId == $arrivalAirportID &&
                array_key_exists($arrivalAirportID, $handlingStations))
            {
                $station = $handlingStations[$airportId];
                $serviceType = ARRIVAL_SERVICE;
                $depArrTime = getFlightArrivalInitialDate($flight);
            }
        }

        if (!$serviceType) {
            if (array_key_exists($departureAirportID, $handlingStations))
            {
                $serviceType = DEPARTURE_SERVICE;
                $depArrTime = getFlightDepartureInitialDate($flight);
                $station = $handlingStations[$departureAirportID];
            }
            elseif (array_key_exists($arrivalAirportID, $handlingStations))
            {
                $serviceType = ARRIVAL_SERVICE;
                $depArrTime = getFlightArrivalInitialDate($flight);
                $station = $handlingStations[$arrivalAirportID];
            }
        }


        return [$serviceType, $station, $depArrTime];
    }

    public static function updateFlightsStaff(Request $request, $flight, $flightComDetails = null, $resetSodUpdatedField = false,
            $updateDepAndArrivalServices = false){

        $staffAirports = StaffStation::currentStaffStationsList();
        $selectedAirportIds = Airport::getSelectedAirportIds($request);
        $selectedAirportIds = (!$selectedAirportIds || !count($selectedAirportIds)) ? $staffAirports[0] : $selectedAirportIds[0];
//        debug("SEL AP ID:{$selectedAirportIds}");
//        $serviceType =  getFlightServiceType($flight->flightNumber, $selectedAirportIds);

        $services = Service::pluck("id")->all();

//        list($serviceType, $service_type, $station) = self::getFlightServiceType($flight, $selectedAirportIds);

        $authUserID = Auth::user()->id;

        // Service And Staff
        $currentDate = date("Y-m-d H:i:s");
        $processedFlightStaffIds = $createdFlightComStaff = [];
        foreach ($services as $serviceId) {
            $staffInput = "s_" . $serviceId;
            $slaTypeInput = "t_" . $serviceId;
            $reportTimeInput = "r_" . $serviceId;
            $releaseTimeInput = "f_" . $serviceId;

            $pReportTimeInput = "pr_" . $serviceId;
            $pReleaseTimeInput = "pf_" . $serviceId;

            $slaType = $request->get($slaTypeInput) == "t" ? TURNAROUND_SERVICE : ($request->get($slaTypeInput) == "a" ? ARRIVAL_SERVICE : DEPARTURE_SERVICE);

            if ($request->exists($staffInput) && $request->get($staffInput)) {
                foreach ($request->get($staffInput) as $j => $userId) {
                    if (!$userId){
                        continue;
                    }

                    $flightStaff = FlightStaff::where("flight_id", $flight->id)
                        ->where("user_id", $userId)
                        ->where("service_id", $serviceId)
                        ->whereNull("deleted_at")
                        ->orderBy("deleted_at")
                        ->first();


                    $pReportTime = $request->get($pReportTimeInput)[$j];
                    $pReleaseTime = $request->get($pReleaseTimeInput)[$j];

                    $reportTime = $request->get($reportTimeInput)[$j];
                    $releaseTime = $request->get($releaseTimeInput)[$j];

                    // check and correct
                    $pReportTime  = $pReportTime  && $pReportTime. ":00" != EMPTY_DATETIME ? $pReportTime. ":00"  : null;
                    $pReleaseTime = $pReleaseTime && $pReleaseTime.":00" != EMPTY_DATETIME ? $pReleaseTime. ":00" : null;
                    $reportTime   = $reportTime   && $reportTime. ":00" != EMPTY_DATETIME ? $reportTime. ":00"  : null;
                    $releaseTime  = $releaseTime  && $releaseTime.":00" != EMPTY_DATETIME ? $releaseTime. ":00" : null;
                    //

                    if ($flightStaff) {
                        $createNew = $flightStaff->planned_report_time != $pReportTime || $flightStaff->planned_release_time != $pReleaseTime
                                    || $flightStaff->report_time != $reportTime || $flightStaff->release_time != $releaseTime;
                    }
                    else {
                        $createNew = true;
                        $flightStaff = new FlightStaff();
                        $flightStaff->flight_id = $flight->id;
                        $flightStaff->user_id = $userId;
                        $flightStaff->service_id = $serviceId;
                        $flightStaff->created_by = $authUserID;
                        // Based on each SLA
                        $flightStaff->service_type = $slaType;
                    }

                    if ($resetSodUpdatedField){
                        $flightStaff->sod_updated_at = null;
                        $flightStaff->sod_updated_by = null;
                    }

                    $flightStaff->deleted_at = null;
                    $flightStaff->planned_report_time   = $pReportTime;
                    if ($pReleaseTime && strtotime($pReleaseTime) > strtotime($pReportTime)){
                        $flightStaff->planned_release_time  = $pReleaseTime;
                    }

                    $flightStaff->report_time   = $reportTime;
                    if ($releaseTime && strtotime($releaseTime) > strtotime($reportTime)){
                        $flightStaff->release_time  = $releaseTime;
                    }
                    $flightStaff->updated_by = $authUserID;
                    $flightStaff->save();

                    $processedFlightStaffIds[] = $flightStaff->id;

                    // Flight Com Details
                    if ($createNew && $flightComDetails){
                        $createdFlightComStaff[] = FlightComFlightStaff::createFromFlightStaff($flightComDetails, $flightStaff);
                    }
                }
            }
        }

        $flightStaffForRemoval = FlightStaff::where("flight_id", $flight->id)
                                            ->whereNotIn("id", $processedFlightStaffIds)
                                            ->whereNull("deleted_at");

//        debug("SERVICE TYPE: {$serviceType}");
//        if (!$updateDepAndArrivalServices){
//            $flightStaffForRemoval->where("service_type", $serviceType);
//        }

        $flightStaffForRemoval = $flightStaffForRemoval->get();

        foreach ($flightStaffForRemoval as &$each) {
            $each->deleted_at = $currentDate;
            $each->updated_by = $authUserID;
            $each->save();

            // Flight Com Details
            if ($flightComDetails){
                $createdFlightComStaff[] = FlightComFlightStaff::createFromFlightStaff($flightComDetails, $each);
            }
        }

        // Remove Flight Com Details if nothing was changed
        if ($flightComDetails){
            return $createdFlightComStaff;
        }

        return true;
    }

    public static function updateFlightsStaffByStation(Request $request, $flight, $flightComDetails = null, $resetSodUpdatedField = false){

        $airportID = $request->get("airport_id");

        $services = Service::pluck("id")->all();
        list($serviceType, $station) = FlightStaff::getFlightServiceType($flight, $airportID);

        $authUserID = Auth::user()->id;

        // Service And Staff
        $currentDate = date("Y-m-d H:i:s");
        $processedFlightStaffIds = $createdFlightComStaff = [];

        $slaTypesArray = [];
        foreach ($services as $serviceId) {
            $staffInput = "s_" . $serviceId;
            $slaTypeInput = "t_" . $serviceId;
            $reportTimeInput = "r_" . $serviceId;
            $releaseTimeInput = "f_" . $serviceId;

            $pReportTimeInput = "pr_" . $serviceId;
            $pReleaseTimeInput = "pf_" . $serviceId;

            $slaType = $request->get($slaTypeInput) == "t" ? TURNAROUND_SERVICE : ($request->get($slaTypeInput) == "a" ? ARRIVAL_SERVICE : DEPARTURE_SERVICE);

            if (!in_array($slaType, $slaTypesArray)){
                $slaTypesArray[] = $slaType;
            }

            if ($request->exists($staffInput) && $request->get($staffInput)) {
                foreach ($request->get($staffInput) as $j => $userId) {
                    if (!$userId){
                        continue;
                    }

                    $flightStaff = FlightStaff::where("flight_id", $flight->id)
                        ->where("user_id", $userId)
                        ->where("service_id", $serviceId)
                        ->whereNull("deleted_at")
                        ->orderBy("deleted_at")
                        ->first();


                    $pReportTime = $request->get($pReportTimeInput)[$j];
                    $pReleaseTime = $request->get($pReleaseTimeInput)[$j];

                    $reportTime = $request->get($reportTimeInput)[$j];
                    $releaseTime = $request->get($releaseTimeInput)[$j];

                    // check and correct
                    $pReportTime  = $pReportTime  && $pReportTime. ":00" != EMPTY_DATETIME ? $pReportTime. ":00"  : null;
                    $pReleaseTime = $pReleaseTime && $pReleaseTime.":00" != EMPTY_DATETIME ? $pReleaseTime. ":00" : null;
                    $reportTime   = $reportTime   && $reportTime. ":00" != EMPTY_DATETIME ? $reportTime. ":00"  : null;
                    $releaseTime  = $releaseTime  && $releaseTime.":00" != EMPTY_DATETIME ? $releaseTime. ":00" : null;
                    //

                    if ($flightStaff) {
                        $createNew = $flightStaff->planned_report_time != $pReportTime || $flightStaff->planned_release_time != $pReleaseTime
                            || $flightStaff->report_time != $reportTime || $flightStaff->release_time != $releaseTime;
                    }
                    else {
                        $createNew = true;
                        $flightStaff = new FlightStaff();
                        $flightStaff->flight_id = $flight->id;
                        $flightStaff->user_id = $userId;
                        $flightStaff->service_id = $serviceId;
                        $flightStaff->created_by = $authUserID;
                        // Based on each SLA
                    }

                    if ($resetSodUpdatedField){
                        $flightStaff->sod_updated_at = null;
                        $flightStaff->sod_updated_by = null;
                    }

                    $flightStaff->service_type = $slaType;
                    $flightStaff->deleted_at = null;
                    $flightStaff->planned_report_time   = $pReportTime;
                    if ($pReleaseTime && strtotime($pReleaseTime) > strtotime($pReportTime)){
                        $flightStaff->planned_release_time  = $pReleaseTime;
                    }

                    $flightStaff->report_time   = $reportTime;
                    if ($releaseTime && strtotime($releaseTime) > strtotime($reportTime)){
                        $flightStaff->release_time  = $releaseTime;
                    }
                    $flightStaff->updated_by = $authUserID;
                    $flightStaff->save();

                    $processedFlightStaffIds[] = $flightStaff->id;

                    // Flight Com Details
                    if ($createNew && $flightComDetails){
                        $createdFlightComStaff[] = FlightComFlightStaff::createFromFlightStaff($flightComDetails, $flightStaff);
                    }
                }
            }
        }

        // IF Only 1 type of SLA is selected

        $flightStaffForRemoval = FlightStaff::where("flight_id", $flight->id)
            ->whereNotIn("id", $processedFlightStaffIds)
            ->whereNull("deleted_at");

        if (count($slaTypesArray) == 1){
            $flightStaffForRemoval->where("service_type", $serviceType);
        }

        $flightStaffForRemoval = $flightStaffForRemoval->get();

//        debug("SERVICE TYPE: {$serviceType}");

        foreach ($flightStaffForRemoval as &$each) {
            $each->deleted_at = $currentDate;
            $each->updated_by = $authUserID;
            $each->save();

            // Flight Com Details
            if ($flightComDetails) {
                $createdFlightComStaff[] = FlightComFlightStaff::createFromFlightStaff($flightComDetails, $each);
            }
        }

        // Remove Flight Com Details if nothing was changed
        if ($flightComDetails){
            return $createdFlightComStaff;
        }

        return true;
    }

    public static function setFlightsStaff(Request $request, $flight, $airlineService, $flightStaff = null, $flightComDetails = null,
                                           $resetSodUpdatedField = false){

        $authUserID = Auth::user()->id;

        // Service And Staff
        $processedFlightStaffIds = $createdFlightComStaff = [];

        $serviceId = $airlineService->service_id;
        $staffInput = "s_" . $serviceId;
        $slaTypeInput = "t_" . $serviceId;

        $reportTimeInput = "r_" . $serviceId;
        $releaseTimeInput = "f_" . $serviceId;

        $pReportTimeInput = "pr_" . $serviceId;
        $pReleaseTimeInput = "pf_" . $serviceId;

        $slaType = $request->get($slaTypeInput) == "t" ? TURNAROUND_SERVICE : ($request->get($slaTypeInput) == "a" ? ARRIVAL_SERVICE : DEPARTURE_SERVICE);

        if ($request->exists($staffInput) && $request->get($staffInput)) {
            foreach ($request->get($staffInput) as $j => $userId) {
                if (!$userId){
                    auditDeleted($flightStaff);
                    continue;
                }

                if (!$flightStaff){
                    $flightStaff = FlightStaff::where("flight_id", $flight->id)
                        ->where("user_id", $userId)
                        ->where("service_id", $serviceId)
                        ->whereNull("deleted_at")
//                        ->orderBy("deleted_at")
                        ->first();
                }

                $pReportTime = $request->get($pReportTimeInput)[$j];
                $pReleaseTime = $request->get($pReleaseTimeInput)[$j];

                $reportTime = $request->get($reportTimeInput)[$j];
                $releaseTime = $request->get($releaseTimeInput)[$j];

                // check and correct
                $pReportTime  = $pReportTime  && $pReportTime. ":00" != EMPTY_DATETIME ? $pReportTime. ":00"  : null;
                $pReleaseTime = $pReleaseTime && $pReleaseTime.":00" != EMPTY_DATETIME ? $pReleaseTime. ":00" : null;
                $reportTime   = $reportTime   && $reportTime. ":00" != EMPTY_DATETIME ? $reportTime. ":00"  : null;
                $releaseTime  = $releaseTime  && $releaseTime.":00" != EMPTY_DATETIME ? $releaseTime. ":00" : null;
                //

                if ($flightStaff) {
                    $createNew = $flightStaff->planned_report_time != $pReportTime || $flightStaff->planned_release_time != $pReleaseTime
                        || $flightStaff->report_time != $reportTime || $flightStaff->release_time != $releaseTime;
                }
                else {
                    $createNew = true;
                    $flightStaff = new FlightStaff();
                    $flightStaff->flight_id = $flight->id;
                    $flightStaff->user_id = $userId;
                    $flightStaff->service_id = $serviceId;
                    $flightStaff->created_by = $authUserID;
                }

                if ($resetSodUpdatedField){
                    $flightStaff->sod_updated_at = null;
                    $flightStaff->sod_updated_by = null;
                }

                $flightStaff->service_type = $slaType;
                $flightStaff->user_id = $userId;
                $flightStaff->deleted_at = null;
                $flightStaff->planned_report_time   = $pReportTime;
                $flightStaff->planned_release_time  = strtotime($pReportTime) > strtotime($pReleaseTime) ? $pReportTime : $pReleaseTime;
                $flightStaff->report_time   = $reportTime;
                $flightStaff->release_time  = strtotime($reportTime) > strtotime($releaseTime) ? $reportTime : $releaseTime;
                $flightStaff->updated_by = $authUserID;
                $flightStaff->save();

                $processedFlightStaffIds[] = $flightStaff->id;

                // Flight Com Details
                if ($createNew && $flightComDetails){
                    $createdFlightComStaff[] = FlightComFlightStaff::createFromFlightStaff($flightComDetails, $flightStaff);
                }
            }
        }

        /*
        $flightStaffForRemoval = FlightStaff::where("flight_id", $flight->id)
            ->whereNotIn("id", $processedFlightStaffIds)
            ->whereNull("deleted_at")
            ->get();

        foreach ($flightStaffForRemoval as &$each) {
            $each->deleted_at = $currentDate;
            $each->updated_by = $authUserID;
            $each->save();

            // Flight Com Details
            if ($flightComDetails){
                $createdFlightComStaff[] = FlightComFlightStaff::createFromFlightStaff($flightComDetails, $each);
            }
        }

        // Remove Flight Com Details if nothing was changed
        if ($flightComDetails){
            return $createdFlightComStaff;
        }
        */

        return true;
    }

}
