<?php namespace App\Models;
use Illuminate\Database\Eloquent\Model as Eloquent;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;

/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov


 * Project: intranet.dev
 */

use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class FlightSchedule extends Eloquent implements AuditableContract
{
    use Auditable;

    protected $auditExclude = [
        'updated_by',
        'created_by',
    ];

    protected $table = "flights__schedule";

    protected $guarded = [
        "created_at",
        "updated_at",
        "id"
    ];

    public $timestamps = TRUE;

    public function flights(){

        return $this->hasMany(FlightScheduleFlight::class, "flight_schedule_id");
    }

    public function airlines(){

        return $this->hasMany(FlightScheduleAirline::class, "flight_schedule_id");
    }

    public function user(){

        return $this->belongsTo("App\\Models\\User", "created_by");
    }

    public static function scheduleByAirline($airline, $currentSchedule = true){
        $today = date("Y-m-d");

        $schedules = FlightSchedule::with([
            "flights",
            "airlines",
            "airlines.airline",
        ])
            ->join("flights__schedule_airlines", "flights__schedule_airlines.flight_schedule_id", "=", "flights__schedule.id")
            ->where("flights__schedule_airlines.airline_id", $airline->id);

        if ($currentSchedule){
            $schedules/*->where("date_from", "<=", $today)*/
                        ->where("date_to", ">=", $today);
        }


        return $schedules->get(["flights__schedule.*"]);
    }

    /**
     * Get frequency numeric value
     * @param $frequencyString
     * @return int
     */
    private static function getFrequencyNumericValue($frequencyString)
    {
        // weekly by default
        $frequency = 1;
        if ($frequencyString == FREQ_ONCE_IN_2_WEEKS)
        {
            $frequency = 2;
        }
        else if ($frequencyString == FREQ_ONCE_IN_3_WEEKS)
        {
            $frequency = 3;
        }
        else if ($frequencyString == FREQ_ONCE_IN_4_WEEKS)
        {
            $frequency = 4;
        }
        return $frequency;
    }

    /**
     * Get stating week numeric value
     * @param $startingWeek
     * @return int
     */
    private static function getStartingWeekNumericValue($startingWeek)
    {
        // first week by default
        $startWeek = 1;
        if ($startingWeek == STARTING_WEEK_SECOND)
        {
            $startWeek = 2;
        }
        else if ($startingWeek == STARTING_WEEK_THIRD)
        {
            $startWeek = 3;
        }
        else if ($startingWeek == STARTING_WEEK_FOURTH)
        {
            $startWeek = 4;
        }
        return $startWeek;
    }

    /**
     * Check week number
     * @param $frequencyString
     * @param $startingWeek
     * @param $weekNumber
     * @return bool
     */
    public static function checkWeekNumber($frequencyString, $startingWeek, $weekNumber)
    {
        $frequency = self::getFrequencyNumericValue($frequencyString);
        $startWeek = self::getStartingWeekNumericValue($startingWeek);

        // if weekly and start week number less or equal to current week number
        if ($frequency == 1 && $startWeek <= $weekNumber)
        {
            return TRUE;
        }
        // if starting week is first week and current week number is 1
        if ($startWeek == 1 && $weekNumber == 1)
        {
            return TRUE;
        }
        if ($startWeek > $weekNumber)
        {
            return FALSE;
        }
        $allowedWeeks = array();
        for ($i = $startWeek; $i < 55; $i += $frequency)
        {
            $allowedWeeks[] = $i;
        }
        if (in_array($weekNumber, $allowedWeeks))
        {
            return TRUE;
        }
        return FALSE;
    }

    public static function getDaysOfOperation($string){
        $days = [];
        for( $i = 0; $i <= strlen($string); $i++ ) {
            $char = substr( $string, $i, 1 );
            if(is_numeric( $char ) ) {
                $days[] = intval($char);
            }
        }
        return $days;
    }

    public static function saveTemplate($templateName, $data, $from, $to){
        if (is_array($data))
        {
            $flightSchedule             = new FlightSchedule();
            $flightSchedule->name       = $templateName;
            $flightSchedule->date_from  = $from;
            $flightSchedule->date_to    = $to;
            $flightSchedule->created_at = date("Y-m-d H:i:s");
            $flightSchedule->created_by = Auth::user()->id;
            $flightSchedule->save();

            $insertData = [];
            $parents    = [];
            $children   = [];

            // Get Parent Flights
            foreach ($data as $each)
            {
                if (!$each->parent_id)
                {
                    $parents[] = $each;
                }
            }
            // Get Child Flights
            foreach ($data as $each)
            {
                if ($each->parent_id)
                {
                    $children[] = $each;
                }
            }
            foreach ($parents as $parent)
            {
                $flightScheduleFlight                     = new FlightScheduleFlight();
                $flightScheduleFlight->flight_schedule_id = $flightSchedule->id;
                $flightScheduleFlight->aircraft_id        = $parent->aircraft_id;
                $flightScheduleFlight->aircraft_type_id   = $parent->aircraft_type_id;
                $flightScheduleFlight->flight_number_id   = $parent->flight_number_id;
                $flightScheduleFlight->std_weekday        = $parent->std_weekday;
                $flightScheduleFlight->std_time           = $parent->std_time;
                $flightScheduleFlight->sta_weekday        = $parent->sta_weekday;
                $flightScheduleFlight->sta_time           = $parent->sta_time;
                $flightScheduleFlight->frequency          = $parent->frequency;
                $flightScheduleFlight->starting_week      = $parent->starting_week;
                $flightScheduleFlight->parent_id          = 0;
                $flightScheduleFlight->save();

                foreach ($children as $key => $child)
                {
                    if ($child->parent_id == $parent->unique_id)
                    {
                        $childScheduleFlight                     = new FlightScheduleFlight();
                        $childScheduleFlight->flight_schedule_id = $flightSchedule->id;
                        $childScheduleFlight->aircraft_id        = $parent->aircraft_id;
                        $childScheduleFlight->aircraft_type_id   = $parent->aircraft_type_id;
                        $childScheduleFlight->flight_number_id   = $child->flight_number_id;
                        $childScheduleFlight->std_weekday        = $child->std_weekday;
                        $childScheduleFlight->std_time           = $child->std_time;
                        $childScheduleFlight->sta_weekday        = $child->sta_weekday;
                        $childScheduleFlight->sta_time           = $child->sta_time;
                        $childScheduleFlight->frequency          = $child->frequency;
                        $childScheduleFlight->starting_week      = $child->starting_week;
                        $childScheduleFlight->parent_id          = $flightScheduleFlight->id;
                        $childScheduleFlight->save();
                        unset($children[$key]);
                    }
                }
            }
            if (is_array($children) && count($children))
            {
                foreach ($children as $key => $child)
                {
                    $childScheduleFlight                     = new FlightScheduleFlight();
                    $childScheduleFlight->flight_schedule_id = $flightSchedule->id;
                    $childScheduleFlight->aircraft_id        = $child->aircraft_id;
                    $childScheduleFlight->flight_number_id   = $child->flight_number_id;
                    $childScheduleFlight->std_weekday        = $child->std_weekday;
                    $childScheduleFlight->std_time           = $child->std_time;
                    $childScheduleFlight->sta_weekday        = $child->sta_weekday;
                    $childScheduleFlight->sta_time           = $child->sta_time;
                    $childScheduleFlight->frequency          = $child->frequency;
                    $childScheduleFlight->starting_week      = $child->starting_week;
                    $childScheduleFlight->parent_id          = 0;
                    $childScheduleFlight->save();
                }
            }

            if (count($insertData))
            {
                FlightScheduleFlight::insert($insertData);
            }
        }
    }

    public static function findFromAndToDates($data){
        $tomorrow = date("Y-m-d", strtotime("+1 days"));
        $minStartDate = $maxEndDate = null;
        foreach ($data as $each) {
            $startDate  = $each['start_date'];
            $endDate    = $each['end_date'];

            if (!$minStartDate){
                $minStartDate = $startDate;
            }
            else {
                if (strtotime($startDate) < strtotime($minStartDate)){
                    $minStartDate = $startDate;
                }
            }

            if (!$maxEndDate){
                $maxEndDate = $endDate;
            }
            else {
                if (strtotime($endDate) > strtotime($maxEndDate)){
                    $maxEndDate = $endDate;
                }
            }
        }

        return [
            'date_from'     => strtotime($minStartDate) < strtotime($tomorrow) ? $tomorrow : $minStartDate,
            'date_to'       => strtotime($maxEndDate) < strtotime($tomorrow) ? $tomorrow : $maxEndDate,
        ];
    }

    public static function getDateTime($day, $time){
        $today = date("Y-m-d");
        $todayDay = date('N', strtotime($today));

        if ($day > $todayDay){
            $std       = date("Y-m-d", strtotime("+" . ($day - $todayDay) . " days", strtotime($today)));
        }
        elseif ($day < $todayDay){
            $std       = date("Y-m-d", strtotime("-" . ($todayDay - $day) . " days", strtotime($today)));
        }
        else {
            $std       = $today;
        }

        return $std." ".$time;
    }

    public static function saveAirlineTemplate(Request $request, $data){
        $flightSchedule = null;

        if ($data && is_array($data))
        {
            $dateFromAndTo = self::findFromAndToDates($data);

            $flightSchedule             = new FlightSchedule();
            $flightSchedule->name       = $request->get("template_name");

            $flightSchedule->airport_ids = $request->get("airport") && count($request->get("airport")) ? json_encode($request->get("airline")) : null;
            $flightSchedule->airline_id = $request->get("airline");
            $flightSchedule->remove_other_flights = $request->get("remove_other_flights");

            $flightSchedule->date_from  = $dateFromAndTo['date_from'];
            $flightSchedule->date_to    = $dateFromAndTo['date_to'];
            $flightSchedule->created_at = date("Y-m-d H:i:s");
            $flightSchedule->created_by = Auth::user()->id;
            $flightSchedule->save();

            $insertData = $parents = $children = $airlines = [];

            $oneFlightOption = $flightSchedule->date_from == $flightSchedule->date_to;

            foreach ($data as $parent)
            {
                $flightScheduleFlight                           = new FlightScheduleFlight();
                //$flightScheduleFlight->parent_id          = 0;
                $flightScheduleFlight->flight_schedule_id       = $flightSchedule->id;
                $flightScheduleFlight->airline_id               = $parent['airline_id'];
                $flightScheduleFlight->operated_by_airline_id   = $parent['aircraft_airline_id'];

                $flightScheduleFlight->aircraft_id              = isset($parent['aircraft_id']) ? $parent['aircraft_id'] : null;
                $flightScheduleFlight->aircraft_type_id         = $parent['aircraft_type_id'];
                // CFG
                $flightScheduleFlight->cfg_c         = $parent['cfg_c'] ? $parent['cfg_c'] : null;
                $flightScheduleFlight->cfg_y         = $parent['cfg_y'] ? $parent['cfg_y'] : null;

                // One Flight
//                if ($oneFlightOption) {
                    $flightScheduleFlight->bkd_c = $parent['bkd_c'] ? $parent['bkd_c'] : null;
                    $flightScheduleFlight->bkd_y = $parent['bkd_y'] ? $parent['bkd_y'] : null;
//                }
                // Notes
                $flightScheduleFlight->notes = $parent['notes'] ? $parent['notes'] : null;

                $flightScheduleFlight->flight_number_id         = $parent['flight_number_id'];


                $flightScheduleFlight->effective_date     = $parent['start_date'];
                $flightScheduleFlight->discontinue_date   = $parent['end_date'];

                // Departure
                if ($parent["days"]){
                    $flightScheduleFlight->days           = implode("",$parent['days']);
                }

                $flightScheduleFlight->departure_time     = $parent['departure_time'];

                // Arrival
                $flightScheduleFlight->arrival_time       = $parent['arrival_time'];

                $flightScheduleFlight->frequency          = isset($parent['frequency']) ? $parent['frequency'] : "weekly";
                $flightScheduleFlight->starting_week      = isset($parent['starting_week']) ? $parent['starting_week'] : "first_week";
                $flightScheduleFlight->save();

                if ($parent['airline_id'] && !in_array($parent['airline_id'], $airlines)){
                    $airlines[] = $parent['airline_id'];
                    $flightScheduleAirline = new FlightScheduleAirline();
                    $flightScheduleAirline->flight_schedule_id = $flightSchedule->id;
                    $flightScheduleAirline->airline_id = $parent['airline_id'];
                    $flightScheduleAirline->save();
                }
            }

            if (count($insertData))
            {
                FlightScheduleFlight::insert($insertData);
            }
        }

       return $flightSchedule;

    }

    public static function uploadAirlineTemplateToFlightWatch($template, $airportIDs, $removeOtherFlights = false)
    {
        $authID = Auth::user()->id;
        if ($template)
        {
            $result = $flightIDs = [];

            $airlineID = null;

            foreach ($template->flights as $flight)
            {
                $fltIDs = self::createScheduledFlights($flight, $result);

                foreach ($fltIDs as $id){
                    $flightIDs[] = $id;
                }

                if (!$airlineID){
                    $airlineID = $flight->airline_id;
                }
            }

            if ($airlineID && $removeOtherFlights)
            {
                $dateFrom = strtotime($template->date_from) <= strtotime(date("Y-m-d"))
                    ? date("Y-m-d", strtotime("+1 days")) : $template->date_from;

                // IF >
                if (strtotime($template->date_to) > strtotime(date("Y-m-d"))) {

                    $dateTo = strtotime($template->date_to) <= strtotime(date("Y-m-d"))
                        ? date("Y-m-d", strtotime("+1 days")) : $template->date_to;

                    $flights = Flight::select(["flights.*"])
                        ->join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
                        ->whereNull('flights.deleted_at')
                        ->where("flights__numbers.airline_id", $airlineID)
                        ->where(function ($sql) use ($dateFrom, $dateTo) {
                            $sql->whereBetween('std', [$dateFrom . ' 00:00:00', $dateTo . " 23:59:59"])
                                ->orWhereBetween('ptd', [$dateFrom . ' 00:00:00', $dateTo . " 23:59:59"]);
                        });

                    if (count($flightIDs)){
                        $flights->whereNotIn('flights.id', $flightIDs);
                    }

                    // Template date-from and date-to

                    // Filter by Airports
                    if ($airportIDs && count($airportIDs)) {
                        $flights->where(function ($sql) use ($airportIDs) {
                            $sql->whereIn("flights__numbers.departure_airport_id", $airportIDs)
                                ->orWhereIn("flights__numbers.arrival_airport_id", $airportIDs)
                                ->orWhereIn("flights.departure_airport_id", $airportIDs)
                                ->orWhereIn("flights.arrival_airport_id", $airportIDs);
                        });
                    }

                    $flights = $flights->get();
                    $now = date("Y-m-d H:i:s");

                    if (count($flights)) {
                        debug("REMOVED FLIGHTS: " . count($flights));
                        foreach ($flights as &$flt) {
                            $flt->removed_by_schedule_id = $template->id;
                            $flt->deleted_at = $now;
                            $flt->updated_by = $authID;
                            $flt->save();
                        }
                    }
                }
            }

            debug($result);
        }
    }

    public static function createScheduledFlights($flight_schedule_flight, &$result){

        $flightIds = [];
        $days = [];
        $from = strtotime($flight_schedule_flight->effective_date);

        if ($flight_schedule_flight->discontinue_date){
            $to   = strtotime($flight_schedule_flight->discontinue_date);
            $oneDayOption = $from == $to;
        }
        else {
            $to = $from;
            $oneDayOption = true;
        }

        $dateNow = date("Y-m-d");
        $dateTimeNow = date("Y-m-d H:i:s");

        $startLoopWeekDay = date('N', $from);

        if ($flight_schedule_flight->days){
            $days = FlightSchedule::getDaysOfOperation($flight_schedule_flight->days);
        }
        elseif ($oneDayOption) {
            $days = [date('N', $from)];
        }

        $staAdd1Day = strtotime($flight_schedule_flight->arrival_time) < strtotime($flight_schedule_flight->departure_time) ? true : false;

//        debug("PERIOD: ".date("Y-m-d", $from)." - ".date("Y-m-d", $to));

        for ($i = $from; $i <= $to; $i = strtotime("+7 days", $i))
        {
            // $week++;

            foreach ($days as $day) {

                /*
                if (!self::checkWeekNumber($flight->frequency, $flight->starting_week, $week))
                {
                    continue;
                }
                */

                $stdWeekDay = ($day - $startLoopWeekDay + 7) % 7;

                $std       = date("Y-m-d", strtotime("+" . $stdWeekDay . " days", $i));

                // If Out of Period => Continue
                if (strtotime($std) < $from || strtotime($std) > $to){
                    debug($std."/". $flight_schedule_flight->effective_date."/".$stdWeekDay);
                    continue;
                }

//                debug("STD: ".$std);

                if ($staAdd1Day){
                    $sta   = date("Y-m-d", strtotime("+" . ($stdWeekDay + 1) . " days", $i));
                }
                else{
                    $sta   = date("Y-m-d", strtotime("+" . $stdWeekDay . " days", $i));
                }

                if (!isset($result[$flight_schedule_flight->flight_number_id])){
                    $result[$flight_schedule_flight->flight_number_id] = [];
                }

                // If flt No/Date combination exists => Continue
                if (in_array($std, $result[$flight_schedule_flight->flight_number_id]))
                {
//                    debug("If flt No/Date combination exists");
                    continue;
                }

                $result[$flight_schedule_flight->flight_number_id][] = $std;

                // Proceed

                // Search Flights only when its > today
                if (strtotime($std) <= strtotime($dateNow)) {
//                    debug("STD:{$std}. Search Flights only when its > today");
                    continue;
                }

                // Search Flight
                $searchFlight = Flight::where('flight_number_id', $flight_schedule_flight->flight_number_id)
                    /*->where("flight_schedule_flight_id", $flightScheduleFlight->id)*/
                    ->where(DB::raw('DATE(std)'), $std)
//                    ->whereNull("deleted_at")
//                    ->whereNull("cancelled_at")
                    ->orderBy("deleted_at")
                    ->orderBy("cancelled_at")
                    ->get();

                if ($foundFlights = $searchFlight->count()) {

                    foreach ($searchFlight as $j => &$flight) {

                        // Update only first record
                        self::updateFlight($flight, $flight_schedule_flight, $std, $sta, $dateTimeNow);

                        $flight->deleted_at     = null;
                        $flight->cancelled_at   = null;
                        $flight->save();

                        // Processed Flight Ids
                        $flightIds[] = $flight->id;

                        break;
                    }

                    continue;
                }

                // Create New
                $flight = self::createNewFlight($flight_schedule_flight, $std, $sta, $dateTimeNow);

                // Processed Flight Ids
                $flightIds[] = $flight->id;

            }
        }

        return $flightIds;
    }

    public static function createNewFlight($flight_schedule_flight, $std, $sta, $dateTimeNow){

        $flight = new Flight;
        $flight->created_by    = Auth::user()->id;
        $flight->created_at    = $dateTimeNow;

        self::updateFlight($flight, $flight_schedule_flight, $std, $sta, $dateTimeNow, false);

        $flight->save();

        return $flight;
    }

    public static function updateFlight(&$flight, $flight_schedule_flight, $std, $sta, $dateTimeNow, $update = true){
        $flight->flight_schedule_id          = $flight_schedule_flight->flight_schedule_id;
        $flight->flight_schedule_flight_id   = $flight_schedule_flight->id;

        $flight->aircraft_type_id   = $flight_schedule_flight->aircraft_type_id;
        $flight->aircraft_id        = $flight_schedule_flight->aircraft_id;
        // CFG
        $flight->capacity_c        = $flight_schedule_flight->cfg_c;
        $flight->capacity_y        = $flight_schedule_flight->cfg_y;

        $flight->flight_number_id   = $flight_schedule_flight->flight_number_id;

        $flight->std = $flight->ptd = $std . " " . $flight_schedule_flight->departure_time;
        $flight->departure_date = $std;

        $flight->sta = $flight->pta = $sta . " " . $flight_schedule_flight->arrival_time;
        $flight->arrival_date = $sta;

        // BKD
        $flight->pax_c_booked = $flight_schedule_flight->bkd_c;
        $flight->pax_y_booked = $flight_schedule_flight->bkd_y;
        $flight->remark = $flight_schedule_flight->notes;

        if ($update){
            $flight->updated_by = Auth::user()->id;
            $flight->updated_at = $dateTimeNow;

            // Update Flight service Data
            Flight::updateFlightServiceData($flight);
        }

        return $flight;
    }


}
