<?php namespace App\Models; use app\Classes\ProjectConstant;
use Illuminate\Database\Eloquent\Model as Eloquent;

/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 27.08.14

 */

use Illuminate\Support\Facades\DB;

use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class FlightNumber extends Eloquent implements AuditableContract
{
    use Auditable;

    protected $auditExclude = [
        'updated_by',
        'created_by',
    ];

    protected $table = "flights__numbers";

    protected $guarded = [
        "created_at",
        "updated_at",
        "id"
    ];

    protected $fillable = [
        'airline_id',
        'flight_number',
        'departure_airport_id',
        'arrival_airport_id',
    ];

    public $timestamps = TRUE;

    public function flight()
    {
        return $this->hasMany("App\\Models\\Flight", "flight_number_id");
    }

    public function passengerFlight()
    {
        return $this->hasMany("App\\Models\\PassengerFlight", "flight_number_id");
    }

    public function capacity()
    {
        return $this->hasMany("App\\Models\\FlightNumberAircraftCapacity", "flight_number_id");
    }

    public function airline(){
        return $this->belongsTo('App\Models\Airline', 'airline_id');
    }

    public function departureAirport(){
        return $this->belongsTo('App\Models\Airport', 'departure_airport_id');
    }

    public function arrivalAirport(){
        return $this->belongsTo('App\Models\Airport', 'arrival_airport_id');
    }

    public function flightType(){
        return $this->belongsTo('App\Models\FlightType', 'flight_type_id');
    }

    public function getFlightSectorAttribute()
    {
        return $this->attributes["departure_airport"] . "-" . $this->attributes["arrival_airport"];
    }

    public function getFlightNumberIataAttribute()
    {
        return ProjectConstant::getConstants(IATA_CODE) . " " . $this->attributes["flight_number"];
    }



    public static function getFlightNumbers(){
        $data = FlightNumber::select('flight_number')
                        ->whereRaw('flight_number != 0 AND flight_number != ""')
                        ->whereNull("deleted_at")
                        ->orderBy('flight_number')
                        ->get();
        return $data;
    }

    // 'Scheduled','Charter','VIP Charter','Hajj','Positioning','Tajik Air','Other Airlines','Historic'

    public static function getArraySectors($type = null, $typeId = null){
        $sector = FlightNumber::selectRaw('CONCAT(departure_airport, "-", arrival_airport) AS sect')
                             ->join('flights__types', 'flights__types.id', '=', 'flights__numbers.flight_type_id');
        if (isset($type)) {
            $type = is_array($type) ? $type : [$type];
            $sector->whereIn('type' , $type);
        }
        elseif (isset($typeId)) {
            $typeId = is_array($typeId) ? $typeId : [$typeId];
            $sector->whereIn('type_id' , $typeId);
        }

        $sector = $sector
            ->whereNull("flights__numbers.deleted_at")
            ->orderBy('flight_number')->pluck('sect', 'sect')->all();
        return $sector;
    }

    public static function arrayFlightNumbersRoutesByRegion($region = null, $type =null, $bound = null, $different_types = null, $combineValue = null)
    {
        $data = FlightNumber::selectRaw('flight_number, CONCAT(s_depAirport.iata,"-", s_arrAirport.iata) as route,
            CONCAT(flight_number," | ", s_depAirport.iata,"-", s_arrAirport.iata) as combined')
            ->join('flights__types', 'flights__numbers.flight_type_id', '=', 'flights__types.id')
            ->join('airports as depAirport', 'flights__numbers.departure_airport_id', '=', 'depAirport.id')
            ->join('airports as arrAirport', 'flights__numbers.arrival_airport_id', '=', 'arrAirport.id');

        if (isset($region)) {
            $data->join('countries', 'depAirport.country_id', '=', 'countries.id')
                 ->join('regions', 'countries.region_id', '=', 'regions.id')
                 ->where('region', $region);

            if ($region == 'Domestic') {
                $data->whereIn('depAirport.iata', ['DYU', 'LBD'])
                    ->whereIn('arrAirport.iata', ['DYU', 'LBD']);

            }
        }

        if (isset($bound))
            $data->where('bound', $bound);

        if (isset($type)) {
            if ($type == 'Scheduled')
                $data->where('type', '0');
            elseif ($type == 'Charter')
                $data->where('type', '1');
        }

        if (isset($different_types))
        {
            if ($different_types == 'Maintenance')
                $data->whereRaw('flight_number like "9%" and char_length(flight_number) = 4');
            elseif ($different_types == 'Tajik Air')
                $data->whereIn('flight_number', [123, 124, 645, 646, 627, 628]);
            elseif($different_types == 'VIP'){
                $data = FlightNumber::selectRaw('flight_number, "LITER" as route, CONCAT(flight_number," | LITER") as combined')
                    ->where('flight_number', '001');
            }
        }

        $data->whereNull("flights__numbers.deleted_at");


        if (isset($combineValue))
            return $data->orderBy('flight_number')->pluck('combined', 'flight_number')->all();
        else
            return $data->orderBy('flight_number')->pluck('route', 'flight_number')->all();


    }

    /**
     * List Flight Numbers
     * @param null $type
     * @param null $typeId
     * @return $this|array
     */
    public static function getArrayFlightNumbers($type = null, $typeId = null){
        $sector = FlightNumber::select(["flight_number", "flights__numbers.id as id"])
                        ->join('flights__types', 'flights__types.id', '=', 'flights__numbers.flight_type_id')
                        ->whereNull("flights__numbers.deleted_at");

        if (isset($type)) {
            $type = is_array($type) ? $type : [$type];
            $sector->whereIn('type' , $type);
        }
        elseif (isset($typeId)) {
            $typeId = is_array($typeId) ? $typeId : [$typeId];
            $sector->whereIn('type_id' , $typeId);
        }

        $sector = $sector->orderBy('flight_number')->pluck("flight_number", "id")->all();
        return $sector;
    }

    public static function getFlightNumbersByAirlines($airlines = null){
        $flightNumbers = FlightNumber::select([
            "flight_number",
            "flights__numbers.id as id",
            DB::raw('CONCAT(s_airlines.iata,"-",flight_number," (",s_depAirport.iata,"-",s_arrAirport.iata, ")") as sector')
        ])
            ->join('airlines', 'airlines.id', '=', 'flights__numbers.airline_id')
            ->leftJoin('airports AS depAirport', 'depAirport.id', '=', 'flights__numbers.departure_airport_id')
            ->leftJoin('airports AS arrAirport', 'arrAirport.id', '=', 'flights__numbers.arrival_airport_id')
            ->whereNull("flights__numbers.deleted_at")
            ->whereNotNull("depAirport.iata")
            ->whereNotNull("arrAirport.iata");

        if ($airlines){
            $airlines = is_array($airlines) ? $airlines : [$airlines];
            $flightNumbers->whereIn("flights__numbers.airline_id", $airlines);
        }

        return $flightNumbers->orderBy(DB::raw("LENGTH(flight_number)"))
                             ->orderBy('flight_number')
                             ->pluck("sector", "id")
                             ->all();
    }

    /**
     * List Flight Numbers With Sector
     * @param bool|false $type
     * @param bool|false $typeId
     * @param bool|false $bound
     * @param null $airlines
     * @param bool|false $object
     * @return mixed
     */
    public static function getWithSector($id){
        $flightNumbers = FlightNumber::leftJoin('airports AS depAirport', 'depAirport.id',  '=', 'flights__numbers.departure_airport_id')
                                       ->leftJoin('airports AS arrAirport', 'arrAirport.id', '=', 'flights__numbers.arrival_airport_id')
                                       ->leftJoin('flights__types', 'flights__types.id', '=', 'flights__numbers.flight_type_id')
                                        ->where("flights__numbers.id", $id)
                                        ->first([
                                            "flight_number",
                                            "airline_id",
                                            "flights__numbers.id as id",
                                            DB::raw('CONCAT(flight_number," ",IF(s_depAirport.iata IS NULL, "", s_depAirport.iata),"-",IF(s_arrAirport.iata IS NULL, "", s_arrAirport.iata)) as sector')
                                        ]);

        return $flightNumbers;
    }

    public static function getArrayFlightNumbersWithSector($type = false, $typeId = false, $bound = false, $airlines = null, $object = false){
        $flightNumbers = FlightNumber::leftJoin('airports AS depAirport', 'depAirport.id',  '=', 'flights__numbers.departure_airport_id')
            ->leftJoin('airports AS arrAirport', 'arrAirport.id', '=', 'flights__numbers.arrival_airport_id');

//        if ($onlyFromPassengerFlights){
//            $flightNumbers->join('passengers__flights', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id');
//        }

        $flightNumbers->leftJoin('flights__types', 'flights__types.id', '=', 'flights__numbers.flight_type_id');

        if ($type) {
            $type = is_array($type) ? $type : [$type];
            $flightNumbers->whereIn('type' , $type);
        }
        elseif ($typeId) {
            $typeId = is_array($typeId) ? $typeId : [$typeId];
            $flightNumbers->whereIn('type_id' , $typeId);
        }

        if ($bound){
            $flightNumbers->where('bound', ($bound == 'outbound') ? 0 : 1);
        }

        if ($airlines){
            $airlines = is_array($airlines) ? $airlines : [$airlines];
            $flightNumbers->whereIn("flights__numbers.airline_id", $airlines);
        }

        $flightNumbers->whereNull("flights__numbers.deleted_at")
            ->where(function($sql){
                $sql->whereNotNull("depAirport.iata")
                    ->orWhereNotNull("arrAirport.iata");
            })
//                    ->orderBy(DB::raw("LENGTH(flight_number)"))
            ->orderBy('flight_number');

        $flightNumbers = $flightNumbers->get([
            "flight_number",
            "airline_id",
            "flights__numbers.id as id",
            DB::raw('CONCAT(flight_number," ",IF(s_depAirport.iata IS NULL, "", s_depAirport.iata),"-",IF(s_arrAirport.iata IS NULL, "", s_arrAirport.iata)) as sector')
        ]);

        if($object){
            return $flightNumbers;
        }
        else {
            return $flightNumbers->pluck("sector", "id")
                ->all();
        }
    }


    public static function getArrayFlightNumbersOutbounds(){
        $data = FlightNumber::selectRaw('CONCAT(flight_number," ",departure_airport,"-",arrival_airport) as route, flight_number')
                            ->whereNull("deleted_at")
                            ->where('bound',0)
                            ->orderBy('flight_number')
                            ->pluck("route", "flight_number")->all();
        return $data;
    }


    /**
     * List Flight Numbers Based on Flight Type, Airport and Choose Array Key
     * @param $flightType
     * @param bool $airport
     * @param string $keyId
     * @return array|bool
     */
     public static function listFlightNumbersByFlightTypeAndAirport($flightType, $airport = false, $bound = false, $keyId = 'id'){
        if ($flightType) {
            $flightType = is_array($flightType) ? $flightType : [$flightType];
            $sectors = FlightNumber::select(['flights__numbers.id as id', 'flight_number'])
                ->join('flights__types', 'flights__types.id', '=', 'flights__numbers.flight_type_id')
                ->whereNull("flights__numbers.deleted_at")
                ->whereIn('type', $flightType);

            if ($airport) {
                $airport = is_array($airport) ? $airport : [$airport];
                $sectors->whereIn('departure_airport', $airport);
            }

            if ($bound){
                $sectors->where('bound', ($bound || $bound =='outbound') ? 1 : 0);
            }

            return $sectors->groupBy('flight_number')
                            ->pluck('flight_number', $keyId)->all();
        }
        return false;
    }

    /**
     * List Distinct Airports By Flight Type : [ie Scheduled, CHarter, VIP Charter, ...]
     * @param $flightType
     * @param string $keyId
     * @return array|bool
     */
    public static function listAirportsByFlightType($flightType, $keyId = 'id'){
        if ($flightType) {
            $flightType = is_array($flightType) ? $flightType : [$flightType];
            $airports = FlightNumber::join('flights__types', 'flights__types.id', '=', 'flights__numbers.flight_type_id')
                ->whereNull("flights__numbers.deleted_at")
                ->whereIn('type', $flightType)
                ->groupBy('departure_airport')
                ->pluck('departure_airport', $keyId)->all();

            return $airports;
        }
        return false;
    }


    public static function createManualFlightNumber($bound = null){
        $manualFlightNo = \request("flt_no");
        $departureAirport = \request("departure_airport");
        $arrivalAirport = \request("arrival_airport");
        $flightNo = FlightNumber::join('airports as depAirport', 'flights__numbers.departure_airport_id', '=', 'depAirport.id')
                                ->join('airports as arrAirport', 'flights__numbers.arrival_airport_id', '=', 'arrAirport.id')
                                ->whereNull("flights__numbers.deleted_at")
                                ->where('flight_number', $manualFlightNo)
                                ->where('depAirport.iata',$departureAirport)
                                ->where('arrAirport.iata', $arrivalAirport);


        $airline = null;
        if ($operatorName = \request("operator_name")) {
            $findOperator = Airline::where("operator", 1)
                                ->where(function ($sql) use($operatorName){
                                    $sql->where("airline", $operatorName)
                                        ->orWhere("iata", $operatorName)
                                        ->orWhere("icao", $operatorName)
                                        ->orWhere("callsign", $operatorName);
                                })
                                ->get();

            if ($findOperator->count() == 1){
                $airline = $findOperator->first();
            }
            else {
                $airline = new Airline();
                $airline->operator = 1;
                $airline->airline = $operatorName;
                $airline->save();
            }
        }
        else {
            if (\request("airline_id")) {
                $airline = Airline::find(\request("airline_id"));
            }
            else {
                if (env(CARRIER) == AIRLINE || !env(MULTIPLE_AIRLINE)) {
                    $airline = Airline::where("handling_airline", true)
                        ->first();
                }
            }
        }

        if ($airline){
            $flightNo->where("airline_id", $airline->id);
        }

        $flightNo = $flightNo->first(['flights__numbers.*']);

        if (!$flightNo) {
            $depAirport = Airport::where("iata", $departureAirport)->first();
            if (!$depAirport){
                $depAirport = Airport::findOrCreateByIATA($departureAirport);
            }

            $arrAirport = Airport::where("iata", $arrivalAirport)->first();
            if (!$arrAirport){
                $arrAirport = Airport::findOrCreateByIATA($arrivalAirport);
            }

            $flightNo = new FlightNumber;
            $flightNo->flight_number = ltrim($manualFlightNo, "0");
            $flightNo->bound = $bound;
            $flightNo->airline_id = $airline ? $airline->id : null;
            $flightNo->departure_airport_id = $depAirport ? $depAirport->id : null;
            $flightNo->arrival_airport_id = $arrAirport ? $arrAirport->id : null;
            $flightNo->flight_type_id = \request("flight_type") ? \request("flight_type") : null;
            $flightNo->save();
        }
        // Get New Flight Number ID
        return $flightNo;
    }

}
