<?php namespace App\Models;

use Illuminate\Database\Eloquent\Model as Eloquent;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 01.09.14

 */
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class FlightCrew extends Eloquent implements AuditableContract
{
    use Auditable;

    protected $auditExclude = [
        'updated_by',
        'created_by',
    ];

    protected $table = "flights__crew";

    public $timestamps = true;

    // Mass Assignment
    protected $fillable = [
        'flight_id',
        'diverted_flight_id',
        'standby_crew_id',
        'user_id',
        'position_id',
        'position_order',
        'is_standby',
        'is_dhc',
        'is_sup',
        'crew_sup_id'
    ];

    public function flight()
    {
        return $this->belongsTo('App\Models\Flight', "flight_id");
    }

    public function position()
    {
        return $this->belongsTo('App\Models\Position', "position_id");
    }

    public function user()
    {
        return $this->belongsTo('App\Models\User', "user_id");
    }

    /**
     * Delete Standby Crew By Standby Id
     * @param $standbyId
     * @throws Exception
     */
    public static function deleteStandbyCrew($standbyId)
    {
        // For Edit Purposes
        if ($standbyId) {
            // Delete Previously Set Standby Crew
            $fCrew = FlightCrew::where('flights_standby_crew_id', $standbyId)
                                ->get();
            // Audit Deleted
            auditDeleted($fCrew);

        }
    }


    public static function deleteAllFlightsCrew($flightId)
    {
        // For Edit Purposes
        if ($flightId) {
            // Delete Previously Set Standby Crew
            $fCrew = FlightCrew::where('flight_id', $flightId)
                                ->get();
            // Audit Deleted
            auditDeleted($fCrew);

        }
    }

    /**
     * @param $standbyCrew
     */
    public static function saveStandbyCrew($standbyCrew)
    {
        $flights = Flight::with(['flightNumber'])
        ->whereBetween("std", [
            $standbyCrew->start_date,
            $standbyCrew->end_date
        ])
        ->orWhereBetween("ptd", [
            $standbyCrew->start_date,
            $standbyCrew->end_date
        ])
            ->get();

        if ($flights->count() && $standbyCrew->airport_id) {
            foreach ($flights as $flight) {

                $flightNumber = $flight->flightNumber;

                if ($flightNumber && $flightNumber->departure_airport_id == $standbyCrew->airport_id)
                 {
                    $flightCrew = FlightCrew::firstOrNew([
                        'flight_id'                 => $flight->id,
                        'flights_standby_crew_id'   => $standbyCrew->id,
                        'user_id'                   => $standbyCrew->user_id,
                        'position_id'               => $standbyCrew->position_id,
                        'position_order'            => 0,
                        'is_standby'                => 1,
                        'is_dhc'                    => 0
                    ]);

                    $flightCrew->flights_standby_crew_id = $standbyCrew->id;
                    $flightCrew->save();
                }
            }
        }
    }

    /**
     * Create flight crew item
     * @param $userId
     * @param $flightId
     * @param null $divertedFlightId
     * @param bool|FALSE $fields
     * @param bool|FALSE $isStandby
     * @param bool|FALSE $isDHC
     * @param bool|FALSE $isSup
     * @return static
     */
    public static function createCrewItem($userId, $flightId, $divertedFlightId = NULL, $fields = FALSE, $isStandby = FALSE, $isDHC = FALSE, $isSup = FALSE)
    {
        $user = User::find($userId);
        $attributes['flight_id'] = $flightId;
        $attributes['diverted_flight_id'] = $divertedFlightId;
        $attributes['user_id'] = $userId;

        $attributes['position_id'] = $user->position && count($user->position) ? $user->position[0]->id : null;

        if ($fields && count($fields)) {
            foreach ($fields as $name => $value) {
                $attributes[$name] = $value;
            }
        }
        $attributes['is_standby'] = $isStandby;
        $attributes['is_dhc'] = $isDHC;
        $attributes['is_sup'] = $isSup;
        $attributes['created_by'] = Auth::user()->id;

        // Check If Such Crew Exists
        $flightCrew = FlightCrew::firstOrNew($attributes);
        $flightCrew->save();

        return $flightCrew;
    }

    /**
     * @param $flightID
     * @param bool|FALSE $isStandby
     * @param bool|FALSE $isDHC
     * @return bool
     */
    public static function removeCrew($flightID, $isStandby = FALSE, $isDHC = FALSE)
    {
        $flightID = (int)$flightID;
        if (!$flightID) {
            return FALSE;
        }

        $flightCrew = FlightCrew::where("flight_id", $flightID);

        // Standby
        if ($isStandby){
            $flightCrew->where("is_standby", $isStandby);
        }

        // DHC
        if ($isDHC){
            $flightCrew->where("is_dhc", $isDHC);
        }

        // Audit Deleted
        $flightCrew = $flightCrew->get();

        auditDeleted($flightCrew);
//        $flightCrew->delete();

        return TRUE;
    }

    public static function removeDuplicateCrew($flightId){
        // Remove Standby Duplicates
        $standbys = FlightCrew::with(["position"])
            ->join("flights", "flights.id", "=", "flights__crew.flight_id")
            ->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at")
            ->where("is_standby", true)
            ->where("is_dhc", false)
            ->where('flight_id', $flightId)
            ->orderBy('position_order')
            ->get(['flights__crew.*']);

        $standbyUsers = $deletedIds = [];
        $positionOrderCAPT = $positionOrderFO = $positionOrderCC = 0;
        foreach ($standbys as $j => $standby) {
            if (in_array($standby->user_id, $standbyUsers)){
                $deletedIds[] = $standby->id;
                continue;
            }
            $standbyUsers[] = $standby->user_id;
            $position = $standby->position;


            if ($position->type == FCM_CPT_TYPE_ID){
                $standbys[$j]->position_order = $positionOrderCAPT++;
            }
            else if ($position->type == FCM_FA_TYPE_ID) {
                $standbys[$j]->position_order = $positionOrderFO++;
            }
            else {
                $standbys[$j]->position_order = $positionOrderCC++;
            }
            $standbys[$j]->save();
        }

        // Remove DHC Duplicates
        $dhcs = FlightCrew::with(["position"])
                           ->join("flights", "flights.id", "=", "flights__crew.flight_id")
                           ->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at")
                           ->where("is_standby", false)
                           ->where("is_dhc", true)
                           ->where('flight_id', $flightId)
                           ->orderBy('position_order')
                           ->get(['flights__crew.*']);

        $dhcUsers = $deletedIds = [];
        $positionOrderCAPT = $positionOrderFO = $positionOrderCC = 0;
        foreach ($dhcs as $j => $dhc) {
            if (in_array($dhc->user_id, $dhcUsers)){
                $deletedIds[] = $dhc->id;
                continue;
            }
            $dhcUsers[] = $dhc->user_id;
            $positionType = $dhc->position ? $dhc->position->type : null;

            if ($positionType == FCM_CPT_TYPE_ID){
                $dhcs[$j]->position_order = $positionOrderCAPT++;
            }
            else if ($positionType == FCM_FA_TYPE_ID) {
                $dhcs[$j]->position_order = $positionOrderFO++;
            }
            else {
                $dhcs[$j]->position_order = $positionOrderCC++;
            }
            $dhcs[$j]->save();
        }


        // Remove Operating Duplicates
        $operatings = FlightCrew::join("flights", "flights.id", "=", "flights__crew.flight_id")
            ->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at")
            ->where("is_standby", false)
            ->where("is_dhc", false)
            ->where('flight_id', $flightId)
            ->orderBy('position_order')
            ->get(['flights__crew.*']);

        $operatingUsers = $deletedIds = [];
        $positionOrderCAPT = $positionOrderFO = $positionOrderCC = 0;
        foreach ($operatings as $j => $operating) {
            if (in_array($operating->user_id, $operatingUsers)){
                $deletedIds[] = $operating->id;
                continue;
            }
            $operatingUsers[] = $operating->user_id;
            /* POSITION ORDER => NOT ALLOWED TO CHANGE
            if ($operating->position_id == USER_CAPTAIN_POSITION_ID){
                $operatings[$j]->position_order = $positionOrderCAPT++;
            }
            else if ($operating->position_id == USER_FIRST_OFFICER_POSITION_ID) {
                $operatings[$j]->position_order = $positionOrderFO++;
            }
            else {
                $operatings[$j]->position_order = $positionOrderCC++;
            }
            $operatings[$j]->save();
            */
        }

        // Remove Duplicates
        if (count($deletedIds)) {
            $fCrew = FlightCrew::whereIn('id', $deletedIds)
                                ->get();
            // Audit Deleted
            auditDeleted($fCrew);

        }
    }

    public static function removeAllCrewDuplicates(){
        $ytd = date("Y-m-01");

        // Remove Standby Duplicates
        $crew = FlightCrew::join("flights", "flights.id", "=", "flights__crew.flight_id")
            ->whereRaw("DATE(std) > '{$ytd}'")
            ->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at")
            ->orderBy('flight_id')
            ->orderBy('user_id')
            ->orderBy('is_standby')
            ->orderBy('is_dhc')
            ->orderBy('position_order')
            ->get(['flights__crew.*']);

        $crewFlight = $deletedIds = [];
        foreach ($crew as $i => $each) {
            if (!isset($crewFlight[$each->flight_id][$each->user_id])){
                $crewFlight[$each->flight_id][$each->user_id] = 1;
                continue;
            }
            $deletedIds[] = $each->id;
        }

        if (count($deletedIds)) {
            $fCrew = FlightCrew::whereIn('id', $deletedIds)
                                ->get();
            // Audit Deleted
            auditDeleted($fCrew);

        }
    }

    public static function removeAllCrewDuplicatesByUser($userId){
        // Remove Standby Duplicates
        $crew = FlightCrew::join("flights", "flights.id", "=", "flights__crew.flight_id")
            ->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at")
            ->where('user_id', $userId)
            ->orderBy('flight_id')
            ->orderBy('is_standby')
            ->orderBy('is_dhc')
            ->orderBy('position_order')
            ->get(['flights__crew.*']);

        $crewFlight = $deletedIds = [];
        foreach ($crew as $i => $each) {
            if (!isset($crewFlight[$each->flight_id])){
                $crewFlight[$each->flight_id] = 1;
                continue;
            }
            $deletedIds[] = $each->id;
        }

        if (count($deletedIds)) {
            $fCrew = FlightCrew::whereIn('id', $deletedIds)
                                ->get();
            // Audit Deleted
            auditDeleted($fCrew);
        }
    }

    /**
     * Get flight crew full information (Option: By Type)
     * @param $flightId
     * @return \Illuminate\Database\Eloquent\Model|null|static
     */
    public static function flightCrewFullInformation($flightId, $crewType = false)
    {
        $flightCrew = FlightCrew::select([
            "flights__crew.id",
            "flights__crew.flight_id",
            "flights__crew.user_id",
            "flights__crew.position_id",
            "flights__crew.position_order",
            "flights__crew.is_standby",
            "flights__crew.is_dhc",
            "users.first_name",
            "users.last_name",
            "users.email",
            "structure__positions.name AS position_title",
            "structure__positions.type AS position_type"
        ])
            ->join("users", "users.id", "=", "flights__crew.user_id")
            ->join("structure__positions", "structure__positions.id", "=", "flights__crew.position_id")
            ->where("flight_id", $flightId);

        // If Type Is Set Filter By Type( fcm, ccm)
        if ($crewType) {
            switch ($crewType) {
                case FCM_CREW:
                    $flightCrew->whereIn('structure__positions.type', [FCM_CPT_TYPE_ID, FCM_FA_TYPE_ID]);
                    break;

                case CCM_CREW:
                    $flightCrew->whereIn('structure__positions.type', [CCM_PSR_TYPE_ID, CCM_CC_TYPE_ID]);
                    break;
            }
        }

        return $flightCrew->get();
    }

    /**
     * @param $usersIds
     * @param null $startDate
     * @param null $endDate
     * @return $this|array|static[]
     */
    public static function flightCrewUsersFlightsHours($usersIds, $startDate = NULL, $endDate = NULL)
    {
        $flights = Flight::select([
            "flights.id",
            "flights.std",
            "flights.sta",
            "flights.atd",
            "flights.ata",
            "flights__crew.user_id"
        ])
            ->join("flights__crew", "flights.id", "=", "flights__crew.flight_id")
            ->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at")
            ->whereIn("flights__crew.user_id", $usersIds);

        if ($startDate && $endDate) {
            $flights->whereRaw("((s_flights.std BETWEEN \"$startDate\" AND \"$endDate\") OR
                                 (s_flights.sta BETWEEN \"$startDate\" AND \"$endDate\"))");
        }
        $users = array_fill_keys($usersIds, 0);
        $flights = $flights->orderBy("flights__crew.user_id")
            ->get();
        if (count($flights)) {
            foreach ($usersIds as $keyUser => $userId) {
                foreach ($flights as $key => $flight) {
                    if ($flight->user_id == $userId) {
                        if (isset($users[$userId])) {
                            $users[$userId] += getFlightFlySeconds($flight);
                        } else {
                            $users[$userId] = getFlightFlySeconds($flight);
                        }
                    }
                }
            }
        }
        return $users;
    }

    /**
     * Get Crew Flights in Date Range (All Crew or Specific), filter (DHC and Standby)
     * @param $fromDate
     * @param $toDate
     * @param bool|false $dateSearch
     * @param bool|false $userIds
     * @param string $sortDate
     * @param bool|false $noDHC
     * @param bool|false $noStandby
     * @param bool|false $noSUP
     * @return mixed
     */
    public static function getCrewFlights($fromDate, $toDate, $dateSearch = false, $userIds = false, $sortDate = 'ASC', $noDHC = false, $noStandby = false, $noSUP = false)
    {

        $crewFlights = FlightCrew::with([
            'flight',
            'flight.flightNumber',
            'flight.flightNumber.departureAirport',
            'flight.flightNumber.arrivalAirport',
        ])
            ->join('flights','flights__crew.flight_id', '=', 'flights.id')
//            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->whereNull('flights.cancelled_at')
            ->whereNull('flights.deleted_at');

        if ($dateSearch && isset($dateSearch['month']) && isset($dateSearch['year'])) {
            $month = is_array($dateSearch['month']) ? $dateSearch['month'] : [ $dateSearch ['month'] ];
            $year = is_array($dateSearch['year']) ? $dateSearch['year'] : [ $dateSearch ['year'] ];

            $crewFlights->whereIn(DB::raw("MONTH(std)"), $month)
                         ->whereIn(DB::raw("YEAR(std)"), $year);

        }
        else {
            $crewFlights->whereRaw("DATE(std) BETWEEN '$fromDate' AND '$toDate' ");
        }


        if ($userIds) {
            $userIds = !is_array($userIds) ? [$userIds] : $userIds;
            $crewFlights->whereIn('user_id', $userIds);
        }

        if ($noDHC){
            $crewFlights->where('is_dhc', 0);
        }

        if ($noStandby){
            $crewFlights->where('is_standby', 0);
        }

        if ($noSUP){
            $crewFlights->where('is_sup', 0);
        }

        return $crewFlights->orderBy('user_id')
                           ->orderBy('std', $sortDate)
                           ->get([
                               'flights__crew.*',
//                               'flight_id as id',
//                               'user_id',
//                               'is_standby',
//                               'is_dhc',
                           ]);
    }


    public static function getCrewStandbyOrDhcFlights($dateSearch = false, $isDhc = false, $userIds = false)
    {
        $crewFlights = FlightCrew::select(['flights.id', 'user_id', 'is_standby', 'is_dhc', 'flight_number', 'departure_airport', 'arrival_airport', 'std', 'atd', 'sta', 'ata'])
            ->join('flights','flights__crew.flight_id', '=', 'flights.id')
            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->whereNull('flights.cancelled_at')
            ->whereNull('flights.deleted_at');

        if (isset($dateSearch['month']) && isset($dateSearch['year'])) {
            $month = is_array($dateSearch['month']) ? $dateSearch['month'] : [ $dateSearch ['month'] ];
            $year = is_array($dateSearch['year']) ? $dateSearch['year'] : [ $dateSearch ['year'] ];

            $crewFlights->whereIn(DB::raw("MONTH(std)"), $month)
                        ->whereIn(DB::raw("YEAR(std)"), $year);

        }
        elseif (isset($dateSearch['dateFrom']) && isset($dateSearch['dateTo'])) {
            $crewFlights->whereRaw("DATE(std) BETWEEN '".$dateSearch['dateFrom']."' AND '".$dateSearch['dateTo']."' ");
        }

        if ($isDhc)
            $crewFlights->where('is_dhc', 1);
        else{
            $crewFlights->where('is_standby', 1);
        }

        if ($userIds) {
            $crewFlights->whereIn('user_id', (!is_array($userIds) ? [$userIds] : $userIds));
        }

        return $crewFlights->orderBy('user_id')->orderBy('std')->get();
    }


    /**
     * Crew Change/Update Procedure
     * @param $flightId
     * @param null $divertedFlightId
     * @param $crewType
     * @param $crewIds
     * @param $crewReportTimes
     * @param $crewReasons
     * @param bool|FALSE $is_standby
     * @param bool|FALSE $is_dhc
     * @param bool|FALSE $is_sup
     * @param null $supIds
     * @return bool
     */
    public static function createChangeCrewProcedure($flightId, $divertedFlightId = NULL, $crewType, $crewIds, $crewReportTimes, $crewReasons, $is_standby = FALSE, $is_dhc = FALSE, $is_sup = FALSE, $supIds = NULL)
    {
        if (!$flightId && !$divertedFlightId)
        {
            return FALSE;
        }

        // Get Previously Saved Crew
        $previousCrew = FlightCrew::leftJoin("structure__positions", "structure__positions.id", "=", "flights__crew.position_id")
            ->where("is_standby", $is_standby)
            ->where("is_dhc", $is_dhc)
            ->where("is_sup", $is_sup)
            ->where("flight_id", $flightId)
            ->where("diverted_flight_id", $divertedFlightId);

        if ($crewType == CREW_CAPTAINS){

            $previousCrew->where("structure__positions.type", FCM_CPT_TYPE_ID);
        }
        else if ($crewType == CREW_FIRST_OFFICERS)
        {
            $previousCrew->where("structure__positions.type", FCM_FA_TYPE_ID);
        }
        if ($crewType == CREW_PURSER_AND_CABIN_CREW)
        {
            $previousCrew->whereIn("structure__positions.type", [CCM_PSR_TYPE_ID, CCM_CC_TYPE_ID]);
        }
        // Array of Inputted Crew Ids which is dealt with
        $processedCrewIds = [];

        $previousCrew = $previousCrew->get([
            "flights__crew.*"
        ]);

        debug($previousCrew);

        if ($previousCrew && count($previousCrew)) {
            foreach ($previousCrew as $i => $each) {

                $positionOrder = $each->position_order;
                // Report Time
                $reportTime = isset($crewReportTimes[$positionOrder]) && $crewReportTimes[$positionOrder] && strlen($crewReportTimes[$positionOrder]) > 0 ? $crewReportTimes[$positionOrder] : null;

                $supId = isset($supIds[$positionOrder]) && $supIds[$positionOrder] && strlen($supIds[$positionOrder]) > 0 ? $supIds[$positionOrder] : null;

                // Crew Reason
                $crewReason = isset($crewReasons[$positionOrder]) && $crewReasons[$positionOrder] && strlen($crewReasons[$positionOrder]) > 0 ? $crewReasons[$positionOrder] : null;

                // If Crew Exists on The Given Position Order
                if (isset($crewIds[$positionOrder]) && $crewIds[$positionOrder] && strlen($crewIds[$positionOrder]) > 0) {
                    // Current Crew ID
                    $currentCrewId = $crewIds[$positionOrder];

                    // Processed User Ids,
                    $processedCrewIds[] = $currentCrewId;

                    // Check If Previous Crew Still Exists in Current Crew List => Continue
                    if (in_array($each->user_id, $crewIds) && $currentCrewId == $each->user_id) {
                        // Update Report Time & Reason And Save
                        $previousCrew[$i]->crew_sup_id = $supId;
                        $previousCrew[$i]->report_time = $reportTime;
                        $previousCrew[$i]->reason = $crewReason;
                        $previousCrew[$i]->updated_by = Auth::user()->id;
                        $previousCrew[$i]->save();
                        continue;
                    }
                    // Create New Crew, Reason Applies For The Previous Crew
                    $fields = [
                        'position_order'    => $positionOrder,
                        'report_time'       => $reportTime,
                        'crew_sup_id'       => $supId
                    ];
                    FlightCrew::createCrewItem($currentCrewId, $flightId, $divertedFlightId, $fields, $is_standby, $is_dhc, $is_sup);
                }

                // Log The Previous Crew (save to FlightCrewLog)
                $fields = [
                    'position_id'       => $each->position_id,
                    'crew_roster_id'    => $each->roster_id,
                    'position_order'    => $each->position_order,
                    'report_time'       => $each->report_time,
                    'reason'            => $crewReason
                ];

                LogFlightCrew::createCrewItem($each->user_id, $each->flight_id, $fields, $each->is_standby, $each->is_dhc);

                // Delete Previous Crew From Active Flight Crew List (FlightCrew)

                // Audit Deleted
                auditDeleted($previousCrew[$i]);
//                $previousCrew[$i]->delete();

            }
        }

        if (!$crewIds){
            return true;
        }

        $remainingCrewIds = array_diff($crewIds, $processedCrewIds);
        // If Remains Not Processed Crew Ids

//        debug("Remaining Crew IDs:". implode(",", $remainingCrewIds));

        if (count($remainingCrewIds)){
            foreach ($remainingCrewIds as $i => $userId) {
                if ($userId && strlen($userId) > 0) {
                    $reportTime = isset($crewReportTimes[$i]) && $crewReportTimes[$i] && strlen($crewReportTimes[$i]) > 0 ? $crewReportTimes[$i] : null;
                    $crewReason = isset($crewReasons[$i]) && $crewReasons[$i] && strlen($crewReasons[$i]) > 0 ? $crewReasons[$i] : null;
                    $supId = isset($supIds[$i]) && $supIds[$i] && strlen($supIds[$i]) > 0 ? $supIds[$i] : null;

                    // Create New Crew
                    $fields = [
                        'position_order' => $i,
                        'report_time'    => $reportTime,
                        'reason'         => $crewReason,
                        'crew_sup_id'    => $supId
                    ];
                    FlightCrew::createCrewItem($userId, $flightId, $divertedFlightId, $fields, $is_standby, $is_dhc, $is_sup);
                }
            }
        }

        return TRUE;
    }

}
