<?php namespace App\Models;

/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 27.08.14

 */

use App\Classes\FlightStaff\StaffFilter;
use App\Classes\Parsing\Parse;
use app\Classes\ProjectConstant;
use App\Repositories\Interfaces\IAgencyCodeRepository;
use App\Repositories\Interfaces\ICountryRepository;
use App\Repositories\Interfaces\IFlightCabinRepository;
use App\Repositories\Interfaces\IFlightDelayRepository;
use App\Repositories\Interfaces\IFlightFareBasisRepository;
use App\Repositories\Interfaces\IFlightOrderStatusRepository;
use App\Repositories\Interfaces\IFlightRbdRepository;
use App\Repositories\Interfaces\IFlightTypeRepository;
use App\Repositories\Interfaces\IRegionRepository;
use App\Repositories\Interfaces\IReportTypeRepository;
use App\Repositories\Interfaces\ISalesChannelRepository;
use Illuminate\Database\Eloquent\Model as Eloquent;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;



use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class Flight extends Eloquent implements AuditableContract
{
    use Auditable;

    protected $auditExclude = [
        'parent_id',
        'updated_by',
        'created_by',
    ];

    protected $auditableEvents = [
        'updated',
        'deleted',
        'restored',
    ];

    protected $table = "flights";

    protected $guarded = [
        "id",
        "created_at",
        "updated_at"
    ];

    public $timestamps = TRUE;

    public function flightChild()
    {
        return $this->belongsTo(Flight::class, "id", "parent_id");
    }

    public function flightParent()
    {
        return $this->belongsTo(Flight::class, "parent_id");
    }

    public function flightPts()
    {
        return $this->hasOne(FlightPTS::class, "flight_id");
    }

    public function fhr()
    {
        return $this->hasOne(FlightFHR::class, "flight_id");
    }

    public function flightCom()
    {
        return $this->hasOne(FlightCom::class, "flight_id");
    }

    public function flightComPts()
    {
        return $this->hasMany(FlightComFlightPTS::class, "flight_id");
    }

    public function flightNumber()
    {
        return $this->belongsTo("App\\Models\\FlightNumber", "flight_number_id");
    }

    public function flightSchedule()
    {
        return $this->belongsTo(FlightSchedule::class, "flight_schedule_id");
    }

    public function flightScheduleFlight()
    {
        return $this->belongsTo("App\\Models\\FlightScheduleFlight", "flight_schedule_flight_id");
    }

    public function ssr()
    {
        return $this->hasMany(FlightSSR::class);
    }

    public function aircraft()
    {
        return $this->belongsTo("App\\Models\\Aircraft", "aircraft_id");
    }

    public function aircraftType()
    {
        return $this->belongsTo("App\\Models\\AircraftType", "aircraft_type_id");
    }

    public function user()
    {
        return $this->belongsTo("App\\Models\\User", "updated_by");
    }

    public function departureAirport()
    {
        return $this->belongsTo("App\\Models\\Airport", "departure_airport_id");
    }

    public function arrivalAirport()
    {
        return $this->belongsTo("App\\Models\\Airport", "arrival_airport_id");
    }

    public function divertedAirport()
    {
        return $this->belongsTo("App\\Models\\Airport", "diverted_airport_id");
    }

    public function nextFlight()
    {
        return $this->belongsTo("App\\Models\\Flight", "next_flight_id");
    }

    public function prevFlight()
    {
        return $this->belongsTo("App\\Models\\Flight", "prev_flight_id");
    }

    public function hbg()
    {
        return $this->hasMany("App\\Models\\FlightHBG", "flight_id");
    }

    public function station()
    {
        return $this->hasMany("App\\Models\\FlightStation", "flight_id");
    }

    public function message()
    {
        return $this->hasMany("App\\Models\\FlightMessage", "flight_id");
    }

    public function message2018()
    {
        return $this->hasMany(FlightMessage2018::class, "flight_id");
    }

    public function container()
    {
        return $this->hasMany("App\\Models\\FlightContainer", "flight_id");
    }

    public function cl()
    {
        return $this->hasMany("App\\Models\\CrewListMessage", "flight_id");
    }

    public function cpm()
    {
        return $this->hasMany("App\\Models\\CPMMessage", "flight_id");
    }

    public function ldmMessage()
    {
        return $this->hasMany("App\\Models\\LDMmessage", "flight_id");
    }

    public function lpm()
    {
        return $this->hasMany("App\\Models\\LPMMessage", "flight_id");
    }

    public function mvt()
    {
        return $this->hasMany("App\\Models\\MVTMessage", "flight_id");
    }

    public function others()
    {
        return $this->hasMany("App\\Models\\OtherMessage", "flight_id");
    }

    public function pnl()
    {
        return $this->hasMany("App\\Models\\PNLMessage", "flight_id");
    }

    public function prl()
    {
        return $this->hasMany("App\\Models\\PRLMessage", "flight_id");
    }

    public function psm()
    {
        return $this->hasMany("App\\Models\\PSMMessage", "flight_id");
    }

    public function ptm()
    {
        return $this->hasMany("App\\Models\\PTMMessage", "flight_id");
    }

    public function tpm()
    {
        return $this->hasMany("App\\Models\\TPMMessage", "flight_id");
    }

    public function ucm()
    {
        return $this->hasMany("App\\Models\\UCMMessage", "flight_id");
    }

    public function flightCrew()
    {
        return $this->belongsToMany("App\\Models\\User", "flights__crew", "flight_id");
    }

    public function flightCrewItems()
    {
        return $this->hasMany("App\\Models\\FlightCrew", "flight_id");
    }

    public function flightStaff(){
        return $this->hasMany("App\\Models\\FlightStaff", "flight_id");
    }

    public function divertedFlights()
    {
        return $this->hasMany("App\\Models\\FlightDiverted", "flight_id");
    }

    public function passengerFlights()
    {
        return $this->hasMany("App\\Models\\PassengerFlight", "flight_id");
    }

    public function delays(){
        return $this->hasMany("App\\Models\\FlightDelay", "flight_id");
    }

    public function catering(){
        return $this->hasOne(FlightCatering::class, "flight_id");
    }

    public static function getFlight($id, $detailed = false){
        if ($detailed){
            return Flight::with([
                'ssr',
                'aircraftType',
                'aircraft',
                'aircraft.type',
                'station',
                'departureAirport',
                'arrivalAirport',
                'delays',
                'flightNumber',
                'delays.delay',
                'delays.delay.airline',
                'flightNumber.airline',
                'flightNumber.departureAirport',
                'flightNumber.arrivalAirport',
            ])->find($id);
        }

        return Flight::with([
            'ldmMessage',
            'aircraft',
            'aircraft.type',
            'aircraftType',
            'flightNumber',
            'flightNumber.departureAirport',
            'flightNumber.arrivalAirport',
            'flightNumber.airline',
            'hbg',
        ])
            ->find($id);
    }

    public static function getFlightsDetailed($IDs, $detailed = false){
        if ($detailed){
            return Flight::with([
                'ssr',
                'aircraftType',
                'aircraft',
                'aircraft.type',
                'station',
                'departureAirport',
                'arrivalAirport',
                'delays',
                'flightNumber',
                'delays.delay',
                'delays.delay.airline',
                'flightNumber.airline',
                'flightNumber.departureAirport',
                'flightNumber.arrivalAirport',
            ])->whereIn("id", $IDs)
                ->whereNull("deleted_at")
                ->get();
        }

        return Flight::with([
            'ldmMessage',
            'aircraft',
            'aircraft.type',
            'aircraftType',
            'flightNumber',
            'flightNumber.departureAirport',
            'flightNumber.arrivalAirport',
            'flightNumber.airline',
            'hbg',
        ])
            ->whereIn("id", $IDs)
            ->whereNull("deleted_at")
            ->get();
    }


    public static function getFlightStaff($flight){
        $staffList = [];
        if ($flight->flightStaff && count($flight->flightStaff)){

            foreach ($flight->flightStaff as $staff) {

                if (!$staff || !$staff->user || $staff->deleted_at){
                    continue;
                }

                if (!isset($staffList[$staff->service->abbr])){
                    $staffList[$staff->service->abbr] = [];
                }

                $staff->location = Location::getUserLocationAirport($staff->user->location_id);

                $staffList[$staff->service->abbr][] = $staff;
            }
        }

        ksort($staffList);

        return $staffList;
    }

    public static function getFlightStaffUserIDs($flight){
        $staffList = [];
        if ($flight->flightStaff && count($flight->flightStaff)){
            foreach ($flight->flightStaff as $staff) {
                if ($staff->deleted_at){
                    continue;
                }

                if (!isset($staffList[$staff->service->abbr])){
                    $staffList[$staff->service->abbr] = [];
                }

                $staffList[$staff->service->abbr][] = $staff->user_id;
            }
        }
        ksort($staffList);

        return $staffList;
    }

    public static function getFlightStaffUsersObject($flight, $airportID = null){
        $staffList = [];

//        $flightStaff = $flight->flightStaff;

        $flightStaff =  FlightStaff::with([
            "service",
            "user",
            "user.location",
//            "user.location.airport",
        ])
            ->where("flight_id", $flight->id)
            ->whereNull("deleted_at")
            ->get();

        if ($airportID){
            $airportID = is_array($airportID) ? $airportID : [$airportID];
        }
//
        if ($flightStaff->count()){
            foreach ($flightStaff as $each) {

                // IT's not active - for some reasons
//                if ($airportID && !in_array($each->user->location->airport_id, $airportID)){
//                    continue;
//                }

                if (!isset($staffList[$each->service->abbr])){
                    $staffList[$each->service->abbr] = [];
                }

                $staffList[$each->service->abbr][$each->user_id] = $each;
            }
        }
        ksort($staffList);

        return $staffList;
    }

    public static function getFlightsStaffServiceData($flights, $airportId = null, $withStaffLocation = null, $includeStaffData = null){
        $data = [];

        $handlingStations = Airport::getHandlingStationsIdAndObject();

        if ($flights instanceof Flight){
            return self::prepareServiceData($flights, $handlingStations, $airportId, $withStaffLocation, $includeStaffData);
        }
        else if (count($flights)){
            foreach ($flights as $flight) {

                $result = self::prepareServiceData($flight, $handlingStations, $airportId, $withStaffLocation, $includeStaffData);

                if (!$result || ($result && !count($result["services"]))){
                    continue;
                }

                $data[$flight->id] = $result;
            }
        }
        else if ($flights instanceof Collection) {
            return self::prepareServiceData($flights->first(), $handlingStations, $airportId, $withStaffLocation, $includeStaffData);
        }

        return $data;
    }

    public static function setFlightsStaffServiceData(&$flights, $airportId = null, $withStaffLocation = null, $includeStaffData = null,
                                                      $occStaffOnly = null, $staffAirportID = null){
        $handlingStations = Airport::getHandlingStationsIdAndObject();

        if ($flights instanceof Flight){
            $flights->serviceData = self::prepareServiceData($flights, $handlingStations, $airportId, $withStaffLocation, $includeStaffData, $occStaffOnly, $staffAirportID);
        }
        else if (count($flights)){
            foreach ($flights as &$flight) {

                $result = self::prepareServiceData($flight, $handlingStations, $airportId, $withStaffLocation, $includeStaffData, $occStaffOnly, $staffAirportID);

                if (!$result || ($result && !count($result["services"]))){
                    continue;
                }

                $flight->serviceData = $result;
            }
        }
        else if ($flights instanceof Collection) {
            $flight = $flights->first();
            if ($flight) {
                $flight->serviceData = self::prepareServiceData($flight, $handlingStations, $airportId, $withStaffLocation, $includeStaffData, $occStaffOnly, $staffAirportID);
            }
        }
    }

    public static function updateFlightServiceData($flight){

        if (!$flight){
            return false;
        }

        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber->airline;

        if (!$airline){
            return false;
        }

        $handlingStations = Airport::getHandlingStationsIdAndObject();

        $airportId = self::getFlightNumbersHandlingStation($handlingStations, $flightNumber);

        $airlineServices = AirlineService::getServices($airline->id, $airportId);

        $station = $serviceType = null;

        if (array_key_exists($flightNumber->departure_airport_id, $handlingStations)){

            $station = $handlingStations[$flight->flightNumber->departure_airport_id];

            $serviceType = DEPARTURE_SERVICE;
        }
        elseif (array_key_exists($flightNumber->arrival_airport_id, $handlingStations)){

            $station = $handlingStations[$flight->flightNumber->arrival_airport_id];

            $serviceType = ARRIVAL_SERVICE;
        }

        // Skip flight if not enough info detected
        if (! ($station && $airline && $airlineServices)){
            return false;
        }

        $depArrTime = $serviceType == ARRIVAL_SERVICE ? getFlightArrivalInitialDate($flight) : getFlightDepartureInitialDate($flight);

        $rosteredStaffUsers = Flight::getFlightStaffUsersObject($flight);

        foreach ($airlineServices as $j => $airlineService) {

            if (!checkSLAEffectivePeriod($airlineService, $depArrTime)){
                continue;
            }

            // SKIP if no appropriate service found
            if ($serviceType == ARRIVAL_SERVICE){
                if ( ($airlineService->service_timings && !$airlineService->service->arrival_service)
                    || (!$airlineService->service_timings && !$airlineService->arrival_service)
                ){
                    continue;
                }
            }
            else {
                if ( ($airlineService->service_timings && !$airlineService->service->departure_service)
                    || (!$airlineService->service_timings && !$airlineService->departure_service)
                ){
                    continue;
                }
            }
            // End

            if ($rosteredStaffUsers && isset($rosteredStaffUsers[$airlineService->service->abbr])
                && count($rosteredStaffUsers[$airlineService->service->abbr])){

                foreach ($rosteredStaffUsers[$airlineService->service->abbr] as $userID => $flightStaff) {
                    /**
                     * Prepared Data
                     */
                    if ($airlineService->service_timings == 1) {
                        if ($airlineService->service[$serviceType] != 1) {
                            continue;
                        }

                        $obj = $airlineService->service;
                    }
                    else {
                        if ($airlineService[$serviceType] != 1) {
                            continue;
                        }

                        $obj = $airlineService;
                    }

                    switch ($serviceType){
                        case ARRIVAL_SERVICE:
                            $req = "arr_staff_req";
                            $min = "arr_staff_min";
                            $arr = getFlightArrivalInitialDatePTA($flight);
                            $plannedReport = date("Y-m-d H:i:s", strtotime($arr) - (60 * $obj->arr_report_time));
                            $plannedRelease = date("Y-m-d H:i:s", strtotime($arr) + (60 * $obj->arr_release_time));
                            $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                            break;

                        case DEPARTURE_SERVICE:
                            $req = "dep_staff_req";
                            $min = "dep_staff_min";
                            $dep = getFlightDepartureInitialDatePTD($flight);
                            $plannedReport = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_report_time));
                            $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_release_time));
                            $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                            break;

                        case TURNAROUND_SERVICE:
                            $req = "turn_staff_req";
                            $min = "turn_staff_min";
                            $arr = getFlightArrivalInitialDatePTA(getParentFlight($flight));
                            $dep = getFlightDepartureInitialDatePTD($flight);

                            $plannedReport = date("Y-m-d H:i", strtotime($arr) - (60 * $obj->turn_report_time));
                            $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->turn_release_time));
                            $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));

                            break;
                    }



                    // PLN time changed
                    if (strtotime($flightStaff->planned_report_time) != strtotime($plannedReport)
                        || strtotime($flightStaff->planned_release_time) != strtotime($plannedRelease)){
                        $flightStaff->planned_report_time = $plannedReport;
                        $flightStaff->planned_release_time = $plannedRelease;
                        $flightStaff->sod_updated_at = date("Y-m-d H:i:s");
                        $flightStaff->sod_updated_by = Auth::user()->id;
                        $flightStaff->save();
                    }
                }
            }
        }

    }

    public static function prepareServiceData($flight, $handlingStations, $airportId = null, $loadAllStationsStaff = null,
                                              $includeStaffData = null, $occStaffOnly = null, $staffAirportID = null){

        if (!$flight){
            return false;
        }

        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber->airline;

        if (!$airline){
            return false;
        }


        list($serviceType, $station, $depArrTime) = FlightStaff::getFlightServiceType($flight, $airportId);

        $airlineServices = AirlineService::getServices($airline->id, $airportId, $occStaffOnly);

        // Skip flight if not enough info detected
//        if (! ($station && $airline && $airlineServices)){
//            return false;
//        }

//        $depArrTime = $serviceType == ARRIVAL_SERVICE ? getFlightArrivalInitialDate($flight) : getFlightDepartureInitialDate($flight);

        $eligibleServices = [];

        $rosteredStaffUserIDs = $includeStaffData ? Flight::getFlightStaffUsersObject($flight, $staffAirportID) : null;

        $depAirportID = $flight->departure_airport_id ? $flight->departure_airport_id : $flight->flightNumber->departure_airport_id;
        $arrAirportID = $flight->arrival_airport_id ? $flight->arrival_airport_id : $flight->flightNumber->arrival_airport_id;
        $stationID = null;

        foreach ($airlineServices as $j => $airlineService) {

//            $depArrTime = $serviceType == ARRIVAL_SERVICE ? getFlightArrivalInitialDate($flight) : getFlightDepartureInitialDate($flight);
//            if (!checkSLAEffectivePeriod($airlineService, $depArrTime)){
//                continue;
//            }

            // SKIP if no appropriate service found
            if ($serviceType == ARRIVAL_SERVICE){
                $stationID = $arrAirportID;
                $service_type = ARRIVAL_SERVICE;

                if ( ($airlineService->service_timings && !$airlineService->service->arrival_service)
                    || (!$airlineService->service_timings && !$airlineService->arrival_service)
                ){
                    continue;
                }
            }
            else {
                $stationID = $depAirportID;
                $service_type = DEPARTURE_SERVICE;
                if ( ($airlineService->service_timings && !$airlineService->service->departure_service)
                    || (!$airlineService->service_timings && !$airlineService->departure_service)
                ){
                    if (($airlineService->service_timings && !$airlineService->service->turnaround_service)
                        || (!$airlineService->service_timings && !$airlineService->turnaround_service)){
                        continue;
                    }
                    else {
                        if (staffFlightIsInactiveByID($flight->parent_id)){
                            continue;
                        }
                        else {
                            $service_type = TURNAROUND_SERVICE;
                        }
                    }
                }

            }
            // End

            if ($loadAllStationsStaff){
                $eligibleStaff = StaffService::getStaffWithLocation($airline->id, $airlineService, $stationID);
            }
            else {
                $eligibleStaff = StaffService::getStaff($stationID, $airline->id, $airlineService);
            }

            $rosteredStaff = [];
            if ($rosteredStaffUserIDs && isset($rosteredStaffUserIDs[$airlineService->service->abbr])
                && count($rosteredStaffUserIDs[$airlineService->service->abbr])){

                foreach ($eligibleStaff as $k => $flightStaff) {
                    if (isset($rosteredStaffUserIDs[$airlineService->service->abbr][$flightStaff->user_id])){
                        $eligibleStaff[$k]->selected = true;

                        $rosteredStaff[] = $rosteredStaffUserIDs[$airlineService->service->abbr][$flightStaff->user_id];
                    }
                }
            }

            // Skip other station SLAs if
            if (!count($rosteredStaff)){
                if ($staffAirportID && $staffAirportID != $stationID){
                    continue;
                }
            }

            /**
             * Prepared Data
             */
            if ($airlineService->service_timings == 1) {
                if ($airlineService->service[$service_type] != 1) {
                    continue;
                }

                $obj = $airlineService->service;
            }
            else {
                if ($airlineService[$service_type] != 1) {
                    continue;
                }

                $obj = $airlineService;
            }

            switch ($service_type){
                case ARRIVAL_SERVICE:
                    $req = "arr_staff_req";
                    $min = "arr_staff_min";
                    $arr = getFlightArrivalInitialDatePTA($flight);
                    $plannedReport = date("Y-m-d H:i:s", strtotime($arr) - (60 * $obj->arr_report_time));
                    $plannedRelease = date("Y-m-d H:i:s", strtotime($arr) + (60 * $obj->arr_release_time));
                    $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                    break;

                case DEPARTURE_SERVICE:
                    $req = "dep_staff_req";
                    $min = "dep_staff_min";
                    $dep = getFlightDepartureInitialDatePTD($flight);
                    $plannedReport = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_report_time));
                    $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->dep_release_time));
                    $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));
                    break;

                case TURNAROUND_SERVICE:
                    $req = "turn_staff_req";
                    $min = "turn_staff_min";
                    $arr = getFlightArrivalInitialDatePTA(getParentFlight($flight));
                    $dep = getFlightDepartureInitialDatePTD($flight);

                    $plannedReport = date("Y-m-d H:i", strtotime($arr) - (60 * $obj->turn_report_time));
                    $plannedRelease = date("Y-m-d H:i", strtotime($dep) - (60 * $obj->turn_release_time));
                    $duration = date("H:i", strtotime($plannedRelease) - strtotime($plannedReport));

                    break;
            }


            $sla = $obj->service ? $obj->service->abbr : $obj->abbr;

            $airlineServices[$j]->sla = $sla;
            $airlineServices[$j]->req = $airlineService->{$req} ? $airlineService->{$req} : $airlineService->service->{$req};
            $airlineServices[$j]->report_time = $plannedReport;
            $airlineServices[$j]->release_time = $plannedRelease;
            $airlineServices[$j]->duration = $duration;

            // Check if doesn't work
            $airlineServices[$j]->serviceType = $service_type;
            /**
             * End
             */

            $airlineServices[$j]->rosteredStaff = $rosteredStaff;
            $airlineServices[$j]->eligibleStaff = $eligibleStaff;

            $eligibleServices[] = $airlineServices[$j];
        }

        $departureAirport = $flightNumber->departureAirport ? $flightNumber->departureAirport->iata : "";
        $arrivalAirport = $flightNumber->arrivalAirport ? $flightNumber->arrivalAirport->iata : "";
        $sector = $departureAirport."-".$arrivalAirport;

        $flightTypeText = $serviceType == ARRIVAL_SERVICE ? "ARR" : "DEP";
        $flightType = $serviceType == ARRIVAL_SERVICE ? "<span class='label label-primary'>ARR</span>" : "<span class='label label-success'>DEP</span>";

        return [
            'id'            => $flight->id,
            'flight'        => "/{$flightTypeText}/ ".$airline->iata."-".$flightNumber->flight_number."(".$sector.") ".baseDateFormat($depArrTime, true, "-", true),
            'serviceType'   => $serviceType,
            'flightType'    => $flightType,
            "depArrTime"    => $depArrTime,
            'services'      => $eligibleServices,
            'station'       => $airportId,
        ];
    }

    public static function getFlightNumbersHandlingStation($handlingStations, $flightNumber){
        if (array_key_exists($flightNumber->departure_airport_id, $handlingStations)){
            return $flightNumber->departure_airport_id;
        }
        elseif (array_key_exists($flightNumber->arrival_airport_id, $handlingStations)){
            return $flightNumber->arrival_airport_id;
        }
        return null;
    }

    protected static function groupSelection($group_period, $sales_departure=null){
        $column = isset($sales_departure) ? $sales_departure : "std";
        switch($group_period) {
            case 0:
                $select = "YEAR($column)";
                break;
            case 1:
                $select  = "QUARTER($column)";
                break;
            case 2:
                $select = "MONTH($column)";
                break;
            case 3:
                $select  = "WEEK($column)";
                break;
            case 4:
                $select  = "WEEKDAY($column)";
                break;
            case 5:
                $select  = "DATE($column)";
                break;
            case 6:
                $select  = "flight_number";
                break;
            case 7:
                $select  = "CONCAT(departure_airport,'-',arrival_airport)";
                break;
            case 8:
                $select  = "DATE_FORMAT(sales_date,'%b %e')";
                break;
            case 9:
                $select = "YEAR($column)";
                break;
        }
        return $select;
    }

    /**
     * @param $flightCrew
     * @return array|bool
     */
    public static function calculateBlockDurationAndDepartureDate($flightCrew){
        if ($flightCrew && is_object($flightCrew)) {
            $flight = $flightCrew->flight;

            $dep = getFlightDepartureDate($flight);
            $arr = getFlightArrivalDate($flight);

            $blockDuration = $dep && $arr && strtotime($arr) > strtotime($dep) ? Calculate_Duration($dep, $arr) : 0;
            $departureDate = date('Y-m-d', strtotime($dep));

            return [$blockDuration, $departureDate];
        }
        return false;
    }

    public static function getFlights($date_from = NULL, $date_to = NULL, $groupbyflight = NULL, $group_period = NULL, $inbound = NULL)
    {
        if (isset($inbound))
        {
            $inbound = "NOT";
        }
        else
        {
            $inbound = "";
        }

        if (isset($group_period)) { // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily, 6-ByFlights
            $select = Flight::groupSelection($group_period);
            if ($group_period == 6) {
                $group = "flight_number";
            }

            if (isset($date_from, $date_to)) {
                $sql = " AND DATE(std) BETWEEN '$date_from' AND '$date_to'";
            } elseif (isset($date_from)) {
                $sql = " AND DATE(std) = '$date_from'";
            } else {
                $sql = " AND DATE(std) IS NOT NULL";
            }


            $data = Flight::selectRaw("s_flights.id as flight_id")
                ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
                ->whereRaw("departure_airport $inbound IN ('DYU', 'LBD') $sql")
                ->get();

            $new_data = array();
            foreach ($data as $each) {
                $new_data[] = $each->flight_id;
            }

            return $new_data;

        }
    }

    public static function getFlightPaxRevenue($date_from = null, $date_to = null, $group_period = null, $flight_number = null)
    {
        $select = "departure_date";
        $group  = "date";
        $select_departure = '';

        if (isset($group_period) && $group_period != 5)
        { // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily, 6-ByFlights
            $select = groupSelectionNonComparison($group_period, 'departure_date');
        }

        if ($date_from != $date_to && in_array($group_period,[6])) {
            $select_departure = "CONCAT ($select,' | ', DATE_FORMAT(departure_date, '%b %d, %Y')) as flight_departure, DATE(departure_date) as departure_date, ";
            $group = 'flight_departure';
        }


        if (isset($date_from, $date_to))
            $sql = " departure_date BETWEEN '$date_from' AND '$date_to'";
        elseif (isset($date_from))
            $sql = " departure_date = '$date_from'";
        else
            $sql = " departure_date IS NOT NULL";


        $data = PassengerFlight::selectRaw("$select AS date, $select_departure
                SUM(IF(bound = '0' AND s_flights__cabin.cabin = 'Business',1,0)) as pax_out_c,
                SUM(IF(bound = '0' AND s_flights__cabin.cabin = 'Economy', 1,0)) as pax_out_y,
                SUM(IF(bound = '1' AND s_flights__cabin.cabin = 'Business',1,0)) as pax_in_c,
                SUM(IF(bound = '1' AND s_flights__cabin.cabin = 'Economy', 1,0)) as pax_in_y,
                SUM(IF(bound = '1' AND s_flights__cabin.cabin = 'Business', s_passengers__flights.fare, 0)) as revenue_in_c,
                SUM(IF(bound = '1' AND s_flights__cabin.cabin = 'Economy',  s_passengers__flights.fare, 0)) as revenue_in_y,
                SUM(IF(bound = '0' AND s_flights__cabin.cabin = 'Business', s_passengers__flights.fare, 0)) as revenue_out_c,
                SUM(IF(bound = '0' AND s_flights__cabin.cabin = 'Economy',  s_passengers__flights.fare, 0)) as revenue_out_y")
            ->join('flights__numbers',     'passengers__flights.flight_number_id',     '=', 'flights__numbers.id')
            ->join('flights__cabin',       'passengers__flights.flight_cabin_id',      '=', 'flights__cabin.id')
            ->join('flights__fare_basis',  'passengers__flights.flight_fare_basis_id', '=', 'flights__fare_basis.id')
            ->join('flights__order_status','passengers__flights.flight_order_status_id','=', 'flights__order_status.id')
            ->join('passengers__orders', 'passengers__flights.passenger_order_id',  '=', 'passengers__orders.id')
            ->leftJoin('flights', 'passengers__flights.flight_id', '=', 'flights.id')
            ->leftJoin('aircraft', 'flights.aircraft_id', '=', 'aircraft.id')
            ->whereRaw($sql)
            ->where('fare_basis', '!=', 'IN')
            ->whereNotIn('order_status',['R', 'V', 'E']);

        if (isset($flight_number))
            $data->whereIn('flight_number', $flight_number);

        $data->orderBy($group)
            ->groupBy($group);

        $data = $data->get();

        return $data;
    }



    public static function getPaxActual($date_from = null, $date_to = null, $period=null, $flight_number = null, $select_value = null, $sector=null, $month=null, $year =null){

        $where = "";

        if (!is_null($select_value)){
            $select = $select_value;
        }
        else {
            $select = "DATE(std)";
        }

        if (isset($month)) {
            $select = "DATE_FORMAT(std,'%b')";
            if (is_array($month)) {
                $where .= "MONTH(std) IN(" . implode(',', $month) . ") AND ";
            } else
                $where .= "MONTH(std) = '" . $month."' AND ";

            if (isset($year))
                $where .= "YEAR(std) = '".$year."' AND ";
        }
        else {
            if (isset($date_from, $date_to))
                $where .= " DATE(std) BETWEEN '$date_from' AND '$date_to' AND ";
            else if (isset($date_from))
                $where .= " DATE(std) = '$date_from' AND ";
            else
                $where .= " DATE(std) IS NOT NULL AND ";
        }

        if (isset($period)){
            if ($period == 'Monthly')
                $select = "DATE_FORMAT(std,'%b')";
            else if($period == 'Daily')
                $select = 'DATE(std)';
        }

        if (isset($flight_number)){
            if (is_array($flight_number))
                $where .= " flight_number IN('" . implode("' , '", $flight_number) . "') AND";
            else
                $where .= ' flight_number = "'.$flight_number.'" AND';
        }
        else if (isset($sector)){
            if (is_array($sector))
            {
                $where .= "(";
                foreach ($sector as $each_sector)
                {
                    $sector_arr = explode("-", $each_sector);
                    $from       = $sector_arr[0];
                    $to         = isset($sector_arr[1]) ? $sector_arr[1] : "";
                    $where .= " (departure_airport = '$from' AND arrival_airport = '$to') OR ";
                }
                $where = rtrim($where, 'OR ');
                $where .= ")";
            }
            else
            {
                $sector_arr = explode("-", $sector);
                $from       = $sector_arr[0];
                $to         = isset($sector_arr[1]) ? $sector_arr[1] : "";
                $where .= " (departure_airport = '$from' AND arrival_airport = '$to')";
            }
        }

        $where = rtrim($where, "AND ");

        $data = Flight::selectRaw("$select AS date,
                SUM(pax_c_actual) as pax_c_actual,
                SUM(pax_y_actual) as pax_y_actual")
            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->whereRaw($where)
            ->groupBy('date');

        return $data->get();
    }


    public static function getFlightCounter($dates){
        $dates = is_array($dates) ? $dates : [$dates];

        $data = Flight::select([DB::raw("COUNT(flight_number_id) AS counter"), DB::raw("DATE(std)". " AS date")]);

        if (count($dates) == 2 && strtotime($dates[1]) > strtotime($dates[0])){
            $data->whereBetween(DB::raw('DATE(std)'), $dates);
        }
        else {
            $data->whereIn(DB::raw('DATE(std)'), $dates);
        }


        $data->whereNull('cancelled_at')->whereNull('deleted_at')
            ->orderBy('date', 'DESC')
            ->groupBy('date');


        return $data->get();

    }

    /**
     * Calculate PAX By Given Date And/Or Bound
     * @param $dates
     * @param null $bound
     * @return mixed
     */
    public static function getPaxCounter($dates, $bound = null){
        $data = Flight::join('flights__numbers', 'flights.flight_number_id','=','flights__numbers.id');

        if (is_array($dates)) {
            // If Array (Len = 2) => Calculate Range
            if (count($dates) == 2 && strtotime($dates[0]) < strtotime($dates[1]))
                $data->whereBetween(DB::raw('DATE(std)'), $dates);
            // List Of Dates
            else
                $data->whereIn(DB::raw('DATE(std)'), $dates);
        }
        else {
            $data->where(DB::raw('DATE(std)'), $dates);
        }

        if (isset($bound))
            $data->where('bound', $bound == 'outbound' ? 0 : 1);

        $data = $data->orderBy('date', 'DESC')
            ->groupBy('date')
            ->get([DB::raw("SUM(pax_c_actual) AS pax_c, SUM(pax_y_actual) AS pax_y, DATE(std)" . " AS date")]);

        return $data;
    }

    /**
     * @param $dateSearch
     * @param null $flightSearch
     * @param null $period
     * @param null $airport
     * @param null $airline
     * @param string $type
     * @param null $aircraftID
     * @param null $airportCountryRelation
     * @return \Illuminate\Database\Eloquent\Collection|static[]
     */
    public static function getFlightsByDateFlightNumber($dateSearch, $flightSearch = null, $period = null, $airport = null, $airline = null, $type = DEPARTURE, $aircraftID = null, $airportCountryRelation = null, $includeCancelled = null){

        $select = "DATE(std)";

        $flightNumberSelect = "flight_number";

        $relationships = [
            'ssr',
            'aircraftType',
            'aircraft',
            'aircraft.type',
            'station',
            'departureAirport',
            'arrivalAirport',
            'delays',
            'flightNumber',
            'delays.delay',
            'delays.delay.airline',
            'flightNumber.airline',
            'flightNumber.departureAirport',
            'flightNumber.arrivalAirport',
        ];

        if ($airportCountryRelation){
            $relationships[] = "departureAirport.country";
            $relationships[] = "arrivalAirport.country";
            $relationships[] = "flightNumber.departureAirport.country";
            $relationships[] = "flightNumber.arrivalAirport.country";
        }

        $flights = Flight::with($relationships)
            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id');

//        if (env("CARRIER") == HANDLING){
//            $flights->join('airlines', 'airlines.id', '=', 'flights__numbers.airline_id');
//            $flightNumberSelect = DB::raw("CONCAT(s_airlines.iata,'-',flight_number) AS flight_number");
//        }

        if ($airline){
            $airline = is_array($airline) ? $airline : [$airline];
            $flights->whereIn("flights__numbers.airline_id", $airline);
        }

        // Filter by aircraft
        if ($aircraftID){
            if (is_array($aircraftID)){
                if (count($aircraftID)){
                    $flights->whereIn("flights.aircraft_id", $aircraftID);
                }
            }
            else {
                $flights->where("flights.aircraft_id", $aircraftID);
            }
        }

        if (isset($dateSearch['month'], $dateSearch['year'])) {
            $month = is_array($dateSearch['month']) ? $dateSearch['month'] : [ $dateSearch ['month'] ];
            $year = is_array($dateSearch['year']) ? $dateSearch['year'] : [ $dateSearch ['year'] ];
            $biMonth = isset($dateSearch['bi_month']) ? $dateSearch['bi_month'] : null;

            $flights->where(function ($sql) use ($month, $year, $biMonth) {
                $sql->whereIn(DB::raw("MONTH(departure_date)"), $month)
                    ->whereIn(DB::raw("YEAR(departure_date)"), $year);

                if ($biMonth){
                    $sql->whereBetween(DB::raw("DAY(departure_date)"), $biMonth == 1 ? [1, 15] : [16, 31]);
                }
            });
        }
        elseif (isset($dateSearch['from'], $dateSearch['to'])) {
            $flights->whereBetween("departure_date",
                [
                    date("Y-m-d", strtotime($dateSearch['from'])),
                    date("Y-m-d", strtotime($dateSearch['to']))
                ]);
//            $flights->where(function($sql) use ($dateSearch){
//                $sql->whereBetween  ("std", [ $dateSearch['from']." 00:00:00", $dateSearch['to']." 23:59:59"])
//                    ->orWhereBetween("ptd", [ $dateSearch['from']." 00:00:00", $dateSearch['to']." 23:59:59"])
//                    ->orWhereBetween("etd", [ $dateSearch['from']." 00:00:00", $dateSearch['to']." 23:59:59"])
//                    ->orWhereBetween("atd", [ $dateSearch['from']." 00:00:00", $dateSearch['to']." 23:59:59"]);
//            });
        }

        if (isset($flightSearch['flightNumberIds'])){
            $flightNumbersId = is_array($flightSearch['flightNumberIds']) ? $flightSearch['flightNumberIds'] : [ $flightSearch['flightNumberIds'] ];
            $flights->whereIn('flights__numbers.id', $flightNumbersId);
        }

        if (isset($flightSearch['sectorIds'])){
            $sectorIds = is_array($flightSearch['sectorIds']) ? $flightSearch['sectorIds'] : [ $flightSearch['sectorIds'] ];
            $flights->whereIn('flights__numbers.flight_sector_id', $sectorIds);
        }

        if($airport){
            $airport = is_array($airport) ? $airport : [ $airport ];
            switch($type){
                case ALL:
                    $flights->where(function($sql) use ($airport){
                        $sql->whereIn('flights.departure_airport_id', $airport)
                            ->orWhereIn('flights.arrival_airport_id', $airport)
                            ->orWhereIn('flights__numbers.departure_airport_id', $airport)
                            ->orWhereIn('flights__numbers.arrival_airport_id', $airport);

                    });
                    break;

                case ARRIVAL:
                    $flights->where(function($sql) use ($airport){
                        $sql->whereIn('flights.arrival_airport_id', $airport)
                            ->orWhereIn('flights__numbers.arrival_airport_id', $airport);

                    });
                    break;

                case DEPARTURE:
                default:
                    $flights->where(function($sql) use ($airport){
                        $sql->whereIn('flights.departure_airport_id', $airport)
                            ->orWhereIn('flights__numbers.departure_airport_id', $airport);
                    });
                    break;
            }
        }

        $flights->whereNull("flights.deleted_at");

        if (!$includeCancelled){
            $flights->whereNull("flights.cancelled_at");
        }

        // If Period IS Selected Group By Period
        if (isset($period) && $period != ''){
            $select = groupSelectionNonComparison($period);
            $flights->groupBy('date');
        }

        $flights->orderBy("std")
            ->orderBy("ptd")
            ->orderBy("etd")
            ->orderBy("atd");

        return $flights->get([
            "flights.id as id",
            DB::raw($select . " AS date"),
            "aircraft_id",
            $flightNumberSelect,
            "flight_number_id",
//            "std",
//            "ptd",
//            "atd",
//            "abn",
//            "sta",
//            "pta",
//            "ata",
//            "tdn",
//            "pax_c_booked",
//            "pax_y_booked",
//            "pax_c_actual",
//            "pax_y_actual",
//            "pax_inf_actual",
//            "gate_parking",
//            "baggage",
//            "mail",
//            "cargo",
//            "deportees",
//            "in_tanks",
            "flights.*"
        ]);
    }

    public static function getFlightsConfig($date_from = null, $date_to = null,$group_period = null,$flight_num=null, $select_value = null, $totals_only = null, $selected_months = null, $year = null, $sector=null)
    {
        $select_departure = '';
        $group  = "date";

        if (!is_null($select_value)){
            $select = $select_value;
        }
        else {
            $select = "DATE(std)";
        }

        if (isset($group_period))
        { // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily, 6-ByFlights
            $select = groupSelectionNonComparison($group_period);
        }
        $where = "";


        if ($date_from != $date_to && in_array($group_period,[6])) {
            $select_departure = "CONCAT ($select,' | ', DATE(std)) as flight_departure, DATE(std) as departure_date, ";
            $group = 'flight_departure';
        }


        if (isset($selected_months)){
            if (is_array($selected_months))
                $where .= "MONTH(std) IN (".implode(',', $selected_months).") AND ";
            else
                $where .= "MONTH(std) = ".$selected_months." AND ";

            if (isset($year))
                $where .= "YEAR(std) = '".$year."' AND ";
        }
        else {
            if (isset($date_from, $date_to))
                $where .= "(DATE(std) BETWEEN '$date_from' AND '$date_to') AND ";
            elseif (isset($date_from))
                $where .= " DATE(std) = '$date_from' AND ";
            else
                $where .= " DATE(std) IS NOT NULL AND ";
        }


        if (isset($sector)){
            if (is_array($sector))
            {
                $where .= "(";
                foreach ($sector as $each_sector)
                {
                    $sector_arr = explode("-", $each_sector);
                    $from       = $sector_arr[0];
                    $to         = isset($sector_arr[1]) ? $sector_arr[1] : "";
                    $where .= " (departure_airport = '$from' AND arrival_airport = '$to') OR ";
                }
                $where = rtrim($where, 'OR ');
                $where .= ")";
            }
            else
            {
                $sector_arr = explode("-", $sector);
                $from       = $sector_arr[0];
                $to         = isset($sector_arr[1]) ? $sector_arr[1] : "";
                $where .= " (departure_airport = '$from' AND arrival_airport = '$to') ";
            }
        }

        $where = rtrim($where, 'AND ');



        $all_selects = " COUNT(DISTINCT s_flights.id) as flight_count,
            SUM(capacity_c) as config_c,
            SUM(capacity_y) as config_y";
        if (is_null($totals_only)){
            $all_selects .= ",  SUM(IF(bound = '0',1,0)) as flight_count_out,
                                SUM(IF(bound = '1',1,0)) as flight_count_in,
                                SUM(IF(bound = '0', capacity_c, 0)) AS config_out_c,
                                SUM(IF(bound = '0', capacity_y, 0)) AS config_out_y,
                                SUM(IF(bound = '1', capacity_c, 0)) AS config_in_c,
                                SUM(IF(bound = '1', capacity_y, 0)) AS config_in_y";
        }


        $flights = Flight::selectRaw("$select as date, $select_departure $all_selects")
            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('aircraft', 'flights.aircraft_id', '=', 'aircraft.id')
            ->whereRaw($where);


        if (isset($flight_num)){
            if (is_array($flight_num))
                $flights->whereIn("flight_number",$flight_num);
            else
                $flights->where("flight_number",$flight_num);
        }

        $flights = $flights->orderBy($group)
            ->groupBy($group)
            ->get();

        return $flights;
    }

    /**
     * Calculate Flight (C, Y) Configuration (Filter: By Date, Flight Number Ids, Period)
     * @param $dateSearch
     * @param $flightSearch
     * @param bool $period
     * @return $this|array|\Illuminate\Database\Eloquent\Collection|static[]
     */
    public static function calculateFlightConfiguration($dateSearch, $flightSearch = null, $period = null)
    {
        $select = "DATE(std)";

        // For Default Viewing

        $flights = Flight::with(["aircraft", "aircraftType"])
            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at");

        if (isset($dateSearch['month']) && isset($dateSearch['year'])) {
            $month = is_array($dateSearch['month']) ? $dateSearch['month'] : [ $dateSearch ['month'] ];
            $year = is_array($dateSearch['year']) ? $dateSearch['year'] : [ $dateSearch ['year'] ];

            $flights->whereIn(DB::raw("MONTH(std)"), $month)
                ->whereIn(DB::raw("YEAR(std)"), $year);

            // If Period is Not Set, Then Make It -> Monthly
            if (!isset($period) || $period == ''){
                $period = 2;
            }
        }
        elseif (isset($dateSearch['from']) && isset($dateSearch['to'])) {
            $flights->whereBetween(DB::raw("DATE(std)"), [$dateSearch['from'], $dateSearch['to']]);
        }

        if ($flightSearch){

            if (isset($flightSearch['flightNumberIds'])){
                $flightNumbersId = is_array($flightSearch['flightNumberIds']) ? $flightSearch['flightNumberIds'] : [ $flightSearch['flightNumberIds'] ];
                $flights->whereIn('flights__numbers.id', $flightNumbersId);
            }

            if (isset($flightSearch['sectorIds'])){
                $sectorIds = is_array($flightSearch['sectorIds']) ? $flightSearch['sectorIds'] : [ $flightSearch['sectorIds'] ];
                $flights->whereIn('flights__numbers.flight_sector_id', $sectorIds);
            }

            // If Period is Not Set Make It By Flight Number
            if (!isset($period) || $period == ''){
                $period = 6;
            }
        }

        if (isset($period) && $period != '')
        {   // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily, 6-ByFlights
            $select = groupSelectionNonComparison($period);

            if ($period == 7){
                $flights->leftJoin('airports AS depAirport', 'depAirport.id', '=', 'flights__numbers.departure_airport_id' )
                    ->leftJoin('airports AS arrAirport', 'arrAirport.id', '=', 'flights__numbers.arrival_airport_id'   );
            }

            $group  = "date";
        }

        $flights->orderBy('date');

        /*
        if (isset($group)){
            $flights->groupBy($group);
        }

        $data2 = $flights->get([
            DB::raw("$select as date"),
            DB::raw("COUNT(DISTINCT s_flights.id) as flight_count"),
            DB::raw("SUM(capacity_c) as config_c"),
            DB::raw("SUM(capacity_y) as config_y")
        ]);

        debug($data2);
        */

        $flights = $flights->get([
            DB::raw("$select as date"),
            "flights.*",
        ]);

        $data = [];
        foreach ($flights as $each) {

            $index = isset($group) ? $each->date : "grouped";

            if (!isset($data[$index])){
                $data[$index] = new \stdClass();

                $data[$index]->date             = $index;
                $data[$index]->flight_count     = 0;
                $data[$index]->config_c         = 0;
                $data[$index]->config_y         = 0;
            }

            $capacity = getFlightCapacity($each);

            $data[$index]->flight_count++;
            $data[$index]->config_c += $capacity[0];
            $data[$index]->config_y += $capacity[1];
        }

        debug($data);

        return $data;
    }

    /**
     * Calculate Flight Actual PAX Load (Filter: By Date, Flight Number Ids, )
     * @param $dateSearch
     * @param $flightSearch
     * @param null $period
     * @return \Illuminate\Database\Eloquent\Collection|static[]
     */
    public static function calculateFlightActualPAXLoad($dateSearch, $flightSearch, $period = null)
    {
        $select = "DATE(std)";

        $flights = Flight::join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id');
        $flights->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at");

        if (isset($dateSearch['month'], $dateSearch['year'])) {
            $month = is_array($dateSearch['month']) ? $dateSearch['month'] : [ $dateSearch ['month'] ];
            $year = is_array($dateSearch['year']) ? $dateSearch['year'] : [ $dateSearch ['year'] ];

            $flights->whereIn(DB::raw("MONTH(std)"), $month)
                ->whereIn(DB::raw("YEAR(std)"), $year);

            // If Period is Not Set, Then Make It -> Monthly
            if (!isset($period) || $period == ''){
                $period = 2;
            }

        }
        elseif (isset($dateSearch['from'], $dateSearch['to'])) {
            $flights->whereBetween(DB::raw("DATE(std)"), [$dateSearch['from'], $dateSearch['to']]);
        }

        if ($flightSearch){
            if (isset($flightSearch['flightNumberIds'])){
                $flightNumbersId = is_array($flightSearch['flightNumberIds']) ? $flightSearch['flightNumberIds'] : [ $flightSearch['flightNumberIds'] ];
                $flights->whereIn('flights__numbers.id', $flightNumbersId);
            }

            if (isset($flightSearch['sectorIds'])){
                $sectorIds = is_array($flightSearch['sectorIds']) ? $flightSearch['sectorIds'] : [ $flightSearch['sectorIds'] ];
                $flights->whereIn('flights__numbers.flight_sector_id', $sectorIds);
            }

            // If Period is Not Set Make It By Flight Number
            if (!isset($period) || $period == ''){
                $period = 6;
            }
        }

        if (isset($period) && $period != '')
        {   // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily, 6-ByFlights
            $select = groupSelectionNonComparison($period);

            if ($period == 7){
                debug("By Sector");
                $flights->leftJoin('airports AS depAirport', 'depAirport.id', '=', 'flights__numbers.departure_airport_id' )
                    ->leftJoin('airports AS arrAirport', 'arrAirport.id', '=', 'flights__numbers.arrival_airport_id'   );
            }


            $group  = "date";
        }

        $flights->orderBy('date');

        /*
        if (isset($group)) {
            $flights->groupBy($group);
        }

        return $flights->get([
            DB::raw("$select as date"),
            DB::raw('COUNT(DISTINCT s_flights.id) as flight_count'),
            DB::raw('SUM(pax_c_actual) as pax_c'),
            DB::raw('SUM(pax_y_actual) as pax_y')
        ]);
        */

        $flights = $flights->get([
            DB::raw("$select as date"),
            "flights.*",
        ]);

        $data = [];
        foreach ($flights as $each) {

            $index = isset($group) ? $each->date : "grouped";

            if (!isset($data[$index])){
                $data[$index] = new \stdClass();

                $data[$index]->date             = $index;
                $data[$index]->flight_count     = 0;
                $data[$index]->pax_c            = 0;
                $data[$index]->pax_y            = 0;

            }

            $data[$index]->flight_count++;
            $data[$index]->pax_c         += $each->pax_c_actual ? $each->pax_c_actual : 0;
            $data[$index]->pax_y         += $each->pax_y_actual ? $each->pax_y_actual : 0;
        }

        debug($data);

        return $data;
    }



    public static function getFlightsConfigComparison($date_from = null, $date_to = null,$group_period = null,$comparing_years = null)
    {
        $select = "DATE(std)";
        $group="date";

        if (isset($group_period)){ // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily, 6-ByFlights
            $select = Flight::groupSelection($group_period);
            if ($group_period == 0)
                $select = "'Year'";
            elseif ($group_period == 5){
                $select = "DATE_FORMAT(std,'%b %e')";
            }
            elseif ($group_period == 8){
                $select = "DATE_FORMAT(std,'%b %e')";
            }
        }
        $sql = "";

        if (isset($flight_num) && is_numeric($flight_num))
            $sql .= "flight_number = '$flight_num' AND";
        if (isset($date_from, $date_to))
            $sql .= "(DATE(std) BETWEEN '$date_from' AND '$date_to')";
        elseif (isset($date_from))
            $sql .= "DATE(std) = '$date_from'";
        else
            $sql .= "DATE(std) IS NOT NULL";

        if (isset($comparing_years) && is_array($comparing_years)){
            $comparing_query = "";
            foreach ($comparing_years as $each){
                $comparing_query .= "SUM(IF(YEAR(std) = '$each' AND bound = '0',1,0)) as flight_count_out_{$each},
                                     SUM(IF(YEAR(std) = '$each' AND bound = '1',1,0)) as flight_count_in_{$each},
                                     SUM(IF(YEAR(std) = '$each' AND bound = '0', config_c, 0)) AS config_out_c_{$each},
                                     SUM(IF(YEAR(std) = '$each' AND bound = '0', config_y, 0)) AS config_out_y_{$each},
                                     SUM(IF(YEAR(std) = '$each' AND bound = '1', config_c, 0)) AS config_in_c_{$each},
                                     SUM(IF(YEAR(std) = '$each' AND bound = '1', config_y, 0)) AS config_in_y_{$each},";
            }
            $comparing_query = rtrim($comparing_query, ',');
        }
        else{
            $comparing_query = "SUM(IF(bound = '0',1,0)) as flight_count_out,
                                SUM(IF(bound = '1',1,0)) as flight_count_in,
                                SUM(IF(bound = '0', config_c, 0)) AS config_out_c,
                                SUM(IF(bound = '0', config_y, 0)) AS config_out_y,
                                SUM(IF(bound = '1', config_c, 0)) AS config_in_c,
                                SUM(IF(bound = '1', config_y, 0)) AS config_in_y";
        }

        $flights = Flight::selectRaw("$select as date, $comparing_query")
            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('aircraft', 'flights.aircraft_id', '=', 'aircraft.id')
            ->whereRaw($sql)
            ->orderBy($group);

        if ($group_period != 0) {
            $flights->groupBy($group);
        }

        return $flights->get();
    }








    public static function getFlightPaxRevenueTotal($date_from = null, $date_to = null, $group_period = null, $economy = null){

        //$inbound = (isset($inbound)) ? "NOT" : "";
        $economy = (isset($economy)) ? "NOT" : "";

        $select = "DATE(std)";

        $group="departure_date";

        if (isset($group_period)){ // $period  0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily
            $select = self::groupSelection($group_period);
        }
        elseif (isset($groupbyflight)) {
            $group = "flight_number";
        }

        if (isset($date_from, $date_to))
            $sql = " AND (DATE(std) BETWEEN '$date_from' AND '$date_to')";
        elseif (isset($date_from))
            $sql = " AND DATE(std) = '$date_from'";
        else
            $sql = "";

        $data = Flight::selectRaw("$select AS date,SUM(IF(bound = '0',1,0)) as pax_out, SUM(IF(bound = '0',s_passengers__flights.fare,null)) as revenue_out,
            SUM(IF(bound = '1',1,0)) as pax_in, SUM(IF(bound = '1',s_passengers__flights.fare,null)) as revenue_in")
            ->whereRaw("rbd $economy IN ('C','J','I','D') $sql")
            ->join('passengers__flights', 'passengers__flights.flight_id', '=', 'flights.id')
            ->join('passengers__orders', 'passengers__flights.passenger_order_id', '=', 'passengers__orders.id')
            ->join('flights__numbers', 'passengers__flights.flight_number_id', '=', 'flights__numbers.id')
            ->orderBy('date')
            ->groupBy('date')
            ->get();
        return $data;
    }

    public static function getParentFlights($date_from = NULL, $date_to = NULL)
    {
        $sql = "";
        if (isset($date_from, $date_to))
        {
            $sql = "(DATE(std) BETWEEN '$date_from' AND '$date_to') AND ";
        }
        elseif (isset($date_from))
        {
            $sql = "DATE(std) = '$date_from' AND ";
        }

        $flights = Flight::selectRaw('s_flights.id as flight_id, s_flights__numbers.id as flight_number_id, flight_number, departure_airport, arrival_airport, DATE(std) as departure_date')
            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->whereRaw("$sql bound = '0' AND NOT ((departure_airport = 'DYU' AND arrival_airport = 'LBD') OR (departure_airport = 'LBD' AND arrival_airport = 'DYU'))")
            ->get();

        return $flights;
    }

    /**
     * Get Parent FLight Number By Child FLight Number
     * @param $flightNumberId
     * @return array
     */
    public static function getParentFlightNumber($flightNumberId)
    {
        if ($flightNumberId) {
            $flightNumber = FlightNumber::find($flightNumberId);
            if ($flightNumber){
                $match = preg_split('/(?<=\d)(?=[a-zA-Z])/i', $flightNumber->flight_number);
                if (count($match)) {
                    // match[0] has Digit Part Of Flight Number, i.e. 101
                    $match[0]--;
                    // match[1] has String Part Of Flight Number If Exists, i.e. if 101Y -> Y
                    $parentFltNum = $match[0] . (isset($match[1]) ? $match[1] : '');
                    // Search Parent Flight Number
                    $parentFlightNumber = FlightNumber::where('flight_number', $parentFltNum)
                        ->where('departure_airport', $flightNumber->arrival_airport)
                        ->where('arrival_airport', $flightNumber->departure_airport)
                        ->whereNull("deleted_at")
                        ->first(['id']);

                    if ($parentFlightNumber) {
                        return ["success" => TRUE,
                            "message" => "Parent Flight Number Found",
                            "flightNumberId" => $parentFlightNumber->id
                        ];
                    }
                }
            }

        }

        return [  "success" => FALSE,
            "message" => "Parent Flight Number Not Found"
        ];
    }


    /**
     * Get Parent Flight Id
     * @param $flightNumberId
     * @param $departureDate
     * @return bool|mixed
     */
    public static function getParentFlight($flightNumberId, $departureDate)
    {
        if ($flightNumberId && $departureDate) {
            $flightNumber = FlightNumber::find($flightNumberId, ['flight_number']);
            if ($flightNumber){
                $match = preg_split('/(?<=\d)(?=[a-zA-Z])/i', $flightNumber->flight_number);
                if (count($match)) {
                    // match[0] has Digit Part Of Flight Number, i.e. 101
                    $match[0]--;
                    // match[1] has String Part Of Flight Number If Exists, i.e. if 101Y -> Y
                    $parentFltNum = $match[0] . (isset($match[1]) ? $match[1] : '');
                    // Search Parent Flight Number
                    $parentFlightNumber = FlightNumber::where('flight_number', $parentFltNum)
                        ->whereNull("deleted_at")
                        ->first(['id']);

                    if ($parentFlightNumber) {
                        // Previous Date In Case Parent Was Not Found In The Same Date As Child Flight
                        $previousDate = date('Y-m-d', strtotime('- 1 day', strtotime($departureDate)));
                        // Search For Parent FLight
                        $flight = Flight::where('flight_number_id', $parentFlightNumber->id)
                            ->whereBetween(DB::raw('DATE(std)'), [$previousDate, $departureDate])
                            ->orderBy('std', 'DESC')
                            ->first(['flights.id AS id']);

                        if ($flight) {
                            return ["success" => TRUE,
                                "message" => "Parent Flight Found",
                                "parentId" => $flight->id
                            ];
                        }
                    }
                }
            }

        }

        return [  "success" => FALSE,
            "message" => "Parent Flight Not Found"
        ];
    }


    public static function getChildFlight($flight_num_id, $departure_date)
    {
        $flight = Flight::select('flights.id')
            ->leftJoin('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->whereRaw("s_flights__numbers.id = '$flight_num_id' AND DATE(std) = '$departure_date'")
            ->first();

        return $flight;
    }

    /**
     * Get parents flight for today and yesterday fot exact flight ID
     * @param $flightNumber
     * @return \Illuminate\Database\Eloquent\Collection|static[]
     */
    public static function flightParents($flightNumber)
    {
        $select = [
            "id",
            "std",
            "etd",
            "atd",
            "abn",
            "sta",
            "eta",
            "ata"
        ];
        array_walk($select, function (&$value, $index)
        {
            $value = "flights." . $value;
        });
        $parentFlights = Flight::select($select)
            ->join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
            ->whereRaw("DATE(std) >= DATE_ADD(CURDATE(), INTERVAL - 1 DAY) AND DATE(std) <= CURDATE()")
            ->where("flights__numbers.flight_number", $flightNumber - 1)
            ->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at")
            ->orderBy("flights.std", "DESC")
            ->get();
        if ($parentFlights->count())
        {
            $result = [
                "message" => "Parent flights were successfully loaded",
                "success" => TRUE
            ];
            foreach ($parentFlights as $parentFlight)
            {
                $result["flights"][] = [
                    "id"    => $parentFlight->id,
                    "label" => ProjectConstant::getConstants(IATA_CODE) . ($flightNumber - 1) . " " . $parentFlight->std
                ];
            }
        }
        else
        {
            $result = [
                "success" => TRUE,
                "message" => "Not found any parent flights for selected flight number"
            ];
        }
        return $result;
    }

    /**
     * Get flights for range
     * @param $startDate
     * @param $endDate
     * @param array $relativeModels
     * @param bool $onlyParents
     * @param bool $orderBy
     * @param bool $flightNumberIds
     * @param bool $addDeleted
     * @return mixed
     */
    public static function flightsRange($startDate, $endDate, $relativeModels = [
        "flightNumber",
        "aircraft",
        "flightNumber.departureAirport",
        "flightNumber.arrivalAirport",
    ], $onlyParents = FALSE, $orderBy = FALSE, $flightNumberIds = FALSE, $addDeleted = FALSE)
    {
        $flightCrewRelative = FALSE;
        if (($key = array_search("flightCrew", $relativeModels)) !== FALSE)
        {
            unset($relativeModels[$key]);
            $flightCrewRelative = TRUE;
        }
        $flights = Flight::with($relativeModels)
            ->select([
                "flights.id",
                "flights.aircraft_id",
                "flights.aircraft_type_id",
                "flights.departure_airport_id",
                "flights.arrival_airport_id",
                "flights.diverted_airport_id",
                "flights.flight_number_id",
//                "flights.flights_delay_id",
//                "flights.delays",
                "flights.parent_id",
                "flights.std",
                "flights.ptd",
                "flights.etd",
                "flights.atd",
                "flights.abn",
                "flights.sta",
                "flights.pta",
                "flights.eta",
                "flights.ata",
                "flights.tdn",
                "flights.capacity_a",
                "flights.capacity_c",
                "flights.capacity_w",
                "flights.capacity_y",
                "flights.pax_a_booked",
                "flights.pax_c_booked",
                "flights.pax_w_booked",
                "flights.pax_y_booked",
                "flights.pax_inf_booked",
                "flights.pax_a_actual",
                "flights.pax_c_actual",
                "flights.pax_w_actual",
                "flights.pax_y_actual",
                "flights.pax_inf_actual",
                "flights.baggage",
                "flights.cargo",
                "flights.mail",
                "flights.deportees",
                "flights.in_tanks",
                "flights.utilised",
                "flights.uplifted",
                "flights.defuel",
                "flights.gate_parking",
                "flights.is_diversion",
                "flights.diversion",
                "flights.cancelled_at",
                "flights.deleted_at",
                "childFlight.id as child_id",
                "childFlight.deleted_at as child_deleted_at",
//                DB::raw('(select id from s_flights AS s_f where s_flights.id  = s_f.id limit 1) as child_id')
            ])
            ->leftJoin("flights as childFlight", "flights.id", "=", "childFlight.parent_id");
//            ->whereNull("childFlight.deleted_at");

        // Include Deleted if set to TRUE
        if (!$addDeleted){
            $flights->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at");
        }

        $flights->whereRaw("((s_flights.std BETWEEN \"$startDate\" AND \"$endDate\") OR (s_flights.atd BETWEEN \"$startDate\" AND \"$endDate\"))");

        if ($onlyParents) {
            $flights->where(function($sql){
                $sql->whereNull("flights.parent_id")
                    ->orWhere("flights.parent_id", "0");
            });
        }

        if ($flightNumberIds && count($flightNumberIds)) {
            $flights->whereIn("flights.flight_number_id", $flightNumberIds);
        }

        if ($orderBy) {
            $flights->orderBy($orderBy);
        }

        $flights = $flights->orderBy('flights.std')->get();

        if ($flightCrewRelative)
        {
            if ($flights->count())
            {
                foreach ($flights as $key => $flight)
                {
                    $flightCrew = FlightCrew::select([
                        DB::raw("CONCAT(s_users.first_name , ' ', s_users.last_name) AS full_name"),
                        "flights__crew.user_id",
                        "flight_id",
                        "flights__crew.position_id",
                        'structure__positions.name as position_name',
                        'structure__positions.type as position_type',
                        "position_order",
                        "is_standby",
                        "is_dhc",
                        "is_sup",
                        "users.first_name",
                        "users.last_name",
                        "users.id"
                    ])
                        ->join("users", "users.id", "=", "flights__crew.user_id")
                        ->join('users__departments', 'users.id', '=', 'users__departments.user_id')
                        ->join('structure__positions', 'structure__positions.id', '=', 'users__departments.position_id')
                        ->where("flight_id", $flight->id)
                        ->get();

                    $flights[$key]->flightCrewItems = $flightCrew;
                }
            }
        }

        $processedFlights = [];
        $flightsChecked = [];
        foreach ($flights as $key => $flight) {
            if (in_array($flight->id, $processedFlights)){
                continue;
            }
            // Remove Child Flight if It's deleted
            if ($flight->child_deleted_at){
                $flight->child_id = null;
                $flight->flightChild = null;
            }
            $flightsChecked[] = $flight;
            $processedFlights[] = $flight->id;
        }

        return $flightsChecked;
    }


    public static function handlingFlightsRange($startDate = null, $endDate = null, $airlines = null, $airports = null, $flightNumbers = null,
                                                $options = [], $flightComRelations = false, $includeCancelled = false, $flightCrewRelations = false, $otherRelations = [],
                                                $orderByTime = true){

//        $startDate = "2016-10-30";
//        $endDate = "2016-10-31";

        $relations = [
            "message",
            "station",
            "divertedAirport",
            "departureAirport",
            "arrivalAirport",
            "aircraft",
            "aircraft.type",
            "aircraftType",
            "flightNumber",
            "flightNumber.airline",
            "flightNumber.departureAirport",
            "flightNumber.arrivalAirport",
            "delays",
            "delays.delay",
            "ssr",
        ];

        //
        if ($flightComRelations){
            $relations[] = "flightCom";
            $relations[] = "flightChild";
        }

        if ($flightCrewRelations){
            $relations[] = "flightCrew";
        }

        if (count($otherRelations)){
            foreach ($otherRelations as $each) {
                $relations[] = $each;
            }
        }

        $flights = Flight::with($relations)
            ->join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id");

        if ($airlines){
            $flights->join("airlines", "flights__numbers.airline_id", "=", "airlines.id");
        }

        $flights->whereNull("flights.deleted_at");

        if (!$includeCancelled){
            $flights->whereNull("flights.cancelled_at");
        }

        $start = date("Y-m-d", strtotime("- 1 day", strtotime($startDate)));
        $end   = date("Y-m-d", strtotime("+ 1 day", strtotime($endDate)));

        if (env(CARRIER) == AIRLINE){
            if ($startDate) {
                $flights->where(function ($sql2) use ($startDate, $endDate) {
//                    $sql->where(function ($sql2) use ($startDate, $endDate) {
                    $sql2->whereBetween("std", [$startDate, $endDate])
                        ->orWhereBetween("ptd", [$startDate, $endDate])
                        ->orWhereBetween("etd", [$startDate, $endDate])
                        ->orWhereBetween("atd", [$startDate, $endDate]);
//                    });
//                    ->orWhere(function ($sql2) use ($startDate, $endDate) {
//                        $sql2->whereBetween("sta", [$startDate, $endDate])
//                            ->orWhereBetween("pta", [$startDate, $endDate])
//                            ->orWhereBetween("eta", [$startDate, $endDate])
//                            ->orWhereBetween("ata", [$startDate, $endDate]);
//                    });
                });
            }
            else {
                $flights->where(function ($sql2) use ($endDate) {
//                    $sql->where(function ($sql2) use ($endDate) {
                    $sql2->where( "std", "<=", $endDate)
                        ->orWhere("ptd", "<=", $endDate)
                        ->orWhere("etd", "<=", $endDate)
                        ->orWhere("atd", "<=", $endDate);
//                    });
//                    ->orWhere(function ($sql2) use ($endDate) {
//                        $sql2->where( "sta", "<=", $endDate)
//                            ->orWhere("pta", "<=", $endDate)
//                            ->orWhere("eta", "<=", $endDate)
//                            ->orWhere("ata", "<=", $endDate);
//                    });
                });
            }
        }
        // HANDLING
        else {
            if (isset($options['arrival_date_search'])) {
                $flights->whereBetween("arrival_date", [$start, $end]);
                if ($startDate) {
                    $flights->where(function ($sql) use ($startDate, $endDate) {
                        $sql->whereBetween("sta", [$startDate, $endDate])
                            ->orWhereBetween("pta", [$startDate, $endDate])
                            ->orWhereBetween("eta", [$startDate, $endDate])
                            ->orWhereBetween("ata", [$startDate, $endDate]);
                    });

                }
                else {
                    $flights->where(function ($sql) use ($endDate) {
                        $sql->where("sta", "<=", $endDate)
                            ->orWhere("pta", "<=", $endDate)
                            ->orWhere("eta", "<=", $endDate)
                            ->orWhere("ata", "<=", $endDate);
                    });
                }
            }
            else {
                $flights->whereBetween("departure_date", [$start, $end]);
                if ($startDate) {
                    $flights->where(function ($sql) use ($startDate, $endDate) {
                        $sql->whereBetween("std", [$startDate, $endDate])
                            ->orWhereBetween("ptd", [$startDate, $endDate])
                            ->orWhereBetween("etd", [$startDate, $endDate])
                            ->orWhereBetween("atd", [$startDate, $endDate]);
                    });
                }
                else {
                    $flights->where(function ($sql) use ($endDate) {
                        $sql->where("std",   "<=", $endDate)
                            ->orWhere("ptd", "<=", $endDate)
                            ->orWhere("etd", "<=", $endDate)
                            ->orWhere("atd", "<=", $endDate);
                    });
                }
            }
        }

        if ($airports){
            $airports = is_array($airports) ? $airports : [$airports];

            if (count($airports)) {
                if (isset($options['only_departure_airports'])) {
                    $flights->where(function($sql) use ($airports){
                        $sql->whereIn("flights__numbers.departure_airport_id", $airports)
                            ->orWhereIn("flights.departure_airport_id", $airports);
                    });
                } else {
                    $flights->where(function ($sql) use ($airports) {
                        $sql->whereIn("flights__numbers.departure_airport_id", $airports)
                            ->orWhereIn("flights__numbers.arrival_airport_id", $airports)
                            ->orWhereIn("flights.departure_airport_id", $airports)
                            ->orWhereIn("flights.arrival_airport_id", $airports);
                    });
                }
            }
        }

        if ($airlines){
            if (is_array($airlines)){
                $flights->whereIn("flights__numbers.airline_id", $airlines);
            }
            else {
                $flights->where("flights__numbers.airline_id", $airlines);
            }
        }

        if ($flightNumbers && count($flightNumbers)) {
            if (is_array($flightNumbers)){
                $flights->whereIn("flights.flight_number_id", $flightNumbers);
            }
            else {
                $flights->where("flights.flight_number_id", $flightNumbers);
            }
        }

        if (!isset($options["off_order_by_airline_id"])){
            if ($airlines){
                $flights->orderBy("airlines.airline");
            }
//            else {
//                $flights->orderBy("airline_id");
//            }
//                    ->orderBy("flights__numbers.departure_airport_id");
//                    ->orderBy("flights__numbers.arrival_airport_id");
        }

        if ($orderByTime){
            return $flights->orderBy("std")
                ->orderBy("atd")
                ->orderBy("parent_id")
                ->get([
                    "flights.*"
                ]);
        }
        else {
            return $flights->get([
                "flights.*"
            ]);
        }
    }

    /**
     * Unserialize Flight Diversion If Exists
     * @param $flight
     */
    public static function getDiversionUnserialized(&$flight){
        if (is_object($flight) && $flight && $flight->is_diversion && !is_array($flight->diversion)){
            $diversion = unserialize($flight->diversion);
            $diversionAirport = City::find((int)$diversion['diversion_airport']);
            $diversion['diversion_airport_code'] = $diversionAirport->iata;
            $flight->diversion = $diversion;
        }
    }


    /**
     * Calculate Baggage, Cargo, Mail Revenue
     * @param $flight
     */
    public static function getBaggageCargoMail(&$flight){
        if (is_object($flight) && $flight) {
            $flight->allowance_baggage = $flight->pax_c_actual * C_BAGGAGE_WEIGHT + $flight->pax_y_actual * Y_BAGGAGE_WEIGHT;
            $flight->excess_baggage = (((int)($flight->baggage) - $flight->allowance_baggage) < 0) ? 0 : ((int)($flight->baggage) - $flight->allowance_baggage);
            $flight->excess_baggage_revenue = $flight->excess_baggage * EXCESS_BAGGAGE_RATE;
            $flight->cargo_revenue = ($flight->cargo) ? (int)($flight->cargo) * CARGO_RATE : 0;
            $flight->mail_revenue = ($flight->mail) ? (int)($flight->mail) * MAIL_RATE : 0;
            $flight->total_revenue = (int)($flight->cargo_revenue) + (int)($flight->mail_revenue) + (int)($flight->excess_baggage_revenue);
        }
    }


    /**
     * Put Unserialized Proper Defined Array in FLight->delays (rewrite)
     * @param $flight
     */
    public static function getDelaysUnserialized(&$flight){
        if (is_object($flight) && $flight->delays)
        {
            $flight->delays = unserialize($flight->delays);
            // Delays
            $delays = [];
            foreach ($flight->delays as $each) {
                if ($each) {
                    $delay = Delay::find($each['id']);
                    $delays[] = 'DL '.$delay->code . ' / ' . $delay->description;
                }
            }
            if (count($delays)){
                $flight->delays = $delays;
            }
        }
    }

    /**
     * Get flight user
     * @param $flightId
     * @param $userId
     * @param int $type
     * @return array|\Illuminate\Database\Eloquent\Model|null|static
     */
    public static function getFlightUser($flightId, $userId, $type = T_OBJECT_CAST)
    {
        $flightCrewUser = FlightCrew::select([
            "users.first_name",
            "users.last_name",
            "flights__crew.*",
            "structure__positions.name"
        ])
            ->join("users", "users.id", "=", "flights__crew.user_id")
            ->join("structure__positions", "structure__positions.id", "=", "flights__crew.position_id")
            ->where("flight_id", $flightId)
            ->where("user_id", $userId)
            ->first();

        return ($type == T_OBJECT_CAST) ? $flightCrewUser : $flightCrewUser->toArray();
    }

    /**
     * Get user roster
     * @param $userId
     * @param $period
     * @param string $order
     * @param bool|false $type
     * @return mixed
     */
    public static function getUserRoster($userId, $period, $order = 'ASC', $type = false)
    {
        $startDate = $period["from"];
        $endDate   = $period["to"];

        $userFlights = Flight::with([
            "flightNumber",
            "flightNumber.airline",
            "flightNumber.departureAirport",
            "flightNumber.arrivalAirport",
        ])
            ->select([
                "flights.id",
                "flights.parent_id",
                "flights.aircraft_id",
                "flights.flight_number_id",
//            "flights.flights_delay_id",
                "flights.std",
                "flights.etd",
                "flights.atd",
                "flights.abn",
                "flights.sta",
                "flights.eta",
                "flights.tdn",
                "flights.ata",
                "aircraft.name AS aircraft_name",
                "flights__numbers.flight_number",
                "depAirport.iata AS departure_airport",
                "arrAirport.iata AS arrival_airport",
                "flights__crew.user_id",
                "flights__crew.position_id",
                "flights__crew.position_order",
                "flights__crew.is_standby",
                "flights__crew.is_dhc"
            ])
            ->join("flights__crew", "flights__crew.flight_id", "=", "flights.id")
            ->join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
            ->leftJoin("aircraft", "aircraft.id", "=", "flights.aircraft_id")
            ->leftjoin('airports AS depAirport', 'flights__numbers.departure_airport_id', '=', 'depAirport.id')
            ->leftjoin('airports AS arrAirport', 'flights__numbers.arrival_airport_id', '=', 'arrAirport.id')
            ->where("flights__crew.user_id", "=", $userId)
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at")
            ->whereRaw("((s_flights.std BETWEEN \"$startDate\" AND \"$endDate\"))");

        if ($type){
            switch($type){
                case CREW_DHC:
                    $userFlights->where("is_dhc", true);
                    break;
                case CREW_STANDBY:
                    $userFlights->where("is_standby", true);
                    break;
            }
        }

        $userFlights->orderBy('std', $order);

        return $userFlights->get();
    }


    /**
     * Get Grouped in one Parent And Child Flights With Aircraft And Flight Number Information
     * @param $from
     * @param $to
     * @return array
     */
    public static function FlightsGroupedByFlightNumber($from, $to){

        $group_flights = [];
        $inbound_flight = Flight::with(['flightNumber','aircraft'])
            ->whereBetween("std", [$from, $to])
//            ->where('parent_id','!=',0)
            ->orderBy('flight_number_id')
            ->orderBy('std')
            ->get([
                'flights.id as id',
                'parent_id',
                'is_diversion',
                'diversion',
                'flight_number_id',
                'aircraft_id',
                'std',
                'sta',
                'pax_a_booked',
                'pax_c_booked',
                'pax_w_booked',
                'pax_y_booked',
                'capacity_a',
                'capacity_c',
                'capacity_w',
                'capacity_y',
                'cancelled_at',
                'deleted_at',
                'closed_sales',
                'remark'
            ])
            ->toArray();

        $flight_number_ids = [];
        $group_flights_by_flightnumber = [];

        // Roundtrip Flights
        foreach($inbound_flight as $i => $flight){
            if (is_null($flight['capacity_c']))
                $inbound_flight[$i]['capacity_c'] = $flight['aircraft']['config_c'];

            if (is_null($flight['capacity_y']))
                $inbound_flight[$i]['capacity_y'] = $flight['aircraft']['config_y'];

            if ($flight['is_diversion'] == 1){
                /*$diversion = unserialize($flight['diversion']);
                if ($diversion['diversion_sta'] != '' && $diversion['diversion_sta'] != '0000-00-00 00:00')
                    $diversion_requirement['sta'] = date('Y-m-d H:i', strtotime($diversion['diversion_sta']));
                if ($diversion['diversion_std'] != '' && $diversion['diversion_std'] != '0000-00-00 00:00')
                    $diversion_requirement['std'] = date('Y-m-d H:i', strtotime($diversion['diversion_std']));
                if (is_numeric($diversion['diversion_airport'])) {
                    $city = Airport::join('cities', 'airports.city','=','cities.city')
                        ->where('cities.id',$diversion['diversion_airport'])
                        ->first(['airports.iata as city']);

                    $diversion_requirement['airport'] = $city->city;
                }
                $inbound_flight[$i]['diversion'] = $diversion_requirement;*/
            }

            $flight_number_ids[] = $flight['flight_number_id'];

            $outbound_flight = Flight::with(['flightNumber','aircraft'])
                ->find($flight['parent_id'],[
                    'flights.id as id',
                    'parent_id',
                    'is_diversion',
                    'flight_number_id',
                    'diversion',
                    'aircraft_id',
                    'std',
                    'sta',
                    'pax_a_booked',
                    'pax_c_booked',
                    'pax_w_booked',
                    'pax_y_booked',
                    'capacity_a',
                    'capacity_c',
                    'capacity_w',
                    'capacity_y',
                    'cancelled_at',
                    'deleted_at',
                    'closed_sales',
                    'remark']);

            if (count($outbound_flight)) {
                $outbound_flight = $outbound_flight->toArray();
                if (is_null($outbound_flight['capacity_c']))
                    $outbound_flight['capacity_c'] = $flight['aircraft']['config_c'];

                if (is_null($outbound_flight['capacity_y']))
                    $outbound_flight['capacity_y'] = $flight['aircraft']['config_y'];

                $group_flights[$i][] = $outbound_flight;
                $index = $outbound_flight['flight_number']['flight_number'];

                $flight_number_ids[] = $outbound_flight['flight_number_id'];
            }
            else
                $index = $flight['flight_number']['flight_number'];

            $group_flights[$i][] = $inbound_flight[$i];

            if (isset($group_flights_by_flightnumber[$index]))
                $group_flights_by_flightnumber[$index][] = $group_flights[$i];
            else
                $group_flights_by_flightnumber[$index] = [$group_flights[$i]];

        }

        // One Way Flights
        $outbound_flight = Flight::with(['flightNumber','aircraft'])
            ->whereBetween("std", [$from, $to])
            ->whereNotIn('flight_number_id', $flight_number_ids)
            ->where('parent_id',0)
            ->orderBy('std')
            ->get([
                'flights.id as id',
                'parent_id',
                'is_diversion',
                'diversion',
                'flight_number_id',
                'aircraft_id',
                'std',
                'sta',
                'capacity_a',
                'capacity_c',
                'capacity_w',
                'capacity_y',
                'pax_a_booked',
                'pax_c_booked',
                'pax_w_booked',
                'pax_y_booked',
                'cancelled_at',
                'deleted_at',
                'closed_sales',
                'remark'])
            ->toArray();

        foreach ($outbound_flight as $j=>$flight){
            if (is_null($flight['capacity_c']))
                $outbound_flight[$j]['capacity_c'] = $flight['aircraft']['config_c'];

            if (is_null($flight['capacity_y']))
                $outbound_flight[$j]['capacity_y'] = $flight['aircraft']['config_y'];

            $index = $flight['flight_number']['flight_number'];
            if (isset($group_flights_by_flightnumber[$index]))
                $group_flights_by_flightnumber[$index][] = [$flight];
            else
                $group_flights_by_flightnumber[$index] = [[$flight]];
        }

        // order by Key (Flight Number)
        if ($group_flights_by_flightnumber) {
            ksort($group_flights_by_flightnumber);
        }

        return $group_flights_by_flightnumber;
    }


    public static function Flights_Grouped_ByFlightNumber_Pax_Capacity($from, $to){

        $group_flights = [];
        $inbound_flight = Flight::with(['flightNumber','aircraft'])
            ->whereRaw("DATE(std) BETWEEN '$from' AND '$to' ")
            ->where('parent_id','!=',0)
            ->orderBy('std')
            ->get([
                'flights.id as id',
                'parent_id',
                'is_diversion',
                'diversion',
                'flight_number_id',
                'aircraft_id',
                'std',
                'sta',
                'pax_a_booked',
                'pax_c_booked',
                'pax_w_booked',
                'pax_y_booked',
                'capacity_a',
                'capacity_c',
                'capacity_w',
                'capacity_y',
                'closed_sales',
                'remark',
                'cancelled_at',
                'deleted_at'])
            ->toArray();
        $flight_number_ids = [];

        // Roundtrip Flights
        foreach($inbound_flight as $i=>$flight){
            if (is_null($flight['capacity_c']))
                $inbound_flight[$i]['capacity_c'] = $flight['aircraft']['config_c'];

            if (is_null($flight['capacity_y']))
                $inbound_flight[$i]['capacity_y'] = $flight['aircraft']['config_y'];

            if ($inbound_flight[$i]['capacity_c'] + $inbound_flight[$i]['capacity_y'] == 0)
                $inbound_flight[$i]['sf'] = 0;
            else
                $inbound_flight[$i]['sf'] = round(($flight['pax_c_booked'] + $flight['pax_y_booked'])*100 / ($inbound_flight[$i]['capacity_c'] + $inbound_flight[$i]['capacity_y']));

            if ($flight['is_diversion'] == 1){
                $diversion = unserialize($flight['diversion']);
                if ($diversion['diversion_sta'] != '' && $diversion['diversion_sta'] != '0000-00-00 00:00')
                    $diversion_requirement['sta'] = date('Y-m-d H:i', strtotime($diversion['diversion_sta']));
                if ($diversion['diversion_std'] != '' && $diversion['diversion_std'] != '0000-00-00 00:00')
                    $diversion_requirement['std'] = date('Y-m-d H:i', strtotime($diversion['diversion_std']));
                if (is_numeric($diversion['diversion_airport'])) {
                    $city = Airport::join('cities', 'airports.city','=','cities.city')
                        ->where('cities.id',$diversion['diversion_airport'])
                        ->first(['airports.iata as city']);

                    $diversion_requirement['airport'] = $city->city;
                }
                $inbound_flight[$i]['diversion'] = $diversion_requirement;
            }

            $flight_number_ids[] = $flight['flight_number_id'];

            $outbound_flight = Flight::with(['flightNumber','aircraft'])
                ->find($flight['parent_id'],
                    [
                        'flights.id as id',
                        'parent_id',
                        'is_diversion',
                        'flight_number_id',
                        'diversion',
                        'aircraft_id',
                        'std',
                        'sta',
                        'pax_a_booked',
                        'pax_c_booked',
                        'pax_w_booked',
                        'pax_y_booked',
                        'capacity_a',
                        'capacity_c',
                        'capacity_w',
                        'capacity_y',
                        'closed_sales',
                        'remark',
                        'cancelled_at',
                        'deleted_at']);

            if (count($outbound_flight)) {
                $outbound_flight = $outbound_flight->toArray();
                if (is_null($outbound_flight['capacity_c']))
                    $outbound_flight['capacity_c'] = $flight['aircraft']['config_c'];

                if (is_null($outbound_flight['capacity_y']))
                    $outbound_flight['capacity_y'] = $flight['aircraft']['config_y'];

                if ($outbound_flight['capacity_c'] + $outbound_flight['capacity_y'] == 0)
                    $outbound_flight['sf'] = 0;
                else
                    $outbound_flight['sf'] = round(($flight['pax_c_booked'] + $flight['pax_y_booked'])*100 / ($outbound_flight['capacity_c'] + $outbound_flight['capacity_y']));

                $group_flights[$i][] = $outbound_flight;
                $index = $outbound_flight['flight_number']['flight_number'];

                $flight_number_ids[] = $outbound_flight['flight_number_id'];
            }
            else
                $index = $flight['flight_number']['flight_number'];

            $group_flights[$i][] = $inbound_flight[$i];

            if (isset($group_flights_by_flightnumber[$index]))
                $group_flights_by_flightnumber[$index] = $group_flights[$i];
            else
                $group_flights_by_flightnumber[$index] = $group_flights[$i];

        }

        // One Way Flights
        $outbound_flight = Flight::with(['flightNumber','aircraft'])
            ->whereRaw("DATE(std) BETWEEN '$from' AND '$to' ")
            ->whereNotIn('flight_number_id', $flight_number_ids)
            ->where('parent_id',0)
            ->orderBy('flight_number_id')
            ->get([
                'flights.id as id',
                'parent_id',
                'is_diversion',
                'diversion',
                'flight_number_id',
                'aircraft_id',
                'std',
                'sta',
                'capacity_a',
                'capacity_c',
                'capacity_w',
                'capacity_y',
                'pax_a_booked',
                'pax_c_booked',
                'pax_w_booked',
                'pax_y_booked',
                'closed_sales',
                'remark',
                'cancelled_at',
                'deleted_at'
            ])
            ->toArray();

        foreach ($outbound_flight as $j=>$flight){
            if (is_null($flight['capacity_c']))
                $outbound_flight[$j]['capacity_c'] = $flight['aircraft']['config_c'];

            if (is_null($flight['capacity_y']))
                $outbound_flight[$j]['capacity_y'] = $flight['aircraft']['config_y'];

            if ($outbound_flight[$j]['capacity_c'] + $outbound_flight[$j]['capacity_y'] == 0)
                $outbound_flight[$j]['sf'] = 0;
            else
                $outbound_flight[$j]['sf'] = round(($flight['pax_c_booked'] + $flight['pax_y_booked'])*100 / ($outbound_flight[$j]['capacity_c'] + $outbound_flight[$j]['capacity_y']));

            $index = $flight['flight_number']['flight_number'];
            if (isset($group_flights_by_flightnumber[$index]))
                $group_flights_by_flightnumber[$index] = [$outbound_flight[$j]];
            else
                $group_flights_by_flightnumber[$index] = [$outbound_flight[$j]];
        }

        // order by Key (Flight Number)
        ksort($group_flights_by_flightnumber);

        $total = [];
        foreach($group_flights_by_flightnumber as $flt_no => $route){
            $total[$flt_no]['capacity_c'] = 0;
            $total[$flt_no]['capacity_y'] = 0;
            $total[$flt_no]['pax_c_booked'] = 0;
            $total[$flt_no]['pax_y_booked'] = 0;
            foreach ($route as $flight){
                $total[$flt_no]['capacity_c'] += $flight['capacity_c'];
                $total[$flt_no]['capacity_y'] += $flight['capacity_y'];
                $total[$flt_no]['pax_c_booked'] += $flight['pax_c_booked'];
                $total[$flt_no]['pax_y_booked'] += $flight['pax_y_booked'];
            }
            if ($total[$flt_no]['capacity_c'] + $total[$flt_no]['capacity_y'] == 0)
                $total[$flt_no]['sf'] = 0;
            else
                $total[$flt_no]['sf'] = round(($total[$flt_no]['pax_c_booked'] + $total[$flt_no]['pax_y_booked'])*100 / ($total[$flt_no]['capacity_c'] + $total[$flt_no]['capacity_y']));

        }

        return [$group_flights_by_flightnumber, $total];
    }


    public static function Pax_Bound_Std($from, $to = null, $flight_numbers = null){
        $to = isset($to) ? $to : $from;
        $group = 'departure_date';
        $pax_by_std = Flight::join('aircraft', 'flights.aircraft_id','=','aircraft.id')
            ->join('flights__numbers','flights.flight_number_id','=','flights__numbers.id')
            ->selectRaw('DATE(std) AS departure_date, flight_number,
                        SUM(IF(bound = "0", capacity_c,0)) as capacity_c_outbound, SUM(IF(bound = "0", capacity_y,0)) as capacity_y_outbound,
                        SUM(IF(bound = "1", capacity_c,0)) as capacity_c_inbound, SUM(IF(bound = "1", capacity_y,0)) as capacity_y_inbound,
                        SUM(IF(bound = "0", IF(pax_c_actual + pax_y_actual = 0, pax_c_booked, pax_c_actual),0)) as pax_c_booked_outbound,
                        SUM(IF(bound = "0", IF(pax_c_actual + pax_y_actual = 0, pax_y_booked, pax_y_actual),0)) as pax_y_booked_outbound,
                        SUM(IF(bound = "1", IF(pax_c_actual + pax_y_actual = 0, pax_c_booked, pax_c_actual),0)) as pax_c_booked_inbound,
                        SUM(IF(bound = "1", IF(pax_c_actual + pax_y_actual = 0, pax_y_booked, pax_y_actual),0)) as pax_y_booked_inbound,
                        SUM(capacity_c) as capacity_c, SUM(capacity_y) as capacity_y, SUM(pax_c_booked) as pax_c_booked, SUM(pax_y_booked) as pax_y_booked')
            ->whereRaw("DATE(std) BETWEEN '$from' AND '$to' ");

        if (isset($flight_numbers) && count($flight_numbers))
            $pax_by_std->whereIn('flight_number',$flight_numbers);

        $pax_by_std = $pax_by_std->groupBy($group)
            ->get()
            ->toArray();

        foreach ($pax_by_std as $i=>$each) {
            if (($each['capacity_c_outbound'] + $each['capacity_y_outbound']) == 0)
                $pax_by_std[$i]['sf_outbound'] = 0;
            else
                $pax_by_std[$i]['sf_outbound'] = round(($each['pax_c_booked_outbound'] + $each['pax_y_booked_outbound']) * 100/($each['capacity_c_outbound'] + $each['capacity_y_outbound']));

            if (($each['capacity_c_inbound'] + $each['capacity_y_inbound']) == 0)
                $pax_by_std[$i]['sf_inbound'] = 0;
            else
                $pax_by_std[$i]['sf_inbound'] = round(($each['pax_c_booked_inbound'] + $each['pax_y_booked_inbound']) * 100/($each['capacity_c_inbound'] + $each['capacity_y_inbound']));
            if (($each['capacity_c'] + $each['capacity_y']) == 0)
                $pax_by_std[$i]['sf'] = 0;
            else
                $pax_by_std[$i]['sf'] = round(($each['pax_c_booked'] + $each['pax_y_booked']) * 100/($each['capacity_c'] + $each['capacity_y']));
        }

        return $pax_by_std;
    }


    /**
     * Check, Correct And Save FLight's Seat Capacity
     * @param $flight
     */
    public static function checkAndCorrectFlightsSeatCapacity(&$flight){
        if (is_object($flight) && !$flight->capacity_y) {
//            $flightNumber = $flight->flightNumber->flight_number;
            $ac = $flight->aircraft;

            /*
            if (in_array($flightNumber, [101, 102])) {
                $flight->capacity_c = 16;
                $flight->capacity_y = 130;
            } elseif (in_array($flightNumber, [103, 104, 203, 204, 219, 220, 251, 252])) {
                $flight->capacity_c = $ac->config_c;
                $flight->capacity_y = $ac->config_y - 1;
            } else {
            */
            if ($ac){
                $flight->capacity_c = $ac->config_c;
                $flight->capacity_y = $ac->config_y;
                $flight->save();
            }
            //}
        }
    }

    /**
     * @param $groupedFlights
     * @param null $totals
     */
    public static function calculateSeatFactors(&$groupedFlights, &$totals = null){
        if ($groupedFlights && is_array($groupedFlights)){
            foreach ($groupedFlights as $std => $flight) {
                // $type = ['total' , 'outbound', 'inbound']
                foreach ($flight as $type => $each) {
                    $sf = ($each['capacity']['c'] + $each['capacity']['y']) ? round( ($each['pax']['c'] + $each['pax']['y']) * 100 / ($each['capacity']['c'] + $each['capacity']['y']) ) : 0;
                    $groupedFlights[$std][$type]['sf'] = $sf > 100 ? 100 : $sf;
                }
            }

            ksort($groupedFlights);
        }

        if ($totals){
            $sf = ($totals["outbound"]['capacity']['c'] + $totals["outbound"]['capacity']['y']) ? round( ($totals["outbound"]['pax']['c'] + $totals["outbound"]['pax']['y']) * 100 / ($totals["outbound"]['capacity']['c'] + $totals["outbound"]['capacity']['y']) ) : 0;
            $totals["outbound"]['sf']   = $sf > 100 ? 100 : $sf;

            $sf = ($totals["inbound"]['capacity']['c'] + $totals["inbound"]['capacity']['y'])   ? round( ($totals["inbound"]['pax']['c'] + $totals["inbound"]['pax']['y']) * 100 / ($totals["inbound"]['capacity']['c'] + $totals["inbound"]['capacity']['y']) ) : 0;
            $totals["inbound"]['sf']    = $sf > 100 ? 100 : $sf;
        }
    }


    /**
     * Group And Calculate Capacity, PAX (flights)
     * @param $flights
     * @return array|bool
     */
    public static function groupAndCalculateCapacityPaxOfFlightsByDate($flights){
        $groupedFlights = $totals = [];
        if (count($flights)){
            $totals = [
                'outbound'  => [
                    'capacity' => [
                        'c' => 0,
                        'y' => 0
                    ],
                    'pax' => [
                        'c' => 0,
                        'y' => 0
                    ]
                ],
                'inbound'   => [
                    'capacity' => [
                        'c' => 0,
                        'y' => 0
                    ],
                    'pax' => [
                        'c' => 0,
                        'y' => 0
                    ]
                ],
            ];
            foreach ($flights as $flight) {
                if ($flight->std && $flight->std != EMPTY_DATETIME){
                    $std = date('Y-m-d', strtotime($flight->std));
                }
                elseif ($flight->atd && $flight->atd != EMPTY_DATETIME) {
                    $std = date('Y-m-d', strtotime($flight->atd));
                }
                else {
                    continue;
                }

                if (!isset($groupedFlights[$std])){
                    $groupedFlights[$std] = [
                        'outbound'  => [
                            'capacity' => [
                                'c' => 0,
                                'y' => 0
                            ],
                            'pax' => [
                                'c' => 0,
                                'y' => 0
                            ]
                        ],
                        'inbound'   => [
                            'capacity' => [
                                'c' => 0,
                                'y' => 0
                            ],
                            'pax' => [
                                'c' => 0,
                                'y' => 0
                            ]
                        ],
//                        'total'     => []
                    ];

                }
                $pax_c_outbound = ($flight->pax_c_actual || $flight->pax_y_actual) ? $flight->pax_c_actual : $flight->pax_c_booked;
                $pax_y_outbound = ($flight->pax_c_actual || $flight->pax_y_actual) ? $flight->pax_y_actual : $flight->pax_y_booked;

//                $groupedFlights[$std]['total']['capacity']['c'] += $flight->capacity_c_total;
//                $groupedFlights[$std]['total']['capacity']['y'] += $flight->capacity_y_total;
//                $groupedFlights[$std]['total']['pax']['c'] += $flight->pax_c_total;
//                $groupedFlights[$std]['total']['pax']['y'] += $flight->pax_y_total;

                $groupedFlights[$std]['outbound']['capacity']['c'] += $flight->capacity_c;
                $groupedFlights[$std]['outbound']['capacity']['y'] += $flight->capacity_y;
                $groupedFlights[$std]['outbound']['pax']['c'] += $pax_c_outbound;
                $groupedFlights[$std]['outbound']['pax']['y'] += $pax_y_outbound;

                $totals['outbound']['capacity']['c'] += $flight->capacity_c;
                $totals['outbound']['capacity']['y'] += $flight->capacity_y;
                $totals['outbound']['pax']['c'] += $pax_c_outbound;
                $totals['outbound']['pax']['y'] += $pax_y_outbound;

                if ($flight->flightChild) {
                    $pax_c_outbound = ($flight->flightChild->pax_c_actual || $flight->flightChild->pax_y_actual) ? $flight->flightChild->pax_c_actual : $flight->flightChild->pax_c_booked;
                    $pax_y_outbound = ($flight->flightChild->pax_c_actual || $flight->flightChild->pax_y_actual) ? $flight->flightChild->pax_y_actual : $flight->flightChild->pax_y_booked;

                    $groupedFlights[$std]['inbound']['capacity']['c'] += $flight->flightChild->capacity_c;
                    $groupedFlights[$std]['inbound']['capacity']['y'] += $flight->flightChild->capacity_y;
                    $groupedFlights[$std]['inbound']['pax']['c'] += $pax_c_outbound;
                    $groupedFlights[$std]['inbound']['pax']['y'] += $pax_y_outbound;

                    $totals['inbound']['capacity']['c'] += $flight->flightChild->capacity_c;
                    $totals['inbound']['capacity']['y'] += $flight->flightChild->capacity_y;
                    $totals['inbound']['pax']['c'] += $pax_c_outbound;
                    $totals['inbound']['pax']['y'] += $pax_y_outbound;
                }
            }
        }
        return [$groupedFlights, $totals];
    }

    public static function handlingGroupAndCalculateCapacityPaxOfFlightsByDate($flights, $handlingStations = null){
        $groupedFlights = $totals = [];

        if (count($flights)){

            if (!$handlingStations){
                $handlingStations = Airport::getHandlingStationsIdAndObject();
            }

            $totals = [
                'outbound' =>
                    [
                        'capacity' => [
                            'c' => 0,
                            'y' => 0
                        ],
                        'pax' => [
                            'c' => 0,
                            'y' => 0
                        ]
                    ],
                'inbound' =>
                    [
                        'capacity' => [
                            'c' => 0,
                            'y' => 0
                        ],
                        'pax' => [
                            'c' => 0,
                            'y' => 0
                        ]
                    ],
            ];

            foreach ($flights as $flight) {
                $std = getFlightDepartureInitialDate($flight, true);

                if (!isset($groupedFlights[$std])){
                    $groupedFlights[$std] = [
                        'outbound'  => [
                            'capacity' => [
                                'c' => 0,
                                'y' => 0
                            ],
                            'pax' => [
                                'c' => 0,
                                'y' => 0
                            ]
                        ],
                        'inbound'   => [
                            'capacity' => [
                                'c' => 0,
                                'y' => 0
                            ],
                            'pax' => [
                                'c' => 0,
                                'y' => 0
                            ]
                        ],
                    ];

                }
                $pax_c_outbound = ($flight->pax_c_actual || $flight->pax_y_actual) ? $flight->pax_c_actual : $flight->pax_c_booked;
                $pax_y_outbound = ($flight->pax_c_actual || $flight->pax_y_actual) ? $flight->pax_y_actual : $flight->pax_y_booked;

                if ($pax_c_outbound || $pax_y_outbound) {
                    $type = getFlightsDepArrType($flight, $handlingStations) == DEPARTURE ? OUTBOUND : INBOUND;

                    $groupedFlights[$std][$type]['capacity']['c'] += $flight->capacity_c;
                    $groupedFlights[$std][$type]['capacity']['y'] += $flight->capacity_y;

                    $groupedFlights[$std][$type]['pax']['c'] += $pax_c_outbound;
                    $groupedFlights[$std][$type]['pax']['y'] += $pax_y_outbound;

                    $totals[$type]['capacity']['c'] += $flight->capacity_c;
                    $totals[$type]['capacity']['y'] += $flight->capacity_y;

                    $totals[$type]['pax']['c'] += $pax_c_outbound;
                    $totals[$type]['pax']['y'] += $pax_y_outbound;
                }
            }
        }
        return [$groupedFlights, $totals];
    }


    /**
     * Get SF For Each Flight
     * @param $flights
     * @param bool|false $skipNullPax
     * @return array
     */
    public static function getFlightsSeatFactors(&$flights, $skipNullPax = false){
        $total = [
            /*
            "total" => [
                "config_c"      => 0,
                "config_y"      => 0,
                "config_total"  => 0,
                "pax_c"         => 0,
                "pax_y"         => 0,
                "pax_total"     => 0,
                "sf"            => 0,
            ],
            */
            "outbound" => [
                "config_c"      => 0,
                "config_y"      => 0,
                "config_total"  => 0,
                "pax_c"         => 0,
                "pax_y"         => 0,
                "pax_total"     => 0,
                "sf"            => 0,
            ],
            "inbound" => [
                "config_c"      => 0,
                "config_y"      => 0,
                "config_total"  => 0,
                "pax_c"         => 0,
                "pax_y"         => 0,
                "pax_total"     => 0,
                "sf"            => 0,
            ],
        ];

        $processedFlights = [];
        if ($flights && count($flights)){
            foreach ($flights as $i => $flight) {
                if (count($processedFlights) && in_array($flight->id, $processedFlights)){
                    unset($flights[$i]);
                    continue;
                }

                // Calculate SF
                $pax = self::calculateFlightsSeatFactors($flights[$i]);

                if ($skipNullPax && !$pax){
                    $flights[$i]->skip = true;
                    continue;
                }

                $total["outbound"]["config_c"] 	    += $flights[$i]->capacity_c_total = $flight->capacity_c;
                $total["outbound"]["config_y"] 	    += $flights[$i]->capacity_y_total = $flight->capacity_y;
                $total["outbound"]["config_total"] 	+= $flight->capacity_c + $flight->capacity_y;
                $total["outbound"]["pax_c"] 	    += $flights[$i]->pax_c_total = $pax['c'];
                $total["outbound"]["pax_y"] 	    += $flights[$i]->pax_y_total = $pax['y'];
                $total["outbound"]["pax_total"]     += $pax['c'] + $pax['y'];


                $flights[$i]->pax_total      = $flights[$i]->pax_c_total + $flights[$i]->pax_y_total;
                $flights[$i]->pax_total_type = $pax['type'];

                $processedFlights[] = $flight->id;

                if ($flight->flightChild){

                    $flightChild = $flight->flightChild;

                    $processedFlights[] = $flightChild->id;

                    // If There is Child Flight Calculate SF As Well
                    $pax = self::calculateFlightsSeatFactors($flights[$i]->flightChild);

                    $total["inbound"]["config_c"]	    += $flightChild->capacity_c;
                    $total["inbound"]["config_y"]	    += $flightChild->capacity_y;
                    $total["inbound"]["config_total"] 	+= $flightChild->capacity_c + $flightChild->capacity_y;
                    $total["inbound"]["pax_c"]		    += $pax['c'];
                    $total["inbound"]["pax_y"]		    += $pax['y'];
                    $total["inbound"]["pax_total"]      += $pax['c'] + $pax['y'];

                    $flights[$i]->capacity_c_total 	+= $flightChild->capacity_c;
                    $flights[$i]->capacity_y_total 	+= $flightChild->capacity_y;
                    $flights[$i]->pax_c_total 		+= $pax['c'];
                    $flights[$i]->pax_y_total 		+= $pax['y'];

                    // Get Total Capacity And SF
                    $flights[$i]->capacity_total += $flightChild->capacity_c + $flightChild->capacity_y;
                    // Total Capacity

                    $flights[$i]->capacity_total   += $flights[$i]->capacity_c_total + $flights[$i]->capacity_y_total;

                    // Total Booked PAX

                    $flights[$i]->pax_total   += $flights[$i]->pax_c_total + $flights[$i]->pax_y_total;
                    $flights[$i]->pax_total_type = $pax['type'];

                    // Total SF
                    $flights[$i]->sf_total = round( ($flights[$i]->sf + $flights[$i]->flightChild->sf) / 2 );
                }


//                $total["total"]["config_c"]	    += $flights[$i]->capacity_c_total;
//                $total["total"]["config_y"]	    += $flights[$i]->capacity_y_total;
//                $total["total"]["config_total"] += $flights[$i]->capacity_c_total + $flights[$i]->capacity_y_total;
//                $total["total"]["pax_c"]	    += $flights[$i]->pax_c_total;
//                $total["total"]["pax_y"]	    += $flights[$i]->pax_y_total;
//                $total["total"]["pax_total"]    += $flights[$i]->pax_c_total + $flights[$i]->pax_y_total;
            }
        }

        if ($total["outbound"]["pax_total"] && $total["outbound"]["config_total"])
        {
            $total["outbound"]["sf"] 		= round( ($total["outbound"]["pax_total"] * 100 / $total["outbound"]["config_total"]));
        }

        if ($total["inbound"]["pax_total"] && $total["inbound"]["config_total"])
        {
            $total["inbound"]["sf"] 		= round( ($total["inbound"]["pax_total"] * 100 / $total["inbound"]["config_total"]));
        }

//        if ($total["total"]["pax_total"] && $total["total"]["config_total"])
//        {
//            $total["total"]["sf"] 		= round( ($total["total"]["pax_total"] * 100 / $total["total"]["config_total"]));
//        }

        return $total;
    }

    public static function getHandlingFlightsSeatFactors(&$flights, $handlingStations = null, $skipNullPax = false){
        $total = [
            "outbound" => [
                "config_c"      => 0,
                "config_y"      => 0,
                "config_total"  => 0,
                "pax_c"         => 0,
                "pax_y"         => 0,
                "pax_total"     => 0,
                "sf"            => 0,
            ],
            "inbound" => [
                "config_c"      => 0,
                "config_y"      => 0,
                "config_total"  => 0,
                "pax_c"         => 0,
                "pax_y"         => 0,
                "pax_total"     => 0,
                "sf"            => 0,
            ],
        ];

        if (!$handlingStations){
            $handlingStations = Airport::getHandlingStationsIdAndObject();
        }

        $processedFlights = [];

        if ($flights && count($flights)){

            foreach ($flights as $i => &$flight) {
                if (count($processedFlights) && in_array($flight->id, $processedFlights)){
                    unset($flights[$i]);
                    continue;
                }

                // Calculate SF
                $pax = self::calculateFlightsSeatFactors($flights[$i]);

                if ($skipNullPax && !$pax){
                    $flights[$i]->skip = true;
                    continue;
                }

                $type = getFlightsDepArrType($flight, $handlingStations) == DEPARTURE ? OUTBOUND : INBOUND;

                $paxC = isset($pax['c']) ? $pax['c'] : 0;
                $paxY = isset($pax['y']) ? $pax['y'] : 0;

                $total[$type]["config_c"] 	    += $flights[$i]->capacity_c_total = $flight->capacity_c;
                $total[$type]["config_y"] 	    += $flights[$i]->capacity_y_total = $flight->capacity_y;
                $total[$type]["config_total"] 	+= $flight->capacity_c + $flight->capacity_y;
                $total[$type]["pax_c"] 	        += $flights[$i]->pax_c_total = $paxC;
                $total[$type]["pax_y"] 	        += $flights[$i]->pax_y_total = $paxY;
                $total[$type]["pax_total"]      += $paxC + $paxY;

                $flight->type = $type;
                $flight->pax_total      = $flights[$i]->pax_c_total + $flights[$i]->pax_y_total;
                $flight->pax_total_type = isset($pax['type']) ? $pax['type'] : "";

                $processedFlights[] = $flight->id;
            }
        }

        if ($total["outbound"]["pax_total"] && $total["outbound"]["config_total"])
        {
            $total["outbound"]["sf"] 		= round( ($total["outbound"]["pax_total"] * 100 / $total["outbound"]["config_total"]));
        }

        if ($total["inbound"]["pax_total"] && $total["inbound"]["config_total"])
        {
            $total["inbound"]["sf"] 		= round( ($total["inbound"]["pax_total"] * 100 / $total["inbound"]["config_total"]));
        }

        return $total;
    }
    /**
     * Calculate SF For A Flight And Return PAX Actual Or Booked
     * @param $flight
     * @return array
     */
    public static function calculateFlightsSeatFactors(&$flight){
        $totalCapacity = 0;
        if ($flight->capacity_y){
            $totalCapacity = $flight->capacity_c + $flight->capacity_y;
        }
        else {
            //Check, Correct And Save Flight's Seat Capacity
            //self::checkAndCorrectFlightsSeatCapacity($flight);

            if ($flight->aircraft){
                $totalCapacity = $flight->aircraft->config_c + $flight->aircraft->config_y;
            }
        }
        if (!$totalCapacity){
            return null;
        }

        // Get Actual Load
        $totalPaxActual = ($flight->pax_c_actual ? $flight->pax_c_actual : 0) + ($flight->pax_y_actual ? $flight->pax_y_actual : 0);
        if (!$totalPaxActual){
            $totalPaxActual = $flight->pax_mvt_total;
        }


        // Get Booked Load
        $totalPaxBooked = $flight->pax_c_booked + $flight->pax_y_booked;

        // If Actual Seat Load Is There -> Priority, Else User Booked Figures
        if ($totalPaxActual){
            $flight->sf = round( ($totalPaxActual / $totalCapacity) * 100 );

            return [ 'c' => $flight->pax_c_actual, 'y' => $flight->pax_y_actual, 'type' => 'actual'];
        }
        else {
            $flight->sf = round( ($totalPaxBooked / $totalCapacity) * 100 );

            return [ 'c' => $flight->pax_c_booked, 'y' => $flight->pax_y_booked, 'type' => 'booked'];
        }
    }

    /**
     * Get Parent And Child Flight And Return Array of Flight Ids
     * @param $date
     * @param $parentFlightNumberId
     * @return array
     */
    public static function Parent_Child_Flight($date, $parentFlightNumberId){
        $group_flights = [];

        $flights = Flight::leftJoin("flights as s_f", "flights.id", "=", "s_f.parent_id")
            ->where(DB::raw("DATE(s_flights.std)"), $date)
            ->where("flights.flight_number_id", $parentFlightNumberId)
            ->whereNull("flights.cancelled_at")->whereNull("flights.deleted_at")
            ->first(['flights.id as id', "s_f.id as child_id"]);

        if ($flights){
            $group_flights = [$flights->id, $flights->child_id];
        }

        return $group_flights;
    }


    //$queries = DB::getQueryLog();
    //$last_query = end($queries);
    //debug($queries);exit;



    public static function FlightQueryModule($flightTypeRepository, $flightFareBasisRepository, $flightOrderStatusRepository,
                                             $flightCabinRepository, $flightRbdRepository, $reportTypeRepository,
                                             $salesChannelRepository, $regionRepository, $countryRepository, $cityReporsitory,
                                             $agencyRepository, $agencyCodeRepository){
        $inputs = \request()->all();

        $form_submitted = count(\request()->all()) ?  true : false;
        $selected_report = "";

        if (\request()->has(REGION_REPORT)){
            $inputs['region_group'] = 1;
            $selected_report = REGION_REPORT;
        }
        else if (\request()->has(COUNTRY_REPORT)){
            $inputs['country_group'] = 1;
            $selected_report = COUNTRY_REPORT;
        }
        else if (\request()->has(CITY_REPORT)){
            $inputs['city_group'] = 1;
            $selected_report = CITY_REPORT;
        }
        else if (\request()->has(AGENCY_REPORT)){
            $inputs['agency_group'] = 1;
            $selected_report = AGENCY_REPORT;
        }
        else if (\request()->has(SALES_CHANNEL_REPORT)){
            $inputs['sales_channel_group'] = 1;
            $selected_report = SALES_CHANNEL_REPORT;
        }
        else if (\request()->has(GDS_REPORT)){
            $inputs['gds_group'] = 1;
            $selected_report = GDS_REPORT;
        }
        else if (\request()->has(FLIGHT_NUMBER_REPORT)){
            $inputs['flight_number_group'] = 1;
            $selected_report = FLIGHT_NUMBER_REPORT;
        }
        else if (\request()->has(SECTOR_REPORT)){
            $inputs['sector_group'] = 1;
            $selected_report = SECTOR_REPORT;
        }
        else if (\request()->has(ROUTE_REPORT)){
            $inputs['route_group'] = 1;
            $selected_report = ROUTE_REPORT;
        }
        else if (\request()->has(CABIN_REPORT)){
            $inputs['cabin_group'] = 1;
            $selected_report = CABIN_REPORT;
        }
        else if (\request()->has(RBD_REPORT)){
            $inputs['rbd_group'] = 1;
            $selected_report = RBD_REPORT;
        }

        // Search only on Submitted Form
        if ($form_submitted){

            // Search Passenger Flights And Total
            list($tableData, $pagination) = PassengerFlight::searchPassengerFlights($inputs, FALSE, TRUE);

            $unique_periods = makeUniqueByColumn($tableData, "period");

            $unique_pos = makeUniqueByColumn($tableData, "pos");

            $unique_route = makeUniqueByColumn($tableData, "route");

            if ($selected_report == ROUTE_REPORT){
                $tableData = collectionToArrayRouteReport($tableData);
            }
            else {
                $tableData = collectionToArray($tableData);
            }

            list($paxChart, $revenueChart, $xAxisValues) = paxAndRevenueChartsData($tableData);

        }


        $sector        = FlightNumber::getArraySectors('Scheduled');
        $flightNumbers = FlightNumber::getArrayFlightNumbers('Scheduled');

        return [
            // Charts
            "xAxisValues"		    => isset($xAxisValues) ? json_encode($xAxisValues) : null,
            "paxChart"			    => isset($paxChart) ? json_encode($paxChart) : null,
            "revenueChart"		    => isset($revenueChart) ? json_encode($revenueChart) : null,
            // Currently Selected Report
            "selectedReport"	    => $selected_report,
            // Data
            "tableData"		 	    => isset($tableData) ? $tableData : [],
            // Unique Period
            'uniquePeriods' 	    => isset($unique_periods) ? $unique_periods : null,
            // Unique POS
            'uniquePos'     	    => isset($unique_pos) ? $unique_pos : null,
            // Unique Routes
            'uniqueRoute'     	    => isset($unique_route) ? $unique_route : null,
            // Pagination (Enable)
            'pagination'            => isset($pagination) ? $pagination : null,

            "flightTypesList"   	=> $flightTypeRepository->listModelVariable(['type', 'id'], true),
            "flightNumbersList" 	=> $flightNumbers,
            "salesChannelsList"	    => $salesChannelRepository->listModelVariable(['channel', 'id']),
            "sectorsList"         	=> $sector,
            "regionsList"        	=> $regionRepository->listModelVariable(['region', 'id']),
            "countriesList"        	=> $countryRepository->listModelVariable(['country', 'id']),
            "citiesList"           	=> $cityReporsitory->listModelVariable(['city', 'id']),
            "agenciesList"         	=> $agencyRepository->listModelVariable(['agency', 'id']),
            "tchCodesList"          => $agencyCodeRepository->findAndListModelVariable([ [WHERE_RAW, 'char_length(code) = 5'] ], ['code', 'id']),
            "couponsList"         	=> [
                '1' => '1',
                '2' => '2',
                '3' => '3',
                '4' => '4'
            ],
            "recordsPerPage" => [
                '50'    => '50',
                '250'   => '250',
                '500'   => '500',
                '1000'  => '1000'
            ],
            "fareBasisList"         => $flightFareBasisRepository->listModelVariable(['fare_basis', 'id']),
            "paymentFormsList"      => [
                "CA" => "CA",
                "CC" => "CC",
                "MS" => "MS"
            ],
            "statusList"            => $flightOrderStatusRepository->listModelVariable(['order_status', 'id']),
            "cabinsList"            => $flightCabinRepository->listModelVariable(['cabin', 'id']),
            "rbdList"               => $flightRbdRepository->listModelVariable(['rbd', 'id']),
            "reportTypesList"       => $reportTypeRepository->listModelVariable(['type', 'id'], true, true)
        ];
    }


    public static function searchFlightByDateAndFlightNumber($date, $flightNumber, $departureAirport, $arrivalAirport, $type){
        $flight = Flight::join('flights__numbers','flights__numbers.id', '=', 'flights.flight_number_id')
            ->join('airports AS depAirport', 'depAirport.id', '=', 'flights__numbers.departure_airport_id')
            ->join('airports AS arrAirport', 'arrAirport.id', '=', 'flights__numbers.arrival_airport_id')
            ->leftJoin('cities AS depCity', 'depCity.id', '=', 'depAirport.city_id')
            ->leftJoin('cities AS arrCity', 'arrCity.id', '=', 'arrAirport.city_id');

        if ($type == 'arrival')
            $time = 'sta';
        else
            $time = 'std';

        if (!$date)
            $date = date('Y-m-d');

        $flight->whereBetween(DB::raw("TIMESTAMPADD(HOUR, 5, $time)"), [$date." 00:00:00", $date." 23:59:59"] );

        if ($flightNumber){
            $flight->where('flight_number', $flightNumber);
        }
        else if ($departureAirport || $arrivalAirport) {
            if ($departureAirport)
                $flight->where('depAirport.iata', $departureAirport);

            if ($arrivalAirport)
                $flight->where('arrAirport.iata', $arrivalAirport);
        }

        return $flight->get([
            'flight_number',
            'depAirport.iata as departure_code',
            'depCity.city as departure_airport',
            'arrAirport.iata as arrival_code',
            'arrCity.city as arrival_airport',
            'std',
            'sta',
            'etd',
            'eta',
            'atd',
            'ata',
            'is_diversion',
            'diversion'
        ]);
    }

    /**
     * Update Flight Delays
     * @param $flight
     * @param IFlightDelayRepository $flightDelayRepository
     */
    public static function UpdateFlightDelays(&$flight, IFlightDelayRepository $flightDelayRepository){
        $delay_id = \request()->get("delay_id");
        $delay_time = \request()->get("delay_time");
        $delay = NULL;

        $flightDelays = $flight->delays;

        if ($flightDelays && $flightDelays->count()){
            $flightDelays = $flightDelays->keyBy('position');
        }

        if (is_array($delay_id) && count($delay_id) > 0 && is_array($delay_time) && count($delay_time) > 0) {
            $actualIds = [];

            foreach ($delay_id as $i => $d_id) {
                if ($d_id && $d_id != '' && $delay_time[$i] && $delay_time[$i] != "00:00") {
                    $position = $i + 1;

                    $delay = [
                        "flight_id" => $flight->id,
                        "delay_id"  => $d_id,
                        "position"  => $position,
                        "duration"  => $delay_time[$i]
                    ];

                    if (isset($flightDelays[$position])){
                        $flightDelayRepository->update($flightDelays[$position], $delay);
                        $actualIds[] = $flightDelays[$position]->id;
                        continue;
                    }
                    $newFlightDelay = $flightDelayRepository->createByAttributes($delay);
                    $actualIds[] = $newFlightDelay->id;
                }
            }
            // Delete Rest of Flights' Delays
            $flightDelayRepository->deleteNotIn(['flight_id' => $flight->id], $actualIds);
        }
        elseif ($flight->delays->count() > 0) {
            debug($flight->delays);
//            debug($flight->delays());
            FlightDelay::where("flight_id", $flight->id)
                ->delete();
//            foreach ($flight->delays as $each);
        }
    }


    public static function SetFlightDelays(&$flight, $delay_code, $delay_time){
        if ($flight && is_array($delay_code) && count($delay_code) > 0 && is_array($delay_time) && count($delay_time) > 0) {
            $existingDelays = [];
            $airline = null;

            if (ghaModule()){
                $airline = $flight->flightNumber && $flight->flightNumber->airline ? $flight->flightNumber->airline : null;
            }

            foreach ($delay_code as $i => $each_code) {

                $each_code = trim($each_code);

                if ($each_code && $each_code != '' && $delay_time[$i] && $delay_time[$i] != EMPTY_TIME_HM) {
                    $position = $i + 1;

                    // Find Delay from code

                    $delay = Delay::whereNull("deleted_at")
                        ->where(function($sql) use ($each_code){
                            $sql->where("code", $each_code)
                                ->orWhere("alpha_code", $each_code);
                        });

                    if ($airline){
                        $delay->where("airline_id", $airline->id);
                    }
                    $delay = $delay->first();

                    // If Delay not found -> ignore it
                    if (!$delay)
                    {
                        $delay = new Delay();
                        $delay->airline_id = $airline ? $airline->id : null;
                        if (ctype_alpha($each_code)){
                            $delay->alpha_code = $each_code;
                        }
                        else {
                            $delay->code = $each_code;
                        }
                        $delay->save();
                    }

                    // Search if given delay exists
                    $flightDelay = FlightDelay::where("flight_id", $flight->id)
                        ->where("delay_id", $delay->id)
                        ->first();
                    // If Found update
                    if ($flightDelay){
                        $flightDelay->position = $position;
                        $flightDelay->duration = $delay_time[$i];
                        $flightDelay->deleted_at = null;
                        $flightDelay->save();

                        $existingDelays[] = $flightDelay->id;
                        continue;
                    }

                    // Otherwise, create new
                    $flightDelay = new FlightDelay();
                    $flightDelay->flight_id = $flight->id;
                    $flightDelay->delay_id = $delay->id;
                    $flightDelay->position = $position;
                    $flightDelay->duration = $delay_time[$i];
                    $flightDelay->save();

                    $existingDelays[] = $flightDelay->id;
                }
            }

            // Delete Rest of Flights' Delays

            $fDelay = FlightDelay::where("flight_id", $flight->id)
                ->whereNotIn('id', $existingDelays)
                ->get();
            // Audit Deleted
            auditDeleted($fDelay);


            try {
                Parse::sendDelayEmailNotification($flight);
            }
            catch (\Exception $e){
                Parse::sendSimpleEmailNotification("Delay Email Notification Error"."\n\n Error: \n".$e->getMessage());
            }
        }
    }

    public static function SetFlightDelaysNoCode(&$flight, $delay_time, $description){
        if ($flight && $delay_time) {
            $existingDelays = [];

            $delay = Delay::whereNull("deleted_at")
                ->where(function($sql) use ($description){
                    $sql->where("explanation", $description)
                        ->orWhere("explanation_ln", $description)
                        ->orWhere("description", $description)
                        ->orWhere("description_ln", $description);
                });

            /*
            $airline = $flight->flightNumber && $flight->flightNumber->airline ? $flight->flightNumber->airline : null;
            if ($airline){
                $delay->where("airline_id", $airline->id);
            }
            */

            $delay = $delay->first();

            // If Delay not found -> ignore it
            if (!$delay)
            {
                $delay = new Delay();
                //$delay->airline_id = $airline ? $airline->id : null;
                if (wordIsInEnglish($description)) {
                    $delay->description = $description;
                }
                else {
                    $delay->description_ln = $description;
                }

                $delay->save();
            }

            // Search if given delay exists
            $flightDelay = FlightDelay::where("flight_id", $flight->id)
                ->where("delay_id", $delay->id)
                ->first();

            // If Found update
            if (!$flightDelay) {
                $flightDelay = new FlightDelay();
                $flightDelay->flight_id = $flight->id;
                $flightDelay->delay_id = $delay->id;
            }

            $flightDelay->position = 1;
            $flightDelay->duration = $delay_time;
            $flightDelay->save();

            $existingDelays[] = $flightDelay->id;

            // Delete Rest of Flights' Delays

            $fDelay = FlightDelay::where("flight_id", $flight->id)
                ->whereNotIn('id', $existingDelays)
                ->get();
            // Audit Deleted
            auditDeleted($fDelay);


            try {
                Parse::sendDelayEmailNotification($flight);
            }
            catch (\Exception $e){
                Parse::sendSimpleEmailNotification("Delay Email Notification Error"."\n\n Error: \n".$e->getMessage());
            }
        }
    }

    public static function ManageDiversion($flightRepository, $flight, $flightNumber, $properties, $previouslyDiverted){
        if (\request()->has("is_diversion")) {
            $diversionAirport = \request()->get("diversion_airport");
            // Previously Diverted
            if ($previouslyDiverted){
                // Inbound Flight
                if ($flight->prev_flight_id){
                    $flight->departure_airport_id = $diversionAirport;

                    // Find and Change Outbound Flight
                    $outboundFlight = Flight::find($flight->prev_flight_id);
                    if ($outboundFlight){
                        $outboundFlight->flight_number_id = $flightNumber->id;
                        $outboundFlight->arrival_airport_id = $diversionAirport;
                        $outboundFlight->diverted_airport_id = $diversionAirport;
                        $outboundFlight->save();
                    }
                }
                // Outbound Flight
                elseif($flight->next_flight_id) {
                    $flight->arrival_airport_id = $diversionAirport;

                    // Find and Change Inbound Flight
                    $inboundFlight = Flight::find($flight->next_flight_id);
                    if ($inboundFlight){
                        $inboundFlight->flight_number_id = $flightNumber->id;
                        $inboundFlight->departure_airport_id = $diversionAirport;
                        $inboundFlight->diverted_airport_id = $diversionAirport;
                        $inboundFlight->save();
                    }
                }

                $flight->diverted_airport_id = $diversionAirport;
                $flight->save();
            }
            // Not Previously diverted
            else {
                // Create Inbound Diverted Flight
                $divertedInbound = self::createDivertedFlights($flight->id, $diversionAirport, $properties, $flightRepository, $flightNumber->id);

                // Change Current Flight To Outbound Diverted Flight
                $flight->next_flight_id = $divertedInbound->id;
                $flight->departure_airport_id = $flightNumber->departure_airport_id;
                $flight->arrival_airport_id = $diversionAirport;
                $flight->diverted_airport_id = $diversionAirport;

                $flight->std = \request()->get("std") ? \request()->get('std') : \request()->get('ptd');
                $flight->ptd = \request()->get("ptd") ? \request()->get('ptd') : \request()->get('std');
                $flight->etd = (\request()->get("etd")) ? \request()->get("etd") : NULL;
                $flight->atd = (\request()->get("atd")) ? \request()->get("atd") : NULL;
                $flight->abn = (\request()->get("abn")) ? \request()->get("abn") : NULL;
                setFlightDepartureDate($flight);

                $flight->sta = \request()->get("diversion_sta") ? \request()->get("diversion_sta") : NULL;
                $flight->pta = \request()->get("diversion_sta") ? \request()->get("diversion_sta") : NULL;
                $flight->eta = \request()->get("diversion_eta") ? \request()->get("diversion_eta") : NULL;
                $flight->tdn = \request()->get("diversion_tdn") ? \request()->get("diversion_tdn") : NULL;
                $flight->ata = \request()->get("diversion_ata") ? \request()->get("diversion_ata") : NULL;
                setFlightArrivalDate($flight);

                $flight->is_diversion = TRUE;
                $flight->save();
            }
        }
        // If No longer diverted
        elseif (!\request()->has("is_diversion") && $previouslyDiverted) {
            // Inbound Flight
            if ($flight->prev_flight_id){
                $flight->cancelled_at = date("Y-m-d H:i:s");
                $flight->save();

                // Find and Change Outbound Flight
                $outboundFlight = Flight::find($flight->prev_flight_id);
                if ($outboundFlight){
                    $outboundFlight->flight_number_id = $flightNumber->id;
                    $outboundFlight->departure_airport_id = null;
                    $outboundFlight->arrival_airport_id = null;
                    $outboundFlight->diverted_airport_id = null;
                    $outboundFlight->is_diversion = 0;
                    // Arrival information of Inbound Flight
                    $outboundFlight->sta = $flight->sta;
                    $outboundFlight->pta = $flight->pta;
                    $outboundFlight->eta = $flight->eta;
                    $outboundFlight->ata = $flight->ata;
                    $outboundFlight->tdn = $flight->tdn;

                    setFlightArrivalDate($outboundFlight);
                    $outboundFlight->save();
                }
            }
            // Outbound Flight
            elseif($flight->next_flight_id) {

                // Find and Change Inbound Flight
                $inboundFlight = Flight::find($flight->next_flight_id);
                if ($inboundFlight){
                    $inboundFlight->cancelled_at = date("Y-m-d H:i:s");;
                    $inboundFlight->save();

                    $flight->sta = $inboundFlight->sta;
                    $flight->pta = $inboundFlight->pta;
                    $flight->eta = $inboundFlight->eta;
                    $flight->ata = $inboundFlight->ata;
                    $flight->tdn = $inboundFlight->tdn;

                    setFlightArrivalDate($flight);
                }

                $flight->flight_number_id = $flightNumber->id;
                $flight->departure_airport_id = null;
                $flight->arrival_airport_id = null;
                $flight->diverted_airport_id = null;
                $flight->is_diversion = 0;
                $flight->save();

            }
        }
    }

    public static function CreateDivertedFlights($flightId, $diversionAirport, $properties, $flightRepository, $flightNumberId){
        $flightNumber = FlightNumber::find($flightNumberId);

        $diversionInbound = $flightRepository->createByAttributes($properties + [
                'prev_flight_id'       => $flightId,
                'departure_airport_id' => $diversionAirport,
                'arrival_airport_id'   => $flightNumber->arrival_airport_id,
                'diverted_airport_id'  => $diversionAirport,
                'is_diversion'         => true,
                'std'                  => \request()->get("diversion_std") ? \request()->get("diversion_std") : NULL,
                'ptd'                  => \request()->get("diversion_std") ? \request()->get("diversion_std") : NULL,
                'etd'                  => \request()->get("diversion_etd") ? \request()->get("diversion_etd") : NULL,
                'atd'                  => \request()->get("diversion_atd") ? \request()->get("diversion_atd") : NULL,
                'abn'                  => \request()->get("diversion_abn") ? \request()->get("diversion_abn") : NULL,
                'sta'                  => \request()->get("sta") ? \request()->get("sta") : \request()->get('pta'),
                'pta'                  => \request()->get("pta") ? \request()->get("pta") : \request()->get('sta'),
                'eta'                  => (\request()->get("eta")) ? \request()->get("eta") : NULL,
                'tdn'                  => (\request()->get("tdn")) ? \request()->get("tdn") : NULL,
                'ata'                  => (\request()->get("ata")) ? \request()->get("ata") : NULL,
                'created_by'           => Auth::user()->id
            ]);

        $diversionInboundId = $diversionInbound->id;

        // Add Crew
        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_CAPTAINS, \request()->get('captain'), \request()->get('captain_report_time'), \request()->get('captain_reason'));
        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_CAPTAINS, \request()->get('captain_standby'), \request()->get('captain_standby_report_time'), \request()->get('captain_standby_reason'), true);
        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_CAPTAINS, \request()->get('captain_dhc'), \request()->get('captain_dhc_report_time'), \request()->get('captain_dhc_reason'), false, true);

        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_FIRST_OFFICERS, \request()->get('first_officer'), \request()->get('first_officer_report_time'), \request()->get('first_officer_reason'));
        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_FIRST_OFFICERS, \request()->get('first_officer_standby'), \request()->get('first_officer_standby_report_time'), \request()->get('first_officer_standby_reason'), true);
        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_FIRST_OFFICERS, \request()->get('first_officer_dhc'), \request()->get('first_officer_dhc_report_time'), \request()->get('first_officer_dhc_reason'), false, true);

        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_PURSER_AND_CABIN_CREW, \request()->get('cabin_crew'), \request()->get('cabin_crew_report_time'), \request()->get('cabin_crew_reason'));
        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_PURSER_AND_CABIN_CREW, \request()->get('cabin_crew_standby'), \request()->get('cabin_crew_standby_report_time'), \request()->get('cabin_crew_standby_reason'), true);
        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_PURSER_AND_CABIN_CREW, \request()->get('cabin_crew_dhc'), \request()->get('cabin_crew_dhc_report_time'), \request()->get('cabin_crew_dhc_reason'), false, true);
        FlightCrew::createChangeCrewProcedure($diversionInboundId, NULL, CREW_PURSER_AND_CABIN_CREW, \request()->get('cabin_crew_sup'), \request()->get('cabin_crew_sup_report_time'), \request()->get('cabin_crew_sup_reason'), false, false, true, \request()->get('cabin_crew_sup_type'));

        return $diversionInbound;
    }

    public static function parseMVTMessages($email){

        $readyData = "";

        $sent = "".baseDateFormat($email->received_datetime, true, "-", false, true)."";

        $emailBody = str_replace("=0D", "", $email->email_body);

        if (preg_match('/(?:^(?:\s*|.)?(DIV[\s\S]*)\s*)/m', $emailBody, $match)){

        }
        elseif (preg_match('/(?:^(?:\s*|.)?((?:^CORR?\s*)?MVT[\s\S]*)\s*)/m', $emailBody, $match)){

        }

        if (isset($match[1])){

            $readyData .= $match[1]."\n";
            $readyData .= "-------------------------------\n";

            if ($email->skipped){
                $readyData .= "<span class='label label-danger'>MESSAGE SKIPPED</span>\n";
            }

            $readyData .= "Source: ";

            if ($email->created_by){
                $user = User::find($email->created_by);

                $readyData .= ($user ? $user->first_name." ".$user->last_name : "")."\n";
            }
            else {
                $readyData .= str_replace(["<", ">"], "/", $email->from_email)."\n";
            }

            $readyData .= "Received: ". $sent;
        }
        return $readyData;
    }

    public static function parseMessages($email, $type){
        if (!$email){
            return null;
        }

        $sent = "".baseDateFormat($email->received_datetime, true, "-", false, true)."";
        $data = $email->data ? unserialize($email->data) : null;

        $emailBody = str_replace("=0D", "", $email->email_body);

        if ($data && count($data)){
            switch($type){
                case PRL:
                case PNL:
                    preg_match('/(?:^(?:\s*|.)?((?:^CORR?\s*)?(?:PNL|ADL|PRL)[\s\S]*)\s*)/m',$emailBody, $match);
                    break;

                case LDM:
                    preg_match('/(?:^(?:\s*|.)?(?:.*LDM)([\s\S]*)\s*)/m',$emailBody, $match);
                    return self::prepareLDMForView($email, $sent, $match);
                    break;

                case CPM:
                case UCM:
                case TPM:
                case PTM:
                case PSM:
                case LPM:
                case SLS:
                case HBG:
                case CREW_LIST:
                    preg_match('/(?:^(?:\s*|.)?((?:^CORR?\s*)?'.strtoupper($type).'[\s\S]*)\s*)/m',$emailBody, $match);
                    break;

                case LDS:
                    preg_match('/(?:^(?:\s*|.)?(.*(?:(?:L O A D S H E E T)|(?:LOADSHEET FINAL)).[\s\S]*)\s*)/m',$emailBody, $match);
                    break;

                case LIR:
                    preg_match('/(?:^(?:\s*|.)?(.*(?:(?:LOADING\s*INSTRUCTION\/REPORT)).[\s\S]*)\s*)/m',$emailBody, $match);
                    break;

                case CAL:
                case PAL:
                    preg_match('/(?:^(?:\s*|.)?(.*(?:CAL|PAL).[\s\S]*)\s*)/m',$emailBody, $match);
                    break;

                case NOT:
                case NOTOC:
                case ONLOAD:
                case OTHS:
                    preg_match('/(?:^(?:\s*|.)?(.[\s\S]*)\s*)/m',$emailBody, $match);
                    break;

                default:
                    break;
            }

            if (!isset($match) || !isset($match[1])){
                return null;
            }

            return self::prepareForView($email, $sent, $match);
        }

        return null;
    }

    static function prepareForView($email, $sent, $match){
        $readyData = (isset($match[1]) ? $match[1] : "")."\n";
        $readyData .= "-------------------------------\n";
        $readyData .= "Source: ";

        if ($email->created_by){
            $user = User::find($email->created_by);

            $readyData .= ($user ? $user->first_name." ".$user->last_name : "")."\n";
        }
        else {
            $readyData .= str_replace(["<", ">"], "/", $email->from_email)."\n";
        }
        $readyData .= "Received: ". $sent;

        return $readyData;
    }

    static function prepareLDMForView($email, $sent, $match){
        $readyData = "LDM\n";
        $readyData .= (isset($match[1]) ? $match[1] : "")."\n";
        $readyData .= "-------------------------------\n";
        $readyData .= "Source: ";

        if ($email->created_by){
            $user = User::find($email->created_by);

            $readyData .= ($user ? $user->first_name." ".$user->last_name : "")."\n";
        }
        else {
            $readyData .= str_replace(["<", ">"], "/", $email->from_email)."\n";
        }
        $readyData .= "Received: ". $sent;

        return $readyData;
    }

    public static function FlightsRangeConditional($selectedAirports = null, $selectedAirlines = null, $type = ALL,
                                                   $aircraftID = null, $includeCancelled = null, $datePeriod = null){

        $flightSearchOption = \request()->get('flightSearch');
        switch ($flightSearchOption) {
            case 'flightNumber':
                $flightSearch = ['flightNumberIds' => \request()->get('flightNumber')];
                break;
            case 'sector':
                $flightSearch = ['sectorId' => \request()->get('sector')];
                break;
            default:
                $flightSearch = null;
                break;
        }

        $dateSearch = $datePeriod ? $datePeriod : getFormPeriod();


        // Get All Flights (filter: Date, Flight Number)
        return Flight::getFlightsByDateFlightNumber($dateSearch, $flightSearch, null, ($selectedAirports ? $selectedAirports : \request()->get("airport")), ($selectedAirlines ? $selectedAirlines : \request()->get("airline")), $type, $aircraftID, null, $includeCancelled);
    }


    public static function getAirportStaffFlights($startDate, $endDate, $processedFlightIDs, $selectedAirportIds, $staffAirports, $staffAirlines, $occ = null)
    {
        $userIDs = User::getStaffWithService($occ, $selectedAirportIds);

        // Get Flights of all users of the selected airport location
        $airportStaffFlights = Flight::with([
            "flightNumber",
            "flightNumber.departureAirport",
            "flightNumber.arrivalAirport",
        ])
            ->join("flights__staff", "flights__staff.flight_id", "=", "flights.id")
            ->join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
            ->where(function ($sql) use ($startDate, $endDate) {
                $sql->whereBetween("std", [$startDate, $endDate])
                    ->orWhereBetween("ptd", [$startDate, $endDate])
                    ->orWhereBetween("etd", [$startDate, $endDate])
                    ->orWhereBetween("atd", [$startDate, $endDate]);
            })
            ->whereNotIn("flights.id",$processedFlightIDs)
            ->whereNull("flights.deleted_at")
            ->where(function($sql){
                $sql->whereNull("flights.cancelled_at")
                    ->orWhere(function ($sql2){
                        $sql2->whereNotNull("cancelled_at")
                            ->where("sod_completed", true);
                    });
            })
            ->whereIn("flights__staff.user_id", $userIDs)
            ->whereNull("flights__staff.deleted_at");

        if ($staffAirports){
            $airports = is_array($staffAirports) ? $staffAirports : [$staffAirports];

            if (count($airports)) {
                if (isset($options['only_departure_airports'])) {
                    $airportStaffFlights->where(function($sql) use ($airports){
                        $sql->whereIn("flights__numbers.departure_airport_id", $airports)
                            ->orWhereIn("flights.departure_airport_id", $airports);
                    });
                } else {
                    $airportStaffFlights->where(function ($sql) use ($airports) {
                        $sql->whereIn("flights__numbers.departure_airport_id", $airports)
                            ->orWhereIn("flights__numbers.arrival_airport_id", $airports)
                            ->orWhereIn("flights.departure_airport_id", $airports)
                            ->orWhereIn("flights.arrival_airport_id", $airports);
                    });
                }
            }
        }

        if ($staffAirlines){
            $airlines = is_array($staffAirlines) ? $staffAirlines : [$staffAirlines];
//            $airportStaffFlights->join("airlines", "flights__numbers.airline_id", "=", "airlines.id");
            if (count($airlines)){
                $airportStaffFlights->whereIn("flights__numbers.airline_id", $airlines);
            }
        }

        $airportStaffFlights = $airportStaffFlights
            ->distinct()
            ->get([
                "flights.*"
            ]);

        $locationFlights = [];
        foreach ($airportStaffFlights as $each) {
            if (!in_array($each->id, $processedFlightIDs)){
                $locationFlights[] = $each;
                $processedFlightIDs[] = $each->id;
            }
        }

        return $locationFlights;
    }


}
