<?php namespace App\Models; use Illuminate\Database\Eloquent\Model as Eloquent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 02.09.14

 */

use OwenIt\Auditing\Audit;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class Airport extends Eloquent implements AuditableContract{

    use Auditable;

    protected $auditExclude = [
        'updated_by',
        'created_by',
    ];

    protected $table = "airports";

    public $timestamps = true;

    protected $fillable = [
        'iata',
        'icao',
        'city_id',
        'country_id',
        'name',
        'elevation',
        'latitude',
        'longitude',
        'timezone'
    ];

    public function city(){
        return $this->belongsTo('App\Models\City', 'city_id');
    }

    public function country(){
        return $this->belongsTo('App\Models\Country', 'country_id');
    }

    public static function checkAndGet($code){
        return Airport::where("iata", $code)
            ->orWhere("icao", $code)
            ->first();
    }

    public static function findOrCreateByIATA($iata){
        return Airport::where("iata", strtoupper($iata))
                    ->firstOrCreate();
    }

    public static function listOperatedStations($value = "iata", $key = "id"){
        return Airport::where("handling_airport", true)
            ->orWhere("operated", true)
            ->orderBy("position", "DESC")
            ->orderBy($value)
            ->pluck($value, $key)
            ->all();

    }

    public static function listHandlingStations($value = "iata", $key = "id"){
        return Airport::where("handling_airport", true)
                        ->orderBy("position", "DESC")
                        ->orderBy($value)
                        ->pluck($value, $key)
                        ->all();

    }

    public static function getHandlingStations($value = "id", $key = null, $showFlightWatchOnly = null){
        $airports = Airport::where("handling_airport", true);

        if ($showFlightWatchOnly){
            $airports->where("flight_watch", true);
        }

        if ($key){
            return $airports->pluck($value, $key)->all();
        }
        else {
            return $airports->pluck($value)->all();

        }
    }

    public static function getOperatedStations($value = "id", $key = null){
        $airports = Airport::where("handling_airport", true)
                            ->orWhere("operated", true);

        if ($key){
            return $airports->pluck($value, $key)->all();
        }
        else {
            return $airports->pluck($value)->all();

        }
    }


    public static function getHandlingStationsIdAndObject($includeOperated = false, $iata = false){
        $airports = Airport::where("handling_airport", true);

        // Added on Apr 26 2019
        if (airlineModule() && $includeOperated){
            $airports->orWhere("operated", true);
        }

        $airports = $airports->get();

        $result = [];
        foreach ($airports as $each) {
            if ($iata){
                $result[$each->id] = $each->iata;
            }
            else {
                $result[$each->id] = $each;
            }
        }

        return $result;

    }

    public static function getSelectedAirportIds(Request $request){
        $selectedAirport = $request->get("airport_id");

        $selectedAirports = $request->get("airport_ids");

        if ($selectedAirport == ""){
            return StaffStation::currentStaffStationsList();
//            return env(CARRIER) == HANDLING ? StaffStation::currentStaffStationsList() : [];
        }
        else {
            if ($selectedAirports && count($selectedAirports)) {
                if ($selectedAirport && in_array($selectedAirport, $selectedAirports)) {
                    $selectedAirportIds = array_diff($selectedAirports, [$selectedAirport]);
                } else if (in_array("", $selectedAirports) && count($selectedAirports) > 1) {
                    $selectedAirportIds = StaffStation::currentStaffStationsList();
                } else {
                    $selectedAirportIds = $selectedAirport ? array_merge([$selectedAirport], $selectedAirports) : $selectedAirports;
                }
            }
            else {
                if ($selectedAirport){
                    if (is_array($selectedAirport)) {
                        $selectedAirportIds = $selectedAirport;
                    }
                    else {
                        $selectedAirportIds = explode(",", $selectedAirport);
                    }
                }
                else {

                    $selectedAirportIds = StaffStation::currentStaffStationsList();
                }
            }
        }

        $result = count($selectedAirportIds) ? $selectedAirportIds : StaffStation::currentStaffStationsList();

        return $result;
    }


    /**
     * @param null $airlineId
     * @return mixed
     */
    public static function listHandlingAirports($airlineId = null){

        $airports = FlightNumber::join("airlines", "airlines.id", "=", "flights__numbers.airline_id")
                                      ->leftJoin("airports AS depAirport", "depAirport.id", "=", "flights__numbers.departure_airport_id")
                                      ->where("handling_airline", true)
                                      ->whereNull("flights__numbers.deleted_at");

        if ($airlineId){
            $airports->where("flights__numbers.airline_id", $airlineId);
        }

        $airports = $airports->select(["depAirport.iata as iata", "depAirport.id as id"])
                             ->whereNotNull("depAirport.iata")
                             ->groupBy("iata", "id")
                             ->orderBy("iata")
                             ->pluck("iata", "id")
                             ->all();

        return $airports;

    }

    public static function pluckAirportsWithCode($value = "icao", $key = "id"){
        $list = Airport::select([
            $value,
            $key
        ])->orderBy($value)
            ->pluck($value, $key)
            ->all();

        return $list;
    }

    public static function listAirportsWithCode($value = "icao", $key = "id"){
        $reserve = $value == IATA ? ICAO : IATA;

        $list = Airport::select([
            DB::raw("CONCAT(name, ' (', IF({$value} is NULL OR {$value} = '', {$reserve}, {$value}), ')') AS code"),
            $key
        ])->orderBy("name")
          ->pluck("code", $key)
          ->all();

        return $list;
    }

    public static function listDiversionAirportsWithCode($value = "icao", $key = "id"){
        $reserve = $value == IATA ? ICAO : IATA;

        $list = Airport::select([
            DB::raw("CONCAT(name, ' (', IF({$value} is NULL OR {$value} = '', {$reserve}, {$value}), ')') AS code"),
            $key
        ])  ->where("diversion_airport", true)
            ->orderBy("name")
            ->pluck("code", $key)
            ->all();

        return $list;
    }


    public static function listHandlingAirportsWithCode($value = "icao", $key = "id"){
        $reserve = $value == IATA ? ICAO : IATA;

        $list = Airport::select([
            DB::raw("CONCAT(name, ' (', IF({$value} is NULL OR {$value} = '', {$reserve}, {$value}), ')') AS code"),
            $key
        ])
            ->where("handling_airport", true);

        if (airlineModule()){
            $list->orWhere("operated", true);
        }

        $list = $list->orderBy("name")
                    ->pluck("code", $key)
                    ->all();

        return $list;
    }


    public static function getAirlinesHandlingAirports($airlineId, $listValue = "id", $listKey = null){

        $airports = Airport::leftJoin("flights__numbers AS depFN", "depFN.departure_airport_id", "=", "airports.id")
                            ->leftJoin("flights__numbers AS arrFN", "arrFN.arrival_airport_id", "=", "airports.id")
                            ->where("airports.handling_airport", true)
                            ->where(function($sql) use ($airlineId){
                                $sql->where("depFN.airline_id", $airlineId)
                                    ->orWhere("arrFN.airline_id", $airlineId);
                            })
                            ->groupBy("airports.id", "airports.iata");

        if ($listKey){
            $airports = $airports->get([
                "airports.id as id",
                "airports.iata as iata",
            ])
                ->pluck($listValue, $listKey)
                ->all();
        }
        else {
            $airports = $airports->get([
                "airports.id as id",
                "airports.iata as iata",
            ])->pluck($listValue)
              ->all();
        }

        return $airports;

    }

    public static function getPaxNoByStation($from, $to, $airports = null, $chart = false){

        $flights =  Flight::handlingFlightsRange($from.' 00:00:00', $to.' 23:59:59', null, $airports);

        $handlingAirports = Airport::listOperatedStations("iata", "id");

        $values =  [
            "capacity_c"        => 0,
            "capacity_y"        => 0,
            "capacity_total"    => 0,
//            "pax_c"             => 0,
//            "pax_y"             => 0,
            "pax_total"         => 0,
            "sf"                => 0,
        ];

        $resultByAirports = [];

        $processedFlights = $reportByFN = $reportBySector = [];

        $resultByAirportsMorris = null;

        $types = [
            DEPARTURE,
//            ARRIVAL,
            TOTAL
        ];

        if ($flights && count($flights)) {
            foreach ($flights as $i => $flight) {
                if (count($processedFlights) && in_array($flight->id, $processedFlights)) {
                    unset($flights[$i]);
                    continue;
                }
                $processedFlights[] = $flight->id;

                // Get Capacity
                $capacity = getFlightCapacity($flight);

                $capacity_c = $capacity[0];
                $capacity_y = $capacity[1];
                $capacity_total = $capacity_c + $capacity_y;

                // Get PAX
//                $pax = getFlightPax($flight);
//                $pax_c = $pax[0];
//                $pax_y = $pax[1];
//                $pax_total = $pax_c + $pax_y;

                $pax_total = getFlightPaxTotal($flight);
                $flightNo = $flight->flightNumber;
                $airport = null;

                if ($flightNo && $pax_total){

                    // Get Airport Part
                    if (array_key_exists($flightNo->departure_airport_id, $handlingAirports)){
                        $airport = $handlingAirports[$flightNo->departure_airport_id];
                    }

                    if ($airport){
                        if (!isset($resultByAirports[$airport])){
                            $resultByAirports[$airport] = [
                                DEPARTURE => $values,
                                TOTAL     => $values,
                            ];
                        }

                        $resultByAirports[$airport][DEPARTURE]["capacity_c"]        += $capacity_c;
                        $resultByAirports[$airport][DEPARTURE]["capacity_y"]        += $capacity_y;
                        $resultByAirports[$airport][DEPARTURE]["capacity_total"]    += $capacity_total;
//                        $resultByAirports[$airport][$type]["pax_c"]             += $pax_c;
//                        $resultByAirports[$airport][$type]["pax_y"]             += $pax_y;
                        $resultByAirports[$airport][DEPARTURE]["pax_total"]         += $pax_total;

                        $resultByAirports[$airport][TOTAL]["capacity_c"]        += $capacity_c;
                        $resultByAirports[$airport][TOTAL]["capacity_y"]        += $capacity_y;
                        $resultByAirports[$airport][TOTAL]["capacity_total"]    += $capacity_total;
//                        $resultByAirports[$airport][TOTAL]["pax_c"]             += $pax_c;
//                        $resultByAirports[$airport][TOTAL]["pax_y"]             += $pax_y;
                        $resultByAirports[$airport][TOTAL]["pax_total"]         += $pax_total;
                    }

                    $sector = getSector($flightNo);

                    if (!isset($reportByFN[$flightNo->flight_number])){
                        $reportByFN[$flightNo->flight_number] = [
                            "pax"       => 0,
                            "flights"   => 0,
                            "capacity"  => 0,
                            "sector"    => $sector,
                        ];
                    }

                    $reportByFN[$flightNo->flight_number]["flights"] ++;
                    $reportByFN[$flightNo->flight_number]["pax"] += $pax_total;
                    $reportByFN[$flightNo->flight_number]["capacity"] += $capacity_total;

                    if (!isset($reportBySector[$sector])){
                        $reportBySector[$sector] = [
                            "pax"       => 0,
                            "flights"   => 0,
                            "capacity"  => 0,
                        ];
                    }

                    $reportBySector[$sector]["flights"] ++;
                    $reportBySector[$sector]["pax"] += $pax_total;
                    $reportBySector[$sector]["capacity"] += $capacity_total;

                }
            }

            // Set SF
            foreach ($resultByAirports as &$each) {
                foreach ($types as $type) {
                    $each[$type]["sf"]  = $each[$type]["capacity_total"]? 100 * round($each[$type]["pax_total"] / $each[$type]["capacity_total"], 2): 0;
                }
            }
        }

        $reportBySectorMorris = $reportBySectorMorrisSF = $sectorLabels = [];
        foreach ($reportBySector as $sector => $each) {
            $averagePAX = $each["flights"] ? round($each["pax"] / $each["flights"]) : 0;
            $averageSF = getSeatFactors($each["pax"], $each["capacity"]);

            $reportBySector[$sector]["average_pax"] = $averagePAX;
            $reportBySector[$sector]["average_sf"] = $averageSF;

            $sectorLabels[] = $sector;
            $reportBySectorMorris[] = $averagePAX;
            $reportBySectorMorrisSF[] = $averageSF;
        }

        if ($chart){
            foreach ($resultByAirports as &$each) {
                foreach ($types as $type) {
                    self::orderByValue($each[$type]);
                }
            }
            $resultByAirportsMorris = self::getMorrisChart($resultByAirports, "pax_total", true);
            $resultByAirports = self::getChart($resultByAirports, "airport", "pax_total", true);
        }

        return [$resultByAirports, $resultByAirportsMorris, $reportByFN, $reportBySector, $reportBySectorMorris,
            $reportBySectorMorrisSF, $sectorLabels];

    }

    public static function getPaxNoByStationAndAirline($from, $to, $airports, $airlines, $chart = false){

        $flights =  Flight::handlingFlightsRange($from.' 00:00:00', $to.' 23:59:59', $airlines, $airports);

        $handlingAirports = Airport::getHandlingStations("iata", "id");

        $values =  [
            "capacity_c"        => 0,
            "capacity_y"        => 0,
            "capacity_total"    => 0,
//            "pax_c"             => 0,
//            "pax_y"             => 0,
            "pax_total"         => 0,
            "sf"                => 0,
        ];

        $resultByAirports = $resultByAirlines = [];

        $processedFlights = [];

        $resultByAirlinesMorris = $resultByAirportsMorris = null;

        $handlingStations = Airport::getHandlingStationsIdAndObject();

        $types = [
            DEPARTURE,
            ARRIVAL,
            TOTAL
        ];

        if ($flights && count($flights)) {
            foreach ($flights as $i => $flight) {
                if (count($processedFlights) && in_array($flight->id, $processedFlights)) {
                    unset($flights[$i]);
                    continue;
                }
                $processedFlights[] = $flight->id;

                // Get Capacity
                $capacity = getFlightCapacity($flight);

                $capacity_c = $capacity[0];
                $capacity_y = $capacity[1];
                $capacity_total = $capacity_c + $capacity_y;

                // Get PAX
//                $pax = getFlightPax($flight);
//                $pax_c = $pax[0];
//                $pax_y = $pax[1];
//                $pax_total = $pax_c + $pax_y;

                $pax_total = getFlightPaxTotal($flight);
                $flightNo = $flight->flightNumber;
                $airport = null;

                if ($flightNo && $pax_total){

                    $type = getFlightsDepArrType($flight, $handlingStations);

                    // Have to redefine to adjust
                    if (!$type){
                        continue;
                    }

                    // Get Airline Part
                    $airline = $flightNo->airline;
                    if ($airline){

                        if (!isset($resultByAirlines[$airline->iata])){
                            $resultByAirlines[$airline->iata] = [
                                DEPARTURE => $values,
                                ARRIVAL   => $values,
                                TOTAL     => $values,
                            ];
                        }


                        $resultByAirlines[$airline->iata][$type]["capacity_c"] += $capacity_c;
                        $resultByAirlines[$airline->iata][$type]["capacity_y"] += $capacity_y;
                        $resultByAirlines[$airline->iata][$type]["capacity_total"] += $capacity_total;
//                        $resultByAirlines[$airline->iata][$type]["pax_c"] += $pax_c;
//                        $resultByAirlines[$airline->iata][$type]["pax_y"] += $pax_y;
                        $resultByAirlines[$airline->iata][$type]["pax_total"] += $pax_total;

                        $resultByAirlines[$airline->iata][TOTAL]["capacity_c"] += $capacity_c;
                        $resultByAirlines[$airline->iata][TOTAL]["capacity_y"] += $capacity_y;
                        $resultByAirlines[$airline->iata][TOTAL]["capacity_total"] += $capacity_total;
//                        $resultByAirlines[$airline->iata][TOTAL]["pax_c"] += $pax_c;
//                        $resultByAirlines[$airline->iata][TOTAL]["pax_y"] += $pax_y;
                        $resultByAirlines[$airline->iata][TOTAL]["pax_total"] += $pax_total;
                    }

                    // Get Airport Part
                    if (array_key_exists($flightNo->departure_airport_id, $handlingAirports)){
                        $airport = $handlingAirports[$flightNo->departure_airport_id];
                    }
                    else if (array_key_exists($flightNo->arrival_airport_id, $handlingAirports)){
                        $airport = $handlingAirports[$flightNo->arrival_airport_id];
                    }

                    if ($airport){
                        if (!isset($resultByAirports[$airport])){
                            $resultByAirports[$airport] = [
                                DEPARTURE => $values,
                                ARRIVAL   => $values,
                                TOTAL     => $values,
                            ];
                        }

                        $resultByAirports[$airport][$type]["capacity_c"]        += $capacity_c;
                        $resultByAirports[$airport][$type]["capacity_y"]        += $capacity_y;
                        $resultByAirports[$airport][$type]["capacity_total"]    += $capacity_total;
//                        $resultByAirports[$airport][$type]["pax_c"]             += $pax_c;
//                        $resultByAirports[$airport][$type]["pax_y"]             += $pax_y;
                        $resultByAirports[$airport][$type]["pax_total"]         += $pax_total;

                        $resultByAirports[$airport][TOTAL]["capacity_c"]        += $capacity_c;
                        $resultByAirports[$airport][TOTAL]["capacity_y"]        += $capacity_y;
                        $resultByAirports[$airport][TOTAL]["capacity_total"]    += $capacity_total;
//                        $resultByAirports[$airport][TOTAL]["pax_c"]             += $pax_c;
//                        $resultByAirports[$airport][TOTAL]["pax_y"]             += $pax_y;
                        $resultByAirports[$airport][TOTAL]["pax_total"]         += $pax_total;
                    }
                }
            }

            // Set SF
            foreach ($resultByAirports as &$each) {
                foreach ($types as $type) {
                    $each[$type]["sf"]  = $each[$type]["capacity_total"]? 100 * round($each[$type]["pax_total"] / $each[$type]["capacity_total"], 2): 0;
                }
            }

            // Set SF
            foreach ($resultByAirlines as &$each) {
                foreach ($types as $type) {
                    $each[$type]["sf"]  = $each[$type]["capacity_total"]? 100 * round($each[$type]["pax_total"] / $each[$type]["capacity_total"], 2): 0;
                }
            }
        }

        if ($chart){
            foreach ($resultByAirports as &$each) {
                foreach ($types as $type) {
                    self::orderByValue($each[$type]);
                }
            }
            $resultByAirportsMorris = self::getMorrisChart($resultByAirports, "pax_total");
            $resultByAirports = self::getChart($resultByAirports, "airport", "pax_total");


            foreach ($resultByAirlines as &$each) {
                foreach ($types as $type) {
                    self::orderByValue($each[$type]);
                }
            }
            $resultByAirlinesMorris = self::getMorrisChart($resultByAirlines, "pax_total");
            $resultByAirlines = self::getChart($resultByAirlines, "airline", "pax_total");
        }

        return [$resultByAirports, $resultByAirportsMorris, $resultByAirlines, $resultByAirlinesMorris];

    }

    public static function getMorrisChart($result, $valueField, $departureAndAll = null){

        if ($departureAndAll){
            $chart = [
                DEPARTURE => [],
            ];
        }
        else {
            $chart = [
                DEPARTURE => [],
                ARRIVAL => [],
            ];
        }

        if ($result && count($result)) {
            foreach ($result as $index => $each) {

                $chart[DEPARTURE][] = [
                    "value"  => $each[DEPARTURE][$valueField], //$each[TOTAL][$valueField] == 0 ? 0 : 100 * round($each[DEPARTURE][$valueField] / $each[TOTAL][$valueField], 2),
                    "label"  => "{$index}",
                ];

                if (!$departureAndAll) {
                    $chart[ARRIVAL][] = [
                        "value" => $each[ARRIVAL][$valueField], //$each[TOTAL][$valueField] == 0 ? 0 : 100 * round($each[ARRIVAL][$valueField] / $each[TOTAL][$valueField], 2),
                        "label" => "{$index}",
                    ];
                }

                $chart[TOTAL][] = [
                    "value"  => $each[TOTAL][$valueField] ? $each[TOTAL][$valueField] : 0, //$each[TOTAL][$valueField] == 0 ? 0 : 100 * round($each[ARRIVAL][$valueField] / $each[TOTAL][$valueField], 2),
                    "label"  => "{$index}",
                ];
            }
        }

        return $chart;
    }

    public static function orderByValue(&$array){
        if ($array && count($array)) {
            debug($array);
            try {
                uasort($array, array("App\\Models\\Airport", "compareByValue"));
            }
            catch (\Exception $e){
                debug($e);
            }
        }
    }

    static function compareByValue($first, $second)
    {
        return intval($first["pax_total"]) < intval($second["pax_total"]);
    }

    public static function getChart($result, $field, $valueField, $departureAndAll = null){
        if ($departureAndAll){
            $chart = [
                $field      => [],
                DEPARTURE   => [],
            ];

            if ($result && count($result)) {
                foreach ($result as $index => $each) {
                    $chart[$field][]    = $index;
                    $chart[DEPARTURE][] = $each[DEPARTURE][$valueField]; //$each[TOTAL][$valueField] == 0 ? 0 : 100 * round($each[DEPARTURE][$valueField] / $each[TOTAL][$valueField], 2);
                }
            }
        }
        else {
            $chart = [
                $field      => [],
                DEPARTURE   => [],
                ARRIVAL     => [],
            ];

            if ($result && count($result)) {
                foreach ($result as $index => $each) {
                    $chart[$field][]    = $index;
                    $chart[ARRIVAL][]   = $each[ARRIVAL][$valueField];   //$each[TOTAL][$valueField] == 0 ? 0 : 100 * round($each[ARRIVAL][$valueField] / $each[TOTAL][$valueField], 2);
                    $chart[DEPARTURE][] = $each[DEPARTURE][$valueField]; //$each[TOTAL][$valueField] == 0 ? 0 : 100 * round($each[DEPARTURE][$valueField] / $each[TOTAL][$valueField], 2);
                }
            }
        }

        return $chart;
    }



}
