<?php

namespace App\Jobs;

use App\Models\Flight;
use App\Models\FlightSchedule;
use App\Models\FlightScheduleAirline;
use App\Models\FlightScheduleFlight;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Http\Request;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class UploadFlightScheduleJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $params;
    protected $data;

    /**
     * Create a new job instance.
     * @var Request $request
     * @return void
     */
    public function __construct($data, $params)
    {
        $this->data = $data;
        $this->params = $params;
    }

    const MAX_EXECUTION_TIME = 400;

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '1024M');

//                    return 1;
        // Save Template
        $template = $this->saveAirlineTemplate();

        // Set Template Parent ids
        $this->setTemplateFlightsConnections($template);

        $airport = isset($this->params["airport"]) ? $this->params["airport"] : null;
        $removeOtherFlights = isset($this->params["remove_other_flights"]) ? $this->params["remove_other_flights"] : null;

        // Load To Flight Watch
        FlightSchedule::uploadAirlineTemplateToFlightWatch($template, $airport, $removeOtherFlights);

        // Set Flights Parent Ids
        $this->setFlightsConnections($template);

        flash()->success("Schedule Successfully Uploaded.");

        $this->viewData = [
            'message_type'  => "success",
            'message'       => 'Schedule Successfully Uploaded.',
        ];
    }

    function saveAirlineTemplate(){
        $flightSchedule = null;

        if ($this->data && is_array($this->data))
        {
            $dateFromAndTo = FlightSchedule::findFromAndToDates($this->data);

            $flightSchedule             = new FlightSchedule();
            $flightSchedule->name       = $this->params["template_name"];

            $flightSchedule->airport_ids = isset($this->params["airport"]) && $this->params["airport"] && count($this->params["airport"])
                ? json_encode($this->params["airport"]) : null;

            $flightSchedule->airline_id = isset($this->params["airline"]) ? $this->params["airline"] : null;
            $flightSchedule->remove_other_flights = isset($this->params["remove_other_flights"]) ? $this->params["remove_other_flights"] : null;

            $flightSchedule->date_from  = $dateFromAndTo['date_from'];
            $flightSchedule->date_to    = $dateFromAndTo['date_to'];
            $flightSchedule->created_at = date("Y-m-d H:i:s");
            $flightSchedule->created_by = Auth::user()->id;
            $flightSchedule->save();

            $insertData = $parents = $children = $airlines = [];

            $oneFlightOption = $flightSchedule->date_from == $flightSchedule->date_to;

            foreach ($this->data as $parent)
            {
                $flightScheduleFlight                           = new FlightScheduleFlight();
                //$flightScheduleFlight->parent_id          = 0;
                $flightScheduleFlight->flight_schedule_id       = $flightSchedule->id;
                $flightScheduleFlight->airline_id               = $parent['airline_id'];
                $flightScheduleFlight->operated_by_airline_id   = $parent['aircraft_airline_id'];

                $flightScheduleFlight->aircraft_id              = isset($parent['aircraft_id']) ? $parent['aircraft_id'] : null;
                $flightScheduleFlight->aircraft_type_id         = $parent['aircraft_type_id'];
                // CFG
                $flightScheduleFlight->cfg_c         = $parent['cfg_c'] ? $parent['cfg_c'] : null;
                $flightScheduleFlight->cfg_y         = $parent['cfg_y'] ? $parent['cfg_y'] : null;

                // One Flight
//                if ($oneFlightOption) {
                $flightScheduleFlight->bkd_c = $parent['bkd_c'] ? $parent['bkd_c'] : null;
                $flightScheduleFlight->bkd_y = $parent['bkd_y'] ? $parent['bkd_y'] : null;
//                }
                // Notes
                $flightScheduleFlight->notes = $parent['notes'] ? $parent['notes'] : null;

                $flightScheduleFlight->flight_number_id         = $parent['flight_number_id'];


                $flightScheduleFlight->effective_date     = $parent['start_date'];
                $flightScheduleFlight->discontinue_date   = $parent['end_date'];

                // Departure
                if ($parent["days"]){
                    $flightScheduleFlight->days           = implode("",$parent['days']);
                }

                $flightScheduleFlight->departure_time     = $parent['departure_time'];

                // Arrival
                $flightScheduleFlight->arrival_time       = $parent['arrival_time'];

                $flightScheduleFlight->frequency          = isset($parent['frequency']) ? $parent['frequency'] : "weekly";
                $flightScheduleFlight->starting_week      = isset($parent['starting_week']) ? $parent['starting_week'] : "first_week";
                $flightScheduleFlight->save();

                if ($parent['airline_id'] && !in_array($parent['airline_id'], $airlines)){
                    $airlines[] = $parent['airline_id'];
                    $flightScheduleAirline = new FlightScheduleAirline();
                    $flightScheduleAirline->flight_schedule_id = $flightSchedule->id;
                    $flightScheduleAirline->airline_id = $parent['airline_id'];
                    $flightScheduleAirline->save();
                }
            }

            if (count($insertData))
            {
                FlightScheduleFlight::insert($insertData);
            }
        }

        return $flightSchedule;

    }

    function setTemplateFlightsConnections($template){

        if ($template){
            // Find all inbound flights
            $parentFlights = FlightScheduleFlight::join("flights__numbers", "flights__numbers.id", "=", "flights__schedule_flights.flight_number_id")
                ->where('flight_schedule_id', $template->id)
                ->where('bound', 1)
                ->get(["flights__schedule_flights.*", "flight_number"]);

            foreach ($parentFlights as $parent) {
                $child = FlightScheduleFlight::join("flights__numbers", "flights__numbers.id", "=", "flights__schedule_flights.flight_number_id")
                    ->where('flight_schedule_id', $template->id)
                    ->where("flight_number",    intval($parent->flight_number) + 1)
                    ->where("aircraft_id", $parent->aircraft_id)
                    ->where("aircraft_type_id", $parent->aircraft_type_id)
                    ->where("effective_date",   $parent->effective_date)
                    ->where("discontinue_date", $parent->discontinue_date)
                    ->first(['flights__schedule_flights.*']);

                if ($child){
                    $child->parent_id = $parent->id;
                    $child->save();
                }
            }
        }
    }

    public function setFlightsConnections($template = null, $flight_number = null)
    {
        if ($template || $flight_number)
        {
            // Find all inbound flights
            $parentFlights = Flight::join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
                ->where('flight_schedule_id', $template->id)
                ->where('bound', 1)
                ->get(["flights.*", "flight_number"]);

            foreach ($parentFlights as $parent) {
                $departure = date('Y-m-d', strtotime($parent->std));
                $departureNextDay = date('Y-m-d', strtotime("+ 1 days", strtotime($parent->std)));

                $child = Flight::join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
                    ->whereNull("flights.cancelled_at")
                    ->whereNull("flights.deleted_at")
                    ->where('flight_schedule_id', $template->id)
                    ->where("flight_number",    intval($parent->flight_number) + 1)
                    ->where("aircraft_type_id", $parent->aircraft_type_id)
                    ->where("std", ">", $parent->std)
                    ->whereBetween(DB::raw("DATE(std)"), [$departure, $departureNextDay])
                    ->orderBy("std")
                    ->first(['flights.*']);

                if ($child){
                    $child->parent_id = $parent->id;
                    $child->save();
                }
            }
        }
    }
}
