<?php

namespace App\Http\Controllers;

use App\Models\Airline;
use App\Models\Condition;
use App\Models\FlightStaff;
use App\Models\LibraryView;
use App\Models\Parameter;
use App\Models\User;
use App\Models\UserHistory;
use App\Models\UserLeaveRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Jenssegers\Agent\Agent;

class UserServicesController extends Controller
{
    public function index(){
        $auth = Auth::user();
        $userID = \request("user_id") ? \request("user_id") : $auth->id;

        $user = User::find($userID);

        if (!$user){
            return redirect()->to('error/page-not-found');
        }

        if ($auth->id != $userID && $auth->user_role_id != ROLE_SUPER_ADMIN){
            return redirect()->to('error/page-not-found');
        }

        $from = \request("from") ? \request("from") : date("Y-m-d H:i:s", strtotime("- 24 hours"));

        $param = Parameter::getDefault(STAFF_PERIOD_TO_ROSTER_VIEW);
        if ($param && $param->value == CURRENT_DATE){
            $to = date("Y-m-d")." 23:59:59";
        }
        else {
            $to = \request("to") ? \request("to") : date("Y-m-d H:i:s", strtotime("+ 24 hours"));
        }

        $period = getSelectedPeriod(\request("period"), $from, $to);

        // ROSTER TAB
        $staffFlights = FlightStaff::getAllUpdated($period, $user);

        // END ROSTER
        $officeDuties = UserHistory::getStaffLeaveRequests($from, $to, $userID, null, true);
//        return 1;

        $agent = new Agent();
        $isMobile = $agent->isMobile();

        $this->viewData = [
            "list"              => $this->merge($staffFlights, $officeDuties),
            "user"              => $user,
            "isMobile"          => $isMobile,
            'airlines'          => Airline::listHandlingAirlines(),
            "staffFlights"      => $staffFlights,
            "officeDuties"      => $officeDuties,
            'conditions'        => ['' => 'Select'] + Condition::orderBy("name")->pluck('name', 'id')->all(),
            "now"               => date("Y-m-d H:i:s")
        ];

        return view("user-services.index", $this->viewData);
    }

    function merge($staffFlights, $officeDuties){
        $list = [];

        foreach ($staffFlights as &$each) {
            $each->merged_date = $each->pReportTime;

            $list[] = $each;
        }

        foreach ($officeDuties as &$each) {
            if ($each instanceof UserLeaveRequest){
                $each->merged_date = $each->date_from;
            }
            else{
                $each->merged_date = $each->condition_from;
            }

            $list[] = $each;
        }

        usort($list, array($this, "compareByDate"));

        return $list;
    }

    function compareByDate($transferA, $transferB)
    {
        return strcmp($transferA->merged_date, $transferB->merged_date);
    }

    public function update($id){

        $auth = Auth::user();
        $flightStaff = FlightStaff::find($id);

        if (!$auth){
            return redirect()->to('error/page-not-found');
        }

        if (!$flightStaff){
            flash()->error("Flight staff record not found. Please try again.");
            return redirect("user-services");
        }

        if ($auth->id != $flightStaff->user_id){
            flash()->error("Changes are allowed only for personal profile.");
            return redirect("user-services");
        }

        $maxPeriod = 18;

        $reportTime = \request("report_time");
        $releaseTime = \request("release_time");

        if (strtotime($reportTime) >= strtotime($releaseTime)){
            flash()->error("Release Time is before Start Time. Please correct and try again.");
            return redirect("user-services");
        }

        if (strtotime($reportTime) < strtotime($flightStaff->planned_report_time)){
            flash()->error("Actual Start Time can not be earlier than Planned Start Time. Please correct and try again.");
            return redirect("user-services");
        }


        if (strtotime($releaseTime) - strtotime($reportTime) > $maxPeriod * 3600){
            flash()->error("Duty period can not exceed 18 hours. Please correct and try again.");
            return redirect("user-services");
        }

        if (strtotime($releaseTime) > strtotime(date("Y-m-d H:i:s"))){
            flash()->error("Release Time cannot be in future.");
            return redirect("user-services");
        }


        $flightStaff->report_time = $reportTime;
        $flightStaff->release_time = $releaseTime;
        $flightStaff->reason = \request("reason");
        $flightStaff->save();

        flash()->success("Successfully saved.");

        return redirect("user-services");
    }
}
