<?php

namespace App\Http\Controllers;

use App\Models\Airline;
use App\Models\Delay;
use App\Models\Department;
use App\Models\FlightDelay;
use App\Models\FlightSchedule;
use App\Models\FlightScheduleFlight;
use App\Models\License;
use App\Models\Module;
use App\Models\ModulePermission;
use App\Models\Position;
use App\Models\Service;
use App\Models\User;
use App\Models\UserDepartment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use OwenIt\Auditing\Models\Audit;

class SystemLogController extends Controller
{
    public function loginAsUser($user){
        if (!$user){
            return;
        }

        Auth::logout();
        Auth::login($user);

        return redirect()->to(route("homepage"));
    }

    public function airlineService($airline_service){

        $authUser = Auth::user();

        if (!$authUser || !$authUser->role || !in_array($authUser->role->role, [ADMIN, SUPER_ADMIN])){
            return [];
        }

        $auditData = Audit::where("auditable_type", AIRLINE_SERVICE_MODEL)
            ->where("auditable_id", $airline_service->id)
            ->orderBy("created_at")
            ->with('user')
            ->get();

        $this->viewData = [
            "object"        => $airline_service,
            "auditData"     => $auditData,
            "licenses"      => License::pluck("abbr","id")->all(),

        ];

        return view("system-log/staff-license", $this->viewData);
    }

    public function staffLicense($staff_license){

        $authUser = Auth::user();

        if (!$authUser || !$authUser->role || !in_array($authUser->role->role, [ADMIN, SUPER_ADMIN])){
            return [];
        }

        $auditData = Audit::where("auditable_type", STAFF_LICENSE_MODEL)
            ->where("auditable_id", $staff_license->id)
            ->orderBy("created_at")
            ->with('user')
            ->get();

        $this->viewData = [
            "object"        => $staff_license,
            "auditData"     => $auditData,
            "licenses"      => License::pluck("abbr","id")->all(),
        ];

        return view("system-log/staff-license", $this->viewData);
    }

    public function flight($flight){

        $authUser = Auth::user();

        if (!$authUser || !$authUser->role || !in_array($authUser->role->role, [ADMIN, SUPER_ADMIN])){
            return [];
        }

        $this->viewData = [
            'flightsAudit'      => $this->getFlightAudit($flight),
            "flight"            => $flight,
            "licenses"          => License::pluck("abbr","id")->all(),
            "flightSchedules"   => FlightSchedule::pluck("name", "id")->all(),
        ];

        return view("system-log/flight", $this->viewData);
    }

    public function flightStaff($flight){

        $authUser = Auth::user();

        if (!$authUser || !$authUser->role || !in_array($authUser->role->role, [ADMIN, SUPER_ADMIN])){
            return [];
        }

        $this->viewData = [
            'staffAudit'  => $this->getStaffAudit($flight),
            "flight"      => $flight,
            "services"    => Service::pluck("abbr","id")->all(),
            "licenses"    => License::pluck("abbr","id")->all(),
            "users"         => User::getListByID()
        ];

        return view("system-log/flight-staff", $this->viewData);
    }

    function getStaffAudit($flight){
        if (!isAdminOrSuperAdmin()){
            return [];
        }

        $flightStaff = Audit::leftJoin("flights__staff", "flights__staff.id", "=", "audits.auditable_id")
            ->where("auditable_type", FLIGHT_STAFF_MODEL)
            ->where(function($sql) use ($flight){
                $sql->where("flight_id", $flight->id)
                    ->orWhere(function($sql) use ($flight){
                        $sql->where("old_values", "LIKE", '%"flight_id":'.$flight->id.'%')
                            ->orWhere("new_values", "LIKE", '%"flight_id":'.$flight->id.'%');
                    });
            })
            ->orderBy("audits.created_at")
            ->with('auditable')
            ->get([
                "audits.*"
            ]);


        return $flightStaff;

    }

    function getFlightAudit($flight){

        if (!isAdminOrSuperAdmin()){
            return [];
        }

        $result = Audit::where("auditable_type", FLIGHT_MODEL)
            ->where("auditable_id", $flight->id)
            ->orderBy("created_at")
            ->with('user')
            ->get();

        $delayIDs = Audit::where("auditable_type", FLIGHT_DELAY_MODEL)
            ->where(function($sql) use ($flight){
                $sql->where("old_values", "LIKE", '%"flight_id":'.$flight->id.'%')
                    ->orWhere("new_values", "LIKE", '%"flight_id":'.$flight->id.'%');
            })
            ->get([
                "auditable_id as id"
            ])
            ->pluck("id")
            ->all();

        $delays = Audit::whereIn("auditable_id", $delayIDs)
            ->orderBy("created_at")
            ->with('user')
            ->get();

        foreach ($delays as $i => $delay) {
            $dl = FlightDelay::with("delay")->find($delay->auditable_id);

            if ($dl){
                $delays[$i]->dl = $dl->delay;
            }
            else {
                $prop = $delay->old_values ? $delay->old_values : $delay->new_values;
                if (isset($prop["delay_id"])){
                    $delays[$i]->dl = Delay::find($prop["delay_id"]);
                }
            }
        }

        $flightsData = [];
        foreach ($result as $each) {

            $dataInserted = false;
            $ignoreData = [];
            foreach($each->old_values as $prop => $val){

                if (array_key_exists($prop, $each->new_values)) {
                    if (in_array($prop, [STD, PTD, ETD, ATD, ABN,  STA, PTA, ETA, TDN, ATA])){
                        if (strtotime($val) == strtotime($each->new_values[$prop])){
                            $ignoreData[] = $prop;
//                            debug($prop."/".strtotime($val)."=".strtotime($each->new_values[$prop]));
                            continue;
                        }
                    }
                    else {
                        if ($val == $each->new_values[$prop]){
                            $ignoreData[] = $prop;
                            continue;
                        }
                    }
                }

                $dataInserted = true;
            }

            foreach($each->new_values as $prop => $val){
                if (in_array($prop, $ignoreData)){
                    continue;
                }
                $dataInserted = true;
            }

            if ($dataInserted){
                $data = new \stdClass();
                $data->created_at = $each->created_at;
                $data->user = $each->user;
                $data->old_values = $each->old_values;
                $data->new_values = $each->new_values;

                foreach ($ignoreData as $item) {
                    unset($data->old_values[$item]);
                    unset($data->new_values[$item]);
                }
                $flightsData[] = $data;
            }
        }

        return [
            "flight"        => $flightsData,
            "delays"        => $delays,
        ];
    }

    public function user($user){

        $authUser = Auth::user();
        $data = [];

        if (!$authUser || !$authUser->role || !in_array($authUser->role->role, [SUPER_ADMIN])){
            return redirect()->to('error/page-not-found');
        }

        $userData = Audit::where("auditable_type", USER_MODEL)
            ->where("auditable_id", $user->id)
            ->where("user_id", "!=", 44)
            ->orderBy("created_at")
            ->with(['user', 'auditable'])
            ->get();


        foreach ($userData as $each) {
            $data[] = $each;
        }

        $userDepartment = UserDepartment::where("user_id", $user->id)->first();

        if ($userDepartment) {
            $userDepartmentData = Audit::where("auditable_type", USER_DEPARTMENT_MODEL)
                ->where("auditable_id", $userDepartment->id)
                ->where("user_id", "!=", 44)
                ->orderBy("created_at")
                ->with(['user'])
                ->get();

            foreach ($userDepartmentData as $each) {
                $data[] = $each;
            }
        }

        $ids = $user->service()->pluck("id");
        $userID = $user->id;
        $serviceData = Audit::where("auditable_type", STAFF_SERVICE_MODEL)
            ->where(function($sql) use ($ids, $userID){
                $sql->whereIn("auditable_id", $ids)
                    ->orWhere("old_values", "LIKE", '%"user_id":'.$userID.'%');
            })
            ->where("user_id", "!=", 44)
            ->orderBy("created_at")
            ->with('user')
            ->get();


        foreach ($serviceData as $each) {
            $data[] = $each;
        }

        $ids = $user->license()->pluck("id");
        $licenseData = Audit::where("auditable_type", STAFF_LICENSE_MODEL)
            ->whereIn("auditable_id", $ids)
            ->where("user_id", "!=", 44)
            ->orderBy("created_at")
            ->with('user')
            ->get();

        foreach ($licenseData as $each) {
            $data[] = $each;
        }

        $ids = $user->userModule()->pluck("id");
        $userID = $user->id;
        $moduleData = Audit::where("auditable_type", USER_MODULE_MODEL)
            ->where(function($sql) use ($ids, $userID){
                $sql->whereIn("auditable_id", $ids)
                    ->orWhere("old_values", "LIKE", '%"user_id":'.$userID.'%');
            })
            ->where("user_id", "!=", 44)
            ->orderBy("created_at")
            ->with('user')
            ->get();


        foreach ($moduleData as $each) {
            $data[] = $each;
        }

        $ids = $user->history()->pluck("id");
        $userID = $user->id;
        $historyData = Audit::where("auditable_type", USER_HISTORY_MODEL)
            ->where(function($sql) use ($ids, $userID){
                if (count($ids)){
                    $sql->whereIn("auditable_id", $ids)
                        ->orWhere("old_values", "LIKE", '%"user_id":'.$userID.'%');
                }
                else {
                    $sql->where("old_values", "LIKE", '%"user_id":'.$userID.'%');
                }
            })
            ->where("user_id", "!=", 44)
            ->orderBy("created_at")
            ->with('user')
            ->get();


        foreach ($historyData as $each) {
            $data[] = $each;
        }

        usort($data, array($this, "compareByDate"));

        $this->viewData = [
            "departments" => Department::withTrashed()->pluck("name","id")->all(),
            "positions"   => Position::withTrashed()->pluck("name","id")->all(),
            "airlines"    => Airline::pluck("iata","id")->all(),
            "services"    => Service::pluck("abbr","id")->all(),
            "licenses"    => License::pluck("abbr","id")->all(),
            "modules"    => Module::pluck("name","id")->all(),
            "modulePermissions"    => ModulePermission::pluck("name","id")->all(),
            "user"        => $user,
            "auditData"   => $data,
        ];

        return view("system-log/user", $this->viewData);
    }

    function compareByDate($transferA, $transferB)
    {
        return strcmp($transferB->created_at, $transferA->created_at);
    }
}
