<?php

namespace App\Http\Controllers;

use App\Imports\Schedule;
use App\Models\Airport;
use App\Models\Department;
use App\Models\Location;
use App\Models\Position;
use App\Models\User;
use App\Models\UserDepartment;
use Carbon\Carbon;
use Illuminate\Http\Request;

use App\Http\Requests;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Maatwebsite\Excel\Collections\SheetCollection;
use Maatwebsite\Excel\Facades\Excel;

class StaffUploadController extends Controller
{

    /**
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        $this->viewData = [
            "staffExistList" => $request->get("staffExistList"),
            "messageType"    => $request->get("message_type"),
            "message"        => $request->get("message"),
        ];

        return view("staff-upload/index", $this->viewData);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if ($request->hasFile('file')) {

            $file = $request->file('file');
            // Load Excel File

            if (!in_array($file->getClientOriginalExtension(), ['xls', 'xlsx'])) {
                flash()->error('Wrong Format(' . $file->getClientOriginalExtension() . '). Please Make Sure The Format is XLS or XLSX.');
            } else {

                $locations = Location::listAirports("id", "iata");

                $loadObject = Excel::toCollection(new Schedule, $file);

                if (!$loadObject || !count($loadObject)) {
                    $this->viewData = [
                        'message_type'  => "danger",
                        'message'       => 'File Is Empty. Found No Data To Upload.',
                    ];
                }

                $data = [];
                foreach ($loadObject as $sheet){
                    $pageData = $this->GetUploadedArray($sheet, $locations);
                    foreach ($pageData as $each) {
                        if ($each && count($each)) {
                            $data[] = $each;
                        }
                    }
                }

                $staffExistList = [];

                $reportsToUserIDs = User::get([
                    DB::raw("CONCAT(last_name, ' ', first_name) as name"),
                    "id",
                ])->pluck("id", "name")
                    ->all();
                debug($reportsToUserIDs);


                foreach ($data as $each) {
                    if ($this->CreateNewStaff($each, $reportsToUserIDs)){
                        $staffExistList[] = $each;
                    }
                }

                $this->viewData = [
                    "staffExistList"    => $staffExistList,
                    'message_type'      => "success",
                    'message'           => 'Successfully Uploaded.',
                ];
            }
        } else {
            $this->viewData = [
                'message_type'  => "danger",
                'message'       => 'No File Uploaded. Please select Excel sheet and press upload.',
            ];
        }

        return redirect()->to("staff-upload")->withInput($this->viewData);
    }

    public function GetUploadedArray($loadObject, $locations){
        $finalData = [];
        // Starting Index
        $k = 0;

        $fieldCount = 11;
        $minHeaderFieldRequired = 2;
        $headerFound = false;

        foreach ($loadObject as $i => $object) {
            // Do Not Include Header Row
            // Find Starting Index $k
            if (!$headerFound){
                foreach ($object as $j => $title) {
                    if ($title){
                        $fieldExists = 0;
                        for($s = $j; $s < $j + $fieldCount; $s++){
                            if (isset($object[$s]) && $object[$s] != ""){
                                $fieldExists++;
                            }
                        }
                        if ($fieldExists > $minHeaderFieldRequired){
                            $headerFound = true;
                            $k = $j;
                            break;
                        }
                    }
                }
                continue;
            }

            if (/*!isset($object[$k]) || $object[$k] == "" || */!isset($object[$k + 1]) || $object[$k + 1] == ""){
                continue;
            }

            foreach ($object as $m => $item) {
                $object[$m] = trim($item);
            }

            $data = [];

            $name = $this->GetName($object[$k + 1]);

            $data['staff_number']   = str_replace([".","/","-"], "", $object[$k]);
            $data['last_name']      = $name[0];
            $data['first_name']     = $name[1];
            $data['dob']            = isset($object[$k + 2]) && $object[$k + 2] ? $this->GetDate($object[$k + 2]) : null;
            $data['doj']            = isset($object[$k + 3]) && $object[$k + 3] ? $this->GetDate($object[$k + 3]) : null;
            $data['cost_center']    = isset($object[$k + 4]) && $object[$k + 4] ?  $object[$k + 4] : null;


            $data['station']        = isset($object[$k + 5]) && $object[$k + 5] ? $object[$k + 5] : null;
            $data['location_id']    = isset($object[$k + 5]) && $object[$k + 5] ? $this->GetLocation($object[$k + 5], $locations) : null;

            $data['email']          = isset($object[$k + 6]) && $object[$k + 6] ? $object[$k + 6] : null;

            $department = isset($object[$k + 7]) && $object[$k + 7] ? Department::getOrCreateDepartment($object[$k + 7]) : null;
            $data['department']     = $department;
            $data['department_id']  = $department ? $department->id : null;


            $position = isset($object[$k + 8]) && $object[$k + 8] ? Position::getOrCreatePosition($object[$k + 8], $department) : null;
            $data['position']       = $position;
            $data['position_id']    = $position ? $position->id : null;

            $data['crew_type']      = isset($object[$k + 9]) && $object[$k + 9] ? $object[$k + 9] : null;

            if (isset($object[$k + 10]) && $object[$k + 10]) {
                $reportsToName = $this->GetName($object[$k + 10]);

                $data['reports_to_full_name'] = $reportsToName[0]." ".$reportsToName[1];
            }


            $finalData[] = $data;
        }

        return $finalData;
    }

    public function GetDate($string){
        if ($string){
            return \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($string)->format("Y-m-d");

            if (contains($string, ".")){
                $symbol = ".";
            }
            elseif (contains($string, "/")){
                $symbol = "/";
            }
            else {
                $symbol = "-";
            }

            if (strlen($string) == 10){
                return \DateTime::createFromFormat("m{$symbol}d{$symbol}Y", $string)->format("Y-m-d");
            }
            elseif (strlen($string) == 8){

                preg_match('/(\d{2})'.$symbol.'(\d{2})'.$symbol.'(\d{2})/', $string, $match);

                if ($match){
                    $currYear = intval(date("y"));
                    if (isset($match[3]) && $match[3] && $match[3] > $currYear){
                        $match[3] = "19".$match[3];
                        $date = \DateTime::createFromFormat("m{$symbol}d{$symbol}Y", $match[1]."-".$match[2]."-".$match[3]);
                        return $date->format("Y-m-d");
                    }

                    return \DateTime::createFromFormat("m{$symbol}d{$symbol}y", $string)->format("Y-m-d");
                }

            }
        }

        return null;
    }

    public function GetName($string){
        $name = explode(",", $string);

        return [(isset($name[0]) ? trim($name[0]) : ""), (isset($name[1]) ? trim($name[1]) : "")];
    }

    /**
     * @param $data
     * @param $reportsToUserIDs
     */
    public function CreateNewStaff($data, &$reportsToUserIDs){

        $user = User::where("last_name", $data['last_name'])
                            ->where("first_name", $data['first_name']);

        if (isset($data['dob']) && $data['dob']){
            //$user->where("dob", $data['dob']);
        }

        $user = $user->first();

        if (!$user) {
            $user = new User;
            $user->staff_number = $data['staff_number'];
            $user->last_name = $data['last_name'];
            $user->first_name = $data['first_name'];
            $email = isset($data['email']) && $data['email'] ? $data['email'] : generateEmail($user->last_name, $user->first_name, "tajikair.avbis.aero");

            $user->email = $email ? trim($email) : null;

            // Random Password
            $generatePassword = generateRandomPassword();
            $user->password = Hash::make($generatePassword);
            $user->tmp_password = $generatePassword;

            $user->save();
        }

        // Create Crew
        if (isset($data["crew_type"])){
            createCrew(ucwords($data["crew_type"]), $user, $data['position']);
        }

        // Users Department
        $userDepartment = UserDepartment::where("user_id", "=", $user->id)
                        ->first();

        if (!$userDepartment) {
            $userDepartment = new UserDepartment();
            $userDepartment->user_id = $user->id;
        }

        if (isset($data["department_id"])){
            $userDepartment->department_id = $data["department_id"];
        }

        $position = null;
        if (isset($data["position_id"])) {
            $userDepartment->position_id = $data["position_id"];
        }
        $userDepartment->save();


        // Update fields
        if (isset($data['dob']) && $data['dob']){
            $user->dob = $data['dob'];
        }
        if (isset($data['doj']) && $data['doj']){
            $user->doj = $data['doj'];
        }
        if (isset($data['cost_center']) && $data['cost_center']){
            $user->cost_center = $data['cost_center'];
        }
        if ($data['location_id']){
            $user->location_id = $data['location_id'];
        }

        // Report to User id
        if (isset($data["reports_to_full_name"]) && $data["reports_to_full_name"]){
            // Found match
            if (array_key_exists($data["reports_to_full_name"], $reportsToUserIDs)){
                $user->reports_to_user_id = $reportsToUserIDs[$data["reports_to_full_name"]];
            }

            // add this user to list if not exists
            if ($user->last_name && $user->first_name && !array_key_exists($user->last_name." ".$user->first_name, $reportsToUserIDs)){
                $reportsToUserIDs[$user->last_name." ".$user->first_name] = $user->id;
            }
        }
        $user->save();

    }

    public function GetLocation($station, $locations){
        if (!$station || !array_key_exists($station, $locations)){
            return null;
        }

        return $locations[$station];
    }

}
