<?php

namespace App\Http\Controllers;

use App\Classes\Staff\Report\OverlapHoursCalculation;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Condition;
use App\Models\Department;
use App\Models\FlightStaff;
use App\Models\LibraryView;
use App\Models\Location;
use App\Models\Module;
use App\Models\Parameter;
use App\Models\Position;
use App\Models\Service;
use App\Models\StaffAirline;
use App\Models\StaffLicense;
use App\Models\StaffService;
use App\Models\StaffStation;
use App\Models\User;
use App\Models\UserDepartment;
use App\Models\UserHistory;
use App\Models\UserLeaveStatus;
use App\Models\UserModule;
use App\Repositories\Interfaces\ILeaveRepository;
use App\Repositories\Interfaces\IUserEntitlementRepository;
use App\Repositories\Interfaces\IUserLeaveRequestRepository;
use App\Repositories\Interfaces\IUserLeaveStatusRepository;
use Illuminate\Http\Request;

use App\Http\Requests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Hash;



class StaffController extends Controller
{
    public function index()
    {
        $handlingAirports = StaffStation::currentStaffStationsList(IATA, null, true);

        if (!count($handlingAirports)){
            $handlingAirports = Airport::listHandlingStations();
        }

        $staffList = $this->staffByAirports(User::getUsersByLocation(array_keys($handlingAirports)), $handlingAirports, true);

        $this->viewData = [
            'staffList'             => $staffList,
            'airports'              => [ALL => ALL],
            'sprAdm'                => isSprAdm(),
            'isAdm'                 => isAdmin(),

            'terminatedPermission'  => checkIfModulePermissionIsEnabled("Staff List", "Deleted List")
        ];

        return view("staff.index", $this->viewData);
    }

    public function byStation()
    {
        $handlingAirports = StaffStation::currentStaffStationsList(IATA, null, true);

        if (!count($handlingAirports)){
            $handlingAirports = Airport::listHandlingStations();
        }

        $staffList = $this->staffByAirports(User::getUsersByLocation(array_keys($handlingAirports)), $handlingAirports);

        $this->viewData = [
            'staffList'             => $staffList,
            'airports'              => $handlingAirports,
            'sprAdm'                => isSprAdm(),
            'isAdm'                 => isAdmin(),

            'terminatedPermission'  => checkIfModulePermissionIsEnabled("Staff List", "Deleted List")
        ];

        return view("staff.by-station", $this->viewData);
    }

    public function terminatedDeleted()
    {
        $handlingAirports = StaffStation::currentStaffStationsList(IATA, null, true);

        if (!count($handlingAirports)){
            $handlingAirports = Airport::listHandlingStations();
        }

        $terminatedStaffList = $this->staffByAirports(User::getUsersByLocation(array_keys($handlingAirports), true), $handlingAirports);
        $deletedStaffList = $this->staffByAirports(User::getUsersByLocation(array_keys($handlingAirports), false, true), $handlingAirports);

        $this->viewData = [
            'terminatedStaffList'   => $terminatedStaffList,
            'deletedStaffList'      => $deletedStaffList,
            'airports'              => $handlingAirports,
            'sprAdm'                => isSprAdm(),
            'isAdm'                 => isAdmin(),
        ];

        return view("staff.terminated-deleted", $this->viewData);
    }


    public function staffByAirports($staffList, &$airports, $onlyAll = false, $includeAll = false){

        $staffByAirports = [];

        if ($onlyAll || $includeAll){
            $staffByAirports[ALL] = [];
        }

        foreach ($staffList as $each) {

            if ($each->location && $each->location->airport) {
                $ap = $each->location->airport->iata;
            }
            else {
                $ap = "N/A";
                if ($each->user_role_id == ROLE_SUPER_ADMIN){
                    continue;
                }
            }

            if (!isset($staffByAirports[$ap])){
                $staffByAirports[$ap] = [];
            }

            $staffByAirports[$ap][] = $each;

            // All
            if ($onlyAll || $includeAll){
                $staffByAirports[ALL][] = $each;
            }
        }

        if (isset($staffByAirports["N/A"]) && !in_array("N/A", $airports)){
            $airports[] = "N/A";
        }

        return $onlyAll ? [ALL => $staffByAirports[ALL]] : $staffByAirports;
    }

    /**
     * @param Request $request
     * @param $user
     * @param IUserLeaveRequestRepository $userLeaveRequest
     * @param IUserLeaveStatusRepository $userLeaveStatus
     * @param ILeaveRepository $leaveRepository
     * @param IUserEntitlementRepository $userEntitlementRepository
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function edit(Request $request, $user, IUserLeaveRequestRepository $userLeaveRequest,
                         IUserLeaveStatusRepository $userLeaveStatus,
                         ILeaveRepository $leaveRepository,
                         IUserEntitlementRepository $userEntitlementRepository)
    {
        ini_set("max_execution_time", 120);
        ini_set('memory_limit', '128M');

        $user = User::with("position", "department")->find($user->id);

        // Redirect If Not Found
        if (!$user || $user == '') {
            return redirect()->to('error/page-not-found');
        }

        $authUser = Auth::user();

        if (!$authUser->user_role_id || $authUser->user_role_id == ROLE_USER){
            if (($authUser->id == $user->id)) {
                return redirect()->to("staff/{$user->id}");
            }
            else {
                return redirect()->to('error/page-not-found');
            }
        }

        $accessGranted = isUserAuthorizedToView($user);

        // Redirect if Have no Rights
        if (!$accessGranted){
            return redirect()->to('error/page-not-found');
        }

        $from = $request->get("from") ? $request->get("from") : date("Y-m-d");

        $param = Parameter::getDefault(STAFF_PERIOD_TO_ROSTER_VIEW);
        if ($param && $param->value == CURRENT_DATE){
            $to = date("Y-m-d");
        }
        else {
            $to = $request->get("to") ? $request->get("to") : date("Y-m-d", strtotime(" + 6 days"));
        }
        $period = getSelectedPeriod($request->get("period"), $from, $to);

        // ROSTER TAB
        $staffFlights = FlightStaff::getAll($period, $user);
        // END ROSTER

        // HOURS TAB
        $staffHours = FlightStaff::getHoursByUser($staffFlights, true);
        // END HOURS

        $departmentId = $user->department && count($user->department) ? $user->department[0]->id : null;

        $positions = [];
        if ($departmentId){
            $positions = Position::getPositionsByDepartment($user->department[0]->id);
        }

        $staffServices = StaffService::where("user_id", $user->id)
            ->orderBy("airline_id")
            ->get();

        $staffServicesByAirlines = [];
        foreach ($staffServices as $each) {
            if (!isset($staffServicesByAirlines[$each->airline_id])){
                $staffServicesByAirlines[$each->airline_id] = [];
            }

            $staffServicesByAirlines[$each->airline_id][] = $each->service_id;
        }

        $staffLicenses = StaffLicense::with(['license'])
            ->where('user_id', $user->id)
            ->get();

        $position = getUserPosition($user);

        // Leaves And Others
        $hasEmployees = hasEmployeesInLine($user);// $this->user->hasUserReportees($user->id, $this->organization);

        $dateFrom = $request->get("dateFrom") ? $request->get("dateFrom") :date("Y-01-01");
        $dateTo = $request->get("dateTo") ? $request->get("dateTo") : date("Y-m-d");

        $employeesStatus = $this->getUserRequests($user, $dateFrom, $dateTo);

        if ($hasEmployees && $position->show_leave_requests){

//            if ($authUser->id == $user->id) {
            $data = UserLeaveStatus::with("leaveRequest")
                ->join("users__leave_requests", "users__leave_requests.id", "=", "users__leave_status.request_id")
                ->where("report_to_user_id", "!=", $authUser->id)
                ->where(function ($sql) use ($dateFrom, $dateTo){
                    $sql->whereBetween("date_from", [$dateFrom, $dateTo." 23:59:59"])
                        ->orWhereBetween("date_to", [$dateFrom, $dateTo." 23:59:59"])
                        ->orWhereBetween("users__leave_requests.created_at", [$dateFrom, $dateTo." 23:59:59"])
                        ->orWhereBetween("users__leave_requests.updated_at", [$dateFrom, $dateTo." 23:59:59"]);
                })
                ->orderBy("updated_at", "DESC")
                ->get([
                    "users__leave_status.*"
                ]);

            $requests = $employeesStatus->pluck("request_id")->all();


            foreach ($data as $item) {
                if (!$item->request){
                    continue;
                }

                if (!in_array($item->request_id, $requests)) {
                    if (in_array($item->leaveRequest->status_id, [1, 2])) {
                        $employeesStatus[] = $item;
                        $requests[] = $item->request_id;
                    }
                }
            }
//            }
        }

        // HRM Staff
        $currentYear = (int)date('Y');
        $last3years = [$currentYear, ++$currentYear];


        $entitlements = $userEntitlementRepository->findAllByAttributes([
            'year'      => [WHERE_IN, $last3years],
            'user_id'   => $user->id
        ]);

        // Get Selected Tab
        $selectedTab = null;
        if (\request()->get("tab") && \request()->get("tab") != ""){
            $selectedTab = \request()->get("tab");
        }
        else if ($hasEmployees){
            $selectedTab = "employees_requests";
        }


        // Modules
        $modules = Module::with(["permission"])->orderBy("category")->get();

        $userModulePermissions = UserModule::where("user_id", $user->id)
            ->orderBy("module_id")
            ->orderBy("module_permission_id")
            ->pluck("module_permission_id")
            ->all();



        // Staff Hours
        list($overlappingHours, $staffHistory, $periodUnique) = $this->getStaffHoursWithOfficeDuties($staffFlights, $period, $user);
        debug($overlappingHours);
//        return 1;

        $this->viewData = [
            'dateFrom'          => $dateFrom,
            'dateTo'            => $dateTo,

            "readAndSignPermission" => checkIfModulePermissionIsEnabled(LIBRARY, "read_and_sign"),

            "page"      => "staff/{$user->id}/edit",

            "libraryViews"      => LibraryView::getAuthLibrarySignDocs($user),


            "from"      => $from,
            "to"        => $to,

            'overlappingHours'  => $overlappingHours,
            "staffHistory"      => $staffHistory,
            'periodUnique'      => $periodUnique,

            'dateYesterday'     => date('Y-m-d',strtotime(" - 1 days")),
//            'periods'           => getPeriods(),
            'monthNames'        => getMonthsArray(),
            'years'             => getYearsArray(),

            "staffStations" => StaffStation::currentStaffStationsList(IATA, $user->id),
            "staffAirlines" => StaffAirline::currentStaffAirlinesList(IATA, $user->id),

            // Modules
            'modules'       => $modules,
            'userModulePermissions'   => $userModulePermissions,

            //  Roster
            'staffFlights'  => $staffFlights,
            'staffHours'    => $staffHours,
            'periods'       => ["" => "Select", "planned" => "Current Roster", "last7days" => "Last 7 Days", "last28days" => "Last 28 Days", "last90days"=>"Last 90 Days", "last365days" => "Last 365 Days"],

            'authUser'      => $authUser,
            // Others
            'userId'        => $user->id,
            'user'          => $user,
            'gender'        => $this->gender,
            'usersList'     => ['' => 'Select'] + User::getUsersList($departmentId, TRUE),

            'position'      => $position,
            'positions'     => $positions,

            'department'    => $departmentId,
            'departments'   => Department::listDepartments(),

            'picture'       => $user->picture ? asset("storage/users/" . $user->picture) : asset("/assets/img/default-user-picture.png"),

            'services'      => Service::pluck("abbr", "id")->all(),
            'staffServices' => $staffServicesByAirlines,
            'staffLicenses' => $staffLicenses,

            'airlines'      => Airline::listHandlingAirlines(),

            'isSuperAdmin'          => isSuperAdmin(),

            // HRM
            'types'                 => ['Calendar' => "Calendar", 'Working' => "Working"],
            'last3years'            => $last3years,
            'entitlementsLast3Year' => $entitlements->keyBy('year'),

            // OTHERS
            'selectedTab'       => $selectedTab,
            'leaves'            => ["" => "Select"] + $leaveRepository->findAndListModelVariable(['is_main' => true], ['name', 'id']),
            'otherLeaves'       => ["" => "Select"] + $leaveRepository->findAndListModelVariable(['is_main' => NULL], ['name', 'id']),
            'entitlements'      => $user->entitlements->keyBy('year'),
            'hasEmployees'      => $hasEmployees,
            'leaveRequests'     => $userLeaveRequest->findAllByAttributes(['user_id' => $user->id], ['leaveStatus']),
            'employeesStatus'   => $employeesStatus,
            'conditions'        => ['' => 'Select'] + Condition::where("type", 1)->orderBy("name")->pluck('name', 'id')->all(),
            'countries'         => isset($countries) ? $countries : null,

        ];

        return view('staff/edit', $this->viewData);
    }

    public function show(Request $request, $user, IUserLeaveRequestRepository $userLeaveRequest,
                         IUserLeaveStatusRepository $userLeaveStatus,
                         ILeaveRepository $leaveRepository,
                         IUserEntitlementRepository $userEntitlementRepository)
    {

        ini_set("max_execution_time", 120);
        ini_set('memory_limit', '128M');

        $user = User::with("position", "department")->find($user->id);;

        $authUser = Auth::user();

        // Redirect If Not Found
        if (!$user || $user == '') {
            debug("NO USER FOUND");
            return redirect()->to('error/page-not-found');
        }

        // Redirect if Have no Rights
        $accessGranted = isUserAuthorizedToView($user);

        // Redirect if Have no Rights
        if (!$accessGranted){
            return redirect()->to('error/page-not-found');
        }

        $from = $request->get("from") ? $request->get("from") : date("Y-m-d");
        $to = $request->get("to") ? $request->get("to") : date("Y-m-d", strtotime(" + 6 days"));

        $period = getSelectedPeriod($request->get("period"), $from, $to);

        // Set Maximum Queryable Period + 14 days
        $maxPeriodDate = date("Y-m-d", strtotime(" + 14 days"));
        if (strtotime($period["to"]) > strtotime($maxPeriodDate)){
            $period["to"] = $maxPeriodDate;
        }

        // Limit Period To
        $param = Parameter::getDefault(STAFF_PERIOD_TO_ROSTER_VIEW);
        if ($param && $param->value == CURRENT_DATE){
            $period["to"] = date("Y-m-d");
        }

        // ROSTER TAB
        $staffFlights = FlightStaff::getAll($period, $user);
        // END ROSTER

        // HOURS TAB
        $staffHours = FlightStaff::getHoursByUser($staffFlights, true);
        // END HOURS

        $departmentId = $user->department && count($user->department) ? $user->department[0]->id : null;

        $positions = [];
        if ($departmentId){
            $positions = Position::getPositionsByDepartment($user->department[0]->id);
        }

        $staffServices = StaffService::where("user_id", $user->id)
            ->orderBy("airline_id")
            ->get();

        $staffServicesByAirlines = [];
        foreach ($staffServices as $each) {
            if (!isset($staffServicesByAirlines[$each->airline_id])){
                $staffServicesByAirlines[$each->airline_id] = [];
            }

            $staffServicesByAirlines[$each->airline_id][] = $each->service_id;
        }

        $staffLicenses = StaffLicense::with(['license'])
            ->where('user_id', $user->id)
            ->get();

        $position = getUserPosition($user);

        // Leaves And Others
        $hasEmployees = hasEmployeesInLine($user);// $this->user->hasUserReportees($user->id, $this->organization);

        $dateFrom = $request->get("dateFrom") ? $request->get("dateFrom") :date("Y-01-01");
        $dateTo = $request->get("dateTo") ? $request->get("dateTo") : date("Y-m-d");

        $employeesStatus = $this->getUserRequests($user, $dateFrom, $dateTo);

        if ($hasEmployees && $position->show_leave_requests){

            $data = UserLeaveStatus::with("leaveRequest")
                ->join("users__leave_requests", "users__leave_requests.id", "=", "users__leave_status.request_id")
                ->where("report_to_user_id", "!=", $authUser->id)
                ->where(function ($sql) use ($dateFrom, $dateTo){
                    $sql->whereBetween("date_from", [$dateFrom, $dateTo." 23:59:59"])
                        ->orWhereBetween("date_to", [$dateFrom, $dateTo." 23:59:59"])
                        ->orWhereBetween("users__leave_requests.created_at", [$dateFrom, $dateTo." 23:59:59"])
                        ->orWhereBetween("users__leave_requests.updated_at", [$dateFrom, $dateTo." 23:59:59"]);
                })
                ->orderBy("updated_at", "DESC")
                ->get([
                    "users__leave_status.*"
                ]);

            $requests = $employeesStatus->pluck("request_id")->all();


            foreach ($data as $item) {
                if (!in_array($item->request_id, $requests)){
                    if (in_array($item->leaveRequest->status_id, [1, 2])){
                        $employeesStatus[] = $item;
                        $requests[] = $item->request_id;
                    }
                }
            }
        }

        // HRM Staff
        $currentYear = (int)date('Y');
        $last3years = [$currentYear, ++$currentYear];


        $entitlements = $userEntitlementRepository->findAllByAttributes([
            'year'      => [WHERE_IN, $last3years],
            'user_id'   => $user->id
        ]);

        // Get Selected Tab
        $selectedTab = null;
        if (\request()->get("tab") && \request()->get("tab") != ""){
            $selectedTab = \request()->get("tab");
        }
        else if ($hasEmployees){
            $selectedTab = "employees_requests";
        }

        // Modules
        $modules = Module::with(["permission"])->orderBy("category")->get();

        $userModulePermissions = UserModule::where("user_id", $user->id)
            ->orderBy("module_id")
            ->orderBy("module_permission_id")
            ->pluck("module_permission_id")
            ->all();

        // Staff Hours
        list($overlappingHours, $staffHistory, $periodUnique) = $this->getStaffHoursWithOfficeDuties($staffFlights, $period, $user);

        $this->viewData = [
            'dateFrom'          => $dateFrom,
            'dateTo'            => $dateTo,

            "readAndSignPermission" => checkIfModulePermissionIsEnabled(LIBRARY, "read_and_sign"),

            "page"      => "staff/{$user->id}",
            "from"      => $from,
            "to"        => $to,

            "libraryViews"      => LibraryView::getAuthLibrarySignDocs($user),
            'isSuperAdmin'          => isSuperAdmin(),


            'overlappingHours'  => $overlappingHours,
            "staffHistory"      => $staffHistory,
            'periodUnique'      => $periodUnique,

            'dateYesterday'     => date('Y-m-d',strtotime(" - 1 days")),
//            'periods'           => getPeriods(),
            'monthNames'        => getMonthsArray(),
            'years'             => getYearsArray(),

            "staffStations" => StaffStation::currentStaffStationsList(IATA, $user->id),
            "staffAirlines" => StaffAirline::currentStaffAirlinesList(IATA, $user->id),

            // Modules
            'modules'       => $modules,
            'userModulePermissions'   => $userModulePermissions,

            //  Roster
            'staffFlights'  => $staffFlights,
            'staffHours'    => $staffHours,
            'periods'       => ["" => "Select", "planned" => "Current Roster", "last7days" => "Last 7 Days", "last28days" => "Last 28 Days", "last90days"=>"Last 90 Days", "last365days" => "Last 365 Days"],

            'authUser'      => $authUser,

            // Others
            'userId'        => $user->id,
            'user'          => $user,
            'gender'        => $this->gender,
            'usersList'     => ['' => 'Select'] + User::getUsersList($departmentId, TRUE),

            'position'      => $position,
            'positions'     => $positions,

            'department'    => $departmentId,
            'departments'   => Department::listDepartments(),

            'picture'       => $user->picture ? asset("storage/users/" . $user->picture) : asset("/assets/img/default-user-picture.png"),

            'services'      => Service::pluck("abbr", "id")->all(),
            'staffServices' => $staffServicesByAirlines,
            'staffLicenses' => $staffLicenses,

            'airlines'      => Airline::listHandlingAirlines(),

            // HRM
            'types'                 => ['Calendar' => "Calendar", 'Working' => "Working"],
            'last3years'            => $last3years,
            'entitlementsLast3Year' => $entitlements->keyBy('year'),

            // OTHERS
            'selectedTab'       => $selectedTab,
            'leaves'            => ["" => "Select"] + $leaveRepository->findAndListModelVariable(['is_main' => true], ['name', 'id']),
            'otherLeaves'       => ["" => "Select"] + $leaveRepository->findAndListModelVariable(['is_main' => NULL], ['name', 'id']),
            'entitlements'      => $user->entitlements->keyBy('year'),
            'hasEmployees'      => $hasEmployees,
            'leaveRequests'     => $userLeaveRequest->findAllByAttributes(['user_id' => $user->id], ['leaveStatus']),
            'employeesStatus'   => $employeesStatus,
            'conditions'        => ['' => 'Select'] + Condition::orderBy("name")->pluck('name', 'id')->all(),
            'countries'         => isset($countries) ? $countries : null,

        ];

        return view('staff/show', $this->viewData);
    }

    function getUserRequests($user, $dateFrom, $dateTo){
        return UserLeaveStatus::with([
            "status",
            "leaveRequest",
            "leaveRequest.user",
            "leaveRequest.user.location",
            "leaveRequest.user.location.airport",
            "leaveRequest.leave",
            "leaveRequest.status",
            "leaveRequest.leaveStatus",
            "leaveRequest.leaveStatus.status",
            "leaveRequest.leaveStatus.user",
            "leaveRequest.leaveStatus.user.position",
            "leaveRequest.leaveStatus.user.location",
            "leaveRequest.leaveStatus.user.location.airport",
            "leaveRequest.leaveStatus.organization",
            "leaveRequest.leaveStatus.organization.position",
        ])
            ->join("users__leave_requests", "users__leave_requests.id", "=", "users__leave_status.request_id")
            ->where(function ($sql) use ($dateFrom, $dateTo){
                $sql->whereBetween("date_from", [$dateFrom, $dateTo." 23:59:59"])
                    ->orWhereBetween("date_to", [$dateFrom, $dateTo." 23:59:59"])
                    ->orWhereBetween("users__leave_requests.created_at", [$dateFrom, $dateTo." 23:59:59"])
                    ->orWhereBetween("users__leave_requests.updated_at", [$dateFrom, $dateTo." 23:59:59"]);
            })
            ->where("report_to_user_id", $user->id)
            ->orderBy("users__leave_status.updated_at", "DESC")
            ->get([
                "users__leave_status.*"
            ]);
    }

    function getStaffHoursWithOfficeDuties($staffFlights, $period, $user){
        list($staffHours, $resultsByAirline, $resultsByFlightNumber, $resultsByFlight, $resultsDaily, $periodUnique) = FlightStaff::getHours($staffFlights, $period);
//        $periodSelected = getSelectedFormPeriod($request->get("dateSearch"), $request->get("dateFrom"), $request->get("dateTo"), $request->get("month"), $request->get("year"));
        $staffHistory = UserHistory::getUserHistory($period["from"], $period["to"], true, true, null, null, null, [$user->id]);

        debug($period);
        $overlappingHours = new OverlapHoursCalculation($staffHours, $staffHistory, $period["from"], $period["to"], true);

        return [$overlappingHours, $staffHistory, $periodUnique];
    }

    /**
     * @param Request $request
     * @param $user
     * @return $this|\Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function update(Request $request, $user)
    {
        if (!$user) {
            return view('error/404');
        }

        // Check Password
        if ($request->get('password') && $request->get("password_confirmation")) {

            if ($user->id == Auth::user()->id || in_array(Auth::user()->user_role_id, [ROLE_SUPER_ADMIN, ROLE_ADMIN])){

                if ($request->get('password') == $request->get('password_confirmation')) {
                    $user->remember_token = null;
                    $user->password = Hash::make($request->get('password'));
                    $user->save();

                    flash()->success("Password successfully changed!");
                }
                else {

                    flash()->error("Password/Confirmation do not match. Please try again");
                    /*
                      return redirect()->back()
                          ->withInput()
                          ->withErrors(['Password/Confirmation Do Not Match']);
                      */
                }
            }
        }

        if ($request->get("sv")){
            StaffService::saveDetails($user, \request()->get("service"));
        }

        if ($request->get("update_page")){
            if ($request->get("update_page") == 1)
                return redirect()->to( route("homepage")."#"."staff/{$user->id}/edit");
            else
                return redirect()->to( route("homepage")."#"."staff/{$user->id}");
        }

        return redirect()->to("staff/{$user->id}/edit");
    }

    /**
     * @return $this|\Illuminate\Http\RedirectResponse
     */
    public function create()
    {
        // Redirect If Not Found
//        if (!Auth::user() || Auth::user() == '' )// || !userAnyAdmin())
//        {
//            return redirect()->to('error/page-not-found');
//        }

        $handlingAirports = StaffStation::currentStaffStationsList();

        $approvedLocations = Location::locationAirports(true, $handlingAirports);

        return view("staff/create", [
            "departments"       => getDepartmentListByRole($this->authObject),
//            "positions"         => Auth::user()->role == "admin" ? [] : Position::getPositionsByDepartment($userDepartmentId),
            "locations"         => ["" => "Select"] + $approvedLocations,
            "user"              => isset($user) ? $user : null
        ]);
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        debug($request->all());

        $lastName   = $request->get("last_name");
        $firstName  = $request->get("first_name");
        $email      = $request->get("email");
        $password   = $request->get("password");
        $passwordConfirmation = $request->get("password_confirmation");

        $searchUser = User::where("email", $email)
            ->orWhere(function($sql) use ($firstName, $lastName){
                $sql->where("last_name", $lastName)
                    ->where("first_name", $firstName);
            })
            ->first();

        if ($searchUser){
            return redirect()->to("staff/create")->withInput($request->all());
        }

        if ($password != $passwordConfirmation){
            return redirect()->to("staff/create")->withInput($request->all());
        }

        $user = new User;
        $user->first_name    = $firstName;
        $user->last_name     = $lastName;
        $user->email         = trim($request->get("email"));
        $user->is_contractor = $request->get("is_contractor");
        $user->company       = $request->get("company");
        $user->dob           = $request->get("dob");
        if ($request->get("location")){
            $user->location_id   = $request->get("location");
        }

        $user->idcard_number = $request->get("idcard_number");
        $user->idcard_expiry = $request->get("idcard_expiry");

        $user->password      = Hash::make($password);
        $user->tmp_password  = $password;
        $user->save();

        // Users Departments
        $userDepartment = UserDepartment::where("user_id", "=", $user->id)
            ->first();

        if (!$userDepartment) {
            $userDepartment = new UserDepartment();
            $userDepartment->user_id = $user->id;
        }

        $position = $departmentID = $subdepartmentID = null;
        debug($request->all());
        if ($departmentID = $request->get("department_id")){
            $userDepartment->department_id = $request->get("department_id");
        }
        if ($subdepartmentID = $request->get("subdepartment_id")){
            $userDepartment->subdepartment_id = $request->get("subdepartment_id");
        }
        if ($request->get("position_id")) {
            if ($request->get("position_id") == "other"){
                if ($request->get("other_position")) {
                    $position = Position::getOrCreatePositionByDepartment($request->get("other_position"), $departmentID, $subdepartmentID);
                }
            }
            else {
                $position = Position::find($request->get("position_id"));
            }

            if ($position){
                $userDepartment->position_id = $position->id;
            }
            // Find Position
        }
        $userDepartment->save();

        // Create crew
        createCrew($request->get("crew"), $user, $position);

        return redirect()->to('staff');
    }

    public function show2(Request $request, $user, IUserLeaveRequestRepository $userLeaveRequest,
                          IUserLeaveStatusRepository $userLeaveStatus,
                          ILeaveRepository $leaveRepository,
                          IUserEntitlementRepository $userEntitlementRepository)
    {
        $user = User::with("position", "department")->find($user->id);;

        // Redirect If Not Found
        if (!$user || $user == '') {
            debug("NO USER FOUND");
            return redirect()->to('error/page-not-found');
        }

        // Redirect if Have no Rights
        $accessGranted = isUserAuthorizedToView($user);

        // Redirect if Have no Rights
        if (!$accessGranted){
            return redirect()->to('error/page-not-found');
        }

        $from = $request->get("from") ? $request->get("from") : date("Y-m-d");
        $to = $request->get("to") ? $request->get("to") : date("Y-m-d", strtotime(" + 6 days"));
        $period = getSelectedPeriod($request->get("period"), $from, $to);

        // ROSTER TAB
        $staffFlights = FlightStaff::getAll($period, $user);
        // END ROSTER

        // HOURS TAB
        $staffHours = FlightStaff::getHoursByUser($staffFlights, true);
        // END HOURS

        $departmentId = $user->department && count($user->department) ? $user->department[0]->id : null;

        $positions = [];
        if ($departmentId){
            $positions = Position::getPositionsByDepartment($user->department[0]->id);
        }

        $staffServices = StaffService::where("user_id", $user->id)
            ->orderBy("airline_id")
            ->get();

        $staffServicesByAirlines = [];
        foreach ($staffServices as $each) {
            if (!isset($staffServicesByAirlines[$each->airline_id])){
                $staffServicesByAirlines[$each->airline_id] = [];
            }

            $staffServicesByAirlines[$each->airline_id][] = $each->service_id;
        }

        $staffLicenses = StaffLicense::with(['license'])
            ->where('user_id', $user->id)
            ->get();


        // Leaves And Others
        $hasEmployees = hasEmployeesInLine($user);// $this->user->hasUserReportees($user->id, $this->organization);

        // HRM Staff
        $currentYear = (int)date('Y');
        $last3years = [$currentYear, ++$currentYear];

        $entitlements = $userEntitlementRepository->findAllByAttributes([
            'year'      => [WHERE_IN, $last3years],
            'user_id'   => $user->id
        ]);

        // Get Selected Tab
        $selectedTab = null;
        if (\request()->get("tab") && \request()->get("tab") != ""){
            $selectedTab = \request()->get("tab");
        }
        else if ($hasEmployees){
            $selectedTab = "employees_requests";
        }

        // Modules
        $modules = Module::with(["permission"])->get();

        $userModulePermissions = UserModule::where("user_id", $user->id)
            ->orderBy("module_id")
            ->orderBy("module_permission_id")
            ->pluck("module_permission_id")
            ->all();


        $this->viewData = [

            "from"      => $from,
            "to"        => $to,

            "staffStations" => StaffStation::currentStaffStationsList(IATA, $user->id),
            "staffAirlines" => StaffAirline::currentStaffAirlinesList(IATA, $user->id),

            // Modules
            'modules'       => $modules,
            'userModulePermissions'   => $userModulePermissions,

            //  Roster
            'staffFlights'  => $staffFlights,
            'staffHours'    => $staffHours,
            'periods'       => ["" => "Select", "planned" => "Current Roster", "last7days" => "Last 7 Days", "last28days" => "Last 28 Days", "last90days"=>"Last 90 Days", "last365days" => "Last 365 Days"],

            // Others
            'userId'        => $user->id,
            'user'          => $user,
            'gender'        => $this->gender,
            'usersList'     => ['' => 'Select'] + User::getUsersList($departmentId, TRUE),

            'position'      => $user->position && count($user->position) ? $user->position[0]->id : null,
            'positions'     => $positions,

            'department'    => $departmentId,
            'departments'   => Department::listDepartments(),

            'picture'       => $user->picture ? asset("storage/users/" . $user->picture) : asset("/assets/img/default-user-picture.png"),

            'services'      => Service::pluck("abbr", "id")->all(),
            'staffServices' => $staffServicesByAirlines,
            'staffLicenses' => $staffLicenses,

            'airlines'      => Airline::listHandlingAirlines(),

            // HRM
            'types'                 => ['Calendar' => "Calendar", 'Working' => "Working"],
            'last3years'            => $last3years,
            'entitlementsLast3Year' => $entitlements->keyBy('year'),

            // OTHERS
            'selectedTab'       => $selectedTab,
            'leaves'            => ["" => "Select"] + $leaveRepository->findAndListModelVariable(['is_main' => true], ['name', 'id']),
            'otherLeaves'       => ["" => "Select"] + $leaveRepository->findAndListModelVariable(['is_main' => NULL], ['name', 'id']),
            'entitlements'      => $user->entitlements->keyBy('year'),
            'hasEmployees'      => $hasEmployees,
            'leaveRequests'     => $userLeaveRequest->findAllByAttributes(['user_id' => $user->id], ['leaveStatus']),
            'employeesStatus'   => $userLeaveStatus->findAllByAttributes(['report_to_user_id' => $user->id], ['leaveRequest']),
            'conditions'        => ['' => 'Select'] + Condition::orderBy("name")->pluck('name', 'id')->all(),
            'countries'         => isset($countries) ? $countries : null,
        ];

        return view('staff/show', $this->viewData);
    }

    /**
     * @param $user
     * @return \Illuminate\Http\RedirectResponse
     */
    public function delete($user)
    {
        if ($user){
            $user->updated_by = Auth::user()->id;
            $user->deleted_at = date("Y-m-d H:i:s");
            $user->save();
        }

        return redirect()->to('staff');
    }

    public function restore($user)
    {
        if ($user){
            $user->updated_by = Auth::user()->id;
            $user->deleted_at = null;
            $user->save();
        }

        return redirect()->to('staff');
    }
}
