<?php namespace App\Http\Controllers;

use App\Http\Requests;
use App\Http\Controllers\Controller;

use App\Repositories\Interfaces\ICurrencyRepository;
use App\Repositories\Interfaces\IReportTypeRepository;
use App\Repositories\Interfaces\IRevenueTargetPeriodRepository;
use App\Repositories\Interfaces\IRevenueTargetRepository;
use Illuminate\Http\Request;

use League\Period\Period;

class RevenueTargetPeriodController extends Controller {

    /**
     * Display a listing of the resource.
     * @param IRevenueTargetPeriodRepository $period
     * @return \Illuminate\View\View
     */
	public function index(IRevenueTargetPeriodRepository $period)
	{
        $this->viewData = [
            'revenue_targets_periods'   => $period->get(['currency', 'organization'], [ ['period_from', 'DESC']])
        ];
        return view('revenue-target-period.index', $this->viewData);
	}

    /**
     * Show the form for creating a new resource.
     * @param ICurrencyRepository $currency
     * @return \Illuminate\View\View
     */
    public function create(ICurrencyRepository $currency)
    {
        $level = $this->organizationLevel->findByAttributes(['level' => 1]);

        $organizationList = [];

        if ($level) {
            $organization = $this->organization->findAllByAttributes([
                'organization_level_id' => $level->id,
                'support_staff'         => 0
            ]);

            foreach ($organization as $each) {
                $user = $each->user;
                $organizationList[$each->id] = $user ? $user->first_name." ".$user->last_name. '/' . $each->position->name . '/' : "";
            }
        }


        // Data To Pass to View
        $this->viewData = [
            'organizations'             => ['' => 'Select'] + $organizationList,
            'currencies'                => $currency->findAndListModelVariable(['code' => [WHERE_IN, ['USD' ,'EUR', 'TJS']]],['code', 'id']),
        ];

        return view('revenue-target-period.create', $this->viewData);
    }

    /**
     * Store a newly created resource in storage.
     * @param IRevenueTargetPeriodRepository $period
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
	public function store(IRevenueTargetPeriodRepository $period)
	{
        $inputs = request()->all();

        // Check For Conflicts
        $check = $this->checkForConflictInPeriods($inputs['period_from'], $inputs['period_to'], $period);

        if ($check) {
            return redirect()->back()
                ->withInput()
                ->with('message', [
                    'type' => 'danger',
                    'text' => 'Period Conflicts. Please Check Period Again.'
                ]);
        }

        removeUnnecessaryField($inputs, ['_token', 'insert', 'organization_level_id']);

        $inputs['target'] = str_replace(',', '',$inputs['target']);

        $period->createByAttributes($inputs);

        return redirect('revenue-target-period');
	}

    /**
     * Check For Conflicts
     * @param $periodFrom
     * @param $periodTo
     * @param $periodRepository
     * @return bool
     */
    protected function checkForConflictInPeriods($periodFrom, $periodTo, $periodRepository){
        $check = $periodRepository->findByAttributes(
            ['period_from' => [WHERE_BETWEEN, [$periodFrom, $periodTo]]]
        );
        if ($check)
            return true;

        $check = $periodRepository->findByAttributes(
            ['period_to' => [WHERE_BETWEEN, [$periodFrom, $periodTo]]]
        );
        if ($check)
            return true;

        return false;
    }

	/**
	 * Display the specified resource.
	 *
	 * @param  int  $id
	 * @return Response
	 */
	public function show($id)
	{
		//
	}


    /**
     * @param $revenue_target_period
     * @param IReportTypeRepository $reportType
     * @param ICurrencyRepository $currency
     * @return \Illuminate\View\View
     */
	public function edit($revenue_target_period, IReportTypeRepository $reportType, ICurrencyRepository $currency)
	{
        $reportTypes = $reportType->findAndListModelVariable(['type' => [WHERE_IN, ['Yearly', 'Quarterly', 'Monthly']]],['type', 'id'], false, false, ['id']);

        debug($reportTypes);

        $this->viewData = [
            'period'                => $revenue_target_period,
            'reportTypes'           => $reportTypes,
            'currencies'            => $currency->listModelVariable(['code', 'id'], true)
        ];
        return view('revenue-target-period.edit', $this->viewData);
	}


	public function update($revenue_target_period, IReportTypeRepository $reportType)
	{
        $target = \request("target") ? removeCommas(\request("target")) : null;
        $periodFrom = \request("period_from");
        $periodTo = \request("period_to");

        // Update All Revenue Targets Only When Records Differ
        if ($revenue_target_period->period_from != $periodFrom) {
            // Extract Period From Given Period
            $extractedPeriods = $this->calculateAllTypePeriodsFromGivenPeriod($periodFrom, $periodTo, $reportType);

            // Get Period's Targets
            $revenueTargets = $revenue_target_period->targets;

            // Match And Update Revenue Targets
            $this->matchPeriodAndUpdateRevenueTargets($revenueTargets, $extractedPeriods);
        }

        $revenue_target_period->target = $target;
        $revenue_target_period->period_from = $periodFrom;
        $revenue_target_period->period_to = $periodTo;
        $revenue_target_period->currency_id = \request("currency_id");
        $revenue_target_period->save();

        flash()->success('Successfully Updated Period');

        return redirect('revenue-target-period');
	}

    /**
     * Match Periods ANd Update Revenue Targets Period From, To Records
     * @param $revenueTargets
     * @param $extractedPeriods
     */
    protected function matchPeriodAndUpdateRevenueTargets(&$revenueTargets, $extractedPeriods){
        foreach ($revenueTargets as &$each) {
            foreach ($extractedPeriods[$each->type_id] as $item) {
                if (isset($item['quarter'])){
                    if ($item['quarter'] != $each->quarter)
                        continue;
                    $each->quarter = $item['quarter'];
                }

                if (isset($item['month'])){
                    if ($item['month'] != $each->month)
                        continue;
                    $each->month = $item['month'];
                }

                $each->period_from = $item['period_from'];
                $each->period_to = $item['period_to'];
                $each->save();
                break;
            }
        }
    }

    /**
     * Get Quarters And Months Targets Out of Submitted Form inputs
     * @param $reportType
     * @param $from
     * @param $to
     * @return array
     */
    protected function calculateAllTypePeriodsFromGivenPeriod($from, $to, $reportType){
        $period = new Period($from, $to);
        $targets = [];
        $yearlyType = $reportType->findByAttributes(['type' => 'Yearly']);
        $quarterlyType = $reportType->findByAttributes(['type' => 'Quarterly']);
        $monthlyType = $reportType->findByAttributes(['type' => 'Monthly']);
        if ($quarterlyType && $monthlyType) {
            $targets[$yearlyType->id][]  = [
                'period_from'      => $from,
                'period_to'        => $to
            ];
            foreach ($period->getDatePeriod('1 MONTH') as $i => $each) {
                $j = $i + 1;
                $date = date('Y-m-d', strtotime($each->format('Y-M-d')));

                if ($i % 3 == 0) {
                    $qt = ($i / 3) + 1;
                    $targets[$quarterlyType->id][]  = [
                        'quarter'       => $qt,
                        'period_from'   => $each->format('Y-m-d'),
                        'period_to'     => date('Y-m-d', strtotime(" + 3 months - 1 days", strtotime($date))),
                    ];
                }

                $targets[$monthlyType->id][] = [
                    'month'            => $j,
                    'period_from'      => $date,
                    'period_to'        => date('Y-m-d', strtotime("last day of this month", strtotime($date))),
                ];
            }
        }
        return $targets;
    }


    /**
     * Remove the specified resource from storage.
     * @param $revenue_target_period
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
	public function destroy($revenue_target_period)
	{
        $revenue_target_period->delete();
        return redirect('revenue-target-period');
	}

}
