<?php namespace App\Http\Controllers;

use App\Http\Requests;
use App\Http\Controllers\Controller;

use App\Models\Currency;
use App\Models\PassengerFlight;
use App\Models\ReportType;
use App\Models\RevenueTarget;
use App\Models\RevenueTargetPeriod;
use App\Repositories\Interfaces\ICurrencyRepository;
use App\Repositories\Interfaces\IOrganizationLevelRepository;
use App\Repositories\Interfaces\IReportTypeRepository;
use App\Repositories\Interfaces\IRevenueTargetPeriodRepository;
use App\Repositories\Interfaces\IRevenueTargetRepository;
use Illuminate\Http\Request;

use League\Period\Period;

class RevenueTargetController extends Controller {

    /**
     * Display a listing of the resource.
     * @param IRevenueTargetRepository $revenueTarget
     * @param IReportTypeRepository $reportTypeRepository
     * @return \Illuminate\View\View
     */
	public function index(IRevenueTargetRepository $revenueTarget, IReportTypeRepository $reportTypeRepository, IRevenueTargetPeriodRepository $period)
	{
        $reportType = $reportTypeRepository->findAllByAttributes(['type' => [WHERE_IN, ['Yearly', 'Quarterly', 'Monthly']] ]);

        $this->viewData = [
            'periods'           => $period->get(['currency', 'organization']),
            'revenue_targets'   => $revenueTarget->findAllByAttributes(['type_id' => [ WHERE_IN, $reportType->modelKeys()]], ['organization', 'type', 'period'], ['*'], [], ['organization_id', 'type_id', 'period_from'])
        ];

        debug($this->viewData);

        return view('revenue-target.index', $this->viewData);
	}

    /**
     * Show the form for creating a new resource.
     * @param ICurrencyRepository $currency
     * @param IReportTypeRepository $reportType
     * @param IRevenueTargetPeriodRepository $period
     * @return \Illuminate\View\View
     */
	public function create(ICurrencyRepository $currency, IReportTypeRepository $reportType, IRevenueTargetPeriodRepository $period)
	{
        $reportTypes = $reportType->findAndListModelVariable(['type' => [WHERE_IN, ['Yearly', 'Quarterly', 'Monthly']]],['type', 'id'], false, false, ['id']);

        $reportTypeSelected = \request("type_selected") ? \request("type_selected") : array_keys($reportTypes, 'Yearly');

        $levels = $this->organizationLevel->listModelVariable(['level', 'id'], true);

        // Data To Pass to View
        $this->viewData = [
            'periods'               => $period->listModelVariable([['period_from', 'period_to'], 'id', '/'], true),
            'organization_levels'   => isset($levels) && $levels && count($levels) ? array_unique($levels) : [],
            'currencies'            => $currency->findAndListModelVariable(['code' => [WHERE_IN, ['USD' ,'EUR', 'TJS']]],['code', 'id']),
            'report_types'          => $reportTypes,
            'report_types_selected' => $reportTypeSelected,
        ];

		return view('revenue-target.create', $this->viewData);
	}

    /**
     * Store a newly created resource in storage.
     * @param IRevenueTargetRepository $revenueTarget
     * @param IReportTypeRepository $reportType
     * @return $this|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|int
     */
	public function store(IRevenueTargetRepository $revenueTarget, IReportTypeRepository $reportType)
	{
        $inputs = request()->all();

        $period = RevenueTargetPeriod::find($inputs['period_id']);

        // If Period Not Found Redirect Back
        if (!$period) {
            return redirect()->back()
                ->with('message', 'Period Not Found')
                ->withInput();
        }

        // If Duplication Then Redirect
        $checkIfDuplicate = $revenueTarget->findByAttributes([
            'period_id'         => $period->id,
            'organization_id'   => $inputs['organization_id']
        ]);
        if ($checkIfDuplicate) {
            return redirect()->back()
                ->with('message', 'Revenue Targets Has Been Already Created For The Given Employee. Please Check Again.')
                ->with('type', 'danger')
                ->withInput();
        }

        // Get Periods From Period Model
        $inputs['period_from'] = $period->period_from;
        $inputs['period_to'] = $period->period_to;

        // Get Quarters And Months Targets Out of Submitted Form inputs
        $targets = $this->getQuarterMonthTargetValues($inputs, $inputs['period_from'], $inputs['period_to'], $reportType);

        // Remove Thousand Separator Commas From Target
        $inputs['target'] = removeCommas($inputs['target']);

        // Remove Unnecessary Fields
        removeUnnecessaryField($inputs, ['_token', 'insert', 'organization_level_id']);

        // Create Yearly Target
        $currentTarget = $revenueTarget->createByAttributes($inputs);

        // Get Upper Level Target Id Attached
        $reportToOrganizationId = $this->bindToUpperLevelOrganizationTarget($inputs, $currentTarget, $inputs['type_id'], $revenueTarget);

        // Create Quarterly And Monthly Targets
        $this->createQuarterlyAndMonthlyTargets($targets, $reportToOrganizationId, $inputs, $revenueTarget, $currentTarget);

        flash()->success('Revenue Targets Successfully Inserted');

		return redirect('revenue-target');
	}

    /**
     * Create Quarterly And Monthly Targets
     * @param $targets
     * @param $reportToOrganizationId
     * @param $inputs
     * @param $revenueTarget
     */
    protected function createQuarterlyAndMonthlyTargets($targets, $reportToOrganizationId, $inputs, $revenueTarget, $yearlyTarget){
        if ($targets && count($targets)) {
            foreach ($targets as $each) {

                $quarter = $each['quarter'];

                // Find Report To Target Id And Save It If Exists
                $reportToTarget = $this->getReportToTarget($revenueTarget, $inputs, $reportToOrganizationId, $quarter);

                // Create Quarterly Targets
                $quarterlyTarget = $revenueTarget->createByAttributes([
                    'parent_id'         => $yearlyTarget->id,
                    'period_id'         => $inputs['period_id'],
                    'organization_id'   => $inputs['organization_id'],
                    'type_id'           => $quarter['type_id'],
                    'period_from'       => $quarter['from'],
                    'period_to'         => $quarter['to'],
                    'target'            => $quarter['target'],
                    'quarter'           => $quarter['quarter'],
                    'target_id'         => $reportToTarget ? $reportToTarget->id : null
                ]);

                // Create Monthly Targets
                foreach ($each['months'] as $month) {
                    // Find Report To Target Id And Save It If Exists
                    $reportToTarget = $this->getReportToTarget($revenueTarget, $inputs, $reportToOrganizationId, $month);

                    $revenueTarget->createByAttributes([
                        'parent_id'         => $quarterlyTarget->id,
                        'period_id'         => $inputs['period_id'],
                        'organization_id'   => $inputs['organization_id'],
                        'type_id'           => $month['type_id'],
                        'period_from'       => $month['from'],
                        'period_to'         => $month['to'],
                        'target'            => $month['target'],
                        'month'             => $month['month'],
                        'target_id'         => $reportToTarget ? $reportToTarget->id : null
                    ]);
                }
            }
        }
    }

    protected function getReportToTarget($revenueTarget, $inputs, $reportToOrganizationId, $each){
        return $revenueTarget->findByAttributes([
            'period_id'         => $inputs['period_id'],
            'organization_id'   => $reportToOrganizationId,
            'type_id'           => $each['type_id'],
            'period_from'       => $each['from'],
            'period_to'         => $each['to']
        ]);
    }


    /**
     * Get Upper Level Target Id Attached And Return Report To Organization Id if Exists
     * @param $inputs
     * @param $currentTarget
     * @param $reportTypeId
     * @param $revenueTarget
     * @return bool
     */
    protected function bindToUpperLevelOrganizationTarget($inputs, &$currentTarget, $reportTypeId, $revenueTarget){
        // Get Upper Level Target Id Attached
        $organization = $this->organization->find($inputs['organization_id']);

        if ($organization){
            $reportToUserId = $organization->report_to_user_id;
            $reportToOrganization = $this->organization->findByAttributes(['user_id' => $reportToUserId]);
            if ($reportToOrganization){
                $reportToTarget = $revenueTarget->findByAttributes([
                    'period_id'       => $inputs['period_id'],
                    'organization_id' => $reportToOrganization->id,
                    'type_id'         => $reportTypeId
                ]);
                if ($reportToTarget){
                    $revenueTarget->update($currentTarget, ['target_id' => $reportToTarget->id]);
                }
                return $reportToOrganization->id;
            }
        }
        return false;
    }


    /**
     * Get Quarters And Months Targets Out of Submitted Form inputs
     * @param $inputs
     * @param $reportType
     * @param $from
     * @param $to
     * @return array
     */
    protected function getQuarterMonthTargetValues(&$inputs, $from, $to, $reportType){
        $period = new Period($from, $to);
        $targets = [];
        $quarterlyType = $reportType->findByAttributes(['type' => 'Quarterly']);
        $monthlyType = $reportType->findByAttributes(['type' => 'Monthly']);
        if ($quarterlyType && $monthlyType) {
            foreach ($period->getDatePeriod('1 MONTH') as $i => $each) {
                $j = $i + 1;
                $date = date('Y-m-d', strtotime($each->format('Y-M-d')));

                if ($i % 3 == 0) {
                    $qt = ($i / 3) + 1;
                    $target_quarter = "target_q" . $qt;
                    $targets[$qt]['quarter']  = [
                        'quarter'   => $qt,
                        'type_id'   => $quarterlyType->id,
                        'from'      => $each->format('Y-m-d'),
                        'to'        => date('Y-m-d', strtotime(" + 3 months - 1 days", strtotime($date))),
                        'target'    => removeCommas($inputs[$target_quarter])
                    ];
                    removeUnnecessaryField($inputs, $target_quarter);
                }

                $target_month = "target_m" . $j;
                $targets[$qt]['months'][] = [
                    'month'     => $j,
                    'type_id'   => $monthlyType->id,
                    'from'      => $date,
                    'to'        => date('Y-m-d', strtotime("last day of this month", strtotime($date))),
                    'target'    => removeCommas($inputs[$target_month])
                ];
                removeUnnecessaryField($inputs, $target_month);
            }
        }
        return $targets;
    }


    /**
     * Remove Commas
     * @param $str
     * @return mixed
     */


	/**
	 * Display the specified resource.
	 *
	 * @param  int  $id
	 * @return Response
	 */
	public function show($id)
	{
		//
	}


	public function edit($revenue_target, IReportTypeRepository $reportType, IRevenueTargetPeriodRepository $period, ICurrencyRepository $currency, IOrganizationLevelRepository $organizationLevelRepository)
	{
        $reportTypes = $reportType->findAndListModelVariable(['type' => [WHERE_IN, ['Yearly', 'Quarterly', 'Monthly']]],['type', 'id'], false, false, ['id']);

        $reportTypeSelected = \request("type_selected") ? \request("type_selected") : array_keys($reportTypes, 'Yearly');

        $levelId = $revenue_target->organization->organizationLevel->id;

        $currentLevelEmployees = $organizationLevelRepository->getEmployeesByLevelId($levelId, TRUE, TRUE);

        $quarterTargets = $revenue_target->children;

        $this->viewData = [
            'periods'               => $period->listModelVariable([['period_from', 'period_to'], 'id', '/'], true),
            'revenueTarget'         => $revenue_target,
            'quarterTargets'        => $quarterTargets->keyBy('quarter'), /// Makes Array Keys Object Ids
            'currentLevelEmployees' => $currentLevelEmployees,
            'report_types_selected' => $reportTypeSelected,
            'reportTypes'           => $reportTypes,
            'organization_levels'   => $this->organizationLevel->listModelVariable(['level', 'id'], true),
            'currencies'            => $currency->findAndListModelVariable(['code' => [WHERE_IN, ['USD' ,'EUR', 'TJS']]],['code', 'id']),

        ];
        return view('revenue-target/edit', $this->viewData);
	}

    /**
     * @param $revenue_target
     * @param IRevenueTargetRepository $revenueTargetRepository
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
	public function update($revenue_target, IRevenueTargetRepository $revenueTargetRepository)
	{
        $inputs = request()->all();

        $this->updateQuarterMonth($inputs, $revenueTargetRepository);

        $target = $inputs['target'] ? removeCommas($inputs['target']) : null;

        $revenueTargetRepository->update($revenue_target, ['target' => $target]);

        flash()->success('Revenue Target Successfully Edited');

        return redirect('revenue-target');
	}

    /**
     * Update Quarter And Month Targets
     * @param $inputs
     * @param $revenueTargetRepository
     */
    protected function updateQuarterMonth($inputs, $revenueTargetRepository){
        foreach ($inputs as $key => $value) {
            if (contains($key, 'target_q')){
                if (preg_match("/target_q([0-9]+)/", $key, $matches))
                {
                    if (isset($matches[1])){
                        $value = removeCommas($value);
                        $elementId = $matches[1];
                        $revenueTargetRepository->updateById($elementId, ['target' => $value]);
                    }
                }
            }
            elseif (contains($key, 'target_m')){
                if (preg_match("/target_m([0-9]+)/", $key, $matches))
                {
                    if (isset($matches[1])){
                        $value = removeCommas($value);
                        $elementId = $matches[1];
                        $revenueTargetRepository->updateById($elementId, ['target' => $value]);
                    }
                }
            }
        }
    }

    /**
     * Remove the specified resource from storage.
     * @param $revenue_target
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
	public function destroy($revenue_target)
	{
        $childrenElements = self::getBindedTargets($revenue_target);

        $userId = $this->authObject->user()->id;

        // Delete Current Revenue Target
        softDeleteObject($revenue_target, $userId);

        if ($childrenElements) {
            // Delete Children Elements
            softDeleteObject($childrenElements, $userId);
        }

        return redirect('revenue-target');
	}

    /**
     * Restore Cannot Be Accessed By Route Model Binding Because of Soft Deletion
     * @param $id
     * @param IRevenueTargetRepository $revenueTargetRepository
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function restore($id, IRevenueTargetRepository $revenueTargetRepository){
        $revenue_target = $revenueTargetRepository->find($id, ONLY_TRASHED);

        if ($revenue_target) {
            $childrenElements = self::getBindedTargets($revenue_target, WITH_TRASHED);

            $userId = $this->authObject->user()->id;

            // Restore Current Revenue Target
            restoreObject($revenue_target, $userId);

            if ($childrenElements) {
                // Restore Children Elements
                restoreObject($childrenElements, $userId);
            }
        }

        return redirect('revenue-target/removed-list');
    }

    public function removedList(IRevenueTargetRepository $revenueTarget, IReportTypeRepository $reportTypeRepository, IRevenueTargetPeriodRepository $period){
        $revenueTargets = $revenueTarget->get(['organization', 'type', 'period'],[],[],ONLY_TRASHED);

        $this->viewData = [
            'periods'           => $period->get(['currency', 'organization']),
            'revenue_targets'   => $revenueTargets
        ];

        return view('revenue-target/removed-list', $this->viewData);
    }


    /**
     * Get All Binded Elements(Child)
     * @param bool|FALSE $revenueTarget
     * @return array|null
     */
    public static function getBindedTargets($revenueTarget = FALSE, $trashed = false){
        // Search
        $bindedElements = self::recursiveSearch($revenueTarget, $trashed);
        // Remove Null Values
        if ($bindedElements) {
            removeNullValues($bindedElements);
        }
        // Return Ready List
        return $bindedElements;
    }


    /**
     * Search All Binded Elements Recursively
     * @param bool|FALSE $revenueTarget
     * @return array|null
     */
    public static function recursiveSearch($revenueTarget = FALSE, $trashed = FALSE){
        if (!$revenueTarget) {
            return null;
        }

        if ($trashed){
            $bindedTargets = $revenueTarget->children()->withTrashed()->get();
        }
        else {
            $bindedTargets = $revenueTarget->children;
        }

        if (isset($bindedTargets) && $bindedTargets) {
            foreach ($bindedTargets as $item) {
                $total[] = clone $item;
                if ($trashed){
                    $otherBindedTargets = $item->children()->withTrashed()->get();
                }
                else {
                    $otherBindedTargets = $item->children;
                }
                if ($otherBindedTargets) {
                    foreach ($otherBindedTargets as $each) {
                        $total[] = $each;
                        $total[] = self::getBindedTargets($each);
                    }
                }
            }
            return isset($total) ? $total : null;
        }
        return null;
    }

}
