<?php namespace App\Http\Controllers;

use App\Commands\CurrencyRatesCommand;
use App\Http\Requests;
use App\Models\Airport;
use App\Models\FlightNumber;
use App\Models\FlightSector;
use App\Models\PassengerFlight;
use App\Models\ReportType;
use App\Repositories\Interfaces\IAgencyCodeRepository;
use App\Repositories\Interfaces\IAgencyRepository;
use App\Repositories\Interfaces\ICityRepository;
use App\Repositories\Interfaces\ICountryRepository;
use App\Repositories\Interfaces\IDivisionRepository;
use App\Repositories\Interfaces\IRegionRepository;
use App\Repositories\Interfaces\ITableRepository;

use App\Repositories\Interfaces\IFlightCabinRepository;
use App\Repositories\Interfaces\IFlightFareBasisRepository;
use App\Repositories\Interfaces\IFlightOrderStatusRepository;
use App\Repositories\Interfaces\IFlightRbdRepository;
use App\Repositories\Interfaces\IFlightTypeRepository;
use App\Repositories\Interfaces\IReportTypeRepository;

class RevenueDataController extends Controller {

    protected $flightTypeRepository;
    protected $flightFareBasisRepository;
    protected $flightOrderStatusRepository;
    protected $flightCabinRepository;
    protected $flightRbdRepository;
    protected $reportTypeRepository;


    /**
     * @param IFlightTypeRepository $flightTypeRepository
     * @param IFlightFareBasisRepository $flightFareBasisRepository
     * @param IFlightOrderStatusRepository $flightOrderStatusRepository
     * @param IFlightCabinRepository $flightCabinRepository
     * @param IFlightRbdRepository $flightRbdRepository
     * @param IReportTypeRepository $reportTypeRepository
     * @param IDivisionRepository $division
     * @param IRegionRepository $region
     * @param ICountryRepository $country
     * @param ICityRepository $city
     * @param IAgencyRepository $agency
     * @param IAgencyCodeRepository $agencyCode
     * @param ITableRepository $table
     */
    public function __construct(IFlightTypeRepository $flightTypeRepository, IFlightFareBasisRepository $flightFareBasisRepository,
                                IFlightOrderStatusRepository $flightOrderStatusRepository, IFlightCabinRepository $flightCabinRepository,
                                IFlightRbdRepository $flightRbdRepository, IReportTypeRepository $reportTypeRepository,
                                IDivisionRepository $division,
                                IRegionRepository $region,
                                ICountryRepository $country,
                                ICityRepository $city,
                                IAgencyRepository $agency,
                                IAgencyCodeRepository $agencyCode,
                                ITableRepository $table
                                )
    {
        $this->flightTypeRepository = $flightTypeRepository;
        $this->flightFareBasisRepository = $flightFareBasisRepository;
        $this->flightOrderStatusRepository = $flightOrderStatusRepository;
        $this->flightCabinRepository = $flightCabinRepository;
        $this->flightRbdRepository = $flightRbdRepository;
        $this->reportTypeRepository = $reportTypeRepository;

        $this->division                     = $division;
        $this->region                       = $region;
        $this->country                      = $country;
        $this->city                         = $city;
        $this->agency                       = $agency;
        $this->agencyCode                   = $agencyCode;
        $this->table                        = $table;
    }

    public function index()
    {
        $this->viewData = $this->GetFormData();

        return view('revenue-data.index', $this->viewData);
    }


    public function GetFormData(){

        return [
            "flight_numbers"    => FlightNumber::getArrayFlightNumbersWithSector(),
            'sectors'           => FlightSector::listSectors(),

            "flight_types"      => $this->flightTypeRepository->listModelVariable(['type', 'id'], true),
            "countries"         => Country::getCountries(),
            "cities"            => $this->city->findAndListModelVariable(['country_id' => [WHERE_IN, [1077,1062,1017,1079,1037,1026,1046,1024]]],['city', 'id']),
            "agencies"          => $this->agency->listModelVariable(['agency', 'id']),
            "tch_codes"         => $this->agencyCode->findAndListModelVariable([ [WHERE_RAW, 'char_length(code) = 5'] ], ['code', 'id']),
            "coupons"           => [
                '1' => '1',
                '2' => '2',
                '3' => '3',
                '4' => '4'
            ],
            "records_per_page" => [
                '50'    => '50',
                '250'   => '250',
                '500'   => '500',
                '1000'  => '1000'
            ],
            "fare_basis"     => $this->flightFareBasisRepository->listModelVariable(['fare_basis', 'id']),
            "payment_forms"   => [
                "CA" => "CA",
                "CC" => "CC",
                "MS" => "MS"
            ],
            "statuses"        => $this->flightOrderStatusRepository->listModelVariable(['order_status', 'id']),
            "cabins"          => $this->flightCabinRepository->listModelVariable(['cabin', 'id']),
            "rbds"            => $this->flightRbdRepository->listModelVariable(['rbd', 'id']),
            "report_types"    => $this->reportTypeRepository->listModelVariable(['type', 'id'], true, true)
        ];
    }

    public function test(){
        $data = [];
        for ($i = 1; $i < 100000; $i++){
            $data[] = [
                "id"    => $i,
				"date"  => "2000-".$i % 12 ."-".$i % 30,
				"name"  => "test3",
				"note"  => "note3",
				"amount"=> "400.00",
				"tax"   => "30.00",
				"total" => "430.00"
            ];
        }

        return view("smart/ajax/jqgrid", ["data" => $data]);
    }


    /**
     * Search Index Given Query
     * @return \Illuminate\View\View
     */
    public function search(){
        // Get All Inputs
        $inputs = request()->all();

        // For Pagination Do Not Update Info Alert Box (Total Result)
        $cacheQuery = isset($inputs['cacheQuery']) ? TRUE : FALSE;

        // Search Passenger Flights And Total
        list($tableData, $pagination, $totalMode, $total) = PassengerFlight::searchPassengerFlights($inputs, $cacheQuery, TRUE, TRUE);

        debug($total);
        debug($tableData);

        // If Report Type is Set -> Show Results View, else Regular
        if (isset($inputs['country_group']) || isset($inputs['city_group']) || isset($inputs['agency_group'])) {
            $view = 'revenue-data/index-result-pos';
            $unique_periods = makeUniqueByColumn($tableData, "period");
            $unique_pos = makeUniqueByColumn($tableData, "pos");
            $tableData = collectionToArray($tableData);
        }
        else {
            $view = $totalMode ? 'revenue-data/index-result-totals' : 'revenue-data/index-result';
        }

        $strInputs = "?";
        debug($inputs);
        foreach ($inputs as $param => $val ) {
            if ($param != "_token"){
                if (is_array($val)){
                    foreach ($val as $item) {
                        $strInputs .= $param."[]=".$item."&";
                    }
                }
                else {
                    $strInputs .= $param."=".$val."&";
                }
            }
        }

        $this->viewData = [
            'strInputs'         => $strInputs,
            // Query Totals Cache
            'cacheQuery'        => $cacheQuery,
            // Passenger Flights
            'tableData'         => $tableData,
            // Total Mode
            'totalMode'         => $totalMode,
            // Total
            'total'             => $total,
            // Pagination (Enable)
            'pagination'        => $pagination,
            // Records Per Page
            'recordsPerPage'    => $inputs['recordsPerPage'],
            // Current Page
            'page'              => isset($inputs['page']) ? $inputs['page'] : 1,
            // Report Type
            'report_type'       => isset($inputs['report_type']) ? ReportType::find($inputs['report_type']) : null,
            // Unique Period
            'unique_periods'    => isset($unique_periods) ? $unique_periods : null,
            // Unique POS
            'unique_pos'        => isset($unique_pos) ? $unique_pos : null,
        ];

        foreach ($this->GetFormData() as $index => $data ) {
            $this->viewData[$index] = $data;
        }

        return view($view, $this->viewData);
    }



    /**
     * Get Ticket Number Details
     * @param null $ticket_number
     * @return \Illuminate\View\View
     */
    public function ticketNumber($ticket_number = NULL)
    {
        if (is_numeric($ticket_number)) {
            list($passengerFlights) = PassengerFlight::searchPassengerFlights(
                ['ticket_number' => [$ticket_number, 'operator' => '=']
                ], FALSE);
            debug($passengerFlights);

            return view('revenue-data.ticket-number', [
                'ticket_number'        => $ticket_number,
                'passengerFlights'     => $passengerFlights,
            ]);
        }
    }


    public function getFixFlightNumberAirport(){
        $flightNumbers = FlightNumber::all();
        foreach ($flightNumbers as $flightNumber) {
            $depAirportId = Airport::where('iata', $flightNumber->departure_airport)->first();
            $arrAirportId = Airport::where('iata', $flightNumber->arrival_airport)->first();
            $flightNumber->departure_airport_id = $depAirportId ? $depAirportId->id : null;
            $flightNumber->arrival_airport_id = $arrAirportId ? $arrAirportId->id : null;
            $flightNumber->save();
        }

    }


    public function testC(){
        $this->dispatch(
            new CurrencyRatesCommand()
        );
        return 123;
    }
}
