<?php namespace App\Http\Controllers;

use App\Models\Agency;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\City;
use App\Models\Country;
use App\Models\FlightDelay;
use App\Models\FlightSector;
use App\Models\PassengerFlight;
use App\Models\StaffAirline;
use App\Models\StaffStation;
use App\Models\User;
use Illuminate\Http\Request;

use App\Models\Flight;
use Illuminate\Support\Facades\DB;

use stdClass;
use App\Models\FlightNumber;

class ReportController extends Controller
{

    public function getIndex()
    {
        return view('report.index');
    }

    public function ops(){
        // Get Flights With Flight Crew

        $flights  = Flight::FlightsRangeConditional();

        // Data Array
        $this->viewData = [
            'opsReport'          => ExportController::getOpsReportDataArray($flights, \request()->get('period')),
            'dateYesterday'      => date('Y-m-d',strtotime(" - 1 days")),

            'flightNumbers'      => FlightNumber::listFlightNumbersByFlightTypeAndAirport('Scheduled'),
            'sectors'            => FlightSector::listSectors(),
            'sectorsByAirlines'     => FlightSector::getSectorsByAirlines(),

            'airports'           => FlightNumber::listAirportsByFlightType('Scheduled', 'departure_airport'),
            'periods'            => getPeriodsArray([6]),
            'monthNames'         => getMonthsArray(),
            'years'              => getYearsArray(),
        ];


        return view('report.ops', $this->viewData);
    }

    /**
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function handlingOps(Request $request){
        // Get Flights With Flight Crew
        // Get Staff Handling AP and Airlines
        list($handlingAirlines, $handlingAirports) = getStaffHandlingAirports();

        debug($handlingAirlines);

        $selectedAirports = \request()->has("airport") ? \request()->get("airport") : array_keys($handlingAirports);
        $selectedAirlines = \request()->has("airline") ? \request()->get("airline") : array_keys($handlingAirlines);

        $flights  = Flight::FlightsRangeConditional($selectedAirports, $selectedAirlines, $request->get("type"), null, true);

        $opsReport = ExportController::getOpsReportDataArray($flights, \request()->get('period'));

        // Data Array

        $this->viewData = [
            'opsReport'         => $opsReport,

//            "handlingStations"  => $handlingStations,

            "airlines"          => $handlingAirlines,
            "airports"          => $handlingAirports,

            "flightNumbers"     => FlightNumber::getArrayFlightNumbersWithSector(false, false, false, null, true),
            'sectors'           => FlightSector::listSectors(),
            'sectorsByAirlines' => FlightSector::getSectorsByAirlines(),

            'periods'           => getPeriodsArray([6]),
            'dateYesterday'     => date('Y-m-d',strtotime(" - 1 days")),
            'monthNames'        => getMonthsArray(),
            'years'             => getYearsArray(),
        ];

        return view('report.h-ops', $this->viewData);
    }


    public function handlingPunctuality(Request $request, $widget = null){

        // Get Staff Handling AP and Airlines
        list($handlingAirlines, $handlingAirports) = getStaffHandlingAirports();

        $selectedAirports = $request->get("airport") ? $request->get("airport") : array_keys($handlingAirports);
        $selectedAirlines = $request->get("airline") ? $request->get("airline") : array_keys($handlingAirlines);

        // Get Punctuality Ready Array
        list($airportsUnique, $airportsByCategory, $airportGraph, $flightsByCategory, $flightsByDelayCode, $categories, $airlinesByCategory, $flights)
            = $this->punctualityArray(null, null, false, $selectedAirports, $selectedAirlines, $request->get("type"));

        debug($request->all());

        $this->viewData = [
            'dateYesterday'     => date('Y-m-d',strtotime(" - 1 days")),
            'flights'           => $flights,
            'categories'        => $categories,
            //      "handlingStations"  => $handlingStations,

            "airlines"          => $handlingAirlines,
            "airports"          => $handlingAirports,

            "flightNumbers"     => FlightNumber::getArrayFlightNumbersWithSector(false, false, false, null, true),
            'sectors'           => FlightSector::listSectors(),
            'sectorsByAirlines' => FlightSector::getSectorsByAirlines(),

            'periods'            => getPeriodsArray(),
            'monthNames'         => getMonthsArray(),
            'years'              => getYearsArray(),

            'flightsByCategory'         => isset($flightsByCategory) ? $flightsByCategory : [],
            'flightsByDelayCode'        => isset($flightsByDelayCode) ? $flightsByDelayCode : [],
            'flightsByCategoryChart'    => isset($flightsByCategory) ? $this->getFlightsByCategoryChart($flightsByCategory, $categories) : [],
            'flightsByDelayCodeChart'   => isset($flightsByDelayCode) ? FlightDelay::getFlightsByDelayChart($flightsByDelayCode) : [],
            'airportsByCategory'        => isset($airportsByCategory) ? $airportsByCategory : null,
            'airportsUnique'            => isset($airportsUnique) ? $airportsUnique : null,
            'airportGraph'              => isset($airportGraph) ? $airportGraph : null,
        ];


        return view('report.h-punctuality', $this->viewData);
    }

    public function handlingPunctualityReport(){

        // Get Staff Handling AP and Airlines
        list($handlingAirlines, $handlingAirports) = getStaffHandlingAirports();

        $selectedAirports = \request()->has("airport") ? \request()->get("airport") : array_keys($handlingAirports);
        $selectedAirlines = \request()->has("airline") ? \request()->get("airline") : array_keys($handlingAirlines);

        // Get Punctuality Ready Array
        list($airportsUnique, $airportsByCategory, $airportGraph, $flightsByCategory, $flightsByDelayCode, $categories) = $this->punctualityArray(null, null, false, $selectedAirports, $selectedAirlines, DEPARTURE);

        return view('report.h-punctuality-report',[
            'dateYesterday'      => date('Y-m-d',strtotime(" - 1 days")),
            'categories'         => $categories,

//            "handlingStations"  => $handlingStations,

            "airlines"          => $handlingAirlines,
            "airports"          => $handlingAirports, //Airport::listHandlingAirports(),

//            "flightNumbers"     => FlightNumber::getArrayFlightNumbersWithSector(),
            "flightNumbers"     => FlightNumber::getArrayFlightNumbersWithSector(false, false, false, null, true),
            'sectors'           => FlightSector::listSectors(),
            'sectorsByAirlines' => FlightSector::getSectorsByAirlines(),

            'periods'            => getPeriodsArray(),
            'monthNames'         => getMonthsArray(),
            'years'              => getYearsArray(),
            'flightsByCategory'  => isset($flightsByCategory) ? $flightsByCategory : null,
            'flightsByDelayCode' => isset($flightsByDelayCode) ? $flightsByDelayCode : null,
            'flightsByCategoryChart'  => isset($flightsByCategory) ? $this->getFlightsByCategoryChart($flightsByCategory, $categories) : [],
            'flightsByDelayCodeChart' => isset($flightsByDelayCode) ? FlightDelay::getFlightsByDelayChart($flightsByDelayCode) : [],
            'airportsByCategory' => isset($airportsByCategory) ? $airportsByCategory : null,
            'airportsUnique'     => isset($airportsUnique) ? $airportsUnique : null,
            'airportGraph'       => isset($airportGraph) ? $airportGraph : null,

        ]);
    }

    public function punctuality($widget = null){

        // Get Punctuality Ready Array
        list($airportsUnique, $airportsByCategory, $airportGraph, $flightsByCategory, $flightsByDelayCode, $categories) = $this->punctualityArray();

        return view('report.punctuality',[
            'dateYesterday'      => date('Y-m-d',strtotime(" - 1 days")),
            'categories'         => $categories,
            'airports'           => FlightNumber::listAirportsByFlightType('Scheduled', 'departure_airport'),

            'flightNumbers'      => FlightNumber::listFlightNumbersByFlightTypeAndAirport('Scheduled'),
            'sectors'            => FlightSector::listSectors(),
            'sectorsByAirlines'  => FlightSector::getSectorsByAirlines(),

            'periods'            => getPeriodsArray(),
            'monthNames'         => getMonthsArray(),
            'years'              => getYearsArray(),
            'flightsByCategory'  => isset($flightsByCategory) ? $flightsByCategory : null,
            'flightsByDelayCode' => isset($flightsByDelayCode) ? $flightsByDelayCode : null,
            'flightsByCategoryChart'  => isset($flightsByCategory) ? $this->getFlightsByCategoryChart($flightsByCategory, $categories) : [],
            'flightsByDelayCodeChart' => isset($flightsByDelayCode) ? FlightDelay::getFlightsByDelayChart($flightsByDelayCode) : [],
            'airportsByCategory' => isset($airportsByCategory) ? $airportsByCategory : null,
            'airportsUnique'     => isset($airportsUnique) ? $airportsUnique : null,
            'airportGraph'       => isset($airportGraph) ? $airportGraph : null,

        ]);
    }

    public function punctualityReport(){

        // Get Punctuality Ready Array
        list($airportsUnique, $airportsByCategory, $airportGraph, $flightsByCategory, $flightsByDelayCode, $categories) = $this->punctualityArray();

        return view('report.punctuality-report',[
            'dateYesterday'      => date('Y-m-d',strtotime(" - 1 days")),
            'categories'         => $categories,
            'airports'           => FlightNumber::listAirportsByFlightType('Scheduled', 'departure_airport'),

            'flightNumbers'      => FlightNumber::listFlightNumbersByFlightTypeAndAirport('Scheduled'),
            'sectors'            => FlightSector::listSectors(),
            'sectorsByAirlines'  => FlightSector::getSectorsByAirlines(),

            'periods'            => getPeriodsArray(),
            'monthNames'         => getMonthsArray(),
            'years'              => getYearsArray(),
            'flightsByCategory'  => isset($flightsByCategory) ? $flightsByCategory : null,
            'flightsByDelayCode' => isset($flightsByDelayCode) ? $flightsByDelayCode : null,
            'flightsByCategoryChart'  => isset($flightsByCategory) ? $this->getFlightsByCategoryChart($flightsByCategory, $categories) : [],
            'flightsByDelayCodeChart' => isset($flightsByDelayCode) ? FlightDelay::getFlightsByDelayChart($flightsByDelayCode) : [],
            'airportsByCategory' => isset($airportsByCategory) ? $airportsByCategory : null,
            'airportsUnique'     => isset($airportsUnique) ? $airportsUnique : null,
            'airportGraph'       => isset($airportGraph) ? $airportGraph : null,

        ]);
    }



    public function getFlightsByCategoryChart($flightsByCategory, $categories){
        $flightsByCategoryChart = [
            "category"  => [],
            "value"     => []
        ];
        if ($flightsByCategory && count($flightsByCategory) && $categories && count($categories)) {
            foreach ($flightsByCategory as $index => $each) {
                if (isset($categories[$index])){
                    $flightsByCategoryChart["category"][] = trans("content.".$categories[$index]);
                    $flightsByCategoryChart["value"][] = $each->percentage;
                }
            }
        }

        return array_reverse($flightsByCategoryChart);
    }



    public static function punctualityArray($dateFrom = null, $dateTo = null, $widget = false, $selectedAirports = null, $selectedAirlines = null, $type = DEPARTURE){
        if ($dateFrom && $dateTo){
            $dateSearch = [
                'from'  => $dateFrom,
                'to'    => $dateTo
            ];
            $flightSearch = null;
        }
        else {
            $dateYesterday = date('Y-m-d', strtotime(" - 1 days"));
            $flightSearchOption = \request()->get('flightSearch');
            switch ($flightSearchOption) {
                case 'flightNumber':
                    $flightSearch = ['flightNumberIds' => \request()->get('flightNumber')];
                    break;
                case 'sector':
                    $flightSearch = ['sectorIds' => \request()->get('sector')];
                    break;
                default:
                    $flightSearch = null;
                    break;
            }

            $dateSearchOption = \request()->get('dateSearch');
            switch ($dateSearchOption) {
                case 'monthYear':
                    $dateSearch = [
                        'month' => \request()->get('month') ? \request()->get('month') : date('m'),
                        'year' => \request()->get('year') ? \request()->get('year') : date('Y')
                    ];
                    break;
                default:
                    $dateSearch = [
                        'from'  => \request()->get('dateFrom')   ? \request()->get('dateFrom') : $dateYesterday,
                        'to'    => \request()->get('dateTo')     ? \request()->get('dateTo')  : $dateYesterday
                    ];
                    break;

            }
        }

        debug($type);


        // Get All Flights (filter: Date, Flight Number)
        $flights = Flight::getFlightsByDateFlightNumber($dateSearch, $flightSearch, null,
            ($selectedAirports ? $selectedAirports : \request()->get("airport")),
            ($selectedAirlines ? $selectedAirlines : \request()->get("airline")),
            $type
        );

        // Get Flights By Category And By Delay Code
        $result =  self::createFlightsByCategoryAndFlightsByDelayCodeArrays($flights);

        $airportsUnique = $airportsByCategory = $flightsByCategory = $flightsByDelayCode = $airportGraph = $airlinesByCategory = [];

        if ($result){
            $flights = $result[0];
            $flightsByCategory = $result[1];
            $flightsByDelayCode = $result[2];

            debug($flightsByCategory);

            // Calculate Each Categories Percentage (Each/Total) By Airports
            list($airportsUnique, $airportsByCategory, $airportGraph, $airlinesByCategory) = self::calculateCategoriesPercentageByAirports($flightsByCategory);

            // Calculate Each Categories Percentage (Each/Total)
            $flightsByCategory = self::calculateCategoriesPercentage($flightsByCategory, count($flights));

            // Calculate Each Delays Percentage
            $flightsByDelayCode = self::calculateDelaysPercentage($flightsByDelayCode);
        }


        // List Of Categories
        $categories = ['on_time', 'within_15_mins', 'within_30_mins', 'within_60_mins', 'above_60_mins' /*,'Total'*/];

        // For Home Page Widget Graph
        if ($widget){
            return [$flightsByCategory, $categories];
        }

        debug($airportsByCategory);

        return [$airportsUnique, $airportsByCategory, $airportGraph, $flightsByCategory, $flightsByDelayCode, $categories, $airlinesByCategory, $flights];

    }


    public function seatfactors(){

        // Prepare Seat Factors Array
        $seatFactors = $this->seatFactorsArray(null, null, null, null, true);

        list($chartCapPax, $chartSF) = $this->prepareChartData($seatFactors);

        return view('report.seatfactors',[
            'dateYesterday'    =>  date('Y-m-d',strtotime(" - 1 days")),
            'periods'          => [ '' => 'Select', 0 => 'Yearly', 1 => 'Quarterly', 2 => 'Monthly', 3 => 'Weekly', 4 => 'Weekdays', 5 => 'Daily', 6 => 'Flight Number', 7 =>'Sector'],
            'airports'         => FlightNumber::listAirportsByFlightType('Scheduled', 'departure_airport'),

            'flightNumbers'     => FlightNumber::listFlightNumbersByFlightTypeAndAirport('Scheduled'),
            'sectors'           => FlightSector::listSectors(),
            'sectorsByAirlines' => FlightSector::getSectorsByAirlines(),

            'weekday'          => ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
            'months'           => [1 => "Jan", 2 => "Feb", 3 => "Mar", 4 => "Apr", 5 => "May", 6 => "Jun", 7 => "Jul", 8 => "Aug", 9 => "Sep", 10 => "Oct", 11 => "Nov", 12 => "Dec"],
            'monthNames'       => [1 => "January", 2 => "February", 3 => "March", 4 => "April", 5 => "May", 6 => "June", 7 => "July", 8 => "August", 9 => "September", 10 => "October", 11 => "November", 12 => "December"],
            'years'            => Flight::select(DB::raw('DISTINCT(YEAR(std)) as year'))->pluck('year', 'year')->all(),
            'seatFactors'      => isset($seatFactors) ? $seatFactors : [],
            'chartCapPax'      => $chartCapPax,
            'chartSF'          => $chartSF,
            'isPeriod'         => \request()->get('dateSearch') == 'fromTo'
        ]);

    }

    public function prepareChartData($seatFactors){
        $chartCapPax = [
            "date"  => [],
            "cap"   => [],
            "pax"   => []
        ];

        $chartSF = [
            "date"  => [],
            "sf"    => [],
        ];

        foreach ($seatFactors as  $date => $each) {
            $chartCapPax["date"][] = $date;
            $chartCapPax["cap"][] = $each->config_c +$each->config_y;
            $chartCapPax["pax"][] = $each->pax_c + $each->pax_y;

            $chartSF["date"][] = $date;
            $chartSF["sf"][] = $each->sf > 100 ? 100 : round($each->sf);
        }

        return [
            $chartCapPax,
            $chartSF
        ];
    }

    public static function getIATACodeFromFltNo($fltNo){
        preg_match("/(\\w{2,3})-.*/", $fltNo, $matches);
        if (count($matches) && isset($matches[1]) && $matches[1]){
            return $matches[1];
        }
        return null;
    }

    /**
     * Calculate Categories Percentage By Airport (Use Flight Count)
     * @param $flightsByCategory
     * @return mixed
     */
    public static function calculateCategoriesPercentageByAirports($flightsByCategory){
        $airports = $airportsByCategory = $airlinesByCategory = [];

        $hAirports = Airport::listHandlingStations();

        $ghaModule = ghaModule();

        debug($flightsByCategory);

        if (isset($flightsByCategory) && $flightsByCategory) {
            for ($i = 0; $i < 5; $i++) {
                if (isset($flightsByCategory[$i])) {
                    $flightNumbers = $flightsByCategory[$i]->flight_number;
                    if (count($flightNumbers)) {
                        foreach ($flightNumbers as $flNo => $monthsData) {
                            $countFlights = 0;
                            $airline = self::getIATACodeFromFltNo($flNo);

                            $currentAirport = null;

                            foreach ($monthsData as $month => $data) {
                                if ($month == "airport"){

                                    // Skip non-handing Airports
                                    if ($ghaModule && !in_array($data, $hAirports)){
                                        continue;
                                    }

                                    $currentAirport = $data;
                                    $airports[$currentAirport] = 0;
                                    if (!isset($airportsByCategory[$i][$currentAirport])) {
                                        $airportsByCategory[$i][$currentAirport] = new stdClass();
                                        $airportsByCategory[$i][$currentAirport]->count = 0;
                                        $airportsByCategory[$i][$currentAirport]->accumulated_count = 0;
                                    }
                                }
                                else {
                                    $countFlights += count($data);
                                }
                            }

                            if ($currentAirport && isset($airportsByCategory[$i][$currentAirport])) {
                                $airportsByCategory[$i][$currentAirport]->count += $countFlights;
                            }

                            if (!isset($airlinesByCategory[$i][$airline])) {
                                $airlinesByCategory[$i][$airline] = new stdClass();
                                $airlinesByCategory[$i][$airline]->count = 0;
                                $airlinesByCategory[$i][$airline]->accumulated_count = 0;
                            }

                            $airlinesByCategory[$i][$airline]->count += $countFlights;
                        }
                    }

                    // Since Next Category Includes Previously Categories, Except For Last Category (Above 60min)
                    $j = $i - 1;
                    if (isset($flightsByCategory[$j]) && isset($airportsByCategory[$j])) {
                        foreach ($airportsByCategory[$j] as $airportName => $airportEach) {
                            if (!isset($airportsByCategory[$i][$airportName])) {
                                $airportsByCategory[$i][$airportName] = new stdClass();
                                $airportsByCategory[$i][$airportName]->count = 0;
                                $airportsByCategory[$i][$airportName]->accumulated_count = 0;
                            }
                        }
                    }

                    if (isset($airlinesByCategory[$j])) {
                        foreach ($airlinesByCategory[$j] as $airline => $obj ) {
                            if (!isset($airlinesByCategory[$i][$airline])) {
                                $airlinesByCategory[$i][$airline] = new stdClass();
                                $airlinesByCategory[$i][$airline]->count = 0;
                                $airlinesByCategory[$i][$airline]->accumulated_count = 0;
                            }
                        }
                    }
                }
            }

            $airports = array_keys($airports);
            $airportGraph = [];

            // Calculate Percentages of Airports
            foreach ($airports as $airport) {
                $airportsByCategory[$airport] = 0;
                // First Loop Counts Total Flights
                $totalFlights = 0;
                for ($i = 0; $i < 5; $i++) {
                    if (!isset($airportsByCategory[$i][$airport])) {
                        $airportsByCategory[$i][$airport] = new stdClass();
                        $airportsByCategory[$i][$airport]->count = 0;
                    }
                    $counter = $airportsByCategory[$i][$airport]->count; //1

                    $totalFlights += $counter;
                    $airportsByCategory[$i][$airport]->accumulated_count = $counter; // 1

                    if ($i < 4) {
                        for ($j = 0; $j < $i; $j++) {
                            $airportsByCategory[$i][$airport]->accumulated_count += $airportsByCategory[$j][$airport]->count;
                        }
                    }

                    $airportsByCategory[$airport] += $counter;
                }

                // Second Loops Counts Percentages
                for ($i = 0; $i < 5; $i++) {

                    $previousCounter = 0;
                    $percentage = $totalFlights == 0 ? 0 : round( ($airportsByCategory[$i][$airport]->accumulated_count + $previousCounter) * 100 / $totalFlights );
                    $airportsByCategory[$i][$airport]->percentage = $percentage;

                    // Graph Google
                    if (!isset($airportGraph[$i])) {
                        $airportGraph[$i] = [
                            "airports"  => [],
                            "value"     => []
                        ];
                    }

//                    $airportGraph[$i][] = [$airport, $percentage];
                    $airportGraph[$i]["airports"][] = $airport;
                    $airportGraph[$i]["value"][] = $percentage;
                }
            }

            // Calculate Percentages of Airlines
//            $airlines = Airline::listHandlingAirlines("iata");
            $airlines = Airline::listCode();

            foreach ($airlines as $airline) {
                $airlinesByCategory[$airline] = 0;
                // First Loop Counts Total Flights
                $totalFlights = 0;
                for ($i = 0; $i < 5; $i++) {
                    if (!isset($airlinesByCategory[$i][$airline])) {
                        $airlinesByCategory[$i][$airline] = new stdClass();
                        $airlinesByCategory[$i][$airline]->count = 0;
                    }
                    $counter = $airlinesByCategory[$i][$airline]->count; //1

                    $totalFlights += $counter;
                    $airlinesByCategory[$i][$airline]->accumulated_count = $counter; // 1

                    if ($i < 4) {
                        for ($j = 0; $j < $i; $j++) {
                            $airlinesByCategory[$i][$airline]->accumulated_count += $airlinesByCategory[$j][$airline]->count;
                        }
                    }

                    $airlinesByCategory[$airline] += $counter;
                }

                // Second Loops Counts Percentages
                for ($i = 0; $i < 5; $i++) {

                    $previousCounter = 0;
                    $percentage = $totalFlights == 0 ? 0 : round( ($airlinesByCategory[$i][$airline]->accumulated_count + $previousCounter) * 100 / $totalFlights );
                    $airlinesByCategory[$i][$airline]->percentage = $percentage;

                    // Graph Google
                    if (!isset($airlineGraph[$i])) {
                        $airlineGraph[$i] = [
                            "airlines"  => [],
                            "value"     => []
                        ];
                    }

//                    $airportGraph[$i][] = [$airline, $percentage];
                    $airlineGraph[$i]["airlines"][] = $airline;
                    $airlineGraph[$i]["value"][] = $percentage;
                }
            }

            return [$airports, $airportsByCategory, $airportGraph, $airlinesByCategory, $airlineGraph];
        }

    }


    /**
     * Prepare Array for Seat Factors Report
     * @param null $dateFrom
     * @param null $dateTo
     * @param null $label
     * @param null $period
     * @param bool|false $sort
     * @return array
     */
    public static function seatFactorsArray($dateFrom = null, $dateTo = null, $label = null, $period = null, $sort = false){
        if ($dateFrom && $dateTo) {
            $dateSearch = [
                0 => [
                    'from'  => $dateFrom,
                    'to'    => $dateTo,
                    'label' => $label
                ]
            ];
            $flightSearch = null;
        }
        else {
            $period = \request()->get('period');
            $dateYesterday = date('Y-m-d', strtotime(" - 1 days"));
            $dateLastWeek = date('Y-m-d', strtotime(" - 7 days"));
            $dateFirstOfMonth = date('Y-m-d', strtotime('first day of this month', strtotime($dateYesterday)));

            // Get Flight Search Filter Based On Options Defined By User
            $flightSearchOption = \request()->get('flightSearch');
            switch ($flightSearchOption) {
                case 'flightNumber':
                    $flightSearch = ['flightNumberIds' => \request()->get('flightNumber')];
                    break;
                case 'sector':
                    $flightSearch = ['sectorIds' => \request()->get('sector')];
                    break;
                default:
                    $flightSearch = null;
                    break;
            }

            // Get Flight Search Filter Based On Options Defined By User
            $dateSearchOption = \request()->get('dateSearch');
            switch ($dateSearchOption) {
                case 'monthYear':
                    $dateSearch = [
                        0 => [
                            'month' => \request()->get('month'), 'year' => \request()->get('year')
                        ]
                    ];
                    break;
                case 'fromTo' :
                    $dateSearch = [
                        0 => [
                            'from' => \request()->has('dateFrom') ? \request()->get('dateFrom') : $dateYesterday,
                            'to' => \request()->has('dateTo') ? \request()->get('dateTo') : $dateYesterday
                        ]
                    ];
                    break;
                default:
                    $dateSearch = [
                        0 => ['from' => $dateYesterday,     'to' => $dateYesterday, 'label' => 'Yesterday'],
                        1 => ['from' => $dateLastWeek,      'to' => $dateYesterday, 'label' => 'Last 7 Days'],
                        2 => ['from' => $dateFirstOfMonth,  'to' => $dateYesterday, 'label' => 'Month To Date']
                    ];
                    break;
            }
        }

        // Compute Seat Factors Array Of Objects Based on Flight Configuration and Actual PAX Load
        $seatFactors = self::getSeatFactorArray($dateSearch, $flightSearch, $period, $sort);

        return isset($seatFactors) ? $seatFactors : [];
    }

    public function anyRevenue(){
        $dateYesterday = date('Y-m-d', strtotime(' - 1 days'));
        $period = \request()->get('period');

        $flightSearchOption = \request()->get('flightSearch');
        switch ($flightSearchOption){
            case 'flightNumber':
                $flightSearch = ['flightNumberIds' => \request()->get('flightNumber') ];
                break;
            case 'sector':
                $flightSearch = ['sectorIds' => \request()->get('sector') ];
                break;
            default:
                $flightSearch = null;
                break;
        }

        $dateSearchOption = \request()->get('dateSearch');
        switch ($dateSearchOption){
            case 'monthYear':
                $dateSearch = ['month' => \request()->get('month'),
                    'year'  => \request()->get('year')];
                break;
            default:
                $dateSearch = ['from' => \request()->has('dateFrom') ? \request()->get('dateFrom') : $dateYesterday,
                    'to'   => \request()->has('dateTo') ? \request()->get('dateTo') : $dateYesterday];
                break;

        }
        // Number Of Flights
        $flights = Flight::getFlightsByDateFlightNumber($dateSearch, $flightSearch, $period, \request()->get('airport'));
        // Get PAX and Revenue By Cabin
        $paxRevenueByCabin = PassengerFlight::getPaxAndRevenueByCabin($dateSearch, "BothCabin", $flightSearch, $period);

        return view('report.revenue',[
            'flights'          => $flights,
            'paxRevenueByCabin'=> $paxRevenueByCabin,
            'dateYesterday'    => $dateYesterday,
            'countries'        => Country::orderBy('country')->pluck('country', 'id')->all(),
            'cities'           => City::orderBy('city')->pluck('city','id')->all(),
            'agency'           => Agency::orderBy('agency')->pluck('agency','id')->all(),
            'reportTypes'      => ['None', 'Country', 'Route', 'POS'],
            'periods'          => [ '' => 'Select', 0 => 'Yearly', 1 => 'Quarterly', 2 => 'Monthly', 3 => 'Weekly', 4 => 'Weekdays', 5 => 'Daily', 6 => 'Flight Number', 7 =>'Sector'],

            'flightNumbers'     => FlightNumber::listFlightNumbersByFlightTypeAndAirport('Scheduled'),
            'sectors'           => FlightSector::listSectors(),
            'sectorsByAirlines' => FlightSector::getSectorsByAirlines(),

            'airports'         => FlightNumber::listAirportsByFlightType('Scheduled', 'departure_airport'),
            'weekday'          => ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
            'months'           => [1 => "Jan", 2 => "Feb", 3 => "Mar", 4 => "Apr", 5 => "May", 6 => "Jun", 7 => "Jul", 8 => "Aug", 9 => "Sep", 10 => "Oct", 11 => "Nov", 12 => "Dec"],
            'monthNames'       => [1 => "January", 2 => "February", 3 => "March", 4 => "April", 5 => "May", 6 => "June", 7 => "July", 8 => "August", 9 => "September", 10 => "October", 11 => "November", 12 => "December"],
            'years'            => Flight::select(DB::raw('DISTINCT(YEAR(std)) as year'))->pluck('year', 'year')->all(),
            'seatFactors'      => isset($seatFactors) ? $seatFactors : []
        ]);

    }

    /*public function getRevenue()
    {
        $date_from = (\request()->has('from_date')) ? \request()->get('from_date') : null;//date('Y-01-01');
        $date_to = (\request()->has('to_date')) ? \request()->get('to_date') : null;//date('Y-01-01');
        $period = (\request()->has('period')) ? \request()->get('period') : null;

        $country = (\request()->has('country')) ? \request()->get('country') : null;//date('Y-01-01');
        $city = (\request()->has('city')) ? \request()->get('city') : null;//date('Y-01-01');
        $agency = (\request()->has('agency')) ? \request()->get('agency') : null;//date('Y-01-01');
        $report_type = (\request()->has('report_type')) ? \request()->get('report_type') : "None";//date('Y-01-01');


        $cur_year =  (int)date('Y');
        $prev_year = $cur_year - 1;
        $comparison_years = [ $cur_year, $prev_year];

        foreach ($comparison_years as $each) {
            $flight_count_out[] = "flight_count_out_{$each}";
            $config_out_c[] = "config_out_c_{$each}";
            $config_out_y[] = "config_out_y_{$each}";
            $pax_out_c[] = "pax_out_c_{$each}";
            $pax_out_y[] = "pax_out_y_{$each}";
            $revenue_out_c[] = "revenue_out_c_{$each}";
            $revenue_out_y[] = "revenue_out_y_{$each}";
            $yield_out_c[] = "yield_out_c_{$each}";
            $yield_out_y[] = "yield_out_y_{$each}";

            $flight_count_in[] = "flight_count_in_{$each}";
            $config_in_c[] = "config_in_c_{$each}";
            $config_in_y[] = "config_in_y_{$each}";
            $pax_in_c[] = "pax_in_c_{$each}";
            $pax_in_y[] = "pax_in_y_{$each}";
            $revenue_in_c[] = "revenue_in_c_{$each}";
            $revenue_in_y[] = "revenue_in_y_{$each}";
            $yield_in_c[] = "yield_in_c_{$each}";
            $yield_in_y[] = "yield_in_y_{$each}";
        }

        if (isset($period)) {   // $period   1-yearly  2-quarterly  3-monthly  4-weekly   5-daily


            $flights_config = Flight::getFlightsConfigComparison($date_from, $date_to, $period, $comparison_years);
            $booked_flights = PassengerFlight::getFlightPaxRevenueComparison($date_from, $date_to, $period, $comparison_years, $country, $city, $agency, $report_type);

            $final_data = array();
            $u = 0;
            foreach ($booked_flights as $i => $each) {
                $final_data[$i] = new stdClass();
                foreach ($each['attributes'] as $k => $value) {
                    $final_data[$i]->$k = $value;
                }


                $found = 0;
                foreach ($flights_config as $j => $item) {
                    if ($item->date == $each->date) {
                        $found = 1;
                        break;
                    }
                }
                foreach ($flights_config[$j]['attributes'] as $k => $value) {
                    if ($k != 'date') {
                        if ($found == 1)
                            $final_data[$i]->$k = $value;
                        else
                            $final_data[$i]->$k = 'N/A';
                    }
                }

                foreach ($comparison_years as $u => $item) {
                    $final_data[$i]->{"yield_out_c_" . $item} = ($final_data[$i]->{"pax_out_c_" . $item} == 0 ? "-" : round($final_data[$i]->{"revenue_out_c_" . $item} / $final_data[$i]->{"pax_out_c_" . $item}));
                    $final_data[$i]->{"yield_out_y_" . $item} = ($final_data[$i]->{"pax_out_y_" . $item} == 0 ? "-" : round($final_data[$i]->{"revenue_out_y_" . $item} / $final_data[$i]->{"pax_out_y_" . $item}));
                    $final_data[$i]->{"yield_in_c_" . $item} = ($final_data[$i]->{"pax_in_c_" . $item} == 0 ? "=" : round($final_data[$i]->{"revenue_in_c_" . $item} / $final_data[$i]->{"pax_in_c_" . $item}));
                    $final_data[$i]->{"yield_in_y_" . $item} = ($final_data[$i]->{"pax_in_y_" . $item} == 0 ? "=" : round($final_data[$i]->{"revenue_in_y_" . $item} / $final_data[$i]->{"pax_in_y_" . $item}));
                }

                foreach ($comparison_years as $u => $item) {

                    $final_data[$i]->revenue_c_total[$u] = round($final_data[$i]->{"revenue_out_c_" . $item} + $final_data[$i]->{"revenue_in_c_" . $item});
                    $final_data[$i]->revenue_y_total[$u] = round($final_data[$i]->{"revenue_out_y_" . $item} + $final_data[$i]->{"revenue_in_y_" . $item});
                    $final_data[$i]->revenue_out_total[$u] = round($final_data[$i]->{"revenue_out_c_" . $item} + $final_data[$i]->{"revenue_out_y_" . $item});
                    $final_data[$i]->revenue_in_total[$u] = round($final_data[$i]->{"revenue_in_c_" . $item} + $final_data[$i]->{"revenue_in_y_" . $item});

                    $final_data[$i]->revenue_total[$u] = round($final_data[$i]->revenue_c_total[$u] + $final_data[$i]->revenue_y_total[$u]);

                    $final_data[$i]->pax_c_total[$u] = $final_data[$i]->{"pax_in_c_" . $item} + $final_data[$i]->{"pax_out_c_" . $item};
                    $final_data[$i]->pax_y_total[$u] = $final_data[$i]->{"pax_in_y_" . $item} + $final_data[$i]->{"pax_out_y_" . $item};
                    $final_data[$i]->pax_in_total[$u] = $final_data[$i]->{"pax_in_c_" . $item} + $final_data[$i]->{"pax_in_y_" . $item};
                    $final_data[$i]->pax_out_total[$u] = $final_data[$i]->{"pax_out_c_" . $item} + $final_data[$i]->{"pax_out_y_" . $item};

                    $final_data[$i]->pax_total[$u] = $final_data[$i]->pax_c_total[$u] + $final_data[$i]->pax_y_total[$u];

                    $final_data[$i]->yield_c_total[$u] = $final_data[$i]->pax_c_total[$u] == 0 ? "-" : round($final_data[$i]->revenue_c_total[$u] / $final_data[$i]->pax_c_total[$u]);
                    $final_data[$i]->yield_y_total[$u] = $final_data[$i]->pax_y_total[$u] == 0 ? "-" : round($final_data[$i]->revenue_y_total[$u] / $final_data[$i]->pax_y_total[$u]);
                    $final_data[$i]->yield_in_total[$u] = $final_data[$i]->pax_in_total[$u] == 0 ? "-" : round($final_data[$i]->revenue_in_total[$u] / $final_data[$i]->pax_in_total[$u]);
                    $final_data[$i]->yield_out_total[$u] = $final_data[$i]->pax_out_total[$u] == 0 ? "-" : round($final_data[$i]->revenue_out_total[$u] / $final_data[$i]->pax_out_total[$u]);

                    $final_data[$i]->yield_total[$u] = $final_data[$i]->pax_total[$u] == 0 ? "-" : round($final_data[$i]->revenue_total[$u] / $final_data[$i]->pax_total[$u]);

                    $final_data[$i]->config_c_total[$u] = $final_data[$i]->{"config_out_c_" . $item} + $final_data[$i]->{"config_in_c_" . $item};
                    $final_data[$i]->config_y_total[$u] = $final_data[$i]->{"config_out_y_" . $item} + $final_data[$i]->{"config_in_y_" . $item};
                    $final_data[$i]->config_in_total[$u] = $final_data[$i]->{"config_in_c_" . $item} + $final_data[$i]->{"config_in_y_" . $item};
                    $final_data[$i]->config_out_total[$u] = $final_data[$i]->{"config_out_c_" . $item} + $final_data[$i]->{"config_out_y_" . $item};

                    $final_data[$i]->config_total[$u] = $final_data[$i]->config_c_total[$u] + $final_data[$i]->config_y_total[$u];

                    $final_data[$i]->flight_count_total[$u] = $final_data[$i]->{"flight_count_out_" . $item} + $final_data[$i]->{"flight_count_in_" . $item};

                    if ($final_data[$i]->config_in_total[$u] == 0)
                        $final_data[$i]->sf_in[$u] = 0;
                    else
                        $final_data[$i]->sf_in[$u] = round(($final_data[$i]->pax_in_total[$u]) * 100 / ($final_data[$i]->config_in_total[$u]));

                    if ($final_data[$i]->config_out_total[$u] == 0)
                        $final_data[$i]->sf_out[$u] = 0;
                    else
                        $final_data[$i]->sf_out[$u] = round(($final_data[$i]->pax_out_total[$u]) * 100 / ($final_data[$i]->config_out_total[$u]));


                    if ($final_data[$i]->config_c_total[$u] + $final_data[$i]->config_y_total[$u] == 0)
                        $final_data[$i]->sf[$u] = 0;
                    else
                        $final_data[$i]->sf[$u] = round($final_data[$i]->pax_total[$u] * 100 / ($final_data[$i]->config_total[$u]));

                }

            }

        }
        $variables = [$pax_out_c, $pax_out_y, $pax_in_c, $pax_in_y, $revenue_out_c, $revenue_out_y, $revenue_in_c,
                $revenue_in_y, $flight_count_out, $flight_count_in, $config_out_c, $config_out_y, $config_in_c, $config_in_y];

        $input_all = \request()->all() ? \request()->all() : ['from_date' => null, 'to_date' => null];
        $countries = Country::orderBy('country')->pluck('country', 'id');
        $cities = City::orderBy('city')->pluck('city','id');
        $agencies = Agency::orderBy('agency')->pluck('agency','id');

        return view('report.revenue', [
            'input_all' => $input_all,
            'pax_out_c_var' => $pax_out_c,
            'pax_out_y_var' =>  $pax_out_y,
            'pax_in_c_var' => $pax_in_c,
            'pax_in_y_var' => $pax_in_y,
            'yield_in_c_var' => $yield_in_c,
            'yield_in_y_var' => $yield_in_y,
            'yield_out_c_var' => $yield_out_c,
            'yield_out_y_var' => $yield_out_y,
            'pax_in_y_var' => $pax_in_y,
            'revenue_out_c_var' =>  $revenue_out_c,
            'revenue_out_y_var' =>  $revenue_out_y,
            'revenue_in_c_var' => $revenue_in_c,
            'revenue_in_y_var' => $revenue_in_y,
            'flight_count_out_var' => $flight_count_out,
            'flight_count_in_var' => $flight_count_in,
            'config_out_c_var' => $config_out_c,
            'config_out_y_var' => $config_out_y,
            'config_in_c_var' => $config_in_c,
            'config_in_y_var' =>$config_in_y,
            'variables' => $variables,
            'comparison_years' => $comparison_years,
            'report_type'=>$report_type,
            'report_types'=>['None', 'Country', 'Route', 'POS'],
            'date_from'=>$date_from,
            'date_to'=>$date_to,
            'booked_flights' =>  isset($booked_flights) ? $booked_flights : null,
            'final_data' => isset($final_data) ? $final_data : null,
            'period' => isset($period) ? $period : 'None',
            "country"               => $country,
            "countries"               => $countries,
            "city"                  => $city,
            "cities"                  => $cities,
            "agency"                => $agency,
            "agencies"                => $agencies,
            'weekday' => ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
            'months' => [1 => "Jan", 2 => "Feb", 3 => "Mar", 4 => "Apr", 5 => "May", 6 => "Jun", 7 => "Jul", 8 => "Aug", 9 => "Sep", 10 => "Oct", 11 => "Nov", 12 => "Dec"]

        ]);
    }*/

    public function postAjaxflighttype(){

        $action = \request()->get('action');

        $final = array(
            'success' => 1
        );

        switch($action){
            case 'FLIGHT_TYPE_CHANGE':
                $type = \request()->get('type');
                $response[0] = FlightNumber::getArraySectors($type);
                $response[1] = FlightNumber::getArrayFlightNumbers($type);
                break;
            case 'AGENCY_CHANGE':
                $country = \request()->get('country');
                $city = \request()->get('city');

                if (!empty($country)){
                    $response[0] = City::select('city','id');
                    $response[1] = Agency::select('agency','id');
                    foreach($response as $each){
                        if (is_array($country) && count($country) > 1)
                            $each->whereIn('country_id',$country);
                        else
                            $each->where("country_id",$country);
                    }
                }
                if (!empty($city)){
                    if (!isset($response[1]))
                        $response[1] = Agency::select('agency','id');
                    if (is_array($city) && count($city) > 1)
                        $response[1]->whereIn('city_id', $city);
                    else
                        $response[1]->where('city_id', $city);
                }

                $final[0] = $response[0]->orderBy('city')->pluck('city','id')->all();
                $final[1] = $response[1]->orderBy('agency')->pluck('agency','id')->all();
                break;
            case 'SEARCH_REVENUE':
                $data = \request()->get('data_all');
                $response[0] = $this::search_parse($data);
        }

        return response()->json($final);
    }


    /**
     * Calculate Categories Percentage (Use Flight Count)
     * @param $flightsByCategory
     * @param $flightCount
     * @return mixed
     */
    public static function calculateCategoriesPercentage($flightsByCategory, $flightCount){
        if (isset($flightsByCategory) && $flightsByCategory) {
            for ($i = 0; $i < 5; $i++) {
                if (isset($flightsByCategory[$i])) {
                    $count = $flightsByCategory[$i]->count;
                    for ($j = 0; $j < $i; $j++) {
                        if (isset($flightsByCategory[$j]))
                            $count += $flightsByCategory[$j]->count;
                    }
                    $flightsByCategory[$i]->percentage = round($count * 100 / $flightCount);
                }
                else {
                    $flightsByCategory[$i] = new stdClass();
                    $flightsByCategory[$i]->count = 0;
                    $flightsByCategory[$i]->flight_number = [];
                    $flightsByCategory[$i]->percentage = 0;
                    if ($i > 0) {
                        for ($k = $i - 1; $k >= 0; $k--) {
                            if (isset($flightsByCategory[$k])) {
                                $flightsByCategory[$i]->percentage = $flightsByCategory[$k]->percentage;
                                break;
                            }
                        }
                    }
                }

                if ($i == 4){
                    $flightsByCategory[$i]->percentage = 100 - $flightsByCategory[$i-1]->percentage;
                }

            }
            ksort($flightsByCategory);
            return $flightsByCategory;
        }

    }


    /**
     * Calculate Each Delays Percentage Based on Total Delay Number
     * @param $flightsByDelayCode
     * @return mixed
     */
    public static function calculateDelaysPercentage($flightsByDelayCode){
        if ($flightsByDelayCode && count($flightsByDelayCode)) {
            $totalDelays = 0;
            foreach ($flightsByDelayCode as $i => $each) {
                $totalDelays += $each->count;
            }

            foreach ($flightsByDelayCode as $i => $each) {
                $count = $each->count;
                $flightsByDelayCode[$i]->percentage = round($count * 100 / $totalDelays, 1);
                $flightsByDelayCode[$i]->duration = Minutes_To_Time($each->duration);
            }
            uasort($flightsByDelayCode, 'sortArray');
            return $flightsByDelayCode;
        }
    }


    /**
     * create Flights By Category And Flights By Delay AgencyCode Arrays
     * @param $flights
     * @return array|bool
     */
    public static function createFlightsByCategoryAndFlightsByDelayCodeArrays($flights){
        if ($flights && count($flights)) {
            $flightsByCategory = [];
            $flightsByDelayCode = [];
            $multipleAirlines = ghaModule() && env(MULTIPLE_AIRLINE);

            $airports = [];
            foreach ($flights as $i => $each) {
                $delayCode = $delayReason = $delayDuration = [];
                $flightDelays = $each->delays;
                $airlineID = getFlightAirlineID($each);

                $departureAirport = $each->departureAirport ? $each->departureAirport : $each->flightNumber->departureAirport;
                if ($departureAirport){
                    $flights[$i]->departure_airport = $departureAirport->iata;
                }

                if (count($flightDelays)) {
                    $flights[$i]->is_delayed = true;
                    // Iterate Each Delays
                    foreach ($flightDelays as $eachDelay) {
                        $code = ($eachDelay->delay->airline ? $eachDelay->delay->airline->iata." " : "")
                            .($airlineID == 253 ? $eachDelay->delay->alpha_code : $eachDelay->delay->code);
                        $reason = $eachDelay->delay->description;
                        $delayCode[] = $code;
                        $delayReason[] = $reason;
                        $delayDuration[] = TimeToMinutes($eachDelay->duration, true);
                    }
                    $flights[$i]->delay_code = $delayCode;
                    $flights[$i]->delay_reason = $delayReason;
                    $flights[$i]->delay_duration = $delayDuration;
                }
                else {
                    $flights[$i]->is_delayed = false;
                }

                $category = getDelayCategory($flights[$i]);
                // DEBUG
                if (!isset($airports[$departureAirport->iata])){
                    $airports[$departureAirport->iata] = 0;
                }
                $airports[$departureAirport->iata]++;
                // END

                $flightNumber = "";
                if ($multipleAirlines){
                    $flightNumber = getAirlineCode($flights[$i]->flightNumber->airline)."-";
                }

                $flightNumber .= $departureAirport->iata.$flights[$i]->flight_number;

                $departureDateDay = date('d', strtotime($flights[$i]->date));
                $departureDateMonth = date('M', strtotime($flights[$i]->date));

                $flights[$i]->category = $category;

                // Flights By Category
                if (!isset($flightsByCategory[$category])){
                    $flightsByCategory[$category] = new stdClass();
                    $flightsByCategory[$category]->count = 0;
                    $flightsByCategory[$category]->flights = [];
                }

                $flightsByCategory[$category]->count++;

                if ($flights[$i]->is_delayed){
                    $flightsByCategory[$category]->flights[] = $each;
                }

                if (isset($flightsByCategory[$category]->flight_number[$flightNumber])) {
                    // Group Date if Found The Same Month (i.e. depDate = Feb 08, Already Have Feb 03, Then -> Feb 03, 08)
                    $foundSameMonth = false;
                    foreach ($flightsByCategory[$category]->flight_number[$flightNumber] as $dateMonth => $dayArray) {
                        if ($dateMonth == $departureDateMonth) {
                            $flightsByCategory[$category]->flight_number[$flightNumber][$dateMonth][] = $departureDateDay;
                            sort($flightsByCategory[$category]->flight_number[$flightNumber][$dateMonth]);
                            $foundSameMonth = true;
                            break;
                        }
                    }
                    if (!$foundSameMonth){
                        $flightsByCategory[$category]->flight_number[$flightNumber]["$departureDateMonth"][] = $departureDateDay;
                    }
                }
                else{
                    $flightsByCategory[$category]->flight_number[$flightNumber] = [
                        $departureDateMonth => [
                            $departureDateDay
                        ],
                        "airport"           => $flights[$i]->departure_airport
                    ];
//                    $flightsByCategory[$category]->flight_number[$flightNumber]["$departureDateMonth"] = [$departureDateDay];
//                    $flightsByCategory[$category]->flight_number[$flightNumber]["airport"] = $flights[$i]->departure_airport;
                }

                // Delayed Part
                if ($flights[$i]->is_delayed) {
                    foreach ($delayCode as $k => $eachDelay) {
                        if (!isset($flightsByDelayCode[$eachDelay])){
                            $flightsByDelayCode[$eachDelay] = new stdClass();
                            $flightsByDelayCode[$eachDelay]->count = 0;
                            $flightsByDelayCode[$eachDelay]->duration = 0;
                        }
                        $flightsByDelayCode[$eachDelay]->count++;
                        $flightsByDelayCode[$eachDelay]->duration += $each->delay_duration[$k];

                        if (isset($flightsByDelayCode[$eachDelay]->flight_number[$flightNumber])) {
                            // Group Date if Found The Same Month (i.e. depDate = Feb 08, Already Have Feb 03, Then -> Feb 03, 08)
                            $foundSameMonth = false;
                            foreach ($flightsByDelayCode[$eachDelay]->flight_number[$flightNumber] as $dateMonth => $dayArray) {
                                if ($dateMonth == $departureDateMonth) {
                                    $flightsByDelayCode[$eachDelay]->flight_number[$flightNumber][$dateMonth][] = $departureDateDay;
                                    sort($flightsByDelayCode[$eachDelay]->flight_number[$flightNumber][$dateMonth]);
                                    $foundSameMonth = true;
                                    break;
                                }
                            }
                            if (!$foundSameMonth){
                                $flightsByDelayCode[$eachDelay]->flight_number[$flightNumber]["$departureDateMonth"][] = $departureDateDay;
                            }
                        }
                        else{
                            $flightsByDelayCode[$eachDelay]->flight_number[$flightNumber]["$departureDateMonth"] = [$departureDateDay];
                        }

                        $flightsByDelayCode[$eachDelay]->delay_reason = $delayReason[$k];
                    }
                }
            }

            return [ $flights, $flightsByCategory, $flightsByDelayCode];
        }
        return false;
    }


    /**
     * Combine Flight Config And Actual Load To Get New Seat Factors Array Of Objects
     * @param $dateSearch
     * @param $flightSearch
     * @param $period
     * @param bool|false $sort
     * @return array
     */
    public static function getSeatFactorArray($dateSearch, $flightSearch, $period, $sort = false){
        $seatFactors = [];
        foreach($dateSearch as $dateItem) {
            // Compute Flight Configuration
            $flightConfig = Flight::calculateFlightConfiguration($dateItem, $flightSearch, $period);

            // Compute Actual PAX Load
            $actualPaxLoad = Flight::calculateFlightActualPAXLoad($dateItem, $flightSearch, $period);
            // Combine Flight Config And Actual Load To Get New Seat Factors Array Of Objects

            foreach ($flightConfig as $j => $each){
                $date = $each->date;

                // For Default Settings Put Different Label
                if (isset($dateItem['label'])){
                    $date = $dateItem['label'];
                    // Not Sort If Default is Set
                    $sort = false;
                }

                if (!isset($seatFactors[$date])){
                    $seatFactors[$date] = new stdClass();
                }

                $seatFactors[$date]->config_c = $each->config_c;
                $seatFactors[$date]->config_y = $each->config_y;
                $seatFactors[$date]->config_flight_count = $each->flight_count;
                $seatFactors[$date]->sf = 0;
            }

//            debug($seatFactors);


            foreach ($actualPaxLoad as $j => $each){
                $date = $each->date;

                // For Default Settings Put Different Label
                if (isset($dateItem['label'])){
                    $date = $dateItem['label'];
                    // Not Sort If Default is Set
                    $sort = false;
                }

                if (!isset($seatFactors[$date])){
                    $seatFactors[$date] = new stdClass();
                }

                $pax_c = $each->pax_c;
                $pax_y = $each->pax_y;

                $seatFactors[$date]->pax_c = $pax_c;
                $seatFactors[$date]->pax_y = $pax_y;
                $seatFactors[$date]->pax_flight_count = $each->flight_count;

                if(isset($seatFactors[$date]->config_c) && isset($seatFactors[$date]->config_y) && $seatFactors[$date]->config_y){
                    $seatFactors[$date]->sf = ($pax_c + $pax_y) * 100 / ($seatFactors[$date]->config_c + $seatFactors[$date]->config_y);

                    if ($seatFactors[$date]->sf > 100){
                        $seatFactors[$date]->sf = 100;
                    }
                }
            }
        }
        if (count($seatFactors) && $sort){
            ksort($seatFactors);
        }
        return $seatFactors;
    }

    /**
     * @return \Illuminate\View\View
     */
    public function getCrewflights(){
        $cabincrew = User::join('users__departments', 'users.id', '=', 'users__departments.user_id')
            ->where('department_id', 12)->where('position_id', 3)->orderBy('last_name')->orderBy('first_name')->get();
        foreach ($cabincrew as $cc) {
            $fligths_total = array();
            for ($t = 1; $t < 10; $t++) {
                $flights = FlightCrew::selectRaw('count(flight_number) as counter, MONTH(std) as period')
                    ->addSelect('flight_number', 'arrival_airport', 'last_name', 'first_name')
                    ->join('flights', 'flights__crew.flight_id', '=', 'flights.id')
                    ->join('users', 'flights__crew.user_id', '=', 'users.id')
                    ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
                    ->whereIn('flight_number_id', [14, 39, 6, 21])
                    ->where('user_id', $cc->user_id)
                    ->where('is_standby', 0)
                    ->where('is_dhc', 0)
                    ->whereRaw("YEAR(std) = '2014' AND MONTH(std) = '$t'")
                    ->groupBy('period', 'flight_number')
                    ->get();

                if (!$flights->isEmpty()) {
                    $ff = new stdClass();
                    foreach ($flights as $i => $fitem) {
                        $ff->period = $fitem->period;
                        if ($i == 0) {
                            $ff->last_name = $fitem->last_name;
                            $ff->first_name = $fitem->first_name;
                        }
                        $ff->flight_number[] = $fitem->flight_number;
                        $ff->arrival_airport[] = $fitem->arrival_airport;
                        $ff->counter[] = $fitem->counter;
                    }
                    $fligths_total[$t] = $ff;
                }
            }
            if (!empty($fligths_total))
                $grand_total[] = $fligths_total;
        }

        return view('report.crewflights', ['flights_total' => $grand_total]);
    }

}
