<?php

namespace App\Http\Controllers;

use App\Classes\Transfer\TransferHandler;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Flight;
use App\Models\FlightNumber;
use App\Models\FlightSector;
use App\Models\PassengerFlight;
use App\Models\PassengerFlightAct;
use App\Models\StaffAirline;
use App\Models\StaffStation;
use Illuminate\Http\Request;

use App\Http\Requests;
use App\Http\Controllers\Controller;


class PassengerReportController extends Controller
{
    const MAX_EXECUTION_TIME = 120;

    protected $ssrList = [
        ASVC    => ASVC,
        BLND    => BLND,
        CKIN    => CKIN,
        DHC     => DHC,
        DPNA    => DPNA,
        INAD    => INAD,
        INFI    => INFI,
        INO     => INO,
        MAAS    => MAAS,
        MEDA    => MEDA,
        NITA    => NITA,
        PAX     => PAX,
        PLAT    => PLAT,
        SPBG    => SPBG,
        SVC     => SVC,
        WCHC    => WCHC,
        WCHR    => WCHR,
        WCHS    => WCHS,
    ];

    protected $transferList = [
        'ob' => "Outbound",
        'ib' => "Inbound",
    ];

    /**
     * PASSENGER REPORT
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '256M');

        // Get Staff Handling AP and Airlines
        list($handlingAirlines, $handlingAirports) = getStaffHandlingAirports();

        $selectedAirports = $request->has("airport") ? $request->get("airport") : array_keys($handlingAirports);
        $selectedAirlines = $request->has("airline") ? $request->get("airline") : array_keys($handlingAirlines);

        if (count($request->all())) {

            $flights = Flight::FlightsRangeConditional($selectedAirports, $selectedAirlines, $request->get("type"));

            if ($flights && count($flights)) {

                $flightIDs = $flights->pluck("id")->all();

                $paxPlanned = PassengerFlight::getFlightsPassengers($flightIDs, $request->get("first_name"), $request->get("last_name"));

                $paxAct = PassengerFlightAct::getFlightsActPassengers($flightIDs, $request->get("first_name"), $request->get("last_name"));

                PassengerFlightAct::combinePassengersPlannedAndAct($paxAct, $paxPlanned);

                $transferList = PassengerFlightAct::transferPassengers($paxAct, $request->get("transfer"));

                $readyList = PassengerFlightAct::ssrPassengers($transferList, $request->get("ssr"));
            }
        }

        $this->viewData = [
            'dateYesterday'     => date('Y-m-d',strtotime(" - 1 days")),
            "paxAct"            => isset($readyList) ? $readyList : [],
            "airlines"          => $handlingAirlines,
            "transferList"      => $this->transferList,
            "ssrList"           => $this->ssrList,
            "airports"          => $handlingAirports,
            "flightNumbers"     => FlightNumber::getArrayFlightNumbersWithSector(false, false, false, null, true),
            'sectors'           => FlightSector::listSectors(),
            'sectorsByAirlines' => FlightSector::getSectorsByAirlines(),
            'monthNames'        => getMonthsArray(),
            'years'             => getYearsArray(),
        ];

        return view("passenger-report.index", $this->viewData);
    }


    public function transfer(Request $request)
    {
        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '256M');

        // Get Staff Handling AP and Airlines
        list($handlingAirlines, $handlingAirports) = getStaffHandlingAirports();

        $selectedAirports = $request->has("airport") ? $request->get("airport") : array_keys($handlingAirports);
        $selectedAirlines = $request->has("airline") ? $request->get("airline") : array_keys($handlingAirlines);

        if (count($request->all())) {

            $flights = Flight::FlightsRangeConditional($selectedAirports, $selectedAirlines, $request->get("type"));

            if ($flights && count($flights)) {

                $flightIDs = $flights->pluck("id")->all();

                $paxPlanned = PassengerFlight::getFlightsPassengers($flightIDs);

                $paxAct = PassengerFlightAct::getFlightsActPassengers($flightIDs);

                PassengerFlightAct::combinePassengersPlannedAndAct($paxAct, $paxPlanned, true);

                $transferPAX = PassengerFlightAct::transferPassengers($paxAct, $request->get("transfer"));

                $summary = $this->summary($transferPAX, $request->get("transfer"));
            }
        }

        $this->viewData = [
            'dateYesterday'     => date('Y-m-d',strtotime(" - 1 days")),
            "transferPAX"       => isset($transferPAX) ? $transferPAX : [],
            "summary"           => isset($summary) ? $summary : [],
            "airlines"          => $handlingAirlines,
            "transferList"      => $this->transferList,
            "airports"          => $handlingAirports,
            "flightNumbers"     => FlightNumber::getArrayFlightNumbersWithSector(false, false, false, null, true),
            'sectors'           => FlightSector::listSectors(),
            'sectorsByAirlines' => FlightSector::getSectorsByAirlines(),
            'monthNames'        => getMonthsArray(),
            'years'             => getYearsArray(),
        ];

        return view("passenger-report.transfer", $this->viewData);
    }

    public function summary($transferPAX, $transfer){
        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '256M');

        if (is_array($transfer)) {
            $onlyInbound = in_array("ib", $transfer);
            $onlyOutbound = in_array("ob", $transfer);
        }
        else {
            $onlyInbound = $transfer == "ib";
            $onlyOutbound = $transfer == "ob";
        }

        $handler = new TransferHandler();

        foreach ($transferPAX as $each) {
            $combinedInfo = $each->combined_info;

            $flight = $each->flight;
            $flightNumber = $flight ? $flight->flightNumber : null;
            $airline = $flightNumber ? $flightNumber->airline : null;

            $flightNumberIATA = $flightNumber ? ($airline ? $airline->iata : "N/A").$flightNumber->flight_number : "N/A";

            $depAirport = $flightNumber ? $flightNumber->departureAirport : null;
            $arrAirport = $flightNumber ? $flightNumber->arrivalAirport : null;

            $depDate = getFlightDepartureDate($flight, true);

            if ($onlyInbound){
                $origin = isset($combinedInfo['inbound']['origin']) ? $combinedInfo['inbound']['origin']    : "";
                $rbd    = isset($combinedInfo['inbound']['rbd'])    ? $combinedInfo['inbound']['rbd']       : "";
                $inboundFN    = isset($combinedInfo['inbound']['airline'])          ? $combinedInfo['inbound']['airline']  : "N/A";
                $inboundFN    .= isset($combinedInfo['inbound']['flight_number'])    ? ltrim($combinedInfo['inbound']['flight_number'], "0")       : "";

                $handler->AddInbound($depDate, $flightNumberIATA, ($depAirport ? $depAirport->iata : ""), ($arrAirport ? $arrAirport->iata : ""), $inboundFN, $origin, $rbd);
            }

            if ($onlyOutbound){
                $destination = isset($combinedInfo['outbound']['destination'])  ? $combinedInfo['outbound']['destination']  : "";
                $rbd         = isset($combinedInfo['outbound']['rbd'])          ? $combinedInfo['outbound']['rbd']          : "";
                $outboundFN    = isset($combinedInfo['outbound']['airline'])            ? $combinedInfo['outbound']['airline']  : "N/A";
                $outboundFN    .= isset($combinedInfo['outbound']['flight_number'])     ? ltrim($combinedInfo['outbound']['flight_number'], "0")       : "";

                $handler->AddOutbound($depDate, $flightNumberIATA, ($depAirport ? $depAirport->iata : ""), ($arrAirport ? $arrAirport->iata : ""), $outboundFN, $destination, $rbd);
            }
        }

        $handler->sortAll();

        return $handler->getTransferList();

    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}
