<?php namespace App\Http\Controllers;



use App\Classes\Image\CropAvatar;
use App\Classes\Messages\Message;
use App\Models\Country;
use App\Models\Flight;
use App\Models\FlightNumber;
use App\Models\PassengerFlight;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

use App\Models\Passenger;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\URL;

class PassengerController extends Controller
{
    const MAX_EXECUTION_TIME = 60;

    public function show($passenger)
    {
        if (!$passenger || !isAnyAdmin()) {
            return redirect()->to('error/page-not-found');
        }

        if ($passenger->picture){
            $picture = asset("/assets/img/passengers/" . $passenger->picture);
        }

        if ($passenger->passengerLanguages){
            foreach ($passenger->passengerLanguages as $each) {
                $languages[]    = $each->language->language;
                $languageIds[]  = $each->language->id;
            }
        }

        $this->viewData = [
            'passenger'     => $passenger,
            'picture'       => isset($picture)      ? $picture      : asset("/assets/img/profile_picture.png"),
            'languages'     => isset($languages)    ? $languages    : [],
            'languageIds'   => isset($languageIds)  ? $languageIds  : []
        ];

        return view('passenger.index', $this->viewData);
    }

    public function paxList(Request $request)
    {
        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '128M');

        $flight = $flightNumber = null;
        if ($request->get("flight_id")){
            $flight = Flight::find($request->get("flight_id"));
        }

        if ($request->get("flight_number")){
            $flightNumber = FlightNumber::find($request->get("flight_number"));
        }

        // Search Passenger Flights And Total
        list($tableData, $pagination, $totalMode, $total) = PassengerFlight::searchPassengerFlights($request->all(), FALSE, FALSE, FALSE);

        $data = $this->preparePaxData($tableData);

        debug($data);

        $totalRevenue = 0;
        foreach ($data["revenue"] as $each) {
            $totalRevenue += ($each->fare ? $each->fare : 0);
        }

        $this->viewData = [
            'flight'            => $flight,
            'flightNumber'      => $flightNumber,
            'departureDate'     => $request->get("departure_from"),

            // Passenger Flights
            'data'              => $data,
            'totalPax'          => count($data["revenue"]) + count($data["child"]) + count($data["infant"]) + count($data["non_revenue"]),
            'revenuePax'        => count($data["revenue"]),
            'totalRevenue'      => $totalRevenue,
            // Total
            'total'             => $total,
            // Pagination (Enable)
            'pagination'        => $pagination,
            // Records Per Page
            'recordsPerPage'    => isset($inputs['recordsPerPage']) ? $inputs['recordsPerPage'] : 1000,
            // Current Page
            'page'              => isset($inputs['page']) ? $inputs['page'] : 1,
        ];

        return view('passenger.pax', $this->viewData);
    }

    function preparePaxData($tableData){
        $data = [
            "child50"       => [],
            "child"         => [],
            "infant"        => [],
            "revenue"       => [],
            "non_revenue"   => [],
            "refunded"      => [],
        ];

        foreach ($tableData as $each) {
            $status = $each->flightOrderStatus ? $each->flightOrderStatus->order_status : null;
            $fareBasis = $each->flightFareBasis ? $each->flightFareBasis->fare_basis : null;

            // Returns
            if ($status == "R"){
                $data["refunded"][] = $each;
                continue;
            }

            $processed = false;
            if ($fareBasis){
                // ID-staff tickets
                if (contains($fareBasis, "ID")){
                    $data["non_revenue"][] = $each;
                    $processed = true;
                }
                else {
                    if ($each->fare){
                        if (contains($fareBasis, ["CH50", "CN50"])){
                            $data["child50"][] = $each;
                        }
                    }
                    // 0-fare tickets
                    else {
                        if (contains($fareBasis, ["CH00", "CN00"])){
                            $data["child"][] = $each;
                        }
                        elseif (contains($fareBasis, ["IN"])){
                            $data["infant"][] = $each;
                        }
                        else {
                            $data["non_revenue"][] = $each;
                        }
                        $processed = true;
                    }
                }
            }

            if ($processed){
                continue;
            }

            // if fare include to
            if ($each->fare){
                $data["revenue"][] = $each;
            }
            else {
                $data["non_revenue"][] = $each;
            }
        }

        debug($data);

        return $data;
    }

    /**
     * Upload Picture
     * @param $id
     * @return bool|string
     */
    public function uploadPhoto($id){

        $passenger = $this->passenger->find($id);

        if (!$passenger){
            return response()->isServerError();
        }

        $crop = new CropAvatar($_POST['avatar_src'], $_POST['avatar_data'], $_FILES['avatar_file'], $passenger, 'passengers' );

        $response = array(
            'state'   => 200,
            'message' => $crop -> getMsg(),
            'result'  => asset($crop -> getResult())
        );

        return  json_encode($response);
    }

    /**
     * Delete Picture
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deletePhoto($id){

        $passenger = $this->passenger->find($id);

        $picture = public_path("/assets/img/passengers/{$passenger->picture}");
        $thumb = public_path("/assets/img/passengers/{$passenger->thumb}");

        File::delete($picture);
        File::delete($thumb);

        $this->passenger->update($passenger, [
            'picture'    => null,
            'thumb'     => null
        ]);

        \Flash::success("Profile Photo Successfully Deleted");

        return redirect()->back();
    }


    public function flight($departure_date = NULL, $flight_number = NULL, $sales_date = NULL)
    {
        $passengers = Passenger::getFlightPassengersBySalesDate($departure_date, $flight_number, $sales_date);
        /* $codes = AgencyCode::getCodeAndAgency($passengers->pluck('agency_code_id'));*/
        debug($passengers);
        return view('passenger.flight', array(
            'date'          => $departure_date,
            'flight_number' => $flight_number,
            'passengers'    => $passengers,
            'sales_date'    => $sales_date
        ));
    }

    /**
     * Get AJAX PNL list
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxGetPnlList()
    {
        $stdDate        = date('Y-m-d', strtotime(\request("std")));
        $flightId       = (int)\request("flight_id");
        $flightNumberId = (int)\request("flight_number_id");
        debug(request()->all());
        if (!$flightId || !$flightNumberId)
        {
            return response()->json([
                "error"   => TRUE,
                "message" => "Not found required parameters, please check you requested data",
                "title"   => "Validation Errors:"
            ]);
        }
        $passengers = Passenger::select([
            "passengers.id",
            "passengers.first_name",
            "passengers.last_name",
            "passengers__flights.flight_id",
            "passengers__flights.flight_number_id",
            "passengers__flights.passenger_order_id",
            "passengers__flights.coupon",
            "passengers__orders.agency_code_id",
            "passengers__orders.sales_date",
            "passengers__orders.ticket_number",
            "passengers__orders.original_fare",
            "passengers__orders.fare",
            "flights__fare_basis.fare_basis",
            "flights__rbd.rbd",
            "passengers__orders.payment_form"
        ])
            ->join("passengers__flights", "passengers__flights.passenger_id", "=", "passengers.id")
            ->join("passengers__orders", "passengers__flights.passenger_order_id", "=", "passengers__orders.id")
            ->leftJoin("flights__fare_basis", "passengers__flights.flight_fare_basis_id", "=", "flights__fare_basis.id")
            ->leftJoin("flights__rbd", "passengers__flights.flight_rbd_id", "=", "flights__rbd.id")
            ->whereNull("passengers__flights.deleted_at")
            ->where(function($query) use ($flightId, $stdDate, $flightNumberId){
                $query->where("passengers__flights.flight_id", $flightId)
                        ->orWhere(function($sql) use ($stdDate, $flightNumberId){
                            $sql->where(DB::raw("DATE(s_passengers__flights.departure)"), $stdDate)
                                ->where("passengers__flights.flight_number_id", "=", $flightNumberId);
                        });
                })
            ->get();

        return response()->json([
            "success"    => TRUE,
            "passengers" => $passengers
        ]);
    }

    /**
     * Get AJAX PL list
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxGetPlList(Request $request)
    {
        $type = $request->get("type") ? $request->get("type") : PNL;

        $stdDate        = date('Y-m-d', strtotime(\request("std")));
        $flightId       = (int)\request("flight_id");
        $flightNumberId = (int)\request("flight_number_id");

        $result = Passenger::getPLList($flightId, $flightNumberId, $stdDate, $type);

        $plData = Message::getMessageBody($type, $flightId);

        return response()->json([
            "success"       => TRUE,
            "pl_data"       => $plData,
            "pl"            => $result["pl"],
            "pl_inf"        => $result["plInf"],
            "pl_outbound"   => $result["plOutbound"],
            "pl_inbound"    => $result["plInbound"],
            "pl_details"    => $result["plDetails"],
            "pl_xbag"       => $result["plXBag"],
            "pl_docs"       => $result["plDocs"],
        ]);
    }

    protected static function period_to_int($period){
        switch ($period) {
            case "Yearly":
                $periodInt = 0;
                break;
            case "Quarterly":
                $periodInt = 1;
                break;
            case "Monthly":
                $periodInt = 2;
                break;
            case "Weekly":
                $periodInt = 3;
                break;
            case "Weekdays":
                $periodInt = 4;
                break;
            case "Daily":
                $periodInt = 5;
                break;
            case "Flight":
                $periodInt = 6;
                break;
            default:
                $periodInt = -1;
        }

        return $periodInt;
    }


}
