<?php namespace App\Http\Controllers;

use App\Http\Requests;
use App\Http\Controllers\Controller;

use App\Models\Table;
use App\Repositories\Interfaces\IDepartmentRepository;
use Illuminate\Http\Request;


class OrganizationLevelController extends Controller {


    /**
     * Display a listing of the resource.
     * @param IDepartmentRepository $department
     * @return \Illuminate\View\View
     */
    public function index(IDepartmentRepository $department)
    {
        // Departments List
        $departments = $department->listModelVariable(['name', 'id'], TRUE);

        // If Form Submit
        $departmentId = \request("structure_department_id");

        $organizationLevels = [];
        if ($departmentId) {
            $organizationLevels = $this->organizationLevel->getByCondition(['structure_department_id' => $departmentId], 'level');

            foreach ($organizationLevels as &$each) {
                $tables = $this->organizationLevel->getRelatedTableNames($each->id, TRUE);
                $each->related_tables = count($tables) ? implode(', ', $tables) : '';
            }
        }

        $this->viewData = [
            'departmentId'       => $departmentId,
            'departments'        => $departments,
            'organizationLevels' => $organizationLevels
        ];

        return view('organization-level/index', $this->viewData);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        // Get Department Id
        $departmentId = \request("structure_department_id");


        // Get Related Table And Capitalize Table Names
        $related_tables = Table::pluck('name', 'id')->all();
        array_walk($related_tables, function(&$each){
            $each = ucwords($each);
        });

        $this->viewData = [
            'departmentId'    => $departmentId,
            'related_tables'  => $related_tables,
        ];

        return view('organization-level/create', $this->viewData);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return Response
     */
    public function store()
    {
        debug(request()->all());
        $inputs = request()->all();

        // Related Table
        $related_tables = isset($inputs['related_tables']) ? $inputs['related_tables'] : FALSE;

        // Remove Unnecessary Fields
        removeUnnecessaryField($inputs, ['_token', 'insert', 'related_tables']);

        // Get Department Id
        $departmentId = $inputs['structure_department_id'];

        try {
            $object = $this->organizationLevel->createByAttributes($inputs);
        }
        catch(\Exception $e){
            flash()->error('Error! Please Make Sure Duplication Not Occurred.');
            return redirect()->back()
                             ->withInput();
        }

        // Create Many Level Details
        if ($related_tables) {
            $this->organizationLevelDetails->firstOrCreateMany(['organization_level_id' => $object->id], 'table_id', $related_tables);
        }

        return redirect()->to("organization-level?structure_department_id={$departmentId}")
            ->with('message', 'Success! New Level Successfully Inserted.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     * @param $id
     * @return int
     */
    public function edit($id)
    {

        // Get Department Id
        $departmentId = \request("structure_department_id");

        // Get Related Table And Capitalize Table Names
        $related_tables = Table::pluck('name', 'id')->all();
        array_walk($related_tables, function(&$each){
            $each = ucwords($each);
        });

        $organizationLevel = $this->organizationLevel->find($id);

        $selected_tables = $this->organizationLevelDetails->findAndListModelVariable(['organization_level_id' => $id], ['table_id']);

        $this->viewData = [
            'departmentId'      => $departmentId,
            'organizationLevel' => $organizationLevel,
            'selected_tables'   => $selected_tables,
            'related_tables'    => $related_tables
        ];

        return view('organization-level/edit', $this->viewData);
    }

    /**
     * Update the specified resource in storage.
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update($id)
    {
        $inputs = request()->all();

        // Related Table
        $related_tables = isset($inputs['related_tables']) ? $inputs['related_tables'] : FALSE;

        // Multiple Selection
        $inputs['multiple_selection'] = isset($inputs['multiple_selection']) ? $inputs['multiple_selection'] : false;

        // Remove unnecessary fields
        removeUnnecessaryField($inputs, ['_token', 'save', 'related_tables']);

        // Get Department Id
        $departmentId = $inputs['structure_department_id'];

        try {
            $this->organizationLevel->updateById($id, $inputs);
        }
        catch (\ErrorException $e){
            return redirect()->back();
        }

        //OrganizationLevelDetails::where('organization_level_id', $id)->delete();
        // Delete
        $this->organizationLevelDetails->deleteByAttributes(['organization_level_id' => $id]);

        if ($related_tables) {
            $related_tables = is_array($related_tables) ? $related_tables : [$related_tables];
            // Create Many Level Details
            foreach ($related_tables as $each) {
                $this->organizationLevelDetails->createByAttributes(['organization_level_id' => $id, 'table_id' => $each]);

            }
        }

        return redirect()->to("organization-level?structure_department_id={$departmentId}");
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return Response
     */
    public function delete($id)
    {
        $departmentId = \request("structure_department_id");

        // Delete Level
        $this->organizationLevel->deleteByAttributes(['id' => $id]);
        // Delete All Related Level Details
        $this->organizationLevelDetails->deleteByAttributes(['organization_level_id' => $id]);

        return redirect()->to("organization-level?structure_department_id={$departmentId}");
    }
}
