<?php namespace App\Http\Controllers;

use App\Http\Requests;
use App\Models\Organization;
use App\Models\Table;
use App\Repositories\Interfaces\IDepartmentRepository;
use App\Repositories\Interfaces\IOrganizationDetailsRepository;
use App\Repositories\Interfaces\IOrganizationLevelDetailsRepository;
use App\Repositories\Interfaces\IOrganizationLevelRepository;
use App\Repositories\Interfaces\IOrganizationRepository;
use App\Repositories\Interfaces\IPositionRepository;
use App\Repositories\Interfaces\ITableRepository;
use App\Repositories\Interfaces\IUserDepartmentRepository;
use App\Repositories\Interfaces\IUserRepository;

use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\URL;
use stdClass;

class OrganizationController extends Controller {


    public function __construct(IOrganizationRepository $organization,
                                IOrganizationDetailsRepository $organizationDetails,
                                IOrganizationLevelRepository $organizationLevel,
                                IOrganizationLevelDetailsRepository $organizationLevelDetails,
                                IUserRepository $user,
                                ITableRepository $table,
                                IPositionRepository $position,
                                IDepartmentRepository $department,
                                IUserDepartmentRepository $userDepartment)
    {
        $this->organization = $organization;
        $this->organizationDetails = $organizationDetails;
        $this->organizationLevel = $organizationLevel;
        $this->organizationLevelDetails = $organizationLevelDetails;
        $this->user = $user;
        $this->table = $table;
        $this->position = $position;
        $this->department = $department;
        $this->userDepartment = $userDepartment;
    }


    public function getValueFromTableId($tableId, $tableValueId){
        $name = $this->table->getAttributeById($tableId, 'name');
        $valueObject = DB::table($name)->find($tableValueId);
        $value = null;
        if ($valueObject) {
            switch ($name) {
                case 'sales__channel':
                    $value = $valueObject->channel;
                    break;
                case 'divisions':
                    $value = $valueObject->division;
                    break;
                case 'regions':
                    $value = $valueObject->region;
                    break;
                case 'countries':
                    $value = $valueObject->country;
                    break;
                case 'cities':
                    $value = $valueObject->city;
                    break;
                case 'agency':
                    $value = $valueObject->agency;
                    break;
                case 'passengers':
                    $value = $valueObject->first_name . " " . $valueObject->last_name;
                    break;
            }
        }
        return $value;
    }

    public function listTableValues($tableIds){
        if ($tableIds) {
            $list = [];
            if (is_array($tableIds)) {
                if (count($tableIds)) {
                    foreach ($tableIds as $id) {
                        $tableName = $this->table->getAttributeById($id, 'name');
                        $table = DB::table($tableName);
                        $list[] = $this->tables($table, $tableName);
                    }
                }
            }
            else {
                $tableName = $this->table->getAttributeById($tableIds, 'name');
                $table = DB::table($tableName);
                $list = $this->tables($table, $tableName);
                debug($table);
            }
            return $list;
        }
        return false;
    }


    public function tables($table, $tableName){
        switch ($tableName) {
            case 'divisions':
                $list = [
                    'values'        => $table->pluck('division', 'id'),//->all(),
                    'table_name'    => $tableName,
                    'label'         => 'Division'
                ];
                break;
            case 'regions':
                $list = [
                    'values'        => $table->pluck('region', 'id'),//->all(),
                    'table_name'    => $tableName,
                    'label'         => 'Region'
                ];
                break;
            case 'countries':
                $list = [
                    'values'        => $table->pluck('country', 'id'),//->all(),
                    'table_name'    => $tableName,
                    'label'         => 'Country'
                ];
                break;
            case 'cities':
                $list = [
                    'values'        => $table->pluck('city', 'id'),//->all(),
                    'table_name'    => $tableName,
                    'label'         => 'City'
                ];
                break;
            case 'agency':
                $list = [
                    'values'        => $table->pluck('agency', 'id'),//->all(),
                    'table_name'    => $tableName,
                    'label'         => 'Agency'
                ];
                break;
            case 'passengers':
                $list = [
                    'values'        => $table->list('CONCAT(first_name," ",last_name)', 'id'),//->all(),
                    'table_name'    => $tableName,
                    'label'         => 'Passenger'
                ];
                break;
            default:
                $list = [];
        }
        return $list;
    }


    /**
     * Display a listing of the resource.
     * @param IDepartmentRepository $department
     * @return \Illuminate\View\View
     */
    public function index(IDepartmentRepository $department)
    {
        $key['not'] = 'null';
        debug(array_keys($key));

        // Departments List
        $departments = $department->listModelVariable(['name', 'id'], TRUE);

        // If Form Submit
        $departmentId = \request("structure_department_id");

        if ($departmentId) {
            $organization = $this->organization->listOrganization(['organization.structure_department_id' => $departmentId], false, ['level']);
            debug($organization);
            // Not Assigned
            $organizationNotAssigned = $this->organization->listOrganization(['organization_level_id' => null, 'organization.structure_department_id' => $departmentId], 'left');
            // Get All User Organization Info For Index Page
            $this->getUserOrganizationInfo($organization);
            // Get Organization User Info
            $this->getUserOrganizationInfo($organizationNotAssigned);
        }


        $this->viewData = [
            'departmentId'              => $departmentId,
            'departments'               => $departments,
            'organization'              => isset($organization) ? $organization : [],
            'organizationNotAssigned'   => isset($organizationNotAssigned) ? $organizationNotAssigned : []
        ];

        return view('organization/index', $this->viewData);
    }


    public function getUserOrganizationInfo(&$organization){
        foreach ($organization as &$each) {
            debug($each);

            $details = $each->details;
            $each->first_name = $each->user ? $each->user->first_name : "";
            $each->last_name = $each->user ? $each->user->last_name : "";
            $each->thumb = $each->user ? $each->user->thumb : "";
            $each->level = $each->organizationLevel ? $each->organizationLevel->level : 'Not Assigned';
            $each->position = $each->position ? $each->position->name : '';
            $each->report_to = $each->reportToUser ? $each->reportToUser->first_name." ".$each->reportToUser->last_name : '';
            $tableInfo = [];
            foreach ($details as $detail) {
                // Get Value Out Of TableId
                $tableName = $this->table->getAttributeById($detail->table_id, 'name');
                $valueName = $this->getValueFromTableId($detail->table_id, $detail->table_value_id);
                $tableInfo[$tableName][] = $valueName;
            }
            $each->tableInfo = $tableInfo;
        }
    }

    public function create(IPositionRepository $position, IDepartmentRepository $deparment){

        $departmentId = \request("structure_department_id");

        // Get Related Table And Capitalize Table Names
        $related_tables = Table::pluck('name', 'id')->all();
        array_walk($related_tables, function(&$each){
            $each = ucwords($each);
        });

        $this->viewData = [
            'departmentId'    => $departmentId,
            'departments'     => $deparment->findAndListModelVariable(['id' => $departmentId],['name','id'], TRUE),
            'users'           => $this->organization->listNotAssignedUsers(['department_id' => $departmentId], TRUE),
            'levels'          => $this->organizationLevel->listModelVariable([ ['level','name'], 'id', '-'], true),
            'employees'       => $this->organization->findAndListModelVariable(['structure_department_id' => $departmentId], ['user_id','id']),
            'related_tables'  => $related_tables,
            'positions'       => $position->findAndListModelVariable(['department_id' => $departmentId],['name','id'], true),
        ];

        return view('organization/create', $this->viewData);
    }

    /**
     * Store a newly created resource in storage.
     * @param IPositionRepository $position
     * @param IUserDepartmentRepository $userDepartment
     * @return \Illuminate\Http\RedirectResponse
     */

    public function store(IPositionRepository $position, IUserDepartmentRepository $userDepartment)
    {
        $inputs = request()->all();

        // Check if Table Inputs Exists
        $tableNameInputs = $this->getTableInputs($inputs, TRUE);

        // Check If New Position Option Is Set
        if (isset($inputs['new_position']) && $inputs['new_position']){
            $newPosition = $position->firstOrCreate(['department_id' => 7, 'name' => $inputs['new_position'] ]);
            $inputs['structure_position_id'] = $newPosition->id;
        }

        // Check If New User Option Is Set
        if (isset($inputs['new_user']) && isset($inputs['first_name']) && isset($inputs['last_name'])){

            $first_name = $inputs['first_name'];
            $last_name = $inputs['last_name'];
            $email = $inputs['email'] ? $inputs['email'] : null;

            $user = $this->user->firstOrCreate([
                'first_name'    => $first_name,
                'last_name'     => $last_name
            ]);

            $inputs['user_id'] = $user->id;

            if ($email)
                $this->user->update($user, ['email' => $email]);

            $userDepartment->firstOrCreate([
                'user_id'           => $user->id,
                'department_id'     => $inputs['structure_department_id'],
                'position_id'       => $inputs['structure_position_id'],
            ]);

            removeUnnecessaryField($inputs, ['first_name', 'last_name', 'email', 'new_user']);
        }

        // Get Department Id
        $departmentId = $inputs['structure_department_id'];

        // Check if user Reports to CEO
        $reportsToCeo = \request("reports_to_ceo");

        // Remove Unnecessary Fields
        removeUnnecessaryField($inputs, ['_token', 'new_position', 'insert', 'report_to_level', 'reports_to_ceo']);

        // Remove Blank Fields
        removeBlankAttributes($inputs);

        // Check If Such User Exists, otherwise Create
        $object = $this->organization->firstOrCreate(['user_id' => $inputs['user_id']]);

        // Update Remaining Fields
        $this->organization->update($object, $inputs);

        // If Reports To CEO
        if ($reportsToCeo){
            $ceoUserId = $this->user->getCeoUserId($this->userDepartment);
            $this->organization->update($object, ['report_to_user_id' => $ceoUserId]);
        }

        // Create Many Level Details
        if (count($tableNameInputs)) {
            foreach ($tableNameInputs as $id => $values) {
                $values = is_array($values) ? $values : [$values];
                foreach ($values as $item) {
                    $details = $this->organizationDetails->firstOrCreate([
                        'organization_id' => $object->id,
                        'table_id'        => $id,
                        'table_value_id'  => $item
                    ]);
                    $ids[] = $details->id;
                }

            }
            // Delete Other Records Not IN Ids List
            $this->organizationDetails->deleteNotIn(['organization_id' => $object->id], $ids);
        }

        return redirect()->to("organization?structure_department_id={$departmentId}")
            ->with('message', 'Success!');
    }


    /**
     * @param $organization
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit($organization)
    {
        $departmentId = $organization->structure_department_id;

        $organizationLevel = $organization->organizationLevel;

        $related_tables = [];
        if ($organizationLevel) {
            // Get Department

            foreach ($organizationLevel->details as $details) {
                $related_table_id = $details->table_id;
                if ($related_table_id) {
                    $selected_table_value_ids = $this->organizationDetails->findAllByAttributes(['organization_id' => $organization->id, 'table_id' => $related_table_id]);
                    $selected = [];
                    if (count($selected_table_value_ids)) {
                        foreach ($selected_table_value_ids as $each) {
                            $selected[] = $each->table_value_id;
                        }
                    }
                    $related_tables[] = [
                        'data'      => $this->listTableValues($related_table_id),
                        'selected'  => $selected_table_value_ids ? $selected : [],
                        'multiple'  => $organizationLevel->multiple_selection
                    ];
                }
            }
        }

        $reportUser = $this->organization->findByAttributes(['user_id' => $organization->report_to_user_id]);

        $report_to_users = [];
        if ($reportUser && $reportUser->organization_level_id) {
            $organizationReportTo = $this->organization->findAllByAttributes(['organization_level_id' => $reportUser->organization_level_id]);

            if ($organizationReportTo) {
                foreach ($organizationReportTo as $each) {
                    $user = $each->user;
                    $report_to_users[$user->id] = $user->first_name . " " . $user->last_name . '/' . ($each->position ? $each->position->name : '-') . '/';
                }
            }
        }

        $position = $this->position->find($organization->structure_position_id);
        if ($position) {
            $positions = $this->position->findAndListModelVariable(['department_id' => $position->department_id], ['name','id'], true);
            $departmentId = isset($departmentId) ? $departmentId : $position->department_id;
        }
        else
            $positions  = $this->position->listModelVariable(['name', 'id'], true);



        $this->viewData = [
            'departments'           => Department::listDepartments(),
            'departmentId'          => $departmentId,
            'users'                 => $this->user->listModelVariable([ ['first_name', 'last_name'], 'id', ' ']),
            'levels'                => $this->organizationLevel->findAndListModelVariable(['structure_department_id' => $departmentId],[ ['level','name'], 'id', '-'], true),
            'employees'             => $this->organization->listModelVariable(['user_id', 'id']),
            'positions'             => $positions,
            'organization'          => $organization,
            'report_to_level_id'    => $reportUser ? $reportUser->organization_level_id : null,
            'related_tables'        => $related_tables,
            'report_to_users'       => $report_to_users ? $report_to_users : []
        ];

        return view('organization/edit', $this->viewData);
    }

    /**
     * Update the specified resource in storage.
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update($id)
    {
        $inputs = request()->all();

        // Get Related Tables
        $related_tables = $this->getTableInputs($inputs, TRUE);

        // Check If New Position Option Is Set
        if (isset($inputs['new_position']) && $inputs['new_position']){
            $newPosition = $this->position->firstOrCreate(['department_id' => 7, 'name' => $inputs['new_position'] ]);
            $inputs['structure_position_id'] = $newPosition->id;
        }

        // Check if user Reports to CEO
        $reportsToCeo = \request("reports_to_ceo");

        // Remove Blank Attributes
        replaceBlankAttributes($inputs);

        // Get Department
        $departmentId = $inputs['structure_department_id'];

        // Remove unnecessary fields
        removeUnnecessaryField($inputs, ['_token', 'save', 'related_tables', 'report_to_level', 'new_position', 'reports_to_ceo']);


        // Update Report To User Ids Of Others
        $organization = $this->organization->find($id);
        if ($organization->user_id !== $inputs['user_id']){
            $this->organization->updateByAttributes(['report_to_user_id' => $organization->user_id], ['report_to_user_id' => $inputs['user_id']]);
        }

        $this->organization->updateById($id, $inputs);

        // If Reports To CEO
        if ($reportsToCeo){
            $ceoUserId = $this->user->getCeoUserId($this->userDepartment);
            $this->organization->updateById($id, ['report_to_user_id' => $ceoUserId]);
        }

        $ids = [];
        if ($related_tables) {
            $related_tables = is_array($related_tables) ? $related_tables : [$related_tables];
            // Create Many Level Details
            foreach ($related_tables as $tableId => $tableValueId) {
                $tableValueId = is_array($tableValueId) ? $tableValueId : [$tableValueId];
                foreach ($tableValueId as $itemId) {
                    $item = $this->organizationDetails->firstOrCreate(['organization_id' => $id, 'table_id' => $tableId, 'table_value_id' => $itemId]);
                    $ids[] = $item->id;
                }
            }
        }

        if (count($ids)){
            $this->organizationDetails->deleteNotIn(['organization_id' => $id], $ids);
        }
        else {
            $this->organizationDetails->deleteByAttributes(['organization_id' => $id]);
        }

        return redirect()->to('organization'."?structure_department_id={$departmentId}");
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return Response
     */
    public function delete($id)
    {

        // Update Report To User Ids Of Others
        $organization = $this->organization->find($id);

        $departmentId = $organization->structure_department_id;

        $this->organization->updateByAttributes(['report_to_user_id' => $organization->user_id], ['report_to_user_id' => null]);

        // Delete Level
        $this->organization->deleteByAttributes(['id' => $id]);
        // Delete All Related Level Details
        $this->organizationDetails->deleteByAttributes(['organization_id' => $id]);

        flash()->success("Successfully Deleted");

        return redirect()->to('organization'."?structure_department_id={$departmentId}");
    }


    /**
     * Organization Chart
     * @return \Illuminate\View\View
     */
    public function chart(){

        $organization = $this->organization->all();

        $this->viewData = [
            'organization' => $organization
        ];

        return view('organization/chart', $this->viewData);
    }





    /**
     * Show Individual Organization CHart Based On Given Organization Id
     * @param $id
     * @return \Illuminate\View\View
     */
    public function chartOrganization($organization){

        $organization = $this->organization->getAllReportToEmployees($organization->id);

        $total = [];
        array_walk_recursive($organization, function(&$each) use (&$total){
            if ($each)
                $total[] = $each;
        });

        debug($total);

        $this->viewData = [
            'personal'     => true,
            'organization' => $total
        ];

        return view('organization/chart', $this->viewData);
    }

    public function chartDepartment($department){

        $organization = $this->organization->listOrganization(['organization_level_id' => [WHERE_NOT_NULL], 'organization.structure_department_id' => $department->id], 'left', ['level']);

        $organizationNotAssigned = $this->organization->listOrganization(['organization_level_id' => null, 'organization.structure_department_id' => $department->id], 'left');

        $this->viewData = [
            'personal'                => false,
            'organization'            => $organization,
            'organizationNotAssigned' => $organizationNotAssigned,
        ];

        return view('organization/chart', $this->viewData);
    }


    /**
     * @param $department
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function newChartDepartment($department){

        $organization = $this->organization->listOrganization(['organization_level_id' => [WHERE_NOT_NULL], 'organization.structure_department_id' => $department->id], 'left', ['level']);

        $undefinedHierarchyOrganization = $this->getUndefinedHierarchy($organization);

        $organizationNotAssigned = $this->organization->listOrganization(['organization_level_id' => null, 'organization.structure_department_id' => $department->id], 'left');

        debug($organization);

        $this->viewData = [
            'personal'                  => false,
            'data'                      => Organization::getOrganizationJsonForChart($organization),
            'undefinedHierarchy'        => $undefinedHierarchyOrganization,
            'organizationNotAssigned'   => $organizationNotAssigned
        ];

        return view('organization/new-chart', $this->viewData);
    }


    /**
     * Subtract Undefined Hierarchy Organization People From Organization List
     * @param $organization
     * @return array
     */
    protected function getUndefinedHierarchy(&$organization){
        $undefinedHierarchy = [];
        $undefinedIds = [];
        foreach ($organization as $i => $each) {
            debug($each);

            if ($each->organizationLevel->level > 1 && !$each->report_to_user_id){

                // Get All Reportees
                $reportees = $this->organization->getAllReportToEmployees($each->id);

                // If This Person {item} Appears In The List -> Disable Him/Her
                $undefinedIds[] = $each->id;
                foreach ($reportees as $item) {
                    if ($item) {
                        // If This Person {item} Appears In The List -> Disable Him/Her
                        $undefinedIds[] = $item->id;
                    }
                }
            }
        }

        // Unique Values Only
        $undefinedIds = array_unique($undefinedIds);

        if (count($undefinedIds)) {
            foreach ($organization as $i => $each) {
                if (in_array($each->id, $undefinedIds)) {
                    $undefinedHierarchy[] = clone $each;
                    unset($organization[$i]);
                }
            }
        }

        return $undefinedHierarchy;
    }

    /**
     * @param $organization
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function newChartOrganization($organization){

        $organization = $this->organization->getAllReportToEmployees($organization->id);

        debug($organization);

        $total = [];
        if ($organization && count($organization)) {
            array_walk_recursive($organization, function (&$each) use (&$total) {
                if ($each)
                    $total[] = $each;
            });
        }

        $this->viewData = [
            'personal'      => true,
            'data'          => Organization::getOrganizationJsonForChart($total),
        ];

        return view('organization/new-chart', $this->viewData);
    }


    public function newChartOrganizationByUserId($user){

        $searchOrganization = $this->organization->findByAttributes(['user_id' => $user->id]);

        $total = [];
        if ($searchOrganization) {
            $organization = $this->organization->getAllReportToEmployees($searchOrganization->id);

            if ($organization && count($organization)) {
                array_walk_recursive($organization, function (&$each) use (&$total) {
                    if ($each)
                        $total[] = $each;
                });
            }
        }

        $this->viewData = [
            'personal'      => true,
            'data'          => Organization::getOrganizationJsonForChart($total),
        ];

        return view('organization/new-chart', $this->viewData);
    }

    /**
     * Order Organization List By Level
     * @param $organization
     * @return array
     */
    protected static function orderOrganizationByLevel($organization){
        $orderedOrganization = [];

        foreach ($organization as $each) {
            $level = $each->organizationLevel ? $each->organizationLevel->level : 'undefined';
            $data[$level][$each->id] = $each;
        }

        if (isset($data)) {
            foreach ($data as $level => $each) {
                foreach ($each as $item) {
                    $orderedOrganization[] = $item;
                }
            }
        }

        return $orderedOrganization;
    }


    protected static function debugOrganizationLevelAndUserId($organization){
        $print = [];
        foreach ($organization as $each) {
            $level = $each->organizationLevel ? $each->organizationLevel->level : 'undefined';
            $print[$level."->".$each->id] = $each->user_id;
        }
        debug($print);
    }



    /**
     * Search Inputs For Table Name Items
     * @param $inputs
     * @param bool $deleteMatches
     * @return array an array of found inputs
     */
    public function getTableInputs(&$inputs, $deleteMatches = FALSE){
        $table = [];
        $tableNames = $this->table->listModelVariable(['name', 'id']);
        foreach ($inputs as $key => $value) {
            if (in_array($key, $tableNames)){
                $i = array_keys($tableNames, $key);
                $table[$i[0]] = $value;
                if ($deleteMatches){
                    unset($inputs[$key]);
                }
            }
        }
        return $table;
    }



}
