<?php

namespace App\Http\Controllers;

use App\Models\Airline;
use App\Models\AirlineLibrary;
use App\Models\Department;
use App\Models\Library;
use App\Models\LibrarySign;
use App\Models\Location;
use App\Models\StaffAirline;
use App\Models\StaffStation;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;


class LibraryController extends Controller
{
    public function index()
    {
        $handlingAirlines = StaffAirline::currentStaffAirlinesList(LIBRARY);

        AirlineLibrary::uploadedAirlineLibraries($handlingAirlines);

        $library = [];
        for($i = 0; $i < 4; $i++) {
            $library[$i] = Library::with([
                "createdBy", "updatedBy",
//                "toLocation", "toDepartment", "toPosition"
            ])->orderBy("created_at", "DESC");
        }

        $this->viewData = [
            "readAndSignPermission" => checkIfModulePermissionIsEnabled(\LIBRARY, "read_and_sign"),
            "library"               => $library[0]->whereNull("deleted_at")->whereNull("department_id")->get(),
            "libraryReadAndSign"    => $library[1]->whereNull("deleted_at")->whereNotNull("read_and_sign")->get(),
            "categories"            => $library[2]->whereNull("deleted_at")->whereNotNull("category")->where("category", "!=", "")->pluck("category")->unique()->all(),

            "libraryDeleted"        => $library[3]->whereNotNull("deleted_at")->get(),
            "departments"           => Department::where("has_library", true)->whereNull("deleted_at")->get(),
            "departmentsLibrary"    => $this->getLibraryByDepartment(),
            "handlingAirlines"      => $handlingAirlines,
        ];

        return view("library/index", $this->viewData);
    }

    function getLibraryByDepartment(){
        $result =  Library::with([
            "createdBy",
            "updatedBy",
        ])
            ->whereNotNull("department_id")
            ->whereNull("deleted_at")
            ->orderBy("department_id")
            ->get();

        $libraries = [];
        foreach ($result as $each) {
            if (!isset($libraries[$each->department_id])){
                $libraries[$each->department_id] = [];
            }

            $libraries[$each->department_id][] = $each;
        }

        return $libraries;
    }

    public function create(Request $request)
    {
        $handlingAirports = StaffStation::currentStaffStationsList();

        $approvedLocations = Location::locationAirports(true, $handlingAirports);

        $this->viewData = [
            "categories"        => ["" => "Select"] + Library::whereNotNull("category")->where("category", "!=", "")->groupBy("category")->pluck("category", "category")->all(),
            "department"        => $request->get("department_id") ? Department::find($request->get("department_id")) : null,

            "departments"       => getDepartmentListByRole($this->authObject, true),
            "locations"         => $approvedLocations,
            "positions"         => [],
            "users"             => [],
        ];

        return view("library/create", $this->viewData);
    }

    public function store(Request $request)
    {
        // create
        $library = new Library();
        $library->created_by = Auth::user()->id;
        $library->save();

        // update
        $this->updateFields($library, $request);

        return redirect()->to("library");
    }


    public function edit(Library $library)
    {
        $handlingAirports = StaffStation::currentStaffStationsList();

        $approvedLocations = Location::locationAirports(true, $handlingAirports);

        $positions = $users = $deptIDs = $positionIDs = [];
        if ($lastSign = $library->lastSign){

            if (count($lastSign->departments)){
                $deptIDs = getLibrarySignDepartments($lastSign);
                $positions = getPositionsByDepartments($deptIDs);
            }

            $positionIDs = getLibrarySignPositions($lastSign);
            $locationIDs = getLibrarySignLocations($lastSign);
            $users = findUsersByDepartmentPositionLocation($locationIDs, $deptIDs, $positionIDs, null, true);
            $users = $users->pluck("name", "id")->all();
        }

        $this->viewData = [
            "library"    => $library,
            "categories" => ["" => "Select"] + Library::whereNotNull("category")->where("category", "!=", "")->groupBy("category")->pluck("category", "category")->all(),
            "department"    => \request("department_id") ? Department::find(\request("department_id")) : null,

//            "subdepartments"    => $subdepartments,
            "lastSign"          => $lastSign,
            "positions"         => $positions,
            "users"             => $users,
            "departments"       => getDepartmentList(),
            "locations"         => $approvedLocations,
        ];

        return view("library/edit", $this->viewData);
    }

    public function update(Request $request, Library $library)
    {
        // update
        $library->updated_by = Auth::user()->id;

        debug($request->all());

        // update
        $this->updateFields($library, $request);

        return redirect()->to("library?current_tab=lbr");
    }


    public function showAirline(Airline $airline)
    {
        $this->viewData = [
            "airline"   => $airline,
            "readAndSignPermission" => checkIfModulePermissionIsEnabled(AIRLINE_LIBRARY, "read_and_sign"),

            "libraryReadAndSign"    =>AirlineLibrary::where("airline_id", $airline->id)->whereNull("deleted_at")->whereNotNull("read_and_sign")->orderBy("created_at", "DESC")->get(),

            'library'	        => AirlineLibrary::where("airline_id", $airline->id)->whereNull("deleted_at")->orderBy("created_at", "DESC")->get(),
            'libraryDeleted'	=> AirlineLibrary::where("airline_id", $airline->id)->whereNotNull("deleted_at")->orderBy("created_at", "DESC")->get(),
            "categories" => AirlineLibrary::where("airline_id", $airline->id)->whereNotNull("category")->where("category", "!=", "")->groupBy("category")->pluck("category")->all(),
            'picture' 	 => $airline->picture ? asset("storage/airlines/img/{$airline->picture}") : asset("/assets/img/default-airline-picture.png"),

        ];

        return view("library/show-airline", $this->viewData);
    }

    public function delete(Library $library){

        if ($library){
            $library->deleted_at = date("Y-m-d H:i:s");
            $library->updated_by = Auth::user()->id;
            $library->save();
        }

        return redirect()->to("library?current_tab=lbr");
    }

    function updateFields(&$library, Request $request){
        $library->department_id = $request->get("department_id");
        $library->name = $request->get("name");
        $library->category = $request->get("new_category") ? $request->get("new_category") : $request->get("category");
        $library->abbr = $request->get("abbr");
        $library->type = $request->get("type");
        $library->issue_date = $request->get("issue_date") && $request->get("issue_date") != EMPTY_DATE_MASK ? $request->get("issue_date") : null;
        $library->effective_date = $request->get("effective_date") && $request->get("effective_date") != EMPTY_DATE_MASK ? $request->get("effective_date") : null;
        $library->edition_no = $request->get("edition_no");
        $library->details = $request->get("details");
//        $library->read_and_sign = $request->get("read_and_sign") ? date("Y-m-d H:i:s") : null;
        $library->save();

        try {
            LibrarySign::libraryCheckSignedDocuments($library, $request);
        }
        catch (\Exception $e){
            debug($e);
        }
    }


    public function showFile(Library $library)
    {
        $filename = $library->attachment_file;
        $url = storage_path("app/public/library/$filename");
        debug(File::exists($url));
        debug($url);

        if ($filename && File::exists($url)) {
            $mimeType = getMimeType($filename);
            $fileContents = File::get($url);

            if (\request("view")){

                // Check and set
                LibrarySign::libraryReadAndSignCheck($library, true);

                return response()->make($fileContents, 200,array('Content-Type' => $mimeType));
            }
            elseif (\request("download")) {
                // Check and set
                LibrarySign::libraryReadAndSignCheck($library, null, true);

                return response()->download($url, $filename, array('Content-Type' => $mimeType));
            }
        }

        App::abort(404);
    }

    public function storeFile(Request $request, Library $library){

        if (!$request->hasFile("file")){
            return redirect()->to("library");
        }

        $mimeType = $request->file('file')->getClientMimeType();
        $file =  ['file' => $request->file('file')];
        $rulesArray = ["application/pdf", "pdf", "image/jpeg", "image/jpg", "image/png", "image/gif",
                        "application/vnd.ms-excel", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
            "application/vnd.openxmlformats-officedocument.spreadsheetml.template",
            "application/vnd.ms-excel.sheet.macroEnabled.12",
            "application/vnd.ms-excel.template.macroEnabled.12",
            "application/vnd.ms-excel.addin.macroEnabled.12",
            "application/vnd.ms-excel.sheet.binary.macroEnabled.12"];


        if (!in_array($mimeType, $rulesArray)){
            return redirect()->to("library")
                ->withInput()
                ->withErrors(["'Current File's extension is {$mimeType}. \nFile has to be in PDF format"]);
        }
        else {
            $fileName = $library->id . '.' . $file['file']->getClientOriginalExtension();
            $storagePath = storage_path("app/public/library/");

            // Picture Save
            $file['file']->move($storagePath, $fileName);

            // Save in DB
            $library->attachment_file = $fileName;
            $library->save();

            // Set
            LibrarySign::libraryCheckSignedDocuments($library, $request, \LIBRARY, true);

            return redirect()->to("library");
        }
    }

    public function deleteFile(Library $library){
        $filename = $library->attachment_file;

        $storageUrl = storage_path("app/public/library/$filename");

        if (File::exists($storageUrl)){
            File::delete($storageUrl);
        }

        $library->attachment_file = null;
        $library->updated_by = Auth::user()->id;
        $library->save();

        return redirect()->to("library");
    }
}
