<?php

namespace App\Http\Controllers;

use App\Imports\Schedule;
use App\Jobs\UploadFlightScheduleJob;
use App\Models\Aircraft;
use App\Models\AircraftType;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Flight;
use App\Models\FlightNumber;
use App\Models\FlightSchedule;
use App\Models\FlightScheduleFlight;
use Carbon\Carbon;
use Illuminate\Http\Request;

use App\Http\Requests;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

use Maatwebsite\Excel\Collections\SheetCollection;
use Maatwebsite\Excel\Facades\Excel;

class HandlingFlightScheduleUploadController extends Controller
{
    const MAX_EXECUTION_TIME = 400;

    public function index(Request $request){

        // Only Considered Stations

        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '1024M');

        debug($request->all());

        if ($this->isPostRequest()) {

            if (\request()->hasFile('file')) {

                $localTimings = $request->get("timezone");

                $file = $request->file('file');
                // Load Excel File

                if (!in_array($file->getClientOriginalExtension(), ['xls', 'xlsx']))
                {
                    flash()->error('Wrong Format(' . $file->getClientOriginalExtension() . '). Please Make Sure The Format is XLS or XLSX.');
                    $this->viewData = [

                        'message_type'  => "danger",
                        'message'       => 'Wrong Format(' . $file->getClientOriginalExtension() . '). Please Make Sure The Format is XLS or XLSX.',
                    ];
                }
                else {

//                    try {
                    $loadObject = Excel::toCollection(new Schedule, $file);
//                    }
//                    catch (\Exception $e){
//                        debug($e->getMessage());
//                    }
                    if (!$loadObject || !count($loadObject)) {
                        flash()->error("File Is Empty. Found No Data To Upload!");

                        $this->viewData = [
                            'message_type' => "danger",
                            'message' => 'File Is Empty. Found No Data To Upload.',
                            "airports" => Airport::listHandlingStations(),
                            "airlines" => ["" => "Select"] + Airline::listHandlingAirlines(),
                            "iataCode" => airlineModule() && env(IATA_CODE) ? env(IATA_CODE) : "IB",
                        ];

                        return view("handling-flight-schedule/upload", $this->viewData);
                    }

                    $stations = Airport::getHandlingStations("iata");

                    $data = [];
                    foreach ($loadObject as $sheet){

                        $pageData = $this->GetScheduleArray($request->get("type"), $sheet, $stations, $localTimings);
//                        debug($pageData);
//                        return  1;
                        foreach ($pageData as $each) {
                            if ($each && count($each)) {
                                $data[] = $each;
                            }
                        }
                    }

                    /*
                    else {
                        try {
                            debug($loadObject);
                            // Get Schedule Array
                            $data = $this->GetScheduleArray($request->get("type"), $loadObject, $stations, $localTimings);
                            debug($data);

                        } catch (\Exception $e) {
                            debug($loadObject);
                            flash()->error("Upload error!" . $e->getMessage());

                            $this->viewData = [
                                'message_type' => "danger",
                                'message' => "Upload error!" . $e->getMessage(),
                                "airports" => Airport::listHandlingStations(),
                                "airlines" => ["" => "Select"] + Airline::listHandlingAirlines(),
                                "iataCode" => airlineModule() && env(IATA_CODE) ? env(IATA_CODE) : "IB",
                            ];

                            return view("handling-flight-schedule/upload", $this->viewData);
                        }
                    }
                    */

//                    debug($request->all());
                    debug($data);
//                    return 1;

                    $params = [];
                    foreach ($request->all() as $p => $item) {
                        if ($p != "file"){
                            $params[$p] = $item;
                        }
                    }

                    UploadFlightScheduleJob::dispatch($data, $params);
//                    dispatch(new UploadFlightScheduleJob($data, $params));
                }
            }
            else {
                flash()->error("No File Uploaded. Please select Excel sheet and press upload.");

                $this->viewData = [
                    'message_type'  => "danger",
                    'message'       => 'No File Uploaded. Please select Excel sheet and press upload.',
                ];
            }
        }

        $this->viewData["airports"] = Airport::listHandlingStations();
        $this->viewData["airlines"] = ["" => "Select"] + Airline::listHandlingAirlines();
        $this->viewData["iataCode"] =airlineModule() && env(IATA_CODE) ? env(IATA_CODE) : "IB";

        return view("handling-flight-schedule/upload", $this->viewData);
    }

    public function GetScheduleArray($type, $loadObject, $stations, $localTimings){
        $finalData = [];
        // Starting Index
        $k = 0;

        $fieldCount = 13;
        $minHeaderFieldRequired = 5;
        $headerFound = false;

        $airports = Airport::whereNotNull("iata")
            ->select([
                "id", DB::raw("UPPER(iata) as code")
            ])
            ->pluck("id", "code")
            ->all();

        $airportTimezones = Airport::whereNotNull("timezone")
            ->pluck("timezone", "id")
            ->all();

        $airlinesIATA = Airline::whereNotNull("iata")
            ->select([
                "id", DB::raw("UPPER(iata) as code")
            ])
            ->pluck("id", "code")
            ->all();

        $airlinesICAO = Airline::whereNotNull("icao")
            ->select([
                "id", DB::raw("UPPER(icao) as code")
            ])
            ->pluck("id", "code")
            ->all();

        $checkStations = env(CARRIER) == HANDLING;

        $n = $type == 1 ? -1 : 0;
        $iAirline = 0 + $n;
        $iFltNo = 1 + $n;
        $iEffDate = 2 + $n;
        $iEndDate = 3 + $n;
        $iFreq = 4 + $n;
        $iOrigin = 5 + $n;
        $iDepTime = 6 + $n;
        $iDest = 7 + $n;
        $iArrTime = 8 + $n;
        $iAcType = 9 + $n;
        $iAC = 10 + $n;
        $iC = 11 + $n;
        $iY = 12 + $n;
        $iBkdC = 13 + $n;
        $iBkdY = 14 + $n;
        $iNotes = 15 + $n;

        debug("TYPE: {$type}");

        foreach ($loadObject as $i => $object) {
            // Do Not Include Header Row
            // Find Starting Index $k

            if (!$headerFound){
                foreach ($object as $j => $title) {
                    if ($title){
                        $fieldExists = 0;
                        for($s = $j; $s < $j + $fieldCount; $s++){
                            if (isset($object[$s]) && $object[$s] != ""){
                                $fieldExists++;
                            }
                        }
                        if ($fieldExists > $minHeaderFieldRequired){
                            $headerFound = true;
                            $k = $j;
                            break;
                        }
                    }
                }
                continue;
            }

            if (!isset($object[$k]) || $object[$k] == ""){
                continue;
            }

            foreach ($object as $m => $item) {
                $object[$m] = trimMax($item);
            }

            $data = [];

            if (!isset($object[$iOrigin]) || !isset($object[$iDest])){
                continue;
            }


            $departureAirport = strtoupper($object[$iOrigin]);
            $arrivalAirport   = strtoupper($object[$iDest]);

//            debug($departureAirport."-".$arrivalAirport);

            if ($checkStations && !(in_array($departureAirport, $stations) || in_array($arrivalAirport, $stations))){
                continue;
            }

//            debug($departureAirport."-".$arrivalAirport);

            $bound = $this->GetFlightBound($stations, $departureAirport, $arrivalAirport);

            $departureAirportId = $this->getOriginDestinationStation($airports, $object[$iOrigin]);
            $arrivalAirportId = $this->getOriginDestinationStation($airports, $object[$iDest]);

            // Type FLT NO (IB3001)
            if ($type == 1) {
                $IATAfltNO = $this->getIATACodeAndFlightNumber($airlinesIATA, $airlinesICAO, $object[$iFltNo], $departureAirportId, $arrivalAirportId, $bound);

                // Get passenger Array
                $data['airline']            = $IATAfltNO['iata'];
                $data['airline_id']         = $IATAfltNO['airline_id'];
                $data['flight_number']      = $IATAfltNO['flight_number'];
                $data['flight_number_id']   = $IATAfltNO['flight_nubmer_id'];
            }
            // TYPE AIRLINE, FLT NO - separate
            else {
                list($iata, $icao, $airline_id) = $this->getAirline($airlinesIATA, $airlinesICAO, $object[$iAirline]);
                $flightNumber = $this->getFlightNumber($airline_id, $object[$iFltNo], $departureAirportId, $arrivalAirportId, $bound);

                // Get passenger Array
                $data['airline']            = $iata;
                $data['airline_id']         = $airline_id;
                $data['flight_number']      = $object[$iFltNo];
                $data['flight_number_id']   = $flightNumber ? $flightNumber->id : null;
            }

            $data['start_date']         = $this->GetDate($object[$iEffDate]);
            $data['end_date']           = $this->GetDate($object[$iEndDate]);

//             debug("PERIOD: " . $data['start_date']. " - ". $data["end_date"]);

//             debug($i .") INITIAL: ". $object[$k + 1]." - ".$object[$k + 2]." Converted: ".$data['start_date']. " - ". $data["end_date"]);

            if (strtotime($data["start_date"]) > strtotime($data["end_date"])){
                $data['start_date']         = $this->GetDate($object[$iEffDate], true);
                $data['end_date']           = $this->GetDate($object[$iEndDate], true);

//                 debug("FIXED: " . $data['start_date']. " - ". $data["end_date"]);
            }


            $data['departure_airport']      = $departureAirport;
            $data['departure_airport_id']   = $departureAirportId;
            $data['departure_time']         = $this->GetTime($object[$iDepTime]);
            $data['arrival_airport']        = $arrivalAirport;
            $data['arrival_airport_id']     = $arrivalAirportId;
            $data['arrival_time']           = $this->GetTime($object[$iArrTime]);

            $data['days']               = FlightSchedule::getDaysOfOperation($object[$iFreq]);

            if ($localTimings){
                list($depLocal, $prevDay) = $this->localTimeToUTC($airportTimezones, $departureAirportId, $data['start_date'], $data['departure_time']);
                list($arrLocal, $ignore)  = $this->localTimeToUTC($airportTimezones, $arrivalAirportId, $data['start_date'], $data['arrival_time']);

                $data['departure_time'] = $depLocal;
                $data['arrival_time'] = $arrLocal;

                if (strtotime($depLocal) > strtotime($arrLocal)){
                    debug("ERR!");
                }

                if ($prevDay){
                    $newDays = [];
                    foreach ($data["days"] as $each) {
                        $newDays[] = $each == 1 ? 7 : $each - 1;
                    }
                    sort($newDays);

                    $data['days'] = $newDays;
                }
            }

            // debug("Prev: ".$data['departure_time']."-".$data['arrival_time']. " | Converted: {$depLocal} - {$arrLocal}");

            // LONG TYPE TEMPLATE
            /*
            if (isset($object[$k + 9]) && isset($object[$k + 10]) && isset($object[$k + 11]) &&
                isset($object[$k + 12]) && isset($object[$k + 13]) && isset($object[$k + 14])){
                $returnDepartureAirport = $this->getOriginDestinationStation($airports, $object[$k + 12]);
                $returnArrivalAirport = $this->getOriginDestinationStation($airports, $object[$k + 14]);
                $returnIATAfltNO = $this->getIATACodeAndFlightNumber($object[$k + 8], $returnDepartureAirport['airport_id'], $returnArrivalAirport['airport_id']);
            }

            // SHORT TYPE TEMPLATE
            else {
            */

            $acReg = isset($object[$iAC]) ? $object[$iAC] : null;

            $IATAacType = $this->getACRegAndType($data['airline_id'], $object[$iAcType], $acReg);

            $data['aircraft_type'] = $IATAacType['ac_type'];
            $data['aircraft_type_id'] = $IATAacType['ac_type_id'];
            $data['aircraft_airline'] = $IATAacType['aircraft_airline_iata'];
            $data['aircraft_airline_id'] = $IATAacType['aircraft_airline_id'];
            $data['aircraft_id'] = $IATAacType['aircraft_id'];

            $data['cfg_c'] = isset($object[$iC]) ? $object[$iC] : null;
            $data['cfg_y'] = isset($object[$iY]) ? $object[$iY] : null;
            $data['bkd_c'] = isset($object[$iBkdC]) ? $object[$iBkdC] : null;
            $data['bkd_y'] = isset($object[$iBkdY]) ? $object[$iBkdY] : null;
            $data['notes'] = isset($object[$iNotes]) ? $object[$iNotes] : null;

            //}
            $finalData[] = $data;
        }

        return $finalData;
    }

    function GetDate($string, $mdy = false){
        if ($string){
            if (is_numeric($string)) {
                $date = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject(intval($string))->format("Y-m-d");
//                debug("CONVERTED DATE $string: {$date}");
                return strtotime($date) < strtotime(date("Y-m-d")) ? date("Y-m-d") : $date;
            }

            debug("HER:" .$string);

            preg_match('/[A-Za-z]{3}/', $string, $match);

            $len = strlen(str_replace(['-', '/', '.', ','], "", $string));

            $string = str_replace(['/', '.', ','], '-', $string);

            if (count($match)){
                //return date("Y-m-d", strtotime($string));
                $carbon = Carbon::parse($string);
                return $carbon->format("Y-m-d");
            }
            else {

                $currentYear = (int)date("Y");
                $nextYear = $currentYear + 1;

                if ($len == 7 || $len == 8){
                    if (!$mdy) {
                        $date = \DateTime::createFromFormat("d-m-Y", $string);

                        if ($date && ($date->format("Y") == $currentYear || $date->format("Y") == $nextYear)) {
                            return $date->format("Y-m-d");
                        }
                    }

                    $date = \DateTime::createFromFormat("m-d-Y", $string);
                    if ($date && ($date->format("Y") == $currentYear || $date->format("Y") == $nextYear)){
                        return $date->format("Y-m-d");
                    }

                    $date = \DateTime::createFromFormat("Y-m-d", $string);
                    return $date ? $date->format("Y-m-d") : null;
                }
                else if ($len == 6){

                    if (!$mdy) {
                        $date = \DateTime::createFromFormat("d-m-y", $string);

                        if ($date && ($date->format("Y") == $currentYear || $date->format("Y") == $nextYear)) {
                            return $date->format("Y-m-d");
                        }
                    }


                    $date = \DateTime::createFromFormat("m-d-y", $string);
                    if ($date && ($date->format("Y") == $currentYear || $date->format("Y") == $nextYear)){
                        return $date->format("Y-m-d");
                    }

                    $date = \DateTime::createFromFormat("y-m-d", $string);
                    return $date ? $date->format("Y-m-d") : null;
                }
            }
        }

        return null;
    }

    function isFloat(string $string) : bool {
        return is_numeric($string) && strpos($string, '.') !== false;
    }

    function GetTime($string){
        if ($string){
            if ($this->isFloat($string)){
                return \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject((float)$string)->format("H:i:s");
            }
            else if (is_string($string)) {
                return date("H:i:s", strtotime($string));
            } else {
                return $string->format('H:i:s');
            }
        }
        else {
            return  null;
        }
    }

    function localTimeToUTC($airportTimezones, $airportID, $depArrDate, $depArrTime){
        if (!$airportID || !isset($airportTimezones[$airportID]) || !$airportTimezones[$airportID]){
            debug("ERR: ". $airportID);
            return [$depArrTime, false];
        }

        $timezone = $airportTimezones[$airportID];
        $utcDateTime = getLocalDateTime($depArrDate." ".$depArrTime, "UTC", "Y-m-d H:i:s", $timezone);
        $utcTime = date("H:i:s", strtotime($utcDateTime));

        /*
        $date = new \DateTime($depArrTime, new \DateTimeZone($timezone)); // USER's timezone
        $date->setTimezone(new \DateTimeZone('UTC'));
        $nLocalTime = $date->format('H:i:s');
        */

        // debug($utcDateTime." | ".$depArrTime." => ".CalculateDifference($utcTime, $depArrTime));

        if (strtotime(date("Y-m-d", strtotime($utcDateTime))) < strtotime(date("Y-m-d"))){
//            debug($depArrTime." -> ".$localTime);
            return [$utcTime, true];
        }

        return [$utcTime, false];
    }

    function GetFlightBound($stations, $departure, $arrival){

        // Outbound
        if (in_array($departure, $stations)){
            return 0;
        }
        // Inbound
        else if (in_array($arrival, $stations)){
            return 1;
        }
        else {
            return null;
        }
    }

    function getACRegAndType($airlineID, $aircraftTypeName, $acReg = null){
        $acTypeMatches = true;
        $aircraftType = $aircraft = null;
        $aircraftTypeName = trimMax(preg_replace('/\s/u', ' ', $aircraftTypeName));
//        debug("AC TYPE Before:".$aircraftTypeName);
        $aircraftTypeName = trimMax($aircraftTypeName);
        $acType = $aircraftTypeName;

        if (strlen($aircraftTypeName) > 3) {
            preg_match('/(\w{2})*(?:[\s|-])*(.{3,4})/', $aircraftTypeName, $match);
            $iata = isset($match[1]) ? trimMax($match[1]) : null;
            $acType = isset($match[2]) ? trimMax($match[2]) : null;

            if (strlen($acType) < 3){
                $acType = $aircraftTypeName;
            }

        }
//        debug("AC TYPE:{$aircraftTypeName}/".$acType);

        if ($acType){
            $aircraftType = AircraftType::where(function($sql) use ($acType){
                $sql->where("iata_name", "LIKE", "%{$acType}%")
                    ->orWhere("icao", "LIKE", "%{$acType}%")
                    ->orWhere("name", "LIKE", "%{$acType}%");
            })
                ->where(function($sql) use ($airlineID){
                    $sql->whereNull("airline_id");
                    if ($airlineID){
                        $sql->orWhere("airline_id", $airlineID);
                    }
                })
                ->orderByDesc("airline_id")
                ->first();

            if (!$aircraftType){
                $aircraftType = new AircraftType();
                if (strlen($acType) == 3){
                    $aircraftType->iata_name = $acType;
                }
                else {
                    $aircraftType->icao = $acType;
                }
                $aircraftType->airline_id = $airlineID;
                $aircraftType->created_by = Auth::user()->id;
                $aircraftType->save();
            }
        }

        if ($acReg){
            $aircraft = Aircraft::where(function($sql) use ($acReg){
                $sql->where("mvt_name", $acReg)
                    ->orWhere("name", $acReg)
                    ->orWhere("mvt_name", str_replace(["-", " "], "", $acReg));
            });

            if ($airlineID){
                $aircraft->where("airline_id", $airlineID);
            }

            $aircraft = $aircraft->first();

            if ($aircraft) {
                if ($aircraftType) {
//                debug($aircraft->mvt_name. "-".$aircraft->aircraft_type_id. " AC TYPE:".$aircraftType->icao." id=".$aircraftType->id);
                    if ($aircraft->aircraft_type_id == $aircraftType->id) {
                        $acTypeMatches = true;
                    } else {
                        $acTypeMatches = false;
                    }
                    $aircraft->deleted_at = null;
                    $aircraft->save();
                }
            }
            else {
                // Create new AC
                $aircraft = new Aircraft();
                $aircraft->mvt_name         = contains($acReg, "-") ? str_replace("-", "", $acReg) : $acReg;
                $aircraft->name             = contains($acReg, "-") ? $acReg : partitionAircraftCode($acReg);
                $aircraft->airline_id       = $airlineID;
                $aircraft->aircraft_type_id = $aircraftType ? $aircraftType->id : null;
                $aircraft->save();
            }
        }

        return [
            'ac_type'               => $acType,
            'ac_type_id'            => $acTypeMatches && $aircraftType ? $aircraftType->id : null,
            'aircraft_airline_iata' => null,
            'aircraft_airline_id'   => null,
            'aircraft_id'           => $aircraft ? $aircraft->id : null,
        ];
    }

    function getOriginDestinationStation(&$airports, $iata){
        $airport_id = null;
        //preg_match('/\s*(\w{3})\s*/', $string, $match);

        if (isset($iata) && $iata){

            $iata = preg_replace('/^\p{Z}+|\p{Z}+$/u', '', $iata);

            $iata = strtoupper($iata);

            $airport_id = isset($airports[$iata]) ? $airports[$iata] : null;

            if (!$airport_id){

                $findAP = Airport::where("iata", strtoupper($iata))
                                 ->first();

                if ($findAP){
                    $findAP->deleted_at = null;
                    $findAP->save();

                    // include
                    $airports[$iata] = $findAP->id;
                }
                else {
                    $airport = Airport::findOrCreateByIATA($iata);


                    $airport_id = $airport->id;

                    // include
                    $airports[$iata] = $airport_id;
                }
            }
        }

        return $airport_id;
    }

    function getIATACodeAndFlightNumber(&$airlinesIATA, &$airlinesICAO, $fltNoStr, $dep_airport_id, $arr_airport_id, $bound){
        $flightNumber = null;
        preg_match('/(\w{2}[A-Z]?)(?:[\s|-])*(\d+\D*)/', $fltNoStr, $match);
        $code = isset($match[1]) ? strtoupper($match[1]) : null;
        $fltNo = isset($match[2]) ? $match[2] : null;

        list($iata, $icao, $airline_id) = $this->getAirline($airlinesIATA, $airlinesICAO, $code);

        if ($airline_id && $fltNo) {
            $flightNumber = $this->getFlightNumber($airline_id, $fltNo, $dep_airport_id, $arr_airport_id, $bound);
        }

        return [
            'iata'              => $iata,
            'icao'              => $icao,
            'airline_id'        => $airline_id,
            'flight_number'     => $fltNo,
            'flight_nubmer_id'  => ($flightNumber ? $flightNumber->id : null)
        ];
    }

    function getAirline(&$airlinesIATA, &$airlinesICAO, $code = null){
        if (!$code){
            return [
                null,
                null,
                null,
            ];
        }

        $iata = $icao = null;

        if (strlen($code) == 3){
            $icao = $code;
            $airline_id = isset($airlinesICAO[$code]) ? $airlinesICAO[$code] : null;
        }
        else {
            $iata = $code;
            $airline_id = isset($airlinesIATA[$code]) ? $airlinesIATA[$code] : null;
        }

        if (!$airline_id){
            $airline = new Airline();
            $airline->iata = $iata;
            $airline->icao = $icao;
            $airline->created_by = Auth::user()->id;
            $airline->save();

            $airline_id = $airline->id;

            // include this iata
            if ($iata){
                $airlinesIATA[$iata] = $airline_id;
            }

            if ($icao){
                $airlinesICAO[$icao] = $airline_id;
            }
        }

        return [
            $iata,
            $icao,
            $airline_id,
        ];
    }

    function getFlightNumber($airline_id, $fltNo, $dep_airport_id, $arr_airport_id, $bound){
        $flightNumber = FlightNumber::where("airline_id", $airline_id)
            ->where("flight_number", $fltNo)
            ->where("departure_airport_id", $dep_airport_id)
            ->where("arrival_airport_id", $arr_airport_id)
            ->whereNull("deleted_at")
            ->first();

        if (!$flightNumber){
            $flightNumber = new FlightNumber();
            $flightNumber->airline_id = $airline_id;
            $flightNumber->flight_number = ltrim($fltNo, "0");
            $flightNumber->bound = $bound;
            $flightNumber->departure_airport_id = $dep_airport_id;
            $flightNumber->arrival_airport_id = $arr_airport_id;
            $flightNumber->created_by = Auth::user()->id;
            $flightNumber->save();
        }

        return $flightNumber;
    }

}
