<?php namespace App\Http\Controllers;

ini_set('max_execution_time', 60000);
ini_set('memory_limit', '1024M');

use App\Classes\Graph;
use App\Models\AgencyCode;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\BspReport;
use App\Models\Currency;
use App\Models\CurrencyRate;
use App\Models\FlightCabin;
use App\Models\FlightFareBasis;
use App\Models\FlightOrderStatus;
use App\Models\FlightRbd;
use App\Models\FlightType;
use App\Models\Passenger;
use App\Models\PassengerFlight;
use App\Models\PassengerOrder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

use App\Models\Flight;
use App\Models\FlightNumber;
use Illuminate\Support\Facades\URL;
use Maatwebsite\Excel\Facades\Excel;
use stdClass;

class ForwardRevenueController extends Controller
{
    /**
     * Initial page
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $dateFrom = \request("date_from") ? \request("date_from") : date("Y-m-d");
        $dateTo = \request("date_to") ? \request("date_to") : date("Y-m-d");
        $period = \request("period") ? \request("period") : 5;

        $flightsConfig = Flight::getFlightsConfig($dateFrom, $dateTo, $period);
        $flightsBooked = Flight::getFlightPaxRevenue($dateFrom, $dateTo, $period);

        $finalData = $this->prepareFinalRevenueData($flightsConfig, $flightsBooked);
        $chartData = $this->calculateTotalsAndGetChartArray($finalData, $period, $dateFrom, $dateTo);

        $this->viewData = [
            'flightNumbers'     => FlightNumber::getArrayFlightNumbers(),
            'dateFrom'          => $dateFrom,
            'dateTo'            => $dateTo,
            'periodLabels'      => getPeriodsArray(),
            'periodColumns'     => getPeriodNamesArray(),
            'finalData'         => isset($finalData) ? $finalData : [],
            'chart'             => isset($chartData) ? $chartData : [],
            'period'            => isset($period) ? $period : null,
        ];

        return view('forwardrevenue.index', $this->viewData);
    }


    public function report($period = null, $from = null, $to = null)
    {
        $dateFrom = (isset($from)) ? $from : date('Y-01-01');
        $dateTo = (isset($to)) ? $to : date('Y-m-d');

        $period = (isset($period)) ? $period : 0; // 0-Yearly, 1-Quarterly, 2-Monthly, 3-Weekly, 4-Weekdays, 5-Daily

        $flightsConfig = Flight::getFlightsConfig($dateFrom, $dateTo,$period, null);
        $bookedFlights = Flight::getFlightPaxRevenue($dateFrom, $dateTo, $period, null);

        $this->viewData = [
            'date_from'         => $dateFrom,
            'date_to'           => $dateTo,
            'booked_flights'    => $bookedFlights,
            'flights_config'    => $flightsConfig,
            'period'            => $period
        ];

        return view('forwardrevenue.report', $this->viewData);
    }

    public function show($date){
        $dateCheck = date_parse($date);
        if ($dateCheck["error_count"] == 0 && checkdate($dateCheck["month"], $dateCheck["day"], $dateCheck["year"])){

            $flightsConfig = Flight::getFlightsConfig($date, $date,null);
            $bookedFlights = Flight::getFlightPaxRevenue($date, $date,null, 1);

            /*$bookedFlights = PassengerFlight::getPaxAndRevenueDateFlights($date);
            $flightsConfig = Flight::getFlightsConfig($bookedFlights->first()->date, $bookedFlights->last()->date,null,1);*/

            $this->viewData = [
                'date'              => $date,
                'booked_flights'    => $bookedFlights,
                'flights_config'    => $flightsConfig
            ];

            return view('forwardrevenue.show', $this->viewData);
        }
    }

    public function flightDaily($widget = null){

        if (isset($widget)){
            $dateFrom = date('Y-m-d');
            $dateTo = date('Y-m-d');
            $period = 6; //FlightNumbers
            $flightNumber = null;
        }
        else {
            if (\request("date_from"))
                $dateFrom = \request("date_from");
            else
                $dateFrom = date('Y-m-d', strtotime("first day of this month"));

            if (\request("date_to"))
                $dateTo = \request("date_to");
            else
                $dateTo = $dateFrom;

            if (\request("period"))
                $period = \request("period");

            if (\request("flight_number"))
                $flightNumber = \request("flight_number");
            else
                $flightNumber = null;
        }


        $flightDaily = Flight::getFlightPaxRevenue($dateFrom, $dateTo, $period, $flightNumber);
        $flightConfig = Flight::getFlightsConfig($dateFrom, $dateTo, $period, $flightNumber);
        $flightConfigProperties = ['flight_count', 'config_c', 'config_y', 'flight_count_in', 'config_out_c',
            'config_out_y', 'config_in_c', 'config_in_y'];

        $flightData = array();
        foreach($flightDaily as $i=>$each){
            $flightData[$i] = new stdClass();
            foreach($each['attributes'] as $k=>$value){
                $flightData[$i]->$k = $value;
            }


            $found = 0;
            foreach($flightConfig as $j=>$item){
                if (isset($item->flight_departure)){
                    if ($item->flight_departure == $each->flight_departure){
                        $found = 1;
                        break;
                    }
                }
                else {
                    if ($item->date == $each->date) {
                        $found = 1;
                        break;
                    }
                }
            }
            if ($found == 1) {
                foreach ($flightConfig[$j]['attributes'] as $k => $value) {
                    if ($k != 'date' && $k != 'departure_date') {
                        $flightData[$i]->$k = $value;
                    }
                }
                $flightData[$i]->config_out = $flightData[$i]->config_out_c + $flightData[$i]->config_out_y;
                $flightData[$i]->config_in = $flightData[$i]->config_in_c + $flightData[$i]->config_in_y;
                $flightData[$i]->config_total = $flightData[$i]->config_in + $flightData[$i]->config_out;
            }
            else {
                foreach ($flightConfigProperties as $value) {
                    $flightData[$i]->$value = '';
                }
                $flightData[$i]->config_out = '';
                $flightData[$i]->config_in = '';
                $flightData[$i]->config_total = '';
            }
        }

        if (isset($widget)){
            return $flightData;
        }

        $this->viewData = [
            'flightNumbers'     => FlightNumber::getArrayFlightNumbers(),
            'periodLabels'      => ['None' => 'None', 0 => 'Yearly', 1 => 'Quarterly', 2 => 'Monthly', 3 => 'Weekly', 4 => 'Weekdays', 5 => 'Daily', 6 => 'Flight Number'],
            'periodColumns'     => ['None' => 'None', 0 => 'Year', 1 => 'Quarter', 2 => 'Month', 3 => 'Weekly', 4 => 'Weekdays', 5 => 'Daily', 6 => 'Flight Number'],
            'finalData'         => isset($flightData) ? $flightData : [],
            'period'            => isset($period) ? $period : null,
            'flightNumber'      => isset($flightNumber) ? $flightNumber : null,
            'date'              => $dateFrom,
            'dateFrom'          => $dateFrom,
            'dateTo'            => $dateTo,
        ];

        return view("forwardrevenue.flightdaily", $this->viewData);
    }

    public function flightDailySales(){

        if (\request("date_from"))
            $dateFrom = \request("date_from");
        else
            $dateFrom = date('Y-m-d', strtotime("first day of this month"));

        if (\request("date_to"))
            $dateTo = \request("date_to");
        else
            $dateTo = $dateFrom;

        $period = \request("period");

        $flightNumber = \request("flight_number");

        $flightDaily = PassengerFlight::getFlightPreviousMonthPaxRevenue($dateFrom, $dateTo, $flightNumber);

        $flightConfig = Flight::getFlightsConfig($dateFrom, $dateTo, $period, $flightNumber);

        $flightConfigProperties = [
            'flight_count',
            'config_c',
            'config_y',
            'flight_count_in',
            'config_out_c',
            'config_out_y',
            'config_in_c',
            'config_in_y'
        ];

        $flightData = array();
        foreach($flightDaily as $i => $each){
            $flightData[$i] = new stdClass();
            foreach($each['attributes'] as $k=>$value){
                $flightData[$i]->$k = $value;
            }


            $found = 0;
            foreach($flightConfig as $j=>$item){
                if ($item->date == $each->date){
                    $found = 1;
                    break;
                }
            }
            if ($found == 1) {
                foreach ($flightConfig[$j]['attributes'] as $k => $value) {
                    if ($k != 'date') {
                        $flightData[$i]->$k = $value;
                    }
                }

                $flightData[$i]->config_out = $flightData[$i]->config_out_c + $flightData[$i]->config_out_y;
                $flightData[$i]->config_in = $flightData[$i]->config_in_c + $flightData[$i]->config_in_y;
                $flightData[$i]->config_total = $flightData[$i]->config_in + $flightData[$i]->config_out;
            }
            else {
                foreach ($flightConfigProperties as $value) {
                    $flightData[$i]->$value = '';
                }
                $flightData[$i]->config_out = '';
                $flightData[$i]->config_in = '';
                $flightData[$i]->config_total = '';
            }
        }

        $this->viewData = [
            'flightNumbers'    => FlightNumber::getArrayFlightNumbers(),
            'periodLabels'     => ['None' => 'None', 0 => 'Yearly', 1 => 'Quarterly', 2 => 'Monthly', 3 => 'Weekly', 4 => 'Weekdays', 5 => 'Daily', 6 => 'Flight Number'],
            'periodColumns'    => ['None' => 'None', 0 => 'Year', 1 => 'Quarter', 2 => 'Month', 3 => 'Weekly', 4 => 'Weekdays', 5 => 'Daily', 6 => 'Flight Number'],
            'finalData'        => isset($flightData) ? $flightData : [],
            'period'           => isset($period) ? $period : null,
            'date'             => $dateFrom,
            'dateFrom'         => $dateFrom,
            'dateTo'           => $dateTo,
            'flightNumber'     => $flightNumber,
        ];


        return view("forwardrevenue.flightdailysales", $this->viewData);
    }

    public function salesDatePassengers(){
        if (\request("date_from"))
            $dateFrom = \request("date_from");
        else
            $dateFrom = date('Y-m-d', strtotime("first day of this month"));
        if (\request("date_to"))
            $dateTo = \request("date_to");
        else
            $dateTo = date('Y-m-d');

        if (\request("period"))
            $period = \request("period");

        if (\request("flight_number"))
            $flightNumber = \request("flight_number");
        if (\request("sales_date"))
            $salesDate = \request("sales_date");

        $passengers = PassengerFlight::getFlightPassengersBySalesDate($dateFrom, $flightNumber, $salesDate);

        $this->viewData = [
            'date'              => $dateFrom,
            'flight_number'     => $flightNumber,
            'sales_date'        => $salesDate,
            'passengers'        => $passengers
        ];

        return view('forwardrevenue.salesDatePassengers', $this->viewData);
    }

    /**
     * Upload page
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Illuminate\View\View|int
     */
    public function upload(){

        if ($this->isPostRequest() && \request()->hasFile('excel_sheet')) {

            debug(request()->all());

            $file = \request()->file('excel_sheet');
            // Load Excel File

            if (!in_array($file->getClientOriginalExtension(), ['xls', 'xlsx'])) {
                return redirect('forwardrevenue/upload')
                    ->with('message', 'Wrong Format('.$file->getClientOriginalExtension().'). Please Make Sure The Format is XLS or XLSX.')
                    ->with('type', 'danger');
            }

            $loadObject = Excel::load($file, function ($reader) {
                $reader->noHeading();
                $reader->toObject();
            })->get();

            // Get Passenger Orders Array
            $passenger = $this->getPassengerOrderArray($loadObject);

            if (!count($passenger))
                return redirect('forwardrevenue/upload')
                    ->with('message', 'File Is Empty. Found No Data To Upload.')
                    ->with('type', 'info');

            // Pass To Parse And Upload Revenue Info
            $this->getParseAndUploadRevenue($passenger);

            flash()->success('Revenue Data Successfully Uploaded');

            return redirect('forwardrevenue/upload');
        }

        return view('forwardrevenue/upload');
    }

    public function prepareFinalRevenueData($flightsConfig, $bookedFlights){
        $flightData = [];
        $flightConfigProperties =
            [
                'flight_count',
                'config_c',
                'config_y',
                'flight_count_in',
                'config_out_c',
                'config_out_y',
                'config_in_c',
                'config_in_y'
            ];

        foreach ($bookedFlights as $i => $each) {
            $flightData[$i] = new stdClass();
            foreach ($each['attributes'] as $k => $value) {
                $flightData[$i]->$k = $value;
            }


            $found = 0;
            foreach ($flightsConfig as $j => $item) {
                if ($item->date == $each->date) {
                    $found = 1;
                    break;
                }
            }
            if ($found == 1){
                foreach ($flightsConfig[$j]['attributes'] as $k => $value) {
                    if ($k != 'date') {
                        if ($found == 1)
                            $flightData[$i]->$k = $value;
                        else
                            $flightData[$i]->$k = 'N/A';
                    }
                }
                $flightData[$i]->config_out = $flightData[$i]->config_out_c + $flightData[$i]->config_out_y;
                $flightData[$i]->config_in = $flightData[$i]->config_in_c + $flightData[$i]->config_in_y;
                $flightData[$i]->config_total = $flightData[$i]->config_in + $flightData[$i]->config_out;
            }
            else {
                foreach ($flightConfigProperties as $value) {
                    $flightData[$i]->$value = '';
                }
                $flightData[$i]->config_out = '';
                $flightData[$i]->config_in = '';
                $flightData[$i]->config_total = '';
            }
        }

        return $flightData;
    }

    public function calculateTotalsAndGetChartArray(&$finalData, $period, $dateFrom, $dateTo){
        $weekday = getWeekdaysArray(true);
        $months  = getMonthsArray(true);

        if ($period == 3) {
            $nextPeriod = (int)$period + 2;
        }
        if ($period == 2) {
            $nextPeriod = (int)$period + 3;
        }
        else {
            $nextPeriod = (int)$period + 1;
        }

        $selectedYear = date('Y', strtotime($dateFrom));

        $chart = [];

        $chart[0] = [
            'period'        => [],
            'pax_outbound'  => [],
            'pax_inbound'   => [],
            'pax_total'     => [],
            'revenue_line'  => [],
        ];

        $chart[1] = [
            'period'        => [],
            'revenue_c'     => [],
            'revenue_y'     => [],
            'revenue_total' => [],
        ];

        $chart[2] = [
            'period'        => [],
            'sf_outbound'   => [],
            'sf_inbound'    => [],
            'sf_total'      => [],
        ];


        $chart[3] = [
            'period'           => [],
            'revenue_outbound' => [],
            'revenue_inbound'  => [],
            'revenue_total'    => [],
        ];


        foreach($finalData as $i => &$each) {

            list($selectedPeriod, $from, $to) = periodText($period, $dateFrom, $dateTo, $months, $weekday, $selectedYear, $each->date);

            if ($period < 4)
                $href = "forwardrevenue?period=$nextPeriod&date_from=$from&date_to=$to";
            else if ($period == 5)
                $href = "forwardrevenue/flightdaily?period=$nextPeriod&date_from=$from&date_to=$to";
            else if ($period == 6)
                $href = "forwardrevenue/flightdailysales?period=$nextPeriod&date_from=$from&date_to=$to";
            else
                $href = "forwardrevenue?period=$nextPeriod&date_from=$from&date_to=$to";

            $each->selected_period = $selectedPeriod;
            $each->href = $href;
            $each->revenue_out_total = $each->revenue_out_c + $each->revenue_out_y;
            $each->revenue_in_total = $each->revenue_in_c + $each->revenue_in_y;

            $each->revenue_c_total = $each->revenue_out_c + $each->revenue_in_c;
            $each->revenue_y_total = $each->revenue_out_y + $each->revenue_in_y;
            $each->revenue_total = $each->revenue_c_total + $each->revenue_y_total;

            $each->pax_out_total = $each->pax_out_c + $each->pax_out_y;
            $each->pax_in_total = $each->pax_in_c + $each->pax_in_y;
            $each->pax_total = $each->pax_out_total + $each->pax_in_total;

            $each->pax_c_total = $each->pax_out_c + $each->pax_in_c;
            $each->pax_y_total = $each->pax_out_y + $each->pax_in_y;

            $each->sf_out = ($each->config_out != 0 && $each->config_out != 'N/A') ? number_format(100 * $each->pax_out_total / $each->config_out) : 0;
            $each->sf_in = ($each->config_in != 0 && $each->config_in != 'N/A') ? number_format(100 * $each->pax_in_total / $each->config_in) : 0;
            $each->sf_total = ($each->config_total != 0 && $each->config_total != 'N/A') ? number_format(100 * $each->pax_total / $each->config_total) : 0;


            $chart[0]['period'][] = $selectedPeriod;
            $chart[0]['pax_outbound'][] =(int)round($each->pax_out_total);
            $chart[0]['pax_inbound'][] =(int)round($each->pax_in_total);
            $chart[0]['pax_total'][] =(int)round($each->pax_total);
            $chart[0]['revenue_line'][] =(int)round($each->revenue_total / 1000);

            $chart[1]['period'][] = $selectedPeriod;
            $chart[1]['revenue_c'][] =(int)round($each->revenue_c_total);
            $chart[1]['revenue_y'][] =(int)round($each->revenue_y_total);
            $chart[1]['revenue_total'][] =(int)round($each->revenue_total);

            $chart[2]['period'][] = $selectedPeriod;
            $chart[2]['sf_outbound'][] = (int)$each->sf_out;
            $chart[2]['sf_inbound'][] = (int)$each->sf_in;
            $chart[2]['sf_total'][] =(int)$each->sf_total;

            $chart[3]['period'][] = $selectedPeriod;
            $chart[3]['revenue_outbound'][] =(int)round($each->revenue_out_total);
            $chart[3]['revenue_inbound'][] =(int)round($each->revenue_in_total);
            $chart[3]['revenue_total'][] =(int)round($each->revenue_total);


        }

        return $chart;
    }

    public static function checkBound($dep_airport, $arr_airport){
        if( ($dep_airport == "DYU" || ($dep_airport == 'LBD' && $arr_airport != 'DYU')))
            $bound = 0;
        elseif (($arr_airport == 'DYU' || $arr_airport == 'LBD'))
            $bound = 1;
        else
            $bound = null;

        return $bound;
    }

    public static function checkCabin($rbd){

        if ($rbd == null)
            $cabin = null;
        else{
            if ($rbd == 'C' || $rbd == 'J' || $rbd == 'I' || $rbd == 'D'){
                $cabin = 'Business';
            }
            else
                $cabin = 'Economy';
        }
        return $cabin;
    }

    protected function generateRandomPassword($length = 10){
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ/*[]@$%';
        $randomString = '';

        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, strlen($characters) - 1)];
        }
        return $randomString;
    }

    protected function days_in_month($month, $year) {
        if(checkdate($month, 31, $year)) return 31;
        if(checkdate($month, 30, $year)) return 30;
        if(checkdate($month, 29, $year)) return 29;
        if(checkdate($month, 28, $year)) return 28;
        return 0; // error
    }

    protected function TchCodeCheck($tch, $iata) {
        // echo '<meta http-equiv="Content-Type" content="text/html; charset=utf-8">';
        $code = AgencyCode::where('code',$iata)
                            ->first();

        if (!$code){
            $contains_cyrillic = (bool) preg_match('/[\p{Cyrillic}]/u', $tch);

            if ($contains_cyrillic) {
                $code = AgencyCode::where('tch_code', $tch)
                                    ->first();
                if (!$code){
                    //$agency = Agency::where('agency', 'Unknown')->first();
                    $code = new AgencyCode;
                    $code->tch_code = $tch;
//                    $code->agency_id = 101466;//$agency->id;
                    $code->save();
                }
            }
            // Doesn't contain cyrillic
            else
            {
                //$agency = Agency::where('agency', 'Unknown')->first();
                $code = new AgencyCode;
                $code->code = $iata;
                //$code->agency_id = 101466;//$agency->id;
                $code->save();
            }
        }

        return $code->id;
    }

    protected function fare_fix($fare_original, $salesDate){

        $year = date("Y",strtotime($salesDate));
        $db_currency = "s_currency_".$year;

        $fare_cur = substr($fare_original, -3, 3);

        preg_match_all('!\d+(?:\.\d+)?!', $fare_original, $matches);
        $floats = array_map('floatval', $matches[0]);
        $fare = $floats[0];

        if ($fare == 0){
            $fare_cur = "";
        }
        else {
            if ($fare_cur == "USD")
                $rate = 1;
            else
            {
                $currency = Currency::select($fare_cur)->where('date',$salesDate)->first();
                if (empty($currency))
                    $currency = Currency::select($fare_cur)->orderBy('date','DESC')->first();

                $rate = 1 /$currency[$fare_cur];
            }

            $fare = round($fare * $rate,2);
        }
        $fixed['fare'] = $fare;
        $fixed['currency'] = $fare_cur;

        return $fixed;
    }

    protected function cyrillicToLatin($string){

        $string = mb_convert_encoding($string, 'UTF-8','windows-1251');

        //$contains_cyrillic = (bool) preg_match('/[\p{Cyrillic}]/u', $string);
        $contains_cyrillic = (bool) preg_match('/[Α-Ωα-ωА-Яа-я]/u', $string);
        if ($contains_cyrillic) {
            $cyr  = array('а','б','в','г','д','е','ж','з','и','й','к','л','м','н','о','п','р','с','т','у','ы','э',
                'ф','х','ц','ч','ш','щ','ъ','ь', 'ю','я','А','Б','В','Г','Д','Е','Ж','З','И','Й','К','Л','М','Н','О','П','Р','С','Т','У','Ы','Э',
                'Ф','Х','Ц','Ч','Ш','Щ','Ъ','Ь', 'Ю','Я' );

            $lat = array( 'a','b','v','g','d','e','zh','z','i','y','k','l','m','n','o','p','r','s','t','u','y','e',
                'f' ,'h' ,'ts' ,'ch','sh' ,'sht' ,'a' ,'y' ,'yu' ,'ya','A','B','V','G','D','E','Zh',
                'Z','I','Y','K','L','M','N','O','P','R','S','T','U','Y','E',
                'F' ,'H' ,'Ts' ,'Ch','Sh' ,'Sht' ,'A' ,'Y' ,'Yu' ,'Ya' );

            $string = str_replace($cyr, $lat, $string);
        }

        //$textlat = str_replace($lat, $cyr, $textlat);
        //echo("$textcyr");
        return $string;
    }

    protected function date_fix($date, $salesDate){

        $date = iconv(mb_detect_encoding($date, mb_detect_order(), true), "UTF-8", $date);

        $date_part[0] =mb_substr($date,0,2);
        $date_part[1] =mb_substr($date,2,3);


        $month = date('m', strtotime($salesDate));
        $year = date('Y', strtotime($salesDate));

        $next_year = date('Y', strtotime('+1 year'));

        $month_rus = array('ЯНВ','ФЕВ','МАР','АПР','МАЙ','ИЮН','ИЮЛ','АВГ','СЕН','ОКТ','НОЯ','ДЕК');

        $month_num = array('-01-', '-02-','-03-', '-04-', '-05-', '-06-', '-07-', '-08-', '-09-', '-10-', '-11-', '-12-' );

        $new_date = $date_part[0];

        foreach($month_rus as $index => $value){
            if ($date_part[1] == $value){
                $new_date.= $month_num[$index];
                $month_order = $index + 1;
                break;
            }
        }

        if (!isset($month_order) || $month_order >= $month)
            $new_date .= $year;
        else {
            if ($month_order < $month){
                $new_date .= $next_year;
            }
        }
        $new_date = date('Y-m-d', strtotime($new_date));
        return $new_date;

    }

    protected function getRbd($value){
        if ($value == 'IN')
            $rbd = null;
        else {
            $firstLetter  = substr($value,0,1);
            $threeLetters = substr($value,0,3);
            $fourLetters  = substr($value,0,4);

            if ($firstLetter == 'A' || $firstLetter == 'X')
                $rbd = 'X';
            elseif ($firstLetter == 'S' || $threeLetters  == 'YID')
                $rbd = 'S';
            elseif ($firstLetter == 'I' || $threeLetters == 'JID' || $fourLetters  == 'CDIP')
                $rbd = 'I';
            elseif ($firstLetter == 'Y' && $threeLetters  != 'YID')
                $rbd = 'Y';
            elseif ($firstLetter == 'C' && $fourLetters  != 'CDIP')
                $rbd = 'C';
            else{
                $rbd = $firstLetter;
            }
        }
        return $rbd;
    }

    public function findOriginalFareId(&$object, $fareOriginal = false){
        if (is_object($object)) {
            $fareOriginal = $fareOriginal ? $fareOriginal : $object->original_fare;

            // Get Fare And Currency Separately
            preg_match_all("/(\d+(?:\.\d+)?)(?:\s*[.]*\s*)(\D*)/", $fareOriginal, $matches);
            list($fareAll, $fare, $currency) = $matches;

            if (isset($fare[0]) && $fare[0] && strlen($fare[0])) {
                $object->original_fare = $fare[0];

                // Check If Currency Exists
                if (isset($currency[0]) && $currency[0] && strlen($currency[0])) {
                    $currencyCheck = Currency::firstOrCreate(['code' => $currency[0]]);
                    $object->original_fare_currency_id = $currencyCheck->id;
                }
                $object->original_fare = $fareOriginal;
            }
        }
    }

    public function getCyrillicRecords(){
        $passengers = Passenger::where('last_name', 'REGEXP','[Α-Ωα-ωА-Яа-я]')
            ->orWhere('first_name', 'REGEXP','[Α-Ωα-ωА-Яа-я]')
            ->get();
        debug($passengers->last_name);
        $value = iconv(mb_detect_encoding($passengers->last_name, mb_detect_order(), true), "UTF-8", $passengers->last_name);
        debug($this->cyrillicToLatin($value));
        return "DONE";
    }

    public function convertFareToUsd(&$object, $date){
        if (is_object($object) && $object){
            if ($object->original_fare_currency_id && $object->original_fare){
                $object->fare_currency_id = 1;
                if ($object->original_fare_currency_id == 1){
                    $object->fare = $object->original_fare;
                }
                else {
                    $rate = CurrencyRate::where('to_currency_id', 1)
                        ->where('from_currency_id', $object->original_fare_currency_id)
                        ->where('date', '<=', $date)
                        ->orderBy('date', 'DESC')
                        ->first();
                    if ($rate) {
                        $object->fare = $rate->rate * $object->original_fare;
                    }
                }
            }
        }
    }

    public function getFixOrdersRbdCabin(){
        $passengerFlights = PassengerFlight::where('fare_basis', '<>', 'IN')

            ->where(function($sql){
                $sql->whereNull('rbd')
                    ->orWhereNull('cabin');
            })
            ->limit(100000)
            ->get(['id', 'fare_basis', 'cabin']);
        foreach ($passengerOrders as $order) {
            $rbd = $this->getRbd($order->fare_basis);
            $cabin = $this->checkCabin($rbd);
            $order->rbd = $rbd;
            $order->cabin = $cabin;
            $order->save();
        }
        return "DONE";
    }

    public function getFixOrdersCurrencyIds(){
        $orders = PassengerOrder::limit(100000)
            ->whereNull('original_fare_currency_id')
            ->get(['id', 'original_fare']);

        foreach ($orders as $i => $order) {
            // Get Fare And Currency Separately
            self::findAndSaveFare($order);
        }
    }

    public function getParseAndUploadRevenue($revenueData = []){

        /*if (!$revenueData) {
            $revenueData = DB::table('revenue_date_corrected')
                ->get();
        }*/
        foreach ($revenueData as $i => $passenger) {

            // If Sales Date is Not Set Then Skip
            if (!$passenger['sales_date'])
                continue;

            preg_match_all("/\b(?!MR|MRS|MS|MIS|DR)\w+/", $passenger['first_name'], $matches);
            list($names) = $matches;

            // Find Or Create Passenger
            $passenger_db_check = Passenger::firstOrCreate([
                'last_name'     => $passenger['last_name'],
                'first_name'    => isset($names[0]) ? $names[0] : null,
                'middle_name'   => isset($names[1]) ? $names[1] : null
            ]);


            $rbd   = $this->getRbd($passenger['fare_code']);
            $cabin = $this->checkCabin($rbd);
            $inf   = !$rbd ? 1 : null;

            if ($rbd){
                $flightRbd = FlightRbd::where('rbd', $rbd)
                                        ->first();
            }

            if ($cabin){
                $flightCabin = FlightCabin::where('cabin', $cabin)
                                            ->first();
            }

            if ($passenger['fare_code']){
                $flightFareBasis  = FlightFareBasis::where('fare_basis', $passenger['fare_code'])
                                                    ->first();
            }

            $flightOrderStatus = null;
            if ($passenger['order_status']){
                $passenger['order_status'] = ($passenger['order_status'] == 'A SZ:ETL') ? 'A' : $passenger['order_status'];
                $flightOrderStatus = FlightOrderStatus::where('order_status', $passenger['order_status'])
                                                    ->first();
            }

            // Find Passenger Order
            $passengerOrder = PassengerOrder::where('ticket_number', $passenger['ticket_number'])
                                            ->first(['id', 'fare']);

            if (!$passengerOrder) {
                $code_id = $this->TchCodeCheck($passenger['agency_code'], $passenger['iata_code']);
                $passengerOrder = new PassengerOrder;

                if ($passenger['fare'] != "") {
                    $this->findOriginalFareId($passengerOrder, $passenger['fare']);
                    // Convert To USD
                    $this->convertFareToUsd($passengerOrder, $passenger['sales_date']);
                }
                $passengerOrder->agency_code_id = $code_id;
                $passengerOrder->sales_date     = $passenger['sales_date'];
                $passengerOrder->ticket_number  = $passenger['ticket_number'];
                $passengerOrder->payment_form   = $passenger['payment_form'];
                $passengerOrder->airport_taxes  = $passenger['airport_taxes'];
                $passengerOrder->uab_taxes      = $passenger['uab_taxes'];
                $passengerOrder->fuel_taxes     = $passenger['fuel_taxes'];
                $passengerOrder->tch_taxes      = $passenger['tch_taxes'];
                $passengerOrder->other_taxes    = $passenger['other_taxes'];
                $passengerOrder->created_by     = Auth::user()->id;
                $passengerOrder->save();
            }
            else {
                $this->findOriginalFareId($passengerOrder, $passenger['fare']);
                // Convert To USD
                $this->convertFareToUsd($passengerOrder, $passenger['sales_date']);
                $passengerOrder->save();
            }


            $sector = explode('-', $passenger['route']);
            $depAirportIATA = (isset($sector[0])) ? $sector[0] : null;
            $arrAirportIATA = (isset($sector[1])) ? $sector[1] : null;


            $departureAirport = $arrivalAirport = null;
            if ($depAirportIATA){
                $departureAirport = Airport::where('iata', $depAirportIATA)
                                            ->first();
            }

            if ($arrAirportIATA){
                $arrivalAirport = Airport::where('iata', $arrAirportIATA)
                                            ->first();
            }

            $bound = $this->checkBound($depAirportIATA, $arrAirportIATA);

            // If Flight Number Starts With SZ Then Take Only Digits That Come After it
            preg_match_all('/(\w{2})(?:[-|\s])*(\d+)/', $passenger['flight_number'], $matches);
            list($flightAll, $airlineIATA, $flightNumber) = $matches;

            // If Matches Then AVBIS Flight
            if (isset($flightNumber[0]) && $flightNumber[0] && strlen($flightNumber[0])) {
                $flightNumber = $flightNumber[0];
            }

            $airline = null;
            if (isset($airlineIATA[0]) && $airlineIATA[0] && strlen($airlineIATA[0])) {
                $airlineIATA = $airlineIATA[0];
                $airline = Airline::where("iata", $airlineIATA)
                                    ->first();

            }

            // VVIP Flights Need Dep, Arr Airports
            if ($flightNumber == "001") {
                $flightNumberCheck = FlightNumber::firstOrCreate([
                    'airline_id'            => $airline ? $airline->id : null,
                    'flight_number'         => $flightNumber,
                    'departure_airport_id'  => $departureAirport->id,
                    'arrival_airport_id'    => $arrivalAirport->id,
                    'deleted_at'            => null,
                ]);
            }
            else {

                $flightNumberCheck = FlightNumber::firstOrCreate([
                    'airline_id'    => $airline ? $airline->id : null,
                    'flight_number' => $flightNumber,
                    'deleted_at'    => null,
                ]);

                if (!$flightNumberCheck->departure_airport_id) {
//                    $flightNumberCheck->departure_airport = $dep_airport;
                    $flightNumberCheck->departure_airport_id = $departureAirport ? $departureAirport->id : null;
                }
                if (!$flightNumberCheck->arrival_airport_id) {
//                    $flightNumberCheck->arrival_airport = $arr_airport;
                    $flightNumberCheck->arrival_airport_id = $arrivalAirport ? $arrivalAirport->id : null;
                }

                if ($airlineIATA != env(IATA_CODE)){
                    $flightType = FlightType::where('type', "Other Airlines")
                                                    ->first();

                    $flightNumberCheck->flight_type_id = $flightType ? $flightType->id : null;
                }

                $flightNumberCheck->save();

            }
            // If Bound Is Determined
            if ($bound) {
                $flightNumberCheck->bound = $bound;
            }

            $flight = null;
            if (isset($passenger['departure_date']) || $passenger['departure_datetime']) {
                if ($passenger['departure_datetime']){
                    $std = date('Y-m-d', strtotime($passenger['departure_datetime']));
                }
                elseif (isset($passenger['departure_date'])){
                    $std = date('Y-m-d', strtotime($passenger['departure_date']));
                }
                else{
                    $std = null;
                }

                $flight = Flight::where('flight_number_id', $flightNumberCheck->id)
                        ->where(DB::raw('DATE(std)'), $std)
                        ->first();

                if (!$flight) {
                    if (!in_array($flightNumberCheck->id, [1, 2])) { /// 1 and 2 Is DYU-DME-DYU Flight
                        $tomorrow = date('Y-m-d', strtotime("$std + 1 days"));
                        $yesterday = date('Y-m-d', strtotime("$std - 1 days"));
                        $flight = Flight::where("flight_number_id", $flightNumberCheck->id)
                                        ->whereBetween(DB::raw("DATE(std)"), [$yesterday, $tomorrow])
                                        ->orderBy('std', 'desc')
                                        ->first(['id']);
                    }
                }

            }

            $passFlightCheck = PassengerFlight::where('passenger_order_id', $passengerOrder->id)
                ->where('flight_number_id', $flightNumberCheck->id)
                ->first(['id', 'flight_id', 'flight_order_status_id']);

            if ($passFlightCheck) {
                if (!$passFlightCheck->flight_id && $flight) {
                    $passFlightCheck->flight_id = $flight->id;
                    $passFlightCheck->updated_by = Auth::user()->id;
                }

                if ($flightOrderStatus && $passFlightCheck->flight_order_status_id != $flightOrderStatus->id) {
                    $passFlightCheck->flight_order_status_id = $flightOrderStatus->id;
                    $passFlightCheck->updated_by = Auth::user()->id;
                }

                //if (!$passFlightCheck->fare){
                $passFlightCheck->fare = $passengerOrder->fare;
                //}

                $passFlightCheck->save();

            } else {

                // Insert Passenger Flight
                Passenger::find($passenger_db_check['id'])->passengerFlight()->create([
                    'flight_id'             => $flight ? $flight->id : null,
                    'flight_number_id'      => $flightNumberCheck->id,
                    'passenger_order_id'    => $passengerOrder->id,
                    'flight_fare_basis_id'  => isset($flightFareBasis) && $flightFareBasis ? $flightFareBasis->id : null,
                    'flight_rbd_id'         => isset($flightRbd) && $flightRbd ? $flightRbd->id : null,
                    'flight_cabin_id'       => isset($flightCabin) && $flightCabin ? $flightCabin->id : null,
                    'flight_order_status_id'=> isset($flightOrderStatus) && $flightOrderStatus ? $flightOrderStatus->id : null,
                    'departure'             => $passenger['departure_datetime'] ? $passenger['departure_datetime'] : (isset($passenger['departure_date']) ? $passenger['departure_date'] : null ),
                    'departure_date'        => $passenger['departure_date'],
                    'fare'                  => $passengerOrder->fare,
                    'coupon'                => $passenger['coupon'],
                    'fare_currency_id'      => 1,
                    'is_inf'                => $inf,
                    'created_by'            => Auth::user()->id,
                    'updated_by'            => 999
                ]);
            }

            // Check If Order Has Many Flights,
            // If So Divide Order / flights count
            if ($passengerOrder->passengerFlight->count() > 1){
                foreach ($passengerOrder->passengerFlight as $each) {
                    $each->fare = $passengerOrder->fare / $passengerOrder->passengerFlight->count();
                    $each->updated_by = Auth::user()->id;
                    $each->save();
                }
            }
        }
    }

    public function getFixFares(){
        $orders = PassengerOrder::where('original_fare_currency_id', 1)
            ->where('fare', '<', 10)
            ->where('original_fare', '>', 20)
            ->get();

        foreach ($orders as $order) {
            $order->fare = $order->original_fare;
            $order->updated_by = 100;
            $order->save();
            $flightCounter = $order->passengerFlight->count();
            if ($flightCounter > 0){
                foreach ($order->passengerFlight as $each) {
                    $each->fare = $order->fare / $flightCounter;
                    $each->updated_by = Auth::user()->id;
                    $each->save();
                }
            }
        }
        return "SCRIPT IS DONE";
    }

    protected function getPassengerOrderArray($loadObject){
        $passenger = [];
        foreach ($loadObject as $i => $object) {
            // Not Include Header Row
            if ($i == 0)
                continue;

            $sameInfo = false;
            if (!($object[1] && $object[2])){
                // If Coupon Not Found Break The Procedure
                if (!($object[14] && $object[15]))
                    break;

                // Check If Previous Record Is Set For Duplication
                if (isset($passenger[$i - 1]))
                    $sameInfo = true;
                else // If Not Set Skip This Order
                    continue;
            }

            // Get passenger Array
            $passenger[$i]['last_name']     = $sameInfo ? $passenger[$i - 1]['last_name']   : $object[1];
            $passenger[$i]['first_name']    = $sameInfo ? $passenger[$i - 1]['first_name']  : $object[2];
            $passenger[$i]['sales_date']    = $sameInfo ? $passenger[$i - 1]['sales_date']  : $object[3]->toDateString();
            $passenger[$i]['agency_code']   = $sameInfo ? $passenger[$i - 1]['agency_code'] : $object[4];
            $passenger[$i]['iata_code']     = $sameInfo ? $passenger[$i - 1]['iata_code']   : (int)$object[5];
            $passenger[$i]['fare']          = $sameInfo ? '' :   $object[6];
            $passenger[$i]['payment_form']  = $sameInfo ? '' :   $object[7];
            $passenger[$i]['airport_taxes'] = $sameInfo ? '' :   $object[8];
            $passenger[$i]['uab_taxes']     = $sameInfo ? '' :   $object[9];
            $passenger[$i]['fuel_taxes']    = $sameInfo ? '' :   $object[10];
            $passenger[$i]['tch_taxes']     = $sameInfo ? '' :   $object[11];
            $passenger[$i]['other_taxes']   = $sameInfo ? '' :   $object[12];
            $passenger[$i]['ticket_number'] = $sameInfo ? $passenger[$i - 1]['ticket_number'] :  $object[13];
            $passenger[$i]['coupon']             = $object[14];
            $passenger[$i]['flight_number']      = $object[15];
            $passenger[$i]['route']              = $object[16];
            $passenger[$i]['departure_datetime'] = $this->correctDepartureDate($object[17], $passenger[$i]['sales_date']);
            $passenger[$i]['departure_date']    = date('Y-m-d', strtotime($passenger[$i]['departure_datetime']));
            $passenger[$i]['order_status']             = $object[18];
            $passenger[$i]['fare_code']          = $object[19];
        }
        return $passenger;
    }

    protected function correctDepartureDate($value, $salesDate){
        $date_sep = explode('/', $value);

        if (!isset($date_sep[1]) || strtotime($date_sep[1]) == ""){
            $mins = "00:00";
        }
        else{
            $mins = $date_sep[1];
        }

        $dateCheck = date('Y-m-d', strtotime($date_sep[0]));

        if ($dateCheck == '1970-01-01')
            $fixed_date = $this->date_fix($date_sep[0], $salesDate)." ".$mins;
        else
            $fixed_date = $dateCheck." ".$mins;

        $datetime = date('Y-m-d H:i:s', strtotime($fixed_date));

        if (isset($date_sep[1]) && strtotime($date_sep[1]) != ""){
            $value = date('Y-m-d H:i:s', strtotime("$datetime - 5 hours"));
        }
        else
            $value = $datetime;

        return $value;
    }

    public function getImportPassengers(){
        //Passenger::truncate();

        $table = "passengers_import";
        $limit = 100; //1000;
        $queryLimit = 1000; //10000;
        $tableRows = DB::table($table)->count();

        $importArray = [];
        for ($i = 0; $i <= $tableRows; $i = $i + $queryLimit){
            $skip = $i;
            $importData =  DB::table($table)->skip($skip)->take($queryLimit)->get();
            foreach ($importData as $j => $each) {

                $importArray[] = [
                    'id'                    => $each->id,
                    'last_name'             => $each->last_name,
                    'first_name'            => $each->first_name,
                    'middle_name'           => $each->middle_name,
                    'email'                 => $each->email,
                    'dob'                   => $each->date_of_birth,
                    'pob_country_id'        => $each->pob_country_id,
                    'gender'                => $each->gender,
                    'nationality_id'        => $each->nationality_id,
                    'passport_num'          => $each->passport_num,
                    'passport_doi'          => $each->passport_doi,
                    'passport_dex'          => $each->passport_doe,
                ];

                if (count($importArray) && ( ($j + 1) % $limit == 0 ) ){
                    Passenger::insert($importArray);
                    $importArray = [];
                }
            }

            if (count($importArray)){
                Passenger::insert($importArray);
            }

        }


        return 1;
    }

    public function getImportPassengersOrders()
    {
        //PassengerOrder::truncate();
        //return 1;
        $table = "passengers__orders_import";
        $limit = 100; //1000;
        $queryLimit = 1000; //10000;
        $tableRows = DB::table($table)->count();

        $importArray = [];
        for ($i = 0; $i <= $tableRows; $i = $i + $queryLimit){
            $skip = $i;
            $importData =  DB::table($table)->skip($skip)->take($queryLimit)->get();
            foreach ($importData as $j => $each) {
                $importArray[] = [
                    'id'                        => $each->id,
                    'agency_code_id'            => $each->code_id,
                    'sales_date'                => $each->sales_date,
                    'ticket_number'             => $each->ticket_number,
                    'original_fare'             => $each->original_fare,
                    'original_fare_currency_id' => $each->original_fare_currency_id,
                    'payment_form'              => $each->payment_form,
                    'fare'                      => $each->fare,
                    'fare_currency_id'          => $each->fare_currency_id,
                    'reservations_system_id'    => $each->gds ? $each->gds : null,
                    'airport_taxes'             => $each->airport_taxes,
                    'uab_taxes'                 => $each->uab_taxes,
                    'fuel_taxes'                => $each->fuel_taxes,
                    'tch_taxes'                 => $each->tch_taxes,
                    'other_taxes'               => $each->other_taxes,
                ];

                if (count($importArray) && ( ($j + 1) % $limit == 0 ) ){
                    PassengerOrder::insert($importArray);
                    $importArray = [];
                }
            }
            if (count($importArray)){
                PassengerOrder::insert($importArray);
            }
        }

        return 1;
    }

    public function getImportPassengersFlights(){

        $table = "passengers__flights_import";
        $limit = 50; //1000;
        $queryLimit = 500; //10000;
        $tableRows = DB::table($table)->count();

        $importArray = [];
        for ($i = 0; $i <= $tableRows; $i = $i + $queryLimit){
            $skip = $i;
            $importData =  DB::table($table)->skip($skip)->take($queryLimit)->get();
            foreach ($importData as $j => $each) {

                if ($each->rbd){
                    $flightRbd = FlightRbd::firstOrCreate(['rbd' => $each->rbd]);
                }
                else {
                    $flightRbd = null;
                }


                $cabin = $each->actual_cabin;
                if ($cabin == "0" || $cabin === 0) {
                    $cabin = "Business";
                }
                else if ($cabin == 1) {
                    $cabin = "Economy";
                }
                else {
                    $cabin = null;

                }
                if ($cabin) {
                    $flightCabin = FlightCabin::firstOrCreate(['cabin' => $cabin]);
                }
                else {
                    $flightCabin = null;
                }

                if ($each->fare_basis){
                    $flightFareBasis  = FlightFareBasis::firstOrCreate(['fare_basis' => $each->fare_basis]);
                }
                else {
                    $flightFareBasis = null;
                }

                if ($each->status) {
                    $each->status = ($each->status == 'A SZ:ETL') ? 'A' : $each->status;
                    $flightOrderStatus = FlightOrderStatus::firstOrCreate(['order_status' => $each->status]);
                }
                else{
                    $flightOrderStatus = null;
                }

                $importArray[] = [
                    'id'                    => $each->id,
                    'passenger_id'          => $each->passenger_id,
                    'flight_id'             => $each->flight_id,
                    'flight_number_id'      => $each->flight_number_id,
                    'passenger_order_id'    => $each->passenger_order_id,
                    'flight_fare_basis_id'  => $flightFareBasis ? $flightFareBasis->id : null,
                    'flight_rbd_id'         => $flightRbd ? $flightRbd->id : null,
                    'flight_cabin_id'       => $flightCabin ? $flightCabin->id : null,
                    'flight_order_status_id'=> $flightOrderStatus ? $flightOrderStatus->id : null,
                    'departure'             => $each->departure,
                    'departure_date'        => $each->departure ? date('Y-m-d', strtotime($each->departure)) : null,
                    'fare'                  => $each->fare,
                    'fare_currency_id'      => $each->fare_currency_id,
                    'coupon'                => $each->coupon,
                    'seat_number'           => $each->seat_number,
                    'is_inf'                => $each->is_inf,
                    'baggage_pcs'           => $each->baggage_pcs,
                    'baggage_weight'        => $each->baggage_weight,
                    'tag_number'            => $each->tag_number,
                    'is_offload'            => $each->is_offload,
                    'reference_number'      => $each->reference_number,
                    'offload_force'         => $each->offload_force,
                    'offload_group'         => $each->offload_group,
                    'offload_rmk'           => $each->offload_rmk,
                ];

                if (count($importArray) && ( ($j + 1) % $limit == 0 ) ){
                    PassengerFlight::insert($importArray);
                    $importArray = [];
                }
            }

            if (count($importArray)){
                PassengerFlight::insert($importArray);
            }

        }
        return 1;
    }
}

