<?php

namespace App\Http\Controllers;

use App\Classes\FlightStaff\StaffFilter;
use App\Classes\FlightWatch\FlightPosition;
use App\Classes\Parsing\Helpers\ParseHelper;
use App\Classes\Parsing\Parse;
use App\Classes\Parsing\ParseOperations;
use App\Forms\FlightForm;
use App\Http\Controllers\Controller;
use App\Models\Aircraft;
use App\Models\AircraftType;
use App\Models\Airline;
use App\Models\AirlineService;
use App\Models\Airport;
use App\Models\Delay;
use App\Models\Flight;
use App\Models\FlightCatering;
use App\Models\FlightCrew;
use App\Models\FlightNumber;
use App\Models\FlightStaff;
use App\Models\FlightStation;
use App\Models\FlightType;
use App\Models\Maintenance;
use App\Models\Service;
use App\Models\StaffAirline;
use App\Models\StaffService;
use App\Models\StaffStation;
use App\Models\User;
use App\Repositories\Interfaces\IAirportRepository;
use App\Repositories\Interfaces\IFlightDelayRepository;
use App\Repositories\Interfaces\IFlightRepository;
use Illuminate\Http\Request;

use App\Http\Requests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;


use Jenssegers\Agent\Agent;

class FlightWatchController extends Controller
{
    const MAX_EXECUTION_TIME = 60;

    static function emptyFlights($staffAirlines = null, $staffAirports = null){
        if (ghaModule()){

            if (!$staffAirlines){
                $staffAirlines = StaffAirline::currentStaffAirlinesList();
            }

            if (!$staffAirports){
                $staffAirports = StaffStation::currentStaffStationsList();
            }

            if (!count($staffAirports) || !count($staffAirlines)){
                return true;
            }
        }
        return false;
    }

    public function flightWatch(Request $request)
    {
        $date = time();
        $airlineModule = airlineModule();

//        if (! ($request->get("go_to_airline_id") || $request->get("go_to_airport_id"))){

        $selectedAirlineIds = Airline::getSelectedAirlineIds($request);
        $selectedAirportIds = Airport::getSelectedAirportIds($request);

        $staffAirlines = StaffAirline::currentStaffAirlinesList();
        $staffAirports = StaffStation::currentStaffStationsList();

        $emptyFlights = self::emptyFlights($staffAirlines, $staffAirports);

//        }
//        else{
//            $selectedAirline = $request->get("go_to_airline_id");
//            $selectedAirport = $request->get("go_to_airport_id");
//        }

        if ($request->has("start_time") && $request->has("end_time") && $request->get("start_time") && $request->get("end_time"))
        {
            $startDate = date("Y-m-d", strtotime("-12 hours", strtotime($request->get("start_time"))));
            $endDate   = date("Y-m-d", strtotime("+12 hours", strtotime($request->get("end_time"))));
        }
        elseif ($request->has("go_to_date"))
        {
            $date = $request->get("go_to_date") ? strtotime($request->get("go_to_date")." 12:00:00") : strtotime(date("Y-m-d H:i:s"));

            $startDate = date("Y-m-d H:i:s", strtotime("-12 hours", $date));
            $endDate   = date("Y-m-d H:i:s", strtotime("+12 hours", $date));

            $aircraftMaintenance = Maintenance::maintenanceRange($startDate, $endDate);

            if ($emptyFlights){
                $flights = $jsonData = [];
                $flightTables = [
                    DEPARTURE   => null,
                    ARRIVAL     => null,
                    ALL         => null,
                ];
            }
            else {
                $flights =  Flight::handlingFlightsRange($startDate, $endDate, $selectedAirlineIds, $selectedAirportIds);

                $jsonData = self::getJsonDataForFlightWatchTimeLine($flights, $aircraftMaintenance);

                $flightsOPS = self::getFlightTrackerPanelFlights($flights);

                $tableLayout = $airlineModule ? "flights-list-airline" : "flights-list";

                $flightTables = [
                    DEPARTURE   => view("flight-tracker/layouts/{$tableLayout}", ["flightsOPS" => $flightsOPS[DEPARTURE], "tableId" => "tableFltDepartures"])->render(),
                    ARRIVAL     => view("flight-tracker/layouts/{$tableLayout}", ["flightsOPS" => $flightsOPS[ARRIVAL],   "tableId" => "tableFltArrivals"])->render(),
                    ALL         => view("flight-tracker/layouts/{$tableLayout}", ["flightsOPS" => $flightsOPS[ALL],       "tableId" => "tableFltAll"])->render(),
                ];
            }

            // Get JSON Ready
            return response()->json([
                "start_visible_period"  => date("Y-m-d 00:00:00", $date),
                "end_visible_period"    => date("Y-m-d 23:59:59", $date),
                "jsonData"              => $jsonData,
                "flights"               => $flights,
                "flightTables"          => $flightTables,
                "success"               => TRUE
            ]);
        }
        else {
            $startDate = date("Y-m-d H:i:s", strtotime("-12 hours", $date));
            $endDate   = date("Y-m-d H:i:s", strtotime("+12 hours", $date));
        }

//        $startDate = date("Y-m-d", strtotime("-1 hours", strtotime($request->get("start_time"))));
//        $endDate   = date("Y-m-d", strtotime("+1 hours", strtotime($request->get("end_time"))));

        // Load Maintenance
        $aircraftMaintenance = Maintenance::maintenanceRange($startDate, $endDate);

        if ($emptyFlights){
            $flights = $jsonData = [];

            $flightsOPS = [
                DEPARTURE   => null,
                ARRIVAL     => null,
                ALL         => null,
            ];
        }
        else {
            // Load Flights
            $flights = Flight::handlingFlightsRange($startDate, $endDate, $selectedAirlineIds, $selectedAirportIds);

            // Get JSON Ready
            $jsonData = self::getJsonDataForFlightWatchTimeLine($flights, $aircraftMaintenance);

            $flightsOPS = self::getFlightTrackerPanelFlights($flights);
        }

        // Aircraft List
        $aircraftList = Aircraft::getList();
        // AC Type List
        $aircraftTypeList = AircraftType::getList();

        $agent = new Agent();
        $isMobile = $agent->isMobile();

        $flightNumbersObj = FlightNumber::getArrayFlightNumbersWithSector(false, false, false, null, true);

        $flightNumbers = [];
        foreach ($flightNumbersObj as $each) {
            $flightNumbers[$each->id] = $each->sector;
        }

        $this->viewData = [
            "airlineModule"             => $airlineModule,
            "airlineName"               => env(AIRLINE),

            "isMobile"                  => $isMobile,
            "staffAirlines"             => $staffAirlines,
            "staffStations"             => $staffAirports,

            "flightsOPS"                => $flightsOPS, //->sortBy("std")->all(),

//            "flightsOPSYesterday"       => Flight::handlingFlightsRange($startDate, date("Y-m-d", strtotime($startDate))." 23:59:59", $selectedAirline, $selectedAirport),
//            "flightsOPSToday"           => Flight::handlingFlightsRange(date("Y-m-d"). " 00:00:00", date("Y-m-d")." 23:59:59", $selectedAirline, $selectedAirport),
//            "flightsOPSTomorrow"        => Flight::handlingFlightsRange(date("Y-m-d", strtotime($endDate))." 00:00:00", $endDate, $selectedAirline, $selectedAirport),

            "json"                      => json_encode($jsonData),

            "supCrew"                   => [], //$supCrew->listModelVariable(['type', 'id'], true),
            "handlingAirports"          => Airport::getHandlingStations($airlineModule ? "icao" : "iata", "id", true),
            "handlingAirlines"          => Airline::listHandlingAirlines("picture", "id"),

            "airlines"            => ["" => "Select"] + Airline::listHandlingAirlinesWithCode(),
            "locationsAirports"   => Airport::listHandlingAirports($selectedAirlineIds),
            "flights"             => $flights,
            "flightTypes"         => ["" => "Select"] + FlightType::pluck('type', 'id')->all(),
            "airports"            => Airport::listAirportsWithCode(),
            "diversionAirports"   => Airport::listDiversionAirportsWithCode(),
            "aircraft"            => ["" => "Select Aircraft"] + $aircraftList,
            "aircraftTypes"       => ["" => "Select AC Type"] + $aircraftTypeList,
            "delays"              => Delay::GetDelayWithAirlineList(),
            "reasons"             => ['' => "Select", 'head_order' => 'Head Order', 'roster_not_received' => 'Roster Not Received', 'sick' => 'Sick', 'no_show' => 'No Show'],

//            "flightNumbers"       => ["" => "Select Flight Number"] + FlightNumber::getArrayFlightNumbersWithSector(),
            "flightNumbers"       => ["" => "Select Flight Number"] + $flightNumbers,

            "flightNumbersObj"    => $flightNumbersObj,
            "aircraftByAirline"   => Aircraft::getAircraftByAirline($selectedAirlineIds),
            "acTypesByAirline"    => AircraftType::byAirline($selectedAirlineIds),
            "aircraftByType"      => Aircraft::byType($selectedAirlineIds),

            "captains"            => ["" => "Select Captain"] + User::getFlightCrew(FCM_CP_CREW),
            "firstOfficers"       => ["" => "Select First Officer"] + User::getFlightCrew(FCM_FO_CREW),
            "cabinCrew"           => ["" => "Select Flight Attendant"] + User::getFlightCrew(CCM_CREW),
        ];

        if (airlineModule()){
            $this->viewData["staff"]  = User::listUsers();
        }
        else {
            $this->viewData["staff"] = User::listUsers(false);
        }


        return view("flight-watch/index", $this->viewData);
    }

    public static function getFlightTrackerPanelFlights($flights){

        $flights = $flights->sortBy("std")->all();

        $data = [
            ALL         => [],
            DEPARTURE   => [],
            ARRIVAL     => [],
        ];

        //$today = date("Y-m-d");
        $handlingStations = Airport::getHandlingStationsIdAndObject();

        foreach ($flights as $flight) {
            //$date = date("Y-m-d", strtotime(getFlightDepartureDate($flight)));

            //if ($date == $today){
            $data[ALL][] = $flight;

            $type = getFlightsDepArrType($flight, $handlingStations);
            if ($type){
                $data[$type][] = $flight;
            }
            //}
        }

        return $data;
    }

    public function setStaffAirlines($airline){
        $user = Auth::user();

        if ($airline) {

            $staffAirline = StaffAirline::where("user_id", $user->id)
                ->where("airline_id", $airline)
                ->first();

            if ($staffAirline){
                // Audit Deleted
                auditDeleted($staffAirline);
//                $staffAirline->delete();
            }
            else {
                $staffAirline = new StaffAirline();
                $staffAirline->user_id = $user->id;
                $staffAirline->airline_id = $airline;
                $staffAirline->save();
            }
        }
        else {
            $sAirlines = StaffAirline::where("user_id", $user->id)
                ->get();
            // Audit Deleted
            auditDeleted($sAirlines);

        }
    }

    public function setStaffStations($airport){
        $user = Auth::user();

        if ($airport) {
            $staffStation = StaffStation::where("user_id", $user->id)
                ->where("airport_id", $airport)
                ->first();

            if ($staffStation){
                // Audit Deleted
                auditDeleted($staffStation);
//                $staffStation->delete();
            }
            else {
                $staffStation = new StaffStation();
                $staffStation->user_id = $user->id;
                $staffStation->airport_id = $airport;
                $staffStation->save();
            }

        }
        else {
            $sStations = StaffStation::where("user_id", $user->id)
                ->get();
            // Audit Deleted
            auditDeleted($sStations);

        }

    }

    public static function ajaxGetRange(Request $request, $from = null, $to = null)
    {
        $airlineModule = airlineModule();

        if ($from && $to) {
            $startDate  = date("Y-m-d H:i:s", strtotime($from));
            $endDate    = date("Y-m-d H:i:s", strtotime($to));
//            $startDate  = date("Y-m-d H:i:s", strtotime("-12 hours", strtotime($from)));
//            $endDate    = date("Y-m-d H:i:s", strtotime("+12 hours", strtotime($to)));
        }
        else {
            $startDate  = date("Y-m-d H:i:s", strtotime($request->get("start_date")));
            $endDate    = date("Y-m-d H:i:s", strtotime($request->get("end_date")));
//            $startDate  = date("Y-m-d H:i:s", strtotime("-12 hours", strtotime($request->get("start_date"))));
//            $endDate    = date("Y-m-d H:i:s", strtotime("+12 hours", strtotime($request->get("end_date"))));
        }

        $selectedAirlineIds = Airline::getSelectedAirlineIds($request);
        $selectedAirportIds = Airport::getSelectedAirportIds($request);

        $aircraftMaintenance = Maintenance::maintenanceRange($startDate, $endDate);

        $emptyFlights = self::emptyFlights();

        if ($emptyFlights){
            $flights = $jsonData = [];

            $flightsOPS = $flightTables = [
                DEPARTURE   => null,
                ARRIVAL     => null,
                ALL         => null,
            ];
        }
        else {
            // Load Flights
            $flights = Flight::handlingFlightsRange($startDate, $endDate, $selectedAirlineIds, $selectedAirportIds);

            // Flight-watch Ready Json
            $jsonData = self::getJsonDataForFlightWatchTimeLine($flights, $aircraftMaintenance);

            $flightsOPS = self::getFlightTrackerPanelFlights($flights);

            $tableLayout = $airlineModule ? "flights-list-airline" : "flights-list";

            $flightTables = [
                DEPARTURE   => view("flight-tracker/layouts/{$tableLayout}", ["flightsOPS" => $flightsOPS[DEPARTURE], "tableId" => "tableFltDepartures"])->render(),
                ARRIVAL     => view("flight-tracker/layouts/{$tableLayout}", ["flightsOPS" => $flightsOPS[ARRIVAL],   "tableId" => "tableFltArrivals"])->render(),
                ALL         => view("flight-tracker/layouts/{$tableLayout}", ["flightsOPS" => $flightsOPS[ALL],       "tableId" => "tableFltAll"])->render(),
            ];
        }


        $flightNumbersObj = FlightNumber::getArrayFlightNumbersWithSector(false, false, false, null, true);

        return response()->json([
            "jsonData"                  => $jsonData,
            "flights"                   => $flights,

            "aircraftByAirline"   => Aircraft::getAircraftByAirline($selectedAirlineIds),
            "acTypesByAirline"    => AircraftType::byAirline($selectedAirlineIds),
            "aircraftByType"      => Aircraft::byType($selectedAirlineIds),


            "flightsOPS"                => isset($flightsOPS) ? $flightsOPS : null,
            "flightTables"              => isset($flightTables) ? $flightTables : null,
            "success"                   => TRUE,

            // ADDED FOR UPDATE
            "airlines"                  => Airline::getHandlingAirlinesAndOperatorsWithCode(),
            "flightNumbersObj"          => $flightNumbersObj,
        ]);
    }

    public function quickLink(Request $request){
        $startDate = date("Y-m-d H:i:s", strtotime($request->get("start_visible_time")));
        $endDate   = date("Y-m-d H:i:s", strtotime($request->get("end_visible_time")));

        $selectedAirlineIds = Airline::getSelectedAirlineIds($request);
        $selectedAirportIds = Airport::getSelectedAirportIds($request);

        $aircraftMaintenance = Maintenance::maintenanceRange($startDate, $endDate);

        // Load Flights
        $flights =  Flight::handlingFlightsRange($startDate, $endDate, $selectedAirlineIds, $selectedAirportIds);

        // Get JSON Ready
        return response()->json([
            "start_visible_period"  => $startDate,
            "end_visible_period"    => $endDate,
            "jsonData"              => self::getJsonDataForFlightWatchTimeLine($flights, $aircraftMaintenance),
            "flights"               => $flights,
            "success"               => TRUE
        ]);
    }

    public function cancel(Request $request){

        $date = date("Y-m-d H:i:s");

        if ($request->get("remove_reason")){

            $flight = Flight::find($request->get("flight_id"));
            if (!$flight){
                return redirect()->to("flight-watch");
            }

            $flight->remark         = $request->get("remove_reason");
            $flight->deleted_at     = date("Y-m-d H:i:s");
            $flight->updated_by     = Auth::user()->id;
            $flight->save();

            flash()->success('Flight Successfully Removed');

            $date = $this->getFlightDate($flight);
        }
        else if ($request->get("cancel_reason")){

            $flight = Flight::find($request->get("flight_id"));
            if (!$flight){
                return redirect()->to("flight-watch");
            }

            $flight->cancellation_received_at   = $request->get("cancellation_received_at");
            $flight->remark         = $request->get("cancel_reason");
            $flight->cancelled_at   = date("Y-m-d H:i:s");
            $flight->sod_completed  = $request->get("sod_completed");
            $flight->updated_by     = Auth::user()->id;
            $flight->save();

            flash()->success('Flight Successfully Cancelled');

            $date = $this->getFlightDate($flight);
        }

        $startDate = $endDate = null;

        if ($date) {
            $startDate = date("Y-m-d H:i:s", strtotime("-12 hours", strtotime($date)));
            $endDate = date("Y-m-d H:i:s", strtotime("+12 hours", strtotime($date)));
        }

        return $this::ajaxGetRange($request, $startDate, $endDate);
    }

    protected $depTimings = [
        STD,
        PTD,
        ETD,
        ATD,
        ABN,
    ];

    protected $arrTimings = [
        STA,
        PTA,
        ETA,
        ATA,
        TDN,
    ];

    function depArrTimingsExists(Request $request){
        $depExists = $arrExists = false;
        foreach ($this->depTimings as $timing) {
            if ($request->get($timing)){
                $depExists = true;
            }
        }

        foreach ($this->arrTimings as $timing) {
            if ($request->get($timing)){
                $arrExists = true;
            }
        }

        return $depExists && $arrExists;
    }

    public function create(Request $request, IFlightDelayRepository $flightDelayRepository)
    {
        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '64M');

        $url = "";
//        $start_time = $request->get("start_visible_period");
//        $end_time   = $request->get("end_visible_period");
        /*
                if ($request->has("start_visible_period"))
                {
                    $url    = "?start_time=$start_time&end_time=$end_time";
                }
        */

        $form = new FlightForm();

        if (!$this->depArrTimingsExists($request)){
            return response()->json([
                "message"   => "Please fill out departure/arrival times",
                "error"     => true,
                "success"   => false,
            ]);
        }

        If ($request->has("restore_flight")){

            $flight = Flight::findOrFail($request->get("flight_id"));
            if (!$flight){
                return redirect()->to("fleetwatch/index$url")
                    ->withErrors("Flight Not Found");
            }

            $flight->cancelled_at = null;
            $flight->updated_by = Auth::user()->id;
            $flight->save();
//                return redirect()->to("fleetwatch/index$url");
        }

        else {
            if ($request->get("flight_id")) {
                if ($form->isValidForEdit(true)) {
                    $flight = Flight::findOrFail($request->get("flight_id"));
                }
            }
            else {
                if ($form->isValidForAdd(true)) {
                    $flight = new Flight;
                    $flight->created_by = Auth::user()->id;
                }
            }

            if (!isset($flight)) {
                return redirect()->back()
                    ->withInput()
                    ->withErrors($form->getErrors());
            }

            // get delay array values

            // Get Submitted Flight Number Id

            // If Manual FLight Number is Set
            if ($request->has('manual_flt_no')) {
                $bound = $this->GetFlightBound($request->get('departure_airport'), $request->get('arrival_airport'));
                $flightNumber = FlightNumber::createManualFlightNumber($bound);
                $flightNumberId = $flightNumber->id;
            }
            else {
                $flightNumberId = $request->get("flight_number_id");
                $flightNumber = FlightNumber::find($flightNumberId);
            }

            $airline = isset($flightNumber) && $flightNumber ? $flightNumber->airline : null;

            $aircraftType = null;
            if ($request->has('manual_ac_type') && $request->get("ac_type")) {
                $manualAcType = $request->get("ac_type");
                /*
                $aircraftType = AircraftType::where("iata_name", $manualAcType)
                                    ->orWhere("name", $manualAcType)
                                    ->first();
                */

                $aircraftType = AircraftType::where(function($sql) use ($manualAcType){
                    $sql->where("iata_name", $manualAcType)
                        ->orWhere("icao", $manualAcType)
                        ->orWhere("name", $manualAcType);
                })
                    ->where(function($sql) use ($airline){
                        $sql->whereNull("airline_id");
                        if ($airline){
                            $sql->orWhere("airline_id", $airline->id);
                        }
                    })
                    ->orderByDesc("airline_id")
                    ->first();

                if (!$aircraftType){
                    $aircraftType = new AircraftType();
                    if (airlineModule()) {
                        $aircraftType->icao = $manualAcType;
                    }
                    else {
                        $aircraftType->iata_name = $manualAcType;
                    }
                    $aircraftType->name = $manualAcType;
                    $aircraftType->airline_id = $airline ? $airline->id : null;
                    $aircraftType->created_by = Auth::user()->id;
                    $aircraftType->save();
                }
            }
            elseif ($request->get("aircraft_type_id")) {
                $aircraftType = AircraftType::find($request->get("aircraft_type_id"));
            }


            if ($request->has('manual_ac') && $request->get("ac")) {
                $aircraft = self::getAircraft($request->get("ac"), $airline, $aircraftType);
            }
            else {
                $aircraft = Aircraft::find($request->get("aircraft_id"));
            }

            // Get Properties
            $properties = $this->getProperties($flight, $flightNumberId, $aircraftType, $aircraft);

            // Update Flight Details
            $scheduleTimingsChanged = $this->UpdateFlight($flight, $properties);

            // Update Flight Delays
            Flight::UpdateFlightDelays($flight, $flightDelayRepository);

            FlightCrew::createChangeCrewProcedure($flight->id, NULL, CREW_CAPTAINS, $request->get('captain'), $request->get('captain_report_time'), $request->get('captain_reason'));
            FlightCrew::createChangeCrewProcedure($flight->id, NULL, CREW_CAPTAINS, $request->get('captain_standby'), $request->get('captain_standby_report_time'), $request->get('captain_standby_reason'), true);
            FlightCrew::createChangeCrewProcedure($flight->id, NULL, CREW_CAPTAINS, $request->get('captain_dhc'), $request->get('captain_dhc_report_time'), $request->get('captain_dhc_reason'), false, true);

            FlightCrew::createChangeCrewProcedure($flight->id, NULL, CREW_FIRST_OFFICERS, $request->get('first_officer'), $request->get('first_officer_report_time'), $request->get('first_officer_reason'));
            FlightCrew::createChangeCrewProcedure($flight->id, NULL, CREW_FIRST_OFFICERS, $request->get('first_officer_standby'), $request->get('first_officer_standby_report_time'), $request->get('first_officer_standby_reason'), true);
            FlightCrew::createChangeCrewProcedure($flight->id, NULL, CREW_FIRST_OFFICERS, $request->get('first_officer_dhc'), $request->get('first_officer_dhc_report_time'), $request->get('first_officer_dhc_reason'), false, true);

            FlightCrew::createChangeCrewProcedure($flight->id, NULL, CREW_PURSER_AND_CABIN_CREW, $request->get('cabin_crew'), $request->get('cabin_crew_report_time'), $request->get('cabin_crew_reason'));
            FlightCrew::createChangeCrewProcedure($flight->id, NULL, CREW_PURSER_AND_CABIN_CREW, $request->get('cabin_crew_standby'), $request->get('cabin_crew_standby_report_time'), $request->get('cabin_crew_standby_reason'), true);
            FlightCrew::createChangeCrewProcedure($flight->id, NULL, CREW_PURSER_AND_CABIN_CREW, $request->get('cabin_crew_dhc'), $request->get('cabin_crew_dhc_report_time'), $request->get('cabin_crew_dhc_reason'), false, true);

            // SNY
            FlightCrew::createChangeCrewProcedure($flight->id, NULL, STAFF_SNY, $request->get('cabin_crew_sup'), $request->get('cabin_crew_sup_report_time'), $request->get('cabin_crew_sup_reason'), false, false, true, $request->get('cabin_crew_sup_type'));

            // Create Diverted Flights
            //Flight::manageDiversion($flightRepository, $flight, $flightNumber, $properties, $previouslyDiverted);

            // Process Diversion
            $this->processDiversion($request, $flight);

            debug($request->all());
            // Update Flight Staff
            if ($scheduleTimingsChanged){
                debug("SCHEDULE TIMINGS CHANGED");
                Flight::updateFlightServiceData($flight);
            }
            else {
                FlightStaff::updateFlightsStaff($request, $flight);
            }

            // Catering
            $this->processCatering($flight, $request);
            //End

            // Stations
            $this->processStations($flight, $request);
            // End Stations

            // Send Delay Notification
            Parse::sendDelayEmailNotification($flight);
        }

        $startDate = $endDate = null;

        if ($date = $this->getFlightDate($flight)) {
            $startDate = date("Y-m-d H:i:s", strtotime("-12 hours", strtotime($date)));
            $endDate = date("Y-m-d H:i:s", strtotime("+12 hours", strtotime($date)));
        }

        return $this::ajaxGetRange($request, $startDate, $endDate);
    }

    function processCatering($flight, Request $request){
        if ($request->get("std_meals_a") || $request->get("std_meals_c") || $request->get("std_meals_w") || $request->get("std_meals_y")
            || $request->get("spcl_meals_a") || $request->get("spcl_meals_c") || $request->get("spcl_meals_w") || $request->get("spcl_meals_y")
            || $request->get("meals_a") || $request->get("meals_c") || $request->get("meals_w") || $request->get("meals_y") ){
            $flightCatering = $flight->catering;

            if (!$flightCatering){
                $flightCatering = FlightCatering::create([
                    "flight_id"     => $flight->id,
                ]);
            }

            $flightCatering->std_meals_a = $request->get("std_meals_a");
            $flightCatering->std_meals_c = $request->get("std_meals_c");
            $flightCatering->std_meals_w = $request->get("std_meals_w");
            $flightCatering->std_meals_y = $request->get("std_meals_y");

            $flightCatering->spcl_meals_a = $request->get("spcl_meals_a");
            $flightCatering->spcl_meals_c = $request->get("spcl_meals_c");
            $flightCatering->spcl_meals_w = $request->get("spcl_meals_w");
            $flightCatering->spcl_meals_y = $request->get("spcl_meals_y");

            $flightCatering->meals_a = $request->get("meals_a");
            $flightCatering->meals_c = $request->get("meals_c");
            $flightCatering->meals_w = $request->get("meals_w");
            $flightCatering->meals_y = $request->get("meals_y");
            $flightCatering->save();
        }
        else {
            $flight->catering()->delete();
        }
    }

    function processStations($flight, Request $request){
        if ($flightStation = $request->get("station")){
            $stationAirport = $request->get("station_airport");
            $pax = [
                "a"     => request("station_pax_a"),
                "c"     => request("station_pax_c"),
                "w"     => request("station_pax_w"),
                "y"     => request("station_pax_y"),
                "inf"   => request("station_pax_inf"),
            ];
            $paxBooked = [
                "a"     => request("station_pax_a_bkd"),
                "c"     => request("station_pax_c_bkd"),
                "w"     => request("station_pax_w_bkd"),
                "y"     => request("station_pax_y_bkd"),
                "inf"   => request("station_pax_inf_bkd"),
            ];
            $baggage = [
                "baggage"       => request("station_baggage"),
                "baggage_pcs"   => request("station_baggage_pcs"),
                "cargo"         => request("station_cargo"),
                "eic"           => request("station_eic"),
                "mail"          => request("station_mail"),
            ];
            foreach ($flightStation as $i => $id){
                $station = FlightStation::find($id);
                if (!$station){
                    $station = FlightStation::create([
                        "flight_id"         => $flight->id,
                        "flight_number_id"  => $flight->flight_number_id,
                        "airport_id"        => $stationAirport[$i],
                    ]);
                }

                $station->baggage = $baggage["baggage"][$i];
                $station->baggage_pcs = $baggage["baggage_pcs"][$i];
                $station->cargo = $baggage["cargo"][$i];
                $station->eic = $baggage["eic"][$i];
                $station->mail = $baggage["mail"][$i];

                $station->pax_a = $pax["a"][$i];
                $station->pax_c = $pax["c"][$i];
                $station->pax_w = $pax["w"][$i];
                $station->pax_y = $pax["y"][$i];
                $station->pax_inf = $pax["inf"][$i];

                $station->pax_a_bkd = $paxBooked["a"][$i];
                $station->pax_c_bkd = $paxBooked["c"][$i];
                $station->pax_w_bkd = $paxBooked["w"][$i];
                $station->pax_y_bkd = $paxBooked["y"][$i];
                $station->pax_inf_bkd = $paxBooked["inf"][$i];
                $station->save();
            }
        }
    }

    public function saveLine(Request $request){

        $airlineModule = airlineModule();

        $airline = $request->get("airline");
        $flightIDs = $request->get("flight_ids");

        $aircraftType = null;
        if ($manualAcType = $request->get("ac_type")) {
            $aircraftType = AircraftType::where(function($sql) use ($manualAcType){
                $sql->where("iata_name", $manualAcType)
                    ->orWhere("icao", $manualAcType)
                    ->orWhere("name", $manualAcType);
            });

            if ($airline){
                $aircraftType->where(function($sql) use ($airline){
                    $sql->whereNull("airline_id");
                    if ($airline){
                        $sql->orWhere("airline_id", $airline->id);
                    }
                });
            }

            $aircraftType = $aircraftType->orderByDesc("airline_id")
                ->first();

            if (!$aircraftType){
                $aircraftType = new AircraftType();
                if ($airlineModule) {
                    $aircraftType->icao = $manualAcType;
                }
                else {
                    $aircraftType->iata_name = $manualAcType;
                }
                $aircraftType->name = $manualAcType;
                $aircraftType->airline_id = $airline ? $airline->id : null;
                $aircraftType->created_by = Auth::user()->id;
                $aircraftType->save();
            }
        }
        elseif ($request->get("aircraft_type_id")) {
            $aircraftType = AircraftType::find($request->get("aircraft_type_id"));
        }

        if ($manualAc = $request->get("ac")) {
            $aircraft = self::getAircraft($manualAc, $airline, $aircraftType);
        }
        else {
            $aircraft = Aircraft::find($request->get("aircraft_id"));
        }

        if (!$aircraft && !$aircraftType){
            return response()->json([
                "message"   => "Please select Aircraft Type or Aircraft Registration",
                "error"     => true,
                "success"   => false,
            ]);
        }

        $flight = null;
        debug($flightIDs);
        foreach ($flightIDs as $flightID) {
            $flight = Flight::find($flightID);

            if ($manualAc){
                $flight->aircraft_id = $aircraft->id;
                $flight->aircraft_type_id = $aircraftType->id;
            }
            else if ($manualAcType){
                $flight->aircraft_id = null;
                $flight->aircraft_type_id = $aircraftType->id;
            }
            else {
                if ($aircraft){
                    $flight->aircraft_id = $aircraft->id;
                }
                else {
                    $flight->aircraft_type_id = $aircraftType->id;
                }
            }

            $flight->save();
        }

        $startDate = $endDate = date("Y-m-d");
        if ($flight && $date = $this->getFlightDate($flight)) {
            $startDate = date("Y-m-d H:i:s", strtotime("-12 hours", strtotime($date)));
            $endDate = date("Y-m-d H:i:s", strtotime("+12 hours", strtotime($date)));
        }

        return $this::ajaxGetRange($request, $startDate, $endDate);

    }

    public static function getAircraft($acReg, $airline = null, $aircraftType = null){
        $mvtName = str_replace("-","", $acReg);

        $aircraft = Aircraft::whereNull("deleted_at")
            ->where(function($sql) use ($mvtName, $acReg){
                $sql->where("mvt_name", $mvtName)
                    ->orWhere("name", $acReg);
            });

        if ($airline){
            $aircraft->where("airline_id", $airline->id);
        }

        $aircraft = $aircraft->first();

        if (!$aircraft){
            $aircraft = new Aircraft();

            $aircraft->mvt_name = contains($mvtName, "-") ? str_replace("-", "", $mvtName) : $mvtName;
            $aircraft->name =     contains($acReg, "-") ? $acReg : partitionAircraftCode($acReg);

            if ($airline){
                $aircraft->airline_id = $airline->id;
            }

            $aircraft->created_by = Auth::user()->id;

            if ($aircraftType){
                if (airlineModule()){
                    if (!$airline || !$aircraftType->airline_id){
                        $aircraft->aircraft_type_id = $aircraftType->id;
                    }
                }
                else {
                    if (!$aircraftType->airline_id || ($airline && $aircraftType->airline_id == $airline->id)){
                        $aircraft->aircraft_type_id = $aircraftType->id;
                    }
                }
            }
        }
        $aircraft->deleted_at = null;
        $aircraft->save();

        return $aircraft;
    }

    public function processDiversion(Request $request, $flight){

        if ($request->has("is_diversion")){
            // Get Airport
            $divertedAirport = $this->getOrCreateAirport($request->get("diversion_airport"));

            // Flight to original destination
            $flightToOriginalDestination = $this->createFlightToOriginalDestination($request, $divertedAirport, $flight);

            // Update Current Flight
            $flight->is_diversion = true;
            $flight->arrival_airport_id = $divertedAirport->id;
            $flight->diverted_airport_id = $divertedAirport->id;
            $flight->next_flight_id = $flightToOriginalDestination ? $flightToOriginalDestination->id : null;
        }
        elseif ($request->get("diverted_flight") && $flight->departure_airport_id) {
            // DONT DO ANYTHING
        }
        else {

            if ($flight->nextFlight){
                $flight->next_flight_id = null;
                $flight->nextFlight->deleted_at = date("Y-m-d H:i:s");
                $flight->nextFlight->save();
            }

            $flight->is_diversion = null;
            $flight->diverted_airport_id = null;
            $flight->arrival_airport_id = null;
        }

        $flight->save();
    }

    public function createFlightToOriginalDestination(Request $request, $divertedAirport, $divertedFlight){

        $departures = [ PTD, ETD, ATD, ABN ];
        $arrivals =   [ PTA, ETA, TDN, ATA ];

        $foundDeparture = $foundArrival = false;

        $flight = Flight::where("prev_flight_id", $divertedFlight->id)
            ->first();

        // Create if not found
        if (!$flight) {
            $flight = new Flight();
        }

        // Update key fields
        $flight->flight_number_id = $divertedFlight->flight_number_id;
        $flight->aircraft_id = $divertedFlight->aircraft_id;
        $flight->aircraft_type_id = $divertedFlight->aircraft_type_id;

        $flight->is_diversion = true;
        $flight->departure_airport_id = $divertedAirport->id;
        $flight->diverted_airport_id = $divertedAirport->id;
        $flight->prev_flight_id = $divertedFlight->id;

        // Departure info
        foreach ($departures as $each) {
            if ($request->get("diversion_{$each}")){
                $flight->{$each} = $request->get("diversion_{$each}");
                $foundDeparture = true;
            }
        }

        foreach ($arrivals as $each) {
            if ($request->get("diversion_{$each}")){
                $flight->{$each} = $request->get("diversion_{$each}");
                $foundArrival = true;
            }
        }

        if (!$foundDeparture){
            $flight->etd = Add_Minutes_To_DateTime(getFlightArrivalDate($divertedFlight), 30);
        }

        if (!$foundArrival){
            $flight->eta = Add_Minutes_To_DateTime(getFlightDepartureDate($flight), 120);
        }

        setFlightDepartureDate($flight);
        setFlightArrivalDate($flight);

        $flight->deleted_at = null;
        $flight->save();

        return $flight;
    }

    public function updateOrCreateNextFlight(){

    }

    public function getOrCreateAirport($diversionAirportIATA){

        $diversionAirport = Airport::where("iata", strtoupper($diversionAirportIATA))
            ->first();

        if (!$diversionAirport){
            $diversionAirport = new Airport();
            $diversionAirport->iata = strtoupper($diversionAirportIATA);
            $diversionAirport->save();
        }

        return $diversionAirport;
    }

    public function getFlightDate($flight){
        if ($flight->atd && $flight->atd != EMPTY_DATETIME){
            $date = $flight->atd;
        }
        elseif ($flight->etd && $flight->etd != EMPTY_DATETIME){
            $date = $flight->etd;
        }
        elseif ($flight->ptd && $flight->ptd != EMPTY_DATETIME){
            $date = $flight->ptd;
        }
        else {
            $date = $flight->std;
        }

        return $date;
    }

    public function ajaxSearchAirport(){
        $search = \request()->get("key");

        $airports = Airport::where("iata", "LIKE", "%$search%")
            ->pluck("iata")
            ->all();

        echo json_encode($airports);
    }

    public function ajaxMVTDeparture(Request $request){

        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '64M');

        // Prepare data
        $data = $this->prepareMVTDeparture($request);

        // Find Flight
        $flight = Flight::with([
            'flightNumber',
            'flightNumber.airline'
        ])->findOrFail($request->get("mvt_flight_id"));

        // Stop -> if flight not found
        if (!$flight){
            return response()->json([
                "success"        => false,
                "message"        => "Flight ID not set",
            ]);
        }

        // Get Airline
        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber && $flightNumber->airline ? $flightNumber->airline : null;

        // Set AC
        $aircraft = self::getAircraft($data["aircraft"], $airline);

        // Edited on Aug 6 2019
//        $date = getFlightDepartureDate($flight, true);
        $date = getFlightDepartureInitialDate($flight, true);

        // Set Flight Delay Data
        self::setFlightDelayData($flight, $data, true);

        $parseOperations = new ParseOperations();
        $parseOperations->mvtType = MVT_DEPARTURE;
        $parse = new ParseHelper($parseOperations);

        $success = false;

        // Don't update DIV messages
        if ($flight->is_diversion && $flight->arrival_airport_id && $flight->diverted_airport_id){
            $parseOperations->mvtType = MVT_DIVERSION;
        }
//        else {
        $success =  $parse->updateFlightInformation($flight, $data, $aircraft, $date);
//        }

        // Update Flight
        //Parse::updateFlightInformation($flight, $data, $aircraft, $date, MVT_DEPARTURE);

        // Set Delays
        //Flight::SetFlightDelays($flight, $data['delayCodes'], $data['delayTimes']);

        // Prepare Message
        list($subject, $content) = prepareMVTDepartureText($flight, $data['mvt_dep_type'], $data["si"], $request->get("mvt_correction")); // $this->parseOperations->emailBody

        // Send Message
        Parse::forwardMVTToStations( $subject, $content, $flightNumber,
            $parseOperations->hubStations, $parseOperations->hubEmails, $parseOperations->stationEmails );

        if ($data['mvt_dep_type'] == 'ad'){
            // Save Message
            // !$success => $skipped
            Parse::createMessage($content, MVT, $flight, $flightNumber, $data, $parseOperations->mvtType, null, !$success);
        }

        // Update record
        $flight = Flight::with([
            'aircraft',
            'flightNumber',
            'flightNumber.airline'
        ])->find($flight->id);

        // Send message for aircraft email group
        Parse::forwardMVTToAircraft($subject, $content, $flight->aircraft);


        $startDate = $endDate = null;

        if ($date = $this->getFlightDate($flight)) {
            $startDate = date("Y-m-d H:i:s", strtotime("-12 hours", strtotime($date)));
            $endDate = date("Y-m-d H:i:s", strtotime("+12 hours", strtotime($date)));
        }

        return $this::ajaxGetRange($request, $startDate, $endDate);
    }

    public static function setFlightDelayData($flight, &$data, $toString = false){
        $initDate = getFlightDepartureInitialDate($flight);
        $actDate = getFlightDepartureDate($flight);

        // in minutes
        $diff = Calculate_Duration($initDate, $actDate, true) * 60;

        $delayCodes = $delayTimes = [];
        for($i = 1; $i <= 3; $i++){
            if ($data["dl{$i}_code"]){
                $delayCodes[] = $data["dl{$i}_code"];
                $strDLTimes[] = $data["dl{$i}_time"];
            }
        }

        foreach ($delayCodes as $j => $each ) {
            if (isset($strDLTimes[$j]) && $strDLTimes[$j]){
                $dTime = makeTime(trim($strDLTimes[$j]));
//                debug("REAL: ".$strDLTimes[$j]." CONVERTED:".$dTime);
            }
            else {

                // If 1 delay mentioned assign all delay time to this delay
                if (count($delayCodes) == 1){
                    $dTime = Minutes_To_Time($diff);
                }
                else {
                    if ($j == 0){
                        $dTime = Minutes_To_Time($diff - (count($delayCodes) - 1));
                    }
                    else {
                        $dTime = makeTime("0001");
                    }
                }
            }

            $delayTimes[] = $dTime;
        }

        $data['delayCodes'] = $toString ? implode("/", $delayCodes) : $delayCodes;
        $data['delayTimes'] = $toString ? implode("/", $delayTimes) : $delayTimes;
    }

    public function prepareMVTDeparture(Request $request){

        $depType = $request->get("mvt_dep_type");

        $data = [
            "flight_id"     => $request->get("mvt_flight_id")   ? $request->get("mvt_flight_id")    : null,
            "aircraft"      => $request->get("mvt_ac")          ? $request->get("mvt_ac")           : null,
            "mvt_dep_type"  => $depType,

            "dl1_code"      => $request->get("mvt_dl1_code")    ? $request->get("mvt_dl1_code")     : null,
            "dl1_time"      => $request->get("mvt_dl1_time")    ? $request->get("mvt_dl1_time")     : null,
            "dl2_code"      => $request->get("mvt_dl2_code")    ? $request->get("mvt_dl2_code")     : null,
            "dl2_time"      => $request->get("mvt_dl2_time")    ? $request->get("mvt_dl2_time")     : null,
            "dl3_code"      => $request->get("mvt_dl3_code")    ? $request->get("mvt_dl3_code")     : null,
            "dl3_time"      => $request->get("mvt_dl3_time")    ? $request->get("mvt_dl3_time")     : null,

            "si"            => $request->get("mvt_si")          ? $request->get("mvt_si")           : null,
        ];

        switch($depType){
            case "ed":
                $add = [
                    "etd"           => $request->get("mvt_etd")         ? $request->get("mvt_etd")          : null,
                ];
                break;

            case "ad":
            default:
                $add = [
                    "flight_id"     => $request->get("mvt_flight_id")   ? $request->get("mvt_flight_id")    : null,
                    "aircraft"      => $request->get("mvt_ac")          ? $request->get("mvt_ac")           : null,

                    "atd"           => $request->get("mvt_atd")         ? $request->get("mvt_atd")          : null,
                    "abn"           => $request->get("mvt_abn")         ? $request->get("mvt_abn")          : null,
                    "eta"           => $request->get("mvt_eta")         ? $request->get("mvt_eta")          : null,

                    "paxF"          => $request->get("mvt_pax_a")       ? $request->get("mvt_pax_a")        : null,
                    "paxC"          => $request->get("mvt_pax_c")       ? $request->get("mvt_pax_c")        : null,
                    "paxW"          => $request->get("mvt_pax_w")       ? $request->get("mvt_pax_w")        : null,
                    "paxY"          => $request->get("mvt_pax_y")       ? $request->get("mvt_pax_y")        : null,

                    "paxMvtTotal"   => $request->get("mvt_pax_total")   ? $request->get("mvt_pax_total")    : null,
                    // Same INF
                    "paxMvtInf"     => $request->get("mvt_pax_inf")     ? $request->get("mvt_pax_inf")      : null,
                    "paxInf"        => $request->get("mvt_pax_inf")     ? $request->get("mvt_pax_inf")      : null,

                    "baggage"       => $request->get("mvt_baggage")     ? $request->get("mvt_baggage")      : null,
                    "cargo"         => $request->get("mvt_cargo")       ? $request->get("mvt_cargo")        : null,
                    "mail"          => $request->get("mvt_mail")        ? $request->get("mvt_mail")         : null,
                    "eic"           => $request->get("mvt_eic")        ? $request->get("mvt_eic")         : null,

                    "fob"           => $request->get("mvt_fob")         ? $request->get("mvt_fob")          : null,
                    "tof"           => $request->get("mvt_tof")         ? $request->get("mvt_tof")          : null,
                    "eet"           => $request->get("mvt_eet")         ? $request->get("mvt_eet")          : null,
                    "pic"           => $request->get("mvt_pic")         ? $request->get("mvt_pic")          : null,
                ];

                break;
        }

        return array_merge($data, $add);
    }

    public function ajaxMVTArrival(Request $request){

        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '64M');

        // Prepare data
        $data = $this->prepareMVTArrival($request);

        // Find Flight
        $flight = Flight::with([
            'flightNumber',
            'flightNumber.airline'
        ])->findOrFail($request->get("arr_mvt_flight_id"));

        // Stop -> if flight not found
        if (!$flight){
            return response()->json([
                "success"        => false,
                "message"        => "Flight ID not set",
            ]);
        }

        // Get Airline
        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber ? $flightNumber->airline : null;

        // Set AC
        $aircraft = self::getAircraft($data["aircraft"], $airline);

        // Get Arrival Date
        $date = getFlightArrivalDate($flight, true);

        // Update Flight
//        Parse::updateFlightInformation($flight, $data, $aircraft, $date, MVT_ARRIVAL);

        $parseOperations = new ParseOperations();
        $parseOperations->mvtType = MVT_ARRIVAL;
        $parse = new ParseHelper($parseOperations);

        $success = $parse->updateFlightInformation($flight, $data, $aircraft, $date);

        list($subject, $content) = prepareMVTArrivalText($flight, $data, $request->get("arr_mvt_correction")); // $this->parseOperations->emailBody

        // Send Message
        Parse::forwardMVTToStations($subject, $content, $flightNumber,
            $parseOperations->hubStations, $parseOperations->hubEmails, $parseOperations->stationEmails);


        // Save Message
        // !$success => $skipped
        Parse::createMessage($content, MVT, $flight, $flightNumber, $data, $parseOperations->mvtType, null, !$success);

        // Update record
        $flight = Flight::with([
            'aircraft',
            'flightNumber',
            'flightNumber.airline'
        ])->find($flight->id);

        // Send message for aircraft email group
        Parse::forwardMVTToAircraft($subject, $content, $flight->aircraft);

        //
        $startDate = $endDate = null;

        if ($date = $this->getFlightDate($flight)) {
            $startDate = date("Y-m-d H:i:s", strtotime("-12 hours", strtotime($date)));
            $endDate = date("Y-m-d H:i:s", strtotime("+12 hours", strtotime($date)));
        }

        return $this::ajaxGetRange($request, $startDate, $endDate);
    }

    public function prepareMVTArrival(Request $request){

        $data = [
            "flight_id"     => $request->get("arr_mvt_flight_id")   ? $request->get("arr_mvt_flight_id")    : null,
            "aircraft"      => $request->get("arr_mvt_ac")          ? $request->get("arr_mvt_ac")           : null,
            "tdn"           => $request->get("arr_mvt_tdn")         ? $request->get("arr_mvt_tdn")          : null,
            "ata"           => $request->get("arr_mvt_ata")         ? $request->get("arr_mvt_ata")          : null,

            "rf"            => $request->get("arr_mvt_rf")         ? $request->get("arr_mvt_rf")             : null,
            "pic_arr"       => $request->get("arr_mvt_pic")         ? $request->get("arr_mvt_pic")           : null,
            "si"            => $request->get("arr_mvt_si")         ? $request->get("arr_mvt_si")             : null,
        ];


        return $data;
    }

    public function ajaxGetFlight(Request $request, $flight)
    {
//        $flightId = (int)$request->get("flight_id");

        $relations = [
            "flightParent",
            "station",
            "catering",
            "aircraft",
            "aircraft.type",
            "divertedAirport",
            "arrivalAirport",
            "departureAirport",
            "nextFlight",
            "aircraftType",
            "flightCrew",
            "flightNumber",
            "flightStaff",
            "flightStaff.service",
            "flightStaff.user",
            "flightNumber.airline",
            "flightNumber.departureAirport",
            "flightNumber.arrivalAirport",
            "delays",
            "delays.delay",
            "message",
            "message2018"
        ];

        $airlineModule = airlineModule();

        if ($airlineModule){
            $relations[] = "message2018";
        }

        $flight   = Flight::with($relations)
            ->find($flight->id);

        // if not found flight, print error JSON
        if (!$flight)
        {
            return response()->json([
                "error" => 1,
                "text"  => "Not found flight"
            ]);
        }
        /*
        if ($flight->is_diversion && $flight->diversion)
        {
            $flight->diversion = @unserialize($flight->diversion);
        }
        */
        /*
        if (!$flight->capacity_y && !$flight->capacity_c)
        {
            if ($flight->aircraft){
                $flight->capacity_c = $flight->aircraft->config_c;
                $flight->capacity_y = $flight->aircraft->config_y;
            }
        }
        */

        // Get staff
        $flight->staffList = Flight::getFlightStaff($flight);

        if (count($flight->delays)){
            $delays = [];
            foreach ($flight->delays as $each) {
                $delay = $each->delay;
                if ($delay){
                    $delays[] = "DL ".$delay->code." | ".
                        ($delay->alpha_code ? $delay->alpha_code." | " : "")
                        .date("H:i", strtotime($each->duration))
                        // if Local description exists -> show one
                        .($delay->description_ln ?  " | ". str_limit($delay->description_ln, 110) : ($delay->description ? " | ". str_limit($delay->description, 110) : ""))
                        ." ";
                }
            }
            $flight->delaysString = $delays;
        }
        else if ($flightDelayedTime = isFlightDelayed($flight, ghaModule())){
            $delays[] = "DL | " . $flightDelayedTime . " | UNKNOWN";
            $flight->delaysString = $delays;
            $flight->is_delayed = true;
        }

        $flightNumber = $flight->flightNumber;
        $flight->departure_date_type = getFlightDepartureTimeType($flight, true);
        $flight->arrival_date_type = getFlightArrivalTimeType($flight, true);

        if ($flightNumber) {
            if ($flightNumber->departureAirport && $timezone = $flightNumber->departureAirport->timezone) {
                $depDateTime = getFlightDepartureDate($flight);
                if ($depDateTime && $depDateTime != EMPTY_DATETIME) {
                    $flight->departure_local_date = getLocalDateTime($depDateTime, $timezone, "D d M Y");
                    $flight->departure_local_time = getLocalDateTime($depDateTime, $timezone, "H:i"); }

            }

            if ($flightNumber->arrivalAirport && $timezone = $flightNumber->arrivalAirport->timezone) {
                $arrDateTime = getFlightArrivalDate($flight);
                if ($arrDateTime && $arrDateTime != EMPTY_DATETIME){
                    $flight->arrival_local_date = getLocalDateTime($arrDateTime, $timezone, "D d M Y");
                    $flight->arrival_local_time = getLocalDateTime($arrDateTime, $timezone, "H:i");
                }
            }
        }

        if ($airlineModule){
            $this->setFlightCrew($flight);
        }

        $staffFilter = new StaffFilter($flight, true);

        $flightNumberObj = FlightNumber::getWithSector($flight->flight_number_id);

        return response()->json([
            "success"        => 1,
            "serviceData"    => $staffFilter->getFlightWatchStaffData(),
            "user"           => Auth::user(),
            "flight"         => $flight,
            "departure_city" => isset($departureCity) ? $departureCity : "",
            "arrival_city"   => isset($arrivalCity) ? $arrivalCity : "",
            "flight_number"  => $flightNumberObj,
        ]);
    }

    public static function ajaxGetFlights(Request $request)
    {
        $airport = $request->get("airport_id");

        $startDate = $request->get("date")." 00:00:00";
        $endDate = $request->get("date")." 23:59:59";

        // Load Flights
        $flights = Flight::handlingFlightsRange($startDate, $endDate, $request->get("airline_id"), $airport);

        $jsonData = Flight::getFlightsStaffServiceData($flights, $airport);

        return response()->json([
            "jsonData"      => $jsonData,
            "success"       => TRUE
        ]);
    }

    public function setFlightCrew(&$flight){
        if ($flight->flightCrew()->count())
        {
            // Remove Duplicates
            FlightCrew::removeDuplicateCrew($flight->id);

            $flightCrew = $flight->flightCrew()
                ->whereNull('flights__crew.deleted_at')
                ->get();

            $captains   = $firstOfficers = $flightAttendants = [
                'general' => [],
                'standby' => [],
                'dhc'     => [],
            ];

            // Only
            $flightAttendants['sup'] = [];

            foreach ($flightCrew as $flightCrewItem)
            {

//                if (count($user_ids) && in_array($flightCrewItem->id, $user_ids))
//                    continue;
//                else
//                    $user_ids[] = $flightCrewItem->id;

                $flightCrewItemSettings = FlightCrew::select([
                    "user_id",
                    "flight_id",
                    "position_id",
                    "position_order",
                    "report_time",
                    "reason",
                    "is_standby",
                    "is_dhc",
                    "is_sup",
                    "crew_sup_id",
                    "structure__positions.name as position_name",
                    "structure__positions.type as position_type",
                ])
                    ->leftJoin('structure__positions', 'structure__positions.id', '=', 'flights__crew.position_id')
                    ->where("flight_id", $flight->id)
                    ->where("user_id", $flightCrewItem->id)
                    ->whereNull('flights__crew.deleted_at')
                    ->get();

                if (count($flightCrewItemSettings)) {
                    foreach ($flightCrewItemSettings as $each_settings) {

                        $flightCrewItem->settings = clone $each_settings;

                        if ($flightCrewItem->settings->is_sup) {
                            // SUP
                            $this->addToCrew($flightCrewItem, $flightAttendants, $each_settings->position_order);
                        }
                        else {
                            $positionType = trim($each_settings->position_type);

                            if ($positionType == FCM_CPT_TYPE_ID) {
                                $this->addToCrew($flightCrewItem, $captains, $each_settings->position_order);
                            }
                            elseif ($positionType == FCM_FA_TYPE_ID) {
                                $this->addToCrew($flightCrewItem, $firstOfficers, $each_settings->position_order);
                            }
                            else if (in_array($positionType, [CCM_PSR_TYPE_ID, CCM_CC_TYPE_ID])) {
                                $this->addToCrew($flightCrewItem, $flightAttendants, $each_settings->position_order);
                            }
                        }

                    }

                    // Otherwise JSON will treat it as Objects instead of Arrays
                    ksort($captains['general']);
                    ksort($firstOfficers['general']);
                    ksort($flightAttendants['general']);

                    /*$types = ['general', 'standby', 'dhc'];
                    $variables = [ 'captains', 'firstOfficers', 'flightAttendants' ];
                    foreach ($variables as $t => $var) {
                        foreach ($types as $s => $each) {
                            for ($p = 0; $p < count($flightAttendants['general']); $p++) {
                                if (!isset(${$var}[$each][$p])) {
                                    ${$var}[$each][$p] = [];
                                }
                            }
                        }
                    }*/

                    // IF POSITION ORDER MATTERS FOR FLIGHT CREW UNCOMMENT THESE LINES
                    for ($p = 0; $p < count($captains['general']); $p++){
                        if (!isset($captains['general'][$p]))
                            $captains['general'][$p] = [];
                    }
                    for ($p = 0; $p < count($firstOfficers['general']); $p++){
                        if (!isset($firstOfficers['general'][$p]))
                            $firstOfficers['general'][$p] = [];
                    }

                    for ($p = 0; $p < count($flightAttendants['general']); $p++){
                        if (!isset($flightAttendants['general'][$p]))
                            $flightAttendants['general'][$p] = [];
                    }
                }
            }

            $flight->flightCrew = [
                "captains"         => $captains,
                "firstOfficers"    => $firstOfficers,
                "flightAttendants" => $flightAttendants,
            ];
        }
    }

    /**
     * Add item to crew
     * @param $crewItem
     * @param $array
     * @param $positionOrder
     * @return mixed
     */
    private function addToCrew($crewItem, &$array, $positionOrder)
    {
        if ($crewItem->settings->is_sup){
            $array["sup"][] = $crewItem;
        }
        elseif ($crewItem->settings->is_standby)
        {
            $array["standby"][] = $crewItem;
        }
        elseif ($crewItem->settings->is_dhc)
        {
            $array["dhc"][] = $crewItem;
        }
        else {
            $array["general"][(int)$positionOrder] = $crewItem;
        }
    }

    public function UpdateFlight(&$flight, $properties){
        $timings = [
            STD,
            STA,
            PTD,
            PTA,
            ETD,
            ETA,
            ATD,
            ATA,
            ABN,
            TDN,
        ];

        $recordUpdated = false;
        $now = strtotime("now");
        $scheduleTimingsChanged = false;
        foreach ($timings as $prop) {

            if ($flight->{$prop} != \request()->get($prop)){
                // Both STA/... and Flights current STA/... is present and not equal (!=)
                if ($flight->{$prop} && \request()->get($prop)){
                    if (strtotime($flight->{$prop}) != strtotime(\request()->get($prop))){
                        $recordUpdated = true;
                    }
                }
                else {
                    if (! ( (!$flight->{$prop} || $flight->{$prop} == "0000-00-00 00:00:00")
                        && (!\request()->get($prop) || \request()->get($prop) == "0000-00-00 00:00:00")) ){
                        debug($prop);
                        if (! (in_array($prop, [STD, STA]) && !\request()->get($prop))){
                            $recordUpdated = true;
                        }
                    }
                }
            }

            switch($prop){
                case STD:
                case STA:
                case PTD:
                case PTA:
                    if (\request()->get($prop)) {
                        $arr = getFlightArrivalInitialDate($flight);
                        if (($flight->sta && $flight->sta != EMPTY_DATETIME) || ($flight->pta && $flight->pta != EMPTY_DATETIME)){
                            // Editable only when ARR + 24 HRS not passed yet
                            // Editable only when ARR + 96 HRS not passed yet changed it on FEB 7 2023
                            if (!$flight->{$prop} || /*(strtotime($arr) + (96*60*60) > $now &&*/
                                (strtotime(date("Y-m-d H:i", strtotime($flight->{$prop}))) != strtotime(\request()->get($prop))))
                            {
                                $flight->{$prop} = \request()->get($prop).":00";
                                $scheduleTimingsChanged = true;
                            }
                            else {
                                debug($prop."/".$flight->{$prop}. " violates 24hr rule");
                            }
                        }
                        else {
                            if (strtotime(date("Y-m-d H:i", strtotime($flight->{$prop}))) != strtotime(\request()->get($prop)))
                            {
                                $flight->{$prop} = \request()->get($prop).":00";
                                $scheduleTimingsChanged = true;
                            }
                            else {
                                debug("111");
                            }
                        }
                    }
                    break;

                default:
                    if (in_array($prop, [ETA, TDN, ATA])){
                        $DEP = getFlightDepartureDate($flight);

                        if (\request()->get($prop) && \request()->get($prop) != EMPTY_DATETIME && strtotime(\request()->get($prop)) < strtotime($DEP)){
                            debug("{$prop} SKIPPED. ARRIVAL TIME < DEP");
                            break;
                        }
                    }

                    $flight->{$prop} = \request()->get($prop) ? \request()->get($prop).":00" : null;
                    break;
            }
        }

        foreach ($properties as $prop => $value) {

            if ($flight->{$prop} != $value){
                $recordUpdated = true;
            }

            $flight->{$prop} = $value || $value === 0 || $value === "0" ? $value : null;
        }

        if ($recordUpdated){
            $flight->updated_at = date("Y-m-d H:i:s");
            $flight->updated_by = Auth::user()->id;
        }

        if (\request()->get("outbound_flight_number_id")){
            //$flight->parent_id        = \request()->get("outbound_flight_number_id");
        }

        setFlightDepartureDate($flight);
        setFlightArrivalDate($flight);

        // save flight
        $flight->save();

        return $scheduleTimingsChanged;
    }

    public function getProperties($flight, $flightNumberId, $aircraftType = null, $aircraft = null){
        $properties = [
            'aircraft_id'      => $aircraft ? $aircraft->id : null,
//            'aircraft_type_id' => null,
            'flight_number_id' => $flightNumberId,
            'pax_a_booked'     => \request()->get("pax_a_booked"),
            'pax_c_booked'     => \request()->get("pax_c_booked"),
            'pax_w_booked'     => \request()->get("pax_w_booked"),
            'pax_y_booked'     => \request()->get("pax_y_booked"),
            'pax_inf_booked'   => \request()->get("pax_inf_booked"),

            'pax_a_actual'     => \request()->get("pax_a_actual"),
            'pax_c_actual'     => \request()->get("pax_c_actual"),
            'pax_w_actual'     => \request()->get("pax_w_actual"),
            'pax_y_actual'     => \request()->get("pax_y_actual"),
            'pax_inf_actual'   => \request()->get("pax_inf_actual"),

            'baggage'          => \request()->get("baggage"),
            'baggage_pcs'      => \request()->get("baggage_pcs"),
            'cargo'            => \request()->get("cargo"),
            'cargo_pcs'        => \request()->get("cargo_pcs"),
            'mail'             => \request()->get("mail"),
            'eic'              => \request()->get("eic"),
            'deportees'        => \request()->get("deportees"),
            'gate_parking'     => \request()->get("gate_parking"),
            'gate_boarding'    => \request()->get("gate_boarding"),
            'in_tanks'         => \request()->get("in_tanks"),
            'uplifted'         => \request()->get("uplifted"),
            'fob'              => \request()->get("departure_fuel"),
            'utilised'         => \request()->get("utilised"),
            'defuel'           => \request()->get("defuel"),
            // Made changes on 190718
            'rf'               => \request()->get("arrival_fuel"),
            'remark'           => \request()->get("remark"),

            'ckin_num'         => \request()->get("ckin_num"),
        ];

        // IF LDM IS NOT RECEIVED -> UPDATE FLIGHT
        if (airlineModule() || (ghaModule() && !$flight->ldm)){
            $properties['capacity_a'] = \request()->get("capacity_a"); // || \request()->get("capacity_c") === 0 || \request()->get("capacity_c") === "0") ? \request()->get("capacity_c") : ($aircraft ? $aircraft->config_c : null),
            $properties['capacity_c'] = \request()->get("capacity_c"); // || \request()->get("capacity_c") === 0 || \request()->get("capacity_c") === "0") ? \request()->get("capacity_c") : ($aircraft ? $aircraft->config_c : null),
            $properties['capacity_w'] = \request()->get("capacity_w"); // || \request()->get("capacity_y") === 0 || \request()->get("capacity_y") === "0") ? \request()->get("capacity_y") : ($aircraft ? $aircraft->config_y : null),
            $properties['capacity_y'] = \request()->get("capacity_y"); // || \request()->get("capacity_y") === 0 || \request()->get("capacity_y") === "0") ? \request()->get("capacity_y") : ($aircraft ? $aircraft->config_y : null),
        }

        // correct utilised fuel
        if ($properties['fob']){
            if ($properties['rf']){
                $properties['utilised'] = ($properties['fob'] - $properties['rf']) >= 0 ? ($properties['fob'] - $properties['rf']) : null;
            }
            else if ($properties['utilised']) {
                $properties['rf'] = ($properties['fob'] - $properties['utilised']) >= 0 ? ($properties['fob'] - $properties['utilised']) : null;
            }

//            $properties['utilised'] = $properties['in_tanks'] + ($properties['uplifted'] ? $properties['uplifted'] : 0) - $properties['rf'];
        }

        if ($aircraftType){
            $properties['aircraft_type_id'] = $aircraftType->id;
        }
        else if ($aircraft && $aircraft->type){
            $properties['aircraft_type_id'] = $aircraft->type->id;
        }

        return $properties;
    }


    public static function getJsonDataForFlightWatchTimeLine(&$flights, $aircraftMaintenance = [])
    {
        $json = [];
        $stations = Airport::getHandlingStations("iata");

        $flightPosition = new FlightPosition($flights, $aircraftMaintenance);
        $flightPosition->run();

        $carrierType = env(CARRIER);
        $gha = ghaModule();

        foreach ($flightPosition->lines as $k => $line) {

            $num = $line->getNum();

            if ($k == 0){
//                debugFlights($line->getFlights());
            }

            foreach ($line->getFlights() as &$flightItem) {

                $flightItem->line = $num;

                $flightNumber = $flightItem->flightNumber;

                // For delay to be colored in flight-watch
                $airlineID = $gha && $flightNumber->airline_id ? $flightNumber->airline_id : null;

                //$aircraftItem = $flightItem->aircraft;
                $airlineItem = $flightNumber ? $flightNumber->airline : null;
                $index = ($num >= 1 ? $num : "");

                $departureDate = getFlightDepartureDate($flightItem);
                $arrivalDate = getFlightArrivalDate($flightItem);

                if (!($departureDate && $arrivalDate)){
                    continue;
                }

                $departureDate = new \DateTime($departureDate);
                $arrivalDate = new \DateTime($arrivalDate);

                $duration = $arrivalDate->diff($departureDate);
                $durationMinutes = $duration->h * 60 + $duration->i;

                $group = self::getFlightPositionGroup($carrierType, $airlineItem, $index, $flightItem);

                $flt = [
                    "start_date" => [
                        $departureDate->format("Y"),
                        (int)$departureDate->format("n") - 1,
                        $departureDate->format("j"),
                        $departureDate->format("G"),
                        (int)$departureDate->format("i"),
                        (int)$departureDate->format("s")
                    ],
                    "end_date" => [
                        $arrivalDate->format("Y"),
                        (int)$arrivalDate->format("n") - 1,
                        $arrivalDate->format("j"),
                        $arrivalDate->format("G"),
                        (int)$arrivalDate->format("i"),
                        (int)$arrivalDate->format("s")
                    ],
                    "group"     => $group,
                    "className" => getFlightClassName($flightItem, $gha, $airlineItem)
                ];

                $depAirport = getFlightDepartureAirport($flightItem);
                $arrAirport = getFlightArrivalAirport($flightItem);

                $departureAirport   = $depAirport ? $depAirport->iata : "";
                $arrivalAirport     = $arrAirport ? $arrAirport->iata : "";

                $type = getAircraftType($flightItem->aircraftType);

                if ($flightItem->aircraft && ($flightItem->aircraft->name || $flightItem->aircraft->mvt_name)){
                    if($flightItem->aircraft->name){
                        $type = $flightItem->aircraft->name;
                    }
                    else {
                        $type = $flightItem->aircraft->mvt_name;
                    }
                }

                $fNo = $flightNumber->airline ? ($flightNumber->airline->iata ? $flightNumber->airline->iata : $flightNumber->airline->icao )." " : "";
                $fNo .= $flightNumber->flight_number;


                $dep_class = in_array($departureAirport, $stations) ? "handling_station" : "";
                $arr_class = in_array($arrivalAirport, $stations)   ? "handling_station" : "";

                if ($durationMinutes < 100) {
                    $flt['content'] = "<div id='flight-" . $flightItem->id . "' class='row fw-flight-item'>" .
                        "<div class='col-sm-12 flightStandard'>" .
                        "<span class='flight_number_box_left {$dep_class}'>" . $departureAirport . "</span>" .
                        "<span class='flight_number_box_center_empty'></span>" .
                        "<span class='flight_number_box_right {$arr_class}'>" . $arrivalAirport . "</span>" .
                        "</div>";
                } else {
                    $flt['content'] = "<div id='flight-" . $flightItem->id . "' class='row fw-flight-item'>" .
                        "<div class='col-sm-12 flightStandard'>" .
                        "<span class='flight_number_box_left {$dep_class}'>" . $departureAirport . "</span>" .
                        "<span class='flight_number_box_center'>" . $fNo ."<br/>".
                        "<span class='flight_number_box_center_row_2'>".$type."</span>".
                        "</span>" .
                        "<span class='flight_number_box_right {$arr_class}'>" . $arrivalAirport . "</span>" .
                        "</div>" .
                        "</div>";
                }

                $json[] = $flt;
            }

            foreach ($line->getMaintenance() as $itemMaintenance) {

                $aircraftItem = $itemMaintenance->aircraft;
                $startDate = new \DateTime($itemMaintenance->arrival_time);
                $endDate = new \DateTime($itemMaintenance->departure_time);

                $airlineItem = $aircraftItem ? $aircraftItem->airline : null;
                $index = ($num >= 1 ? $num : "");

                $group = self::getFlightPositionGroup($carrierType, $airlineItem, $index, $itemMaintenance);

                $json[] = [
                    "start_date" => [
                        $startDate->format("Y"),
                        (int)$startDate->format("n") - 1,
                        $startDate->format("j"),
                        $startDate->format("G"),
                        (int)$startDate->format("i"),
                        (int)$startDate->format("s")
                    ],
                    "end_date" => [
                        $endDate->format("Y"),
                        (int)$endDate->format("n") - 1,
                        $endDate->format("j"),
                        $endDate->format("G"),
                        (int)$endDate->format("i"),
                        (int)$endDate->format("s")
                    ],
                    "content"   => "<div id='maintenance-{$itemMaintenance->id}' class='fw-flight-item'>$itemMaintenance->note ($itemMaintenance->location_code)</div>",
                    "group"     => $group,
                    "className" => "maintenance"
                ];
            }
        }

        /*
        foreach ($aircraftMaintenance as $itemMaintenance) {
            $aircraftItem = $itemMaintenance->aircraft()->first();
            $startDate = new \DateTime($itemMaintenance->arrival_time);
            $endDate = new \DateTime($itemMaintenance->departure_time);

            $json[] = [
                "start_date" => [
                    $startDate->format("Y"),
                    (int)$startDate->format("n") - 1,
                    $startDate->format("j"),
                    $startDate->format("G"),
                    (int)$startDate->format("i"),
                    (int)$startDate->format("s")
                ],
                "end_date" => [
                    $endDate->format("Y"),
                    (int)$endDate->format("n") - 1,
                    $endDate->format("j"),
                    $endDate->format("G"),
                    (int)$endDate->format("i"),
                    (int)$endDate->format("s")
                ],
                "content" => "<div id='maintenance-{$itemMaintenance->id}' class='fw-flight-item'>$itemMaintenance->note ($itemMaintenance->location_code)</div>",
                "group" => $aircraftItem->name,
                "className" => "maintenance"
            ];
        }
        */

        return $json;
    }

    public static function getFlightPositionGroup($carrierType, $airline, $index, $flight = null, $aircraft = null){
        if ($flight && $flight->cancelled_at){
            $group = "<span data-num='".$index."' class='header-cancel'>-Cancelled-</span>";
        }
        else {
            switch($carrierType){
                case AIRLINE:
                    $ac = $flight && $flight->aircraft ? $flight->aircraft : $aircraft;
                    $acType = $ac ? $ac->type : $flight->aircraftType;

                    $group = "<a class='left_sidebar' data-line=".$flight->line." data-ac_id=".($ac ? $ac->id : 0)
                        ." data-ac_type_id=".($acType ? $acType->id : 0).">";

                    /*
                    if ($ac){
                        $group = "<a data-num='".$index."' target='_blank' href='".route("homepage")."#aircraft/{$ac->id}'>";
                    }
                    else if ($flight->aircraft_type_id){
                        $group = "<a data-num='".$index."' target='_blank' href='".route("homepage")."#aircraft?type_id={$flight->aircraft_type_id}'>";
                    }
                    else {
                        $group = "<a data-num='".$index."' target='_blank'>";
                    }
                    */

                    if ($flight){
                        $group .= getACorTypeFromFlight($flight, "icao");
                    }
                    else {
//                        debug($aircraft);
                        $group .= getACorTypeFromAC($aircraft, "icao");
                    }

                    $group .= "</a>";
                    break;

                case HANDLING:
                default:
                    if (!$airline){
                        $group = "<span data-num='".$index."' class='header-cancel'>-Unknown-</span>";
                    }
                    else {
                        $picture = $airline->picture ? asset("storage/airlines/img/{$airline->picture}") : asset("/assets/img/default-airline-picture.png");

                        $group = "<a data-airline='".$airline->airline."' data-num='".$index."' target='_blank' href='".route("homepage")."#". url("/airline/{$airline->id}") ."'>";

                        $group .= "<img class='fleetwatch-airline-logo' src='".$picture."' />";

                        $group .= "</a>";
                    }
                    break;
            }
        }

        return $group;
    }

    public static function flightConflictSearchProcess($flightItem, &$flightsArray)
    {
        $num = 1;

        $index = self::getIndex($flightItem);

        if (!isset($flightsArray[$index])){
            $flightsArray[$index][] = [];
        }

        if (self::isFlightConflicts($flightItem, $flightsArray[$index][0])){
            $found = false;
            $i = 0;
            if (count($flightsArray[$index]) > 1){
                for ($i = 1; $i < count($flightsArray[$index]); $i++){
                    if (isset($flightsArray[$index][$i]) && !self::isFlightConflicts($flightItem, $flightsArray[$index][$i])){
                        $flightsArray[$index][$i][] = $flightItem;
                        $found = true;
                        $num = $i + 1;
                        break;
                    }
                }

                if (!$found){
                    $flightsArray[$index][$i + 1] = [$flightItem];
                    $num = $i + 2;
                }
            }
            else {
                $flightsArray[$index][$i + 1] = [$flightItem];
                $num = $i + 2;
            }
        }
        else {
            $flightsArray[$index][0][] = $flightItem;
        }

        return $num;
    }

    public function setFlightNumbersBounds(){
        $flightNumbers = FlightNumber::with(['departureAirport', 'arrivalAirport'])
            ->whereNull("deleted_at")
            ->get();

        foreach ($flightNumbers as $i => $each ) {
            if ($each->departureAirport && $each->arrivalAirport){
                $bound = $this->GetFlightBound($each->departureAirport->iata, $each->arrivalAirport->iata);
                $flightNumbers[$i]->bound = $bound;
                $flightNumbers[$i]->save();
            }
        }
    }

    public function GetFlightBound($departure, $arrival){
        $stations = Airport::getHandlingStations("iata");
        // Outbound
        if (in_array($departure, $stations)){
            return 0;
        }
        // Inbound
        else if (in_array($arrival, $stations)){
            return 1;
        }
        else {
            return null;
        }
    }

    public static function isFlightConflicts($flight, $flightsArray){
        $departure = strtotime(getFlightDepartureDate($flight));
        $arrival = strtotime(getFlightArrivalDate($flight));

        $lastFlight = end($flightsArray);
        $lastFlightDep =  strtotime(getFlightDepartureDate($lastFlight));
        $lastFlightArr =  strtotime(getFlightArrivalDate($lastFlight));

        if ( ($departure <= $lastFlightArr)
        ){
            return true;
        }
        /*
        foreach ($flightsArray as $each) {
            $eachDep =  strtotime(getFlightDepartureDate($each));
            $eachArr =  strtotime(getFlightArrivalDate($each));

            if ( ($departure >= $eachDep && $departure <= $eachArr) ||
                ($arrival >= $eachDep && $arrival <= $eachArr) ||
                ($departure <= $eachDep && $arrival >= $eachArr)
            ){
                return true;
            }
        }
        */
        return false;
    }

    public static function getIndex($flight){
        return $flight->deleted_at ? "deleted" : ( $flight->cancelled_at ? "cancelled" : ($flight->flightNumber ? $flight->flightNumber->airline_id : "-"));
    }
}
