<?php

namespace App\Http\Controllers;

use App\Classes\FlightStaff\StaffFilter;
use App\Classes\Messages\Message;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Delay;
use App\Models\Flight;
use App\Models\FlightDelay;
use App\Models\FlightNumber;
use App\Models\FlightTracker;
use App\Models\FlightTrackerDetail;
use App\Models\MVTMessage;
use App\Models\OtherMessage;
use App\Models\Passenger;
use App\Models\PNLMessage;
use App\Models\PRLMessage;
use App\Models\Service;
use App\Models\StaffAirline;
use App\Models\StaffStation;
use App\Models\User;
use Illuminate\Http\Request;

use App\Http\Requests;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use OwenIt\Auditing\Models\Audit;

class FlightTrackerController extends Controller
{
    protected $messages = [
        PRL,
        PNL,
        MVT,
        LDM,
        LIR,
        CPM,
        CAL,
        PSM,
        PTM,
        UCM,
        TPM,
        LDS,
        NOTOC,
        ONLOAD,
        OTHS
    ];

    protected $airlineMessages = [
        LPM,
    ];



    /**
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        // Get Staff Handling AP and Airlines
        list($handlingAirlines, $handlingAirports) = getStaffHandlingAirports();

        $selectedAirports = $request->has("airport") ? $request->get("airport") : array_keys($handlingAirports);
        $selectedAirlines = $request->has("airline") ? $request->get("airline") : array_keys($handlingAirlines);

        $dateFrom  = $request->get("dateFrom") && $request->get("dateFrom") != "____-__-__" ?  $request->get("dateFrom")  : date("Y-m-d");
        $dateTo    = $request->get("dateTo")   && $request->get("dateTo")   != "____-__-__" ?  $request->get("dateTo")    : date("Y-m-d");

        $flights = Flight::handlingFlightsRange($dateFrom." 00:00:00", $dateTo." 23:59:59", $selectedAirlines, $selectedAirports, $request->get("flightNumber"));

        $this->viewData = [
            'dateFrom'                  => $dateFrom,
            'dateTo'                    => $dateTo,

            'airports'                  => $handlingAirports,
            'airlines'                  => $handlingAirlines,
            'selectedAirports'          => $selectedAirports,
            'selectedAirlines'          => $selectedAirlines,

            'monthNames'                => getMonthsArray(),
            'years'                     => getYearsArray(),

            "flightNumbers"             => FlightNumber::getArrayFlightNumbersWithSector(false, false, false, null, true),

            'historicFlights'           => $flights->sortBy("std")->all(),
        ];

        return view('flight-tracker/index', $this->viewData);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }


    public function show(Request $request, $flight)
    {
//        $flight->staffList = Flight::getFlightStaff($flight);
//        $this->viewData = [
//            "users"         => User::getListByID(),
//            "services"      => Service::getServicesListByID(),
//            "flight"        => $flight,
//            'staffAudit'    => $this->getStaffAudit($flight),
//        ];
//        return view("flight-tracker.layouts.flight-staff-log-tab", $this->viewData );

        list($handlingAirlines, $handlingAirports) = getStaffHandlingAirports();

        $selectedAirports = $request->has("airport") ? $request->get("airport") : array_keys($handlingAirports);
        $selectedAirlines = $request->has("airline") ? $request->get("airline") : array_keys($handlingAirlines);

        $tomorrow = date("Y-m-d", strtotime("+ 1 days"));
        $minus1month = date("Y-m-d", strtotime("- 1 month", strtotime($tomorrow)));

        $tracker = FlightTracker::where("flight_id", $flight->id)
                                ->first();

        $chartAltGs = [];

        $trackerDetails = FlightTrackerDetail::where("flight_id", $flight->id)
                                            ->orderBy("fch")
                                            ->get();

        foreach ($trackerDetails as $i => $each) {
            $chartAltGs[] = [
                "Date"      => date("Y-m-d H:i", strtotime($each->fch)),
                "Altitude"  => $each->fal / 100,
                "Speed"     => $each->fgs,
            ];
        }

        $historicFlights = Flight::with([
                    "message",
                    "delays",
                    "delays.delay",
                    "aircraftType",
                    "aircraft",
                    "aircraft.type",
                    "departureAirport",
                    "arrivalAirport",
                    "divertedAirport",
                    "flightNumber",
                    "flightNumber.departureAirport",
//                    "flightNumber.departureAirport.city",
//                    "flightNumber.departureAirport.country",
                    "flightNumber.arrivalAirport",
//                    "flightNumber.arrivalAirport.city",
//                    "flightNumber.arrivalAirport.country",
                    "flightNumber.airline",
                ])
                ->whereNull("cancelled_at")
                ->whereNull("deleted_at")
                ->where("flight_number_id", $flight->flight_number_id)
                ->where(function($sql) use ($tomorrow){
                    $sql->where(DB::raw("DATE(std)"), "<=", "{$tomorrow}")
                        ->orWhere(DB::raw("DATE(etd)"), "<=", "{$tomorrow}")
                        ->orWhere(DB::raw("DATE(atd)"), "<=", "{$tomorrow}");
                })
                ->where(function($sql) use ($minus1month){
                    $sql->where(DB::raw("DATE(std)"), ">=", "{$minus1month}")
                        ->orWhere(DB::raw("DATE(eta)"), ">=", "{$minus1month}")
                        ->orWhere(DB::raw("DATE(atd)"), ">=", "{$minus1month}");
                })
                ->orderBy("atd", "desc")
                ->orderBy("etd", "desc")
                ->orderBy("std", "desc")
                ->get();

        if ($flight->delays){
            $delays = [];
            foreach ($flight->delays as $each) {
                $delay = $each->delay;
                if ($delay){
                    $delays[] = "Code: ".$delay->code." | ".str_limit($delay->description, 100);
                }
            }
            $flight->delaysString = $delays;
        }

        $fn1 = $fn2 = $fn3 = "";
        if ($flight->flightNumber){
            $fn = $flight->flightNumber->flight_number;
            if ($flight->flightNumber->airline){
                $airline = $flight->flightNumber->airline;
                $fn1 .= $airline->airline." ";

                $fn2 .= $airline->icao;
                $fn3 .= $airline->iata;
            }
            $fn1 .= $fn;

            if ($fn2){
                $fn2 .= $fn;
            }

            if ($fn3){
                $fn3 .= $fn;
            }
        }

        $departureAirport = $flight->flightNumber ? $flight->flightNumber->departureAirport : null;
        $depLocation = "";
        $selectedAirport = null;
        if ($departureAirport){
            $depLocation .= $departureAirport->city ? $departureAirport->city->city.", " : "";
            $depLocation .= $departureAirport->country ? $departureAirport->country->country : "";

            if (array_key_exists($departureAirport->id, $handlingAirports)){
                $selectedAirport = $departureAirport->id;
            }
        }

        $arrivalAirport = $flight->flightNumber ? $flight->flightNumber->arrivalAirport : null;
        $arrLocation = "";
        if ($arrivalAirport){
            $arrLocation .= $arrivalAirport->city ? $arrivalAirport->city->city.", " : "";
            $arrLocation .= $arrivalAirport->country ? $arrivalAirport->country->country : "";

            if (!$selectedAirport && array_key_exists($arrivalAirport->id, $handlingAirports)){
                $selectedAirport = $arrivalAirport->id;
            }
        }

        $airlineObj = $flight->flightNumber ? $flight->flightNumber->airline : null;

        $receivedMessages = Message::getMessages($flight);
        debug($receivedMessages);

        $flight->staffList = Flight::getFlightStaff($flight);

        $filter = new StaffFilter($flight, true);
        $serviceData = $filter->getFlightWatchStaffData();

        $this->viewData = [
            "serviceData"               => $serviceData,

            'airlineModule'             => airlineModule(),
            'receivedMessages'          => $receivedMessages,
            'messages'                  => $this->messages,

            'dateFrom'                  => $minus1month,
            'dateTo'                    => $tomorrow,

            'picture' 			        => getAirlineLogo($airlineObj),

            'airports'                  => $handlingAirports,
            'airlines'                  => $handlingAirlines,
            'selectedAirports'          => $selectedAirports,
            'selectedAirlines'          => $selectedAirlines,
            "flightNumbers"             => FlightNumber::getFlightNumbersByAirlines(),
            'monthNames'                => getMonthsArray(),
            'years'                     => getYearsArray(),

            'flight'                    => $flight,
            'tracker'                   => $tracker,
            'fn1'                       => $fn1,
            'fn2'                       => $fn2,
            'fn3'                       => $fn3,
            'airlineObj'                => $airlineObj,
            'selectedAirport'           => $selectedAirport,

            'departureAirport'          => $departureAirport,
            'depLocation'               => $depLocation,
            'arrLocation'               => $arrLocation,
            'arrivalAirport'            => $arrivalAirport,
            'historicFlights'           => $historicFlights,
            'chartAltGs'                => $chartAltGs,

//            'staffAudit'                => $this->getStaffAudit($flight),
//            'flightsAudit'              => $this->getFlightAudit($flight),

            "users"         => User::getListByID(),
            "services"      => Service::getServicesListByID(),
        ];

        return view('flight-tracker/show', $this->viewData);
    }

}
