<?php

namespace App\Http\Controllers;

use App\Models\AircraftType;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Flight;
use App\Models\FlightNumber;
use App\Models\FlightSchedule;
use App\Models\FlightScheduleFlight;
use Illuminate\Http\Request;

use App\Http\Requests;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

use Maatwebsite\Excel\Collections\SheetCollection;
use Maatwebsite\Excel\Facades\Excel;

class FlightScheduleUploadController extends Controller
{
    public function index(Request $request){

        // Only Considered Stations
        if ($this->isPostRequest()) {


            if ($request->hasFile('file')) {

                $file = $request->file('file');
                // Load Excel File

                if (!in_array($file->getClientOriginalExtension(), ['xls', 'xlsx'])) {
                    flash()->error('Wrong Format(' . $file->getClientOriginalExtension() . '). Please Make Sure The Format is XLS or XLSX.');
                } else {

                    $loadObject = Excel::load($file, function ($reader) {
                        $reader->noHeading();
                        $reader->toObject();
                        $reader->formatDates(false);
                    })->get();


                    if (!$loadObject || !count($loadObject)) {
                        $this->viewData = [
                            'message_type'  => "danger",
                            'message'       => 'File Is Empty. Found No Data To Upload.',
                        ];
                    }

                    if ($loadObject instanceof SheetCollection){

                        $pageData = [];
                        foreach ($loadObject as $each) {
                            $pageData[] = $this->GetScheduleArray($each);
                        }

                        $data = [];
                        foreach ($pageData as $each) {
                            if ($each && count($each)){
                                foreach ($each as $item) {
                                    if (count($item)){
                                        $data[] = $item;
                                    }
                                }

                            }
                        }

                    }
                    else {
                        // Get Schedule Array
                        $data = $this->GetScheduleArray($loadObject);
                    }

                    debug($data);

                    // Save Template
                    $template = FlightSchedule::saveAirlineTemplate($request->get("template_name"), $data);

                    // Set Template Parent ids
                    $this->setTemplateFlightsConnections($template);

                    // Load To Flight Watch
                    FlightSchedule::uploadAirlineTemplateToFlightWatch($template);

                    // Set Flights Parent Ids
                    $this->setFlightsConnections($template);

                    $this->viewData = [
                        'message_type'  => "success",
                        'message'       => 'Schedule Successfully Uploaded.',
                    ];
                }
            } else {
                $this->viewData = [
                    'message_type'  => "danger",
                    'message'       => 'No File Uploaded. Please select Excel sheet and press upload.',
                ];
            }
        }

        $this->viewData["airlines"] = Airline::listHandlingAirlines();

        return view("flight-schedule/upload", $this->viewData);
    }

    public function setTemplateFlightsConnections($template){

        if ($template){
            // Find all inbound flights
            $parentFlights = FlightScheduleFlight::join("flights__numbers", "flights__numbers.id", "=", "flights__schedule_flights.flight_number_id")
                ->where('flight_schedule_id', $template->id)
                ->where('bound', 1)
                ->get(["flights__schedule_flights.*", "flight_number"]);

            foreach ($parentFlights as $parent) {
                $child = FlightScheduleFlight::join("flights__numbers", "flights__numbers.id", "=", "flights__schedule_flights.flight_number_id")
                    ->where('flight_schedule_id', $template->id)
                    ->where("flight_number",    intval($parent->flight_number) + 1)
                    ->where("aircraft_type_id", $parent->aircraft_type_id)
                    ->where("effective_date",   $parent->effective_date)
                    ->where("discontinue_date", $parent->discontinue_date)
                    ->first(['flights__schedule_flights.*']);

                if ($child){
                    $child->parent_id = $parent->id;
                    $child->save();
                }
            }
        }
    }

    public function setFlightsConnections($template = null, $flight_number = null){

        if ($template || $flight_number){
            // Find all inbound flights
            $parentFlights = Flight::join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
                ->where('flight_schedule_id', $template->id)
                ->where('bound', 1)
                ->get(["flights.*", "flight_number"]);

            foreach ($parentFlights as $parent) {
                $departure = date('Y-m-d', strtotime($parent->std));
                $departureNextDay = date('Y-m-d', strtotime("+ 1 days", strtotime($parent->std)));

                $child = Flight::join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
                    ->whereNull("flights.cancelled_at")
                    ->whereNull("flights.deleted_at")
                    ->where('flight_schedule_id', $template->id)
                    ->where("flight_number",    intval($parent->flight_number) + 1)
                    ->where("aircraft_type_id", $parent->aircraft_type_id)
                    ->where("std", ">", $parent->std)
                    ->whereBetween(DB::raw("DATE(std)"), [$departure, $departureNextDay])
                    ->orderBy("std")
                    ->first(['flights.*']);

                if ($child){
                    $child->parent_id = $parent->id;
                    $child->save();
                }
            }
        }
    }

    public function GetDate($string){
        if ($string){
            preg_match('/[A-Za-z]{3}/', $string, $match);

            if (strlen($string) == 8 && count($match) == 0){
                $date = \DateTime::createFromFormat("m-d-y", $string);
                return $date->format("Y-m-d");
            }
            else {
                //return date("Y-m-d", strtotime($string));
                $string = str_replace(['/', '.', ','], '-', $string);
                $carbon = Carbon::parse($string);
                return $carbon->format("Y-m-d");
            }
        }
        else {
            return null;
        }
    }

    public function GetTime($string){
        if ($string){
            if (is_string($string)) {
                return date("H:i:s", strtotime($string));
            } else {
                return $string->format('H:i:s');
            }
        }
        else {
            return  null;
        }
    }

    public function GetScheduleArray($loadObject){
        $finalData = [];
        // Starting Index
        $k = 0;

        $fieldCount = 9;
        $minHeaderFieldRequired = 5;
        $headerFound = false;

        $airports = Airport::whereNotNull("iata")
            ->pluck("id", "iata")
            ->all();

        $airlinesIATA = Airline::whereNotNull("iata")
            ->pluck("id", "iata")
            ->all();

        $airlinesICAO = Airline::whereNotNull("icao")
            ->pluck("id", "icao")
            ->all();

        foreach ($loadObject as $i => $object) {
            // Do Not Include Header Row
            // Find Starting Index $k
            if (!$headerFound){
                foreach ($object as $j => $title) {
                    if ($title){
                        $fieldExists = 0;
                        for($s = $j; $s < $j + $fieldCount; $s++){
                            if (isset($object[$s]) && $object[$s] != ""){
                                $fieldExists++;
                            }
                        }
                        if ($fieldExists > $minHeaderFieldRequired){
                            $headerFound = true;
                            $k = $j;
                            break;
                        }
                    }
                }
                continue;
            }

            if (!isset($object[$k]) || $object[$k] == ""){
                continue;
            }

            foreach ($object as $m => $item) {
                $object[$m] = trim($item);
            }


            $data = [];

            if (!isset($object[$k + 4]) || !isset($object[$k + 6])){
                continue;
            }

            $departureAirport = $object[$k + 4];
            $arrivalAirport   = $object[$k + 6];

            $bound = $this->GetFlightBound($departureAirport, $arrivalAirport);

            $departureAirportId = $this->getOriginDestinationStation($airports, $object[$k + 4]);
            $arrivalAirportId = $this->getOriginDestinationStation($airports, $object[$k + 6]);

            $IATAfltNO = $this->getIATACodeAndFlightNumber($airlinesIATA, $airlinesICAO, $object[$k], $departureAirportId, $arrivalAirportId, $bound);

            // Get passenger Array
            $data['airline']            = $IATAfltNO['iata'];
            $data['airline_id']         = $IATAfltNO['airline_id'];
            $data['flight_number']      = $IATAfltNO['flight_number'];
            $data['flight_number_id']   = $IATAfltNO['flight_nubmer_id'];

            $data['start_date']         = $this->GetDate($object[$k + 1]);
            $data['end_date']           = $this->GetDate($object[$k + 2]);

            $data['days']               = FlightSchedule::getDaysOfOperation($object[$k + 3]);

            $data['departure_airport']      = $departureAirport;
            $data['departure_airport_id']   = $departureAirportId;
            $data['departure_time'] = $this->GetTime($object[$k + 5]);


            $data['arrival_airport']        = $arrivalAirport;
            $data['arrival_airport_id']     = $arrivalAirportId;
            $data['arrival_time'] = $this->GetTime($object[$k + 7]);

            // LONG TYPE TEMPLATE
            /*
            if (isset($object[$k + 9]) && isset($object[$k + 10]) && isset($object[$k + 11]) &&
                isset($object[$k + 12]) && isset($object[$k + 13]) && isset($object[$k + 14])){
                $returnDepartureAirport = $this->getOriginDestinationStation($airports, $object[$k + 12]);
                $returnArrivalAirport = $this->getOriginDestinationStation($airports, $object[$k + 14]);
                $returnIATAfltNO = $this->getIATACodeAndFlightNumber($object[$k + 8], $returnDepartureAirport['airport_id'], $returnArrivalAirport['airport_id']);
            }

            // SHORT TYPE TEMPLATE
            else {
            */
            $IATAacType = $this->getIATACodeAndAircraftType($airlinesIATA, $object[$k + 8]);

            $data['aircraft_type'] = $IATAacType['ac_type'];
            $data['aircraft_type_id'] = $IATAacType['ac_type_id'];
            $data['aircraft_airline'] = $IATAacType['aircraft_airline_iata'];
            $data['aircraft_airline_id'] = $IATAacType['aircraft_airline_id'];

            //}
            $finalData[] = $data;
        }

        return $finalData;
    }

    public function GetFlightBound($departure, $arrival){
        $stations = Airport::getHandlingStations("iata");
        // Outbound
        if (in_array($departure, $stations)){
            return 0;
        }
        // Inbound
        else if (in_array($arrival, $stations)){
            return 1;
        }
        else {
            return null;
        }
    }

    public function getIATACodeAndAircraftType($airlines, $string){
        $aircraftType = $airline_id = null;
        preg_match('/(\w{2}(?:[\s|-]{1}))*(?:[\s|-])*([\s|\S]*)/', $string, $match);
        $iata = isset($match[1]) ? $match[1] : null;
        $acType = isset($match[2]) ? $match[2] : null;


        if ($iata){
            $airline_id = isset($airlines[$iata]) ? $airlines[$iata] : null;

            if (!$airline_id){
                $airline = new Airline();
                $airline->iata = $iata;
                $airline->created_by = Auth::user()->id;
                $airline->save();

                $airline_id = $airline->id;
            }
        }

        // AC Type
        if ($acType){
            $aircraftType = AircraftType::where(function($sql) use ($acType){
                    $sql->where("iata_name", "LIKE", "%{$acType}%")
                        ->orWhere("icao", "LIKE", "%{$acType}%")
                        ->orWhere("name", "LIKE", "%{$acType}%");
                })
                ->where(function($sql) use ($airline_id){
                    $sql->whereNull("airline_id");
                    if ($airline_id){
                        $sql->orWhere("airline_id", $airline_id);
                    }
                })
                ->orderByDesc("airline_id")
                ->first();

            if (!$aircraftType){
                $aircraftType = new AircraftType();
                $aircraftType->iata_name = $acType;
                $aircraftType->airline_id = $airline_id;
                $aircraftType->created_by = Auth::user()->id;
                $aircraftType->save();
            }
        }


        return [
            'ac_type'               => $acType,
            'ac_type_id'            => ($aircraftType ? $aircraftType->id : null),
            'aircraft_airline_iata' => $iata,
            'aircraft_airline_id'   => $airline_id
        ];
    }

    public function getOriginDestinationStation($airports, $iata){
        $airport_id = null;
        //preg_match('/\s*(\w{3})\s*/', $string, $match);

        if (isset($iata) && $iata){
            $airport_id = isset($airports[$iata]) ? $airports[$iata] : null;

            if (!$airport_id){

                $airport = Airport::findOrCreateByIATA($iata);
                $airport_id = $airport->id;
            }
        }

        return $airport_id;
    }

    public function getIATACodeAndFlightNumber(&$airlinesIATA, &$airlinesICAO, $string, $dep_airport_id, $arr_airport_id, $bound){
        $airline_id = $flightNumber = null;
        preg_match('/(\w{2}\D?)(?:[\s|-])*(\d+\D*)/', $string, $match);
        $code = isset($match[1]) ? $match[1] : null;
        $fltNo = isset($match[2]) ? $match[2] : null;


        $iata = $icao = null;
        if ($code){
            if (strlen($code) == 3){
                $icao = $code;
                $airline_id = isset($airlinesICAO[$code]) ? $airlinesICAO[$code] : null;
            }
            else {
                $iata = $code;
                $airline_id = isset($airlinesIATA[$code]) ? $airlinesIATA[$code] : null;
            }

            if (!$airline_id){
                $airline = new Airline();
                $airline->iata = $iata;
                $airline->icao = $icao;
                $airline->created_by = Auth::user()->id;
                $airline->save();

                $airline_id = $airline->id;

                // include this iata
                if ($iata){
                    $airlinesIATA[$iata] = $airline_id;
                }

                if ($icao){
                    $airlinesICAO[$icao] = $airline_id;
                }
            }

        }

        if ($airline_id && $fltNo) {
            $flightNumber = FlightNumber::where("airline_id", $airline_id)
                ->where("flight_number", $fltNo)
                ->where("departure_airport_id", $dep_airport_id)
                ->where("arrival_airport_id", $arr_airport_id)
                ->whereNull("deleted_at")
                ->first();

            if (!$flightNumber){
                $flightNumber = new FlightNumber();
                $flightNumber->airline_id = $airline_id;
                $flightNumber->flight_number = ltrim($fltNo, "0");
                $flightNumber->bound = $bound;
                $flightNumber->departure_airport_id = $dep_airport_id;
                $flightNumber->arrival_airport_id = $arr_airport_id;
                $flightNumber->created_by = Auth::user()->id;
                $flightNumber->save();
            }
        }

        return [
            'iata'              => $iata,
            'icao'              => $icao,
            'airline_id'        => $airline_id,
            'flight_number'     => $fltNo,
            'flight_nubmer_id'  => ($flightNumber ? $flightNumber->id : null)
        ];
    }
}
