<?php namespace App\Http\Controllers;

use App\Forms\FlightNumberForm;

use App\Models\Airline;
use App\Models\Airport;
use App\Models\Flight;
use App\Models\FlightNumber;
use App\Models\FlightType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;


class FlightNumberController extends Controller {

	/**
	 * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
	 */
	public function index()
	{
		$flightNumbers = FlightNumber::with(['airline', 'departureAirport', 'arrivalAirport', 'flightType'])
									->whereNull("deleted_at")
									->orderBy("airline_id")
									->orderBy("flight_number")
									->get();

		$this->viewData = [
			'flightNumbers'			=> $flightNumbers
		];

		return view('flight-number/index', $this->viewData);
	}


	public function edit($flight_number)
	{
		$distanceSet = true;
		$distance = $flight_number->distance;

		if (!$distance){
			$departureAirport = $flight_number->departureAirport ? $flight_number->departureAirport : "";
			$arrivalAirport = $flight_number->arrivalAirport ? $flight_number->arrivalAirport : "";

			if ($flight_number->departureAirport && $flight_number->arrivalAirport) {
				$distance = getDistanceBetweenPoints($departureAirport->latitude, $departureAirport->longitude, $arrivalAirport->latitude, $arrivalAirport->longitude, "Km");
			}
			$distanceSet = false;
		}

		$this->viewData = [
//			'airports'				=> ['' => 'Select'] + Airport::pluckAirportsWithCode(),
			'airlines'				=> ['' => 'Select'] + Airline::listHandlingAirlinesWithCode(),
			'flightTypes'			=> ['' => 'Select'] + FlightType::pluck("type", "id")->all(),
			'bound'					=> [0	=> 'Outbound', 1 => 'Inbound'],
			'flightNumber'			=> $flight_number,
			'distance'				=> $distance,
			'distanceSet'			=> $distanceSet,
			'currentPage'			=> \request("page")
		];

		return view('flight-number/edit', $this->viewData);
	}

	function CallSignDuplicateCheck($flightNumberID = null, $cs = null, $airlineID = null){
        if (!$cs) {
            return false;
        }

        $csSearch = FlightNumber::where("cs", $cs);

        if ($flightNumberID){
            $csSearch->where("id", "!=", $flightNumberID);
        }

        if ($airlineID){
            $csSearch->where("airline_id", $airlineID);
        }

        return $csSearch->first();
    }

	/**
	 * @param Request $request
	 * @param $flight_number
	 * @return $this|\Illuminate\Http\RedirectResponse
	 */
	public function update(Request $request, $flight_number)
	{
		$form = new FlightNumberForm();

		if ($form->isEditValid()){

		    if ($this->CallSignDuplicateCheck($flight_number->id, $request->get("cs"), $request->get("airline"))){
                return redirect()->back()
                    ->withErrors($form->getErrors())
                    ->withInput();
            }

			$flight_number->flight_type_id = $request->get('flight_type');
			$flight_number->airline_id = $request->get('airline');
			$flight_number->flight_number = $request->get('flight_number');
			$flight_number->cs =  $request->get("cs");

			$depAp = getAirportOrCreate($request->get('departure_airport'));
			$flight_number->departure_airport_id = $depAp ? $depAp->id : null;

			$arrAp = getAirportOrCreate($request->get('arrival_airport'));
			$flight_number->arrival_airport_id = $arrAp ? $arrAp->id : null;

			$flight_number->bound = $request->get('bound');
			$flight_number->duration = $request->get('duration') ? $request->get('duration') : null;
			$flight_number->distance = $request->get('distance') ? $request->get('distance') : null;
			$flight_number->save();

			flash()->success("Changes Successfully Saved");

			if ($request->get("current_airline_id")){
				return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=fltno");
			}

			return redirect()->to("flight-number");
		}
		else {
			return redirect()->back()
				->withErrors($form->getErrors())
				->withInput();
		}
	}

	public function create()
	{
		$this->viewData = [
			'airports'				=> ['' => 'Select'] + Airport::listHandlingAirportsWithCode("iata", "id"),
			'airlines'				=> ['' => 'Select'] + Airline::getHandlingAirlinesAndOperatorsWithCode("iata")->pluck("code", "id")->all(),
			'flightTypes'			=> ['' => 'Select'] + FlightType::pluck("type", "id")->all(),
			'bound'					=> [0	=> 'Outbound', 1 => 'Inbound'],
		];

		return view('flight-number/create', $this->viewData);
	}

	/**
	 * @param Request $request
	 * @return \Illuminate\Http\RedirectResponse
	 */
	public function store(Request $request)
	{
        if ($this->CallSignDuplicateCheck(null, $request->get("cs"), $request->get("airline"))){
            return redirect()->back()
                ->withInput();
        }

		$flight_number = new FlightNumber();
		$flight_number->flight_type_id = \request("flight_type");
		$flight_number->airline_id = \request("airline");
		$flight_number->flight_number = ltrim(\request("flight_number"), "0");
		$flight_number->departure_airport_id = \request("departure_airport");
		$flight_number->arrival_airport_id = \request("arrival_airport");
		$flight_number->bound = \request("bound");
		$flight_number->duration = \request("duration") ? \request("duration") : null;
		$flight_number->distance = \request("distance") ? \request("distance") : null;
		$flight_number->save();

		flash()->success("Changes Successfully Saved");

		if ($request->get("current_airline_id")){
			return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=fltno");
		}

		return redirect()->to("flight-number");
	}


	/**
	 * @param $flight_number
	 * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
	 */
	public function show($flight_number)
	{
        ini_set("max_execution_time", 120);
        ini_set('memory_limit', '256M');


        $this->viewData = [
			"flight_number"		=> $flight_number,
			"flights"			=> Flight::with("message")
										->where("flight_number_id", $flight_number->id)
                                        ->whereNull("deleted_at")
                                        ->whereNull("cancelled_at")
										->orderBy("std", "DESC")
										->orderBy("ptd", "DESC")
										->get(),
		];

		return view('flight-number/show', $this->viewData);
	}

	/**
	 * @param Request $request
	 * @param $flight_number
	 * @return \Illuminate\Http\RedirectResponse
	 */
	public function delete(Request $request, $flight_number)
	{
		$now = date("Y-m-d H:i:s");
		$flight_number->deleted_at = $now;
		$flight_number->updated_at = $now;
		$flight_number->updated_by = Auth::user()->id;
		$flight_number->save();
//		$flight_number->delete();

		if ($request->get("current_airline_id")){
			return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=fltno");
		}

		return redirect()->to("flight-number");
	}


	/**
	 * @param $id
	 */
	public function destroy($id)
	{
		//
	}

}
