<?php

namespace App\Http\Controllers;

use App\Models\Flight;
use Barryvdh\DomPDF\Facade as PDF;
use Illuminate\Http\Request;

use App\Http\Requests;
use App\Http\Controllers\Controller;

use Maatwebsite\Excel\Facades\Excel;

class FlightDocsController extends Controller
{
    const GENERAL_DECLARATION = "GENERAL_DECLARATION",
          FLIGHT_ORDER = "FLIGHT_ORDER",
          FLIGHT_ANALYSIS = "FLIGHT_ANALYSIS",
          FLIGHT_LOG = "FLIGHT_LOG",
          DOMESTIC = "domestic",
          INTERNATIONAL = "international";

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
    }

    public function show(Request $request, $flight)
    {
        $type = $request->get("type");

        switch($type){
            case self::FLIGHT_LOG:
                return view('flight-docs/flight-log', $this->prepareCrewData($flight));

            case self::FLIGHT_ORDER:
                return view('flight-docs/flight-order', $this->prepareCrewData($flight));

            case self::GENERAL_DECLARATION:
            default:
                return view('flight-docs/general-declaration', $this->prepareCrewData($flight));

        }
    }

    public function analysis(Request $request){

        $date = $request->get("date") ? $request->get("date") : date("Y-m-d", strtotime("-1 month"));

        $this->viewData = $this->prepareAnalysisData($date);

//        return 1;

        return view('flight-docs/flight-analysis',  $this->viewData);
    }

    function prepareAnalysisData($date){

        $dateSearch = [
            'month'     => date('m', strtotime($date)),
            'year'      => date('Y', strtotime($date)),
        ];

        // Get All Flights (filter: Date, Flight Number)
        $flights = Flight::getFlightsByDateFlightNumber($dateSearch, null, null, \request("airport"), \request("airline"), ALL, null, true);

        // Get Flights By Category And By Delay Code
        $flightTypes = $this->getFlightsDelay($flights);
        debug($flightTypes);

        return [
            "date"          => $date,
            "flights"       => $flights,
            "flightTypes"   => $flightTypes,
        ];
    }

    function getFlightsDelay(&$flights){

        $all = [
            "total"         => count($flights),
            "on_time"       => 0,
            "on_time_perc"  => 0,
            "delayed"       => 0,

            self::DOMESTIC      => [
                "total"         => 0,
                "on_time"       => 0,
                "on_time_perc"  => 0,
                "delayed"       => 0,
            ],

            self::INTERNATIONAL      => [
                "total"         => 0,
                "on_time"       => 0,
                "on_time_perc"  => 0,
                "delayed"       => 0,
            ]
        ];

        $gha = ghaModule();

        if ($flights && count($flights)) {
            $j = 0;
            foreach ($flights as $i => $each) {

                $departureAirport = $each->departureAirport ? $each->departureAirport   : $each->flightNumber->departureAirport;
                $arrivalAirport   = $each->arrivalAirport   ? $each->arrivalAirport     : $each->flightNumber->arrivalAirport;

                $index = null;
                if ($departureAirport){
                    $flights[$i]->departure_airport = $departureAirport->iata;

                    if ($departureAirport->country){
                        $index = $departureAirport->country->country == TAJIKISTAN ? self::DOMESTIC : self::INTERNATIONAL;
                    }
                }

                if ($arrivalAirport){
                    $flights[$i]->arrival_airport = $arrivalAirport->iata;

                    if ($arrivalAirport->country && $index != self::INTERNATIONAL){
                        $index = $arrivalAirport->country->country == TAJIKISTAN ? self::DOMESTIC : self::INTERNATIONAL;
                    }
                }

                if (!$index){
                    //continue;
                }

                try{
                    $all[$index]["total"]++;
                }
                catch(\Exception $e){
                }

                $delayed = isFlightDelayed($each, $gha);
                if ($delayed){
                    $all["delayed"]++;

                    if ($index){
                        $all[$index]["delayed"]++;
                    }
                }
                else {
                    $all["on_time"]++;

                    if ($index){
                        $all[$index]["on_time"]++;
                    }
                }

                if ($each->ata && $each->ata !== EMPTY_DATETIME){

                }

                $delayCode = $delayReason = $delayDuration = [];
                $flightDelays = $each->delays;

                if ($delayed || count($flightDelays)) {
                    $flights[$i]->is_delayed = true;
                    // Iterate Each Delays
                    $delayTotal = 0;
                    foreach ($flightDelays as $eachDelay) {
                        $code = ($eachDelay->delay->airline ? $eachDelay->delay->airline->iata." " : "").$eachDelay->delay->code;
                        $reason = $eachDelay->delay->description_ln ? $eachDelay->delay->description_ln : $eachDelay->delay->description;

                        $delayCode[] = $code;
                        $delayReason[] = $reason;
                        $delayDuration[] = $time = TimeToMinutes($eachDelay->duration, true);

                        $delayTotal += $time;

                    }

                    $stdBased = false;
                    if ((($each->std && $each->std != EMPTY_DATETIME) || ($each->ptd && $each->ptd != EMPTY_DATETIME))
                    && ($each->atd && $each->atd != EMPTY_DATETIME)){
                        $depInitial = getFlightDepartureInitialDatePTD($each);
                        $delayTotal = CalculateDifference($depInitial, $each->atd);

                        if ($delayTotal < 0){
                            $stdBased = true;
                            $depInitial = getFlightDepartureInitialDate($each);
                            $delayTotal = CalculateDifference($depInitial, $each->atd);
                        }
                        debug(++$j. " - STD Based.". $stdBased. " | INIT=". $depInitial." / ACT ".$each->atd. " DIFF = ".$delayTotal. " | ready".hoursToDayHourTime($delayTotal / 60, true, ""));
                    }

                    $flights[$i]->std_based = $stdBased;

                    $flights[$i]->delay_code = $delayCode;
                    $flights[$i]->delay_reason = $delayReason;
                    $flights[$i]->delay_duration = $delayDuration;
                    $flights[$i]->delay_total = $delayTotal < 0 ? "" : hoursToDayHourTime($delayTotal / 60, true, "");
                }
                else {
                    $flights[$i]->is_delayed = false;
                }

                $flights[$i]->category = getDelayCategory($each);;
            }
        }

        $all["on_time_perc"] = !$all["total"] ? 0 : 100 * round($all["on_time"] / $all["total"], 3);

        $all[self::DOMESTIC]["on_time_perc"] = !$all[self::DOMESTIC]["total"] ? 0 : 100 * round($all[self::DOMESTIC]["on_time"] / $all[self::DOMESTIC]["total"], 3);
        $all[self::INTERNATIONAL]["on_time_perc"] = !$all[self::INTERNATIONAL]["total"] ? 0 : 100 * round($all[self::INTERNATIONAL]["on_time"] / $all[self::INTERNATIONAL]["total"], 3);

        return $all;
    }


    function prepareCrewData($flight){
        $crew = $this->prepareCrew($flight);
        $capt = count($crew["capt"]);
        $rowSpan = ($capt <= 1 ? 0: $capt - 1) + count($crew["fo"]);

        $viewData = [
            'rowSpan'           => 16 + $rowSpan,
            'crew'              => $crew,
            'flight'            => $flight,
        ];

        debug($viewData);

        return $viewData;
    }

    public function download($flight){

        $crew = $this->prepareCrew($flight);
        $capt = count($crew["capt"]);
        $rowspan = ($capt <= 1 ? 0: $capt - 1) + count($crew["fo"]);

        $this->viewData = [
            'flight'            => $flight,
            'crew'              => $crew,
            'rowspan'           => 16 + $rowspan
        ];

        $pdf = PDF::loadView('flight-docs/downloads/general-declaration', $this->viewData);
        return $pdf->stream('general-declaration.pdf');
    }

    function prepareCrew($flight){

        $crew = [
            "capt"              => [],
            "fo"                => [],
            "cc"                => [],
            "sup"               => [],
        ];

        foreach ($flight->flightCrewItems as $each) {

            if ($each->deleted_at || $each->is_standby || $each->is_dhc){
                continue;
            }

            $positionType = $each->position ? $each->position->type : "";

            if ($each->is_sup) {
                $crew["sup"][] = $each->user;
            }
            else {
                if ($positionType == FCM_CPT_TYPE_ID) {
                    if (!isset($crew["capt"][$each->position_order])) {
                        $crew["capt"][$each->position_order] = $each->user;
                    } else {
                        $crew["capt"][] = $each->user;
                    }
                }
                elseif ($positionType == FCM_FA_TYPE_ID) {
                    if (!isset($crew["fo"][$each->position_order])) {
                        $crew["fo"][$each->position_order] = $each->user;
                    } else {
                        $crew["fo"][] = $each->user;
                    }
                }
                elseif (in_array($positionType, [CCM_PSR_TYPE_ID, CCM_CC_TYPE_ID])) {
                    if (!isset($crew["cc"][$each->position_order])) {
                        $crew["cc"][$each->position_order] = $each->user;
                    } else {
                        $crew["cc"][] = $each->user;
                    }
                }
            }
        }

        return $crew;

    }

}
