<?php namespace App\Http\Controllers;

/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 29.08.14

 */

use App\Classes\FlightStaff\StaffFilter;
use App\Classes\Parsing\Helpers\ParseHelper;
use App\Mail\TemplateEmail;
use App\Models\Aircraft;
use App\Models\AircraftType;
use App\Models\FlightNumber;
use App\Repositories\Interfaces\IFlightDelayRepository;
use App\Repositories\Interfaces\IFlightRepository;
use Illuminate\Contracts\Auth\Guard;
use Illuminate\Support\Facades\Auth;

use App\Models\Flight;
use App\Models\City;
use App\Forms\FlightForm;
use App\Models\FlightCrew;
use Illuminate\Support\Facades\Mail;

use Illuminate\Support\Facades\App;

class FlightController extends Controller
{

    const MIN_HOURS_TO_UPDATE_FLIGHT_TIME = 24;

    public function restore($flight)
    {
        $authUser = Auth::user();
        if (!$flight || !in_array($authUser->user_role_id, [ROLE_DEPARTMENT_ADMIN, ROLE_ADMIN, ROLE_SUPER_ADMIN]))
        {
            return redirect()->to(route("homepage")."#flight-view");
        }

        if ($authUser->user_role_id == ROLE_DEPARTMENT_ADMIN && $authUser->location_id != 3){
            return redirect()->to(route("homepage")."#flight-view");
        }

        $flight->cancelled_at = null;
        $flight->deleted_at = null;
        $flight->save();

        return redirect()->to(route("homepage")."#flight-view");
    }

    public function cancel($flight)
    {
        $authUser = Auth::user();
        if (!$flight || !in_array($authUser->user_role_id, [ROLE_DEPARTMENT_ADMIN, ROLE_ADMIN, ROLE_SUPER_ADMIN]))
        {
            return redirect()->to(route("homepage")."#flight-view");
        }

        if ($authUser->user_role_id == ROLE_DEPARTMENT_ADMIN && $authUser->location_id != 3){
            return redirect()->to(route("homepage")."#flight-view");
        }

        $flight->cancelled_at = date("Y-m-d H:i:s");
        $flight->deleted_at = null;
        $flight->save();

        return redirect()->to(route("homepage")."#flight-view");
    }


    public function remove($flight)
    {
        $authUser = Auth::user();
        if (!$flight || !in_array($authUser->user_role_id, [ROLE_DEPARTMENT_ADMIN, ROLE_ADMIN, ROLE_SUPER_ADMIN]))
        {
            return redirect()->to(route("homepage")."#flight-view");
        }

        if ($authUser->user_role_id == ROLE_DEPARTMENT_ADMIN && $authUser->location_id != 3){
            return redirect()->to(route("homepage")."#flight-view");
        }

        $flight->cancelled_at = null;
        $flight->deleted_at = date("Y-m-d H:i:s");
        $flight->save();

        return redirect()->to(route("homepage")."#flight-view");
    }


    /**
     * Get flight details page
     * @return \Illuminate\View\View
     */
    public function details()
    {
        $flightId = \request("id");
        $flight   = Flight::with([
            "Aircraft",
            "flightCrewItems",
            "FlightNumber"
        ])
            ->find($flightId);

        if (!$flight)
        {

        }
        return view("flight/details", ["flight" => $flight]);
    }

    public function edit($flight)
    {
        $airlineID = getFlightAirlineID($flight);

        // Aircraft List
        $aircraftList = ["" => "Select"] + Aircraft::getAircraftByAirline($airlineID, true);
        // AC Type List
        $aircraftTypeList = ["" => "Select"] + AircraftType::getAirlineTypesList($airlineID);

        return view("flight/edit", [
            "flight"            => $flight,
            "aircraft"          => $aircraftList,
            "aircraftTypes"     => $aircraftTypeList,
        ]);
    }

    public function update($flight, \Illuminate\Http\Request $request){

        debug($request->all());

        $messageDetails = [
            STD                 => $request->get("std"),
            PTD                 => $request->get("ptd"),
            ETD                 => $request->get("etd"),
            ATD                 => $request->get("atd"),
            ABN                 => $request->get("abn"),

            STA                 => $request->get("sta"),
            PTA                 => $request->get("pta"),
            ETA                 => $request->get("eta"),
            TDN                 => $request->get("tdn"),
            ATA                 => $request->get("ata"),

//            "paxMvtA"           => $request->get("atd"),
//            "paxMvtC"           => $request->get("atd"),
//            "paxMvtY"           => $request->get("atd"),
//            "paxMvtInf"         => $request->get("atd"),
        ];

        $aircraft = null;
        if ($request->get("aircraft_id")){
            $aircraft = Aircraft::find($request->get("aircraft_id"));
        }
        else {
            $flight->aircraft_id = null;
        }

        $date = getFlightDepartureInitialDate($flight, true);

        $parseHelper = new ParseHelper(null);
        $parseHelper->skipDateTimeCreation = true;
        $parseHelper->updateInitialTimings = true;
        $parseHelper->updateFlightInformation($flight, $messageDetails, $aircraft, $date);

        // Null empty
        foreach($messageDetails as $timing => $value){
            if (!$value){
                $flight->{$timing} = null;
            }
        }

        $flight->aircraft_type_id = $request->get("aircraft_type_id");
        $flight->save();

        return redirect()->to("flight/{$flight->id}/edit");

    }

    /**
     * Create/Update flight action
     * @param Guard $auth
     * @param IFlightRepository $flightRepository
     * @param IFlightDelayRepository $flightDelayRepository
     * @return \Illuminate\Http\RedirectResponse|int
     */
    public function add(Guard $auth, IFlightRepository $flightRepository, IFlightDelayRepository $flightDelayRepository)
    {
        $url = "";
        $start_time = \request("start_visible_period");
        $end_time   = \request("end_visible_period");

        if (\request("start_visible_period"))
        {
            $url    = "?start_time=$start_time&end_time=$end_time";
        }

        if ($this->isPostRequest()) {

            $form = new FlightForm;
            $previouslyDiverted = false;

            If (\request("restore_flight")){

                $flight = Flight::findOrFail(\request("flight_id"));
                if (!$flight){
                    return redirect()->to("fleetwatch/index$url")
                                     ->withErrors("Flight Not Found");
                }

                $flight->cancelled_at = null;
                $flight->updated_by = Auth::user()->id;
                $flight->save();

                flash()->success('Flight Successfully Restored');

                return redirect()->to("fleetwatch/index$url");
            }
            else if (\request("cancel_flight")){
                $cancel_reason = \request("cancel_reason");

                if (!$cancel_reason || $cancel_reason == ""){
                    return redirect()->to("fleetwatch/index$url")
                        ->withErrors("Cancellation Reason not specified. Please input cancellation reason.");
                }


                $flight = Flight::findOrFail(\request("flight_id"));
                if (!$flight){
                    return redirect()->to("fleetwatch/index$url")
                            ->withErrors("Flight Not Found");
                }

                $flight->remark = $cancel_reason;
                $flight->cancelled_at = date("Y-m-d H:i:s");
                $flight->updated_by = Auth::user()->id;
                $flight->save();

                flash()->success('Flight Successfully Cancelled');

                return redirect()->to("fleetwatch/index$url");
            }
            else {
                if (\request("flight_id")) {
                    if ($form->isValidForEdit()) {
                        $flight = Flight::findOrFail(\request("flight_id"));
                        $previouslyDiverted = $flight->is_diversion;
                    }
                } else {
                    if ($form->isValidForAdd()) {
                        $flight = new Flight;
                        $flight->created_by = Auth::user()->id;
                    }
                }

                if (!isset($flight)) {
                    return redirect()->back()
                        ->withInput()
                        ->withErrors($form->getErrors());
                }

                // get delay array values

                // Get Submitted Flight Number Id
                $flightNumberId = \request("flight_number_id");

                // If Manual FLight Number is Set
                if (\request("manual_flt_no")) {
                    $flightNumberId = FlightNumber::createManualFlightNumber();
                }

                $aircraft = Aircraft::find(\request("aircraft_id"));

                // Get Properties
                $properties = $this->getProperties($auth, $flightNumberId, $aircraft);

                // Update Flight Details
                $this->UpdateFlight($flight, $properties);

                // Update Flight Delays
                Flight::UpdateFlightDelays($flight, $flightDelayRepository);

                $flightNumber = FlightNumber::find($flightNumberId);

                // Check If New Flight Is Created Or It's An Update Form
                $flightId = $flight->id;

                // Create Diverted Flights
                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_CAPTAINS, \request("captain"), \request("captain_report_time"), \request("captain_reason"));
                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_CAPTAINS, \request("captain_standby"), \request("captain_standby_report_time"), \request("captain_standby_reason"), true);
                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_CAPTAINS, \request("captain_dhc"), \request("captain_dhc_report_time"), \request("captain_dhc_reason"), false, true);

                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_FIRST_OFFICERS, \request("first_officer"), \request("first_officer_report_time"), \request("first_officer_reason"));
                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_FIRST_OFFICERS, \request("first_officer_standby"), \request("first_officer_standby_report_time"), \request("first_officer_standby_reason"), true);
                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_FIRST_OFFICERS, \request("first_officer_dhc"), \request("first_officer_dhc_report_time"), \request("first_officer_dhc_reason"), false, true);

                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_PURSER_AND_CABIN_CREW, \request("cabin_crew"), \request("cabin_crew_report_time"), \request("cabin_crew_reason"));
                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_PURSER_AND_CABIN_CREW, \request("cabin_crew_standby"), \request("cabin_crew_standby_report_time"), \request("cabin_crew_standby_reason"), true);
                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_PURSER_AND_CABIN_CREW, \request("cabin_crew_dhc"), \request("cabin_crew_dhc_report_time"), \request("cabin_crew_dhc_reason"), false, true);
                FlightCrew::createChangeCrewProcedure($flightId, NULL, CREW_PURSER_AND_CABIN_CREW, \request("cabin_crew_sup"), \request("cabin_crew_sup_report_time"), \request("cabin_crew_sup_reason"), false, false, true, \request("cabin_crew_sup_type"));

                // Manage Diversion
                Flight::ManageDiversion($flightRepository, $flight, $flightNumber, $properties, $previouslyDiverted);
            }
        }

        return FleetWatchController::ajaxGetRange($start_time, $end_time);
    }

    public function UpdateFlight(&$flight, $properties){
        if (\request("std") && \request("std")) {
            $flight->std = \request("std");
        }
        if (\request("sta") && \request("sta")) {
            $flight->sta = \request("sta");
        }

        $flight->ptd               = \request("ptd")   ? \request("ptd") : NULL;
        $flight->etd               = (\request("etd")) ? \request("etd") : NULL;
        $flight->atd               = (\request("atd")) ? \request("atd") : NULL;
        $flight->abn               = (\request("abn")) ? \request("abn") : NULL;
        setFlightDepartureDate($flight);

        $flight->pta               = \request("pta")   ? \request("pta") : NULL;
        $flight->eta               = (\request("eta")) ? \request("eta") : NULL;
        $flight->ata               = (\request("ata")) ? \request("ata") : NULL;
        $flight->tdn               = (\request("tdn")) ? \request("tdn") : NULL;
        setFlightArrivalDate($flight);

        $flight->gate_parking      = \request("gate_parking");

        if (\request("outbound_flight_number_id")){
            $flight->parent_id        = \request("outbound_flight_number_id");
        }

        foreach ($properties as $prop => $value) {
            $flight->$prop = $value;
        }

        // set diversions
        //$this->setDiversions($flight);

        // save flight
        $flight->save();
    }


    public function getProperties($auth, $flightNumberId, $aircraft){
        return $properties = [
            'aircraft_id'      => \request("aircraft_id"),
            'flight_number_id' => $flightNumberId,
            'capacity_a'       => \request("capacity_a"),// ? \request("capacity_c") : ($aircraft ? $aircraft->config_c : null),
            'capacity_c'       => \request("capacity_c"),// ? \request("capacity_c") : ($aircraft ? $aircraft->config_c : null),
            'capacity_w'       => \request("capacity_w"),// ? \request("capacity_c") : ($aircraft ? $aircraft->config_c : null),
            'capacity_y'       => \request("capacity_y"),// ? \request("capacity_y") : ($aircraft ? $aircraft->config_y : null),
            'pax_a_booked'     => \request("pax_a_booked"),
            'pax_c_booked'     => \request("pax_c_booked"),
            'pax_w_booked'     => \request("pax_w_booked"),
            'pax_y_booked'     => \request("pax_y_booked"),
            'pax_inf_booked'   => \request("pax_inf_booked"),
            'pax_a_actual'     => \request("pax_a_actual"),
            'pax_c_actual'     => \request("pax_c_actual"),
            'pax_w_actual'     => \request("pax_w_actual"),
            'pax_y_actual'     => \request("pax_y_actual"),
            'pax_inf_actual'   => \request("pax_inf_actual"),
            'baggage'          => \request("baggage"),
            'cargo'            => \request("cargo"),
            'mail'             => \request("mail"),
            'deportees'        => \request("deportees"),
            'in_tanks'         => \request("in_tanks"),
            'uplifted'         => \request("uplifted"),
            'utilised'         => \request("utilised"),
            'updated_by'       => Auth::user()->id,
        ];
    }

    /**
     * Get flight's detailed information
     * @param Guard $auth
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxGetFlight(Guard $auth)
    {
        $flightId = (int)\request("flight_id");
        $flight   = Flight::with([
            "aircraft",
            "aircraftType",
            "flightCrew",
            "flightNumber",
            "flightNumber.airline",
            "flightNumber.flightType",
            "flightNumber.departureAirport",
            "flightNumber.arrivalAirport",
            "delays",
            "delays.delay",
        ])
            ->find($flightId);

        // if not found flight, print error JSON
        if (!$flight)
        {
            return response()->json([
                "error" => 1,
                "text"  => "Not found flight"
            ]);
        }
        if ($flight->is_diversion && $flight->diversion)
        {
            $flight->diversion = @unserialize($flight->diversion);
        }
        /*
        if (!$flight->capacity_y && !$flight->capacity_c)
        {
            if ($flight->aircraft){
                $flight->capacity_c = $flight->aircraft->config_c;
                $flight->capacity_y = $flight->aircraft->config_y;
            }
        }
        */

        // if flight does not have in_tanks, then look at previous flight
        $in_tanks = (int)$flight->in_tanks;
        if (!$in_tanks)
        {
            $previousFlight = Flight::where("aircraft_id", "=", $flight->aircraft_id)
                ->where("sta", "<", $flight->std)
                ->orderBy("std", "desc")
                ->first();

            if ($previousFlight)
            {
                if ($previousFlight->in_tanks != "" && $previousFlight->uplifted != "" &&
                    $previousFlight->utilised != ""
                )
                {
                    $in_tanks_auto    = $previousFlight->in_tanks + $previousFlight->uplifted - $previousFlight->utilised;

                    if ($previousFlight->defuel && ($in_tanks_auto - $previousFlight->defuel) >= 0){
                        $in_tanks_auto -= $previousFlight->defuel;
                    }

                    $flight->in_tanks = $in_tanks_auto;
                }
            }
        }

//        $flightCrew = FlightCrew::where('flight_id', $flight->id)->get();

        if ($flight->flightCrew()->count())
        {
            // Remove Duplicates
            FlightCrew::removeDuplicateCrew($flight->id);

            $flightCrew = $flight->flightCrew()
                ->whereNull('flights__crew.deleted_at')
                ->get();

            $captains   = $firstOfficers = $flightAttendants = [
                'general' => [],
                'standby' => [],
                'dhc'     => [],
            ];

            $flightAttendants['sup'] = [];

            foreach ($flightCrew as $flightCrewItem)
            {

//                if (count($user_ids) && in_array($flightCrewItem->id, $user_ids))
//                    continue;
//                else
//                    $user_ids[] = $flightCrewItem->id;


                $flightCrewItemSettings = FlightCrew::select([
                    "user_id",
                    "flight_id",
                    "position_id",
                    "position_order",
                    "report_time",
                    "reason",
                    "is_standby",
                    "is_dhc",
                    "is_sup",
                    "crew_sup_id",
                    "structure__positions.name as position_name",
                    "structure__positions.type as position_type",
                ])
                    ->join('structure__positions', 'structure__positions.id', '=', 'flights__crew.position_id')
                    ->where("flight_id", $flight->id)
                    ->where("user_id", $flightCrewItem->id)
                    ->whereNull('flights__crew.deleted_at')
                    ->get();

                if (count($flightCrewItemSettings)) {
                    foreach ($flightCrewItemSettings as $each_settings) {
                        $flightCrewItem->settings = clone $each_settings;

                        $pos = trim($each_settings->position_name);

                        if ($each_settings->position_type == FCM_CPT_TYPE_ID)
                        {
                            $this->addToCrew($flightCrewItem, $captains, $each_settings->position_order);
                        }
                        elseif ($each_settings->position_type = FCM_FA_TYPE_ID)
                        {
                            $this->addToCrew($flightCrewItem, $firstOfficers, $each_settings->position_order);
                        }
                        else if (in_array($each_settings->position_type, [CCM_PSR_TYPE_ID, CCM_CC_TYPE_ID])){
                            $this->addToCrew($flightCrewItem, $flightAttendants, $each_settings->position_order);
                        }

                        /*
                        if ($each_settings->position_id == USER_CAPTAIN_POSITION_ID) {
                            $captains = $this->addToCrew($flightCrewItem, $captains, $each_settings->position_order);
                        } else if ($each_settings->position_id == USER_FIRST_OFFICER_POSITION_ID) {
                            $firstOfficers = $this->addToCrew($flightCrewItem, $firstOfficers, $each_settings->position_order);
                        } else if (in_array($each_settings->position_id, [USER_PURSER_ID, USER_FLIGHT_ATTENDANT_ID])) {
                            $flightAttendants = $this->addToCrew($flightCrewItem, $flightAttendants, $each_settings->position_order);
                        }
                        */
                    }
                    // Otherwise JSON will treat it as Objects instead of Arrays
                    ksort($captains['general']);
                    ksort($firstOfficers['general']);
                    ksort($flightAttendants['general']);

                    /*$types = ['general', 'standby', 'dhc'];
                    $variables = [ 'captains', 'firstOfficers', 'flightAttendants' ];
                    foreach ($variables as $t => $var) {
                        foreach ($types as $s => $each) {
                            for ($p = 0; $p < count($flightAttendants['general']); $p++) {
                                if (!isset(${$var}[$each][$p])) {
                                    ${$var}[$each][$p] = [];
                                }
                            }
                        }
                    }*/

                    // IF POSITION ORDER MATTERS FOR FLIGHT CREW UNCOMMENT THESE LINES
                    /*for ($p = 0; $p < count($captains['general']); $p++){
                        if (!isset($captains['general'][$p]))
                            $captains['general'][$p] = [];
                    }
                    for ($p = 0; $p < count($firstOfficers['general']); $p++){
                        if (!isset($firstOfficers['general'][$p]))
                            $firstOfficers['general'][$p] = [];
                    }*/
                    for ($p = 0; $p < count($flightAttendants['general']); $p++){
                        if (!isset($flightAttendants['general'][$p]))
                            $flightAttendants['general'][$p] = [];
                    }
                }

            }

            $flight->flightCrew = [
                "captains"         => $captains,
                "firstOfficers"    => $firstOfficers,
                "flightAttendants" => $flightAttendants
            ];

        }

        $departureCity = City::where("abbr", "=", $flight->flightNumber->departureAirport->iata)->first();
        $arrivalCity   = City::where("abbr", "=", $flight->flightNumber->arrivalAirport->iata)->first();

        $departureCity = ($departureCity) ? $departureCity->city : "";
        $arrivalCity   = ($arrivalCity) ? $arrivalCity->city : "";

        if ($flight->delays){
            $delays = [];
            foreach ($flight->delays as $each) {
                $delay = $each->delay;
                if ($delay){
                    $delays[] = "Code: ".$delay->code." | ".$delay->description;
                }
            }
            $flight->delaysString = $delays;

        }

        return response()->json([
            "success"        => 1,
            "user"           => Auth::getUser(),
            "flight"         => $flight,
            "departure_city" => $departureCity,
            "arrival_city"   => $arrivalCity
        ]);
    }

    /**
     * Get AJAX list of parent flights for given current flight ID
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxFlightGetParentFlight()
    {
        $flightNumber  = \request("flight_number");
        $parentFlights = Flight::flightParents($flightNumber);
        return response()->json($parentFlights);
    }

    /**
     * Get Parent Flight By Flight Number And Departure Date
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxParentFlightId(){
        $flightNumberId = \request("flightNumberId");
        $departureDate = \request("departureDate");
        $parentFlight = Flight::getParentFlight($flightNumberId, $departureDate);

        return response()->json($parentFlight);

    }

    /**
     * Get Parent FLight Number Id By Child Flight Number ID
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxParentFlightNumberId(){
        $parentFlightNumber = Flight::getParentFlightNumber(\request("flightNumberId"));
        return response()->json($parentFlightNumber);
    }


    /**
     * AJAX delete flight action
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxDeleteFlight()
    {
        $flightId               = \request("flight_id");
        $flight                 = Flight::find($flightId);
        $flight->cancelled_at   = date("Y-m-d H:i:s");
        $flight->save();
        return response()->json([
            "message" => "Flight was successfully cancelled",
            "success" => TRUE
        ]);
    }

    /**
     * AJAX get flight fuel
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxFuel()
    {
        $flightId = \request("flight_id");
        $flight   = Flight::find($flightId);

        $previousFlight = Flight::where("aircraft_id", "=", $flight->aircraft_id)
            ->where("sta", "<", $flight->std)
            ->whereNull("deleted_at")
            ->whereNull("cancelled_at")
            ->orderBy("std", "desc")
            ->first();

        if ($previousFlight)
        {
            if ($previousFlight->rf){
                $in_tanks_auto = $previousFlight->rf ;

                if ($previousFlight->defuel && ($in_tanks_auto - $previousFlight->defuel) >= 0){
                    $in_tanks_auto -= $previousFlight->defuel;
                }

                return response()->json([
                    "success"   => 1,
                    "in_tanks"  => $in_tanks_auto,
                ]);
            }

            if ($previousFlight->in_tanks && $previousFlight->utilised)
            {
                $in_tanks_auto = $previousFlight->in_tanks + ($previousFlight->uplifted ? $previousFlight->uplifted : 0) - $previousFlight->utilised;

                if ($previousFlight->defuel && ($in_tanks_auto - $previousFlight->defuel) >= 0){
                    $in_tanks_auto -= $previousFlight->defuel;
                }

                return response()->json([
                    "success"   => 1,
                    "flight"    => $flight,
                    "in_tanks"  => $in_tanks_auto
                ]);
            }
        }

        return response()->json([
            "error"   => 1,
            "message" => "Not found previous flight"
        ]);
    }

    /**
     * Update flight start/end dates via AJAX request
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxChangeFlight()
    {
        $form = new FlightForm();
        if ($form->isValidForUpdateFlight())
        {
            $aircraftName = \request("aircraft_name");
            $flight       = Flight::with("Aircraft")
                ->find(\request("flight_id"));

            // validate flight time
            if ($this->getFlightHoursFromNow($flight) >= self::MIN_HOURS_TO_UPDATE_FLIGHT_TIME)
            {
                return response()->json(array(
                    "message" => "Flight cannot be updated due to " . self::MIN_HOURS_TO_UPDATE_FLIGHT_TIME .
                        " hours period",
                    "title"   => "Validation Errors:",
                    "error"   => TRUE
                ));
            }
            $originalFlight = clone $flight;
            $changeAircraft = FALSE;
            if ($flight->aircraft->name != $aircraftName)
            {
                $aircraft            = Aircraft::where("name", $aircraftName)
                                                ->whereNull("deleted_at")
                                                ->first();

                $flight->aircraft_id = $aircraft ? $aircraft->id : null;
                $changeAircraft      = TRUE;
            }

            $startDate = new \DateTime(\request("start_date"));
            $endDate   = new \DateTime(\request("end_date"));

            // update flight
            if ($flight->atd && $flight->atd != EMPTY_DATETIME)
            {
                $arrivalTime = new \DateTime(getFlightArrivalDate($flight));
                $atd         = new \DateTime($flight->atd);
                $now         = new \DateTime();

                // flight currently is flying
                if ($atd < $now && $arrivalTime > $now)
                {
                    // move forward
                    if ($startDate > $atd)
                    {
                        if ($startDate < $now)
                        {
                            $flight->atd = $startDate->format("Y-m-d H:i:s");
                        }
                        else
                        {
                            $flight->etd = $flight->atd = $startDate->format("Y-m-d H:i:s");
                            $flight->atd = NULL;
                        }
                        $flight->ata = NULL;
                        $flight->eta = $endDate->format("Y-m-d H:i:s");
                    }
                    // move back
                    else if ($startDate < $atd)
                    {
                        $flight->atd = $startDate->format("Y-m-d H:i:s");
                        if ($endDate > $now)
                        {
                            $flight->ata = NULL;
                            $flight->eta = $endDate->format("Y-m-d H:i:s");
                        }
                        else
                        {
                            $flight->ata = $endDate->format("Y-m-d H:i:s");
                        }
                    }
                }
                else
                {
                    $flight->atd = $startDate->format("Y-m-d H:i:s");
                    $flight->ata = $endDate->format("Y-m-d H:i:s");
                }
            }
            else
            {
                $flight->etd = $startDate->format("Y-m-d H:i:s");
                $flight->eta = $endDate->format("Y-m-d H:i:s");
            }
            $flight->save();

            // if flight is PARENT then need to update child one as well
            if (!$flight->parent_id)
            {
                $childFlight = Flight::with("Aircraft")
                    ->where("parent_id", "=", $flight->id)
                    ->first();
                if ($childFlight)
                {
                    $arrivalDateParent  = new DateTime(getFlightArrivalDate($originalFlight));
                    $arrivalDateChild   = new DateTime(getFlightArrivalDate($childFlight));
                    $departureDateChild = new DateTime(getFlightDepartureDate($childFlight));
                    $diff               = $endDate->diff($arrivalDateParent);
                    $sign               = ($arrivalDateParent < $endDate) ? "+" : "-";
                    if ($diff->y)
                    {
                        $departureDateChild->modify($sign . $diff->y . " years");
                        $arrivalDateChild->modify($sign . $diff->y . " years");
                    }
                    if ($diff->m)
                    {
                        $departureDateChild->modify($sign . $diff->m . " months");
                        $arrivalDateChild->modify($sign . $diff->m . " months");
                    }
                    if ($diff->d)
                    {
                        $departureDateChild->modify($sign . $diff->d . " days");
                        $arrivalDateChild->modify($sign . $diff->d . " days");
                    }
                    if ($diff->h)
                    {
                        $departureDateChild->modify($sign . $diff->h . " hours");
                        $arrivalDateChild->modify($sign . $diff->h . " hours");
                    }
                    if ($diff->i)
                    {
                        $departureDateChild->modify($sign . $diff->i . " minutes");
                        $arrivalDateChild->modify($sign . $diff->i . " minutes");
                    }
                    if ($diff->s)
                    {
                        $departureDateChild->modify($sign . $diff->s . " seconds");
                        $arrivalDateChild->modify($sign . $diff->s . " seconds");
                    }
                    if ($childFlight->atd)
                    {
                        $childFlight->atd = $departureDateChild->format("Y-m-d H:i:s");
                    }
                    else
                    {
                        $childFlight->etd = $departureDateChild->format("Y-m-d H:i:s");
                    }
                    if ($childFlight->ata)
                    {
                        $childFlight->ata = $arrivalDateChild->format("Y-m-d H:i:s");
                    }
                    else
                    {
                        $childFlight->eta = $arrivalDateChild->format("Y-m-d H:i:s");
                    }
                    if ($changeAircraft && isset($aircraft) && is_object($aircraft))
                    {
                        $childFlight->aircraft_id = $aircraft->id;
                    }
                    $childFlight->save();
                }
            }

            return response()->json([
                "message" => "Flight information was successfully updated",
                "success" => TRUE
            ]);
        }
        return response()->json([
            "message" => $form->getErrors()
                ->toJson(),
            "title"   => "Validation Errors:",
            "error"   => TRUE
        ]);
    }

    /**
     * Get flight hours from not to flight std
     * @param $flight
     * @return int|mixed
     */
    private function getFlightHoursFromNow($flight)
    {
        if ($flight->ata)
        {
            $date = $flight->ata;
        }
        else if ($flight->sta)
        {
            $date = $flight->sta;
        }
        else
        {
            return FALSE;
        }

        $now   = new DateTime();
        $std   = new DateTime($date);
        $hours = 0;
        $diff  = $now->diff($std);
        if ($diff->days > 0)
        {
            $hours = $diff->days * 24;
        }
        $hours += $diff->h;
        return $hours;
    }

    /**
     * Get AJAX MVT messages list for exact flight
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxGetMvtMessagesList()
    {
//        if (!\request()->ajax() || request()->server("REQUEST_METHOD") !== "GET")
//        {
//            App::abort(404);
//        }

        $flightId = \request("flight_id");

        $messages = getMessages($flightId, MVT);

        $departureMVT = $arrivalMVT = [];

        foreach ($messages as $j => $each) {
            $each->readyData =  Flight::parseMVTMessages($each);

            if ($each->type == strtolower(ARRIVAL)){
                $arrivalMVT[] = $each;
            }
            else {
                $departureMVT[] = $each;
            }
        }

        return response()->json([
                "departureMVT"  => $departureMVT,
                "arrivalMVT"    => $arrivalMVT,
            ]);
    }

    /**
     * Post AJAX send MVT message
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxSendMvtMessage()
    {
        if (\request("message") && \request("message"))
        {
            Mail::to("mvt@avbis.aero")
//                ->bcc("dilovar88@mail.ru")
                ->send(new TemplateEmail(\request("message"), \request("subject")));

            return response()->json([
                "message" => "MVT Message was successfully sent",
                "title"   => "Success!",
                "success" => 1
            ]);
        }
        return response()->json([
            "message" => "Please specify the MVT message",
            "title"   => "Error!",
            "error"   => 1
        ]);
    }

    public function ajaxGetMessages(\Illuminate\Http\Request $request)
    {
        if (//!\request()->ajax() ||
            request()->server("REQUEST_METHOD") !== "GET")
        {
            App::abort(404);
        }

//        getMessages($request->get("flight_id"), strtoupper($request->get("type")), true);
//        return  1;

        return response()->json([
            "messages"  => getMessages($request->get("flight_id"), strtoupper($request->get("type")), true),
//            "messages"  => Message::getMessageBody(strtoupper($request->get("type")), $request->get("flight_id")),
        ]);
    }

    public function ajaxGetSOD(\Illuminate\Http\Request $request)
    {
        $flightID = $request->get("flight_id");

        $flight = Flight::with(["flightParent", "flightStaff", "flightStaff.user"])->find($flightID);

        if (!$flightID || !$flight || !\request()->ajax() || request()->server("REQUEST_METHOD") !== "GET")
        {
            App::abort(404);
        }

        $filterStaff = new StaffFilter($flight);
        $serviceData = $filterStaff->getFlightWatchStaffData();

        // Optimized only needed data obj
        $flightObj = new \stdClass();
        $flightObj->staffList = Flight::getFlightStaff($flight);
        $this->setFlightTimings($flightObj, $flight);

        $flightParent = new \stdClass();
        if ($this->setFlightTimings($flightParent, $flight->flightParent)){
            $flightObj->flight_parent = $flightParent;
        }
        // with parent flight

        return response()->json([
            "flight"        => $flightObj,
            "serviceData"   => $serviceData,
        ]);
    }

    function setFlightTimings(&$flightObj, $flight){
        if (!$flight){
            return null;
        }

        $flightObj->std = $flight->std && $flight->std != EMPTY_DATETIME ? $flight->std : "";
        $flightObj->ptd = $flight->ptd && $flight->ptd != EMPTY_DATETIME ? $flight->ptd : "";
        $flightObj->etd = $flight->etd && $flight->etd != EMPTY_DATETIME ? $flight->etd : "";
        $flightObj->atd = $flight->atd && $flight->atd != EMPTY_DATETIME ? $flight->atd : "";
        setFlightDepartureDate($flightObj, false);

        $flightObj->sta = $flight->sta && $flight->sta != EMPTY_DATETIME ? $flight->sta : "";
        $flightObj->pta = $flight->pta && $flight->pta != EMPTY_DATETIME ? $flight->pta : "";
        $flightObj->eta = $flight->eta && $flight->eta != EMPTY_DATETIME ? $flight->eta : "";
        $flightObj->ata = $flight->ata && $flight->ata != EMPTY_DATETIME ? $flight->ata : "";
        setFlightArrivalDate($flightObj, false);

        return true;
    }

    /**
     * Set diversions
     * @param object $flight
     */
    private function setDiversions(&$flight)
    {
        if (\request("is_diversion"))
        {
            $diversion            = [
                "diversion_airport" => \request("diversion_airport"),
                "diversion_sta"     => \request("diversion_sta"),
                "diversion_eta"     => \request("diversion_eta"),
                "diversion_ata"     => \request("diversion_ata"),
                "diversion_tdn"     => \request("diversion_tdn"),

                "diversion_std"     => \request("diversion_std"),
                "diversion_etd"     => \request("diversion_etd"),
                "diversion_atd"     => \request("diversion_atd"),
                "diversion_abn"     => \request("diversion_abn")

            ];
            $flight->is_diversion = TRUE;
            $flight->diversion    = serialize($diversion);
        }
        else
        {
            $flight->is_diversion = FALSE;
            $flight->diversion    = NULL;
        }
    }

    /**
     * Add item to crew
     * @param $crewItem
     * @param $array
     * @param $positionOrder
     * @return mixed
     */
    private function addToCrew($crewItem, &$array, $positionOrder)
    {
        if ($crewItem->settings->is_standby)
        {
            $array["standby"][] = $crewItem;
        }
        else if ($crewItem->settings->is_dhc)
        {
            $array["dhc"][] = $crewItem;
        }
        else if ($crewItem->settings->is_sup){
            $array["sup"][] = $crewItem;
        }
        else {
            $array["general"][(int)$positionOrder] = $crewItem;
        }
    }

    /**
     * Create flight crew items
     * @param array $crew
     * @param $flightId
     * @param $isDHC
     * @return void
     */
    private function createFlightCrewItem($crew, $flightId, $isStandby = false, $isDHC = false)
    {
        if (is_array($crew))
        {
            foreach ($crew as $key => $userId)
            {
                $userId = (int)$userId;
                if (!$userId)
                {
                    continue;
                }
                FlightCrew::createCrewItem($userId, $flightId, NULL, ['position_order' => $key], $isStandby, $isDHC);
            }
        }
    }


}
