<?php

namespace App\Http\Controllers;

use App\Classes\FlightStaff\StaffFilter;
use App\Events\FlightComCreated;
use App\Events\NewMessage;
use App\Models\Aircraft;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Container;
use App\Models\ContainerType;
use App\Models\Flight;
use App\Models\FlightCom;
use App\Models\FlightComCheckin;
use App\Models\FlightComContainer;
use App\Models\FlightComDetails;
use App\Models\FlightComDetailsArchive;
use App\Models\FlightComFlightPTS;
use App\Models\FlightComPicture;
use App\Models\FlightComSSR;
use App\Models\FlightComUserViews;
use App\Models\FlightComUserViewsArchive;
use App\Models\FlightContainer;
use App\Models\FlightContainerInfo;
use App\Models\FlightFHR;
use App\Models\FlightMessage;
use App\Models\FlightMessage2018;
use App\Models\FlightNumber;
use App\Models\FlightPTS;
use App\Models\FlightSSR;
use App\Models\FlightStaff;
use App\Models\Location;
use App\Models\StaffAirline;
use App\Models\StaffStation;
use App\Models\UserModule;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Jenssegers\Agent\Agent;

class FlightComController extends Controller
{
    function isArchivedDate($date = null){
        if (!$date){
            return false;
        }

        $priorTo2Days = date("Y-m-d", strtotime("- 2 days"));

        return strtotime($date) < strtotime($priorTo2Days);
    }

    public function fhr(Request $request, $flight){

        $this->viewData = [
            "fhr"           => $flight->fhr,
            "flight"        => $flight,
            "flightNumber"  => getFlightNumberFull($flight->flightNumber),
        ];

        return view("flight-com.fhr", $this->viewData);
    }

    public function postFhr(Request $request, $flight){

        $record = FlightFHR::where("flight_id", $flight->id)
                        ->first();

        if (!$record) {
            $record = new FlightFHR();
        }

        $record->user_id = Auth::user()->id;
        $record->flight_id = $flight->id;
        $record->missed_artg = $request->input("missed_artg");
        $record->delay = $request->input("delay");
        $record->ckn = $request->input("ckn");
        $record->ramp = $request->input("ramp");
        $record->safety = $request->input("safety");
        $record->other = $request->input("other");
        for($i = 1; $i <= 6; $i++) {
            $input = "denied_boarding".$i;
            $record->{$input} = $request->input($input);
        }
        $record->save();

        return redirect()->back();
    }

    public function index(Request $request){
        return $this->init($request);
    }

    public function openFlight(Request $request, $flight){
        return $this->init($request, $flight);
    }

    function init(Request $request, $flight = null){
        $airlineModule = airlineModule();

        if ($airlineModule) {
            $selectedAirlineIds = null;
        }
        else{
            $selectedAirlineIds = Airline::getSelectedAirlineIds($request);
        }

        $selectedAirportIds = Airport::getSelectedAirportIds($request);

        $aircraftByAirline = Aircraft::getAircraftByAirline($selectedAirlineIds);

        if ($flight){
            $date = getFlightDepartureInitialDate($flight, true);
        }
        else {
            $date = $request->get("date") ? $request->get("date") : date("Y-m-d");
            $submit = $request->get("submit");
            switch ($submit) {
                // Next
                case "3":
                    $date = date("Y-m-d", strtotime("+ 1 days", strtotime($date)));
                    break;

                case "2":
                    $date = date("Y-m-d", strtotime("- 1 days", strtotime($date)));
                    break;

                default:
                    break;
            }
        }

        // ArchivedData
        $archivedData = $this->isArchivedDate($date);

        $flightsData = $this->getFlights($selectedAirlineIds, $selectedAirportIds, $date);

        $flightIDs = $keys = $all = $airlineByFlight = [];

        foreach ($flightsData as $type => &$flights) {
            if ($type == ALL){
                foreach ($flights as $i => $each) {
                    foreach ($each as $flightItem) {
                        foreach ($flightItem as $item) {
                            $all[] = $item;
                            $airlineByFlight[$item->id] = $item->flightNumber->airline_id;

                            if ($airlineModule && !in_array($item->id, $flightIDs)){
                                $flightIDs[] = $item->id;
                            }
                        }


                    }
                }
            }
            else {
                foreach ($flights as $each) {
                    if ($each->flightCom) {
                        $keys[$each->id] = $each->flightCom->key;
                    }
                    $flightIDs[] = $each->id;
                }
            }
        }

//        debug($flightsData);
//        return 1;

        $flightsData[ALL] = $all;

        $agent = new Agent();
        $isMobile = $agent->isMobile();

        $authUser = Auth::user();

        // Staff List Permission
        $staffPermission = UserModule::join("modules", "modules.id", "=", "users__modules.module_id")
            ->where("user_id", $authUser->id)
            ->where("modules.name", STAFF_LIST)
            ->whereNull("users__modules.deleted_at")
            ->first(["users__modules.id"]);

        $broadcastPermission =  UserModule::join("modules__permissions", "users__modules.module_permission_id", "=", "modules__permissions.id")
            ->where("user_id", $authUser->id)
            ->where("modules__permissions.name", BROADCAST)
            ->whereNull("users__modules.deleted_at")
            ->whereNull("modules__permissions.deleted_at")
            ->first(["users__modules.id"]);


        $keysArray = [];
        foreach ($keys as $id => $each) {
            $keysArray[$id] = PRE_KEY . $each . POST_KEY;
        }

        /// Airlines
        $airlines = Airline::select([DB::raw("CONCAT(iata, '-', airline) as name"), "id"]);
        if ($selectedAirlineIds){
            $airlines->whereIn("id", $selectedAirlineIds);
        }
        $airlines = $airlines->pluck("name", "id")->all();

        // Airports
        $airports = Airport::select("iata", "id");
        if ($selectedAirportIds){
            $airports->whereIn("id", $selectedAirportIds);
        }
        $airports = $airports->pluck("iata", "id")->all();

        $this->viewData = [
            "airlines"          => $airlines,
            "airports"          => $airports,
            "depArr"            => [
                ALL                 => ALL,
                DEPARTURE           => DEPARTURE,
                ARRIVAL             => ARRIVAL,
            ],

            "airlineModule"     => $airlineModule,
            "archivedData"      => $archivedData,
            "currentDate"       => $date,

            "sprAdm"            => $this->checkIfSprAdmin() ? 1 : 0,
            "messageDeletion"   => $this->messageDeletionPermitted() ? 1 : 0,

            'flight_id'         => $flight ? $flight->id :
                                    ($request->get("refresh") && $request->get("flight_id") ? $request->get("flight_id") : null),
            'dateToday'         => date("Y-m-d"),
            "staffPermission"   => $staffPermission != null,
            "broadcastPermission"   => $broadcastPermission != null,
            "broadcastName"     => env(AIRLINE) == GSRM ? "OCC" : "OPS",
            'user'              => $authUser,
            'flightsData'       => $flightsData,

            'keysArray'         => $keysArray,
            'flightIDs'         => $flightIDs,

            'isMobile'          => $isMobile ? 1 : 0,
            'uldRow'            => ["" => ""] + $this->getULDRows(),
            'uldPos'            => [
                ""  => "",
                "L" => "L",
                "R" => "R",
            ],

            'uldTypes'          => ["" => ""] + ContainerType::orderBy("position")
                    ->orderBy("name")
                    ->pluck("name", "name")
                    ->all(),
            // ["AKH" => "AKH", "PKC" => "PKC"],
            'contents'          => [
                ""    => "",
                "B"   => "B",     "C" => "C",      "M"     => "M",     "E"     => "E",
                "B/C" => "B/C",   "B/M" => "B/M",  "B/E"   => "B/E",   "B/C/M" => "B/C/M",  "B/C/M/E" => "B/C/M/E",
                "C/M" => "C/M",   "C/E" => "C/E",  "C/M/E" => "C/M/E",
                "M/E" => "M/E",
            ],

            "aircraft"      => ["" => "Select"] + Aircraft::whereNull("deleted_at")
                    ->orderBy("name")
                    ->pluck("name", "id")
                    ->all(),

            "airlineByFlight"   => $airlineByFlight,
            "aircraftByAirline" => $aircraftByAirline,

            'picNotes'      => [
                "CARGO"      => "CARGO",
                "CKIN"       => "CKIN",
                "DOCS"       => "DOCS",
                "FPLN"       => "FPLN",
                "FUEL"       => "FUEL",
                "GATE"       => "GATE",
                "LDS"        => "LDS",
                "LIR"        => "LIR",
                "MISC"       => "MISC",
                "RAMP"       => "RAMP",
            ],
        ];

        if ($isMobile){
            return view("flight-com/m-index", $this->viewData);
        }

        return view("flight-com/index", $this->viewData);
    }

    function messageDeletionPermitted(){

        $airline = env(AIRLINE);

        switch($airline){
            case GSRM:
            case IAG:
            case FRAAI:
            case AVIAM:
            case SCO:
                return false;

            case TAJIK_AIR:
            case CAA:
                return true;
        }

        return false;
    }

    public function loadMessages(Request $request, $flight = null){
        $authUserID = Auth::user()->id;
        $initialLoad = $request->get("initial_load");
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        if (!$flightID){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flight = Flight::with(["flightPts", "flightStaff"])->find($flightID);

        $flightDate = getFlightDepartureDate($flight, true);

        //$archivedData = $this->isArchivedDate($flightDate);

        $messages = $this->getFlightMessages($authUserID, $flightID, $flightDate, $request->get("last_message_id"), $request->get("first_message_id"), $initialLoad);

        $flightCom = isset($messages["com"]) && $messages["com"] ? $messages["com"] : null;

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs, $flightDate);

        // $uld = $this->getContainers($flightID);

        $flightObj = $this->prepareFlightObj($flight);


        return response()->json([
            "success"           => true,
            "serviceData"       => null,
            "uld"               => null,
            "unseen"            => $flightCom && $flightCom->archived ? [] : $unseen,
            "flight"            => $flightObj,
            "counter"           => $counter,
            // DO NOT UPLOAD THE SAME WAY
            "created"           => $messages["created"],
            "com"               => count($messages["messages"]) ? $messages["com"] : null,

            "messages"          => $messages["messages"],
            "deleted"           => $messages["deleted"],
            "key"               => $messages["key"],
            "pictures"          => $messages["pictures"],
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    function loadStaff(Request $request){

        $flightID = $request->get("flight_id");

        if (!$flightID){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flight = Flight::with([
                                "flightParent",
                                "flightStaff",
                                "flightStaff.user",
                                "flightNumber",
                                "flightNumber.airline"
                            ])->find($flightID);

        if (!$flight){
            return response()->json([
                "success"      => false,
            ]);
        }

        // Optimized only needed data obj
        $flightObj = new \stdClass();
        $flightObj->staffList = Flight::getFlightStaff($flight);
        $this->setFlightTimings($flightObj, $flight);

        $flightParent = new \stdClass();
        if ($this->setFlightTimings($flightParent, $flight->flightParent)) {
            $flightObj->flight_parent = $flightParent;
        }
        // with parent flight

        // Get staff
        $staffFilter = new StaffFilter($flight, true);
        $serviceData = $staffFilter->getFlightWatchStaffData();

        return response()->json([
            "success"           => true,
            "serviceData"       => $serviceData,
            "flight"            => $flightObj,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    function setFlightTimings(&$flightObj, $flight){
        if (!$flight){
            return null;
        }

        $flightObj->std = $flight->std && $flight->std != EMPTY_DATETIME ? $flight->std : "";
        $flightObj->ptd = $flight->ptd && $flight->ptd != EMPTY_DATETIME ? $flight->ptd : "";
        $flightObj->etd = $flight->etd && $flight->etd != EMPTY_DATETIME ? $flight->etd : "";
        $flightObj->atd = $flight->atd && $flight->atd != EMPTY_DATETIME ? $flight->atd : "";
        setFlightDepartureDate($flightObj, false);

        $flightObj->sta = $flight->sta && $flight->sta != EMPTY_DATETIME ? $flight->sta : "";
        $flightObj->pta = $flight->pta && $flight->pta != EMPTY_DATETIME ? $flight->pta : "";
        $flightObj->eta = $flight->eta && $flight->eta != EMPTY_DATETIME ? $flight->eta : "";
        $flightObj->ata = $flight->ata && $flight->ata != EMPTY_DATETIME ? $flight->ata : "";
        setFlightArrivalDate($flightObj, false);

        return true;
    }

    /// Save staff data
    public function staff(Request $request){
        if (!$request->get("flight_id")){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flight = Flight::with(["flightNumber", "flightStaff"])
            ->find($request->get("flight_id"));

        if (!$flight){
            return response()->json([
                "success"      => false,
            ]);
        }

        $authUserID = Auth::user()->id;
        $flightCom = FlightCom::getOrCreateFlightCom($flight->id, $authUserID, $request->get("key"));

        // Update Flight Staff
        $flightComDetails = new FlightComDetails();
        $flightComDetails->flight_com_id = $flightCom->id;
        $flightComDetails->created_by = $authUserID;
        $flightComDetails->save();

        debug($request->all());
        $createdFlightComStaff = FlightStaff::updateFlightsStaff($request, $flight, $flightComDetails, true);

        if (!count($createdFlightComStaff)){
            $flightComDetails->delete();
        }


        // Added
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        $flightDate = getFlightDepartureDate($flight, true);

        // $archivedData = $this->isArchivedDate($flightDate);

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $flightDate, $request->get("last_message_id"));

        $flightCom = isset($messages["com"]) && $messages["com"] ? $messages["com"] : null;

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs, $flightDate);

        $uld = [];// $this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $flightCom && $flightCom->archived ? [] : $unseen, //$archivedData ? [] : $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "deleted"           => $messages["deleted"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    function loadTrc(Request $request){
        $flightID = $request->get("flight_id");

        if (!$flightID){
            return response()->json([
                "success"      => false,
            ]);
        }

        $uld = $this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "uld"               => $uld,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    /**
     * Update Flight CKIN Info
     * ULDs
     */
    public function updateFlightDetails(){

    }

    function getULDRows(){
        return [
            "11"    => "11",
            "12"    => "12",
            "13"    => "13",
            "14"    => "14",

            "21"    => "21",
            "22"    => "22",
            "23"    => "23",

            "31"    => "31",
            "32"    => "32",
            "33"    => "33",

            "41"    => "41",
            "42"    => "42",
            "43"    => "43",

            "5"     => "5",
            "51"    => "51",
            "52"    => "52",
            "53"    => "53",
        ];
    }

    public function checkInTrc(Request $request){

        if (!$request->get("flight_id")){
            return response()->json([
                "success"      => false,
            ]);
        }

        debug($request->all());

        $authUserID = Auth::user()->id;
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        // Process Message
        list($flight) = $this->processMessage($request, $flightID);

        $flightDate = getFlightDepartureDate($flight, true);

        // $archivedData = $this->isArchivedDate($flightDate);

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $flightDate, $request->get("last_message_id"));

        $flightCom = isset($messages["com"]) && $messages["com"] ? $messages["com"] : null;

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs, $flightDate);

        $uld = []; //$this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $flightCom && $flightCom->archived ? [] : $unseen, // $archivedData ? [] : $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "deleted"           => $messages["deleted"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }


    /**
     * PRIOR TO DEC 1 5:38 UTC
     * all PTS Timings were in local
     * so if previous data needs to be converted to UTC
     * then write a code to convert all up to this date
     * from DEC 1 5:38 all timings are in UTC now
     */
    public function pts(Request $request){

        if (!$request->get("flight_id")){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flight = Flight::with(["flightNumber", "flightPts"])
            ->find($request->get("flight_id"));

        if (!$flight){
            return response()->json([
                "success"      => false,
            ]);
        }

        $authUserID = Auth::user()->id;
        $flightCom = FlightCom::getOrCreateFlightCom($flight->id, $authUserID, $request->get("key"));

        // Create Com Details
        $flightComDetails = new FlightComDetails();
        $flightComDetails->flight_com_id = $flightCom->id;
        $flightComDetails->created_by = $authUserID;
        $flightComDetails->save();

        $pts = FlightPTS::updateFields($request, $flightComDetails, $request->get("flight_id"));

        if (!$pts){
            $flightComDetails->delete();
        }

        // Added
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        $flightDate = getFlightDepartureDate($flight, true);

        // $archivedData = $this->isArchivedDate($flightDate);

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $flightDate, $request->get("last_message_id"));

        $flightCom = isset($messages["com"]) && $messages["com"] ? $messages["com"] : null;

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs, $flightDate);

        $uld = []; // $this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $flightCom && $flightCom->archived ? [] : $unseen, // $archivedData ? [] : $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "deleted"           => $messages["deleted"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);

    }

    public function ssr(Request $request){

        $flightID = $request->get("flight_id");

        if (!$flightID){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flight = Flight::with(["flightNumber", "flightPts"])
            ->find($flightID);

        if (!$flight){
            return response()->json([
                "success"      => false,
            ]);
        }

        $ssrs = ["umnr", "vip", "maas"];
        $ssrData = [];
        foreach ($ssrs as $ssr) {
            if ($request->has($ssr) && $request->get($ssr) !== ""){
                $this->createSSR($flightID, $ssr, $request->get($ssr));
                $ssrData[$ssr] = $request->get($ssr);
            }
        }

        if (!count($ssrData)){
            return response()->json([
                "success"      => false,
            ]);
        }

        $authUserID = Auth::user()->id;
        $flightCom = FlightCom::getOrCreateFlightCom($flight->id, $authUserID, $request->get("key"));

        // Create Com Details
        $flightComDetails = new FlightComDetails();
        $flightComDetails->flight_com_id = $flightCom->id;
        $flightComDetails->created_by = $authUserID;
        $flightComDetails->save();

        //
        $this->saveSSR($flightID, $flightComDetails, $ssrData);

        // Added
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;
        $flightDate = getFlightDepartureDate($flight, true);

        // $archivedData = $this->isArchivedDate($flightDate);

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $flightDate, $request->get("last_message_id"));

        $flightCom = isset($messages["com"]) && $messages["com"] ? $messages["com"] : null;

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs, $flightDate);

        $uld = []; // $this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $flightCom && $flightCom->archived ? [] : $unseen, // $archivedData ? [] : $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "deleted"           => $messages["deleted"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);

    }

    function saveSSR($flightID, $flightComDetails, $ssrData = []){
        if (!count($ssrData)){
            return;
        }

        $obj = new FlightComSSR();
        $obj->flight_com_id = $flightComDetails->flight_com_id;
        $obj->flight_com_details_id = $flightComDetails->id;
        $obj->flight_id = $flightID;

        foreach ($ssrData as $ssr => $val) {
            $obj->{$ssr} = $val;
        }
        $obj->save();
    }

    function createSSR($flightID, $ssr, $total){

        if (!$flightID){
            return;
        }

        $findSSR = FlightSSR::firstOrCreate([
            "flight_id"     => $flightID,
            "ssr"           => strtoupper($ssr),
        ]);

        $findSSR->pax_total = $total;
        $findSSR->save();
    }

    public static function setFlightProperties(&$flight, $type){

        $flight->type = $type;

        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber ? $flightNumber->airline : null;
        list($fn1, $fn2, $fn3) = getFlightNumberStrings($flight);

        $depAP = getFlightDepartureAirport($flight);
        $dep = $depAP ? $depAP->iata : "";

        $arrAP = getFlightArrivalAirport($flight);
        $arr = $arrAP ? $arrAP->iata : "";

        $depType = getFlightDepartureTimeType($flight, true);
        $depInitialType = getFlightDepartureInitialTimeType($flight, true);

        $arrType = getFlightArrivalTimeType($flight, true);
        $arrInitialType = getFlightArrivalInitialTimeType($flight, true);


        $depSpan = departureArrivalTypeSpan($depType, null);
        $depInitialSpan = departureArrivalTypeSpan($depInitialType, null);

        $arrSpan = departureArrivalTypeSpan($arrType, null);
        $arrInitialSpan = departureArrivalTypeSpan($arrInitialType, null);

        $depInitialTxt = $depInitialSpan . "<span class='dep-time-initial'>" . baseDateFormat(getFlightDepartureInitialDate($flight), true) . "</span>";
        $depTxt = $depSpan . "<span id='flt_dep_arr_time' class='dep-time'>" . baseDateFormat(getFlightDepartureDate($flight), true) . "</span>";

        $arrInitialTxt = $arrInitialSpan . "<span class='arr-time-initial'>" . baseDateFormat(getFlightArrivalInitialDate($flight), true) . "</span>";
        $arrTxt = $arrSpan . "<span id='flt_dep_arr_time' class='arr-time'>" . baseDateFormat(getFlightArrivalDate($flight), true ). "</span>";


        $ac = getFlightAircraft($flight);
        $acType = getFlightAircraftType($flight);

        $cap =  getFlightCapacity($flight);
        $bookedPax = getFlightBookedPax($flight);
        $actualPax = getFlightActualPax($flight);

        $capReady = "CFG ".($cap[0] ? $cap[0]."C " : "").$cap[1]."Y";
        $bookedPaxReady = ($bookedPax[0] || $bookedPax[1]) ? "BKD ".($bookedPax[0] ? $bookedPax[0]."C " : "").$bookedPax[1]."Y" : "";
        $actualPaxReady = ($actualPax[0] || $actualPax[1]) ? "ACT ".($actualPax[0] ? $actualPax[0]."C " : "").$actualPax[1]."Y" : "";

        $aircraftDetails = $acType.($ac ? " | ".$ac : "");
        //

        if ($type == DEPARTURE){
            $acDetails = $aircraftDetails." | ".$capReady.($bookedPaxReady ? " | ".$bookedPaxReady : "").($actualPaxReady ? " | ".$actualPaxReady : "");
        }
        else {
            $acDetails = $aircraftDetails." | ".$capReady.($actualPaxReady ? " | ".$actualPaxReady : "");
        }

        $icon =  strtolower($type)."64.png";

        $flight->dep_arr_icon = asset("assets/img/icons/others/{$icon}");
        $flight->status = getFlightClassNameStrict($flight, $type);
        $flight->airline_logo = getAirlineLogo($airline);
        $flight->fn_ready = $fn3;
        $flight->sector_ready = " | ".$dep."-".$arr."";
        $flight->aircraft_details = $acDetails;


        if ($type == DEPARTURE){
            $fltDepArrDetails  = ($depType == $depInitialType ? "" :  $depInitialTxt) ." ". $depTxt;
        }
        else{
            $fltDepArrDetails = ($arrType == $arrInitialType ? "" :  $arrInitialTxt) ." ". $arrTxt;
        }

        $flight->dep_arr_details = $fltDepArrDetails;
    }

    public static function getFlights($selectedAirlineIds, $selectedAirportIds, $date){

        $startDate  = $date . " 00:00:00";
        $endDate    = $date . " 23:59:59";

        $data = [
            ALL             => [],
            DEPARTURE       => [],
            ARRIVAL         => [],
            CANCELLED       => [],
        ];

        $handlingStations = Airport::getHandlingStationsIdAndObject();

        $gha = ghaModule();
        $processedFlightIDs = [];

        if ($gha) {

            if (!$selectedAirlineIds || !$selectedAirportIds) {
                return [
                    ALL             => [],
                    DEPARTURE       => [],
                    ARRIVAL         => [],
                    CANCELLED       => [],
                ];
            }

            // Load Arrival Flights
            $flights = Flight::handlingFlightsRange($startDate, $endDate, $selectedAirlineIds, $selectedAirportIds, null,
                ["arrival_date_search" => true], true, true);
            $flights = $flights->sortBy("sta")->all();

            $allIParentIDs = [];
            $allIChildrenIDs = [];
            foreach ($flights as $flight) {

                $type = getFlightsDepArrFullType($flight, $handlingStations);

                if ($type && in_array($type, [ARRIVAL, FULL])) {
                    $type = ARRIVAL;

                    // Set Properties
                    self::setFlightProperties($flight, $type);

                    $time = strtotime(getFlightArrivalDate($flight));

                    if ($flight->cancelled_at){
                        $type = CANCELLED;
                    }

                    if (isset($data[$type][$time])) {
                        do {
                            $time++;
                        } while (isset($data[$type][$time]));
                    }

                    $data[$type][$time] = $flight;

                    if ($type == CANCELLED){
                        // Added on Feb 11
                        if (isset($data[ARRIVAL][$time])){
                            do {
                                $time++;
                            }
                            while(isset($data[ARRIVAL][$time]));
                        }
                        $data[ARRIVAL][$time] = $flight;
                        continue;
                    }

                    /// All flights
                    $index = $flight->parent_id ? $flight->parent_id : $flight->id;
                    if ($flight->parent_id) {
                        //debug("ARR child");
                        $allIChildrenIDs[$index] = $time;
                        $data[ALL][$time][$index][1] = $flight;
                    }
                    else {
                        $allIParentIDs[$index] = $time;
                        $data[ALL][$time][$index][] = $flight;
                    }

                    $processedFlightIDs[] = $flight->id;
                }
            }
        }
        else {
            if (!$selectedAirportIds) {
                return [
                    ALL             => [],
                    DEPARTURE       => [],
                    ARRIVAL         => [],
                    CANCELLED       => [],
                ];
            }
        }

        // Load Departure Flights
        $flights = Flight::handlingFlightsRange($startDate, $endDate, $selectedAirlineIds, $selectedAirportIds, null, [], true, true);
//        $flights = $flights->sortBy("std")->all();

        $j = 0;

        foreach ($flights as $flight) {

            if ($gha){
                $type = getFlightsDepArrFullType($flight, $handlingStations);
            }
            else {
                $type = DEPARTURE;
            }

//            if ($type && $type == DEPARTURE){
            if (!$type || in_array($type, [DEPARTURE, FULL])){
                $type = DEPARTURE;

                // Set Properties
                self::setFlightProperties($flight, $type);

                $j++;
                $time = strtotime(getFlightDepartureDate($flight));

                if ($flight->cancelled_at){
                    $type = CANCELLED;
                }

                if (isset($data[$type][$time])){
                    do {
                        $time++;
                    }
                    while(isset($data[$type][$time]));
                }

                $data[$type][$time] = $flight;

                if ($type == CANCELLED){
                    // Added on Feb 11
                    if (isset($data[DEPARTURE][$time])){
                        do {
                            $time++;
                        }
                        while(isset($data[DEPARTURE][$time]));
                    }
                    $data[DEPARTURE][$time] = $flight;
                    continue;
                }

                /// All flights
                if (in_array($flight->id, $processedFlightIDs)){
                    continue;
                }

                $index = $flight->parent_id ? $flight->parent_id : $flight->id;
                if ($flight->parent_id){
                    if (isset($allIParentIDs[$index])){
                        //debug("DEP child = found match");
                        //debug(getFlightNumber($flight));
                        $data[ALL][$allIParentIDs[$index]][$index][] = $flight;
                        continue;
                    }
                }
                else {
                    if (isset($allIChildrenIDs[$index])){
                        //debug("DEP parent = found arrival child");
                        $data[ALL][$allIChildrenIDs[$index]][$index][0] = $flight;
                        continue;
                    }
                }

                $data[ALL][$time][$index][] = $flight;
            }
        }

       ///debug($data[ALL]);

        foreach ($data as &$each) {
            ksort($each);
        }

        if (airlineModule()){
            $data[DEPARTURE] = [];
            $data[ARRIVAL] = [];
        }

        return $data;
    }

    function prepareFlightObj($flight){
        if (!$flight){
            return null;
        }

        $cap = getFlightCapacity($flight);
        $bookedPax = getFlightBookedPax($flight);
        $actualPax = getFlightActualPax($flight);

        $flightPts = $flight->flightPts;

        $flightObj = [
            "id"            => $flight->id,
            "has_staff"     => flightHasStaff($flight),
            "aircraft_id"   => $flight->aircraft_id,
            "gate_boarding" => $flight->gate_boarding,
            "gate_parking"  => $flight->gate_parking,
            "message"       => $this->prepareFlightMessages($flight),

            "std"           => $flight->std && $flight->std != EMPTY_DATETIME ? $flight->std : "",
            "ptd"           => $flight->ptd && $flight->ptd != EMPTY_DATETIME ? $flight->ptd : "",
            "etd"           => $flight->etd && $flight->etd != EMPTY_DATETIME ? $flight->etd : "",
            "atd"           => $flight->atd && $flight->atd != EMPTY_DATETIME ? $flight->atd : "",

            "sta"           => $flight->sta && $flight->sta != EMPTY_DATETIME ? $flight->sta : "",
            "pta"           => $flight->pta && $flight->pta != EMPTY_DATETIME ? $flight->pta : "",
            "eta"           => $flight->eta && $flight->eta != EMPTY_DATETIME ? $flight->eta : "",
            "ata"           => $flight->ata && $flight->ata != EMPTY_DATETIME ? $flight->ata : "",

            "ac_type"       => getFlightAircraftType($flight),
            "ac"            => getFlightAircraft($flight),
            "cfg"           => "CFG ".($cap[0] ? $cap[0]."C " : "").$cap[1]."Y",
            "bkd"           => ($bookedPax[0] || $bookedPax[1]) ? "BKD ".($bookedPax[0] ? $bookedPax[0]."C " : "").$bookedPax[1]."Y" : "",
            "act"           => ($actualPax[0] || $actualPax[1]) ? "ACT ".($actualPax[0] ? $actualPax[0]."C " : "").$actualPax[1]."Y" : "",

            "pax_a"         => $flight->pax_a_actual,
            "pax_c"         => $flight->pax_c_actual,
            "pax_y"         => $flight->pax_y_actual,
            "pax_jmp"       => $flight->pax_jmp_actual,
            "pax_adults"    => $flight->pax_adults_actual,
            "pax_m"         => $flight->pax_m_actual,
            "pax_f"         => $flight->pax_f_actual,
            "pax_ch"        => $flight->pax_ch_actual,
            "pax_inf"       => $flight->pax_inf_actual,

            // New fields
            "dow"           => $flight->dow,
            "doi"           => $flight->doi,
            "mtow"          => $flight->mtow,
            "rtow"          => $flight->rtow,

            "tof"           => $flight->tof,
            "trip"          => $flight->trip,
            "eet"           => $flight->eet,
            "taxi"          => $flight->taxi,
            "block"         => $flight->block,
            "altn"          => $flight->altn,

            // LOFO + others

            "lofo_staff"       => $flightPts ? $flightPts->lofo_staff : null,
            "lofo_rmks"        => $flightPts ? $flightPts->lofo_rmks : null,
            "lofo_start"       => $flightPts ? $flightPts->lofo_start : null,
            "lofo_end"         => $flightPts ? $flightPts->lofo_end : null,
            "mhb_ahl"          => $flightPts ? $flightPts->mhb_ahl : null,
            "mhb_ohd"          => $flightPts ? $flightPts->mhb_ohd : null,
            "mhb_dpr"          => $flightPts ? $flightPts->mhb_dpr : null,
        ];

        return $flightObj;
    }

    function prepareFlightMessages($flight){
        if (!$flight){
            return null;
        }

        if (isArchiveMessages($flight)){
            $flightMessages = FlightMessage2018::select("message_type");
        }
        else {
            $flightMessages = FlightMessage::select("message_type");
        }

        $flightMessages = $flightMessages->where("flight_id", $flight->id)->get();

        if (!$flightMessages->count()){
            return null;
        }

        $result = [];
        foreach ($flightMessages as $each) {
            $result[$each->message_type] = $each->message_type;
        }

        return array_values($result);
    }


    /**
     * For
     * @param $flightID
     * @return array
     */
    public function getContainers($flightID){
        // ULD
        $flightContainerInfo = $this->getFlightContainerInfo($flightID);

        $result = [];

        if ($flightContainerInfo) {
            $containers = FlightContainer::with([
                "destination",
                "container",
                "container.type",
            ])
                ->where("info_id", $flightContainerInfo->id)
                ->orderBy("row_number", "DESC")
                ->orderBy("position", "DESC")
                ->get();

            $flight = Flight::with(["flightNumber.arrivalAirport"])->find($flightID);
            $destAP = getFlightArrivalAirport($flight, false, true);

            foreach ($containers as $each) {
                $pos = $each->row_number . ($each->position ? $each->position : "");

                $index = $pos;
                if ($each->row_number == 5){
                    $index = $pos . "0";
                }

                if (isset($result[$index])){
                    $k = 0;
                    do {
                        $index .= $k++;
                    }while(isset($result[$index]));
                }

                $result[$index] = [
                    "dest"      => $each->destination && $each->destination->iata ? $each->destination->iata : $destAP,
                    "pos"       => $pos,
                    "content"   => $each->loaded_type ? $each->loaded_type : "",
                    "type"      => $each->container && $each->container->type ? $each->container->type->name : ($each->containerType ? $each->containerType->name : ""),
                    "num"       => $each->container && $each->container->num ? $each->container->num : "",
                    "owner"     => $each->container ? ($each->container->owner ? $each->container->owner : ($each->container->airline ? $each->container->airline->iata : "" )) : "",
                    "wt"        => $each->weight ? $each->weight : "",
                    "vr"        => $each->vr || $each->vr == 0 || $each->vr == "0" ? $each->vr : "",
                    "pcs"       => $each->bag_pcs ? $each->bag_pcs : "",
                    "notoc"     => $each->notoc ? "YES" : "NO",
                ];

            }

            ksort($result);
            $result = array_reverse(array_values($result));
        }

        return $result;
    }

    public function sendMessage(Request $request){

        debug($request->all());

        $authUserID = Auth::user()->id;
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        // Process Message
        list($flight) = $this->processMessage($request, $flightID);

        $flightDate = getFlightDepartureDate($flight, true);

        // $archivedData = $this->isArchivedDate($flightDate);

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $flightDate, $request->get("last_message_id"));

        $flightCom = isset($messages["com"]) && $messages["com"] ? $messages["com"] : null;

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs, $flightDate);

        $uld = []; //$this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $flightCom && $flightCom->archived ? [] : $unseen, // $archivedData ? [] : $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "deleted"           => $messages["deleted"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);

    }

    public function sendOCC(Request $request){

        $authUserID = Auth::user()->id;
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        $flight = Flight::with([
            "message",
            "flightNumber",
            "flightNumber.airline",
            "flightStaff"
        ])->find($flightID);

        $message = $request->get("occ_message");
        $date = $request->get("occ_date") ? $request->get("occ_date") : date("Y-m-d") ;
        $type = $request->get("occ_dep_arr");

        $airlineIDs = $request->get("occ_airline");
        if (!$airlineIDs || !count($airlineIDs)){
            $airlineIDs = StaffAirline::currentStaffAirlinesList();
        }

        $airportIDs = $request->get("occ_airport");
        if (!$airportIDs || !count($airportIDs)){
            $airportIDs = StaffStation::currentStaffStationsList();
        }

        // Process Message
        $flights = Flight::FlightsRangeConditional($airportIDs, $airlineIDs, $type, null, null, ["from" => $date, "to" => $date]);
        debug($flights);
        $coms = FlightCom::createMessageForFlights($flights, $authUserID, $message);

        $flightDate = getFlightDepartureDate($flight, true);

        // $archivedData = $this->isArchivedDate($flightDate);

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $flightDate, $request->get("last_message_id"));
        $flightCom = isset($messages["com"]) && $messages["com"] ? $messages["com"] : null;
        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs, $flightDate);

        $uld = []; //$this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $flightCom && $flightCom->archived ? [] : $unseen, // $archivedData ? [] : $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "deleted"           => $messages["deleted"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);

    }

    function checkIfSprAdmin(){
        $authUser = Auth::user();

        return in_array($authUser->user_role_id, [3, 4]);

        return ($authUser->user_role_id == 4 || in_array($authUser->id, [
            482, // Chris Bedikian
            531, // Burak Kahraman
            888, // Sergiu Grosu
            911, // Ahmad Mukhtar
            912, // Eline Koek
            894, // Thomas Rosenthal
            796, // Marcel Ehrlich
        ]));
    }

     public function deleteMessage(Request $request){
         $authUserID = Auth::user()->id;
         $messageID = $request->get("message_id");

         debug($request->all());

         $msg = FlightComDetails::find($messageID);

         if ($msg && ($msg->created_by == $authUserID || $this->checkIfSprAdmin())){
             $msg->updated_by = $authUserID;
             $msg->deleted_at = date("Y-m-d H:i:s");
             $msg->save();
         }

         return response()->json([
             "success"           => true,
         ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    function processMessage(Request $request, $flightID){
        $authUserID = Auth::user()->id;
        $key = $request->get('key');
        $message = $request->get('message');

        // Parse Message


        // Gate Boarding & Parking
        $flight = Flight::with([
            "message",
            "flightNumber",
            "flightNumber.airline",
            "flightStaff"
        ])->find($flightID);

        if ($flight){
            if ($request->get("gate_boarding")){
                $flight->gate_boarding = $request->get("gate_boarding") ? strtoupper($request->get("gate_boarding")) : null;
            }
            if ($request->get("gate_parking")){
                $flight->gate_parking = $request->get("gate_parking") ? strtoupper($request->get("gate_parking")) : null;
            }
            if ($request->get("ckin_num")){
                $flight->ckin_num = $request->get("ckin_num") ? strtoupper($request->get("ckin_num")) : null;
            }
            $flight->save();
        }

        $properties = [
            "gate"          => $request->get("gate_boarding") ? strtoupper($request->get("gate_boarding")) : null,
            "pos"           => $request->get("gate_parking") ? strtoupper($request->get("gate_parking")) : null,

            "seat_a"        => $request->get("seat_a"),
            "seat_b"        => $request->get("seat_b"),
            "seat_c"        => $request->get("seat_c"),
            "seat_d"        => $request->get("seat_d"),
            "seat_e"        => $request->get("seat_e"),

            "chkd_bag_pcs"  => $request->get("chkd_bag_pcs"),
            "chkd_bag_wt"   => $request->get("chkd_bag_wt"),
            "gate_bag_pcs"  => $request->get("gate_bag_pcs"),
            "gate_bag_wt"   => $request->get("gate_bag_wt"),

            "trc_rmks"      => $request->get("trc_rmks") ? strtoupper($request->get("trc_rmks")) : null,
            "spvr_rmks"     => $request->get("spvr_rmks") ? strtoupper($request->get("spvr_rmks")) : null,
            "ckin_num"      => $request->get("ckin_num") ? strtoupper($request->get("ckin_num")) : null,
            "ckin_ctr"      => $request->get("ckin_ctr") ? strtoupper($request->get("ckin_ctr")) : null,
            "ckin_rmks"     => $request->get("ckin_rmks") ? strtoupper($request->get("ckin_rmks")) : null,

            "materials_secured_ckin"      => $request->get("materials_secured_ckin"),
            "materials_secured_gate"      => $request->get("materials_secured_gate"),

            "ckin_staff"    => $request->get("ckin_staff") ? strtoupper($request->get("ckin_staff")) : null,
            "gate_staff"    => $request->get("gate_staff") ? strtoupper($request->get("gate_staff")) : null,

            // Added
            "gate_spvr"     => $request->get("gate_spvr") ? strtoupper($request->get("gate_spvr")) : null,
            "specials"     => $request->get("specials") ? strtoupper($request->get("specials")) : null,

            "trc"           => $request->get("trc")  ? strtoupper($request->get("trc")) : null,
            "spvr"          => $request->get("spvr") ? strtoupper($request->get("spvr")) : null,

            "crew"          => $request->get("crew"),
            "capt"          => $request->get("capt")  ? strtoupper($request->get("capt")) : null,

            "lofo_staff"    => $request->get("lofo_staff")  ? strtoupper($request->get("lofo_staff")) : null,
            "lofo_rmks"     => $request->get("lofo_rmks")  ? strtoupper($request->get("lofo_rmks")) : null,
            "lofo_start"    => $request->get("lofo_start"),
            "lofo_end"      => $request->get("lofo_end"),
        ];

        $flightCom = FlightCom::getOrCreateFlightCom($flightID, $authUserID, $key);

        $changesFound = false;
        foreach ($properties as $var => $value ) {
            if (!($value || $value === 0 || $value === "0")){
                continue;
            }

            if ($flightCom->{$var} != $value){
                $changesFound = true;
            }

            $flightCom->{$var} = $value;
        }
        if ($changesFound)
        {
            $flightCom->updated_by = $authUserID;
        }
        $flightCom->save();

        /*
         * OLD structure
        $flightComDetails = FlightComDetails::create([
                "flight_com_id" => $flightCom->id,
                "created_by"    => $authUserID,
                "message"       => $message ? $message : null,
                "trc_rmks"      => $request->get("trc_rmks"),
                "spvr_rmks"     => $request->get("spvr_rmks"),
            ] + $properties);
        */

        // NEW structure
        $flightComDetails = FlightComDetails::createMessage($flightCom, $authUserID, $message,
            $request->get('enc'), $request->get('ct'), $request->get('iv'), $request->get('s'));

        // Add to prop
        $checkInProps = [
            "aircraft_id"       => $request->get("aircraft_id") === "0" ? null : $request->get("aircraft_id"),

            "pax_a_booked"      => $request->get("pax_a_booked"),
            "pax_c_booked"      => $request->get("pax_c_booked"),
            "pax_w_booked"      => $request->get("pax_w_booked"),
            "pax_y_booked"      => $request->get("pax_y_booked"),
            "pax_inf_booked"    => $request->get("pax_inf_booked"),

            "pax_a"             => $request->get("pax_a"),
            "pax_c"             => $request->get("pax_c"),
            "pax_w"             => $request->get("pax_w"),
            "pax_y"             => $request->get("pax_y"),
            "pax_jmp"           => $request->get("pax_jmp"),
            "pax_adults"        => $request->get("pax_adults"),
            "pax_m"             => $request->get("pax_m"),
            "pax_f"             => $request->get("pax_f"),
            "pax_ch"            => $request->get("pax_ch"),
            "pax_inf"           => $request->get("pax_inf"),

            "dow"               => $request->get("dow"),
            "doi"               => $request->get("doi"),
            "mtow"              => $request->get("mtow"),
            "rtow"              => $request->get("rtow"),

            "tof"               => $request->get("tof"),
            "trip"              => $request->get("trip"),
            "eet"               => $request->get("eet"),
            "taxi"              => $request->get("taxi"),
            "block"             => $request->get("block"),
            "altn"              => $request->get("altn"),

//            FHR
            "missed_artg"       => $request->get("missed_artg"),
            "delay"             => $request->get("delay"),
            "ckn"               => $request->get("ckn"),
            "ramp"              => $request->get("ramp"),
            "safety"            => $request->get("safety"),
            "denied_boarding1"  => $request->get("denied_boarding1"),
            "denied_boarding2"  => $request->get("denied_boarding2"),
            "denied_boarding3"  => $request->get("denied_boarding3"),
            "denied_boarding4"  => $request->get("denied_boarding4"),
            "denied_boarding5"  => $request->get("denied_boarding5"),
            "denied_boarding6"  => $request->get("denied_boarding6"),
        ];


        if ($flight && (airlineModule() || (ghaModule() && !$flight->ldm))){
            $checkInProps["capacity_c"] = $request->get("capacity_c");
            $checkInProps["capacity_y"] = $request->get("capacity_y");
        }

        FlightComCheckin::createRecord($flight, $flightComDetails, $properties + $checkInProps);
        // End

        // ULD Containers
        $rowULD = $request->get("uld_row");
        if ($rowULD && count($rowULD)) {
            // ULD
            $flightContainerInfo = $this->getFlightContainerInfo($flightID);

            $posULD     = $request->get("uld_pos");
            $destULD    = $request->get("uld_dest");
            $contentULD = $request->get("uld_content");
            $typeULD    = $request->get("uld_type");
            $numULD     = $request->get("uld_num");
            $ownerULD   = $request->get("uld_owner");
            $wtULD      = $request->get("uld_wt");
            $pcsULD     = $request->get("uld_pcs");
            $vrULD      = $request->get("uld_vr");
            $notocULD   = $request->get("uld_notoc");
            $notocDetailsULD = $request->get("uld_notoc_details");


            $airportIDs     = Airport::whereNotNull("iata")->pluck("id", "iata")->all();
            $airline        = $flight && $flight->flightNumber ? $flight->flightNumber->airline : null;
            $airlineIDs     = Airline::whereNotNull("iata")->pluck("id", "iata")->all();
            $airlineICAOIDs = Airline::whereNotNull("icao")->pluck("id", "icao")->all();

            // Create Info
            if (!$flightContainerInfo){
                $flightContainerInfo = new FlightContainerInfo();
                $flightContainerInfo->flight_id = $flightID;
                $flightContainerInfo->manual = true;
                $flightContainerInfo->save();
            }

            foreach ($rowULD as $i => $rowNumber) {
                if (!$rowNumber){
                    continue;
                }

                /*
                if (preg_match('/(\d+)\s*(\w+)?/', $pos, $posAndNum)) {
                    $rowNumber = isset($posAndNum[1]) ? $posAndNum[1] : null;
                    $position  = isset($posAndNum[2]) ? $posAndNum[2] : null;
                }
                else {
                    continue;
                }
                */

                $position = isset($posULD[$i]) && $posULD[$i] ? strtoupper(trim($posULD[$i])) : null;

                $containerDetails = [
                    "rowNumber"         => $rowNumber,
                    "position"          => $position,
                    "containerTypeName" => isset($typeULD[$i]) && $typeULD[$i] ? strtoupper(trim($typeULD[$i])) : null,
                    "containerNumber"   => isset($numULD[$i]) && $numULD[$i] ? trim($numULD[$i]) : null,

                    "owner"             => isset($ownerULD[$i]) && $ownerULD[$i] ? strtoupper(trim($ownerULD[$i])) : null,
                    "destinationAirport"=> isset($destULD[$i]) && $destULD[$i] ? strtoupper(trim($destULD[$i])) : null,

                    "weight"            => isset($wtULD[$i]) && $wtULD[$i] ? trim($wtULD[$i]) : null,
                    "loadedType"        => isset($contentULD[$i]) && $contentULD[$i] ? strtoupper(trim($contentULD[$i])) : null,
                    "vr"                => isset($vrULD[$i]) && ($vrULD[$i] || $vrULD[$i] == 0 || $vrULD[$i] == "0") ? trim($vrULD[$i]) : null,

                    "bag_pcs"           => isset($pcsULD[$i]) && $pcsULD[$i] ? $pcsULD[$i] : null,
                    "notoc"             => isset($notocULD[$i]) && $notocULD[$i] ? true : null,
                    "notoc_details"     => isset($notocDetailsULD[$i]) && $notocDetailsULD[$i] ? $notocDetailsULD[$i] : null,
                ];

                // Historic Data Com Containers
                FlightComContainer::insert([
                    "flight_com_id"         => $flightCom->id,
                    "flight_com_details_id" => $flightComDetails ? $flightComDetails->id : null,
                    "dest"                  => $containerDetails["destinationAirport"],
                    "pos"                   => $rowNumber.($position ? $position : ""),
                    "owner"                 => $containerDetails["owner"],
                    "content"               => $containerDetails["loadedType"],
                    "type"                  => $containerDetails["containerTypeName"],
                    "num"                   => $containerDetails["containerNumber"],
                    "pcs"                   => $containerDetails["bag_pcs"],
                    "wt"                    => $containerDetails["weight"],
                    "vr"                    => $containerDetails["vr"],
                    "notoc"                 => $containerDetails["notoc"],
                    "notoc_details"         => $containerDetails["notoc_details"],
                    "created_by"            => $authUserID,
                ]);

                // Find Container Type
                $containerType = ContainerType::findByName($containerDetails);

                // Find Container
                $container = Container::find($containerType, $containerDetails, $airline, $airlineIDs, $airlineICAOIDs);

                // Insert To Flights List
                $flightContainerIds[] = FlightContainer::insertContainerToFlight(null, $flightContainerInfo, $containerType, $container, $containerDetails, $airportIDs);

                $readyData[] = $containerDetails;
            }
        }

        // Remove Positions
        $removeUldPos = $request->get("remove_uld_pos");
        if ($removeUldPos){
            if (!isset($flightContainerInfo)){
                // ULD
                $flightContainerInfo = $this->getFlightContainerInfo($flightID);
            }

            foreach ($removeUldPos as $pos) {
                if (preg_match('/(\d+)\s*(\w+)?/', $pos, $posAndNum)) {
                    $rowNumber = isset($posAndNum[1]) ? $posAndNum[1] : null;
                    $position  = isset($posAndNum[2]) ? $posAndNum[2] : null;
                }
                else {
                    continue;
                }

                $container = FlightContainer::where("info_id", $flightContainerInfo->id)
                                            ->where("row_number", $rowNumber)
                                            ->where("position", $position)
                                            ->get();

                $notes = "Container not found";
                if ($container && count($container)){
                    FlightContainer::where("info_id", $flightContainerInfo->id)
                                    ->where("row_number", $rowNumber)
                                    ->where("position", $position)
                                    ->delete();

                    $notes = "POS removed";
                }

                FlightComContainer::insert([
                    "flight_com_id"         => $flightCom->id,
                    "flight_com_details_id" => $flightComDetails ? $flightComDetails->id : null,
                    "pos"                   => $pos,
                    "remove_uld"            => true,
                    "notes"                 => $notes,
                    "created_by"            => $authUserID,
                ]);
            }
        }

        return [$flight];
    }

    function getFlightContainerInfo($flightID){

        return FlightContainerInfo::where("manual", true)
                                ->where("flight_id", $flightID)
                                ->whereNull("deleted_at")
                                ->first();

    }

    public function unseenMessages(Request $request){
        $date = $request->get("date");

        $archivedData = $this->isArchivedDate($date);

        $authUserID = Auth::user()->id;
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs, $date);

        return response()->json([
            "success"           => true,
            "unseen"            => $archivedData ? [] : $unseen,
            "counter"           => $counter,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }


    const PREV_MESSAGES_LENGTH = 30;
    const READ_MESSAGES_LENGTH = 10;

    public function getFlightMessages($userID, $flightID, $date, $lastMessageID = null, $firstMessageID = null, $initialLoad = null){
        $flightCom = FlightCom::where("flight_id", $flightID)
            ->first();

        $created = $flightCom == null;

        if (!$flightCom) {
            $flightCom = new FlightCom();
            $flightCom->created_by = $userID;
            $flightCom->flight_id = $flightID;
            $flightCom->key = generateRandomPassword(10);
            $flightCom->save();
        }

        // Archived data or not
        $archivedData = $flightCom->archived; // $this->isArchivedDate($date);

        $pictures = [];
        if ($archivedData){
            $details = FlightComDetailsArchive::with([
                                            "checkin", "checkin.aircraft", "pts", "ssr", "containers", "pictures", "createdUser",
                                            "createdUser.location","createdUser.location.airport",
                                            "flightStaff", "flightStaff.user", "flightStaff.service",
                                        ])
                ->where("flight_com_id", $flightCom->id);

            if ($firstMessageID){
                $details->where("flight_com_details_id", "<", $firstMessageID);
            }
            else if ($lastMessageID){
                $details->where("flight_com_details_id", ">", $lastMessageID);
            }


            // Deleted Messages
            $deletedMessages = FlightComDetailsArchive::select(["id", "updated_by"])
                                                ->where("flight_com_id", $flightCom->id)
                                                ->whereNotNull("deleted_at");

            if ($initialLoad){
                $pictures = FlightComPicture::join("flights__com_details_archive", "flights__com_details_archive.flight_com_details_id", "=", "flights__com_pictures.flight_com_details_id" )
                                ->whereNull("flights__com_details_archive.deleted_at")
                                ->whereNull("flights__com_pictures.deleted_at")
                                ->where("flights__com_pictures.flight_com_id", $flightCom->id)
                                ->orderBy("flights__com_pictures.id")
                                ->get([
                                    "flights__com_pictures.*"
                                ]);
            }

        }
        else {
            $details = FlightComDetails::with([
                                            "checkin", "checkin.aircraft", "pts", "ssr", "containers", "pictures", "createdUser",
                                            "createdUser.location","createdUser.location.airport",
                                            "flightStaff", "flightStaff.user", "flightStaff.service",
                                        ])->where("flight_com_id", $flightCom->id);

            if ($firstMessageID){
                $details->where("id", "<", $firstMessageID);
            }
            else if ($lastMessageID){
                $details->where("id", ">", $lastMessageID);
            }


            // Deleted Messages
            $deletedMessages = FlightComDetails::select(["id", "updated_by"])
                                                ->where("flight_com_id", $flightCom->id)
                                                ->whereNotNull("deleted_at");

            if ($initialLoad){
                $pictures = FlightComPicture::join("flights__com_details", "flights__com_details.id", "=", "flights__com_pictures.flight_com_details_id")
                    ->whereNull("flights__com_details.deleted_at")
                    ->whereNull("flights__com_pictures.deleted_at")
                    ->where("flights__com_pictures.flight_com_id", $flightCom->id)
                    ->orderBy("flights__com_pictures.id")
                    ->get([
                        "flights__com_pictures.*"
                    ]);
            }

        }

        $details = $details->orderBy("id")
                            ->get();

        // Deleted Messages
        $deletedMessages = $deletedMessages->pluck("updated_by", "id")->all();

        $uIndex = $this->isMessagesUnseen($userID, $flightCom, $details, $archivedData);

//        $enc = cryptoJsAesEncrypt("agt", PRE_KEY. $flightCom->key.POST_KEY);
//        debug($enc);
//        debug(cryptoJsAesDecrypt($enc, PRE_KEY. $flightCom->key.POST_KEY));

        foreach ($details as $i => $each) {
            if ($each->enc){
                $each->message = cryptoJsAesDecrypt($each->enc, PRE_KEY. $flightCom->key.POST_KEY);
                debug($each->message);
            }
        }

        $totalMsgs = count($details);

        if ($uIndex){
            $k = $uIndex - self::READ_MESSAGES_LENGTH > 0 ? $uIndex - self::READ_MESSAGES_LENGTH : 0;
        }
        else {
            $length = $firstMessageID ? self::PREV_MESSAGES_LENGTH : self::READ_MESSAGES_LENGTH;
            $k = $totalMsgs - $length > 0 ? $totalMsgs - $length : 0;
        }

        $messages = [];
        for($i = $k; $i < $totalMsgs; $i++){
            $record = $this->addDetailsToRecord($details[$i]);

            $relations = $record->getRelations();
            unset($relations['createdUser']);

            if ($record->checkin && $record->checkin->aircraft){
                $record->checkin->ac = getAircraft($record->checkin->aircraft);
                unset($relations['checkin']['aircraft']);
            }

            $record->setRelations($relations);

            $messages[] = $record;
        }


        return [
            "key"               => $flightCom ? $flightCom->key     : null,
            "messages"          => $messages,
            "deleted"           => $deletedMessages,
            "com"               => $flightCom ? $flightCom : null,
            "created"           => $created,
            "pictures"          => $pictures,
//            "com"       => count($messages) ? $com : null,
//            "key"       => $com && count($messages) ? $com->key     : null,
        ];
    }

    function addDetailsToRecord($flightComDetails){
        $flightComDetails->timestamp = baseDateFormat($flightComDetails->created_at, true);
        $flightComDetails->deleted_timestamp = $flightComDetails->deleted_at ?  baseDateFormat($flightComDetails->deleted_at, true) : null;

        $flightComDetails->user = getUserName($flightComDetails->createdUser);
        $flightComDetails->base = getUserLocation($flightComDetails->createdUser);

        $flightComDetails->user_thumb = null;
        $userThumb = $flightComDetails->createdUser ? $flightComDetails->createdUser->thumb : null;
        if ($userThumb && File::exists(public_path("storage/users/{$userThumb}"))){
            $flightComDetails->user_thumb = $userThumb;
        }
        $flightComDetails->created_user = null;

        if ($flightComDetails->flightStaff){
            foreach ($flightComDetails->flightStaff as &$each) {
                $each->user_name = getUserName($each->user);
                $each->location = Location::getUserLocationAirport($each->user->location_id);
                $each->service_name = $each->service ? $each->service->abbr : "";

                $relations = $each->getRelations();
                unset($relations['user'], $relations["service"]);
                $each->setRelations($relations);
            }
        }

        return $flightComDetails;
    }

    function isMessageUnseen($userID, $flightCom, $flightComDetails){

        if ($flightComDetails->created_by == $userID){
            return false;
        }

        $userViewed = FlightComUserViews::where("flight_com_id", $flightCom->id)
            ->where("flight_com_details_id", $flightComDetails->id)
            ->where("user_id", $userID)
            ->first();

        if (!$userViewed) {

            FlightComUserViews::create([
                "flight_com_id"         => $flightCom->id,
                "flight_com_details_id" => $flightComDetails->id,
                "user_id"               => $userID,
                "created_at"            => date("Y-m-d H:i:s")
            ]);
        }

        return $userViewed == null;
    }


    function isMessagesUnseen($userID, $flightCom, $flightComDetails, $archivedData){

        if ($archivedData){
            $detailsIDs = $flightComDetails->pluck("flight_com_details_id")->all();
        }
        else {
            $detailsIDs = $flightComDetails->pluck("id")->all();
        }

        if ($archivedData){
            $allUserViews = FlightComUserViewsArchive::where("flight_com_id", $flightCom->id)
                ->whereIn("flight_com_details_id", $detailsIDs)
                ->where("user_id", $userID)
                ->get([
                    "flight_com_details_id"
                ])
                ->pluck("flight_com_details_id")
                ->all();

        }
        else {
            $allUserViews = FlightComUserViews::where("flight_com_id", $flightCom->id)
                ->whereIn("flight_com_details_id", $detailsIDs)
                ->where("user_id", $userID)
                ->get([
                    "flight_com_details_id"
                ])
                ->pluck("flight_com_details_id")
                ->all();

        }

        $insertionArray = [];
        $uIndex = null;
        $curDate = date("Y-m-d H:i:s");
        foreach ($flightComDetails as $i => $each) {
            if (!in_array($each->id, $allUserViews)){

                $insertionArray[] = [
                    "flight_com_id"         => $flightCom->id,
                    "flight_com_details_id" => $each->id,
                    "user_id"               => $userID,
                    "created_at"            => $curDate,
                ];

                if (!$uIndex){
                    $uIndex = $i;
                }
            }
        }

        if (!$archivedData && count($insertionArray)){
            FlightComUserViews::insert($insertionArray);
        }

        return $uIndex;
    }


    function getUnseenFlightMessagesCount2($userID, $flightIDs){
        if (!$userID || !count($flightIDs)){
            return [[], []];
        }

        $coms = FlightCom::with(["details",
            "flight",
            "flight.flightNumber",
            "flight.flightNumber.airline",
            "flight.flightNumber.departureAirport",
            "flight.flightNumber.arrivalAirport"
        ])
            ->whereNull("deleted_at")
            ->whereIn("flight_id", $flightIDs)
            ->get();

        $unseen = $counter = [];
        if ($coms && count($coms)) {
            foreach ($coms as $each) {
                $detailIDs = $each->details->where("created_by", "!=", $userID)->pluck("id")->toArray();
                $seen = FlightComUserViews::where("user_id", $userID)
                                        ->whereIn("flight_com_details_id", $detailIDs)
                                        ->count();

                if ((int)(count($detailIDs) - $seen) > 0) {
                    $unseen[$each->flight_id] = [
                        "f"     => getFlightNumber($each->flight)." | " .getFlightSector($each->flight),
                        "c"     => (int)(count($detailIDs) - $seen),
                    ] ;
                }

                if ((int)(count($each->details))){
                    $counter[] = $each->flight_id;

                }
//                debug(count($detailIDs) . " SEEN:" . $seen);
            }
        }

        return [$unseen, $counter];
    }

    function getUnseenFlightMessagesCount($userID, $flightIDs, $date = null){
        if (!$userID || !$flightIDs || !count($flightIDs)){
            return [[], []];
        }

        $archivedData = $this->isArchivedDate($date);

        if ($archivedData){
            $details = FlightComDetailsArchive::join("flights__com", "flights__com.id", "=", "flights__com_details_archive.flight_com_id")
                                                ->join("flights", "flights.id", "=", "flights__com.flight_id")
                                                ->whereNull("flights__com.deleted_at")
                                                ->whereIn("flights__com.flight_id", $flightIDs)
                                                ->get([
                                                    "flights__com_details_archive.flight_com_details_id as id",
                                                    "flights__com_details_archive.created_by",
                                                    "flights__com_details_archive.flight_com_id",
                                                    "flights__com.flight_id",
                                                    "flights.flight_number_id",
                                                ]);
        }
        else {
            $details = FlightComDetails::join("flights__com", "flights__com.id", "=", "flights__com_details.flight_com_id")
                                        ->join("flights", "flights.id", "=", "flights__com.flight_id")
                                            ->whereNull("flights__com.deleted_at")
                                            ->whereIn("flights__com.flight_id", $flightIDs)
                                            ->get([
                                                "flights__com_details.id",
                                                "flights__com_details.created_by",
                                                "flights__com_details.flight_com_id",
                                                "flights__com.flight_id",
                                                "flights.flight_number_id",
                                            ]);
        }

        $allDetailIDs = $detailsIDs = $detailsIDsByCom = [];
        $flightIDsByCom = $flightNumbersIDsByCom =  $flightNumbersIDs = [];
        foreach ($details as $each) {

            $flightIDsByCom[$each->flight_com_id] = $each->flight_id;
            $flightNumbersIDsByCom[$each->flight_com_id] = $each->flight_number_id;


            if (!isset($detailsIDsByCom[$each->flight_com_id])){
                $detailsIDsByCom[$each->flight_com_id] = [];

                $allDetailIDs[$each->flight_com_id] = [];
            }
            // Add Details
            $allDetailIDs[$each->flight_com_id][] = $each->id;

            if ($each->created_by != $userID){
                $detailsIDsByCom[$each->flight_com_id][] = $each;
                $detailsIDs[] = $each->id;
            }

            if (!in_array($each->flight_number_id, $flightNumbersIDs)){
                $flightNumbersIDs[] = $each->flight_number_id;
            }
        }

        $flightNumbers = FlightNumber::with(["airline", "departureAirport", "arrivalAirport"])
                                        ->whereIn("id", $flightNumbersIDs)
                                        ->get();

        $fnByFlightNumberID = $sectorsByFlightNumberID = [];
        foreach ($flightNumbers as $each) {
            $sectorsByFlightNumberID[$each->id] = getSector($each);
            $fnByFlightNumberID[$each->id] = getFlightNumberFull($each);
        }

        if ($archivedData){
            $userViews = FlightComUserViewsArchive::where("user_id", $userID);
        }
        else {
            $userViews = FlightComUserViews::where("user_id", $userID);
        }


        $userViews = $userViews->whereIn("flight_com_details_id", $detailsIDs)
                                ->groupBy("flight_com_id")
                                ->get([
                                    "flight_com_id",
                                    DB::raw("COUNT(*) as counter"),
                                ])
                                ->pluck("counter", "flight_com_id")
                                ->all();

        $unseen = $counter = [];
        foreach ($detailsIDsByCom as $comID => $details) {
            $counterDetailIDs = count($details);

            $flightID = $flightIDsByCom[$comID];
            $flightNumberID = $flightNumbersIDsByCom[$comID];

            $seen = isset($userViews[$comID]) ? $userViews[$comID] : 0;

            $fn = isset($fnByFlightNumberID[$flightNumberID]) ? $fnByFlightNumberID[$flightNumberID] : "";
            $sector = isset($sectorsByFlightNumberID[$flightNumberID]) ? $sectorsByFlightNumberID[$flightNumberID] : "";

            if ( ($counterDetailIDs - $seen) > 0) {
                $unseen[$flightID] = [
                    "f"     => $fn." | " .$sector,
                    "c"     => $counterDetailIDs - $seen,
                ] ;
            }

            if (isset($allDetailIDs[$comID]) && count($allDetailIDs[$comID])){
                $counter[] = $flightID;
            }
        }

        return [$unseen, $counter];
    }

    const MAX_EXECUTION_TIME = 10000,
            MAX_INPUT_TIME = 1000;

    public function uploadPicture(Request $request){

        debug($request->all());

        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('max_input_time', self::MAX_INPUT_TIME);
        ini_set('upload_max_filesize', '64M');
        ini_set('post_max_size', '64M');
        ini_set('memory_limit', '256M');

        $authUserID = Auth::user()->id;
        $flightID = $request->get("flight_id");
        $key = $request->get('key');
        $message = $request->get('message');
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        $flightCom = FlightCom::where("flight_id", $flightID)
                                ->first();

        if (!$flightCom){
            $flightCom = new FlightCom();
            $flightCom->created_by = $authUserID;
            $flightCom->flight_id = $flightID;
            $flightCom->key = generateRandomPassword(10);
            $flightCom->save();
        }

        $flightComDetails = null;

        if ($fileImage = $request->file("file")) {
            // foreach ($request->file("img") as $i => $fileImage) {
                $ext = $fileImage->getClientOriginalExtension();

            if (in_array(strtolower($ext), ["pdf", "jpg", "jpeg", "png", "gif"])) {

                $flightComDetails = FlightComDetails::createMessage($flightCom, $authUserID, $message);

                $size = $fileImage->getSize() ? $fileImage->getSize() : $fileImage->getClientSize();
                if ($size){
                    $size = round($size / (1024 * 1024), 1);
                }

                $flightComPicture = new FlightComPicture();
                $flightComPicture->flight_com_id = $flightCom->id;
                $flightComPicture->flight_com_details_id = $flightComDetails->id;
                $flightComPicture->notes = $request->get("picture_notes");
                $flightComPicture->size = $size;
                $flightComPicture->type = $ext;
                $flightComPicture->created_by = $authUserID;
                $flightComPicture->save();


                $title = "f" . $flightID . "_u" . $authUserID . "_" . date("ymdHis"); // . "_" . $i;

                try {
                    saveImage($fileImage, strtolower($ext), $flightComPicture, "flight-com/pictures/", $title, false);
                }
                catch(\Exception $e){
                    $flightComPicture->error = ($e->getCode() ? $e->getCode()." / " : ""). $e->getMessage();
                    $flightComPicture->save();
                }
            }
        }

        $flight = Flight::with(["message", "flightStaff"])->find($flightID);

        $flightDate = getFlightDepartureDate($flight, true);

        //$archivedData = $this->isArchivedDate($flightDate);

        $messages = $this->getFlightMessages($authUserID, $flightID, $flightDate, $request->get("last_message_id"));

        $flightCom = isset($messages["com"]) && $messages["com"] ? $messages["com"] : null;

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs, $flightDate);

        $uld = []; //$this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $flightCom && $flightCom->archived ? [] : $unseen, // $archivedData ? [] : $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "deleted"           => $messages["deleted"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    public function picture($flight_com_picture, Request $request)
    {
        $type = $request->get("type") ? $request->get("type") : "thumb";

        if ($flight_com_picture->{$type}){
            $filename = $flight_com_picture->{$type};
        }
        else {
            $filename = $flight_com_picture->file;
        }

        $url = storage_path("app/public/flight-com/pictures/$filename");

        if ($filename && File::exists($url)) {
            $mimeType = getMimeType($filename);
            $fileContents = File::get($url);

            if ($request->has('download')){
                return response()->download($url, $filename, array('Content-Type' => $mimeType));
            }
            else {
                return response()->make($fileContents, 200,array('Content-Type' => $mimeType));
            }
        }
    }
}


/*
 * array:28 [
  "_token" => "TtqxvMuU4SvkU0eQj2UEUIWItjN3DsepbfTHXgza"
  "pr_0" => null
  "pf_0" => null
  "r_0" => null
  "f_0" => null
  "s_2" => array:2 [
    0 => "1180"
    1 => "926"
  ]
  "pr_2" => array:2 [
    0 => "2022-01-11 16:00"
    1 => "2022-01-11 16:00"
  ]
  "pf_2" => array:2 [
    0 => "2022-01-11 17:45"
    1 => "2022-01-11 17:00"
  ]
  "r_2" => array:2 [
    0 => "2022-01-11 16:00"
    1 => null
  ]
  "f_2" => array:2 [
    0 => null
    1 => null
  ]
  "s_3" => array:1 [
    0 => "1180"
  ]
  "pr_3" => array:1 [
    0 => "2022-01-11 17:45"
  ]
  "pf_3" => array:1 [
    0 => "2022-01-11 18:30"
  ]
  "r_3" => array:1 [
    0 => null
  ]
  "f_3" => array:1 [
    0 => null
  ]
  "s_6" => array:1 [
    0 => "926"
  ]
  "pr_6" => array:1 [
    0 => "2022-01-11 17:00"
  ]
  "pf_6" => array:1 [
    0 => "2022-01-11 19:00"
  ]
  "r_6" => array:1 [
    0 => null
  ]
  "f_6" => array:1 [
    0 => null
  ]
  "s_14" => array:1 [
    0 => "1180"
  ]
  "pr_14" => array:1 [
    0 => "2022-01-11 18:30"
  ]
  "pf_14" => array:1 [
    0 => "2022-01-11 18:45"
  ]
  "r_14" => array:1 [
    0 => null
  ]
  "f_14" => array:1 [
    0 => null
  ]
  "flight_id" => "539763"
  "flight_ids" => "539607,544103,539763,543978,539337,539468"
  "key" => "P!R#E@#K$1E25YS94*KP5%O8^S9&amp;T1(K)*E&amp;Y"
]
debug

 */
