<?php

namespace App\Http\Controllers;

use App\Models\Airline;
use App\Models\Delay;
use App\Models\DelayGroup;
use App\Models\EmailNotification;
use Illuminate\Http\Request;

use App\Http\Requests;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DelayCodeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {

        $delays = Delay::with(["group", "emailNotification", "airline"])
                            ->whereNull("deleted_at")
                            ->orderBy("airline_id")
                            ->orderBy("code")
                            ->get();


        $delaysAirline = Delay::with(["group", "emailNotification", "airline"])
                                ->whereNull("deleted_at")
                                ->whereNotNull("airline_id")
                                ->orderBy("airline_id")
                                ->orderBy("code")
                                ->get();


        $this->viewData = [
            'delays'            => $delays,
            'delaysAirline'     => $delaysAirline,
            'delaysIATA'        => $delays->where("airline_id", null)->all(),
        ];

        return view("delay-code/index", $this->viewData);
    }

    public function applyChanges(Request $request){
        debug($request->all());

        if ($request->get("ids")) {
            $ids = explode(",", $request->get("ids"));
            $groupID = $request->get("group_id");
            $airlineID = $request->get("airline_id");

            Delay::whereIn("id", $ids)
                    ->update(["email_notification_id" => $groupID]);

        }

        return redirect()->back();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $airlineID = $request->get("current_airline_id");

        $this->viewData = [
            'airlines'      => ["" => "IATA"] + Airline::listHandlingAirlinesWithCode("iata"),
            'groups'        => ["" => "Select"] + DelayGroup::get([DB::raw("CONCAT(IF(name is NULL, '', name), ' | ', IF (name_ln is NULL, '', name_ln)) AS fullname"),  "id"])->pluck("fullname", "id")->all(),
            'emailGroups'   => ["" => "Select"] + EmailNotification::pluck("name", "id")->all(),
        ];

        return view("delay-code/create", $this->viewData);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $code = $request->get("code");
        $alphaCode = $request->get("alpha_code");

        $airline = $request->get("airline");

        if ($this->duplicateCode($code, $alphaCode, $airline)){
            return redirect()->back()->withInput($request->all());
        }

        $delay = new Delay();
        $delay->code = $code;
        $delay->alpha_code = $alphaCode;
        $delay->description = $request->get("description");
        $delay->description_ln = $request->get("description_ln");
        $delay->explanation = $request->get("explanation");
        $delay->explanation_ln = $request->get("explanation_ln");
        $delay->airline_id = $airline;
        $delay->email_notification_id = $request->get("email_notification") ? $request->get("email_notification") : null;
        if ($request->get("new_group") || $request->get("new_group_ln")){
            $delay->delay_group_id = $this->findOrCreateDelayGroup($request->get("new_group"), $request->get("new_group_ln"));
        }
        else {
            $delay->delay_group_id = $request->get("group");
        }
        $delay->save();

        if ($request->get("current_airline_id")){
            return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=dl");
        }

        return redirect()->to("delay-code");
    }

    public function findOrCreateDelayGroup($newGroup, $newGroupLn){
        if ($newGroup){
            $search = DelayGroup::where("name", $newGroup)->first();

            if (!$search){
                $search = new DelayGroup();
                $search->name = $newGroup;
            }

            $search->name_ln = $newGroupLn;
            $search->save();

            return $search->id;
        }

        if ($newGroupLn){
            $search = DelayGroup::where("name_ln", $newGroupLn)->first();

            if (!$search){
                $search = new DelayGroup();
                $search->name_ln = $newGroupLn;
            }
            $search->save();

            return $search->id;
        }


        return null;
    }

    public function duplicateCode($code, $alphaCode = null, $airline = null, $excludeId = null){

        if ($alphaCode){
            $search = Delay::where(function($sql) use ($code, $alphaCode){
                                $sql->where("code", $code)
                                    ->where("alpha_code", $alphaCode);
            });
        }
        else {
            $search = Delay::where("code", $code);
        }

        if ($excludeId){
            $search->where("id", "!=", $excludeId);
        }

        if ($airline){
            $search->where("airline_id", $airline);
        }

        $search = $search->whereNull("deleted_at")
                         ->get();

        return count($search) > 0;
    }


    public function delete(Request $request, $delay)
    {
        $delay->deleted_at = date("Y-m-d H:i:s");
        $delay->updated_by = Auth::user()->id;
        $delay->save();

        if ($request->get("current_airline_id")){
            return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=dl");
        }

        return redirect()->to("delay-code");
    }

    /**
     * @param $delay
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit($delay)
    {
        $this->viewData = [
            'delay'         => $delay,
            'airlines'      => ["" => "IATA"] + Airline::listHandlingAirlinesWithCode("iata"),
            'groups'        => ["" => "Select"] + DelayGroup::get([DB::raw("CONCAT(IF(name is NULL, '', name), ' | ', IF (name_ln is NULL, '', name_ln)) AS fullname"),  "id"])->pluck("fullname", "id")->all(),
            'emailGroups'   => ["" => "Select"] + EmailNotification::pluck("name", "id")->all(),
        ];

        return view("delay-code/edit", $this->viewData);
    }

    /**
     * @param Request $request
     * @param $delay
     * @return $this|\Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $delay)
    {
        $code = $request->get("code");
        $alphaCode = $request->get("alpha_code");

        $airline = $request->get("airline");
        if ($this->duplicateCode($code, $alphaCode, $airline, $delay->id)){
            return redirect()->back()->withInput($request->all());
        }

        $delay->code = $code;
        $delay->alpha_code = $alphaCode;
        $delay->description = $request->get("description");
        $delay->description_ln = $request->get("description_ln");
        $delay->explanation = $request->get("explanation");
        $delay->explanation_ln = $request->get("explanation_ln");
        $delay->email_notification_id = $request->get("email_notification") ? $request->get("email_notification") : null;
        $delay->airline_id = $airline;

        if ($request->get("new_group") || $request->get("new_group_ln")){
            $delay->delay_group_id = $this->findOrCreateDelayGroup($request->get("new_group"), $request->get("new_group_ln"));
        }
        else {
            $delay->delay_group_id = $request->get("group");
        }
        $delay->save();

        if ($request->get("current_airline_id")){
            return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=dl");
        }

        return redirect()->to("delay-code");
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}
