<?php namespace App\Http\Controllers;

/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 12.09.14

 */

use app\Classes\ProjectConstant;
use App\Http\Requests\Request;
use App\Mail\TemplateEmail;
use App\Models\Airline;
use App\Models\Airport;

use Illuminate\Support\Facades\Validator;
use App\Models\Flight;
use App\Models\User;
use App\Models\FlightCrew;
use Illuminate\Support\Facades\Mail;

class CrewScheduleController extends Controller
{
    const INBOUND = "INBOUND";
    const OUTBOUND = "OUTBOUND";

    /**
     * Get index action crew schedule
     * @return $this|\Illuminate\View\View
     */
    public function index()
    {
        $data = [];
        if (\request("from") && \request("to"))
        {
            $currentDate = \request("date");
            if (!\request("date"))
            {
                $currentDate = \request("from");
            }
            $validator = Validator::make(request()->all(), [
                "from" => "required",
                "to"   => "required"
            ]);
            if ($validator->fails())
            {
                return redirect()->to("crew-schedule/index")
                    ->withInput()
                    ->withErrors($validator);
            }

            $flights = Flight::flightsRange($currentDate . " 00:00:00", $currentDate . " 23:59:59", [
                "flightNumber",
                "Aircraft",
                "flightCrew"
            ], false);

//            $flights =  Flight::handlingFlightsRange($currentDate . " 00:00:00", $currentDate . " 23:59:59",
//                null, null, null, [], null, false, true );

            foreach ($flights as $flight)
            {
                if ($flight->flightCrewItems->count())
                {
                    $flightCrewItems = $flight->flightCrewItems;

                    $captains = $firstOfficers = $flightAttendants = [
                        'general' => [],
                        'standby' => [],
                        'dhc'     => []
                    ];

                    $flightAttendants['sup'] = [];

                    foreach ($flightCrewItems as $flightCrewItem)
                    {
                        if ($flightCrewItem->is_sup) {
                            // SUP
                            $this->addToCrew($flightCrewItem, $flightAttendants, $flightCrewItem->position_order);
                        }
                        else {
                            if ($flightCrewItem->position_type == FCM_CPT_TYPE_ID) {
                                $this->addToCrew($flightCrewItem, $captains, $flightCrewItem->position_order);
                            }
                            else if ($flightCrewItem->position_type == FCM_FA_TYPE_ID) {
                                $this->addToCrew($flightCrewItem, $firstOfficers, $flightCrewItem->position_order);
                            }
                            else if (in_array($flightCrewItem->position_type, [CCM_PSR_TYPE_ID, CCM_CC_TYPE_ID])) {
                                $this->addToCrew($flightCrewItem, $flightAttendants, $flightCrewItem->position_order);
                            }
                        }
                    }

                    ksort($captains['general']);
                    ksort($firstOfficers['general']);
                    ksort($flightAttendants['general']);

                    $flight->flightCrew = [
                        "captains"         => $captains,
                        "firstOfficers"    => $firstOfficers,
                        "flightAttendants" => $flightAttendants
                    ];
                }
            }

            $data = [
                "captains"      => ["" => "Select Captain"] + User::getFlightCrew(FCM_CP_CREW),
                "firstOfficers" => ["" => "Select First Officer"] + User::getFlightCrew(FCM_FO_CREW),
                "cabinCrew"     => ["" => "Select Flight Attendant"] + User::getFlightCrew(CCM_CREW),
                "staff"         => ["" => "Select Staff"] + User::listUsers(),
                "flights"       => $flights,
                "from"          => \request("from"),
                "to"            => \request("to"),
                "date"          => $currentDate
            ];
        }
        return view("crew-schedule/index", $data);
    }

    /**
     * AJAX save flight crew action
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxSaveCrew()
    {
        if (!\request()->ajax())
        {
            App::abort(404);
        }
        $flightId = \request("flight_id");
        $childId  = \request("child_id");
        $captains = \request("captains");
        $first_officers = \request("first_officers");
        $cabin_crew = \request("cabin_crew");

        debug(request()->all());

        $flight   = Flight::find($flightId);
        if ($childId)
        {
            $childFlight = Flight::find($childId);
            if ($childFlight)
            {
                // update child flight crew (using Inputs, like captains, ...)
                $this->updateFlightCrew($childId, self::INBOUND);
            }
        }
        if (!$flight)
        {
            return response()->json([
                "message" => "Not found flight by given ID",
                "error"   => TRUE
            ]);
        }

        // update parent flight crew (using Inputs, like captains, ...)
        $this->updateFlightCrew($flightId, self::OUTBOUND);

        return response()->json([
            "message" => "Flight Crew was successfully updated",
            "success" => TRUE
        ]);
    }

    /**
     * AJAX request send one flight crew emails
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxSendFlightCrewEmail()
    {
        if (!\request()->ajax())
        {
            App::abort(404);
        }
        $flightId = \request("flight_id");
        $childId  = \request("child_id");
        $crewType =  \request("crewType");
        $flight  = Flight::with([
                "flightNumber",
                "aircraft",
                "flightCrewItems"
            ])
            ->find($flightId);

        if (!$flight)
        {
            return response()->json([
                "message" => "Not found flight by given ID",
                "error"   => TRUE
            ]);
        }
        if ($childId)
        {
            $childFlight = Flight::with([
                "flightNumber",
                "Aircraft",
                "flightCrewItems"
            ])
                ->find($childId);
            $flight->child = $childFlight;
        }

        // get flight crew team
        $flightCrew = FlightCrew::flightCrewFullinformation($flightId, $crewType);

        debug($flightCrew);

        $notFoundEmails = [];

        //Crew Admin Emails
        $crewAdminEmails = $this->getCrewAdminEmails($crewType);

        // send for each crew person email
        foreach($flightCrew as $flightCrewItem) {
            // Check If Email Is Set, Otherwise Compile And Later Send As A Log
            if (isset($flightCrewItem->email) && $flightCrewItem->email && !is_null($flightCrewItem->email)) {

                if (in_array(trim($flightCrewItem->position_type), [FCM_CPT_TYPE_ID, FCM_FA_TYPE_ID])) {
                    $subject = "Flight Crew Roster For " . $flightCrewItem->first_name . " " . $flightCrewItem->last_name;
                } else {
                    $subject = "Cabin Crew Roster For " . $flightCrewItem->first_name . " " . $flightCrewItem->last_name;
                }

                Mail::to($crewAdminEmails + [$flightCrewItem->email])
//                    ->bcc("dilovar88@mail.ru")
                    ->send(new TemplateEmail(view("emails/flightCrew/flight_schedule", [
                                            "crewAndFlights" => [
                                                0 => [
                                                    'crew'  =>  $flightCrewItem,
                                                    'flight' => $flight
                                                ]
                                            ]
                                        ])->render()
                    , $subject));


            }else {
                $notFoundEmails[] = $flightCrewItem->first_name . " " . $flightCrewItem->last_name . ' | Email: '. $flightCrewItem->email;
            }
        }

        // Send Log If Crew Emails Were Not Found
        $this::sendNotFoundEmailsAsLog($notFoundEmails, $crewAdminEmails);

        return response()->json([
            "message" => "Crew Emails Successfully Sent",
            "success" => TRUE
        ]);
    }

    /**
     * AJAX request to send email notification for the flights crew
     */
    public function ajaxSendAllFlightsScheduleEmail()
    {
        if (!\request()->ajax())
        {
            App::abort(404);
        }
        $flights =  \request("flights");
        $from =  \request("from");
        $to =  \request("to");
        $crewType =  \request("crewType");

        if (!(is_array($flights) && count($flights)))
        {
            return response()->json([
                "message" => "Wrong data given to controller action",
                "error" => true
            ]);
        }

        $crew = [];

        $flights = Flight::flightsRange($from . " 00:00:00", $to . " 23:59:59", [
            "flightNumber",
            "Aircraft",
            /*"flightCrew"*/
        ], false);

        foreach ($flights as $flight)
        {
            $flightCrew = FlightCrew::flightCrewFullinformation($flight->id, $crewType);
            foreach ($flightCrew as $flightCrewItem) {
                $id = $flightCrewItem->user_id;

                if (!isset($crew[$id])) {
                    $crew[$id] = [];
                }

                if ($flight->child_id) {
                    $childFlight = Flight::with(["flightNumber", "Aircraft" ])
                                            ->find($flight->child_id);
                    $flight->child = $childFlight;
                }

                array_push($crew[$id], ['crew' => $flightCrewItem, 'flight' => $flight]);
            }

        }
        $crewAdminEmails = $this->getCrewAdminEmails($crewType);
        $notFoundEmails = [];

        foreach ($crew as $flightCrewItem) {
            $crewDetails = $flightCrewItem[0]['crew'];
            if (isset($crewDetails->email) && $crewDetails->email && !is_null($crewDetails->email)) {
                if (in_array($crewDetails->position_type, [FCM_CPT_TYPE_ID, FCM_FA_TYPE_ID])) {
                    $subject = "Flight Crew Roster For " . $crewDetails->first_name . " " . $crewDetails->last_name;
                } else {
                    $subject = "Cabin Crew Roster For " . $crewDetails->first_name . " " . $crewDetails->last_name;
                }

                Mail::queue("emails/main_template", view("emails/flightCrew/flight_schedule", [
                        "crewAndFlights" => $flightCrewItem
                    ])
                        ->render()
                , function ($message) use ($crewDetails, $subject, $crewAdminEmails) {
                        // Crew Email
                        $message->to($crewDetails->email);
                        // Add Admin Crew Emails
                        foreach ($crewAdminEmails as $email) {
                            $message->to($email);
                        }
                        // Subject of the Email
                        $message->subject($subject);
                });
            }
            else {
                $notFoundEmails[] = $crewDetails->first_name . " " . $crewDetails->last_name . ' | Email: '. $crewDetails->email;
            }
        }

        // Send Log If Crew Emails Were Not Found
        $this::sendNotFoundEmailsAsLog($notFoundEmails, $crewAdminEmails);

        return response()->json([
            "message" => "Crew Emails Successfully Sent",
            "success" => TRUE
        ]);

    }

    /**
     * Get Crew Admin Email List
     * @param $crewType
     * @return array|bool
     */
    public static function getCrewAdminEmails($crewType){
        if ($crewType) {
            switch ($crewType) {
                case FCM_CREW:
                    $emails = ["fcc@avbis.aero"];
                    break;
                case CCM_CREW:
                    $emails = ["crew.admin@avbis.aero"];
                    break;
            }

            return $emails;
        }
        return false;
    }

    /**
     * Send Log of Not Found Crew Emails to Crew Admins
     * @param $notFoundEmails
     */
    public static function sendNotFoundEmailsAsLog($notFoundEmails, $crewAdminEmails){
        if (count($notFoundEmails)){
            $subject = '/LOG/Not Found Or Wrong Crew Email Addresses.';

            Mail::to($crewAdminEmails)
//                ->bcc("dilovar88@mail.ru")
                ->send(new TemplateEmail(view("emails/general", [
                                        "header"   => "List Of Crew With Wrong Emails:",
                                        "variable" => $notFoundEmails
                                    ])
                                        ->render()
                , $subject));
        }
    }

    /**
     * AJAX request to send cancel notification for the flight crew
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxSendCancelNotification()
    {
        if (!\request()->ajax())
        {
            App::abort(404);
        }
        $flight = Flight::with([
            "flightNumber",
            "Aircraft"
        ])
            ->find(\request("flight_id"));
        if (!$flight)
        {
            return response()->json([
                "message" => "Not found flight by given ID",
                "error"   => TRUE
            ]);
        }
        $subject = "Notification. Flight ".ProjectConstant::getConstants(IATA_CODE)."-".$flight->flightNumber->first()->flight_number." will be operated by other crew.";

        debug(request()->all());
        $users   = array();
        if (is_array(\request("captains")))
        {
            $users = array_merge($users, \request("captains"));
        }
        if (is_array(\request("first_officers")))
        {
            $users = array_merge($users, \request("first_officers"));
        }
        if (is_array(\request("cabin_crew")))
        {
            $users = array_merge($users, \request("cabin_crew"));
        }
        if (is_array($users) && count($users) > 0)
        {
            foreach ($users as $userId)
            {
                $user = User::find($userId);
                if (!$user)
                {
                    continue;
                }

                Mail::to($user->email)
                    ->cc("crew.admin@avbis.aero")
//                    ->bcc("dilovar88@mail.ru")
                    ->send(new TemplateEmail(view("emails/flightCrew/flight_cancel", [
                                            "flight"        => $flight,
                                            "user"          => $user
                                        ])
                                            ->render()
                    , $subject));
            }
        }
        return response()->json([
            "message" => "All cancel notification was successfully sent",
            "success" => TRUE
        ]);
    }

    /**
     * Add item to crew
     * @param $crewItem
     * @param $array
     * @param $positionOrder
     * @return mixed
     */
    private function addToCrew($crewItem, &$array, $positionOrder)
    {
        if ($crewItem->is_sup)
        {
            $array["sup"][] = $crewItem;
        }
        else {
            if ($crewItem->is_standby)
            {
                $array["standby"][] = $crewItem;
            }
            else if ($crewItem->is_dhc)
            {
                $array["dhc"][] = $crewItem;
            }
            else
            {
                $array["general"][(int)$positionOrder] = $crewItem;
            }
        }
    }

    /**
     * Update flight crew items
     * @param $flightId
     * @param $type
     */
    private function updateFlightCrew($flightId, $type)
    {
        // Remove Previously Set Crew
        FlightCrew::removeCrew($flightId);

        // create new crew items
        $this->createFlightCrewItem(\request("captains"), $flightId, $type);
        $this->createFlightCrewItem(\request("first_officers"), $flightId, $type);
        $this->createFlightCrewItem(\request("cabin_crew"), $flightId, $type);
        debug(\request("cabin_crew"));
    }

    /**
     * Create flight crew items
     * @param array $crew
     * @param $flightId
     * @param $flightType
     * @return void
     */
    private function createFlightCrewItem($crew, $flightId, $flightType)
    {
        if (is_array($crew))
        {
            foreach ($crew as $each)
            {
                $userId = (int)$each["user_id"];

                $isSUP = isset($each["is_sup"]) && $each["is_sup"];

                if (!$userId || ($flightType == self::INBOUND && ($each["is_dhc"] || $each["is_standby"]))) {
                    continue;
                }

                $argument = ['position_order' => $each["position_order"]];

                FlightCrew::createCrewItem($userId, $flightId, NULL, $argument, $each["is_standby"], $each["is_dhc"], $isSUP);
            }
        }
    }
}
