<?php namespace App\Http\Controllers;
/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 19.08.14

 */

use App\Models\CabinCrew;
use App\Models\CaptainFirstOfficer;
use App\Models\Flight;
use App\Models\FlightCrew;
use App\Models\Position;
use App\Models\UserDepartment;
use App\Repositories\CrewRosterDetailRepository;
use App\Repositories\Interfaces\ICrewRosterDetailRepository;
use App\Repositories\Interfaces\ICrewRosterRepository;
use App\Repositories\Interfaces\IFlightCrewRepository;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use App\Models\FlightNumber;

use App\Models\CrewRosterSettings;
use App\Classes\RosterInitialize;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Session;
use stdClass;
use App\Models\AircraftType;
use App\Models\Hub;
use App\Models\CrewRoster;
use App\Models\Location;

class CrewRosterController extends Controller
{
    public $ccm_documents = [
        'license_validity'  => 'License Validity',
        'medical_next'      => 'Medical Next',
        'sep_to'            => 'SEP',
        'crm_to'            => 'CRM',
        'security_to'       => 'Security',
        'line_check_to'     => 'Line Check',
        'dgr_to'            => 'DGR',
        'practical_to'      => 'Practicals'
    ];

    public $fcm_documents = [
        'aruba_validation_to'   => 'Aruba Validation',
        'sep_12'                => 'SEP 12',
        'sep_36'                => 'SEP 36',
        'crm_to'                => 'CRM',
        'security'              => 'Security',
        'medical_to'            => 'Medical',
        'flying_license_to'     => 'Flying License',
        'line_check_to'         => 'Line Check',
        'license_attachment_to' => 'License Attachment',
        'dgr_to'                => 'DGR',
        'lpc_ir_to'             => 'LPC/IR',
        'opc_to'                => 'OPC',
        'tri_sfi_to'            => 'TRI/SFI'
    ];

    public function load(ICrewRosterRepository $crewRosterRepository, ICrewRosterDetailRepository $crewRosterDetailRepository){
        $roster_id = \request("roster_id");

        if ($roster_id){
            $crewRoster = $crewRosterRepository->findByAttributes(['id' => $roster_id], ['details', 'user']);
            $crewRosterDetails = $crewRosterDetailRepository->findAllByAttributes(['crew_roster_id' => $roster_id], ['flight', 'user', 'position'], [], [], ['user_id']);

            $rosterList = $this->rosterList($crewRoster->type);

            $groupedDetails = [];
            $flights = [];
            $hours = [];
            $groupedByFlights = [];

            if ($crewRoster->type == FCM_CREW){
                $properties = [
                    FLT       => null,
                    TOTAL     => 0,
                    CAPT      => [],
                    CAPT_SB   => [],
                    CAPT_DHC  => [],
                    FO        => [],
                    FO_SB     => [],
                    FO_DHC    => []
                ];
            }
            else {
                $properties = [
                    FLT       => null,
                    TOTAL     => 0,
                    CC        => [],
                    CC_SB     => [],
                    CC_DHC    => []
                ];
            }

            // Get Crew
            $crew = User::getCrew($crewRoster->type, false, false, false, []);
            // Put Id as Array Key
            $crew = $crew->keyBy('id');

            foreach ($crewRosterDetails as $each) {
                $user_id = $each->user_id;
                $flight = clone $each->flight;
                $class =  $each->is_dhc ? "flight_with_dhc" : ($each->is_standby ? "standby" : "");

                $this->makeRosterByFlights($crew, $flight, $groupedByFlights, $each, $properties);

                $flight->position_order = $each->position_order;
                $flight->is_standby = $each->is_standby;
                $flight->is_dhc = $each->is_dhc;
                $flight->class = $class;

                if (!isset($groupedDetails[$user_id])){
                    $groupedDetails[$user_id] = $each;
                    $flights[$user_id] = [];
                    $hours[$user_id] = 0;

                    // This is Irrelevant info
                    $groupedDetails[$user_id]->flight_id = null;
                    $groupedDetails[$user_id]->position_order = null;
                    $groupedDetails[$user_id]->is_standby = null;
                    $groupedDetails[$user_id]->is_dhc = null;
                }

                $flights[$user_id][] = $flight;
                if (! ($each->is_standby && $each->is_dhc)) {
                    $hours[$user_id] += Calculate_Duration($flight->std, $flight->sta);
                }

                $flight = null;
            }


            foreach ($groupedDetails as $j => $each) {
                $user_id = $each->user_id;
                $groupedDetails[$j]->flight = isset($flights[$user_id]) ? $flights[$user_id] : [];
                $groupedDetails[$j]->hours = isset($hours[$user_id]) ? $hours[$user_id] : null;
            }

            $groupedDetails = array_values($groupedDetails);
        }

        $perPage = \request("perPage") ? \request("perPage") : 10;

        $this->viewData = [
            'perPage'				=> $perPage,
            'groupedByFlights'		=> isset($groupedByFlights) ? $groupedByFlights : [],
            'currentPage'			=> \request("page") ? \request("page") : 1,
            'crewType'              => isset($crewRoster) ? $crewRoster->type : '',
            'crewRoster'            => isset($crewRoster) ? $crewRoster : [],
            'crewRosterDetails'		=> isset($groupedDetails) ? paginate($groupedDetails, (\request("page") ? \request("page") : 1), $perPage) : [],
            'rosters_list'          => isset($rosterList) ? $rosterList : [],
            'roster_id'             => $roster_id
        ];

        return view('crewroster.load', $this->viewData);
    }

    public function makeRosterByFlights($crew, $flight, &$groupedByFlights, $rosterDetail, $properties){
        if (!isset($groupedByFlights[$rosterDetail->flight_id])){
            $groupedByFlights[$rosterDetail->flight_id] = $properties;
            $groupedByFlights[$rosterDetail->flight_id][FLT] = $flight;
        }

        if (!isset($crew[$rosterDetail->user_id])){
            return;
        }

        $user = $crew[$rosterDetail->user_id];
//        $user = User::getCrew($crewType, false, $rosterDetail->user_id);

        if (in_array($rosterDetail->position_id, [
            USER_CAPTAIN_POSITION_ID,
            USER_CHIEF_PILOT_POSITION_ID,
            USER_PILOT_INSTRUCTOR_POSITION_ID,
        ])){
            if ($rosterDetail->is_standby){
                $groupedByFlights[$rosterDetail->flight_id][CAPT_SB][] = $user;
            }
            else if ($rosterDetail->is_dhc){
                $groupedByFlights[$rosterDetail->flight_id][CAPT_DHC][] = $user;
            }
            else {
                $groupedByFlights[$rosterDetail->flight_id][CAPT][$rosterDetail->position_order] = $user;
                $groupedByFlights[$rosterDetail->flight_id][TOTAL]++;
            }
        }
        else if (in_array($rosterDetail->position_id, [
            USER_FIRST_OFFICER_POSITION_ID,
            USER_COPILOT_JUNIOR_POSITION_ID,
        ])){
            if ($rosterDetail->is_standby){
                $groupedByFlights[$rosterDetail->flight_id][FO_SB][] = $user;
            }
            else if ($rosterDetail->is_dhc){
                $groupedByFlights[$rosterDetail->flight_id][FO_DHC][] = $user;
            }
            else {
                $groupedByFlights[$rosterDetail->flight_id][FO][$rosterDetail->position_order] = $user;
                $groupedByFlights[$rosterDetail->flight_id][TOTAL]++;
            }
        }
        else if (in_array($rosterDetail->position_id, [
            USER_PURSER_ID,
            USER_INSTRUCTOR_FLIGHT_ATTENDANT_ID,
            USER_SENIOR_INSTRUCTOR_FLIGHT_ATTENDANT_ID,
            USER_FLIGHT_ATTENDANT_ID,
            USER_FLIGHT_ATTENDANT_N2_ID,
            USER_FLIGHT_ATTENDANT_JUNIOR_ID,
        ])){
            if ($rosterDetail->is_standby){
                $groupedByFlights[$rosterDetail->flight_id][CC_SB][] = $user;
            }
            else if ($rosterDetail->is_dhc){
                $groupedByFlights[$rosterDetail->flight_id][CC_DHC][] = $user;
            }
            else {
                $groupedByFlights[$rosterDetail->flight_id][CC][$rosterDetail->position_order] = $user;
                $groupedByFlights[$rosterDetail->flight_id][TOTAL]++;
            }
        }
    }

    public function ccmSettings(){

        $admin_settings = CrewRosterSettings::where('type', 'ccm')->first();
        $admin_docs_list = [];
        if ($admin_settings){
            $docs = unserialize($admin_settings->documents_validity);
            if (count($docs)) {
                foreach ($docs as $license => $valid) {
                    if ($valid)
                        $admin_docs_list[] = $license;
                }
            }
        }

        $data_parameters = [
            'admin_docs_list' => $admin_docs_list,
            'documents_validity' => $this->ccm_documents,
            "flight_numbers" => ["" => "Select Flight"] + FlightNumber::getArrayFlightNumbersWithSector(['Scheduled', 'Charter']),
            'admin_settings' => $admin_settings,
            'aircraft_types' => AircraftType::get(),
            'hubs' => Location::with('city')->whereNull('deleted_at')->get(),
            'dep_arr_type'  => ["" => "Select Type", 'block_hours' => "Block Hours", "flight_hours" =>'Flight Hours', 'office_duty'=>'Office Duty'],
            'sector_allowed' => [
                ['num'=> '1', 'total_hour'=>14, 'rest_hour'=>10],
                ['num'=> '2', 'total_hour'=>12, 'rest_hour'=>12],
                ['num'=> '3', 'total_hour'=>10, 'rest_hour'=>14],
                ['num'=> '4', 'total_hour'=>8, 'rest_hour'=>16],
                ['num'=> '5', 'total_hour'=>6, 'rest_hour'=>18],
                ['num'=> '6', 'total_hour'=>4, 'rest_hour'=>20 ]
            ],
            'sector_allowed_selected' => 6,
            'regulation' => ["" => "Select Type", 'faa'=>'FAA', 'easa'=>'EASA', 'car'=>'CAR', 'custom'=>'CUSTOM']
        ];

        return view('crewroster.ccm-settings', $data_parameters);
    }

    public function saveCcmSettings(){
        $table = CrewRosterSettings::where('type', 'ccm')->first(['id']);
        $submitted_docs = \request("documents_validity") ? \request("documents_validity") : [];
        $valid_docs = [];

        foreach ($this->ccm_documents as $index => $license){
            if (in_array($index, $submitted_docs))
                $valid_docs[$index] = true;
            else
                $valid_docs[$index] = false;
        }

        $docs_serialized = serialize($valid_docs);
        CrewRosterSettings::where('id',$table->id)->update([
            'roster_type'           => \request("roster_type"),
            'ftl_7'                 => \request("ftl_7"),
            'ftl_28'                => \request("ftl_28"),
            'ftl_365'               => \request("ftl_365"),
            'policy_flight_numbers' => \request("policy_flight_numbers") ? serialize(\request("policy_flight_numbers")) : "",
            'documents_validity'    => isset($docs_serialized) ? $docs_serialized : null
        ]);

        flash()->success('Successfully Done');

        return response()->json(['success' => true]);
    }

    public function fcmSettings(){
        $admin_settings = CrewRosterSettings::where('type', 'fcm')->first();
        $admin_docs_list = [];
        if ($admin_settings){
            if ($admin_settings->documents_validity && !is_null($admin_settings->documents_validity)) {
                $docs = unserialize($admin_settings->documents_validity);
                if (count($docs)) {
                    foreach ($docs as $license => $valid) {
                        if ($valid)
                            $admin_docs_list[] = $license;
                    }
                }
            }
        }

        $data_parameters = [
            'admin_docs_list' => $admin_docs_list,
            'documents_validity' => $this->fcm_documents,
            'flight_numbers' => FlightNumber::getArrayFlightNumbersOutbounds(),
            'admin_settings' => $admin_settings,
            'aircraft_types' => AircraftType::get(),
            'hubs' => Location::with('City')->whereNull('deleted_at')->get(),
            'dep_arr_type'  => ["" => "Select Type", 'block_hours' => "Block Hours", "flight_hours" =>'Flight Hours', 'office_duty'=>'Office Duty'],
            'sector_allowed' => [['num'=> '1', 'total_hour'=>14, 'rest_hour'=>10], ['num'=> '2', 'total_hour'=>12, 'rest_hour'=>12],
                ['num'=> '3', 'total_hour'=>10, 'rest_hour'=>14], ['num'=> '4', 'total_hour'=>8, 'rest_hour'=>16],
                ['num'=> '5', 'total_hour'=>6, 'rest_hour'=>18], ['num'=> '6', 'total_hour'=>4, 'rest_hour'=>20 ]],
            'sector_allowed_selected' => 6,
            'regulation' => ["" => "Select Type", 'faa'=>'FAA', 'easa'=>'EASA', 'car'=>'CAR', 'custom'=>'CUSTOM']

        ];

        return view('crewroster.fcm-settings', $data_parameters);
    }

    public function saveFcmSettings(){
        $table = CrewRosterSettings::where('type', 'fcm')->first(['id']);
        $submitted_docs = \request("documents_validity") ? \request("documents_validity") : [];
        $valid_docs = [];
        foreach ($this->fcm_documents as $index=>$license){
            if (in_array($index, $submitted_docs))
                $valid_docs[$index] = true;
            else
                $valid_docs[$index] = false;
        }
        $docs_serialized = serialize($valid_docs);
        CrewRosterSettings::where('id',$table->id)->update([
            'roster_type'           => \request("roster_type"),
            'ftl_7'                 => \request("ftl_7"),
            'ftl_28'                => \request("ftl_28"),
            'ftl_365'               => \request("ftl_365"),
            'policy_flight_numbers' => \request("policy_flight_numbers") ? serialize(\request("policy_flight_numbers")) : "",
            'documents_validity'    => isset($docs_serialized) ? $docs_serialized : null
        ]);

        flash()->success('Successfully Done');

        return response()->json(['success' => true]);
    }

    public function expirySummary(){

        if (\request("crew_chosen")){
            $crew_chosen = \request("crew_chosen");
            switch ($crew_chosen){
                case "fcm":
                    $crewObject = User::with(['userHistory'])
                            ->join('crew__flight', 'users.id', '=', 'crew__flight.user_id')
                            ->leftJoin('users__departments', 'users.id', '=', 'users__departments.user_id')
                            ->where(function($sql){
                                $sql->whereNull("users.resigned_date")
                                    ->orWhere("users.resigned_date", EMPTY_DATE)
                                    ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                            })
                            ->orderBy('last_name')
                            ->get(['users.id as id', 'position_id', 'first_name', 'last_name','passport', 'staff_number', 'email', 'dob', 'languages', 'phone', 'doj', 'picture', 'thumb', 'location', 'aircraft_types', 'aruba_validation_to', 'cmc_from', 'cmc_to', 'sep_12', 'sep_36', 'crm_from', 'crm_to', 'security', 'medical_from', 'medical_to', 'flying_license_from', 'flying_license_to', 'line_check_from', 'line_check_to',
                                'license_attachment_from', 'license_attachment_to', 'dgr_from', 'dgr_to', 'lpc_ir_from', 'lpc_ir_to', 'opc_from', 'opc_to', 'tri_sfi_from', 'tri_sfi_to'])
                            ->toArray();
                    break;
                case "ccm":
                    $crewObject = User::with(['userHistory'])
                            ->join('crew__cabin', 'users.id', '=', 'crew__cabin.user_id')
                            ->leftJoin('users__departments', 'users.id', '=', 'users__departments.user_id')
                            ->where(function($sql){
                                $sql->whereNull("users.resigned_date")
                                    ->orWhere("users.resigned_date", EMPTY_DATE)
                                    ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                            })
                            ->orderBy('last_name')
                            ->get(['users.id as id', 'first_name', 'last_name','passport', 'staff_number', 'email', 'dob', 'languages', 'phone', 'doj', 'picture', 'thumb', 'location',  'is_contractor', 'is_purser', 'aircraft_types','license_validity', 'cmc_from', 'cmc_to', 'sep_from', 'sep_to', 'crm_from', 'crm_to', 'security_from', 'security_to',  'practical_from','practical_to','dgr_from', 'dgr_to', 'line_check_from', 'line_check_to', 'medical_next'])
                            ->toArray();
                    break;

            }

            foreach ($crewObject as $i=>$crewEach){
                if ($crewEach['passport'] != ''){
                    $crewObject[$i]['passport'] = unserialize($crewEach['passport']);
                }
            }

            $condition = Condition::get()->toArray();
            $new_conditions = [];
            foreach ($condition as $each){
                $new_conditions[$each['id']] = $each['name'];
            }
        }

        $data = [
            'crew_chosen'   => isset($crew_chosen) ? $crew_chosen : null,
            'crewObject' => isset($crewObject) ? $crewObject : null,
            'condition'  => isset($new_conditions) ? $new_conditions : null
        ];

        return view('crewroster.crew-report-expiry-tab', $data);

    }

    public function ccmRoster(){

        //--Admin Settings--------------------------
        $admin_settings = CrewRosterSettings::first();
        if ($admin_settings) {
            $this->crew_policy_flight_numbers = unserialize($admin_settings->policy_flight_numbers);
            $this->ftl_7days = $admin_settings->ftl_7;
            $this->ftl_28days = $admin_settings->ftl_28;
            $this->ftl_12months = $admin_settings->ftl_365;
            $this->roster_type = $admin_settings->roster_type;
            $this->different_hub_stay_period = 5;
            if ($admin_settings->documents_validity) {
                $docs = unserialize($admin_settings->documents_validity);
                if (count($docs)) {
                    foreach ($docs as $license => $valid) {
                        if ($valid)
                            $admin_docs_list[] = $license;
                    }
                }
            }
        }
        //-----------------------------------------------


        $data = [
            'documents_validity'=> $this->ccm_documents,
            'admin_docs_list'   => isset($admin_docs_list) ? $admin_docs_list : [],
            'records_number'    => \request("records_number") ? \request("records_number") : 20,
            'from'              => null,
            'to'                => null,
            'cabincrew_list'    => ['' => 'select'] + User::getCabinCrewList(),
            'admin_settings'    => CrewRosterSettings::first(),
            'flight_numbers'    => ["" => "Select Flight"] + FlightNumber::getArrayFlightNumbersWithSector(['Scheduled', 'Charter']),
        ];

        return view('crewroster.ccm-roster',$data);
    }

    public function fcmRoster(){

        //--Admin Settings--------------------------
        $admin_settings = CrewRosterSettings::where('type', 'fcm')->first();
        if ($admin_settings) {
            $this->crew_policy_flight_numbers = unserialize($admin_settings->policy_flight_numbers);
            $this->ftl_7days = $admin_settings->ftl_7;
            $this->ftl_28days = $admin_settings->ftl_28;
            $this->ftl_12months = $admin_settings->ftl_365;
            $this->roster_type = $admin_settings->roster_type;
            $this->different_hub_stay_period = 5;
            if ($admin_settings->documents_validity) {
                $docs = unserialize($admin_settings->documents_validity);
                if (count($docs)) {
                    foreach ($docs as $license => $valid) {
                        if ($valid)
                            $admin_docs_list[] = $license;
                    }
                }
            }
        }
        //-----------------------------------------------


        $data = [
            'documents_validity'=> $this->fcm_documents,
            'admin_docs_list'   => isset($admin_docs_list) ? $admin_docs_list : [],
            'records_number'    => \request("records_number") ? \request("records_number") : 20,
            'from'              => null,
            'to'                => null,
            'cabincrew_list'    => [''=>'select'] + User::getFlightCrewList(),
            'admin_settings'    => CrewRosterSettings::where('type', 'fcm')->first(),
            'flight_numbers'    => ["" => "Select Flight"] + FlightNumber::getArrayFlightNumbersWithSector(['Scheduled', 'Charter']),
        ];

        return view('crewroster.fcm-roster',$data);
    }

    public function ajaxGenerateCrewRoster(CrewRosterDetailRepository $crewRosterDetailRepository){
        $all_variables = request()->all();

        $crewType = $all_variables['crewType'];
        $from = $all_variables['from'] ? $all_variables['from'] : false;
        $to = $all_variables['to'] ? $all_variables['to'] : false;
        $lbd_crew_requirement = isset($all_variables['lbd_min']) ? $all_variables['lbd_min'] : 0;
        $dyu_pseudo = isset($all_variables['dyu_pseudo']) ? $all_variables['dyu_pseudo'] : false;


        $pseudo_dyu_set = ($dyu_pseudo == 'none' ? false : true);
        $lbd_dhc_pseudo = $all_variables['lbd_dhc_pseudo'] ? $all_variables['lbd_dhc_pseudo'] : false;

        $pseudo_lbd_set = false;
        $dhc_lbd = false;
        if ($lbd_dhc_pseudo){
            switch ($lbd_dhc_pseudo){
                case 'pseudo':
                    $pseudo_lbd_set = true;
                    break;
                case 'dhc':
                    $dhc_lbd = true;
                    break;
                case 'none':
                    break;
            }
        }

        $pseudo_dyu_rank1 = isset($all_variables['pseudo_dyu_rank1']) ? $all_variables['pseudo_dyu_rank1'] : 0;
        $pseudo_dyu_rank2 = isset($all_variables['pseudo_dyu_rank2']) ? $all_variables['pseudo_dyu_rank2'] : 0;
        $pseudo_lbd_rank1 = isset($all_variables['pseudo_lbd_rank1']) ? $all_variables['pseudo_lbd_rank1'] : 0;
        $pseudo_lbd_rank2 = isset($all_variables['pseudo_lbd_rank2']) ? $all_variables['pseudo_lbd_rank2'] : 0;
        $records_number   = isset($all_variables['records_number'])   ? $all_variables['records_number']   : 5;

        //--Admin Settings--------------------------
        $admin_settings = CrewRosterSettings::where('type', $crewType)->first();
        $crew_policy_flight_numbers = null;
        if ($admin_settings->policy_flight_numbers) {
            $crew_policy_flight_numbers = unserialize($admin_settings->policy_flight_numbers);
        }
        $ftl_7days = $admin_settings->ftl_7;
        $ftl_28days = $admin_settings->ftl_28;
        $ftl_12months = $admin_settings->ftl_365;
        $roster_type = $admin_settings->roster_type;

        // Different Hub stay period
        $different_hub_stay_period = 4;

        $documents_validity = is_null($admin_settings->documents_validity) ? [] : unserialize($admin_settings->documents_validity);
        //-----------------------------------------------

        // Get All Available Cabin Crew and Set Array of FCM objects
        $from=  $from ? $from : date('Y-m-d');
        $to = $to ? $to : date('Y-m-d', strtotime("$from + 1 weeks - 1 days"));

        switch ($roster_type) {
            case 'monthly':
                $from_1month = date('Y-m-d', strtotime("$from + 1 months - 1 days"));
                if (strtotime($from_1month) != strtotime($to))
                    $to = $from_1month;
                break;
            case 'fortnightly':
                $from_2week = date('Y-m-d', strtotime("$from + 2 weeks - 1 days"));
                if (strtotime($from_2week) != strtotime($to))
                    $to = $from_2week;
                break;
            case 'weekly':
                $from_1week = date('Y-m-d', strtotime("$from + 1 weeks - 1 days"));
                if (strtotime($from_1week) != strtotime($to))
                    $to = $from_1week;
                break;
            case 'custom':
                break;

        }

        // --- DESIGNATED CREW
        if (isset($all_variables['add_flight_numbers'])) {
            $add_flight_numbers = $all_variables['add_flight_numbers'];
            $add_date = $all_variables['add_date'];

            if (count($add_flight_numbers) > 0) {
                $designated = [];
                foreach ($add_flight_numbers as $i => $flight_number) {
                    $flight_date = date('Y-m-d', strtotime($add_date[$i]));
                    $crew = isset($all_variables['crew'][$i]) ? $all_variables['crew'][$i] : [];
                    if (count($crew)) {
                        $flights = Flight::Parent_Child_Flight($flight_date, $flight_number);
                        debug("Date: " . $flight_date . " | FN: " . $flight_number);
                        if (count($flights)) {
                            $designated['flights'][] = $flights;
                            $designated['flight_number'][] = $flight_number;
                            $designated['date'][] = $flight_date;
                            $designated['crew'][] = $crew;
                        }
                    }
                }
            }
        }

        $schedule_conditions = [
            'from'                      => $from,
            'to'                        => $to,
            'pseudo_lbd_set'            => $pseudo_lbd_set,
            'dhc_lbd'                   => $dhc_lbd,
            'lbd_crew_requirement'      => $lbd_crew_requirement,
            'pseudo_dyu_set'            => $pseudo_dyu_set,
            'pseudo_dyu_rank1'          => $pseudo_dyu_rank1,
            'pseudo_dyu_rank2'          => $pseudo_dyu_rank2,
            'pseudo_lbd_rank1'          => $pseudo_lbd_rank1,
            'pseudo_lbd_rank2'          => $pseudo_lbd_rank2,
            'ftl_7days'                 => $ftl_7days,
            'ftl_28days'                => $ftl_28days,
            'ftl_12months'              => $ftl_12months,
            'fdp_7days'                 => 60,
            'fdp_28days'                => 190,
            'crew_policy_flight_numbers'=> $crew_policy_flight_numbers,
            'different_hub_stay_period' => $different_hub_stay_period
        ];

        // Documents Included by Admin To Be Valid
        $crew_license_conditions = $documents_validity;

        $designated_flights = [];
        if (isset($designated) && count($designated)) {
            $designated_flights = [ 'flights' => $designated['flights'],
                'crew'   => $designated['crew'] ];
        }

        $settings = ['initial' => $schedule_conditions, 'crew_license' => $crew_license_conditions];

        // crewType => FCM or CCM
        $roster = new RosterInitialize(strtoupper($crewType), $settings);

        list($crew, $flights_scheduled, $flights_crew, $scheduled_keys, $lbd_crew_requirement, $pseudo) = $roster->generateRoster($designated_flights);

        $flights_no_crew = 0;
        $flights_crew_less_min = 0;
        $standby_shortage = 0;

        foreach ($flights_scheduled as $item) {
            if ($item->crew_planned == 0)
                $flights_no_crew++;
            else {
                if ($item->crew_planned < $item->crew_min)
                    $flights_crew_less_min++;
            }

            if (!isset($item->flights[0]['crew_standby']))
                $standby_shortage++;
        }


        $flight_crew = isset($flights_no_crew) && isset($flights_crew_less_min) ?
            ['no_crew' => $flights_no_crew, 'less_min' => $flights_crew_less_min, 'standby_shortage' => $standby_shortage]
            : ['no_crew' => 'N/A', 'less_min' => 'N/A', 'standby_shortage' => 'N/A'];

        $other_settings = new stdClass();
        $other_settings->from = $from;
        $other_settings->to = $to;
        $other_settings->published_date = date('Y-m-d H:i');
        $other_settings->user_id = $this->authObject->user()->id;
        $other_settings->pseudo = $pseudo;
        $other_settings->lbd_min = $lbd_crew_requirement;
        $other_settings->keys = $scheduled_keys;
        $other_settings->flights_crew = $flight_crew;
        $other_settings->admin_settings = $admin_settings;


        $data = [
            'crewType'          => $crewType,
            'designated'        => (isset($designated) && count($designated)) ? $designated : [],
            'cabincrew_list'    => [''=>'select'] + User::getCabinCrewList(),
            'admin_settings'    => CrewRosterSettings::where('type', 'fcm')->first(),
            'from'              => $from,
            'to'                => $to,
            'flight_numbers'    => FlightNumber::getArrayFlightNumbersOutbounds(),
            'crew'              => count($crew) ? $crew : [],
            'keys'              => $scheduled_keys,
            'flights_scheduled' => count($flights_scheduled) ? $flights_scheduled : [],
            'flights_crew'      => $flight_crew,
            'pseudo'            => $pseudo,
            'lbd_min'           => $lbd_crew_requirement,
            'pseudo_dyu_rank1'  => $pseudo['pseudo_dyu_rank1'],
            'pseudo_dyu_rank2'  => $pseudo['pseudo_dyu_rank2'],
            'pseudo_lbd_rank1'  => $pseudo['pseudo_lbd_rank1'],
            'pseudo_lbd_rank2'  => $pseudo['pseudo_lbd_rank2'],
            'records_number'    => $records_number,
            'other_settings'    => $other_settings
        ];

        // Save Roster as NOT SAVED status
        if ($crewType == CCM_CREW){
            $data['rosterId'] = $this->SaveCCMRoster($crewRosterDetailRepository, $data);
        }
        elseif ($crewType == FCM_CREW){
            $data['rosterId'] = $this->SaveFCMRoster($crewRosterDetailRepository, $data);
        }

        return response()->json([
            "message"   => "Roster Generated Successfully",
            "success"   => TRUE,
            "data"      =>  $data
        ]);
    }

    public function ajaxLoadRosterList(){
        $crew_type = \request("crew_type");

        $rosters_list = $this->rosterList($crew_type);

        return response()->json([
            "message"       => "Loaded Roster List Successfully",
            "success"       => TRUE,
            "rosters_list"  => $rosters_list
        ]);

    }

    public function rosterList($crew_type){
        return $rosters_list = CrewRoster::selectRaw('id, CONCAT(DATE_FORMAT(roster_from,"%d/%b/%y")," - ",DATE_FORMAT(roster_to,"%d/%b/%y")," | Saved: ", DATE_FORMAT(created_at,"%d/%b/%y")) as name')
                                        ->where('type', $crew_type)
                                        ->where('status', '<>', APPLICATION_STATUS_NOT_SAVED)
                                        ->pluck('name', 'id')->all();

    }


    public function ajaxSaveChanges(){
        $type = \request("type");
        $crew_id = \request("crew_id");
        $item = \request("item");
        $table = \request("table") ? \request("table") : 'none';
        $value = \request("value");
        $date_format = \request("date_format");

        $date_format = (is_null($date_format) || $date_format == "not_applicable") ? 'dMy' : $date_format;

        if (!is_array($value) && dateValidation($value, $date_format)){
            $value =  date('Y-m-d', strtotime($value));
        }

        if ($table == 'users' || in_array($item, ['passport_doi', 'passport_dex'])){
            $user = User::find($crew_id);
            if ($user){
                $value = is_null($value) ? "" : $value;
                if (in_array($item, ['passport_doi', 'passport_dex'])) {
                    $passport =  !is_null($user->passport) ? unserialize($user->passport) : ['p_exp_date' => '', 'p_issued_date' => ''];

                    switch ($item) {
                        case 'passport_dex':
                            $passport['p_exp_date'] = $value;
                            break;
                        case "passport_doi":
                            $passport['p_issued_date'] = $value;
                            break;
                    }
                    $user->passport = serialize($passport);
                }
                else if (in_array($item, ['phone_home', 'phone_mobile1'])) {
                    $phone = !is_null($user->phone) ? unserialize($user->phone) : ['home' => '', 'mobile1' => ''];

                    switch ($item) {
                        case 'phone_home':
                            $phone['home'] = $value;
                            break;
                        case "phone_mobile1":
                            $phone['mobile1'] = $value;
                            break;
                    }
                    $user->phone = serialize($phone);
                }
                elseif ($item == 'languages'){
                    /*$languages = $user->languages;
                    $value_before = !is_null($languages)  ? unserialize($languages) : [];
                    $user->languages = serialize($value);*/
                }
                else {
                    $user->$item = ($value == '' ? null : $value);
                }
                $user->save();
            }
        }
        else {
            switch ($type){
                case "fcm":
                    $crew = CaptainFirstOfficer::where('user_id', $crew_id)->first();
                    break;
                case "ccm":
                    $crew = CabinCrew::where('user_id', $crew_id)->first();
                    break;
                default:
                    $crew = false;
                    break;
            }

            if ($crew){
                if ($item == 'aircraft_types'){
                    $crew->$item = serialize($value);
                }
                elseif (in_array($item, ['is_captain', 'is_purser'])){
                    if ($value == 0) {
                        if ($type == 'fcm')
                            UserDepartment::where('user_id', $crew_id)->update(['position_id' => Position::where('title', 'First Officer')->first(['id'])->id]);
                        elseif($type == 'ccm')
                            UserDepartment::where('user_id', $crew_id)->update(['position_id' => Position::where('title', 'Flight Attendant')->first(['id'])->id]);

                        $crew->$item = null;
                    }
                    elseif ($value == 1) {
                        if ($type == 'fcm')
                            UserDepartment::where('user_id', $crew_id)->update(['position_id' => Position::where('title', 'Captain')->first(['id'])->id]);
                        elseif($type == 'ccm')
                            UserDepartment::where('user_id', $crew_id)->update(['position_id' => Position::where('title', 'Purser')->first(['id'])->id]);

                        $crew->$item = 1;
                    }

                }
                else {
                    $crew->$item = ($value == '' ? null : $value);
                }
                $crew->save();
            }
        }
        return response()->json([
            "message" => "Record Was Successfully Updated",
            "success" => TRUE,
        ]);
    }

    public function ajaxSaveRoster(){
        $rosterId = \request("rosterId");

        $message = "Roster Not Saved";
        $success = false;

        $findRoster = CrewRoster::find($rosterId);
        if ($findRoster){
            $findRoster->status = APPLICATION_STATUS_NOT_PUBLISHED;
            $findRoster->save();
            $message = "Roster Successfully Saved";
            $success = true;
        }

        return response()->json([
            'success'   => $success,
            'message'   => $message
        ]);
    }

    public function SaveCCMRoster(ICrewRosterDetailRepository $crewRosterDetailRepository, $data, $status = null){
        $flights_scheduled    = $data['flights_scheduled'];

        $insert_array = [];
        $crew = User::CabinCrewWithPosition();

        // Create
        $roster = $this->CreateNewRoster($data, $status);

        foreach ($flights_scheduled as $i => $fdp){
            if (isset($fdp->flights) && count($fdp->flights)) {
                foreach ($fdp->flights as $flight) {
                    if (isset($fdp->crew) && count($fdp->crew)) {
                        foreach ($fdp->crew as $pos => $id) {
                            $positionId = $crew[$id]['position_id'];
                            $insert_array[] = $this->GetInsertArray($roster->id, $flight['id'], $id, $positionId, $pos);
                        }
                    }
                    if (isset($flight['crew_standby'])&& count($flight['crew_standby'])) {
                        $pos = 0;
                        foreach ($flight['crew_standby'] as $j => $id) {
                            $positionId = $crew[$id]['position_id'];
                            $insert_array[] = $this->GetInsertArray($roster->id, $flight['id'], $id, $positionId, $pos, true);
                            $pos++;
                        }
                    }
                    if (isset($flight['crew_dhc']) && count($flight['crew_dhc'])) {
                        $pos = 0;
                        foreach ($flight['crew_dhc'] as $j => $id) {
                            $positionId = $crew[$id]['position_id'];
                            $insert_array[] = $this->GetInsertArray($roster->id, $flight['id'], $id, $positionId, $pos, false, true);
                            $pos++;
                        }
                    }
                }
            }
        }

        $crewRosterDetailRepository->insert($insert_array);

        return $roster->id;
    }

    public function SaveFCMRoster(ICrewRosterDetailRepository $crewRosterDetailRepository, $data, $status = null){
        $flights_scheduled    = $data['flights_scheduled'];

        $insert_array = [];
        $crew = User::FlightCrewWithPosition();

        // Create
        $roster = $this->CreateNewRoster($data, $status);

        foreach ($flights_scheduled as $i => $fdp){
            if (isset($fdp->flights) && count($fdp->flights)) {
                foreach ($fdp->flights as $flight) {
                    if (isset($fdp->crew) && count($fdp->crew)) {
                        $cpPosition = $foPosition = 0;
                        foreach ($fdp->crew as $j => $id) {
                            $positionId = $crew[$id]['position_id'];
                            if (in_array($positionId, [
                                USER_CAPTAIN_POSITION_ID,
                                USER_CHIEF_PILOT_POSITION_ID,
                                USER_PILOT_INSTRUCTOR_POSITION_ID,
                            ])){
                                $insert_array[] = $this->GetInsertArray($roster->id, $flight['id'], $id, $positionId, $cpPosition);
                                $cpPosition++;
                            }
                            else {
                                $insert_array[] = $this->GetInsertArray($roster->id, $flight['id'], $id, $positionId, $foPosition);
                                $foPosition++;
                            }
                        }
                    }

                    if (isset($flight['crew_standby'])&& count($flight['crew_standby'])) {
                        $cpPosition = $foPosition = 0;
                        foreach ($flight['crew_standby'] as $j => $id) {
                            $positionId = $crew[$id]['position_id'];
                            if (in_array($positionId, [
                                USER_CAPTAIN_POSITION_ID,
                                USER_CHIEF_PILOT_POSITION_ID,
                                USER_PILOT_INSTRUCTOR_POSITION_ID,
                            ])){
                                $insert_array[] = $this->GetInsertArray($roster->id, $flight['id'], $id, $positionId, $cpPosition, true);
                                $cpPosition++;
                            }
                            else {
                                $insert_array[] = $this->GetInsertArray($roster->id, $flight['id'], $id, $positionId, $foPosition, true);
                                $foPosition++;
                            }
                        }
                    }
                    if (isset($flight['crew_dhc']) && count($flight['crew_dhc'])) {
                        $cpPosition = $foPosition = 0;
                        foreach ($flight['crew_dhc'] as $j => $id) {
                            $positionId = $crew[$id]['position_id'];
                            if (in_array($positionId, [
                                USER_CAPTAIN_POSITION_ID,
                                USER_CHIEF_PILOT_POSITION_ID,
                                USER_PILOT_INSTRUCTOR_POSITION_ID,
                            ])){
                                $insert_array[] = $this->GetInsertArray($roster->id, $flight['id'], $id, $positionId, $cpPosition, false, true);
                                $cpPosition++;
                            }
                            else {
                                $insert_array[] = $this->GetInsertArray($roster->id, $flight['id'], $id, $positionId, $foPosition, false, true);
                                $foPosition++;
                            }
                        }
                    }
                }
            }
        }

        $crewRosterDetailRepository->insert($insert_array);

        return $roster->id;
    }

    public function GetInsertArray($rosterId, $flightId, $userId, $positionId, $positionOrder, $isStandby = false, $isDhc = false ){
        return [
            'crew_roster_id'    => $rosterId,
            'flight_id'         => $flightId,
            'user_id'           => $userId,
            'position_id'       => $positionId,
            'position_order'    => $positionOrder,
            'is_standby'        => $isStandby,
            'is_dhc'            => $isDhc,
            'created_at'        => date('Y-m-d H:i:s'),
            'created_by'        => Auth::user()->id,
            'updated_at'        => date('Y-m-d H:i:s'),
            'updated_by'        => Auth::user()->id];
    }


    /**
     * Publish Roster
     * @param IFlightCrewRepository $flightCrewRepository
     * @param ICrewRosterRepository $crewRosterRepository
     * @param ICrewRosterDetailRepository $crewRosterDetailRepository
     * @return mixed
     */
    public function ajaxPublishRoster(IFlightCrewRepository $flightCrewRepository,
                                          ICrewRosterRepository $crewRosterRepository,
                                          ICrewRosterDetailRepository $crewRosterDetailRepository){

        $roster_id = \request("roster_id");
        $force_publish = \request("force_publish");

        if ($roster_id){

            $currentRoster = $crewRosterRepository->find($roster_id);

            if (!$currentRoster){
                return response()->json([
                    'success'   => false,
                    'message'   => "Roster Not Found."
                ]);
            }

            if ($currentRoster->status == APPLICATION_STATUS_PUBLISHED) {
                return response()->json([
                    'success'   => false,
                    'message'   => "Error! Current Roster is Already Published. Can Not Publish Twice."
                ]);
            }

            $from = $currentRoster->roster_from;
            $to = $currentRoster->roster_to;

            // Get Crew From Roster Details
            $crew = $this->GetCrewFromCrewRosterDetails($crewRosterDetailRepository, $roster_id);

            // Check if Roster From not Lesser Than Today
            if ($from < date('Y-m-d')) {
                return response()->json([
                    'success'   => false,
                    'message'   => "Selected Roster's Date From is Earlier Than Today. Roster Not Published."
                ]);
            }

            // If Any Previously Published Roster Conflicts With Current Rosters Period
            if ($crewRosterRepository->isRosterConflicts($currentRoster->type, $from, $to)) {
                if ($force_publish){
                    // Delete All Crew On The Roster Period
                    $flightCrewRepository->deleteCrewOnPeriod($currentRoster->type, $from, $to);

                    // Publish Roster
                    $this->publishRoster($crewRosterRepository, $flightCrewRepository, $crew, $currentRoster);

                    return response()->json([
                        'success'   => true,
                        'message'   => "Roster Published(Forced). Note: Current Roster Conflicts with Previously Published Rosters."
                    ]);
                }

                return response()->json([
                    'success'   => false,
                    'message'   =>"Error! Current Roster Conflicts with Previously Published Rosters. Please Check Force Roster Publish."
                ]);

            }

            // Check If Any Crew Were Previously Scheduling in The Current Roster Period
            if ($flightCrewRepository->isCrewOnPeriod($currentRoster->type, $from, $to)){
                if ($force_publish){
                    // Delete All Crew On The Roster Period
                    $flightCrewRepository->deleteCrewOnPeriod($currentRoster->type, $from, $to);

                    // Publish Roster
                    $this->publishRoster($crewRosterRepository, $flightCrewRepository, $crew, $currentRoster);

                    return response()->json([
                        'success'   => true,
                        'message'   => "Roster Published(Forced). Note: All Crew That Were Earlier Scheduled on Flights in The Selected Period Were Deleted."
                    ]);
                 }

                return response()->json([
                    'success'   => false,
                    'message'   =>"Error! There are Crew Found Scheduled on the Selected Roster Period."
                ]);
            }

            // Publish Roster
            $this->publishRoster($crewRosterRepository, $flightCrewRepository, $crew, $currentRoster);

            return response()->json([
                'success'   => true,
                'message'   => "Roster Successfully Published."
            ]);

        }
    }


    public function publishRoster($crewRosterRepository, $flightCrewRepository, $crew, $currentRoster){

        array_walk($crew, function(&$each){
            $each["created_by"] = Auth::user()->id;
            $each["created_at"] = date("Y-m-d H:i:s");
        });

        // Add Crew To Flight Crew
        $flightCrewRepository->insert($crew);

        // Update Current Roster And Set it To Published
        $crewRosterRepository->update($currentRoster, [
            'status'            => APPLICATION_STATUS_PUBLISHED,
            'published_date'    => date("Y-m-d H:i:s")
        ]);
    }


    public function GetCrewFromCrewRosterDetails(ICrewRosterDetailRepository $crewRosterDetailRepository, $roster_id){
        return $crewRosterDetailRepository->findAllByAttributes([
            'crew_roster_id'        => $roster_id
        ], [], [
            'flight_id',
            'user_id',
            'position_id',
            'crew_roster_id',
            'position_order',
            'is_standby',
            'is_dhc'
        ])->toArray();
    }


    public function addCrewToFleetwatch($array, $roster_id){
        foreach ($array as $i=>$item){
            $array[$i]['roster_id'] = $roster_id;
        }
        $check = FlightCrew::insert($array);

        return ($check ? true : false);
    }

    /**
     * By Default Status is NOT SAVED
     * @param $data
     * @param null $status
     * @return CrewRoster
     */
    public function CreateNewRoster($data, $status = null){
        $crewType             = $data['crewType'];
        $flights_scheduled    = $data['flights_scheduled'];
        $other_settings       = $data['other_settings'];

        $roster = new CrewRoster;

        switch ($crewType){
            case "fcm":
                $roster->type = 'fcm';
                break;
            case "ccm":
                $roster->type = 'ccm';
                break;
        }
        //$roster->published_date = $other_settings['published_date'];
        $roster->user_id = $other_settings->user_id;
        $roster->roster_from = $other_settings->from;
        $roster->roster_to = $other_settings->to;
        $roster->status = $status ? $status : APPLICATION_STATUS_NOT_SAVED;
        $roster->total_crew = count($other_settings->keys);
        $roster->total_flights = count($flights_scheduled);
        $roster->flt_no_crew = $other_settings->flights_crew['no_crew'];
        $roster->flt_no_min_req = $other_settings->flights_crew['less_min'];
        $roster->flt_no_standby = $other_settings->flights_crew['standby_shortage'];

        if ($other_settings->pseudo['total'] > 0) {
            $roster->is_pseudo = true;
        }

        $roster->admin_settings = json_encode($other_settings->admin_settings);

        $other_settings_important = $other_settings->pseudo;

        $other_settings_important['lbd_min'] = $other_settings->lbd_min;
        $roster->other_settings = json_encode($other_settings_important);
        $roster->save();

        return $roster;
    }

    public function SavePublishedRosterJson($from, $to, $fcm, $flights_scheduled, $other_settings, $crew_type){
        //$folder = date('M.d.Y', strtotime($from))."_".date('M.d.Y', strtotime($to))."_PD_".date('M.d.Y');

        $root_folder = storage_path()."/json/crewroster";

        if (!isset($message)) {

            $roster = $this->CreateNewRoster($crew_type, $other_settings, []);

            $message = "Roster Published Successfully";
            $path = $root_folder."/".$roster->id;

            File::makeDirectory("$path", $mode = 0755, true, true);
            $check['fcm'] = file_put_contents("$path/fcm.json", json_encode($fcm));
            $check['flights_scheduled'] = file_put_contents("$path/flights.json", json_encode($flights_scheduled));
            $check['other_settings'] = file_put_contents("$path/settings.json", json_encode($other_settings));

            foreach ($check as $index=>$each){
                if ($each == false){
                    $message = 'Error! Files Were Not Saved';
                    return ['message'=>$message, 'success'=>false, 'roster_id' => null];
                }
            }
            return ['message'=>$message, 'success'=>true, 'roster_id' => $roster->id];
        }

        return ['message'=>'Error', 'success'=>false, 'roster_id' => null];
    }

    public function ajaxLoadRoster(ICrewRosterDetailRepository $crewRosterDetailRepository){

        $rosterId = \request("roster_id");

/*
        $parent = storage_path()."/json/crewroster";
        $files = File::files("$parent/$folder");
        foreach ($files as $file) {
            $json_files[] = json_decode(file_get_contents($file, true));
        }

        $published_agent = User::find($roster_details['user_id'],['last_name', 'first_name']);
        $roster_details['user_id'] = !is_null($roster_details['user_id']) ? strtoupper($published_agent->last_name." ".$published_agent->first_name) : '-';
        $roster_details['updated_by'] = !is_null($roster_details['updated_by']) ? strtoupper($published_agent->last_name." ".$published_agent->first_name) : '-';
*/



        $roster_details = $crewRosterDetailRepository->findAllByAttributes(['crew_roster_id' => $rosterId],[],[],[],['user_id'])->toArray();

        $roster = CrewRoster::find($rosterId);
        $admin_settings = $roster->admin_settings;
        $other_settings = $roster->other_settings;

        $crew_ids = $crewRosterDetailRepository->findAndListModelVariable(['crew_roster_id' => $rosterId], ['user_id']);

        debug($roster_details);


        return response()->json([
            "message"           => "Roster Successfully Loaded",
            "success"           => TRUE,
            "from"              => $roster->roster_from,
            "to"                => $roster->roster_to,
            'flt_no_crew'       => $roster->flt_no_crew,
            'flt_no_min_req'    => $roster->flt_no_min_req,
            'flt_no_standby'    => $roster->flt_no_standby,
            'roster_details'    => $roster_details,
            'crew_ids'          => $crew_ids,
            'admin_settings'    => $admin_settings,
            'other_settings'    => $other_settings,
        ]);
    }

    public function ajaxCancelRoster(){
        $roster_id = \request("roster_id");

        $fCrew = FlightCrew::where("crew_roster_id", $roster_id)
                            ->get();
        // Audit Deleted
        auditDeleted($fCrew);

        $roster = CrewRoster::find($roster_id);
        $roster->status = 'Cancelled';
        $roster->updated_by = Auth::user()->id;
        $roster->save();

        return response()->json([
            "message" => "Roster was successfully cancelled",
            "success" => TRUE,
        ]);
    }


}
