<?php namespace App\Http\Controllers;
/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 19.08.14

 */

use app\Classes\ProjectConstant;
use App\Models\CrewRestrictionAircraftType;
use App\Models\CrewRestrictionPairing;
use App\Models\CrewRestrictionRoute;
use App\Models\FlightRoute;
use App\Models\FlightSector;
use App\Models\License;
use App\Models\LogFlightCrew;
use App\Models\StaffLicense;
use App\Models\UserPageView;
use App\Repositories\Interfaces\ICrewRestrictionRouteRepository;
use App\Repositories\Interfaces\IFlightRouteRepository;
use App\Repositories\Interfaces\ILeaveRepository;
use App\Repositories\Interfaces\ILocationRepository;
use App\Repositories\Interfaces\IUserLanguageRepository;
use App\Repositories\Interfaces\IUserLeaveRequestRepository;
use Doctrine\DBAL\Exception\NonUniqueFieldNameException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

use App\Models\Flight;
use App\Models\User;
use App\Models\FlightCrew;
use App\Models\FlightNumber;
use App\Models\Condition;

class CrewController extends Controller
{
   const FLIGHT_WITH_REST = 'FLIGHT_WITH_REST',
        FLIGHT_NO_REST = 'FLIGHT_NO_REST',
        ONE_FLIGHT = 'ONE_FLIGHT',
        LATE_NO_SHOW_REPORT = 'Late_No_Show',
        HOURS_REPORT = 'Hours',
        DAILY_REPORT = 'Daily',
        MONTHLY_HOURS_REPORT = 'Monthly_Hours',
        STANDBY_REPORT = 'Standby',
        DHC_REPORT = 'DHC',
        ROUTES_REPORT = 'Route',
        EXPIRY_SUMMARY_REPORT = 'Expiry Summary',
        FDP_REPORT = 'FDP',
        FATIGUE_REPORT = 'Fatigue',
        LATE = 'Late',
        NO_SHOW = 'No Show',
        SICK = 'Sick',
        HEAD_ORDER = 'head_order',
        ROSTER_NOT_RECEIVED = 'roster_not_received';

    // 30 Mins after the flight Departs
    const STANDBY_AFTER_DEPARTURE_OFFSET = 1800;

    const CREW_REPORT_TIME_MIN = 90; // 1:30 in Min Before Flight's Departure
    const CREW_RELEASE_TIME_MIN = 0; //FDP + 0 min After Flight's Arrival


    /**
     * @param ILocationRepository $locationRepository
     * @param IUserLanguageRepository $userLanguage
     * @param IUserLeaveRequestRepository $userLeaveRequest
     * @param ILeaveRepository $leaveRepository
     * @param IFlightRouteRepository $flightRouteRepository
     * @param ICrewRestrictionRouteRepository $crewRestrictionRouteRep
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function profile(Request $request, ILocationRepository $locationRepository, IUserLanguageRepository $userLanguage,
                            IUserLeaveRequestRepository $userLeaveRequest, ILeaveRepository $leaveRepository,
                            IFlightRouteRepository $flightRouteRepository, ICrewRestrictionRouteRepository $crewRestrictionRouteRep)
    {
        $userId = $request->get("id");
        $period = $request->get("period");
        $date = date("Y-m-d");
        $sort = 'ASC';

        // Default Page Settings ( Default OR Roster + 28 Days Select Option)
        $rosterAndLast28Days = (!$period || $period == 'default')
        && (!$request->get("dateFrom") || !$request->get("dateTo")) ? true : false;
        if ($rosterAndLast28Days){
            $period = 'last28days';
            $periodPlanned = 'planned';
        }

        $user = User::with(['CabinCrew', 'FlightCrew'])->find($userId);

        // Get Crew Type
        $crewType = User::getCrewType($user);

        // Redirect If Not Found
        if (!$crewType || $crewType == '') {
            return redirect()->to("staff/{$user->id}");
        }

        // Get User
        $user = User::getCrew($crewType, false, $userId);


        // Redirect If Not Found
        if (!$user || $user == '') {
            return redirect()->to('error/page-not-found');
        }

        // Redirect if Have no Rights
        $accessGranted = isUserAuthorizedToView($user);

        // Redirect if Have no Rights
        if (!$accessGranted){
            return redirect()->to('error/page-not-found');
        }

        // Profile Photo
        if ($request->has('imageSave') || $request->has('imageDelete')) {
            $message = profilePhoto($user);
        }

        debug($period);
        // From And To is Selected
        if (!$period || $period == ''){
            $period = [
                'from' => $request->get('dateFrom'),
                'to'   => $request->get('dateTo')
            ];
            $label = 'PERIOD | '. date('d M Y', strtotime($period['from'])).' - '. date('d M Y', strtotime($period['to']));
        }
        else {
            // Get Label
            $label = getLabelOutOfCounterType($period);

            // Get Counter Type
            $period = getCrewHoursCounterTypesArray($date, $period);

            // Get Period From And To
            $period = getFromToOutOfCounterTypeArray($period);
        }
        // ----------------- Crew Roster --------------------------
        if (in_array($crewType, [CCM_CREW, FCM_CREW])) {


            FlightCrew::removeAllCrewDuplicatesByUser($userId);

            // Get Crew Roster Details
            $userFlights = Flight::getUserRoster($userId, $period, $sort);

            // Get Crew Roster Total Hours
            $userFlightsTotal = self::getCrewRosterTotalHours($userFlights);

            // Convert Hours Total To Time Format
            foreach ($userFlightsTotal as $i => $total)
                $userFlightsTotal[$i] = $total != 0 ? hoursToTime($total / 60, 'ready') : [];

            // If Planned Or Default Current Roster + 28 Days IS SET
            if (isset($periodPlanned)){

                // Get Planned Counter Type
                $periodPlanned = getCrewHoursCounterTypesArray($date, $periodPlanned);

                // Get Planned Period From And To
                $periodPlanned = getFromToOutOfCounterTypeArray($periodPlanned);

                // Get Crew Planned Roster Details
                $userFlightsPlanned = Flight::getUserRoster($userId, $periodPlanned);

                // Get Crew Planned Roster Total Hours
                $userFlightsPlannedTotal = self::getCrewRosterTotalHours($userFlightsPlanned);

                // Set Report Times
                $fdpPlannedFlights = setFdpAndReportTimes($userFlightsPlanned, $crewType, $locationRepository);

                $fdpPlannedFlightsTotal = calculateFdpTotalTimes($fdpPlannedFlights);

                foreach ($userFlightsPlannedTotal as $i => $total)
                    $userFlightsPlannedTotal[$i] = $total != 0 ? hoursToTime($total / 60, 'ready') : [];
            }

            // Counter Types for Info Page
            $counterTypes = ['yearToDate', 'last365days', 'last28days', 'last7days', 'planned'];

            // Get Crew Hours & DHC - Standby Counters
            $crewHours = self::getCrewHoursFlightHistoryDHCStandby($user, $date, $counterTypes, 'yearToDate', 'yearToDate', $userId);

            // Hours Tab Report
            $hoursPeriod = $request->get('hoursPeriod');
            $hoursFrom = $request->get('hoursFrom');
            $hoursTo = $request->get('hoursTo');

            if (is_null($hoursPeriod) || is_null($hoursFrom) && is_null($hoursTo))
                $hoursPeriod = 'last28days';

            // Hours History 3 Columns
            $hoursByDateTypes = ['last7days', 'last28days', 'yearToDate'];

            // Get Crew Hours History By Date
            $crewHours = self::getCrewHoursHistoricByDaily($crewHours, $userId, $hoursByDateTypes, $hoursPeriod, $hoursFrom, $hoursTo);


            // Get Crew Docs
            $crewDocuments = self::getCrewDocsByCrewType($crewType);

            // Check if Any of The Docs is Expired/Valid Temp
            $tabClass = self::checkDocumentValidityStatus($user, $crewDocuments);

            // Set Report Times
            $fdpFlights = array_reverse(setFdpAndReportTimes($userFlights, $crewType, $locationRepository));

            $fdpFlightsTotal = calculateFdpTotalTimes($fdpFlights);

            // Get DHC Flights
            $dhcFlights = Flight::getUserRoster($userId, $period, $sort, CREW_DHC);

            // Get Standby Flights
            $standbyFlights = Flight::getUserRoster($userId, $period, $sort, CREW_STANDBY);
        }

        // Get All Scheduled Sectors By Hubs
        $sectors = FlightSector::listSectors('Scheduled', 'DYU');
//        $lbdSectors = FlightSector::listSectors('Scheduled', 'LBD');

        $fdpFlights = isset($fdpFlights) ? $fdpFlights : [];
        $fdpPlannedFlights = isset($fdpPlannedFlights) ? $fdpPlannedFlights : [];

        $staffLicenses = StaffLicense::with(['license'])
            ->where('user_id', $user->id)
            ->get();

        $this->viewData = [
            'restrictionRoutes'         => CrewRestrictionRoute::listRoutes($userId),
            'restrictionAircraftTypes'  => CrewRestrictionAircraftType::listAircraftTypes($userId),
            'restrictionPairs'          => CrewRestrictionPairing::listPairs($userId),
            'crewAircraftTypes'         => User::getCrewAircraftTypes($user),
            'crewRestrictionRoutes'     => $user->crewRestrictionRoutes->count() > 0 ? $crewRestrictionRouteRep->listRouteRestrictions($userId) : [],
            'otherLeaves'               => ["" => "Select"] + $leaveRepository->findAndListModelVariable(['is_main' => NULL], ['name', 'id']),
            'leaves'                    => ["" => "Select"] + $leaveRepository->findAndListModelVariable(['is_main' => true], ['name', 'id']),
            'leaveRequests'             => $userLeaveRequest->findAllByAttributes(['user_id' => $userId], ['leaveStatus']),
            'hasEmployees'              => $this->user->hasUserReportees($userId, $this->organization),
            'languages'                 => $user->languages->count() > 0 ? $userLanguage->listLanguages($userId) : [],
            'picture'                   => $user->picture ? asset("storage/users/" . $user->picture) : asset("/assets/img/profile_picture.png"),
            'fdpPlannedFlightsTotal'    => isset($fdpPlannedFlightsTotal) ? $fdpPlannedFlightsTotal : null,
            'fdpPlannedFlights'         => $fdpPlannedFlights,
            'fdpFlightsTotal'           => isset($fdpFlightsTotal) ? $fdpFlightsTotal : null,
            'fdpFlights'                => $fdpFlights,

            // For Calendar JS
            'fdpAllFlightsJs'           => $this->prepareFlightsForJsCalendar($fdpPlannedFlights, $fdpFlights),
            // End
            'staffLicenses'             => $staffLicenses,

            'currentDate'               => date('Y-m-d'),
            'message'                   => isset($message) ? $message : null,
            'rosterAndLast28Days'       => $rosterAndLast28Days,
            'maritalStatus'             => ['' => 'Select', 'Single' => 'Single', 'Married' => 'Married'],
            'conditions'                => ['' => 'Select'] + Condition::get()->pluck('name', 'id')->all(),
            "label"                     => $label,
            "sectors"                   => $sectors,
//            "sectors"                   => ['dyu' => $dyuSectors, 'lbd' => $lbdSectors],
            "crewHours"                 => isset($crewHours) ? $crewHours : [],
            "user"                      => isset($user) ? $user : null,
            "userId"                    => $userId,
            "crewType"                  => $crewType,
            "aircraft"                  => isset($user) ? unserialize($user->aircraft_types) : null,
            "crewDocuments"             => isset($crewDocuments) ? $crewDocuments : [],
            "tabClass"                  => isset($tabClass) ? $tabClass : '',
            "dhcFlights"                => isset($dhcFlights) ? $dhcFlights : [],
            "standbyFlights"            => isset($standbyFlights) ? $standbyFlights : [],
            "userFlights"               => isset($userFlights) ? $userFlights : [],
            "userFlightsPlanned"        => isset($userFlightsPlanned) ? $userFlightsPlanned : [],
            "userFlightsTotal"          => isset($userFlightsTotal) ? $userFlightsTotal : [],
            "userFlightsPlannedTotal"   => isset($userFlightsPlannedTotal) ? $userFlightsPlannedTotal : [],
        ];

        return view("crew.profile", $this->viewData);
    }


    /**
     * @return \Illuminate\View\View
     */
    public function report(Request $request){
        $reportType = $request->get('reportType');

        if ($request->all()) {
            // CCM or FCM
            $crewType = $request->has('crew') ? $request->get('crew') : 'fcm';

            $designation = $request->get('designation');
            // Get Crew
            $crew = User::getCrew($crewType, $designation);

            // Redirect If Not Found
//            if (!count($crew)) {  return redirect()->to('error/page-not-found');  }

            $crewIds = null;

            // Put Id as Array Key
            if (count($crew)) {
                $crew = $crew->keyBy('id');

                // Get All Ids
                $crewIds = getIdsFromObjectOrArray($crew);
            }

            // Designation List For Select Boxes
            $designationList = User::getCrewDesignationArray($crewType);

            // Get Conditions Listed
            $conditions = Condition::pluck('name', 'id')->all();

            $counterTypes = ['yearToDate', 'last90days', 'last28days', 'last7days', 'planned'];

            FlightCrew::removeAllCrewDuplicates();

            debug($request->all());

            switch($reportType){

                case self::HOURS_REPORT:
                    $dateFrom = $request->get('hoursReportDate');
                    $crewHours = self::getCrewHoursFlightHistoryDHCStandby($crew, $dateFrom, $counterTypes, 'yearToDate', 'yearToDate', $crewIds);
                    break;

                case self::ROUTES_REPORT:
                    $dateFrom = $request->get('routesReportDate');
                    $crewHours = self::getCrewHoursFlightHistoryDHCStandby($crew, $dateFrom, $counterTypes, 'yearToDate', 'yearToDate', $crewIds);
                    break;

                case self::EXPIRY_SUMMARY_REPORT:
                    $dateFrom = date('Y-m-d');
                    //$crewHours = self::getCrewHoursFlightHistoryDHCStandby($crew, $dateFrom, $counterTypes, 'yearToDate', 'yearToDate', $crewIds);
                    $staffData = $this->getExpiryArrayByDepartment($crewIds);
                    debug($staffData);
                    break;

                case self::DAILY_REPORT:
                    $dateFrom = $request->has('dateFrom') ? $request->get('dateFrom') : date('Y-m-d');
                    $dateTo   = $request->has('dateTo') ? $request->get('dateTo') : date('Y-m-d');

                    $dateTo = strtotime($dateTo) < strtotime($dateFrom) ? $dateFrom : $dateTo;
                    // Get Flights With Flight Crew
                    $flights = Flight::flightsRange($dateFrom.' 00:00:00', $dateTo.' 23:59:59', ['flightNumber', 'Aircraft', 'flightCrew'], false, 'flights.aircraft_id');

                    // Data Array
                    $dailyReport = ExportController::getCrewReportDataArray($flights, $request->get("one_page"));
                    break;

                case self::MONTHLY_HOURS_REPORT:
                    $periodDateFrom = $request->get('monthlyHoursPeriodFrom') ? $request->get('monthlyHoursPeriodFrom') : date('Y-m-d', strtotime('first day of this month'));
                    $periodDateTo   = $request->get('monthlyHoursPeriodTo') ? $request->get('monthlyHoursPeriodTo') : date('Y-m-d');
                    $monthArray = getMonthArrayFromDates($periodDateFrom, $periodDateTo, $request->get('month'));
                    list($crewPeriodHours, $crewPeriodMonthlyHours) = self::getCrewHours($crew, $periodDateFrom, $periodDateTo, false, $request->get('dateSearch'), $request->get('month'), $request->get('year'));
                    break;

                case self::STANDBY_REPORT:
                    if ($request->has('dateSearch') && $request->get('dateSearch') == 'monthYear'){
                        $dateSearch['month'] = $request->get('month');
                        $dateSearch['year'] = $request->get('year');
                        $monthArray = getMonthArrayFromDates(null, null, $request->get('month'));
                    }
                    else {
                        $dateSearch['dateFrom'] = $request->get('dateFrom');
                        $dateSearch['dateTo'] = $request->get('dateTo');
                        $monthArray = getMonthArrayFromDates($dateSearch['dateFrom'], $dateSearch['dateTo']);
                    }
                    list($crewStandbyHours, $crewStandbyMonthlyHours, $standbyCount) = $this->getStandbyQtyAndHours($dateSearch, $crewIds);
                    break;

                case self::DHC_REPORT:
                    if ($request->has('dateSearch') && $request->get('dateSearch') == 'monthYear'){
                        $dateSearch['month'] = $request->get('month');
                        $dateSearch['year'] = $request->get('year');
                        $monthArray = getMonthArrayFromDates(null, null, $request->get('month'));
                    }
                    else {
                        $dateSearch['dateFrom'] = $request->get('dateFrom');
                        $dateSearch['dateTo'] = $request->get('dateTo');
                        $monthArray = getMonthArrayFromDates($dateSearch['dateFrom'], $dateSearch['dateTo']);
                    }

                    list($crewDHCHours, $crewDHCMonthlyHours, $dhcCount) = $this->getDHCQtyAndHours($dateSearch, $crewIds);
                    break;


                case self::FDP_REPORT:
                    $dateFrom = $request->get('dateFrom') ? $request->get('dateFrom') : date('Y-m-d');
                    $dateTo = $request->get('dateTo') ? $request->get('dateTo') : $dateFrom;
                    $flights = self::fdpCalculator($dateFrom, $dateTo);
                    // Pagination (Used For Pagination Links Only)
                    if (count($flights)) {
                        $pagination = paginate($flights, ($request->get('page') ? $request->get('page') : 1), $request->get('perPage') ? $request->get('perPage') : count($flights));
                    }
                    break;

                case self::LATE_NO_SHOW_REPORT:
                    $dateFrom = $request->get('dateFrom') ? $request->get('dateFrom') : date('Y-m-d');
                    $dateTo = $request->get('dateTo') ? $request->get('dateTo') : $dateFrom;
                    $lateNoShow = self::listLateNoShowCrew($dateFrom, $dateTo, $crewType);
                    $lateNoShow = self::listLateNoShowCrew($dateFrom, $dateTo, $crewType, $lateNoShow, TRUE);
                    $reasons =  ['' => "Late", 'head_order' => 'Head Order', 'roster_not_received' => 'Roster Not Received', 'sick' => 'Sick', 'no_show' => 'No Show'];
                    $lateNoShow = array_values($lateNoShow);
                    if (count($lateNoShow)) {
//                        $pagination = $paginator->make($lateNoShow, count($lateNoShow), $request->get('perPage') ? $request->get('perPage') : count($lateNoShow));
//                        $pagination = $paginator->make($lateNoShow, count($lateNoShow), $request->get('perPage') ? $request->get('perPage') : count($lateNoShow));
                    }
                    break;

                case self::FATIGUE_REPORT:
                    $dateFrom = $request->get('hoursReportDate');
                    $crewHours = self::getCrewHoursFlightHistoryDHCStandby($crew, $dateFrom, $counterTypes, 'yearToDate', 'yearToDate', $crewIds);
                    break;
            }
        }

        $routes = FlightRoute::listRoutes();

        $data = [
            'licenses'              => License::get(),
            'staffData'             => isset($staffData) ? $staffData : null,
            'routes'                => $routes,
            'periods'               => getPeriodsArray(),
            'monthNames'            => getMonthsArray(),
            'years'                 => getYearsArray(),
            'reportType'            => $reportType,
            'conditions'            => isset($conditions) ? $conditions : [],
            'date'                  => isset($date) ? $date : date('Y-m-d'),
            'crewType'              => isset($crewType) ? $crewType : null,
            'crew'                  => isset($crew) ? $crew : [],
            'dailyReport'           => isset($dailyReport) ? $dailyReport : [],
            'designationList'       => isset($designationList) ? $designationList : ['' => 'Select'],
            'crewHours'             => isset($crewHours) ? $crewHours : [],
            'monthArray'            => isset($monthArray) ? $monthArray : [],
            'flights'               => isset($flights) ? $flights : [],
            'pagination'            => isset($pagination) ? $pagination : [],
            'lateNoShow'            => isset($lateNoShow) ? $lateNoShow : [],
            'reasons'               => isset($reasons) ? $reasons : [],
            'standbyCount'          => isset($standbyCount) ? $standbyCount : [],
            'dhcCount'              => isset($dhcCount) ? $dhcCount : [],
            'crewDHCHours'          => isset($crewDHCHours) ? $crewDHCHours : [],
            'crewStandbyHours'      => isset($crewStandbyHours) ? $crewStandbyHours : [],
            'crewPeriodHours'       => isset($crewPeriodHours) ? $crewPeriodHours : [],
            'crewPeriodMonthlyHours'    => isset($crewPeriodMonthlyHours) ? $crewPeriodMonthlyHours : [],
            'crewStandbyMonthlyHours'   => isset($crewStandbyMonthlyHours) ? $crewStandbyMonthlyHours : [],
            'crewDHCMonthlyHours'       => isset($crewDHCMonthlyHours) ? $crewDHCMonthlyHours : [],
        ];
        return view('crew.report', $data);
    }


    public function getExpiryArrayByDepartment($userIDs){
        $staffByDepartment = [];
        $licenseIds = [];
        $userIds = [];


        $staffLicenses = StaffLicense::with(["user", "user.location", "user.location.airport"])
                                    ->join("users", "users.id", "=", "staff__licenses.user_id");


        if ($userIDs && count($userIDs)){
            $staffLicenses->whereIn("users.id", is_array($userIDs) ? $userIDs : [$userIDs]);
        }

        $staffLicenses = $staffLicenses->where(function($sql){
                                            $sql->whereNull("users.resigned_date")
                                                ->orWhere("users.resigned_date", EMPTY_DATE)
                                                ->orWhere("users.resigned_date", ">", date("Y-m-d"));
                                        })
                                        ->whereNull('users.deleted_at')
                                        ->orderBy("user_id")
                                        ->get(["staff__licenses.*"]);


        foreach ($staffLicenses as $each) {

            if (!in_array($each->license_id, $licenseIds)){
                $licenseIds[] = $each->license_id;
            }
            if (!in_array($each->user_id, $userIds)){
                $userIds[] = $each->user_id;
            }

            if (!isset($staffByDepartment[$each->user_id])){
                $staffByDepartment[$each->user_id] = [
                    'user'      => $each->user
                ];
            }
            if (!isset($staffByDepartment[$each->user_id][$each->license_id])){
                $staffByDepartment[$each->user_id][$each->license_id] = $each->expiry_date;
            }
        }

        return $staffByDepartment;
    }


    public function prepareFlightsForJsCalendar($fdpPlannedFlights, $fdpFlights){
        $fdpAllFlights = [];
        if ($fdpPlannedFlights){
            $fdpAllFlights = $this->convertFlightsForJsCalendar($fdpPlannedFlights);
        }
        if ($fdpFlights){
            $fdpAllFlights = array_merge($fdpAllFlights, $this->convertFlightsForJsCalendar($fdpFlights));
        }
        return json_encode($fdpAllFlights);
    }


    public function convertFlightsForJsCalendar($fdpFlights){
        $crewFlights = [];
        foreach ($fdpFlights as $fdp) {
            foreach($fdp->getFlights() as $each) {
                if ($each->atd && $each->ata) {
                    $depIndex = "atd";
                    $arrIndex = "ata";
                } elseif ($each->etd && $each->eta) {
                    $depIndex = "etd";
                    $arrIndex = "eta";
                } elseif ($each->ptd && $each->pta) {
                    $depIndex = "ptd";
                    $arrIndex = "pta";
                } else {
                    $depIndex = "std";
                    $arrIndex = "sta";
                }

                $start = $each->{$depIndex};
                $end = $each->{$arrIndex};
                $newFlight = [
                    'start'     => str_replace(" ", "T", $start),
                    'title'     => ProjectConstant::getConstants(IATA_CODE) . " " . $each->flight_number . " " .$each->departure_airport . "-" . $each->arrival_airport,
                ];

                if ($each->is_standby){
                    $newFlight['backgroundColor'] = "#D9534F";
                    $end = Add_Minutes_To_DateTime($start, $each->other_time * 60);

                }
                elseif($each->is_dhc){
                    $newFlight['backgroundColor'] = "#5CB85C";
                }

                $newFlight['end'] = str_replace(" ", "T", $end);

                $crewFlights[] = $newFlight;
            }
        }

        return $crewFlights;
    }


    /**
     * @param $crewFlights
     * @return array
     */
    public function getCrewRosterTotalHours($crewFlights){
        $userFlightsTotal = ['block' => 0, 'flight' => 0];
        foreach ($crewFlights as $crewFlightItem) {
            if (!($crewFlightItem->is_standby || $crewFlightItem->is_dhc)) {
                $time = getFlightTime($crewFlightItem, null, 'h-m-seperated');
                $userFlightsTotal['block'] += $time['h'] * 60 + $time['m'];

                $time = getFlightTime($crewFlightItem, 'flight_time', 'h-m-seperated');
                if ($time != '-')
                    $userFlightsTotal['flight'] += $time['h'] * 60 + $time['m'];
            }
        }
        return $userFlightsTotal;
    }



    /**
     * Get List of All No Show/Late/Sick,.. Crew
     * @param $dateFrom
     * @param $dateTo
     * @param $crewType
     * @param array $defaultArray
     * @param bool $crewLog
     * @param bool $reason
     * @return array
     */
    public static function listLateNoShowCrew($dateFrom, $dateTo, $crewType, $defaultArray = [], $crewLog = FALSE, $reason = FALSE){

        if ($crewLog){
            $table = 'log__flights__crew';
            $flightCrew = LogFlightCrew::join('flights', 'flights.id', '=', "$table.flight_id");
        }
        else {
            $table = 'flights__crew';
            $flightCrew = FlightCrew::join('flights', 'flights.id', '=', "$table.flight_id");
        }
        $flightCrew->join('flights__numbers', 'flights__numbers.id', '=', 'flights.flight_number_id')
            ->join('users', 'users.id', '=', "$table.user_id");

        switch($crewType) {
            case FCM_CREW:
                $flightCrew->join('crew__flight', 'crew__flight.user_id', '=', "$table.user_id");
                break;
            case CCM_CREW:
                $flightCrew->join('crew__cabin', 'crew__cabin.user_id', '=', "$table.user_id");
                break;
            default:
                return false;
        }

        $flightCrew->whereNull('flights.deleted_at')
            ->whereRaw("DATE(std) BETWEEN '$dateFrom' AND '$dateTo' ")
            ->whereRaw('(reason IS NOT NULL OR TIMEDIFF(std, report_time) < "01:15:00")')
            ->where(function($sql) use ($crewLog){
                $sql->where('reason', '<>', 'head_order');
                // If Using Crew Log Then Do Not Include WHERE Reason is Null
                if (!$crewLog)
                    $sql->orWhereNull('reason');
            });
//                    ->whereRaw('reason <> "head_order" OR reason IS NULL');


        if ($reason){
            $flightCrew->where('reason', $reason);
        }

        $flightCrew->orderBy('user_id');

        $flightCrew = $flightCrew->get([
            'users.id AS user_id',
            'is_contractor',
            'location_id',
            'thumb',
            'flight_id',
            'first_name',
            'last_name',
            'std',
            'report_time',
            'reason',
            'flight_number',
            'departure_airport',
            'arrival_airport',
            'users.location_id as base',
            ($crewType == 'fcm' ? 'is_captain' : 'is_purser'),
        ]);

        // Group By Key = Id
        foreach ($flightCrew as $each) {
            if (isset($defaultArray[$each->user_id])){
                $defaultArray[$each->user_id][] = $each;
            }
            else {
                $defaultArray[$each->user_id] = [$each];
            }
        }

        return $defaultArray;


    }

    /**
     * FDP Report
     * @param $dateFrom
     * @param $dateTo
     * @return array|\Illuminate\Database\Eloquent\Collection|mixed|static[]
     */
    public static function fdpCalculator($dateFrom, $dateTo){

        // Since Below Function Looks For Range (TO not included) and if dateFrom == dateTo => THEN dateFrom will Not Be Included
        $dateTo = date('Y-m-d', strtotime('+1 day', strtotime($dateTo)));

        // Get All Parent Flight With Child Id Included
        $flights = Flight::flightsRange($dateFrom, $dateTo, ['flightNumber', 'Aircraft', 'flightChild', 'flightCrew'], false);

        // Add Fdp Calculator Variables Into Flight Object
        $flights = self::addFdpCalculatorToFlights($flights);

        // Data Passed To View
        return $flights;
    }

    /**
     * Add Fdp Calculator Variables Into Flight Object
     * @param $flights
     * @return mixed
     */
    public static function addFdpCalculatorToFlights($flights){

        foreach ($flights as $i => $flight) {
            // Set
            $flights[$i]->flightWithRest = null;

            $type = self::getFlightType($flight);

            $childFlight = $flight->flightChild;

            switch($type){
                case self::ONE_FLIGHT:
                    // Get Departure, Arrival, FDP Start and Sector Counter
                    self::getFlightsDetails($flights[$i], $flight);

                    // CREW Count For FDP Extension Check
                    self::getCrewCounter($flights[$i]);

                    break;
                case self::FLIGHT_NO_REST:

                    self::getFlightsDetails($flights[$i], $flight);

                    self::getFlightsDetails($flights[$i], $childFlight, [ARRIVAL, FDP_END]);

                    // CREW Count For FDP Extension Check
                    self::getCrewCounter($flights[$i]);

                    break;

                case self::FLIGHT_WITH_REST:

                    $flights[$i]->flightChild = null;

                    $flights[$i]->flightWithRest = clone $childFlight;

                    // Get First Flight Details
                    self::getFlightsDetails($flights[$i], $flight);

                    // Get First Flight Crew Counter
                    self::getCrewCounter($flights[$i]);

                    // Get 2nd Flight Details
                    self::getFlightsDetails($flights[$i]->flightWithRest, $childFlight);

                    // Get 2nd Flight Crew Counter
                    self::getCrewCounter($flights[$i]->flightWithRest);

                    break;
            }
        }

        return $flights;
    }

    public static function getFlightsDetails(&$flight, $flightCheck, $type = [FDP_START, FDP_END, DEPARTURE, ARRIVAL]){
        $flightEndPlanned = $flightEndEstimate = $flightEndActual = null;
        $flight->sectorCount = $flight->child_id ? 2 : 1;
        $flight->fdpCabinCrewViolate = false;
        $flight->fdpFlightCrewViolate = false;

        // Flight Departure
        if (in_array(DEPARTURE, $type)) {
            $flight->departure = getFlightDepartureDate($flightCheck);
            $flight->departureType = getFlightDepartureTimeType($flightCheck);
        }

        // Flight Arrival
        if (in_array(ARRIVAL, $type)) {

            $flight->arrival = getFlightArrivalInitialDate($flightCheck);
            $flight->arrivalType = getFlightArrivalInitialTimeType($flightCheck);

            if ($flightCheck->pta && $flightCheck->pta != EMPTY_DATETIME) {
                $flight->arrival = $flightCheck->pta;
                $flight->arrivalType = PTA;
                $flightEndPlanned = $flightCheck->pta;
            }
            else if ($flightCheck->sta && $flightCheck->sta != EMPTY_DATETIME) {
                $flight->arrival = $flightCheck->sta;
                $flight->arrivalType = STA;
                $flightEndPlanned = $flightCheck->sta;
            }
            else {
                $flight->arrival = getFlightArrivalInitialDate($flightCheck);
                $flight->arrivalType = getFlightArrivalInitialTimeType($flightCheck);
                $flightEndPlanned = $flight->arrival;
            }

            if ($flightCheck->eta && $flightCheck->eta != EMPTY_DATETIME) {
                $flight->arrival = $flightCheck->eta;
                $flight->arrivalType = ETA;
                $flightEndEstimate = $flightCheck->eta;
            }

            if ($flightCheck->ata && $flightCheck->ata != EMPTY_DATETIME) {
                $flight->arrival = $flightCheck->ata;
                $flight->arrivalType = ATA;
                $flightEndActual = $flightCheck->ata;
            }
        }

        if (in_array(FDP_START, $type)) {
            if ($flightCheck->ptd && $flightCheck->ptd != EMPTY_DATETIME) {
                $flight->fdpStart = Add_Minutes_To_DateTime($flightCheck->ptd, self::CREW_REPORT_TIME_MIN, null, 'subtract');
            }
            else if ($flightCheck->std && $flightCheck->std != EMPTY_DATETIME) {
                $flight->fdpStart = Add_Minutes_To_DateTime($flightCheck->std, self::CREW_REPORT_TIME_MIN, null, 'subtract');
            }
            else {
                $flight->fdpStart = Add_Minutes_To_DateTime(getFlightDepartureInitialDate($flightCheck), self::CREW_REPORT_TIME_MIN, null, 'subtract');
            }
        }

        if (in_array(FDP_END, $type)) {
            if (isset($flightEndActual)){
                $flight->fdpEnd = Add_Minutes_To_DateTime($flightEndActual, self::CREW_RELEASE_TIME_MIN);
            }
            elseif (isset($flightEndEstimate)){
                $flight->fdpEnd = Add_Minutes_To_DateTime($flightEndEstimate, self::CREW_RELEASE_TIME_MIN);
            }
            elseif (isset($flightEndPlanned)){
                $flight->fdpEnd = Add_Minutes_To_DateTime($flightEndPlanned, self::CREW_RELEASE_TIME_MIN);
            }

            // FDP Planned
            $flight->fdpPlanned = isset($flightEndPlanned) ? Calculate_Duration($flight->fdpStart, Add_Minutes_To_DateTime($flightEndPlanned, self::CREW_RELEASE_TIME_MIN)) : '';
            // FDP Estimate
            $flight->fdpEstimate = isset($flightEndEstimate) ? Calculate_Duration($flight->fdpStart, Add_Minutes_To_DateTime($flightEndEstimate, self::CREW_RELEASE_TIME_MIN)) : '';
            // FDP Estimate
            $flight->fdpActual = isset($flightEndActual) ? Calculate_Duration($flight->fdpStart, Add_Minutes_To_DateTime($flightEndActual, self::CREW_RELEASE_TIME_MIN)) : '';
        }

        return [
            $flightEndPlanned,
            $flightEndEstimate,
            $flightEndActual
        ];
    }

    public static function getCrewCounter(&$flight){
        $flightCrewCounter = $fcmWithoutJuniorCounter = 0;
        $cabinCrewCounter = $ccmWithoutJuniorCounter = 0;

        $childFlight = $flight->flightChild;

        $childFlightCrew = $childFlight ? $childFlight->flightCrewItems : null;

        foreach ($flight->flightCrewItems as $flightCrew) {
            // Check If Crew Is Not Standby Or DHC
            $userId = $flightCrew->user_id;
            $position = $flightCrew->user->position()->first();

            if (!($flightCrew->is_standby || $flightCrew->is_dhc)) {
                if (in_array($position->type, [FCM_CPT_TYPE_ID, FCM_FA_TYPE_ID])) {
                    $flightCrewCounter++;
                    if ($position->name != COPILOT_JUNIOR_POSITION){
                        $fcmWithoutJuniorCounter++;
                    }
                }
                else {
                    $cabinCrewCounter++;
                    if ($position->name != FLIGHT_ATTENDANT_JUNIOR_POSITION){
                        $ccmWithoutJuniorCounter++;
                    }
                }
            }

            // Double Crew Detection
            if ($childFlightCrew && $flightCrew->is_dhc){
                $searchIfDhcOperatesNextSector = $childFlightCrew
                    ->filter(function($item) use($userId){
                        return $item->user_id == $userId;
                    })
                    ->filter(function($item){
                        return $item->is_dhc == false;
                    });

                if ($searchIfDhcOperatesNextSector && count($searchIfDhcOperatesNextSector)){
                    if (in_array($position->type, [FCM_CPT_TYPE_ID, FCM_FA_TYPE_ID])) {
                        $flightCrewCounter++;
                        if ($position->name != COPILOT_JUNIOR_POSITION){
                            $fcmWithoutJuniorCounter++;
                        }
                    }
                    else {
                        $cabinCrewCounter++;
                        if ($position->name != FLIGHT_ATTENDANT_JUNIOR_POSITION){
                            $ccmWithoutJuniorCounter++;
                        }
                    }
                }
            }
        }

        // FDP Planned And Estimate
        $fdpPlanned = $flight->fdpPlanned;

        // Crew Counters
        $flight->flightCrewCounter = $flightCrewCounter;
        $flight->cabinCrewCounter = $cabinCrewCounter;

        // FDP Max For Cabin Crew
        $fdpMaxCabinCrew = FdpCalculator($flight->fdpStart, $flight->sectorCount, TAJIKISTAN_CAA, true);

        $minCCM = $minFCM = 2;
        if ($flight->aircraft){
            $minCCM = $flight->aircraft->min_ccm ? $flight->aircraft->min_ccm : 2;
            $minFCM = $flight->aircraft->min_fcm ? $flight->aircraft->min_fcm : 2;
        }
        else if ($flight->aircraftType){
            $minCCM = $flight->aircraftType->min_ccm ? $flight->aircraftType->min_ccm : 2;
            $minFCM = $flight->aircraftType->min_fcm ? $flight->aircraftType->min_fcm : 2;
        }

        $fdpMaxCabinCrew = extendedFdp($flight->fdpStart, $flight->fdpEnd, $flight->sectorCount, $minCCM, $ccmWithoutJuniorCounter, $fdpMaxCabinCrew);
        $flight->fdpMaxCabinCrew = $fdpMaxCabinCrew;

        // FDP Limitation Crossed
        if ($fdpPlanned > $flight->fdpMaxCabinCrew){
            $flight->fdpCabinCrewViolate = true;
        }

        // FDP Max For Flight Crew
        $fdpMaxFlightCrew = FdpCalculator($flight->fdpStart, $flight->sectorCount, TAJIKISTAN_CAA, true);
        $fdpMaxFlightCrew = extendedFdp($flight->fdpStart, $flight->fdpEnd, $flight->sectorCount, $minFCM, $fcmWithoutJuniorCounter, $fdpMaxFlightCrew);
        $flight->fdpMaxFlightCrew = $fdpMaxFlightCrew;

        // FDP Limitation Crossed
        if ($fdpPlanned > $flight->fdpMaxFlightCrew){
            $flight->fdpFlightCrewViolate = true;
        }
    }

    /**
     * Check IF Flight Is With Rest Or Without It
     * @param $flight
     * @return string
     */
    public static function getFlightType($flight){
        // Check If Flight Is With Rest

        $dep = getFlightDepartureInitialDate($flight);
        $arr = getFlightArrivalInitialDate($flight);

        $plannedFdpStart =  Add_Minutes_To_DateTime($dep, self::CREW_REPORT_TIME_MIN, null, 'subtract');
        $plannedFdpEnd =  Add_Minutes_To_DateTime($arr, self::CREW_RELEASE_TIME_MIN);

        $plannedFdpDuration = Calculate_Duration($plannedFdpStart, $plannedFdpEnd);
        // Required Rest Calculate
        $requiredRest = max($plannedFdpDuration, 10);

        if ($flight->flightChild){
            $plannedNextFdpStart = getFlightDepartureInitialDate($flight->flightChild); // $flight->flightChild->std;

            if (Calculate_Duration($plannedFdpEnd, $plannedNextFdpStart) < $requiredRest){
                $type = self::FLIGHT_NO_REST;
            }
            else
                $type = self::FLIGHT_WITH_REST;
        }
        else {
            $type = self::ONE_FLIGHT;
        }


        return $type;
    }

        /**
     * Get An Array of Types of Counters
     * @param $date
     * @param $counterTypes
     * @return array|bool
     */
    public static function getCrewHoursCounterTypesArray($date, $counterTypes){
        if ($date && $counterTypes) {
            if (!is_array($counterTypes))
                $counterTypes = [$counterTypes];

            foreach ($counterTypes as $type) {
                switch ($type) {
                    case 'last365days':
                        $types[$type] = date('Y-m-d', strtotime("$date - 1 years"));
                        break;
                    case 'yearToDate':
                        $types[$type] = date('Y-01-01', strtotime($date));
                        break;
                    case 'monthToDate':
                        $types[$type] = date('Y-m-01', strtotime($date));
                        break;
                    case 'last90days':
                        $types[$type] = date('Y-m-d', strtotime("$date - 90 days"));
                        break;
                    case 'last28days':
                        $types[$type] = date('Y-m-d', strtotime("$date - 28 days"));
                        break;
                    case 'last7days':
                        $types[$type] = date('Y-m-d', strtotime("$date - 7 days"));
                        break;
                    case 'planned':
                        $types[$type] = date('Y-m-d', strtotime($date));
                        break;
                }
            }
            return $types;
        }
        return false;
    }

    /**
     * Get Label From Counter Type
     * @param $counterType
     * @return bool|string
     */
    public static function getLabelOutOfCounterType($counterType)
    {
        $label = false;
        if ($counterType) {

            switch ($counterType) {
                case "planned":
                    $label = "PLANNED ROSTER TOTAL";
                    break;
                case "last28days":
                    $label = "LAST 28 DAYS TOTAL";
                    break;
                case "last90days":
                    $label = "LAST 90 DAYS TOTAL";
                    break;
                case "last365days":
                    $label = "LAST 365 DAYS TOTAL";
                    break;
                case "fromTo":
                    $label = "PERIOD";
                    break;
                case 'default':
                    $label = "LAST 28 DAYS TOTAL";
                    break;

            }
            return $label;
        }
        return false;
    }

    /**
     * Calculate From And To Dates For Crew Hours Purposes
     * @param $counterTypes
     * @param bool $dateNow
     * @return array|bool
     */
    public static function getFromToOutOfCounterTypeArray($counterTypes, $dateNow = false){
        if ($counterTypes){
            $counterTypes = is_array($counterTypes) ? $counterTypes : [$counterTypes];
            $to = $dateNow ? $dateNow : date('Y-m-d');
            $from = date('Y-m-d');
            foreach ($counterTypes as $type => $date) {
                if ($type == 'planned')
                    $date = date('Y-m-d', strtotime("$date + 30 days"));

                if (strtotime($date) < strtotime($from))
                    $from = $date;

                if (strtotime($date) > strtotime($to))
                    $to = $date;
            }
            return ['from' => $from, 'to' => $to];
        }
        return false;
    }


    /**
     * Check Validity Of Crew Documents And Return Tab Class Of (Red And Orange or Empty)
     * @param $user
     * @param $crewDocuments
     * @return bool|string
     */
    public static function checkDocumentValidityStatus($user, $crewDocuments){
        if ($user && $crewDocuments) {
            foreach ($crewDocuments as $j => $groupedDoc) {
                $filters[] = Expiration_Date($user->{end($groupedDoc)})['filter'];
            }
            if (isset($filters) && $filters && count($filters)) {
                if (in_array('expired', $filters))
                    $tabClass = 'expired';
                elseif (in_array('valid_temp', $filters))
                    $tabClass = 'valid_temp';
                else
                    $tabClass = '';

                return $tabClass;
            }
        }
        return false;
    }

    /**
     * Get Crew Documents Array
     * @param $crewType
     * @return array|bool
     */
    public static function getCrewDocsByCrewType($crewType){
        if ($crewType && in_array($crewType, ['ccm', 'fcm'])) {
            $docs = [];

            switch ($crewType) {
                case 'fcm':
                    $docs = [
                        'Passport'           => ['passport_doi', 'passport_dex'],
                        'Aruba Validation'   => ['aruba_validation_from', 'aruba_validation_to'],
                        'SMS'                => ['cmc_from', 'cmc_to'],
                        'SEP 12'             => ['sep_12'],
                        'SEP 36'             => ['sep_36'],
                        'CRM'                => ['crm_from', 'crm_to'],
                        'Security'           => ['security'],
                        'Medical'            => ['medical_from', 'medical_to'],
                        'Flying License'     => ['flying_license_from', 'flying_license_to'],
                        'Line Check'         => ['line_check_from', 'line_check_to'],
                        'License Attachment' => ['license_attachment_from', 'license_attachment_to'],
                        'DGR'                => ['dgr_from', 'dgr_to'],
                        'LPC/IR'             => ['lpc_ir_from', 'lpc_ir_to'],
                        'OPC'                => ['opc_from', 'opc_to'],
                        'TRI/SFI'            => ['tri_sfi_from', 'tri_sfi_to']
                    ];
                    break;
                case 'ccm':
                    $docs = [
                        'Passport'       => ['passport_doi', 'passport_dex'],
                        'License'        => ['license_validity'],
                        'Medical'        => ['medical_next'],
                        'SMS'            => ['cmc_from', 'cmc_to'],
                        'SEP'            => ['sep_from', 'sep_to'],
                        'CRM'            => ['crm_from', 'crm_to'],
                        'DGR'            => ['dgr_from', 'dgr_to'],
                        'Security'       => ['security_from', 'security_to'],
                        'Line Check'     => ['line_check_from', 'line_check_to'],
                        'Practicals'     => ['practical_from', 'practical_to']
                    ];
                    break;
            }
            return $docs;
        }
        return false;
    }

    /**
     * Calculate Crew Hours, Flight History, DHC & Standby Counter Based On Counter Type
     * @param $crew
     * @param $date
     * @param bool $hoursType
     * @param bool $dhcStandbyType
     * @param bool $flightHistoryType
     * @param bool $userIds
     * @return array|bool
     */
    public static function getCrewHoursFlightHistoryDHCStandby($crew, $date, $hoursType = false, $dhcStandbyType = false, $flightHistoryType = false, $userIds = false){

        if ($crew && count($crew)) {
            $crewArray = [];
            // Get From And To Depending on Counter Types Selected
            if ($hoursType) {
                $counterTypes = getCrewHoursCounterTypesArray($date, $hoursType);
                $period = getFromToOutOfCounterTypeArray($counterTypes, $date);
                $from = $period['from'];
                $to = $period['to'];
            }
            elseif ($dhcStandbyType) {
                $counterTypes = getCrewHoursCounterTypesArray($date, $dhcStandbyType);
                $period = getFromToOutOfCounterTypeArray($counterTypes, $date);
                $from = $period['from'];
                $to = $period['to'];
            }
            else
                return false;

            if (isset($from) && isset($to)) {
                // Get Crew Flights
                $crewFlights = FlightCrew::getCrewFlights($from, $to, false, $userIds);
            }
            else {
                return false;
            }

            if (count($crewFlights)) {

                foreach ($crewFlights as $crewFlightItem) {
                    $crewId = $crewFlightItem->user_id;

                    // Get Block Duration and Departure Date
                    list($blockDuration, $departureDate) = Flight::calculateBlockDurationAndDepartureDate($crewFlightItem);

                    if ($crewFlightItem->is_sup){
                        if (!isset($crewArray[$crewId]['is_sup'])){
                            $crewArray[$crewId]['is_sup'] = 0;
                        }
                        $crewArray[$crewId]['is_sup']++;
                    }
                    elseif ($crewFlightItem->is_dhc || $crewFlightItem->is_standby){
                        // DHC & Standby Counter


                        if ($dhcStandbyType && $departureDate >= $counterTypes[$dhcStandbyType]) {
                            if ($crewFlightItem->is_dhc) {
                                if (!isset($crewArray[$crewId]['is_dhc'])){
                                    $crewArray[$crewId]['is_dhc'] = 0;
                                }

                                $crewArray[$crewId]['is_dhc']++;
                            }
                            else {
                                if (!isset($crewArray[$crewId]['is_standby'])){
                                    $crewArray[$crewId]['is_standby'] = 0;
                                }

                                $crewArray[$crewId]['is_standby']++;
                            }
                        }
                    }
                    else {
                        foreach ($counterTypes as $type => $itemDate) {
                            if ($hoursType) {
                                if ($departureDate >= $itemDate && ($type == 'planned' || $departureDate < $date)) {
                                    if (!isset($crewArray[$crewId][$type])) {
                                        $crewArray[$crewId][$type] = 0;
                                    }

                                    $crewArray[$crewId][$type] += $blockDuration;
                                }
                            }
                            // Flight History ( Routes Counter )
                            if ($flightHistoryType && $type == $flightHistoryType) {

                                $sector = getFlightSector($crewFlightItem->flight);

                                if (!isset($crewArray[$crewId]['flight_history'])){
                                    $crewArray[$crewId]['flight_history'] = [];
                                }

                                if (!isset($crewArray[$crewId]['flight_history'][$sector])) {
                                    $crewArray[$crewId]['flight_history'][$sector] = 0;
                                }

                                $crewArray[$crewId]['flight_history'][$sector]++;
                            }
                        }
                    }
                }
            }
            debug($crewArray);

            return $crewArray;

        }
        return false;
    }


    /**
     * Count Standby Hours (Total And Monthly Formatted)
     * @param $dateSearch
     * @param array $crewIds
     * @return array
     */
    public static function getStandbyQtyAndHours($dateSearch, $crewIds = []){

        $crewArray = []; $crewMonthlyArray = []; $qty = [];

        // Get Standby Flights
        $standbyFlights = FlightCrew::getCrewStandbyOrDhcFlights($dateSearch, false, $crewIds);

        // Count Standby Hours And Create Crew Array (Total And Monthly)
        if (count($standbyFlights)){
            foreach ($standbyFlights as $flight) {
                $crewId = $flight->user_id;
                // Get Block Duration and Departure Date
                $delayDuration = 0;
                if ($flight->atd && $flight->atd != EMPTY_DATETIME && strtotime($flight->atd) > strtotime($flight->std)){
                    $delayDuration = Calculate_Duration($flight->std, $flight->atd);
                }

                $standbyDuration = $delayDuration + (self::STANDBY_AFTER_DEPARTURE_OFFSET / 3600);

                $crewArray[$crewId] = isset($crewArray[$crewId]) ? $crewArray[$crewId] + $standbyDuration : $standbyDuration;

                // Count Number Of Times
                $qty[$crewId] = isset($qty[$crewId]) ? $qty[$crewId] + 1: 1;

                // Monthly Report
                $month = date('M', strtotime($flight->std));
                if (isset($crewMonthlyArray[$crewId][$month])){
                    $crewMonthlyArray[$crewId][$month] += $standbyDuration;
                }
                else {
                    $crewMonthlyArray[$crewId][$month] = $standbyDuration;
                }
            }
        }
        return [$crewArray, $crewMonthlyArray, $qty];
    }


    /**
     * Count DHC Hours (Total And Monthly Formatted)
     * @param $dateSearch
     * @param array $crewIds
     * @return array
     */
    public static function getDHCQtyAndHours($dateSearch, $crewIds = []){

        $crewArray = []; $crewMonthlyArray = []; $qty = [];

        // Get Standby Flights
        $dhcFlights = FlightCrew::getCrewStandbyOrDhcFlights($dateSearch, true, $crewIds);

        // Count Standby Hours And Create Crew Array (Total And Monthly)
        if (count($dhcFlights)){
            foreach ($dhcFlights as $flight) {
                $crewId = $flight->user_id;
                // Get Block Duration and Departure Date

                list($blockDuration, $departureDate) = Flight::calculateBlockDurationAndDepartureDate($flight);

                $crewArray[$crewId] = isset($crewArray[$crewId]) ? $crewArray[$crewId] + $blockDuration : $blockDuration;

                // Count Number Of Times
                $qty[$crewId] = isset($qty[$crewId]) ? $qty[$crewId] + 1: 1;

                // Monthly Report
                $month = date('M', strtotime($departureDate));

                if (!isset($crewMonthlyArray[$crewId][$month])){
                    $crewMonthlyArray[$crewId][$month] = 0;
                }

                $crewMonthlyArray[$crewId][$month] += $blockDuration;
            }
        }
        return [$crewArray, $crewMonthlyArray, $qty];
    }

    /**
     * Calculate Crew Hours Based On Period From And To
     * @param $crew
     * @param $dateFrom
     * @param $dateTo
     * @param bool|false $userIds
     * @param bool|false $dateSearch
     * @param bool|false $month
     * @param bool|false $year
     * @return array|bool
     */
    public static function getCrewHours($crew, $dateFrom, $dateTo, $userIds = false, $dateSearch = false, $month = false, $year = false){
        if ($crew && count($crew)) {
            if ($dateSearch == "monthYear" && $month && $year){
                $dateSearch = [];
                $dateSearch["month"] = $month;
                $dateSearch["year"] = $year;
                // Get Crew Flights
                $crewFlights = FlightCrew::getCrewFlights($dateFrom, $dateTo, $dateSearch, $userIds, null, true, true, true);
            }
            elseif (isset($dateFrom) && isset($dateTo)) {
                // Get Crew Flights
                $crewFlights = FlightCrew::getCrewFlights($dateFrom, $dateTo, false, $userIds, null, true, true, true);
            }
            else
                return false;

            list($crewArray, $crewMonthlyArray) = self::countHours($crewFlights);

            return [$crewArray, $crewMonthlyArray];
        }
        return false;
    }

    public static function countHours($crewFlights = []){
        $crewArray = []; $crewMonthlyArray = [];

        if (count($crewFlights)) {
            foreach ($crewFlights as $crewFlightItem) {

                $crewId = $crewFlightItem->user_id;
                // Get Block Duration and Departure Date
                list($blockDuration, $departureDate) = Flight::calculateBlockDurationAndDepartureDate($crewFlightItem);

                if (!isset($crewArray[$crewId])){
                    $crewArray[$crewId] = 0;
                }

                $crewArray[$crewId] += $blockDuration;

                // Monthly Report
                $month = date('M', strtotime($departureDate));

                if (!isset($crewMonthlyArray[$crewId][$month])){
                    $crewMonthlyArray[$crewId][$month] = 0;
                }

                $crewMonthlyArray[$crewId][$month] += $blockDuration;
            }
        }
        return [$crewArray, $crewMonthlyArray];
    }

    /**
     * Crew Hours History Daily (By Types And Range)
     * @param $crew
     * @param $userId
     * @param $counterTypes
     * @param null $period
     * @param null $periodFrom
     * @param null $periodTo
     * @return bool
     */
    public function getCrewHoursHistoricByDaily($crew, $userId, $counterTypes, $period = null, $periodFrom = null, $periodTo = null)
    {
        $date = date('Y-m-d');
        if ($periodFrom && $periodTo) {
            $history['from'] = date('Y-m-d', strtotime($periodFrom));
            $history['to'] = date('Y-m-d', strtotime($periodTo));
        }
        else if ($period) {
            $historyPeriod = getCrewHoursCounterTypesArray($date, $period);
            $historyPeriod = getFromToOutOfCounterTypeArray($historyPeriod);
            $history['from'] = $historyPeriod['from'];
            $history['to'] = $historyPeriod['to'];

        } else {
            return false;
        }


        $type = getCrewHoursCounterTypesArray($history['from'], $counterTypes);
        $dates = getFromToOutOfCounterTypeArray($type, $history['to']);


        $crewFlights = FlightCrew::getCrewFlights($dates['from'], $dates['to'], false, $userId, 'DESC', true, true, true);

        foreach ($crewFlights as $crewFlightItem) {
            $crewId = $crewFlightItem->user_id;

            // Get Block Duration and Departure Date
            list($blockDuration, $departureDate) = Flight::calculateBlockDurationAndDepartureDate($crewFlightItem);

            // For Historical Crew Hours Data
            if (isset($crewHoursByDate[$crewId][$departureDate]))
                $crewHoursByDate[$crewId][$departureDate] += $blockDuration;
            else
                $crewHoursByDate[$crewId][$departureDate] = $blockDuration;
        }

        if (isset($crewHoursByDate) && count($crewHoursByDate)) {
            for ($i = $history['to']; $i >= $history['from']; $i = date('Y-m-d', strtotime($i . '-1 day'))) {
                foreach ($counterTypes as $type) {
                    $option = false;
                    switch ($type) {
                        case 'last7days':
                            $option = ['range' => ['from' => date('Y-m-d', strtotime($i . '- 6 days')),  'to' => $i], 'value' => 0];
                            break;
                        case 'last28days':
                            $option = ['range' => ['from' => date('Y-m-d', strtotime($i . '- 27 days')), 'to' => $i], 'value' => 0];
                            break;
                        case 'last90days':
                            $option = ['range' => ['from' => date('Y-m-d', strtotime($i . '- 89 days')), 'to' => $i], 'value' => 0];
                            break;
                        case 'last365days':
                            $option = ['range' => ['from' => date('Y-m-d', strtotime($i . '- 1 years')),'to' => $i], 'value' => 0];
                            break;
                        case 'yearToDate':
                            $option = ['range' => ['from' => date('Y-01-01', strtotime($i)), 'to' => $i], 'value' => 0];
                            break;
                    }
                    if ($option)
                        $crew[$crewId]['historicHours'][$i][$type] = $option;
                }
            }

            foreach ($crewHoursByDate as $id => $data) {
                foreach ($data as $date => $total_hours) {
                    foreach ($crew[$id]['historicHours'] as $dateIndex => $lastArray) {
                        foreach ($lastArray as $i => $lastEach) {
                            if (dateWithinRange($date, $lastEach['range']['from'], $lastEach['range']['to']))
                                $crew[$id]['historicHours'][$dateIndex][$i]['value'] += $total_hours;
                        }
                    }
                }
            }
            return $crew;
        }
        return false;
    }
}
