<?php namespace App\Http\Controllers;

use App\Http\Requests;

use App\Models\Flight;
use App\Models\FlightNumber;
use App\Models\FlightSector;
use App\Models\PassengerFlight;
use App\Repositories\Interfaces\IFlightCabinRepository;
use App\Repositories\Interfaces\IFlightFareBasisRepository;
use App\Repositories\Interfaces\IFlightOrderStatusRepository;
use App\Repositories\Interfaces\IFlightRbdRepository;
use App\Repositories\Interfaces\IFlightTypeRepository;
use App\Repositories\Interfaces\IReportTypeRepository;
use App\Repositories\Interfaces\ISalesChannelRepository;


class CommercialController extends Controller {


	/**
	 * @param IFlightTypeRepository $flightTypeRep
	 * @param IFlightFareBasisRepository $flightFareBasisRep
	 * @param IFlightOrderStatusRepository $flightOrderStatusRep
	 * @param IFlightCabinRepository $flightCabinRep
	 * @param IFlightRbdRepository $flightRbdRep
	 * @param IReportTypeRepository $reportTypeRep
	 * @param ISalesChannelRepository $salesChannelRep
	 * @return \Illuminate\View\View
	 */
	public function index(IFlightTypeRepository $flightTypeRep, IFlightFareBasisRepository $flightFareBasisRep,
						  IFlightOrderStatusRepository $flightOrderStatusRep, IFlightCabinRepository $flightCabinRep,
							IFlightRbdRepository $flightRbdRep, IReportTypeRepository $reportTypeRep,
							ISalesChannelRepository $salesChannelRep)
	{
		$this->viewData = Flight::FlightQueryModule($flightTypeRep, $flightFareBasisRep, $flightOrderStatusRep, $flightCabinRep,
				$flightRbdRep, $reportTypeRep, $salesChannelRep, $this->region,$this->country, $this->city, $this->agency, $this->agencyCode);

		return view('commercial.index', $this->viewData);
	}


	public function indicators()
	{
		$data = $charts = [];

		if (request()->all()) {
			$data = $this->CalculateIndicators();
			$charts = $this->IndicatorsCharts($data);

			debug($data);
		}

		$this->viewData = [
			"data"					=> $data,
			"charts"				=> $charts,
			'flightNumbers'     	=> FlightNumber::listFlightNumbersByFlightTypeAndAirport('Scheduled'),
//			'sectorsList'           => FlightSector::listSectors(),

		];

		return view('commercial.indicators', $this->viewData);
	}

	public static function CalculateIndicators(){

		$from  = \request("departure_from");
		$to  = \request("departure_to");
		$flightNumber  = \request("flight_number");

		$flightNumber = !is_array($flightNumber) ? [$flightNumber] : $flightNumber;

		// Find Flights
		$to = date('Y-m-d', strtotime(" + 1 days", strtotime($to)));
		$flights = Flight::flightsRange($from, $to, ['flightNumber', 'aircraft'], false, false, $flightNumber);

		// Calculate Revenue By Date
		$revenue = PassengerFlight::getRevenue(request()->all(), [
			'field'         => 'departure_date',
			'period'        => 6,  // Daily,
			'date_format'   => TRUE
		], 'departure_date');


		$data = [];

		// Group Results
		foreach($flights as $flight){
			$departure = date('Y-m-d', strtotime($flight->std));

			if (!isset($data[$departure])){
				$data[$departure] = [
					'flight'	=> null,
					'revenue'	=> null
				];
			}

			if ($flight->std){
				$data[$departure]['flight'] = $flight;
			}
		}

		foreach ($revenue as $each) {
			if (!isset($data[$each->grouped_field])){
				$data[$each->grouped_field] = [
					'flight'	=> null,
					'revenue'	=> null
				];
			}

			$data[$each->grouped_field]['revenue'] = $each;
		}

		// Calculate Indicators

		foreach ($data as $date => $each) {
			if (!($each['flight'] && $each['flight']->flightNumber && $each['flight']->flightNumber->distance)) {
				continue;
			}

			if (!($each['flight']->aircraft && ($each['flight']->aircraft->config_c + $each['flight']->aircraft->config_y) > 0)) {
				continue;
			}

			$distance = $each['flight']->flightNumber->distance;
			$capacity = $each['flight']->aircraft->config_c + $each['flight']->aircraft->config_y;


			$yield = $each['revenue'] && $each['revenue']->pax != 0 > 0 ? round($each['revenue']->revenue / $each['revenue']->pax) : 0;
			$ask = round($distance * $capacity);
			$rask = $each['revenue'] && $ask != 0 ? round($each['revenue']->revenue / $ask, 2) : 0;
			$rpk = $each['revenue'] ? round($each['revenue']->pax * $distance) : 0;
			$loadFactor = $ask != 0 ? 100 * round($rpk / $ask, 2) : 0;
			$yieldKm = $each['revenue'] && $rpk != 0 ? 100 * round($each['revenue']->revenue / $rpk, 2) : 0;

			$data[$date]['indicators'] = [
				'capacity'		=> $capacity,
				'pax'			=> $each['revenue'] ? $each['revenue']->pax : 0,
				'revenue'		=> $each['revenue']? $each['revenue']->revenue : 0,
				'yield'			=> $yield,
				'ask'			=> $ask,
				'rask'			=> $rask,
				'rpk'			=> $rpk,
				'loadFactor'	=> $loadFactor,
				'yieldKm'		=> $yieldKm,
			];
		}


		return $data;
	}

	public static function IndicatorsCharts($data){
		$chart = [];

		foreach ($data as $date => $each) {

			// ASK-RPK
			if (!isset($chart['ask-rpk'])) {
				$chart['ask-rpk'] = [
					'label' => json_encode(['ASK', 'RPK']),
					'x' 	=> 'date',
					'y' 	=> json_encode(['ask', 'rpk']),
					'data' 	=> [],
				];
			}
			$chart['ask-rpk']['data'][] = [
				'date'		=> $date,
				'ask'		=> isset($each['indicators']['ask']) ? $each['indicators']['ask'] : 0,
				'rpk'		=> isset($each['indicators']['rpk']) ? $each['indicators']['rpk'] : 0,
			];

			// CAP-PAX-Yield
			if (!isset($chart['cap-pax-yield'])){
				$chart['cap-pax-yield'] = [
					'label'			=> json_encode(['CAP', 'PAX', 'Yield']),
					'x'				=> 'date',
					'y'				=> json_encode(['cap', 'pax', 'yield']),
					'data'			=> [],
				];
			}
			$chart['cap-pax-yield']['data'][] = [
				'date'		=> $date,
				'cap'		=> isset($each['indicators']['capacity']) ? $each['indicators']['capacity'] : 0,
				'pax'		=> isset($each['indicators']['pax']) ? $each['indicators']['pax'] : 0,
				'yield'		=> isset($each['indicators']['yield']) ? $each['indicators']['yield'] : 0,
			];
		}

		foreach ($chart as $i => $each) {
			$chart[$i]['data'] = json_encode($chart[$i]['data']);
		}

		return $chart;
	}

	/**
	 * Show the form for creating a new resource.
	 *
	 * @return Response
	 */
	public function search()
	{

	}

	/**
	 * Store a newly created resource in storage.
	 *
	 * @return Response
	 */
	public function store()
	{
		//
	}

	/**
	 * Display the specified resource.
	 *
	 * @param  int  $id
	 * @return Response
	 */
	public function show($id)
	{
		//
	}

	/**
	 * Show the form for editing the specified resource.
	 *
	 * @param  int  $id
	 * @return Response
	 */
	public function edit($id)
	{
		//
	}

	/**
	 * Update the specified resource in storage.
	 *
	 * @param  int  $id
	 * @return Response
	 */
	public function update($id)
	{
		//
	}

	/**
	 * Remove the specified resource from storage.
	 *
	 * @param  int  $id
	 * @return Response
	 */
	public function destroy($id)
	{
		//
	}

}
