<?php

namespace App\Http\Controllers;

use App\Events\FlightComCreated;
use App\Events\NewMessage;
use App\Http\Controllers\Controller;
use App\Models\Aircraft;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Container;
use App\Models\ContainerType;
use App\Models\Flight;
use App\Models\FlightCom;
use App\Models\FlightComCheckin;
use App\Models\FlightComContainer;
use App\Models\FlightComDetails;
use App\Models\FlightComPicture;
use App\Models\FlightComUserViews;
use App\Models\FlightContainer;
use App\Models\FlightContainerInfo;
use App\Models\FlightMessage;
use App\Models\FlightNumber;
use App\Models\FlightPTS;
use App\Models\FlightStaff;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Session;

class ChatController extends Controller
{
    public function __construct()
    {
        //$this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {

        $users = User::where('id', '<>', auth()->user()->id)->get();
        $user = auth()->user();

        // Added
        $selectedAirlineIds = Airline::getSelectedAirlineIds($request);
        $selectedAirportIds = Airport::getSelectedAirportIds($request);

        $aircraftByAirline = Aircraft::getAircraftByAirline($selectedAirlineIds);

        $date = $request->get("date") ? $request->get("date") : date("Y-m-d");

        $flightsData = $this->getFlights($selectedAirlineIds, $selectedAirportIds, $date);

        $flightIDs = $keys = $all = $airlineByFlight = [];
        foreach ($flightsData as $type => &$flights) {
            if ($type == ALL){
                foreach ($flights as $i => $each) {
                    foreach ($each as $flight) {
                        foreach ($flight as $item) {
                            $all[] = $item;
                            $airlineByFlight[$item->id] = $item->flightNumber->airline_id;
                        }
                    }
                }
            }
            else {
                foreach ($flights as $each) {
                    if ($each->flightCom) {
                        $keys[$each->id] = $each->flightCom->key;
                    }
                    $flightIDs[] = $each->id;
                }
            }
        }

        $flightsData[DEPARTURE] = array_values($flightsData[DEPARTURE]);
        $flightsData[ARRIVAL] = array_values($flightsData[ARRIVAL]);
        $flightsData[ALL] = $all;

        $this->viewData = [
            'dateToday'         => date("Y-m-d"),
            'users'             => $users,
            'user'              => $user,
            'flightsData'       => Collection::make($flightsData),
        ];

        return view('chat.test', $this->viewData);
    }


    public function loadMessages(Request $request){
        $authUserID = Auth::user()->id;
        $initialLoad = $request->get("initial_load");
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        if (!$flightID){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flight = Flight::find($flightID);

        $messages = $this->getFlightMessages($authUserID, $flightID, $request->get("last_message_id"), $request->get("first_message_id"), $initialLoad);

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs);

        // $uld = $this->getContainers($flightID);

        $flightObj = $this->prepareFlightObj($flight);

        return response()->json([
            "success"           => true,
            "serviceData"       => null,
            "uld"               => null,
            "unseen"            => $unseen,
            "flight"            => $flightObj,
            "counter"           => $counter,
            // DO NOT UPLOAD THE SAME WAY
            "com"               => $messages["com"],
            "created"           => $messages["created"],
//            "com"               => count($messages["messages"]) ? $messages["com"] : null,

            "messages"          => $messages["messages"],
            "key"               => $messages["key"],
            "pictures"          => $messages["pictures"],
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    function loadStaff(Request $request){

        $flightID = $request->get("flight_id");

        if (!$flightID){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flight = Flight::with([
            "flightStaff",
            "flightNumber",
            "flightNumber.airline"
        ])->find($flightID);

        if (!$flight){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flightObj = [
            "id"    => $flight->id,

            "std"   => $flight->std && $flight->std != EMPTY_DATETIME ? $flight->std : "",
            "ptd"   => $flight->ptd && $flight->ptd != EMPTY_DATETIME ? $flight->ptd : "",
            "etd"   => $flight->etd && $flight->etd != EMPTY_DATETIME ? $flight->etd : "",
            "atd"   => $flight->atd && $flight->atd != EMPTY_DATETIME ? $flight->atd : "",

            "sta"   => $flight->sta && $flight->sta != EMPTY_DATETIME ? $flight->sta : "",
            "pta"   => $flight->pta && $flight->pta != EMPTY_DATETIME ? $flight->pta : "",
            "eta"   => $flight->eta && $flight->eta != EMPTY_DATETIME ? $flight->eta : "",
            "ata"   => $flight->ata && $flight->ata != EMPTY_DATETIME ? $flight->ata : "",

            "staffList" => Flight::getFlightStaff($flight),
        ];

        // Get staff
        $serviceData = Flight::getFlightsStaffServiceData($flight);

        return response()->json([
            "success"           => true,
            "serviceData"       => $serviceData,
            "flight"            => $flightObj,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    function loadTrc(Request $request){
        $flightID = $request->get("flight_id");

        if (!$flightID){
            return response()->json([
                "success"      => false,
            ]);
        }

        $uld = $this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "uld"               => $uld,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }


    /**
     * Update Flight CKIN Info
     * ULDs
     */
    public function updateFlightDetails(){

    }

    function getULDRows(){
        return [
            "11"    => "11",
            "12"    => "12",
            "13"    => "13",
            "14"    => "14",

            "21"    => "21",
            "22"    => "22",
            "23"    => "23",

            "31"    => "31",
            "32"    => "32",
            "33"    => "33",

            "41"    => "41",
            "42"    => "42",
            "43"    => "43",

            "5"     => "5",
            "51"    => "51",
            "52"    => "52",
            "53"    => "53",
        ];
    }

    public function staff(Request $request){
        if (!$request->get("flight_id")){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flight = Flight::with(["flightNumber"])
            ->find($request->get("flight_id"));

        if (!$flight){
            return response()->json([
                "success"      => false,
            ]);
        }

        $authUserID = Auth::user()->id;
        $flightCom = FlightCom::getOrCreateFlightCom($flight->id, $authUserID, $request->get("key"), true);

        // Update Flight Staff
        $flightComDetails = FlightComDetails::create([
            "flight_com_id" => $flightCom->id,
            "created_by"    => $authUserID,
        ]);

        $createdFlightComStaff = FlightStaff::updateFlightsStaff($request, $flight, $flightComDetails);

        if (!count($createdFlightComStaff)){
            $flightComDetails->delete();
        }


        // Added
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $request->get("last_message_id"));

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs);

        $uld = [];// $this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);


    }

    public function checkInTrc(Request $request){

        if (!$request->get("flight_id")){
            return response()->json([
                "success"      => false,
            ]);
        }

        debug($request->all());

        $authUserID = Auth::user()->id;
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        // Process Message
        list($flight) = $this->processMessage($request, $flightID);

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $request->get("last_message_id"));

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs);

        $uld = []; //$this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    public function pts(Request $request){

        if (!$request->get("flight_id")){
            return response()->json([
                "success"      => false,
            ]);
        }

        $flight = Flight::with(["flightNumber"])
            ->find($request->get("flight_id"));

        if (!$flight){
            return response()->json([
                "success"      => false,
            ]);
        }

        $authUserID = Auth::user()->id;
        $flightCom = FlightCom::getOrCreateFlightCom($flight->id, $authUserID, $request->get("key"), true);

        // Create Com Details
        $flightComDetails = FlightComDetails::create([
            "flight_com_id" => $flightCom->id,
            "created_by"    => $authUserID,
        ]);

        $pts = FlightPTS::updateFields($request, $flightComDetails, $request->get("flight_id"));

        if (!$pts){
            $flightComDetails->delete();
        }

        // Added
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $request->get("last_message_id"));

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs);

        $uld = []; // $this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);

    }

    public static function setFlightProperties(&$flight, $type){

        $flight->type = $type;

        $flightNumber = $flight->flightNumber;
        $airline = $flightNumber ? $flightNumber->airline : null;
        list($fn1, $fn2, $fn3) = getFlightNumberStrings($flight);

        $depAP = getFlightDepartureAirport($flight);
        $dep = $depAP ? $depAP->iata : "";

        $arrAP = getFlightArrivalAirport($flight);
        $arr = $arrAP ? $arrAP->iata : "";

        $depType = getFlightDepartureTimeType($flight, true);
        $depInitialType = getFlightDepartureInitialTimeType($flight, true);

        $arrType = getFlightArrivalTimeType($flight, true);
        $arrInitialType = getFlightArrivalInitialTimeType($flight, true);


        $depSpan = departureArrivalTypeSpan($depType, null);
        $depInitialSpan = departureArrivalTypeSpan($depInitialType, null);

        $arrSpan = departureArrivalTypeSpan($arrType, null);
        $arrInitialSpan = departureArrivalTypeSpan($arrInitialType, null);

        $depInitialTxt = $depInitialSpan . "<span class='dep-time-initial'>" . baseDateFormat(getFlightDepartureInitialDate($flight), true) . "</span>";
        $depTxt = $depSpan . "<span id='flt_dep_arr_time' class='dep-time'>" . baseDateFormat(getFlightDepartureDate($flight), true) . "</span>";

        $arrInitialTxt = $arrInitialSpan . "<span class='arr-time-initial'>" . baseDateFormat(getFlightArrivalInitialDate($flight), true) . "</span>";
        $arrTxt = $arrSpan . "<span id='flt_dep_arr_time' class='arr-time'>" . baseDateFormat(getFlightArrivalDate($flight), true ). "</span>";


        $ac = getFlightAircraft($flight);
        $acType = getFlightAircraftType($flight);

        $cap =  getFlightCapacity($flight);
        $bookedPax = getFlightBookedPax($flight);
        $actualPax = getFlightActualPax($flight);

        $capReady = "CFG ".($cap[0] ? $cap[0]."C " : "").$cap[1]."Y";
        $bookedPaxReady = ($bookedPax[0] || $bookedPax[1]) ? "BKD ".($bookedPax[0] ? $bookedPax[0]."C " : "").$bookedPax[1]."Y" : "";
        $actualPaxReady = ($actualPax[0] || $actualPax[1]) ? "ACT ".($actualPax[0] ? $actualPax[0]."C " : "").$actualPax[1]."Y" : "";

        $aircraftDetails = $acType.($ac ? " | ".$ac : "");
        //

        if ($type == DEPARTURE){
            $acDetails = $aircraftDetails." | ".$capReady.($bookedPaxReady ? " | ".$bookedPaxReady : "").($actualPaxReady ? " | ".$actualPaxReady : "");
        }
        else {
            $acDetails = $aircraftDetails." | ".$capReady.($actualPaxReady ? " | ".$actualPaxReady : "");
        }

        $icon =  strtolower($type)."64.png";

        $flight->dep_arr_icon = asset("assets/img/icons/others/{$icon}");
        $flight->status = getFlightClassNameStrict($flight, $type);
        $flight->airline_logo = getAirlineLogo($airline);
        $flight->fn_ready = $fn3;
        $flight->sector_ready = " | ".$dep."-".$arr."";
        $flight->aircraft_details = $acDetails;


        if ($type == DEPARTURE){
            $fltDepArrDetails  = ($depType == $depInitialType ? "" :  $depInitialTxt) ." ". $depTxt;
        }
        else{
            $fltDepArrDetails = ($arrType == $arrInitialType ? "" :  $arrInitialTxt) ." ". $arrTxt;
        }

        $flight->dep_arr_details = $fltDepArrDetails;
    }

    public static function getFlights($selectedAirlineIds, $selectedAirportIds, $date){

        $startDate  = $date . " 00:00:00";
        $endDate    = $date . " 23:59:59";

        $data = [
            ALL         => [],
            DEPARTURE   => [],
            ARRIVAL     => [],
        ];

        $handlingStations = Airport::getHandlingStationsIdAndObject();

        if (!$selectedAirlineIds || !$selectedAirportIds){
            return [
                ALL         => [],
                DEPARTURE   => [],
                ARRIVAL     => [],
            ];
        }

        // Load Arrival Flights
        $flights = Flight::handlingFlightsRange($startDate, $endDate, $selectedAirlineIds, $selectedAirportIds, null, ["arrival_date_search" => true], true);
        $flights = $flights->sortBy("sta")->all();

        $allIParentIDs = [];
        $allIChildrenIDs = [];
        foreach ($flights as $flight) {
            $type = getFlightsDepArrType($flight, $handlingStations);
            if ($type && $type == ARRIVAL){

                // Set Properties
                self::setFlightProperties($flight, $type);

                $time = strtotime(getFlightArrivalDate($flight));

                if (isset($data[$type][$time])){
                    do {
                        $time++;
                    }
                    while(isset($data[$type][$time]));
                }

                $data[$type][$time] = $flight;

                /// All flights
                $index = $flight->parent_id ? $flight->parent_id : $flight->id;
                if ($flight->parent_id){
                    //debug("ARR child");
                    $allIChildrenIDs[$index] = $time;
                    $data[ALL][$time][$index][1] = $flight;
                }
                else {
                    $allIParentIDs[$index] = $time;
                    $data[ALL][$time][$index][] = $flight;
                }
            }
        }

        // Load Departure Flights
        $flights = Flight::handlingFlightsRange($startDate, $endDate, $selectedAirlineIds, $selectedAirportIds, null, [], true);
//        $flights = $flights->sortBy("std")->all();

        $j = 0;
        foreach ($flights as $flight) {

            $type = getFlightsDepArrType($flight, $handlingStations);
            if ($type && $type == DEPARTURE){

                // Set Properties
                self::setFlightProperties($flight, $type);

                $j++;
                $time = strtotime(getFlightDepartureDate($flight));

                if (isset($data[$type][$time])){
                    do {
                        $time++;
                    }
                    while(isset($data[$type][$time]));
                }

                $data[$type][$time] = $flight;

                /// All flights
                $index = $flight->parent_id ? $flight->parent_id : $flight->id;
                if ($flight->parent_id){
                    if (isset($allIParentIDs[$index])){
                        //debug("DEP child = found match");
                        //debug(getFlightNumber($flight));
                        $data[ALL][$allIParentIDs[$index]][$index][] = $flight;
                        continue;
                    }
                }
                else {
                    if (isset($allIChildrenIDs[$index])){
                        //debug("DEP parent = found arrival child");
                        $data[ALL][$allIChildrenIDs[$index]][$index][0] = $flight;
                        continue;
                    }
                }

                $data[ALL][$time][$index][] = $flight;
            }
        }

        ///debug($data[ALL]);

        foreach ($data as &$each) {
            ksort($each);
        }

        return $data;
    }

    function prepareFlightObj($flight){
        $cap = getFlightCapacity($flight);
        $bookedPax = getFlightBookedPax($flight);
        $actualPax = getFlightActualPax($flight);

        $flightObj = [
            "id"            => $flight->id,
            "aircraft_id"   => $flight->aircraft_id,
            "gate_boarding" => $flight->gate_boarding,
            "gate_parking"  => $flight->gate_parking,
            "message"       => $this->prepareFlightMessages($flight),

            "std"           => $flight->std && $flight->std != EMPTY_DATETIME ? $flight->std : "",
            "ptd"           => $flight->ptd && $flight->ptd != EMPTY_DATETIME ? $flight->ptd : "",
            "etd"           => $flight->etd && $flight->etd != EMPTY_DATETIME ? $flight->etd : "",
            "atd"           => $flight->atd && $flight->atd != EMPTY_DATETIME ? $flight->atd : "",

            "sta"           => $flight->sta && $flight->sta != EMPTY_DATETIME ? $flight->sta : "",
            "pta"           => $flight->pta && $flight->pta != EMPTY_DATETIME ? $flight->pta : "",
            "eta"           => $flight->eta && $flight->eta != EMPTY_DATETIME ? $flight->eta : "",
            "ata"           => $flight->ata && $flight->ata != EMPTY_DATETIME ? $flight->ata : "",

            "ac_type"       => getFlightAircraftType($flight),
            "ac"            => getFlightAircraft($flight),
            "cfg"           => "CFG ".($cap[0] ? $cap[0]."C " : "").$cap[1]."Y",
            "bkd"           => ($bookedPax[0] || $bookedPax[1]) ? "BKD ".($bookedPax[0] ? $bookedPax[0]."C " : "").$bookedPax[1]."Y" : "",
            "act"           => ($actualPax[0] || $actualPax[1]) ? "ACT ".($actualPax[0] ? $actualPax[0]."C " : "").$actualPax[1]."Y" : "",

            "pax_a"         => $flight->pax_a_actual,
            "pax_c"         => $flight->pax_c_actual,
            "pax_y"         => $flight->pax_y_actual,
            "pax_jmp"       => $flight->pax_jmp_actual,
            "pax_adults"    => $flight->pax_adults_actual,
            "pax_m"         => $flight->pax_m_actual,
            "pax_f"         => $flight->pax_f_actual,
            "pax_ch"        => $flight->pax_ch_actual,
            "pax_inf"       => $flight->pax_inf_actual,
        ];

        return $flightObj;
    }

    function prepareFlightMessages($flight){
        if (!$flight){
            return null;
        }

        $flightMessages = FlightMessage::where("flight_id", $flight->id)
            ->get([
                "message_type"
            ]);

        if (!$flightMessages->count()){
            return null;
        }

        $result = [];
        foreach ($flightMessages as $each) {
            $result[$each->message_type] = $each->message_type;
        }

        return array_values($result);
    }


    /**
     * For
     * @param $flightID
     * @return array
     */
    public function getContainers($flightID){
        // ULD
        $flightContainerInfo = $this->getFlightContainerInfo($flightID);

        $result = [];

        if ($flightContainerInfo) {
            $containers = FlightContainer::with([
                "destination",
                "container",
                "container.type",
            ])
                ->where("info_id", $flightContainerInfo->id)
                ->orderBy("row_number", "DESC")
                ->orderBy("position", "DESC")
                ->get();

            $flight = Flight::with(["flightNumber.arrivalAirport"])->find($flightID);
            $destAP = getFlightArrivalAirport($flight, false, true);

            foreach ($containers as $each) {
                $pos = $each->row_number . ($each->position ? $each->position : "");

                $index = $pos;
                if ($each->row_number == 5){
                    $index = $pos . "0";
                }

                if (isset($result[$index])){
                    $k = 0;
                    do {
                        $index .= $k++;
                    }while(isset($result[$index]));
                }

                $result[$index] = [
                    "dest"      => $each->destination && $each->destination->iata ? $each->destination->iata : $destAP,
                    "pos"       => $pos,
                    "content"   => $each->loaded_type ? $each->loaded_type : "",
                    "type"      => $each->container && $each->container->type ? $each->container->type->name : ($each->containerType ? $each->containerType->name : ""),
                    "num"       => $each->container && $each->container->num ? $each->container->num : "",
                    "owner"     => $each->container ? ($each->container->owner ? $each->container->owner : ($each->container->airline ? $each->container->airline->iata : "" )) : "",
                    "wt"        => $each->weight ? $each->weight : "",
                    "vr"        => $each->vr || $each->vr == 0 || $each->vr == "0" ? $each->vr : "",
                    "pcs"       => $each->bag_pcs ? $each->bag_pcs : "",
                    "notoc"     => $each->notoc ? "YES" : "NO",
                ];

            }

            ksort($result);
            $result = array_reverse(array_values($result));
        }

        return $result;
    }

    public function sendMessage(Request $request){
        $authUserID = Auth::user()->id;
        $flightID = $request->get("flight_id");
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        // Process Message
        list($flight) = $this->processMessage($request, $flightID);

        // Get Messages
        $messages = $this->getFlightMessages($authUserID, $flightID, $request->get("last_message_id"));


        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs);

        $uld = []; //$this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);

    }

    function processMessage(Request $request, $flightID){
        $authUserID = Auth::user()->id;
        $key = $request->get('key');
        $message = $request->get('message');

        // Gate Boarding & Parking
        $flight = Flight::with([
            "message",
            "flightNumber",
            "flightNumber.airline",
        ])->find($flightID);

        if ($flight){
            if ($request->get("gate_boarding")){
                $flight->gate_boarding = $request->get("gate_boarding") ? strtoupper($request->get("gate_boarding")) : null;
            }
            if ($request->get("gate_parking")){
                $flight->gate_parking = $request->get("gate_parking") ? strtoupper($request->get("gate_parking")) : null;
            }
            $flight->save();
        }

        $properties = [
            "gate"          => $request->get("gate_boarding") ? strtoupper($request->get("gate_boarding")) : null,
            "pos"           => $request->get("gate_parking") ? strtoupper($request->get("gate_parking")) : null,

            "seat_a"        => $request->get("seat_a"),
            "seat_b"        => $request->get("seat_b"),
            "seat_c"        => $request->get("seat_c"),
            "seat_d"        => $request->get("seat_d"),
            "seat_e"        => $request->get("seat_e"),

            "chkd_bag_pcs"  => $request->get("chkd_bag_pcs"),
            "chkd_bag_wt"   => $request->get("chkd_bag_wt"),
            "gate_bag_pcs"  => $request->get("gate_bag_pcs"),
            "gate_bag_wt"   => $request->get("gate_bag_wt"),

            "trc_rmks"      => $request->get("trc_rmks") ? strtoupper($request->get("trc_rmks")) : null,
            "spvr_rmks"     => $request->get("spvr_rmks") ? strtoupper($request->get("spvr_rmks")) : null,
            "ckin_num"      => $request->get("ckin_num") ? strtoupper($request->get("ckin_num")) : null,
            "ckin_ctr"      => $request->get("ckin_ctr") ? strtoupper($request->get("ckin_ctr")) : null,
            "ckin_rmks"     => $request->get("ckin_rmks") ? strtoupper($request->get("ckin_rmks")) : null,

            "ckin_staff"    => $request->get("ckin_staff") ? strtoupper($request->get("ckin_staff")) : null,
            "gate_staff"    => $request->get("gate_staff") ? strtoupper($request->get("gate_staff")) : null,
            "trc"           => $request->get("trc")  ? strtoupper($request->get("trc")) : null,
            "spvr"          => $request->get("spvr") ? strtoupper($request->get("spvr")) : null,


        ];

        $flightCom = FlightCom::getOrCreateFlightCom($flightID, $authUserID, $key, true);

        $changesFound = false;
        foreach ($properties as $var => $value ) {
            if (!($value || $value === 0 || $value === "0")){
                continue;
            }

            if ($flightCom->{$var} != $value){
                $changesFound = true;
            }

            $flightCom->{$var} = $value;
        }
        if ($changesFound)
        {
            $flightCom->updated_by = $authUserID;
        }
        $flightCom->save();

        /*
         * OLD structure
        $flightComDetails = FlightComDetails::create([
                "flight_com_id" => $flightCom->id,
                "created_by"    => $authUserID,
                "message"       => $message ? $message : null,
                "trc_rmks"      => $request->get("trc_rmks"),
                "spvr_rmks"     => $request->get("spvr_rmks"),
            ] + $properties);
        */

        // NEW structure
        $flightComDetails = FlightComDetails::create([
            "flight_com_id" => $flightCom->id,
            "created_by"    => $authUserID,
            "message"       => $message ? $message : null,
        ]);


        // Broadcast
        broadcast(new NewMessage($flightComDetails))->toOthers();


        // Add to prop
        $checkInProps = [
            "aircraft_id"       => $request->get("aircraft_id") === "0" ? null : $request->get("aircraft_id"),
            "capacity_c"        => $request->get("capacity_c"),
            "capacity_y"        => $request->get("capacity_y"),
            "pax_a_booked"      => $request->get("pax_a_booked"),
            "pax_c_booked"      => $request->get("pax_c_booked"),
            "pax_y_booked"      => $request->get("pax_y_booked"),
            "pax_inf_booked"    => $request->get("pax_inf_booked"),

            "pax_a"             => $request->get("pax_a"),
            "pax_c"             => $request->get("pax_c"),
            "pax_y"             => $request->get("pax_y"),
            "pax_jmp"           => $request->get("pax_jmp"),
            "pax_adults"        => $request->get("pax_adults"),
            "pax_m"             => $request->get("pax_m"),
            "pax_f"             => $request->get("pax_f"),
            "pax_ch"            => $request->get("pax_ch"),
            "pax_inf"           => $request->get("pax_inf"),
        ];

        FlightComCheckin::createRecord($flight, $flightComDetails, $properties + $checkInProps);
        // End

        // ULD Containers
        $rowULD = $request->get("uld_row");
        if ($rowULD && count($rowULD)) {
            // ULD
            $flightContainerInfo = $this->getFlightContainerInfo($flightID);

            $posULD     = $request->get("uld_pos");
            $destULD    = $request->get("uld_dest");
            $contentULD = $request->get("uld_content");
            $typeULD    = $request->get("uld_type");
            $numULD     = $request->get("uld_num");
            $ownerULD   = $request->get("uld_owner");
            $wtULD      = $request->get("uld_wt");
            $pcsULD     = $request->get("uld_pcs");
            $vrULD      = $request->get("uld_vr");
            $notocULD   = $request->get("uld_notoc");
            $notocDetailsULD = $request->get("uld_notoc_details");


            $airportIDs     = Airport::whereNotNull("iata")->pluck("id", "iata")->all();
            $airline        = $flight && $flight->flightNumber ? $flight->flightNumber->airline : null;
            $airlineIDs     = Airline::whereNotNull("iata")->pluck("id", "iata")->all();
            $airlineICAOIDs = Airline::whereNotNull("icao")->pluck("id", "icao")->all();

            // Create Info
            if (!$flightContainerInfo){
                $flightContainerInfo = new FlightContainerInfo();
                $flightContainerInfo->flight_id = $flightID;
                $flightContainerInfo->manual = true;
                $flightContainerInfo->save();
            }

            foreach ($rowULD as $i => $rowNumber) {
                if (!$rowNumber){
                    continue;
                }

                /*
                if (preg_match('/(\d+)\s*(\w+)?/', $pos, $posAndNum)) {
                    $rowNumber = isset($posAndNum[1]) ? $posAndNum[1] : null;
                    $position  = isset($posAndNum[2]) ? $posAndNum[2] : null;
                }
                else {
                    continue;
                }
                */

                $position = isset($posULD[$i]) && $posULD[$i] ? strtoupper(trim($posULD[$i])) : null;

                $containerDetails = [
                    "rowNumber"         => $rowNumber,
                    "position"          => $position,
                    "containerTypeName" => isset($typeULD[$i]) && $typeULD[$i] ? strtoupper(trim($typeULD[$i])) : null,
                    "containerNumber"   => isset($numULD[$i]) && $numULD[$i] ? trim($numULD[$i]) : null,

                    "owner"             => isset($ownerULD[$i]) && $ownerULD[$i] ? strtoupper(trim($ownerULD[$i])) : null,
                    "destinationAirport"=> isset($destULD[$i]) && $destULD[$i] ? strtoupper(trim($destULD[$i])) : null,

                    "weight"            => isset($wtULD[$i]) && $wtULD[$i] ? trim($wtULD[$i]) : null,
                    "loadedType"        => isset($contentULD[$i]) && $contentULD[$i] ? strtoupper(trim($contentULD[$i])) : null,
                    "vr"                => isset($vrULD[$i]) && ($vrULD[$i] || $vrULD[$i] == 0 || $vrULD[$i] == "0") ? trim($vrULD[$i]) : null,

                    "bag_pcs"           => isset($pcsULD[$i]) && $pcsULD[$i] ? $pcsULD[$i] : null,
                    "notoc"             => isset($notocULD[$i]) && $notocULD[$i] ? true : null,
                    "notoc_details"     => isset($notocDetailsULD[$i]) && $notocDetailsULD[$i] ? $notocDetailsULD[$i] : null,
                ];

                // Historic Data Com Containers
                FlightComContainer::insert([
                    "flight_com_id"         => $flightCom->id,
                    "flight_com_details_id" => $flightComDetails ? $flightComDetails->id : null,
                    "dest"                  => $containerDetails["destinationAirport"],
                    "pos"                   => $rowNumber.($position ? $position : ""),
                    "owner"                 => $containerDetails["owner"],
                    "content"               => $containerDetails["loadedType"],
                    "type"                  => $containerDetails["containerTypeName"],
                    "num"                   => $containerDetails["containerNumber"],
                    "pcs"                   => $containerDetails["bag_pcs"],
                    "wt"                    => $containerDetails["weight"],
                    "vr"                    => $containerDetails["vr"],
                    "notoc"                 => $containerDetails["notoc"],
                    "notoc_details"         => $containerDetails["notoc_details"],
                    "created_by"            => $authUserID,
                ]);

                // Find Container Type
                $containerType = ContainerType::findByName($containerDetails);

                // Find Container
                $container = Container::find($containerType, $containerDetails, $airline, $airlineIDs, $airlineICAOIDs);

                // Insert To Flights List
                $flightContainerIds[] = FlightContainer::insertContainerToFlight(null, $flightContainerInfo, $containerType, $container, $containerDetails, $airportIDs);

                $readyData[] = $containerDetails;
            }
        }

        // Remove Positions
        $removeUldPos = $request->get("remove_uld_pos");
        if ($removeUldPos){
            if (!isset($flightContainerInfo)){
                // ULD
                $flightContainerInfo = $this->getFlightContainerInfo($flightID);
            }

            foreach ($removeUldPos as $pos) {
                if (preg_match('/(\d+)\s*(\w+)?/', $pos, $posAndNum)) {
                    $rowNumber = isset($posAndNum[1]) ? $posAndNum[1] : null;
                    $position  = isset($posAndNum[2]) ? $posAndNum[2] : null;
                }
                else {
                    continue;
                }

                $container = FlightContainer::where("info_id", $flightContainerInfo->id)
                    ->where("row_number", $rowNumber)
                    ->where("position", $position)
                    ->get();

                $notes = "Container not found";
                if ($container && count($container)){
                    FlightContainer::where("info_id", $flightContainerInfo->id)
                        ->where("row_number", $rowNumber)
                        ->where("position", $position)
                        ->delete();

                    $notes = "POS removed";
                }

                FlightComContainer::insert([
                    "flight_com_id"         => $flightCom->id,
                    "flight_com_details_id" => $flightComDetails ? $flightComDetails->id : null,
                    "pos"                   => $pos,
                    "remove_uld"            => true,
                    "notes"                 => $notes,
                    "created_by"            => $authUserID,
                ]);
            }
        }

        return [$flight];
    }

    function getFlightContainerInfo($flightID){

        return FlightContainerInfo::where("manual", true)
            ->where("flight_id", $flightID)
            ->whereNull("deleted_at")
            ->first();

    }

    public function unseenMessages(Request $request){
        $authUserID = Auth::user()->id;
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs);

        return response()->json([
            "success"           => true,
            "unseen"            => $unseen,
            "counter"           => $counter,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }


    const PREV_MESSAGES_LENGTH = 30;
    const READ_MESSAGES_LENGTH = 10;

    public function getFlightMessages($userID, $flightID, $lastMessageID = null, $firstMessageID = null, $initialLoad = null){
        $flightCom = FlightCom::where("flight_id", $flightID)
            ->first();

        $created = $flightCom == null;

        $flightCom = FlightCom::getOrCreateFlightCom($flightID, $userID, generateRandomPassword(5), true);

        $details = FlightComDetails::with([
            "checkin", "checkin.aircraft",
            "pts",
            "containers",
            "pictures",
            "createdUser",
            "createdUser.location","createdUser.location.airport",
            "flightStaff", "flightStaff.user", "flightStaff.service",
        ])->where("flight_com_id", $flightCom->id);

        if ($firstMessageID){
            $details->where("id", "<", $firstMessageID);
        }
        else if ($lastMessageID){
            $details->where("id", ">", $lastMessageID);
        }

        $details = $details->orderBy("id")
            ->get();

        $uIndex = $this->isMessagesUnseen($userID, $flightCom, $details);
        /*
        foreach ($details as $i => &$each) {
            if ($this->isMessageUnseen($userID, $flightCom, $each)){
                if (!$uIndex){
                    $uIndex = $i;
                }
            }
        }
        */

        $totalMsgs = count($details);

        if ($uIndex){
            $k = $uIndex - self::READ_MESSAGES_LENGTH > 0 ? $uIndex - self::READ_MESSAGES_LENGTH : 0;
        }
        else {
            $length = $firstMessageID ? self::PREV_MESSAGES_LENGTH : self::READ_MESSAGES_LENGTH;
            $k = $totalMsgs - $length > 0 ? $totalMsgs - $length : 0;
        }

        $messages = [];
        for($i = $k; $i < $totalMsgs; $i++){
            $record = $this->addDetailsToRecord($details[$i]);

            $relations = $record->getRelations();
            unset($relations['createdUser']);

            if ($record->checkin && $record->checkin->aircraft){
                $record->checkin->ac = getAircraft($record->checkin->aircraft);
                unset($relations['checkin']['aircraft']);
            }


            $record->setRelations($relations);

            $messages[] = $record;
        }

        $pictures = [];
        if ($initialLoad){
            $pictures = FlightComPicture::where("flight_com_id", $flightCom->id)
                ->orderBy("id")
                ->get();
        }

        return [
            "key"       => $flightCom ? $flightCom->key     : null,
            "messages"  => $messages,
            "com"       => $flightCom ? $flightCom : null,
            "created"   => $created,
            "pictures"  => $pictures,
//            "com"       => count($messages) ? $com : null,
//            "key"       => $com && count($messages) ? $com->key     : null,
        ];
    }

    function addDetailsToRecord($flightComDetails){
        $flightComDetails->timestamp = baseDateFormat($flightComDetails->created_at, true);
        $flightComDetails->user = getUserName($flightComDetails->createdUser);
        $flightComDetails->base = getUserLocation($flightComDetails->createdUser);

        $flightComDetails->user_thumb = null;
        $userThumb = $flightComDetails->createdUser ? $flightComDetails->createdUser->thumb : null;
        if ($userThumb && File::exists(public_path("storage/users/{$userThumb}"))){
            $flightComDetails->user_thumb = $userThumb;
        }
        $flightComDetails->created_user = null;

        if ($flightComDetails->flightStaff){
            foreach ($flightComDetails->flightStaff as &$each) {
                $each->user_name = getUserName($each->user);
                $each->service_name = $each->service ? $each->service->abbr : "";

                $relations = $each->getRelations();
                unset($relations['user'], $relations["service"]);
                $each->setRelations($relations);
            }
        }

        return $flightComDetails;
    }

    function isMessageUnseen($userID, $flightCom, $flightComDetails){

        if ($flightComDetails->created_by == $userID){
            return false;
        }

        $userViewed = FlightComUserViews::where("flight_com_id", $flightCom->id)
            ->where("flight_com_details_id", $flightComDetails->id)
            ->where("user_id", $userID)
            ->first();

        if (!$userViewed) {

            FlightComUserViews::create([
                "flight_com_id"         => $flightCom->id,
                "flight_com_details_id" => $flightComDetails->id,
                "user_id"               => $userID,
                "created_at"            => date("Y-m-d H:i:s")
            ]);
        }

        return $userViewed == null;
    }


    function isMessagesUnseen($userID, $flightCom, $flightComDetails){

        $detailsIDs = $flightComDetails->pluck("id")->all();


        $allUserViews = FlightComUserViews::where("flight_com_id", $flightCom->id)
            ->whereIn("flight_com_details_id", $detailsIDs)
            ->where("user_id", $userID)
            ->get([
                "flight_com_details_id"
            ])
            ->pluck("flight_com_details_id")
            ->all();

        $insertionArray = [];
        $uIndex = null;
        $curDate = date("Y-m-d H:i:s");
        foreach ($flightComDetails as $i => $each) {
            if (!in_array($each->id, $allUserViews)){

                $insertionArray[] = [
                    "flight_com_id"         => $flightCom->id,
                    "flight_com_details_id" => $each->id,
                    "user_id"               => $userID,
                    "created_at"            => $curDate,
                ];

                if (!$uIndex){
                    $uIndex = $i;
                }
            }
        }

        if (count($insertionArray)){
            FlightComUserViews::insert($insertionArray);
        }

        return $uIndex;
    }


    function getUnseenFlightMessagesCount2($userID, $flightIDs){
        if (!$userID || !count($flightIDs)){
            return [[], []];
        }

        $coms = FlightCom::with(["details",
            "flight",
            "flight.flightNumber",
            "flight.flightNumber.airline",
            "flight.flightNumber.departureAirport",
            "flight.flightNumber.arrivalAirport"
        ])
            ->whereNull("deleted_at")
            ->whereIn("flight_id", $flightIDs)
            ->get();

        $unseen = $counter = [];
        if ($coms && count($coms)) {
            foreach ($coms as $each) {
                $detailIDs = $each->details->where("created_by", "!=", $userID)->pluck("id")->toArray();
                $seen = FlightComUserViews::where("user_id", $userID)
                    ->whereIn("flight_com_details_id", $detailIDs)
                    ->count();

                if ((int)(count($detailIDs) - $seen) > 0) {
                    $unseen[$each->flight_id] = [
                        "f"     => getFlightNumber($each->flight)." | " .getFlightSector($each->flight),
                        "c"     => (int)(count($detailIDs) - $seen),
                    ] ;
                }

                if ((int)(count($each->details))){
                    $counter[] = $each->flight_id;

                }
//                debug(count($detailIDs) . " SEEN:" . $seen);
            }
        }

        return [$unseen, $counter];
    }

    function getUnseenFlightMessagesCount($userID, $flightIDs){
        if (!$userID || !count($flightIDs)){
            return [[], []];
        }

        $details = FlightComDetails::join("flights__com", "flights__com.id", "=", "flights__com_details.flight_com_id")
            ->join("flights", "flights.id", "=", "flights__com.flight_id")
            ->whereNull("flights__com.deleted_at")
            ->whereIn("flights__com.flight_id", $flightIDs)
            ->get([
                "flights__com_details.id",
                "flights__com_details.created_by",
                "flights__com_details.flight_com_id",
                "flights__com.flight_id",
                "flights.flight_number_id",
            ]);

        $allDetailIDs = $detailsIDs = $detailsIDsByCom = [];
        $flightIDsByCom = $flightNumbersIDsByCom =  $flightNumbersIDs = [];
        foreach ($details as $each) {

            $flightIDsByCom[$each->flight_com_id] = $each->flight_id;
            $flightNumbersIDsByCom[$each->flight_com_id] = $each->flight_number_id;


            if (!isset($detailsIDsByCom[$each->flight_com_id])){
                $detailsIDsByCom[$each->flight_com_id] = [];

                $allDetailIDs[$each->flight_com_id] = [];
            }
            // Add Details
            $allDetailIDs[$each->flight_com_id] = $each->id;

            if ($each->created_by != $userID){
                $detailsIDsByCom[$each->flight_com_id][] = $each;
                $detailsIDs[] = $each->id;
            }

            if (!in_array($each->flight_number_id, $flightNumbersIDs)){
                $flightNumbersIDs[] = $each->flight_number_id;
            }
        }

        $flightNumbers = FlightNumber::with(["airline", "departureAirport", "arrivalAirport"])
            ->whereIn("id", $flightNumbersIDs)
            ->get();

        $fnByFlightNumberID = $sectorsByFlightNumberID = [];
        foreach ($flightNumbers as $each) {
            $sectorsByFlightNumberID[$each->id] = getSector($each);
            $fnByFlightNumberID[$each->id] = getFlightNumberFull($each);
        }

        $userViews = FlightComUserViews::where("user_id", $userID)
            ->whereIn("flight_com_details_id", $detailsIDs)
            ->groupBy("flight_com_id")
            ->get([
                "flight_com_id",
                DB::raw("COUNT(*) as counter"),
            ])
            ->pluck("counter", "flight_com_id")
            ->all();

        $unseen = $counter = [];
        foreach ($detailsIDsByCom as $comID => $details) {
            $counterDetailIDs = count($details);

            $flightID = $flightIDsByCom[$comID];
            $flightNumberID = $flightNumbersIDsByCom[$comID];

            $seen = isset($userViews[$comID]) ? $userViews[$comID] : 0;

            $fn = isset($fnByFlightNumberID[$flightNumberID]) ? $fnByFlightNumberID[$flightNumberID] : "";
            $sector = isset($sectorsByFlightNumberID[$flightNumberID]) ? $sectorsByFlightNumberID[$flightNumberID] : "";

            if ( ($counterDetailIDs - $seen) > 0) {
                $unseen[$flightID] = [
                    "f"     => $fn." | " .$sector,
                    "c"     => $counterDetailIDs - $seen,
                ] ;
            }

            if (isset($allDetailIDs[$comID]) && count($allDetailIDs[$comID])){
                $counter[] = $flightID;
            }
        }

        return [$unseen, $counter];
    }

    const MAX_EXECUTION_TIME = 10000,
        MAX_INPUT_TIME = 1000;

    public function uploadPicture(Request $request){

        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('max_input_time', self::MAX_INPUT_TIME);
        ini_set('upload_max_filesize', '64M');
        ini_set('post_max_size', '64M');
        ini_set('memory_limit', '256M');

        $authUserID = Auth::user()->id;
        $flightID = $request->get("flight_id");
        $key = $request->get('key');
        $message = $request->get('message');
        $flightIDs = $request->get("flight_ids") ? is_array($request->get("flight_ids")) ? $request->get("flight_ids") : explode(",", $request->get("flight_ids")) : null;

        $flightCom = FlightCom::where("flight_id", $flightID)
            ->first();

        if (!$flightCom){
            $flightCom = FlightCom::create([
                "created_by"    => $authUserID,
                "flight_id"     => $flightID,
                "key"           => generateRandomPassword(5),
            ]);
        }

        $flightComDetails = null;
//        debug($request->all());
//        debug($request->get("img"));
//        debug($request->file("img"));

        if ($request->file("img")) {
            // foreach ($request->file("img") as $i => $fileImage) {
            $fileImage = $request->file("img");
            if ($fileImage){

                $ext = $fileImage->getClientOriginalExtension();

                if (in_array(strtolower($ext), ["pdf", "jpg", "jpeg", "png", "gif"])) {

                    if (!$flightComDetails) {
                        $flightComDetails = FlightComDetails::create([
                            "flight_com_id" => $flightCom->id,
                            "created_by"    => $authUserID,
                            "message"       => $message ? $message : null,
                        ]);

                    }

                    $size = $fileImage->getSize() ? $fileImage->getSize() : $fileImage->getClientSize();
                    if ($size){
                        $size = round($size / (1024 * 1024), 1);
                    }

                    $flightComPicture = FlightComPicture::create([
                        "flight_com_id"         => $flightCom->id,
                        "flight_com_details_id" => $flightComDetails->id,
                        "notes"                 => $request->get("picture_notes"), //[$i],
                        "size"                  => $size,
                        "created_by"            => $authUserID,
                    ]);

                    $title = "f" . $flightID . "_u" . $authUserID . "_" . date("ymdHis"); // . "_" . $i;

                    saveImage($fileImage, strtolower($ext), $flightComPicture, "flight-com/pictures/", $title, false);
                }
            }
        }

        $flight = Flight::with(["message"])->find($flightID);

        $messages = $this->getFlightMessages($authUserID, $flightID, $request->get("last_message_id"));

        list($unseen, $counter) = $this->getUnseenFlightMessagesCount($authUserID, $flightIDs);

        $uld = []; //$this->getContainers($flightID);

        return response()->json([
            "success"           => true,
            "unseen"            => $unseen,
            "flight"            => $this->prepareFlightObj($flight),
            "counter"           => $counter,
            "uld"               => $uld,
            "com"               => count($messages["messages"]) ? $messages["com"] : null,
            "messages"          => $messages["messages"],
            "key"               => count($messages["messages"]) ? $messages["key"] : null,
        ], 200, ['Content-Type' => 'application/json; charset=UTF-8', 'charset' => 'utf-8'], JSON_UNESCAPED_UNICODE);
    }

    public function picture($flight_com_picture, Request $request)
    {
        $type = $request->get("type") ? $request->get("type") : "thumb";

        if ($flight_com_picture->{$type}){
            $filename = $flight_com_picture->{$type};
        }
        else {
            $filename = $flight_com_picture->file;
        }

        $url = storage_path("app/public/flight-com/pictures/$filename");

        if ($filename && File::exists($url)) {
            $mimeType = getMimeType($filename);
            $fileContents = File::get($url);

            if ($request->has('download')){
                return response()->download($url, $filename, array('Content-Type' => $mimeType));
            }
            else {
                return response()->make($fileContents, 200,array('Content-Type' => $mimeType));
            }
        }
    }

}
