<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Airline;
use App\Models\AirlineService;
use App\Models\Airport;
use App\Models\Service;
use Illuminate\Http\Request;

use App\Http\Requests;



class AirlineServiceController extends Controller
{
    protected $departureFields = [
        "departure_service",
        "dep_staff_req",
        "dep_staff_min",
        "dep_report_time",
        "dep_release_time",
    ];

    protected $arrivalFields = [
        "arrival_service",
        "arr_staff_req",
        "arr_staff_min",
        "arr_report_time",
        "arr_release_time",
    ];

    protected $turnaroundFields = [
        "turnaround_service",
        "turn_staff_req",
        "turn_staff_min",
        "turn_report_time",
        "turn_release_time",
    ];

    public function index()
    {
        $airlineId = \request()->get("airline_id");

        $sla = AirlineService::getAirlineServices($airlineId);

        list($slaByAirports, $airports) = AirlineService::getSlaListAndAirports($sla);

        $this->viewData = [
            'airline'			=> $airlineId ? Airline::find($airlineId) : null,
            'sla'				=> $sla,
            'slaByAirports'		=> $slaByAirports,
            'airports'		    => $airports,
        ];

        return view("airline-service.index", $this->viewData);
    }

    /**
     * @var AirlineService $airline_service
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit($airline_service)
    {
        $this->viewData = [
            'airlineService'	=> $airline_service,
        ];

        return view('airline-service.edit', $this->viewData);
    }

    /**
     * @param Request $request
     * @param $airline_service
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $airline_service)
    {
        if (!$airline_service){
            return redirect()->back();
        }

        debug($request->all());
//        return 1;

        //
        if ($this->startAndEndDateConflicts($airline_service, $request->get("start_date"), $request->get("end_date"))){
            debug(1);
            flash()->error("Effective/Discontinue dates conflicts with existing records. Please check it out");
            return redirect()->back();
        }
        else {
            if ($request->get("service_timings")) {
                debug(2);
                $this->makeServiceDefault($airline_service);
            }
            else {
                if ($this->checkIfServiceDefaults($airline_service->service)) {
                    debug(3);
                    // Set Defaults
                    $this->makeServiceDefault($airline_service);
                } else {
                    debug(4);
                    // Update Record
                    $this->updateFields($airline_service);
                }
            }
        }

        return redirect()->to("airline/{$airline_service->airline_id}?current_tab=sla");
    }

    function startAndEndDateConflicts($airline_service, $startDate = null, $endDate = null){

        if (!$startDate && !$endDate){
            return false;
        }

        // start date < cur date
        $startDateRemainsSame = $endDateRemainsSame = false;

        if ($startDate && $airline_service->start_date && (strtotime($airline_service->start_date) <= strtotime(date("Y-m-d")) ||
                strtotime($startDate) <= strtotime(date("Y-m-d")))){
            $startDate = $airline_service->start_date;
            $startDateRemainsSame = true;
        }

        if ($endDate && $airline_service->end_date && (strtotime($airline_service->end_date) <= strtotime(date("Y-m-d"))
                || strtotime($endDate) <= strtotime(date("Y-m-d")))){
            $endDate = $airline_service->end_date;
            $endDateRemainsSame = true;
        }

        if ($startDateRemainsSame && $endDateRemainsSame){
            return true;
        }

        $records = AirlineService::join("services", "services.id", "=", "airlines__services.service_id")
            ->orderBy("services.position")
            ->where("airline_id", $airline_service->airline_id)
            ->where("airport_id", $airline_service->airport_id)
            ->where("service_id", $airline_service->service_id)
            ->where("airlines__services.id", "!=", $airline_service->id)
            ->orderBy("end_date")
            ->get([
                "airlines__services.*"
            ]);

        $error = false;

        debug("START DATE: ".$startDate);
        debug("END DATE: ".$endDate);
        foreach ($records as $each) {
            if ($startDate && $each->start_date) {
                if (strtotime($startDate) >= strtotime($each->start_date) && strtotime($startDate) <= strtotime($each->end_date)) {
                    debug("ER-1");
                    $error = true;
                    break;
                }
            }

            if ($endDate && $each->end_date) {
                if (strtotime($endDate) >= strtotime($each->start_date) && strtotime($endDate) <= strtotime($each->end_date)) {
                    debug("ER-2");
                    $error = true;
                    break;
                }
            }

            if ($startDate && $each->end_date) {
                if (strtotime($startDate) == strtotime($each->end_date)) {
                    debug("ER-3");
                    $error = true;
                    break;
                }
            }

            if ($startDate && $endDate && $each->start_date && $each->end_date){
                if (strtotime($startDate) <= strtotime($each->start_date) && strtotime($endDate) >= strtotime($each->end_date)) {
                    debug("ER-4");
                    $error = true;
                    break;
                }
            }
        }

        if (!$error){
            if (!$airline_service->start_date && $startDate && strtotime($startDate) <= strtotime(date("Y-m-d"))){
                debug("START DATE IS <= TODAY: ".$startDate);
                return true;
            }
            if (!$airline_service->end_date && $endDate && strtotime($endDate) <= strtotime(date("Y-m-d"))){
                debug("END DATE IS <= TODAY: ".$endDate);
                return true;
            }
            $airline_service->start_date = $startDate;
            $airline_service->end_date = $endDate;
            $airline_service->save();
        }


        return $error;
    }

    function serviceStartAndEndDateConflicts($airlineID, $airportID, $serviceID, $startDate = null, $endDate = null){

        if (!$startDate && !$endDate){
            return false;
        }

        $records = AirlineService::where("airline_id", $airlineID)
            ->where("airport_id", $airportID)
            ->where("service_id", $serviceID)
            ->orderBy("end_date")
            ->get();

        $error = false;

        debug("START DATE: ".$startDate);
        debug("END DATE: ".$endDate);
        foreach ($records as $each) {
            if ($startDate && $each->start_date) {
                if (strtotime($startDate) >= strtotime($each->start_date) && strtotime($startDate) <= strtotime($each->end_date)) {
                    debug("ER-1");
                    $error = true;
                    break;
                }
            }

            if ($endDate && $each->end_date) {
                if (strtotime($endDate) >= strtotime($each->start_date) && strtotime($endDate) <= strtotime($each->end_date)) {
                    debug("ER-2");
                    $error = true;
                    break;
                }
            }

            if ($startDate && $each->end_date) {
                if (strtotime($startDate) == strtotime($each->end_date)) {
                    debug("ER-3");
                    $error = true;
                    break;
                }
            }

            if ($startDate && $endDate && $each->start_date && $each->end_date){
                if (strtotime($startDate) <= strtotime($each->start_date) && strtotime($endDate) >= strtotime($each->end_date)) {
                    debug("ER-4");
                    $error = true;
                    break;
                }
            }
        }

        if (!$error){
            if ($startDate && strtotime($startDate) <= strtotime(date("Y-m-d"))){
                debug("START DATE: ".$startDate);
                return true;
            }
            if ($endDate && strtotime($endDate) <= strtotime(date("Y-m-d"))){
                debug("END DATE: ".$endDate);
                return true;
            }
        }

        return $error;
    }


    public function makeServiceDefault(&$airline_service){

        $airline_service->service_timings = true;
        $airline_service->include_all_stations = null;
        $airline_service->effective_from = null;
        $airline_service->effective_to = null;

        foreach($this->departureFields as $field){
            $airline_service->{$field} = null;
        }
        foreach($this->arrivalFields as $field){
            $airline_service->{$field} = null;
        }
        foreach($this->turnaroundFields as $field){
            $airline_service->{$field} = null;
        }

        $airline_service->save();
    }

    public function checkIfServiceDefaults($service){
        $defaults = true;

        if ($service->include_all_stations != \request()->get("include_all_stations")){
            return false;
        }

        if ($service->effective_from != \request()->get("effective_from")){
            return false;
        }
        if ($service->effective_to != \request()->get("effective_to")){
            return false;
        }

        foreach($this->departureFields as $field){
            $value = \request()->get($field);
            if (($field == "dep_report_time" || $field == "dep_release_time") && \request()->get($field."_type")){
                $value *= -1;
            }
            if($service->{$field} != $value){
                $defaults = false;
                break;
            }
        }

        if ($defaults){
            foreach($this->arrivalFields as $field){
                $value = \request()->get($field);
                if (($field == "arr_report_time" || $field == "arr_release_time") && \request()->get($field."_type")){
                    $value *= -1;
                }
                if($service->{$field} != $value){
                    $defaults = false;
                    break;
                }
            }
        }
        if ($defaults){
            foreach($this->turnaroundFields as $field){
                $value = \request()->get($field);
                if (($field == "turn_report_time" || $field == "turn_release_time") && \request()->get($field."_type")){
                    $value *= -1;
                }
                if($service->{$field} != $value){
                    $defaults = false;
                    break;
                }
            }
        }
        return $defaults;
    }

    public function create()
    {
        $airlineId = \request()->get("airline_id");
        $airportId = \request()->get("airport_id");

        $assignedServices = AirlineService::join("services", "services.id", "=", "airlines__services.service_id")
            ->orderBy("services.position")
            ->where("airline_id", $airlineId)
            ->where("airport_id", $airportId)
            ->whereNull("end_date")
            ->pluck("service_id")
            ->toArray();

        $airports = Airport::getAirlinesHandlingAirports($airlineId, "iata", "id");

        if (!$airports || !count($airports)){
            $airports = Airport::listHandlingStations();
        }

        $this->viewData = [
            'airline'           => Airline::find($airlineId),
            'airport'           => $airportId ? Airport::find($airportId) : null,
            'airports'          => $airports,
            'services'          => Service::whereNotIn("id", $assignedServices)->pluck("abbr", "id")->all()
        ];

        return view("airline-service.create", $this->viewData);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Create New Record
        if (!\request()->get("airline_id") || !\request()->get("airport_id") || !\request()->get("service_id"))
        {
            flash()->error("Required information missing.");
            return redirect()->back();
        }

        $airline = \request()->get("airline_id");
        $services = \request()->get("service_id");
        $airports = is_array(\request()->get("airport_id")) ? \request()->get("airport_id") : [\request()->get("airport_id")];

        $startDate = \request()->get("start_date");
        $endDate = \request()->get("end_date");

        $startDateStr = $startDate ? strtotime($startDate) : null;
        $endDateStr = $endDate ? strtotime($endDate) : null;

        foreach ($airports as $airport) {
            foreach ($services as $service) {

                if (!$this->serviceStartAndEndDateConflicts($airline, $airport, $service, $request->get("start_date"), $request->get("end_date"))){
                    $airline_service = new AirlineService();
                    $airline_service->airline_id = $airline;
                    $airline_service->airport_id = $airport;
                    $airline_service->service_id = $service;
                    $airline_service->service_timings = true;
                    $airline_service->start_date = $startDate;
                    $airline_service->end_date = $endDate;
                    $airline_service->save();
                }
            }
        }

        //
        flash()->success("New service successfully created");

        return redirect()->to("airline/{$airline}?current_tab=sla");
    }

    /**
     * @param $airline_service
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($airline_service)
    {
        $airlineId = $airline_service->airline_id;

        // Audit Deleted
        auditDeleted($airline_service);
//        $airline_service->delete();

        return redirect()->to("airline/{$airlineId}?current_tab=sla");
    }

    function getAirlineAllStationsSLA($airline_service){
        return AirlineService::where("airline_id", $airline_service->airline_id)
            ->where("airport_id", $airline_service->airport_id)
            ->where("service_id", $airline_service->service_id);
    }

    public function updateFields(&$airline_service){

        $airline_service->service_timings = \request()->get("service_timings") ? true : null;

        $airline_service->include_all_stations = \request()->get("include_all_stations") ? true : null;

        $airline_service->start_date = \request()->get("start_date");
        $airline_service->end_date = \request()->get("end_date");

        $airline_service->effective_from = \request()->get("effective_to") ? \request()->get("effective_from") : null;
        $airline_service->effective_to = \request()->get("effective_from") ? \request()->get("effective_to") : null;

        // Departure
        $airline_service->departure_service = \request()->get("departure_service") ? true : null;
        $airline_service->dep_staff_req = \request()->get("dep_staff_req");
        $airline_service->dep_staff_min = \request()->get("dep_staff_min");

        $depReportTime = \request()->get("dep_report_time");
        if (\request()->get("dep_report_time_type") == "1"){
            $airline_service->dep_report_time = $depReportTime ? (-1) * $depReportTime : null;
        }
        else {
            $airline_service->dep_report_time = $depReportTime;
        }

        $depReleaseTime = \request()->get("dep_release_time");
        if (\request()->get("dep_release_time_type") == "1"){
            $airline_service->dep_release_time = $depReleaseTime ? (-1) * $depReleaseTime : null;
        }
        else {
            $airline_service->dep_release_time = $depReleaseTime;
        }

        // Arrival
        $airline_service->arrival_service = \request()->get("arrival_service") ? true : null;
        $airline_service->arr_staff_req = \request()->get("arr_staff_req");
        $airline_service->arr_staff_min = \request()->get("arr_staff_min");

        $arrReportTime = \request()->get("arr_report_time");
        if (\request()->get("arr_report_time_type") == "1"){
            $airline_service->arr_report_time = $arrReportTime ? (-1) * $arrReportTime : null;
        }
        else {
            $airline_service->arr_report_time = $arrReportTime;
        }

        $arrReleaseTime = \request()->get("arr_release_time");
        if (\request()->get("arr_release_time_type") == "1"){
            $airline_service->arr_release_time = $arrReleaseTime ? (-1) * $arrReleaseTime : null;
        }
        else {
            $airline_service->arr_release_time = $arrReleaseTime;
        }

        // Turnaround
        $airline_service->turnaround_service = \request()->get("turnaround_service") ? true : null;
        $airline_service->turn_staff_req = \request()->get("turn_staff_req");
        $airline_service->turn_staff_min = \request()->get("turn_staff_min");

        $arrReportTime = \request()->get("turn_report_time");
        if (\request()->get("turn_report_time_type") == "1"){
            $airline_service->turn_report_time = $arrReportTime ? (-1) * $arrReportTime : null;
        }
        else {
            $airline_service->turn_report_time = $arrReportTime;
        }

        $depReleaseTime = \request()->get("turn_release_time");
        if (\request()->get("turn_release_time_type") == "1"){
            $airline_service->turn_release_time = $depReleaseTime ? (-1) * $depReleaseTime : null;
        }
        else {
            $airline_service->turn_release_time = $depReleaseTime;
        }

        $airline_service->save();

    }



}
