<?php

namespace App\Http\Controllers;

use App\Models\Airline;
use App\Models\AirlineLibrary;

use App\Http\Controllers\Controller;
use App\Models\Department;
use App\Models\Library;
use App\Models\LibrarySign;
use App\Models\Location;
use App\Models\StaffAirline;
use App\Models\StaffStation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;


class AirlineLibraryController extends Controller
{
    public function index()
    {
        $handlingAirlines = StaffAirline::currentStaffAirlinesList(LIBRARY);

        AirlineLibrary::uploadedAirlineLibraries($handlingAirlines);

        $this->viewData = [
            "readAndSignPermission" => checkIfModulePermissionIsEnabled(\LIBRARY, "read_and_sign"),
            "departments"           => Department::where("has_library", true)->whereNull("deleted_at")->get(),
            "handlingAirlines"      => $handlingAirlines,
        ];

        return view("airline-library/index", $this->viewData);
    }

    public function create(Airline $airline)
    {
        $this->viewData = [
            "airline"       => $airline,
            "categories"   => ["" => "Select"] + AirlineLibrary::whereNotNull("category")->where("category", "!=", "")->groupBy("category")->pluck("category", "category")->all(),

            "departments"       => getDepartmentListByRole($this->authObject, true),
            "locations"         => [],
            "positions"         => [],
            "users"             => [],
        ];

        return view("airline-library/create", $this->viewData);
    }

    public function store(Request $request)
    {
        $airlineID = $request->get("airline_id");

        // create
        $airline_library = new AirlineLibrary();
        $airline_library->airline_id = $airlineID;
        $airline_library->created_by = Auth::user()->id;

        // update
        $this->updateFields($airline_library, $request);

        return redirect()->to("airline/{$airlineID}?current_tab=lbr");
    }

    public function edit(AirlineLibrary $airline_library)
    {
        $handlingAirports = StaffStation::currentStaffStationsList();

        $approvedLocations = Location::locationAirports(true, $handlingAirports);

        $positions = $users = $deptIDs = $positionIDs = [];
        if ($lastSign = $airline_library->lastSign){

            if (count($lastSign->departments)){
                $deptIDs = getLibrarySignDepartments($lastSign);
                $positions = getPositionsByDepartments($deptIDs);
            }

            $positionIDs = getLibrarySignPositions($lastSign);
            $locationIDs = getLibrarySignLocations($lastSign);
            $users = findUsersByDepartmentPositionLocation($locationIDs, $deptIDs, $positionIDs, null, true);
            $users = $users->pluck("name", "id")->all();
        }

        $this->viewData = [
            "airline"           => $airline_library->airline,
            "airlineLibrary"    => $airline_library,
            "categories"        => ["" => "Select"] + AirlineLibrary::whereNotNull("category")->where("category", "!=", "")->groupBy("category")->pluck("category", "category")->all(),

//            "subdepartments"    => $subdepartments,
            "lastSign"          => $lastSign,
            "positions"         => $positions,
            "users"             => $users,
            "departments"       => getDepartmentList(),
            "locations"         => $approvedLocations,
        ];

        return view("airline-library/edit", $this->viewData);
    }

    public function update(Request $request, AirlineLibrary $airline_library)
    {
        // update
        $airline_library->updated_by = Auth::user()->id;

        // update
        $this->updateFields($airline_library, $request);

        return redirect()->to("airline/{$airline_library->airline_id}?current_tab=lbr");
    }

    public function delete(AirlineLibrary $airline_library){

        if ($airline_library){
            $airline_library->deleted_at = date("Y-m-d H:i:s");
            $airline_library->updated_by = Auth::user()->id;
            $airline_library->save();
        }

        return redirect()->to("airline/{$airline_library->airline_id}?current_tab=lbr");
    }

    function updateFields(&$library, Request $request){
        $library->name = $request->get("name");
        $library->abbr = $request->get("abbr");
        $library->category = $request->get("new_category") ? $request->get("new_category") : $request->get("category");
        $library->issue_date = $request->get("issue_date") && $request->get("issue_date") != EMPTY_DATE_MASK ? $request->get("issue_date") : null;
        $library->effective_date = $request->get("effective_date") && $request->get("effective_date") != EMPTY_DATE_MASK ? $request->get("effective_date") : null;
        $library->edition_no = $request->get("edition_no");
        $library->details = $request->get("details");
        $library->save();

        try {
            LibrarySign::libraryCheckSignedDocuments($library, $request, AIRLINE_LIBRARY);
        }
        catch (\Exception $e){
            debug($e);
        }
    }

    public function showFile(AirlineLibrary $airline_library)
    {
        $filename = $airline_library->attachment_file;
        $url =storage_path("app/public/airlines/library/{$airline_library->airline_id}/$filename");

        if ($filename && File::exists($url)) {
            $mimeType = getMimeType($filename);
            $licenseExists = true;
            $fileContents = File::get($url);

            if (\request("view")){
                // Check and set
                LibrarySign::libraryReadAndSignCheck($airline_library, true, null, AIRLINE_LIBRARY);

                return response()->make($fileContents, 200,array('Content-Type' => $mimeType));
            }
            elseif (\request("download")){
                // Check and set
                LibrarySign::libraryReadAndSignCheck($airline_library, null, true, AIRLINE_LIBRARY);

                return response()->download($url, $filename, array('Content-Type' => $mimeType));
            }
        }
    }

    public function storeFile(Request $request, AirlineLibrary $airline_library){

        $url = "airline/{$airline_library->airline_id}?current_tab=lbr";

        if (!$request->hasFile("file")){
            return redirect()->to($url);
        }

        $mimeType = strtolower($request->file('file')->getClientMimeType());
        $file =  ['file' => $request->file('file')];
        $rulesArray = [
            "application/vnd.ms-excel.sheet.macroenabled.12",
            "application/pdf", "pdf", "image/jpeg", "image/jpg", "image/png", "image/gif",
            "application/vnd.ms-excel", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
            "application/vnd.openxmlformats-officedocument.spreadsheetml.template",
            "application/vnd.ms-excel.template.macroenabled.12",
            "application/vnd.ms-excel.addin.macroenabled.12",
            "application/vnd.ms-excel.sheet.binary.macroenabled.12"
        ];

        if (!in_array($mimeType, $rulesArray)){
            return redirect()->to($url)
                ->withInput()
                ->withErrors(["'Current File's extension is {$mimeType}. \nFile has to be in PDF format"]);
        }
        else {
            $fileName = $airline_library->id . '.' . $file['file']->getClientOriginalExtension();
            $storagePath = storage_path("app/public/airlines/library/{$airline_library->airline_id}");

            // Picture Save
            $file['file']->move($storagePath, $fileName);

            // Save in DB
            $airline_library->attachment_file = $fileName;
            $airline_library->save();

            // Set
            LibrarySign::libraryCheckSignedDocuments($airline_library, $request, AIRLINE_LIBRARY, true);

            return redirect()->to($url);
        }
    }

    public function deleteFile(AirlineLibrary $airline_library){
        $filename = $airline_library->attachment_file;

        $storageUrl = storage_path("app/public/airlines/library/{$airline_library->airline_id}/$filename");

        if (File::exists($storageUrl)){
            File::delete($storageUrl);
        }

        $airline_library->attachment_file = null;
        $airline_library->updated_by = Auth::user()->id;
        $airline_library->save();

        return redirect()->to("airline/{$airline_library->airline_id}?current_tab=lbr");
    }
}
