<?php namespace App\Http\Controllers;

use App\Classes\BindFlights\FlightPosition;
use App\Classes\Image\CropAvatar;
use App\Classes\Staff\Roster\Staff\Staff;
use App\Http\Requests;
use App\Http\Controllers\Controller;

use App\Models\Aircraft;
use App\Models\AircraftType;
use App\Models\Airline;
use App\Models\AirlineLibrary;
use App\Models\AirlineLicense;
use App\Models\AirlineService;
use App\Models\Airport;
use App\Models\Country;
use App\Models\Flight;
use App\Models\FlightNumber;
use App\Models\FlightSchedule;
use App\Models\Maintenance;
use App\Models\Service;
use App\Models\StaffAirline;
use App\Models\StaffService;
use App\Models\StaffStation;
use App\Repositories\Interfaces\IAirlineRepository;
use Illuminate\Foundation\Auth\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;


class AirlineController extends Controller {

	protected $properties = [
		"Founded" 					=> ["founded"],
		"Commenced" 				=> ["commenced"],
		"Operating bases" 			=> "operating_bases",
		"Hubs" 						=> "hubs",
		"Secondary hubs" 			=> "secondary_hubs",
		"Focus cities" 				=> "focus_cities",
		"Frequent-flyer program" 	=> "ffp",
		"Alliance" 					=> "alliance",
		"Subsidiaries" 				=> "subsidiaries",
		"Fleet size" 				=> "fleet_size",
		"Destinations" 				=> "destinations",
		"Slogan" 					=> "slogan",
		"Parent company"			=> "parent_company",
		"Headquarters" 				=> "headquarters",
		"Key people" 				=> "key_people",
		"Employees" 				=> "employees",
		"Website" 					=> "website",

	];

	/**
	 * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
	 */
	public function index()
	{
		$staffAirlines = StaffAirline::currentStaffAirlinesList(WITH_COUNTRY);

        $handlingAirlines = Airline::listHandlingAirlines();
        debug($handlingAirlines);

        $currentStaffAirlines = $airlineServices = [];
		foreach($staffAirlines as $i => $each){
		    if (!array_key_exists($each->airline_id, $handlingAirlines)){
		        continue;
            }

            $currentStaffAirlines[] = $each;

			foreach ($each->airline->airlineService as $airlineService) {
				if (!isset($airlineServices[$i])){
					$airlineServices[$i] = [];
				}
				$service = $airlineService->service->abbr;
				if (!in_array($service, $airlineServices[$i])){
					$airlineServices[$i][] = $service;
				}
			}
		}

		foreach($airlineServices as &$each){
			sort($each);
		}

		$this->viewData = [
			'handlingAirlines'	=> $currentStaffAirlines, //Airline::where("handling_airline", true)->get(),
			'airlineServices'	=> $airlineServices,
			'services'      	=> Service::get(),
//			'airlines'			=> Airline::whereNull("handling_airline")->get(),
            "auditData"         => isSprAdm() ? getServiceAudit() : [],
        ];

		return view("airline.index", $this->viewData);
	}

	public function online($airline)
	{
		$page = \request("page");

		$airline->handling_airline = true;
		$airline->save();

		flash()->success("Changes Successfully Saved");
		return redirect()->to("airline?page={$page}");
	}

	public function offline($airline)
	{
		$page = \request("page");

		$airline->handling_airline = null;
		$airline->save();

		flash()->success("Changes Successfully Saved");

		return redirect()->to(route("homepage")."#airline");

		//return redirect()->to("airline?page={$page}");
	}

	public function show(Request $request, $airline)
	{
        ini_set("max_execution_time", 120);
        ini_set('memory_limit', '512M');

		$airline = Airline::with([
			"airlineService",
			"airlineService.service",
			"airlineService.airport",
			"country",
			"aircraft",
			"aircraft.airline",
			"aircraft.type",
			"aircraftType",
			"delay",
			"delay.group",
			"delay.airline",
			"delay.emailNotification",
			"flightNumber",
			"flightNumber.airline",
			"flightNumber.departureAirport",
			"flightNumber.arrivalAirport",
		])
			->find($airline->id);

        $airlineServices = AirlineService::getAirlineServices($airline->id);

        list($slaByAirports, $airports) = AirlineService::getSlaListAndAirports($airlineServices);

		list($services, $selectedLicenses) = AirlineLicense::getServicesAndSelectedLicenses($airline);

        list($handlingAirlines, $hAirports) = getStaffHandlingAirports();

        $selectedAirports = $request->has("airport") ? $request->get("airport") : array_keys($hAirports);

        $dateFrom  = $request->get("dateFrom") && $request->get("dateFrom") != "____-__-__" ?  $request->get("dateFrom")  : date("Y-m-d");
        $dateTo    = $request->get("dateTo")   && $request->get("dateTo")   != "____-__-__" ?  $request->get("dateTo")    : date("Y-m-d");

		$flights  = null;
        if ($request->get("current_tab") == "flights"){

            $startDate = date("Y-m-d H:i:s", strtotime("-9 hours", strtotime($dateFrom." 00:00:00")));
            $endDate   = $dateTo." 23:59:59";

            $flights =  Flight::handlingFlightsRange($startDate, $endDate, $airline->id, $selectedAirports);

            // Data Array
            list($flights, $ssrList) = $this->getBindedFlights($flights, $dateFrom);
        }

//		$handlingAirports = [ALL => ALL] + StaffStation::currentStaffStationsList(IATA);

//		if (!count($handlingAirports)){
        $handlingAirports = StaffStation::currentStaffStationsList();
//		}
        $airlineApprovedStaff = StaffService::getAirlineApprovedStaff($airline->id, $handlingAirports);

		$excludeIDs = $airlineApprovedStaff->pluck("id");
		$staffList = \App\Models\User::getUsersByLocation($handlingAirports, false, false, $excludeIDs, true, true);

        debug($flights);

		$this->viewData = [
            "readAndSignPermission" => checkIfModulePermissionIsEnabled(AIRLINE_LIBRARY, "read_and_sign"),

            'dateFrom'                  => $dateFrom,
            'dateTo'                    => $dateTo,
            'flights'                   => $flights,
            'hAirports'                 => $hAirports,
            "fNumbers"             => FlightNumber::getArrayFlightNumbersWithSector(false, false, false, $airline->id, true),

            "categories"            => AirlineLibrary::where("airline_id", $airline->id)->whereNotNull("category")->where("category", "!=", "")->groupBy("category")->pluck("category")->all(),
            'picture' 				=> $airline->picture ? asset("storage/airlines/img/{$airline->picture}") : asset("/assets/img/default-airline-picture.png"),
			'airline'				=> $airline,
			'airlineApprovedStaff'	=> $airlineApprovedStaff,
			'properties'			=> $this->properties,
			'slaByAirports'     	=> $slaByAirports,
			'airports'     			=> $airports,
			'aircraft'				=> Aircraft::getAirlineAircraft($airline->id),
			'flightNumbers'			=> $airline->flightNumber,
			'delays'				=> $airline->delay,
			'services'          	=> $services,
			'selectedLicenses'  	=> $selectedLicenses,
			'types'				    => AircraftType::getAirlineTypes($airline->id),
			'schedules'     		=> FlightSchedule::scheduleByAirline($airline),
            'maintenance'           => Maintenance::getByAirline($airline),

            'library'				=> AirlineLibrary::where("airline_id", $airline->id)->whereNull("deleted_at")->orderBy("created_at", "DESC")->get(),
            'libraryDeleted'	    => AirlineLibrary::where("airline_id", $airline->id)->whereNotNull("deleted_at")->orderBy("created_at", "DESC")->get(),
            "libraryReadAndSign"    => AirlineLibrary::where("airline_id", $airline->id)->whereNull("deleted_at")->whereNotNull("read_and_sign")->orderBy("created_at", "DESC")->get(),

			'staffList'				=> $staffList,
            'isAdmin'               => isSprAdm(),
            "auditData"             => isSprAdm() ? getAirlineServiceAudit($airline->id) : [],
            "handlingAirports"      => Airport::listHandlingAirports(),
            "servicesList"          => Service::pluck("abbr", "id")->all(),
        ];

		return view('airline.show', $this->viewData);
	}

    function getBindedFlights($flights, $date = null, $includeSSR = null){
        $flightPosition = new FlightPosition($flights);
        $flightPosition->date = $date;
        $flightPosition->run();

        $ssrList = [];
        if ($includeSSR) {
            foreach ($flightPosition->lines as $k => $line) {

                foreach ($line->getFlights() as &$flightItem) {
                    if ($flightItem->ssr->count()) {
                        $itemSSR = [];
                        foreach ($flightItem->ssr as $flightSSR) {


                            $total = $flightSSR->pax_total ? $flightSSR->pax_total : $flightSSR->pax_c + $flightSSR->pax_y;
                            if ($total) {
                                $itemSSR[$flightSSR->ssr] = $total;

                                if (!in_array($flightSSR->ssr, $ssrList)) {
                                    $ssrList[] = $flightSSR->ssr;
                                }
                            }
                        }

                        $flightItem->ssr_ready = $itemSSR;
                    }
                }
            }
        }

        return [$flightPosition, $ssrList];
    }

	public function upload($airline)
	{
		if (!$airline) {
			return response()->isServerError();
		}

		new CropAvatar(\request("avatar_src"), \request("avatar_data"), $_FILES["avatar_file"], $airline, 'airlines/img', [750, 300], [250, 100]);

		return redirect()->back();
	}

	public function picture($airline){
	    return getAirlineLogoWithSize($airline);
    }

	public function delete($airline)
	{
		$picture = asset("storage/airlines/img/{$airline->picture}");
		$thumb =asset("storage/airlines/img/{$airline->thumb}");

		File::delete($picture);
		File::delete($thumb);

		$this->user->update($airline, [
			'picture' => null,
			'thumb' => null
		]);

		flash()->success("Profile Photo Successfully Deleted");

		return redirect()->back();
	}

	/**
	 * @param $airline
	 * @return \Illuminate\View\View
	 */
	public function edit($airline)
	{
		$this->viewData = [
			'currentPage'	=> \request("page"),
			'airline'		=> $airline,
			'countries'		=> Country::pluck("country", "id")->all(),
		];

		return view('airline.edit', $this->viewData);
	}

	/**
	 * @param $airline
	 * @return \Illuminate\Http\RedirectResponse
	 */
	public function update($airline)
	{
		$page = \request("page");

		$airline->airline = \request("airline");
		$airline->iata = \request("iata");
		$airline->icao = \request("icao");
		$airline->callsign = \request("callsign");
		$airline->country_id = \request("country");
		$airline->save();

		flash()->success("Changes Successfully Saved");
		return redirect()->to("airline?page={$page}");
	}


	public function addStaff(Request $request){

		debug($request->all());

		$airlineID = $request->get("airline_id");
		$authUserID = Auth::user()->id;

		if ($airlineID && $request->get("staff")) {
			foreach ($request->get("staff") as $each) {
				$staffAirline = StaffAirline::firstOrCreate([
					"user_id" 		=> $each,
					"airline_id" 	=> $airlineID,
				]);

				$staffAirline->updated_by = $authUserID;
				$staffAirline->save();
			}
		}

		return redirect()->to("airline/{$airlineID}?current_tab=approved_staff");
	}


	public function add()
	{
		return view("airline/add", $this->viewData);
	}

	public function save(Request $request)
	{
		if (!$request->get("iata") && !$request->get("icao")){
			return redirect()->to("airline/create");
		}

		$iata = $request->get("iata");
		$icao = $request->get("icao");

		$airline = null;

		if ($iata){
		    $iata = strtoupper($iata);

            $airlines = Airline::where("iata", $iata)
                                ->orderBy("deleted_at")
                                ->orderBy("handling_airline", "DESC")
                                ->orderBy("picture", "DESC")
                                ->orderBy("updated_at", "DESC")
								->get();

            if (count($airlines) == 1){
                // FOUND
                $airline = $airlines->first();
            }
            else {
                $flightNumbers = FlightNumber::join("airlines", "airlines.id", "=", "flights__numbers.airline_id")
                                            ->where("airlines.iata", $iata)
                                            ->groupBy("airlines.iata", "airline_id")
                                            ->get([
                                                "iata",
                                                "airline_id"
                                            ]);

                if ($flightNumbers->count() == 1){
                    // FOUND
                    $airline = Airline::find($flightNumbers->first()->airline_id);
                }
            }
		}

		if (!$airline) {
            if ($icao) {
                $icao = strtoupper($icao);
                $airlines = Airline::where("icao", $icao)
                    ->orderBy("handling_airline", "DESC")
                    ->orderBy("picture", "DESC")
                    ->orderBy("updated_at", "DESC")
                    ->get();

                if (count($airlines) == 1) {
                    // FOUND
                    $airline = $airlines->first();
                }
                else {
                    $flightNumbers = FlightNumber::join("airlines", "airlines.id", "=", "flights__numbers.airline_id")
                        ->where("airlines.icao", $icao)
                        ->groupBy("airlines.icao", "airline_id")
                        ->get([
                            "icao",
                            "airline_id"
                        ]);

                    if ($flightNumbers->count() == 1) {
                        // FOUND
                        $airline = Airline::find($flightNumbers->first()->airline_id);
                    }
                }
            }
        }

        if (!$airline){
            return redirect()->to("airline/create")
                        ->withInput($request->all());
        }

		if ($iata) {
			$airline->iata = $iata;
		}
		if ($icao) {
			$airline->icao = $icao;
		}

		$airline->handling_airline = true;
        $airline->deleted_at = null;
        if (Auth::user() != null){
            $airline->updated_by = Auth::user()->id;
        }
		$airline->save();

        StaffAirline::createCurrentStaffAirline($airline);

		return redirect()->to("airline");
	}

	public function create(Request $request)
	{
		$this->viewData = [
			'countries'		=> Country::pluck("country", "id")->all(),
		];

		return view("airline/create", $this->viewData);
	}

	/**
	 * Store a newly created resource in storage.
	 *
	 * @return Response
	 */
	public function store(Request $request)
	{
		$airline = null;
		if ($request->get("iata")){
			$airline = Airline::where("iata", $request->get("iata"))
				->first();
		}

		if (!$airline){
			if ($request->get("icao")) {
				$airline = Airline::where("icao", $request->get("icao"))
					->first();
			}
		}

		if (!$airline){
			$airline = new Airline();
		}

		$airline->handling_airline = $request->get("handling_airline");
		if ($request->get("iata")){
			$airline->iata = $request->get('iata');
		}
		if ($request->get("icao")){
			$airline->icao = $request->get('icao');
		}
		if ($request->get("airline")){
			$airline->airline = $request->get('airline');
		}
		if ($request->get("callsign")){
			$airline->callsign = $request->get('callsign');
		}
		if ($request->get("country_id")){
			$airline->country_id = $request->get('country_id');
		}

		$airline->save();

		return redirect()->to("airline");
	}




}
