<?php namespace App\Http\Controllers;

/**
 * Created by PhpStorm.
 * User: Dilovar Tursunov
 * Date: 21.08.14

 */

use App\Forms\AircraftForm;
use App\Models\AircraftCertificate;
use App\Models\AircraftEngine;
use App\Models\Airline;
use App\Models\Flight;
use App\Models\FlightNumber;
use App\Models\FlightSector;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

use Illuminate\Support\Facades\URL;
use App\Models\Maintenance;
use App\Models\Aircraft;
use App\Models\AircraftPart;
use App\Models\AircraftType;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Session\Session;

class AircraftController extends Controller
{
    protected $aircraftParameters = [

        "max_take_off"  => "Max Take-Off",
        "max_ramp"      => "Max Ramp",
        "max_landing"   => "Max Landing",
        "max_zero_fuel" => "Max Zero Fuel",
        "max_payload"   => "Max Payload",

        "usable_fuel"   => "Usable Fuel",
        "total_fuel"    => "Total Fuel",

        "basic_weight" => "Basic Weight",
        "basic_index"  => "Basic Index",
        "mzfw"         => "MZFW",
        "mtow"         => "MTOW",
        "mldw"         => "MLDW"
    ];

    protected $perPage = 50;

    /**
     * Render dashboard page
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $airlineIds = Airline::where("handling_airline", true)->pluck("id");

        $aircraft = Aircraft::with(["type", "airline", "engine"])
                                ->whereNull("deleted_at");

        if (ghaModule()){
            if ($request->get("airline_id")){
                $aircraft->where("airline_id", $request->get("airline_id"));
            }
            else {
                $aircraft->whereIn("airline_id", $airlineIds);
            }
        }

        if ($request->get("type_id")){
            $aircraft->where("aircraft_type_id", $request->get("type_id"));
        }

        $aircraft = $aircraft->orderBy("airline_id")->get();

        if (!$aircraft->count())
        {
            flash()->error("Not found any aircraft by given conditions");
        }

        return view("aircraft/index", [
            "certificates"  => AircraftCertificate::get(),
            "engines"       => AircraftEngine::get(),

            "aircraftType"  => AircraftType::find($request->get("type_id")),
            "aircraft"      => $aircraft,
            'isAdmin'       => Auth::user()->id == 44,
        ]);
    }


    public function types()
    {
        $types = AircraftType::whereNull("deleted_at")->get();
//                ->paginate($this->perPage);

        return view("aircraft/type", [
            "types"    => $types,
        ]);
    }


    public function showFlights($aircraft, Request $request)
    {
        $perPage = 20;

        $flights = Flight::with("message")
            ->where("aircraft_id", $aircraft->id)
            ->orderBy("std", "DESC")
            ->orderBy("ptd", "DESC")
            ->paginate($perPage, ['*'], "page", $request->get("page"));


        $this->viewData = [
            "perPage"       => $perPage,
            "aircraft"		=> $aircraft,
            "flights"		=> $flights,
        ];


        return view('aircraft/show-flights', $this->viewData);
    }

    /**
     * Add new A/C
     * @param Request $request
     * @return $this|\Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function add(Request $request)
    {
        $airlineId = \request("airline_id");

        $types = AircraftType::leftJoin("airlines", "airlines.id", "=", "aircraft__types.airline_id")
                                ->select([
                                    "aircraft__types.id AS id",
                                    DB::raw("CONCAT(IFNULL(s_airlines.iata, ''), '-', IFNULL(s_aircraft__types.iata_name, s_aircraft__types.icao)) AS fName"),
                                ])
                                ->where(function($sql){
                                    $sql->whereNotNull("aircraft__types.iata_name")
                                        ->orWhereNotNull("aircraft__types.icao");
                                });

        if ($airlineId){
            $types->where("airline_id", $airlineId);
        }

        $types = $types->orderBy("airline_id")
                        ->pluck("fName", "id")
                        ->all();


        $url   = URL::full();

        if ($this->isPostRequest())
        {
            $form = new AircraftForm();

            if ($form->isValidAircraft())
            {
                $aircraftName =  \request("name");

                if ($aircraftName && $aircraftName != ""){

                    $aircraft = Aircraft::select();


                    if (contains($aircraftName, "-")){
                        $aircraft->where("name", $aircraftName);
                    }
                    else {
                        $aircraft->where("mvt_name", $aircraftName);
                    }

                    if ($airlineId){
                        $aircraft->where("airline_id", $airlineId);
                    }

                    $aircraft = $aircraft->first();

                    if (!$aircraft){
                        $aircraft = new Aircraft();
                    }

                    $aircraft->mvt_name = contains($aircraftName, "-") ? str_replace("-", "", $aircraftName) : $aircraftName;
                    $aircraft->name =     contains($aircraftName, "-") ? $aircraftName : partitionAircraftCode($aircraftName);

                    $aircraft->airline_id       = $airlineId;

                    // New fields
                    $aircraft->hex_code         = \request("hex_code");
                    $aircraft->engine_id        = \request("engine_id");
//                    $aircraft->wide_board       = $request->get("wide_board") == "1";
                    $aircraft->flexi_config     = $request->get("flexi_config") == "1";
                    $aircraft->delivery_date    = \request("delivery_date");

                    $aircraft->aircraft_type_id = \request("aircraft_type_id");
                    $aircraft->config_c         = \request("config_c");
                    $aircraft->config_y         = \request("config_y");
                    $aircraft->msn              = \request("msn");
                    $aircraft->line_number      = \request("line_number");
                    $aircraft->first_flight     = \request("first_flight");
                    $aircraft->airframe_status  = \request("airframe_status");
                    $aircraft->created_by       = Auth::user()->id;
                    $aircraft->deleted_at = null;
                    $aircraft->save();

                    if ($request->get("current_airline_id")){
                        return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=ac");
                    }
                    if ($request->get("airline_id")){
                        return redirect()->to("airline/{$request->get('airline_id')}?current_tab=ac");
                    }

                    return redirect()->to("aircraft")
                        ->with("aircraft_message", [
                            "type" => MESSAGE_SUCCESS,
                            "text" => "New Aircraft was successfully added"
                        ]);
                }
            }

            return redirect()->to($url)
                ->withInput()
                ->withErrors($form->getErrors());
        }

        return view("aircraft/add", [
            "airlineId"     => $airlineId,
            "engines"       => ["" => "Select"] + AircraftEngine::pluck("name", "id")->all(),
            "airlines"      => ["" => "Select Airline"] + Airline::listHandlingAirlinesWithCode(),
            "types"         => ["" => "Select Type"] + $types
        ]);
    }

    /**
     * @param Request $request
     * @param Aircraft $aircraft
     * @return $this|\Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Request $request, Aircraft $aircraft)
    {
        $airlineModule = airlineModule();

        $airlineId = $request->get("airline_id") ? $request->get("airline_id") : $request->get("current_airline_id");

        $aircraftParts = AircraftPart::where("aircraft_id", "=", $aircraft->id)
                                        ->get();

        $types         = AircraftType::join("airlines", "airlines.id", "=", "aircraft__types.airline_id")
                                    ->where("iata_name", "!=", "");

        if ($airlineId){
            $types->where(function($sql) use ($airlineId){
                $sql->whereNull("airline_id")
                    ->orWhere("airline_id", $airlineId);
            });
        }

        $typeFieldName = $airlineModule ? "s_aircraft__types.icao" : "s_aircraft__types.iata_name";

        $types = $types->whereNotNull("iata_name")
                        ->get([
                            DB::raw("CONCAT(s_airlines.iata,'-',{$typeFieldName}) AS name"),
                            "aircraft__types.id as id"
                        ])
                        ->pluck("name", "id")
                        ->all();

        if ($this->isPostRequest() && $request->get("name"))
        {
            debug($request->all());

            $aircraftName = $request->get("name");
            $findAircraft = Aircraft::select();

            if (contains($aircraftName, "-")){
                $findAircraft->where("name", "=", $aircraftName);
            }
            else {
                $findAircraft->where("mvt_name", "=", $aircraftName);
            }


            $findAircraft = $findAircraft->where("airline_id", $airlineId)
                                         ->where("id", "!=", $aircraft->id)
                                         ->whereNull("deleted_at")
                                         ->first();

            if ($findAircraft){
                return redirect()->back()->withInput();
            }

            $valueBefore                = $aircraft->toArray();

            $userId = Auth::user()->id;
            $aircraft->mvt_name         = contains($aircraftName, "-") ? str_replace("-", "", $aircraftName) : $aircraftName;
            $aircraft->name             = contains($aircraftName, "-") ? $aircraftName : partitionAircraftCode($aircraftName);
            $aircraft->airline_id       = $request->get("airline_id");
            $aircraft->aircraft_type_id = $request->get("aircraft_type_id");
            $aircraft->description      = $request->get("description");

            // New fields
            $aircraft->hex_code         = \request("hex_code");
            $aircraft->engine_id        = \request("engine_id");
//            $aircraft->wide_board       = $request->get("wide_board") == "1";
            $aircraft->flexi_config     = $request->get("flexi_config") == "1";
            $aircraft->delivery_date    = \request("delivery_date");
            // End

            $aircraft->config_a         = $request->get("config_a");
            $aircraft->config_c         = $request->get("config_c");
            $aircraft->config_w         = $request->get("config_w");
            $aircraft->config_y         = $request->get("config_y");

            $aircraft->msn              = $request->get("msn");
            $aircraft->line_number      = $request->get("line_number");
            $aircraft->first_flight     = $request->get("first_flight") ? $request->get("first_flight") : null;
            $aircraft->airframe_status  = $request->get("airframe_status");


            $aircraft->email_recipients = $request->get('email_recipients');
            $aircraft->stop_emails      = $request->has('stop_emails') ? true : null;

            $aircraft->aisles                   = $request->get("aisles");
            $aircraft->noise_category           = $request->get("noise_category");
            $aircraft->decks                    = $request->get("decks");
            $aircraft->cargo_compartments       = $request->get("cargo_compartments");
            $aircraft->cargo_volume             = $request->get("cargo_volume");
            $aircraft->hourly_fuel_consumption  = $request->get("hourly_fuel_consumption");
            $aircraft->description              = $request->get("description");

            $aircraft->updated_by       = $userId;
            $aircraft->deleted_at = null;

            // Update seat fields
            $this->updateSeatFields($request, $aircraft, $request->get("aircraft_type_defaults"));

            $aircraft->save();

            flash()->success("Aircraft \"$aircraft->name\" was successfully saved");

            if ($request->get("current_airline_id")){
                return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=ac");
            }

            /*
            return redirect()->to("aircraft")
                ->with("aircraft_message", [
                    "type" => MESSAGE_SUCCESS,
                    "text" => "Aircraft \"$aircraft->name\" was successfully saved"
                ]);
            */
        }

        // Get Staff Handling AP and Airlines
        list($handlingAirlines, $handlingAirports) = getStaffHandlingAirports();

        $selectedAirports = $request->has("airport") ? $request->get("airport") : array_keys($handlingAirports);
        $selectedAirlines = $request->has("airline") ? $request->get("airline") : array_keys($handlingAirlines);

        $flights  = Flight::FlightsRangeConditional($selectedAirports, $selectedAirlines, ALL, $aircraft->id);
        $data = Aircraft::groupData($flights, ["flights"]);

        $this->viewData = [
            "airlines"      => ["" => "Select Airline"] + Airline::listHandlingAirlinesWithCode(),
            "engines"       => ["" => "Select"] + AircraftEngine::pluck("name", "id")->all(),

            "aircraft"      => $aircraft,
            "wide_board"    => aircraftWideBoard($aircraft),
            "seatConfig"    => Aircraft::getAircraftSeatConfig($aircraft),
            "aircraftParameters"          => $this->aircraftParameters,
            "aircraftCurrentParameters"   => Aircraft::getParameters($aircraft, $this->aircraftParameters),
            "aircraftParts" => $aircraftParts,
            "types"         => ["" => "Select Type"] + $types,

            // Flight Data TAB
            'data'              => $data,
            'airports'          => $handlingAirports,
            'dateYesterday'     => date('Y-m-d',strtotime(" - 1 days")),
            "flightNumbers"     => FlightNumber::getArrayFlightNumbersWithSector(false, false, false, null, true),
            'sectors'           => FlightSector::listSectors(),
            'sectorsByAirlines' => FlightSector::getSectorsByAirlines(),

            'periods'           => getPeriodsArray([6]),
            'monthNames'        => getMonthsArray(),
            'years'             => getYearsArray(),
        ];

        if ($request->get("show")){
            return view("aircraft/show", $this->viewData);
        }

        return view("aircraft/edit", $this->viewData);
    }

    public function show($aircraft)
    {
        return redirect()->to("aircraft/{$aircraft->id}/edit?s=".rand(100, 999)."&show=1");
    }

    public function updateSeatFields(Request $request, &$aircraft, $aircraftTypeDefaults = null)
    {
        $cabins = ["a_", "c_", "w_", "y_"];

        $fields = [
            "rowStart",
            "rowEnd",
        ];

        $arrayFields = [
            "columns_l",
            "columns_m",
            "columns_r",
            "emptyRows",
            "emptySeats",
            "extraSpace"
        ];

        $aircraftType = $aircraft->type;

        foreach ($cabins as $cabin) {
//            if ($request->get($cabin . "rowStart") && $request->get($cabin . "rowEnd")
//                && $request->get($cabin . "columns_l") && $request->get($cabin . "columns_r")
//            ) {
                $details = [];

                foreach ($fields as $field) {
                    if ($request->get($cabin . $field)) {
                        $details[$field] = trim($request->get($cabin . $field));
                    }
                }

                foreach ($arrayFields as $field) {
                    if ($request->get($cabin . $field)) {
                        $ready = explode(";", trim($request->get($cabin . $field)));

                        foreach ($ready as &$item) {
                            $item = trim($item);
                        }

                        $details[$field] = $ready;
                    }
                }

                $aircraft->{"config_" . $cabin . "details"} = count($details) ? serialize($details) : null;

                if ($aircraftTypeDefaults && $aircraftType) {
                    $aircraftType->{"config_" . $cabin . "details"} = count($details) ? serialize($details) : null;
                }
//            }
        }

        if ($aircraftType){

            if ($aircraftTypeDefaults) {
                $aircraftType->wide_board = $request->get("aisles") == 2;

                $aircraftType->config_a         = \request("config_a");
                $aircraftType->config_c         = \request("config_c");
                $aircraftType->config_w         = \request("config_w");
                $aircraftType->config_y         = \request("config_y");
            }

            $aircraftType->save();
        }
    }

    public function maintenanceList(){
        $maintenance = Maintenance::with(["aircraft"])
                                    ->orderBy('departure_time', "DESC")
                                    ->get();

        $this->viewData = [
            'maintenance'   => $maintenance,
        ];

        return view('aircraft/show-maintenance', $this->viewData);
    }

    public function maintenance(Aircraft $aircraft){
        $search = null;
        $maintenance = Maintenance::where('aircraft_id', $aircraft->id);

        if ($maintenance) {
            if ($search = \request("search")) {

                $maintenance->where(function ($sql) use ($search) {
                    $sql->where("id", $search)
                        ->orWhere("note", "LIKE", "%$search%")
                        ->orWhere("location_code", "LIKE", "%$search%")
                        ->orWhere("arrival_time", "LIKE", "%$search%")
                        ->orWhere("departure_time", "LIKE", "%$search%");
                });
            }
        }

        $maintenance = $maintenance->orderBy('departure_time', "DESC")
                                     ->get();

        return view('aircraft/maintenance', [
            'maintenance'   => $maintenance,
            "search"        => $search,
            'aircraft'      => $aircraft
        ]);
    }

    public function maintenanceAdd(Request $request, Aircraft $aircraft){
        debug($request->all());

//        $form = new AircraftForm();
//        if ($form->isValidMaintenanceRecord()) {
            $arrival_time = \request("arrival_time");
            $departure_time = \request("departure_time");
            $maintenanceAdd = new Maintenance;
            $maintenanceAdd->aircraft_id = $aircraft->id;
            $maintenanceAdd->user_id = Auth::user()->id;
            $maintenanceAdd->note = \request("note");
            $maintenanceAdd->location_code = \request("location_code");
            $maintenanceAdd->arrival_time = $arrival_time;
            $maintenanceAdd->departure_time = $departure_time;

            if (strtotime($maintenanceAdd->arrival_time) < strtotime($maintenanceAdd->departure_time)) {
                if (Maintenance::checkMaintenanceConflicts($aircraft->id, $arrival_time, $departure_time)) {
                    $maintenanceAdd->save();
                    $type = MESSAGE_SUCCESS;
                    $text = "Maintenance Record Successfully Added";

                    return redirect()->to("aircraft/maintenance/{$aircraft->id}")
                        ->with("message", [
                            "type" => $type,
                            "text" =>  $text
                        ]);
                }
                else {
                    $type = MESSAGE_DANGER;
                    $text = "Error! Selected Dates Conflicts With Other Records, Please Check Again.";
                }
            }
            else {
                $type = MESSAGE_DANGER;
                $text = "Error! Start Date Cannot Be Set After End Date. Please Check Again.";
            }

            return redirect()->to("aircraft/maintenance/{$aircraft->id}")
                ->withInput()
                ->with("message", [
                    "type" => $type,
                    "text" =>  $text
                ]);
//        }

    }


    public function maintenanceDelete(Maintenance $maintenance){
        if ($maintenance) {
            $aircraft = $maintenance->aircraft;
            // Audit Deleted
            auditDeleted($maintenance);
//            $maintenance->delete();

            return redirect()->to("aircraft/maintenance/{$aircraft->id}")
                ->with("message", [
                    "type" => MESSAGE_SUCCESS,
                    "text" => "Maintenance Record Successfully Deleted"
                ]);
        }

        return redirect()->back();
    }

    public function ajaxGetAircraftId(){
        $aircraftName = \request("aircraftName");
        $aircraft = Aircraft::where('name', $aircraftName)
                            ->whereNull("deleted_at")
                            ->first();

        if ($aircraft){
            return response()->json([
                'success'   => TRUE,
                'result'    => $aircraft->id
            ]);
        }
        return response()->json([
            'success'       => false,
            'result'        => false
        ]);
    }

    public function ajaxAircraft(){
        $data = [];
        $type = \request("type");
        switch ($type){
            case "type" :
                $data = AircraftType::pluck('name', 'name')->all();
                break;
        }
        return response()->json([
            "success"   => TRUE,
            'result'    => $data
        ]);
    }

    /**
     * @param Request $request
     * @param $aircraft
     * @return \Illuminate\Http\RedirectResponse
     */
    public function delete(Request $request, $aircraft)
    {
        $aircraft->deleted_at = date("Y-m-d H:i:s");
        $aircraft->save();

        if ($request->get("current_airline_id")){
            return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=ac");
        }

        return redirect()->to("aircraft")
            ->with("aircraft_message", [
                "type" => MESSAGE_SUCCESS,
                "text" => "Aircraft \"$aircraft->name\" was successfully deleted"
            ]);
    }


    /**
     * Edit A/C type
     * @param Request $request
     * @return $this|\Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function editType(Request $request)
    {
        $id = \request("id");
        $aircraftType = AircraftType::with("airline")->findOrFail($id);
        $url          = URL::full();

        if ($this->isPostRequest())
        {
            debug($request->all());

            $name  = \request("name") ? trim(\request("name")) : null;
            $iata  = \request("iata_name") ? trim(\request("iata_name")) : null;

            /*
            $findAircraftType = AircraftType::where(function($sql) use ($name, $iata){
                                                $sql->where("name", $name)
                                                    ->orWhere("name", $iata);
                                            })
                                            ->where(function($sql) use ($name, $iata){
                                                $sql->where("iata_name", $name)
                                                    ->orWhere("iata_name", $iata);
                                            })
                                            ->where(function($sql) use ($name, $iata){
                                                $sql->where("iata_name", $name)
                                                    ->orWhere("iata_name", $iata);
                                            })
                                            ->where("id", "!=", $id)
                                            ->first();
            if ($findAircraftType){
                return redirect()->to("aircraft/edit-type?id=".$id)
                    ->withInput();
            }
            */

            $aircraftType->name       = $name;
            $aircraftType->airline_id = $request->get("airline_id");

            $aircraftType->iata_name  = $iata;
            $aircraftType->wide_board       = $request->get("wide_board") == "1";
            $aircraftType->flexi_config     = $request->get("flexi_config") == "1";
            $aircraftType->icao             = $request->get("icao");
            $aircraftType->wake_category    = $request->get("wake_category");
            $aircraftType->details    = $request->get("details");

            $aircraftType->config_a    = $request->get("config_a");
            $aircraftType->config_c    = $request->get("config_c");
            $aircraftType->config_w    = $request->get("config_w");
            $aircraftType->config_y    = $request->get("config_y");

            $aircraftType->min_fcm    = $request->get("min_fcm");
            $aircraftType->policy_fcm = $request->get("policy_fcm");
            $aircraftType->min_ccm    = $request->get("min_ccm");
            $aircraftType->policy_ccm = $request->get("policy_ccm");
            $aircraftType->updated_by = Auth::user()->id;
            $aircraftType->save();

            if ($request->get("current_airline_id")){
                return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=actypes");
            }

            return redirect()->to("aircraft/types")
                ->with("aircraft_type_message", [
                    "type" => MESSAGE_SUCCESS,
                    "text" => "Aircraft type \"$aircraftType->name\" was successfully updated"
                ]);

        }

        return view("aircraft/edit-type", [
            "airlines"      => ["" => "Select Airline"] + Airline::listHandlingAirlinesWithCode(),
            "aircraftType"  => $aircraftType
        ]);
    }

    public function showType(AircraftType $aircraft_type)
    {
        return view("aircraft/show-type", [
            "aircraftType" => $aircraft_type
        ]);
    }

    /**
     * Add new aircraft type
     * @param Request $request
     * @return $this|\Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function addType(Request $request)
    {
        if ($this->isPostRequest())
        {
            $form = new AircraftForm();
            if ($form->isValidType())
            {
                $name  = $request->get("name") && trim($request->get("name")) ? trim($request->get("name")) : null;
                $iataName  = $request->get("iata_name") && trim($request->get("iata_name")) ? trim($request->get("iata_name")) : null;
                $icao  = $request->get("icao") && trim($request->get("icao")) ? trim($request->get("icao")) : null;

                $airlineID = $request->get("airline_id");

                $aircraftType = AircraftType::where("icao", $icao)
                                             ->where("iata_name", $iataName);

                if ($airlineID){
                    $aircraftType->where("airline_id", $airlineID);
                }

                $aircraftType = $aircraftType->first();

                if ($aircraftType){
                    return redirect()->to("aircraft/add-type")
                        ->withInput()
                        ->withErrors($form->getErrors());
                }

                $aircraftType                    = new AircraftType;
                $aircraftType->name              = $name;
                $aircraftType->airline_id        = $airlineID;

                $aircraftType->iata_name         = $iataName;
                $aircraftType->icao              = $request->get("icao");
                $aircraftType->wake_category     = $request->get("wake_category");
                $aircraftType->details           = $request->get("details");
                $aircraftType->min_fcm           = $request->get("min_fcm");
                $aircraftType->policy_fcm        = $request->get("policy_fcm");
                $aircraftType->min_ccm           = $request->get("min_ccm");
                $aircraftType->policy_ccm        = $request->get("policy_ccm");
                $aircraftType->created_by        = Auth::user()->id;
                $aircraftType->save();

                if ($request->get("current_airline_id")){
                    return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=actypes");
                }

                return redirect()->to("aircraft/types")
                    ->with("aircraft_type_message", [
                        "type" => MESSAGE_SUCCESS,
                        "text" => "New Aircraft type was successfully added"
                    ]);
            }

            return redirect()->to("aircraft/add-type")
                ->withInput()
                ->withErrors($form->getErrors());
        }

        return view("aircraft/add-type", [
            "airlines"      => ["" => "Select Airline"] + Airline::listHandlingAirlinesWithCode(),
            "airlineID"     => $request->get("current_airline_id")
        ]);
    }

    /**
     * Delete aircraft type
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deleteType(Request $request)
    {
        $aircraftType             = AircraftType::findOrFail(\request("id"));
        $aircraftType->deleted_at = date("Y-m-d H:i:s");
        $aircraftType->save();

        if ($request->get("current_airline_id")){
            return redirect()->to("airline/{$request->get('current_airline_id')}?current_tab=actypes");
        }

        return redirect()->to("aircraft/types")
            ->with("aircraft_type_message", [
                "type" => MESSAGE_SUCCESS,
                "text" => "Aircraft type \"$aircraftType->name\" was successfully deleted"
            ]);
    }

    /**
     * Get AJAX maintenance aircraft
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxGetMaintenance()
    {
        $id        = \request("maintenance_id");
        $validator = Validator::make(request()->all(), ["maintenance_id" => "required|exists:aircraft__maintenance,id"]);
        if ($validator->fails())
        {
            $messages = $validator->messages()
                ->all();
            return response()->json([
                "error"    => TRUE,
                "messages" => $messages,
                "title"    => "Validation Errors:"
            ]);
        }
        $maintenance = Maintenance::with(["aircraft", "user"])
                                ->find($id);

        return response()->json([
            "maintenance" => $maintenance,
            "success"     => TRUE
        ]);
    }

    /**
     * AJAX request to delete aircraft maintenance
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajaxDeleteMaintenance()
    {
        if (!\request()->ajax())
        {
            App::abort(404);
        }
        $aircraft                 = Aircraft::findOrFail(\request("aircraft_id"));
        $aircraft->maintenance_id = NULL;
        $aircraft->save();

        return response()->json([
            "success" => TRUE,
            "message" => "Maintenance was successfully removed. Now aircraft is available in fleet watch"
        ]);
    }


    /**
     * Add aircraft part
     * @return $this|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function addPart()
    {
        $aircraft = Aircraft::find(\request("aircraft_id"));
        if ($this->isPostRequest())
        {
            $aircraftPart                         = new AircraftPart;
            $aircraftPart->aircraft_id            = (\request("aircraft_id")) ? \request("aircraft_id") : NULL;
            $aircraftPart->part_type              = (\request("part_type")) ? \request("part_type") : NULL;
            $aircraftPart->part_number            = (\request("part_number")) ? \request("part_number") : NULL;
            $aircraftPart->serial_number          = (\request("serial_number")) ? \request("serial_number") : NULL;
            $aircraftPart->type_model             = (\request("type_model")) ? \request("type_model") : NULL;
            $aircraftPart->position               = (\request("position")) ? \request("position") : NULL;
            $aircraftPart->tsn                    = (\request("tsn")) ? \request("tsn") : NULL;
            $aircraftPart->csn                    = (\request("csn")) ? \request("csn") : NULL;
            $aircraftPart->installation_date      = (\request("installation_date")) ? \request("installation_date") : NULL;
            $aircraftPart->tslsv                  = (\request("tslsv")) ? \request("tslsv") : NULL;
            $aircraftPart->cslsv                  = (\request("cslsv")) ? \request("cslsv") : NULL;
            $aircraftPart->thrust_category        = (\request("thrust_category")) ? \request("thrust_category") : NULL;
            $aircraftPart->first_llp_remaining_fc = (\request("first_llp_remaining_fc")) ? \request("first_llp_remaining_fc") : NULL;
            $aircraftPart->last_overhaul_date     = (\request("last_overhaul_date")) ? \request("last_overhaul_date") : NULL;
            $aircraftPart->next_overhaul_date     = (\request("next_overhaul_date")) ? \request("next_overhaul_date") : NULL;
            $aircraftPart->created_by             = Auth::user()->id;
            $aircraftPart->save();

            return redirect()->to("aircraft/".\request("aircraft_id")."/edit")
                ->with("aircraft_message", [
                    "type" => MESSAGE_SUCCESS,
                    "text" => "Aircraft part was successfully added"
                ]);
        }

        return view("aircraft/add-part", [
            "types"    => [
                "APU"           => "APU",
                "ENGINES"       => "ENGINES",
                "LANDING GEARS" => "LANDING GEARS",
                "Other"         => "Other"
            ],
            "positions" => ["" => "N/A", "LH" => "LH", "RH" => "RH", "LMLG" => "LMLG", "RMLG" => "RMLG", "NLG" => "NLG"],
            "aircraft" => $aircraft
        ]);
    }

    /**
     * Edit aircraft part action
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function editPart()
    {
        $part     = AircraftPart::find(\request("id"),  ['id', 'aircraft_id', 'part_type','part_number','serial_number', 'type_model', 'position', 'tsn', 'csn', 'installation_date',
                                                            'tslsv', 'cslsv', 'thrust_category', 'first_llp_remaining_fc', 'last_overhaul_date', 'next_overhaul_date']);
        $aircraft = Aircraft::find($part->aircraft_id);

        if ($this->isPostRequest())
        {
            $userId = Auth::user()->id;
            $valueBefore = $part->toArray();

            $part->part_type              = (\request("part_type")) ? \request("part_type") : NULL;
            $part->part_number            = (\request("part_number")) ? \request("part_number") : NULL;
            $part->serial_number          = (\request("serial_number")) ? \request("serial_number") : NULL;
            $part->type_model             = (\request("type_model")) ? \request("type_model") : NULL;
            $part->position               = (\request("position")) ? \request("position") : NULL;
            $part->tsn                    = (\request("tsn")) ? \request("tsn") : NULL;
            $part->csn                    = (\request("csn")) ? \request("csn") : NULL;
            $part->installation_date      = (\request("installation_date")) ? \request("installation_date") : NULL;
            $part->tslsv                  = (\request("tslsv")) ? \request("tslsv") : NULL;
            $part->cslsv                  = (\request("cslsv")) ? \request("cslsv") : NULL;
            $part->thrust_category        = (\request("thrust_category")) ? \request("thrust_category") : NULL;
            $part->first_llp_remaining_fc = (\request("first_llp_remaining_fc")) ? \request("first_llp_remaining_fc") : NULL;
            $part->last_overhaul_date     = (\request("last_overhaul_date")) ? \request("last_overhaul_date") : NULL;
            $part->next_overhaul_date     = (\request("next_overhaul_date")) ? \request("next_overhaul_date") : NULL;
            $part->updated_by             = Auth::user()->id;
            $part->save();

            return redirect()->to("aircraft/".\request("aircraft_id")."/edit")
                ->with("aircraft_message", [
                    "type" => MESSAGE_SUCCESS,
                    "text" => "Aircraft part was successfully updated"
                ]);
        }
        return view("aircraft/edit-part", [
            "aircraft" => $aircraft,
            "part"     => $part,
            "positions" => ["" => "N/A", "LH" => "LH", "RH" => "RH", "LMLG" => "LMLG", "RMLG" => "RMLG", "NLG" => "NLG"],
            "types"    => [
                "APU"           => "APU",
                "ENGINES"       => "ENGINES",
                "LANDING GEARS" => "LANDING GEARS",
                "Other"         => "Other"
            ],
        ]);
    }

    /**
     * Delete aircraft part
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deletePart()
    {
        $aircraftPart = AircraftPart::find(\request("id"));
        $aircraftId   = $aircraftPart->aircraft_id;

        // Audit Deleted
        auditDeleted($aircraftPart);
//        $aircraftPart->delete();

        return redirect()->to("aircraft/".$aircraftId."/edit")
            ->with("aircraft_message", [
                "type" => MESSAGE_SUCCESS,
                "text" => "Aircraft part was successfully deleted"
            ]);
    }
}
