<?php namespace App\Http\Controllers;

use App\Classes\Image\CropAvatar;
use App\Http\Requests;
use App\Http\Controllers\Controller;

use App\Models\Agency;
use App\Models\AgencyCode;
use App\Models\City;
use App\Models\Country;
use App\Models\SalesChannel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\URL;

class AgencyController extends Controller
{
    /**
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $perPage = \request("perPage") ? \request("perPage") : 1000;

        $agencies = Agency::get();

        $this->viewData = [
            'perPage'				=> $perPage,
            'currentPage'			=> \request("page") ? \request("page") : 1,
            'agencies'			    => $agencies, // paginate($agencies, (\request("page") ? \request("page") : 1), $perPage)
        ];

        debug($this->viewData['agencies']);

        return view('agency/index', $this->viewData);
    }

    public function add()
    {
        $countries = getApprovedCountryList();

        $countryIDs = array_keys($countries);

        $citiesByCountry = City::getCitiesByCountry($countryIDs);

        $salesChannels = SalesChannel::orderBy("sales_channel_type_id")
                                    ->orderBy("channel")
                                    ->pluck("channel", "id")
                                    ->all();

        $this->viewData = [
            'salesChannels'     => ["" => "Select"] + $salesChannels,
            'countries'		    => ["" => "Select"] + $countries,
            'citiesByCountry'   => $citiesByCountry,
        ];

        return view('agency/add', $this->viewData);
    }


    public function create(Request $request){

        $attributes = $request->all();
        unset($attributes["_token"]);

        $agency = Agency::create($attributes);

        return redirect()->to("agency");
    }

    public function edit($agency)
    {
        $countries = getApprovedCountryList();

        $countryIDs = array_keys($countries);

        $citiesByCountry = City::getCitiesByCountry($countryIDs);

        $salesChannels = SalesChannel::orderBy("sales_channel_type_id")
            ->orderBy("channel")
            ->pluck("channel", "id")
            ->all();

        $this->viewData = [
            'agency'            => $agency,
            'salesChannels'     => ["" => "Select"] + $salesChannels,
            'countries'		    => ["" => "Select"] + $countries,
            'citiesByCountry'   => $citiesByCountry,
        ];

        return view('agency/edit', $this->viewData);
    }

    public function show($agency)
    {
        $this->viewData = [
            'agency'            => $agency,
        ];

        return view('agency/show', $this->viewData);
    }

    public function update(Request $request, $agency){

        $attributes = $request->all();
        unset($attributes["_token"]);

        $agency->update($attributes);

        debug($agency);

        return redirect()->to("agency");
    }

    /**
     * Display a listing of the resource.
     * @param Agency $agency
     * @return \Illuminate\View\View
     */
	public function editPrev($agency)
	{
        $salesFrom = \request("dateFrom");
        $salesTo   = \request("dateTo");

        if ($salesFrom && $salesTo){
            $conditions = [
                0 => ['condition' => 'whereBetween',
                      'field'     => 'sales_date',
                      'values'    => [$salesFrom, $salesTo]
                ]
            ];
        }
        else {
            $conditions = [
                0 => ['condition' => 'orderBy',
                      'field'     => 'sales_date',
                      'values'    => 'DESC']
                ,
                1 => ['condition' => 'limit',
                      'values'    => '20'
                ]
            ];
        }

        // Picture

        // Get Passenger Orders
        $passengerOrders = $agency->passengerOrdersWithCondition($conditions);

        $this->viewData = [
            'agency'            => $agency,
            'passengerOrders'   => $passengerOrders,
            'picture'           => $agency->picture ? asset("/assets/img/agencies/". $agency->picture) : asset("/assets/img/company_logo.png"),
        ];

        return view('agency/edit', $this->viewData);
	}


    /**
     * Randomly Assign Agencies To Channels( By Channels Share)
     * @return string
     */
    public function assignShare(){

        // Get All Channels
        $channels = SalesChannel::orderBy('estimate_share', "DESC")
        ->get([
            'id',
            'channel',
            'estimate_share'
        ]);

        // Check If Total Share Does Not Exceed 100 And Assign Current Share To 0
        $totalChannelShare = 0;
        foreach ($channels as $i => $each) {
            $totalChannelShare += $each->estimate_share;
            $channels[$i]->current_share = 0;
        }
        if ($totalChannelShare > 100)
            return "Total Channel Share Exceed 100. Please Change Shares And Then Proceed";

        // Get All Agencies With Total Revenue
        $agencies = Agency::select([
            'agency.id',
            DB::raw('COUNT(*) AS order_count'),
            DB::raw('SUM(fare) AS order_revenue')
        ])
            ->leftJoin('agency__codes', 'agency__codes.agency_id', '=', 'agency.id')
            ->leftJoin('passengers__orders', 'passengers__orders.agency_code_id', '=', 'agency__codes.id')
            ->groupBy('agency.id')
            ->orderBy('order_revenue', 'DESC')
            ->get();


        // Get Total Revenue By All Agencies
        $totalRevenue = 0;
        foreach ($agencies as $i => $each) {
            $totalRevenue += $each->order_revenue;
        }

        // Assign Share Of Each Agency
        foreach ($agencies as $i => $each) {
            $agencies[$i]->share = 100 * round($each->order_revenue / $totalRevenue, 4);
        }

        // Assign Channels For Each Agency
        foreach ($channels as $i => $channel) {
            $totalForThis = 0;
            foreach ($agencies as $j => $agency) {
                if ($channel->current_share >= $channel->estimate_share) {
                    break;
                }
                if ($agency->share + $channel->current_share <= $channel->estimate_share){
                    $totalForThis += $agency->share;
                    $channel->current_share += $agency->share;

                    unset($agency->share);
                    $agency->sales_channel_id = $channel->id;
                    $agency->save();
                    unset($agencies[$j]);
                }
            }
            if ($totalForThis > 0) {
                debug('Channel:' . $channel->channel . " Estimate Share:" . $channel->estimate_share . " Total Assigned Share:" . $totalForThis);
            }
        }

        // Assign Unidentified Channel (id = 19) For The Remaining Agencies
        if (count($agencies)){
            array_walk($agencies, function (&$each) {
                unset($each->share);
                $each->sales_channel_id = 19;
                $each->save();
                }
            );
        }

        return "Completed";
    }


    /**
     * Upload Picture
     * @param $agency
     * @return bool|string
     */
    public function uploadPhoto($agency){

        if (!$agency){
            return response()->isServerError();
        }

        $crop = new CropAvatar($_POST['avatar_src'], $_POST['avatar_data'], $_FILES['avatar_file'], $agency, 'agencies' );

        $response = array(
            'state'   => 200,
            'message' => $crop -> getMsg(),
            'result'  => asset($crop -> getResult())
        );

        return  json_encode($response);
    }

    /**
     * Delete Picture
     * @param $agency
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deletePhoto($agency){

        $picture = public_path("/assets/img/agencies/{$agency->picture}");
        $thumb = public_path("/assets/img/agencies/{$agency->thumb}");

        File::delete($picture);
        File::delete($thumb);

        $this->agency->update($agency, [
            'picture'    => null,
            'thumb'      => null
        ]);

        \Flash::success("Profile Photo Successfully Deleted");

        return redirect()->back();
    }

    /**
     * @param $agency
     */
	public function destroy($agency)
	{
		//
	}

}
