<?php

namespace App\Console\Commands;

use App\Mail\TemplateEmail;
use App\Models\Notification;
use App\Models\StaffLicense;
use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;

class NotificationSendEmailCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'notification:send_email';

    const MAX_EXECUTION_TIME = 120;

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Send notification emails';

    /**
     *
     */
    public function __construct()
    {
        parent::__construct();
    }

    protected $subject = "Staff License Expiry Notification";
    protected $title = "Email Notifications";

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        ini_set("max_execution_time", self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', '128M');

        $notifications = Notification::whereNull("deleted_at")
            ->whereNull("disabled")
            ->get();

        $periods = Notification::getPeriods(DATE);
        $periodsTitle = Notification::getPeriods(TITLE);

        debug($periods);

        $users = [];
        $userIDs = User::whereNull("deleted_at")
            ->where(function($sql){
                $sql->whereNull("users.resigned_date")
                    ->orWhere("users.resigned_date", EMPTY_DATE)
                    ->orWhere("users.resigned_date", ">", date("Y-m-d"));
            })
            ->pluck("id")
            ->all();


        foreach ($notifications as $notification) {

            $period = $periods[$notification->period];
            $pTitle = $periodsTitle[$notification->period];

            $staffLicenses = StaffLicense::with(["user", "user.reportsTo", "user.location", "user.location.airport", "license"])
                ->whereIn("user_id", $userIDs)
                ->whereNotNull("expiry_date");

            if ($notification->daily_reminder){
                $staffLicenses->where("expiry_date", "<=", $period);
            }
            else {
                $staffLicenses->where("expiry_date", $period);
            }

            $staffLicenses = $staffLicenses->orderBy("expiry_date")
                ->get();

            if (count($staffLicenses)){

                $personalEmails = $directorEmails = $list = [];
                foreach ($staffLicenses as $j => $item) {

                    if (!$item->user || !$item->license){
                        continue;
                    }

                    $list[] = $item;

                    $users[$item->user->id] = $item->user;
                    if ($item->user->reportsTo) {
                        $users[$item->user->reportsTo->id] = $item->user->reportsTo;
                    }


                    if (!isset($personalEmails[$item->user->id])){
                        $personalEmails[$item->user_id] = [];
                    }
                    $personalEmails[$item->user_id][] = $item;

                    //
                    if ($item->user->reports_to_user_id) {

                        if (!isset($directorEmails[$item->user->reports_to_user_id])) {
                            $directorEmails[$item->user->reports_to_user_id] = [];
                        }
                        $directorEmails[$item->user->reports_to_user_id][] = $item;
                    }

//                    if (!$notification->daily_reminder){
//                        debug(getUserName($item->user) ." | License: {$item->license->name} | Expiry: ". $item->expiry_date);
//                    }
                }

                // 1.Personal Emails
                foreach ($personalEmails as $userID => $sLicenses) {
                    if (isset($users[$userID]) && $users[$userID]){
                        $this->sendEmailByUser($notification, $users[$userID], $sLicenses, $pTitle);
                    }
                }

                // 2.Director Emails
                foreach ($directorEmails as $userID => $sLicenses) {
                    if (isset($users[$userID]) && $users[$userID]) {
                        $this->sendEmailByUser($notification, $users[$userID], $sLicenses, $pTitle, true);
                    }
                }

                // 3.Group Emails
                $this->sendEmailByGroup($notification, $list, $pTitle);
            }
        }

        return 0;
    }

    public function sendIndividualEmail($user, $license, $licenseExpiry, $reminderTitle){
        if (!$user || !$user->email || !trim($user->email) || !ends_with(trim($user->email), "@gsrm.de")){
            return;
        }

        Mail::to($user->email)
//            ->bcc("ejaz.1.khan@gmail.com")
//            ->bcc("dilovar88@mail.ru")
            ->send(new TemplateEmail(view("emails/notification/staff_license_expiration", [
                    "user"              => $user,
                    "reminderTitle"     => $reminderTitle,
                    "license"           => $license,
                    "expiryDate"        => $licenseExpiry,
                    "expired"           => strtotime($licenseExpiry) < strtotime(date("Y-m-d"))
                ])->render()
            ,  $this->subject." (".$reminderTitle.")"));
    }

    public function sendEmailByUser($notification, $recipientUser, $staffLicenses, $reminderTitle, $director = false){

        if (!$recipientUser || !$recipientUser->email || !trim($recipientUser->email) || !ends_with(trim($recipientUser->email), "@gsrm.de")){
            return;
        }

        //debug($notification->name. " | Director:{$director} | By user: {$recipientUser->id} | ".$recipientUser->last_name. " Counter: ".count($staffLicenses));
        Mail::to($recipientUser->email)
//            ->bcc("ejaz.1.khan@gmail.com")
//            ->bcc("dilovar88@mail.ru")
            ->send(new TemplateEmail(view("emails/notification/staff_license_expiration_by_user", [
//                    "user"              => $user,
//                    "license"           => $license,
//                    "expired"           => strtotime($licenseExpiry) < strtotime(date("Y-m-d"))
                    "user"              => !$director ? $recipientUser : null,
                    "today"             => date("Y-m-d"),
                    "reminderTitle"     => $reminderTitle,
                    "director"          => $director,
                    "staffLicenses"     => $staffLicenses,
                ])->render()
            , $this->subject." (".$reminderTitle.")"));
    }

    public function sendEmailByGroup($notification, $staffLicenses, $reminderTitle){

        $emailList = [];

        $emails = explode(";", $notification->emails);
        foreach ($emails as $email) {
            if (!$email || !trim($email) || !ends_with(trim($email), "@gsrm.de")){
                continue;
            }

            if (!in_array(trim($email), $emailList)){
                $emailList[] = trim($email);
            }
        }

//        debug($notification->name. " | By group: ".implode(",", $emailList)." | Counter: ".count($staffLicenses));

        if (!count($emailList)){
            return;
        }

        Mail::to($emailList)
//            ->bcc("ejaz.1.khan@gmail.com")
//            ->bcc("dilovar88@mail.ru")
            ->send(new TemplateEmail(view("emails/notification/staff_license_expiration_multiple", [
//                    "user"              => $user,
//                    "license"           => $license,
//                    "expired"           => strtotime($licenseExpiry) < strtotime(date("Y-m-d"))
                    "today"             => date("Y-m-d"),
                    "reminderTitle"     => $reminderTitle,
                    "staffLicenses"     => $staffLicenses,
                ])->render()
            , $this->subject." (".$reminderTitle.")"));
    }

}
