<?php
/**
 * Created by PhpStorm.
 * User: user
 * Date: 06.03.18

 */

namespace App\Classes\Sirena\XML;

use App\Classes\Parsing\Helpers\ParseHelper;
use App\Classes\Parsing\ParseOperations;
use App\Models\AgencyCode;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Country;
use App\Models\Currency;
use App\Models\CurrencyRate;
use App\Models\Flight;
use App\Models\FlightCabin;
use App\Models\FlightFareBasis;
use App\Models\FlightNumber;
use App\Models\FlightOrderStatus;
use App\Models\FlightRbd;
use App\Models\FlightType;
use App\Models\Passenger;
use App\Models\PassengerFlight;
use App\Models\PassengerOrder;
use App\Models\ReservationSystem;
use App\Models\SirenaCommand;
use App\Models\SirenaDetailCommand;
use App\Models\SirenaErrorCommand;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Nathanmac\Utilities\Parser\Facades\Parser;
use PhpSpec\Exception\Exception;

class SirenaXML
{
    protected $currentCommand;

    protected $revenueData = [];
    protected $processedFiles = [];
    protected $existingFiles = [];

    protected $currentIATA;
    protected $baseCurrency;
    protected $baseCurrencyID;

    /* @var ParseHelper $parseHelper */
    protected $parseHelper;

    /* @var $parseOperations */
    protected $parseOperations;

    // Vars
    protected $hubAirports = [];
    protected $countryISOs = [];
    protected $airportsIATA = [];
    protected $airlinesIATA = [];
    protected $currencies = [];
    protected $flightRBDs = [];
    protected $flightCabins = [];
    protected $flightFareBasis = [];
    protected $flightOrderStatus = [];
    protected $reservationSystems = [];
    protected $agencyTap = [];
    protected $agencyCode = [];

    const REFUND = "REFUND",
          SALE = "SALE";

    private $dateFrom = null;
    private $dateTo = null;

    public function __construct($dateFrom = null, $dateTo = null){
        $this->dateFrom = $dateFrom;
        $this->dateTo = $dateTo;
    }

    public function run(){
        // GLOBAL PARAMETERS

        $skipPreviousFileCheck = false;

        $maxFiles = 10000; // 10000
        ///

        if (!$skipPreviousFileCheck && $this->commandRunning()){
            return;
        }

        // Initiate
        $this->initiateCommand();

        // Get XMLs
        $files = $this->getXMLs();


        if (is_array($files)) {

            $parsedCounter = 0;

            $this->setVariables();

            foreach($files as $i => $file) {

                // Get Name
                $filename = $this->getFileName($file);

                if (!$filename){
                    continue;
                }

                // Skip existing
                if (!$skipPreviousFileCheck && in_array($filename, $this->existingFiles)){
                    continue;
                }

                $xml = file_get_contents($file);
                $parser = Parser::xml($xml);

                try {
                    $this->prepareArray($parser, $filename);
                }
                catch(\Exception $e) {
                    $this->reportError($e, $parser, "GLOBAL");
                }

                if ($parsedCounter++ == $maxFiles){
                    break;
                }
            }
            debug($this->types);

            debug($this->revenueData);

            //return;

            $this->parseRevenueData();

            // Complete
            $this->completeCommand();
        }

    }

    function setVariables(){

        $this->currentIATA = env(IATA_CODE);

        $this->existingFiles = array_unique(SirenaDetailCommand::pluck("filename")->all());

        $this->parseOperations = new ParseOperations();
        $this->parseOperations->messageType = PNL;
        $this->parseHelper = new ParseHelper($this->parseOperations);

        $this->baseCurrency = Currency::where("code", USD)->first();
        $this->baseCurrencyID = $this->baseCurrency ? $this->baseCurrency->id : null;

        $this->hubAirports = Airport::listHandlingStations("id", "iata");

        $this->countryISOs = Country::whereNotNull("iso_alpha_3")
            ->where("iso_alpha_3", "!=", "")
            ->pluck("id", "iso_alpha_3")
            ->all();

        $this->airportsIATA = Airport::whereNotNull("iata")
            ->where("iata", "!=", "")
            ->pluck("id", "iata")
            ->all();


        $this->airlinesIATA = Airline::whereNotNull("iata")
            ->where("iata", "!=", "")
            ->pluck("id", "iata")
            ->all();

        $this->currencies = Currency::whereNotNull("code")
            ->where("code", "!=", "")
            ->pluck("id", "code")
            ->all();

        $this->flightRBDs = FlightRbd::whereNotNull("rbd")
            ->where("rbd", "!=", "")
            ->pluck("id", "rbd")
            ->all();

        $this->flightCabins = FlightCabin::whereNotNull("cabin")
            ->where("cabin", "!=", "")
            ->pluck("id", "cabin")
            ->all();

        $this->flightFareBasis = FlightFareBasis::whereNotNull("fare_basis")
            ->where("fare_basis", "!=", "")
            ->pluck("id", "fare_basis")
            ->all();

        $this->flightOrderStatus = FlightOrderStatus::whereNotNull("order_status")
            ->where("order_status", "!=", "")
            ->pluck("id", "order_status")
            ->all();

        $this->reservationSystems = ReservationSystem::whereNotNull("abbr")
            ->where("abbr", "!=", "")
            ->pluck("id", "abbr")
            ->all();

        $this->agencyCode = AgencyCode::whereNotNull("code")
            ->where("code", "!=", "")
            ->pluck("id", "code")
            ->all();

        $this->agencyTap = AgencyCode::whereNotNull("tap")
            ->where("tap", "!=", "")
            ->pluck("id", "tap")
            ->all();

    }

    public function commandRunning(){

        $limit = date("Y-m-d H:i:s", strtotime("- 1 hours"));

        $prevCommand = SirenaCommand::where("status", PENDING)
                            ->where("created_at", ">", $limit)
                            ->whereNull("completed_at")
                            ->first();

        return $prevCommand != null;
    }

    public function initiateCommand(){
        $this->currentCommand = new SirenaCommand();
        $this->currentCommand->status = PENDING;
        $this->currentCommand->date_from = $this->dateFrom;
        $this->currentCommand->date_to = $this->dateTo;
        if (Auth::user()){
            $this->currentCommand->user_id = Auth::user()->id;
        }
        $this->currentCommand->save();
    }

    public function completeCommand(){
        $this->currentCommand->status = COMPLETED;
        $this->currentCommand->completed_at = date("Y-m-d H:i:s");
        $this->currentCommand->files = count($this->processedFiles);
        $this->currentCommand->save();
    }


    function getFileName($filename){
        preg_match('/.*\/(\w+).xml/', $filename, $match );

        return isset($match[1]) ? $match[1] : null;
    }

    function getXMLs(){
        /*
        $files = glob(storage_path("app/public/sirena/files/AAA*xml"));

        debug(count($files));

        return $files;
        */

        if ($this->dateFrom && $this->dateTo){

            $dateFrom = strtotime($this->dateFrom);
            $dateTo = strtotime($this->dateTo);

            $data = [];
            for ($i = $dateFrom; $i <= $dateTo; $i += 60 * 60 * 24){
                $date = date("Ymd", $i);
                $files = glob(storage_path("app/public/sirena/files/AAA{$date}*xml"));
                debug("SEARCH DATE: " .$date." FOUND: ".count($files). " files");
                if (count($files)){
                    $data = array_merge($data, $files);
                }
            }

            debug(count($data));

            return $data;

        }
        else if ($this->dateFrom){
            $date = date("Ymd", strtotime($this->dateFrom));
            $files = glob(storage_path("app/public/sirena/files/AAA{$date}*xml"));

            debug(count($files));

            return $files;
        }
        else {

            // AAA20180223_143618_341485579
            $date = date("Ymd");
            $date1days = date("Ymd", strtotime(" + 1 days"));

            $files = glob(storage_path("app/public/sirena/files/AAA{$date}*xml"));
            $files1days = glob(storage_path("app/public/sirena/files/AAA{$date1days}*xml"));

            return $files + $files1days;
        }
    }

    protected $types = [];

    function prepareArray($content, $filename){
        $ticketInfo = $content['TICKET'];

        // CHECK
        if (!array_key_exists($ticketInfo['TYPE'], $this->types)){
            $this->types[$ticketInfo['TYPE']] = 0;
        }
        $this->types[$ticketInfo['TYPE']]++;
        // END CHECK

        if (!in_array($ticketInfo['TYPE'], [ETICKET, TICKET])){
            return;
        }

        // REVIEW THIS OPTION
        if (!$ticketInfo['SEGMENTS'] && !isset($ticketInfo['SEGMENTS']['SEGMENT'])){
            return;
        }

        /*
        if (isset($ticketInfo['TAXES']['TAX'][0]) && is_array($ticketInfo['TAXES']['TAX'][0])){
            debug($ticketInfo);
            debug($filename);
            return;
        }
        */

        $passportCountryID = null;
        if($ticketInfo['CITIZENSHIP']){
            $passportCountryID = isset($this->countryISOs[$ticketInfo['CITIZENSHIP']]) ? $this->countryISOs[$ticketInfo['CITIZENSHIP']] : null;
        }

        // Multiple segments
        if (isset($ticketInfo['SEGMENTS']['SEGMENT'][0]) && is_array($ticketInfo['SEGMENTS']['SEGMENT'][0])){

            foreach ($ticketInfo['SEGMENTS']['SEGMENT'] as $each) {
                $this->prepareSegments($filename, $each, $ticketInfo, $passportCountryID);
            }
        }
        else {
            $this->prepareSegments($filename, $ticketInfo['SEGMENTS']['SEGMENT'], $ticketInfo, $passportCountryID);
        }

    }

    function prepareSegments($filename, $segmentData, $ticketInfo, $passportCountryID = null){

        $system = $ticketInfo['SYSTEM'];

        $order = $ticketInfo['DEAL'];
        $booked = isset($ticketInfo['SALE']['@date']) ? $ticketInfo['SALE'] :
                 (isset($ticketInfo['BOOK']['@date']) ? $ticketInfo['BOOK'] :
                     (isset($ticketInfo['SALE']['@agency']) ? $ticketInfo['SALE'] :
                         (isset($ticketInfo['BOOK']['@agency']) ? $ticketInfo['BOOK'] : null)));

        // ADT,CNN,INF
        $inf = $ticketInfo['PASSENGER_TYPE'] == "INF" ? true : null;

        $depDate = $this->fixDate($segmentData['FLYDATE']);
        $depTime = $this->fixTime($segmentData['FLYTIME']);

        $oFare = $ticketInfo['FARE'];
        if ($oFare && is_array($oFare)){
            $oFare = $oFare["#text"];
        }

        $obj = [
            'filename'              => $filename,
            'type'                  => $ticketInfo['OPTYPE'],
            'booked_date'           => $booked && isset($booked['@date']) && $booked['@date'] ? $this->fixDate($booked['@date']) : null,
            'booked_tch_code'       => $booked && isset($booked['@agency'])     ? $booked['@agency']    : null,
            'booked_iata_code'      => $booked && isset($booked['@stamp'])      ? $booked['@stamp']     : null,
            'booked_tap'            => $booked && isset($booked['@tap'])        ? $booked['@tap']       : null,
            'booked_opr_num'        => $booked && isset($booked['@opr_num'])    ? $booked['@opr_num']   : null,

            'last_name'             => $ticketInfo['SURNAME'],
            'first_name'            => $ticketInfo['NAME'],
            'gender'                => $ticketInfo['GENDER'],
            'dob'                   => $this->fixDate($ticketInfo['BIRTH_DATE']),
            'passport_num'          => $ticketInfo['PASS'],
            'passport_country_id'   => $passportCountryID,
            'sales_date'            => $this->fixDate($order['@date']),

            'tch_code'              => $order['@agency'],
            'iata_code'             => $order['@stamp'],
            'tap'                   => $order['@tap'],
            'opr_num'               => $order['@opr_num'],

            'pnr'                   => $ticketInfo['PNR_LAT'],
            'reservation'           => $system['@id'],
            // PASS ORDER FARE
            //  ORIGINAL CURRENCY
            'ofare'                 => $oFare,
            'ocurrency'             => $ticketInfo['CURRENCY'],
            // PAYMENT CURRENCY
            'fare'                  => $ticketInfo['OFARE'],
            'currency'              => $ticketInfo['OCURRENCY'],
            'rate'                  => $ticketInfo['NRATE'],

            // PASS FLIGHT FARE
            'sector_currency'       => $ticketInfo['NCURRENCY'],
            'sector_fare'           => $segmentData['NFARE'],

            'payment_form'         => $ticketInfo['FARECALC'],
            // 'airport_taxes'         => $ticketInfo[',
            // 'uab_taxes'             => $ticketInfo[',
            // 'fuel_taxes'            => $ticketInfo[',
            // 'tch_taxes'             => $ticketInfo[',
            // 'other_taxes'           => $ticketInfo[',
            'ticket_number'         => $segmentData['SEG_BSONUM'] ? trim($segmentData['SEG_BSONUM']) : null,
            'coupon'                => $segmentData['COUPON_NO'],
            'airline'               => $segmentData['CARRIER'],
            'flight_number'         => $segmentData['REIS'],
            'departure_airport'     => $this->stationsCheck($segmentData['PORT1CODE'] ? $segmentData['PORT1CODE'] : $segmentData['CITY1CODE']),
            'arrival_airport'       => $this->stationsCheck($segmentData['PORT2CODE'] ? $segmentData['PORT2CODE'] : $segmentData['CITY2CODE']),
            'departure_date'        => $depDate,
            'departure_time'        => $depTime,
            'departure_datetime'    => $depDate ? $depDate.($depTime ? " ".$depTime : "") : null,
            'order_status'          => $segmentData['IS_VOID'],
            'fare_code'             => $segmentData['BASICFARE'],
            'rbd'                   => $segmentData['CLASS'],
            'cabin'                 => $this->getCabin($segmentData['CLASS']),
            'inf'                   => $inf,
        ];

        /*
        if (strtotime($obj['sales_date']) > strtotime($obj['departure_date'])){
            debug($ticketInfo);
            debug($obj);
        }
        */

        $this->revenueData[] = $obj;
    }

    function stationsCheck($station){
        if (!$station || !trim($station)){
            return null;
        }

        $station = trim($station);

        switch($station){
            case "ОВБ":
            case "ТЛЧ":
                return "OVB";

            case "СУР":
                return "SGC";

            case "МОВ":
            case "MOW":
                return "DME";

            case "СПТ":
            case "ПЛК":
                return "LED";

            case "АЛА":
                return "ALA";

            case "БИШ":
                return "FRU";

           case "ДШБ":
                return "DYU";

           case "ХДТ":
                return "LBD";

           case "ХРГ":
                return "KHO";

        }

        return $station;
    }

    public function parseRevenueData(){

        if (!$this->revenueData){
            return;
        }

        foreach ($this->revenueData as $i => $obj) {

            try{
                $this->parseItem($obj);

                if (in_array($obj['filename'], $this->processedFiles)) {
                    continue;
                }

                // Add
                $this->processedFiles[] = $obj['filename'];

                // create
                $this->createFileParsed($obj['filename']);

            }
            catch(\Exception $e){
                $this->reportError($e, $obj);
            }
        }
    }


    function reportError(\Exception $e, $data, $type = null){
        $error = new SirenaErrorCommand();
        $error->command_id = $this->currentCommand ? $this->currentCommand->id : null;
        $error->filename = isset($data['filename']) ? $data['filename'] : null;
        $error->data = $data ? json_encode($data) : null;
        $error->type = $type;
        $error->code = $e->getCode();
        $error->file = $e->getFile();
        $error->line = $e->getLine();
        $error->message = $e->getMessage();
        $error->save();
    }

    function reportSimpleError($data, $filename, $type = null){
        $error = new SirenaErrorCommand();
        $error->command_id = $this->currentCommand->id;
        $error->filename = $filename;
        $error->data = json_encode($data);
        $error->type = $type;
        $error->save();
    }


    function createFileParsed($filename){
        if (in_array($filename, $this->existingFiles)){
            return;
        }

        $sDetails = new SirenaDetailCommand();
        $sDetails->command_id = $this->currentCommand->id;
        $sDetails->filename = $filename;
        $sDetails->save();
    }

    function getOrderStatusID($type, $departureDate = null){
        $status = "F";

        if ($type == self::REFUND){
            $status = "R";
        }
        else {

            if (!$departureDate || ($departureDate && strtotime($departureDate) > strtotime(date("Y-m-d")))){
                $status = "O";
            }
        }

        return $this->findFlightOrderStatusProcess($status);
    }

    function parseItem($obj){
        // If Sales Date is Not Set Then Skip
        if (!$obj['sales_date']){
            return;
        }

        preg_match_all('/\b(?!MR|MRS|MS|MIS|DR)\w+/', $obj['first_name'], $matches);
        list($names) = $matches;

        // Find Or Create Passenger
        $passenger = Passenger::where('last_name', $obj['last_name'])
                             ->where('first_name', $obj['first_name']);

        if ($obj['dob']){
            $passenger->where('dob', $obj['dob']);
        }

        $passenger = $passenger->first();

        if (!$passenger){
            $passenger = Passenger::create([
                "first_name"        => $obj['first_name'],
                "last_name"         => $obj['last_name'],
                "dob"               => $obj['dob'],
            ]);
        }

        if ($obj["passport_country_id"]){
            $passenger->passport_country_id = $obj['passport_country_id'];
        }

        if ($obj["passport_num"]){
            $passenger->passport_num = $obj['passport_num'];
        }

        $passenger->gender = $obj['gender'];
        $passenger->save();

        $flightRbdID = $this->findFlightRBDProcess($obj['rbd']);

        $flightCabinID  = $this->findFlightCabinProcess($obj['cabin']);

        $flightFareBasisID  = $this->findFlightFareBasisProcess($obj['fare_code']);

        $flightOrderStatusID = $this->getOrderStatusID($obj['type'], $obj['departure_date']);

        // Find Passenger Order
        $passengerOrder = PassengerOrder::where('ticket_number', $obj['ticket_number'])
                                        ->first();

        if (!$passengerOrder) {

            $passengerOrder = new PassengerOrder;
            $passengerOrder->ticket_number  = $obj['ticket_number'];

//            $passengerOrder->airport_taxes  = $obj['airport_taxes'];
//            $passengerOrder->uab_taxes      = $obj['uab_taxes'];
//            $passengerOrder->fuel_taxes     = $obj['fuel_taxes'];
//            $passengerOrder->tch_taxes      = $obj['tch_taxes'];
//            $passengerOrder->other_taxes    = $obj['other_taxes'];
//            $passengerOrder->created_by     = Auth::user()->id;
        }


        if ($obj['type'] == self::REFUND){
            $passengerOrder->actions_date   = $obj['sales_date'];
            $passengerOrder->sales_date     = $obj['booked_date'];
        }
        if ($obj['type'] == self::SALE){
            $passengerOrder->sales_date     = $obj['sales_date'];
        }

        $passengerOrder->payment_form   = $obj['payment_form'];

        if ($obj['reservation']){
            $passengerOrder->reservations_system_id = $this->findReservationSystemProcess($obj['reservation']);
        }

        if ($obj['iata_code'] || $obj['tap']){
            $passengerOrder->agency_code_id = $this->getAgencyCode($obj['iata_code'], $obj['tap'], $obj['tch_code'], $obj['opr_num']);
        }
        else if ($obj['booked_iata_code'] || $obj['booked_tap']){
            $passengerOrder->agency_code_id = $this->getAgencyCode($obj['booked_iata_code'], $obj['booked_tap'], $obj['booked_tch_code'], $obj['booked_opr_num']);
        }

        if ($obj['ofare']) {
            $passengerOrder->original_fare = $obj['ofare'];
        }

        if ($obj['ocurrency']){
            $passengerOrder->original_fare_currency_id = $this->findCurrencyProcess($obj['ocurrency']);;
        }

        if ($obj['fare']) {
            $passengerOrder->fare = $obj['fare'];
        }

        if ($obj['currency']){
            $passengerOrder->fare_currency_id = $this->findCurrencyProcess($obj['currency']);
        }

        if ($obj['rate']){
            $this->setCurrencyRate($passengerOrder, $obj['sales_date'], $obj['rate']);
        }

        if ($obj['pnr']) {
            $passengerOrder->pnr = $obj['pnr'];
        }
        $passengerOrder->save();

        // Added to skip Passenger Flights search
        if ($this->dateFrom || $this->dateTo){
            return;
        }


        $depAirportIATA = $obj['departure_airport'];
        $arrAirportIATA = $obj['arrival_airport'];
        $airlineIATA    = $obj['airline'];
        $flightNumber   = $obj['flight_number'];

        /*
        $departureAirportID = $this->findAirportProcess($depAirportIATA);
        $arrivalAirportID   = $this->findAirportProcess($arrAirportIATA);
        $bound = $this->getFlightBound($departureAirportID, $arrivalAirportID);
        $airlineID = $this->findAirlineProcess($airlineIATA);
        */

        $flight = $flightNumberObj = null;

        if ($obj['departure_date'] || $obj['departure_datetime']) {

            $depDate = $obj['departure_date'] ? $obj['departure_date'] : date("Y-m-d", strtotime($obj['departure_datetime']));

            $data = [
                "airline"           => $airlineIATA,
                "flightNumber"      => $flightNumber,
                "departureDay"      => date("d", strtotime($depDate)),
                "departureAirport"  => $depAirportIATA,
                "arrivalAirport"    => $arrAirportIATA,
                "std"               => $obj['departure_datetime'],
            ];

            // Without DB update
            $findFlight = $this->parseHelper->findFlightProcess($data, $depDate, true, null, false);

            $flight = $findFlight["flight"];
            $flightNumberObj = $findFlight["flightNumber"];
        }

        // DOUBLE SEARCH WITHOUT PASS-ORDER
        $passengerFlight = PassengerFlight::where("passenger_id", $passenger->id)
            ->where(function($sql) use ($flight, $flightNumberObj, $obj){
                if ($flight){
                    $sql->where("flight_id", $flight->id);

                    if ($flightNumberObj && ($obj['departure_datetime'] || $obj['departure_date'])) {
                        $sql->orWhere(function($sql2) use($flightNumberObj, $obj) {
                            $sql2->where('flight_number_id', $flightNumberObj->id);
                            if ($obj['departure_date']) {
                                $sql2->where("departure_date", $obj['departure_date']);
                            }
                            else {
                                $sql2->where("departure", $obj['departure_datetime']);
                            }
                        });
                    }
                }
                else {
                    if ($flightNumberObj && ($obj['departure_datetime'] || $obj['departure_date'])) {
                        $sql->where('flight_number_id', $flightNumberObj->id);

                        if ($obj['departure_date']) {
                            $sql->where("departure_date", $obj['departure_date']);
                        } else {
                            $sql->where("departure", $obj['departure_datetime']);
                        }
                    }
                }
            });

        $passengerFlight = $passengerFlight->orderBy("passenger_order_id", "DESC")
                                           ->first();
        // END

        if (!$passengerFlight) {
            // Insert Passenger Flight
            $passengerFlight = PassengerFlight::create([
                'passenger_id'              => $passenger       ? $passenger->id        : null,
            ]);
        }

        if ($passengerOrder){
            $passengerFlight->passenger_order_id = $passengerOrder->id;
        }
        if ($flight){
            $passengerFlight->flight_id          = $flight->id;
        }
        if ($flightNumberObj){
            $passengerFlight->flight_number_id   = $flightNumberObj->id;
        }

        $passengerFlight->flight_fare_basis_id   = $flightFareBasisID;
        $passengerFlight->flight_rbd_id          = $flightRbdID;
        $passengerFlight->flight_cabin_id        = $flightCabinID;
        $passengerFlight->flight_order_status_id = $flightOrderStatusID;
        $passengerFlight->departure              = $obj['departure_datetime'];
        $passengerFlight->departure_date         = $obj['departure_date'];
        $passengerFlight->coupon                 = $obj['coupon'];
        $passengerFlight->is_inf                 = $obj['inf'];

        $originalFare = $obj['sector_fare'];
        $originalCurrencyID = $this->findCurrencyProcess($obj['sector_currency']);

        $passengerFlight->original_fare = $originalFare;
        $passengerFlight->original_fare_currency_id = $originalCurrencyID;

        // Convert To Base Currency
        $passengerFlight->fare = $this->convertFare($obj['sales_date'], $originalFare, $originalCurrencyID);
        $passengerFlight->fare_currency_id = $this->baseCurrencyID;
        ///

        $passengerFlight->save();

        // Check If Order Has Many Flights,
        // If So Divide Order / flights count
        /*
        if ($passengerOrder->passengerFlight->count() > 1){
            foreach ($passengerOrder->passengerFlight as $each) {
                $each->fare = $passengerOrder->fare / $passengerOrder->passengerFlight->count();
                $each->updated_by = Auth::user()->id;
                $each->save();
            }
        }
        */
    }

    function findAgencyCodeProcess($iataCode, $tap, $tchCode, $oprNum, $autoCreate = true){
        if (!$iataCode){
            return null;
        }

        if (array_key_exists($iataCode, $this->agencyCode)){
            return $this->agencyCode[$iataCode];
        }

        if (!$autoCreate){
            return null;
        }

        $obj = new AgencyCode();
        $obj->code = $iataCode;
        $obj->tap = $tap;
        $obj->tch_code = $tchCode;
        $obj->opr_num = $oprNum;
        $obj->save();

        // Add to array
        $this->agencyCode[$iataCode] = $obj->id;

        return $obj->id;
    }

    function findAgencyTapProcess($tap, $tchCode, $oprNum, $autoCreate = true){
        if (!$tap){
            return null;
        }

        if (array_key_exists($tap, $this->agencyTap)){
            return $this->agencyTap[$tap];
        }

        if (!$autoCreate){
            return null;
        }

        $obj = new AgencyCode();
        $obj->tap = $tap;
        $obj->tch_code = $tchCode;
        $obj->opr_num = $oprNum;
        $obj->save();

        // Add to array
        $this->agencyTap[$tap] = $obj->id;

        return $obj->id;
    }

    function findReservationSystemProcess($name){
        if (!$name){
            return null;
        }

        if (array_key_exists($name, $this->reservationSystems)){
            return $this->reservationSystems[$name];
        }

        $obj = new ReservationSystem();
        $obj->abbr = $name;
        $obj->save();

        // Add to array
        $this->reservationSystems[$name] = $obj->id;

        return $obj->id;
    }

     function findFlightOrderStatusProcess($name){
        if (!$name){
             return null;
        }

        $name = ($name == 'A SZ:ETL') ? 'A' : $name;


        if (array_key_exists($name, $this->flightOrderStatus)){
            return $this->flightOrderStatus[$name];
        }

        $obj = new FlightOrderStatus();
        $obj->order_status = $name;
        $obj->save();

        // Add to array
        $this->flightOrderStatus[$name] = $obj->id;

        return $obj->id;
    }

    function findFlightFareBasisProcess($name){
        if (!$name){
            return null;
        }

        if (array_key_exists($name, $this->flightFareBasis)){
            return $this->flightFareBasis[$name];
        }

        $obj = new FlightFareBasis();
        $obj->fare_basis = $name;
        $obj->save();

        // Add to array
        $this->flightFareBasis[$name] = $obj->id;

        return $obj->id;
    }

    function findFlightCabinProcess($name){
        if (!$name){
            return null;
        }

        if (array_key_exists($name, $this->flightCabins)){
            return $this->flightCabins[$name];
        }

        $obj = new FlightCabin();
        $obj->cabin = $name;
        $obj->save();

        // Add to array
        $this->flightCabins[$name] = $obj->id;

        return $obj->id;
    }

    function findFlightRBDProcess($name){
        if (!$name){
            return null;
        }

        if (array_key_exists($name, $this->flightRBDs)){
            return $this->flightRBDs[$name];
        }

        $obj = new FlightRbd();
        $obj->rbd = $name;
        $obj->save();

        // Add to array
        $this->flightRBDs[$name] = $obj->id;

        return $obj->id;
    }

    function findCurrencyProcess($name){
        if (!$name){
            return null;
        }

        // Check and correct currency
        $name = $this->checkCurrency($name);

        if (array_key_exists($name, $this->currencies)){
            return $this->currencies[$name];
        }

        $obj = new Currency();
        $obj->code = $name;
        $obj->save();

        // Add to array
        $this->currencies[$name] = $obj->id;

        return $obj->id;
    }

    function checkCurrency($currency){
        if (!$currency || !trim($currency)){
            return null;
        }

        $currency = trim($currency);

        switch($currency){
            case "ДОЛ":
            case "NUC":
                return "USD";

            case "ЕВР":
                return "EUR";

            case "РУБ":
                return "RUB";
        }

        return $currency;
    }

    function findAirlineProcess($name){
        if (!$name){
            return null;
        }

        if (array_key_exists($name, $this->airlinesIATA)){
            return $this->airlinesIATA[$name];
        }

        $obj = new Airline();
        $obj->iata = $name;
        $obj->save();

        // Add to array
        $this->airlinesIATA[$name] = $obj->id;

        return $obj->id;
    }

    function findAirportProcess($name){
        if (!$name){
            return null;
        }

        if (array_key_exists($name, $this->airportsIATA)){
            return $this->airportsIATA[$name];
        }

        $obj = new Airport();
        $obj->iata = $name;
        $obj->save();

        // Add to array
        $this->airportsIATA[$name] = $obj->id;

        return $obj->id;
    }

    function getFlightBound($depAirportID, $arrAirportID){
        if (in_array($depAirportID, $this->hubAirports)){
            return 0;
        }
        else if (in_array($arrAirportID, $this->hubAirports)){
            return 1;
        }

        return null;
    }

    public function setCurrencyRate($passengerOrder, $date, $rate){
        if ($passengerOrder && $passengerOrder->original_fare_currency_id && $passengerOrder->fare_currency_id){

            $currencyRate = CurrencyRate::firstOrCreate(
                [
                    'from_currency_id'  => $passengerOrder->fare_currency_id,
                    'to_currency_id'    => $passengerOrder->original_fare_currency_id,
                    'date'              => $date,
                ]);

            $currencyRate->rate = $rate;
            $currencyRate->save();
        }
    }

    function convertFare($salesDate, $fare, $fromCurrencyID, $toCurrencyID = null){
        if (!$toCurrencyID){
            $toCurrencyID = $this->baseCurrencyID;
        }

        if ($fromCurrencyID == $toCurrencyID){
            return $fare;
        }

        $currencyRate = CurrencyRate::where("from_currency_id", $fromCurrencyID)
                            ->where("to_currency_id", $toCurrencyID)
                            ->where("date", "<=", $salesDate)
                            ->orderBy("date", "DESC")
                            ->first();

        if (!$currencyRate){
            return null;
        }

        return $fare * $currencyRate->rate;

    }

    function fixDate($date, $format = "dmY"){
        if (!$date){
            return null;
        }

        $original = \DateTime::createFromFormat($format, $date);

        return $original->format("Y-m-d");
    }

    function fixTime($time, $format = "Hi"){
        if (!$time){
            return null;
        }

        $original = \DateTime::createFromFormat($format, $time);

        return $original->format("H:i");
    }

    function getCabin($rbd){

        if ($rbd == null)
            $cabin = null;
        else{
            if ($rbd == 'C' || $rbd == 'J' || $rbd == 'I' || $rbd == 'D'){
                $cabin = 'Business';
            }
            else
                $cabin = 'Economy';
        }
        return $cabin;
    }

    function getAgencyCode($iataCode, $tap, $tchCode = null, $oprNum = null) {
        // echo '<meta http-equiv="Content-Type" content="text/html; charset=utf-8">';
        if (!$tap && !$iataCode){
            return null;
        }

        $agencyCode = null;
        if ($iataCode){
            $agencyCodeID = $this->findAgencyCodeProcess($iataCode, $tap, $tchCode, $oprNum);
        }
        elseif ($tap) {
            $agencyCodeID = $this->findAgencyTapProcess($tap, $tchCode, $oprNum);
        }

        return $agencyCodeID;
        /*
        $contains_cyrillic = (bool) preg_match('/[\p{Cyrillic}]/u', $tch);

        if ($contains_cyrillic) {
            return $this->findAgencyTchProcess($tch);
        }

        return $this->findAgencyCodeProcess($iataCode);
        */
    }

}
