<?php namespace App\Classes;
/**
 * Created by PhpStorm.

 */


use App\Models\Airport;
use App\Models\CabinCrew;
use App\Models\Flight;
use App\Models\FlightCrew;
use Illuminate\Support\Facades\DB;
use stdClass;

abstract class RosterSettings {
    // Crew Object Could Be FCM or CCM
    protected $crew;
    protected $crew_license_settings;

    protected $roster_log;

    protected $from;
    protected $to;
    protected $roster_type;
    protected $scheduled_keys = [];
    protected $flights_crew;

    protected $crew_policy_flight_numbers = [];

    protected $pseudo_container;
    protected $pseudo_lbd_set = false;
    protected $pseudo_dyu_set = false;
    protected $pseudo_crew_count = 0;
    protected $pseudo_id = 1000000;
    protected $different_hub_stay_period;
    protected $dhc_lbd = false;

    protected $ftl_7days;
    protected $ftl_28days;
    protected $ftl_12months;

    protected $fdp_7days;
    protected $fdp_28days;

    protected $flights_sector;
    protected $flight_type;
    protected $flight_many_type;
    protected $flights_scheduled;
    protected $lbd_crew_requirement;

    protected $current_law = TAJIKISTAN_CAA;

    protected $aircraft_min_crew = [];

    protected $location = [];

    const HOUR_RESTRICTION = 'HOUR_RESTRICTION';

    const LAST_7_DAYS = 'LAST_7_DAYS';
    const LAST_28_DAYS = 'LAST_28_DAYS';
    const LAST_12_MONTHS = 'LAST_12_MONTHS';

    const FDP_LAST_7_DAYS = 'FDP_LAST_7_DAYS';
    const FDP_LAST_28_DAYS = 'FDP_LAST_28_DAYS';

    const FO = "FO";
    const FO_JNR = "FO-JNR";

    const GMT = 0; // + seconds
    const LOCAL = 18000;
    const CREW_REPORT_TIME_SEC = 5400; // 1:30 in Seconds Before Flight's Departure
    const CREW_RELEASE_TIME_MIN = 0; //FDP + 0 min After Flight's Arrival

    const STANDBY_CREW_REPORT_TIME = 100; // 100 minutes
    const STANDBY_FLIGHT_WAIT_PERIOD = 30; // 30 mins from STD of Standby Flight
    const STANDBY_MAX_FDP_PERIOD = 220;   // 3:40 hrs. 1:40 Report Time + 2 HOURS Standby
    const WOCL_START = '02:00';
    const WOCL_END = '05:59';
    const WOCL_LOCAL = 18000; // + 5 Hours in Seconds

    const ONE_FLIGHT = 'ONE_FLIGHT';
    const ONE_FLIGHT_WITH_DIVERSION = 'ONE_FLIGHT_WITH_DIVERSION';

    const MANY_FLIGHTS = 'MANY_FLIGHTS';
    const MANY_FLIGHTS_WITH_DIVERSION = 'MANY_FLIGHTS_WITH_DIVERSION';

    const FLIGHT_WITH_REST = 'FLIGHT_WITH_REST';
    const FLIGHT_NO_REST = 'FLIGHT_NO_REST';
    const FLIGHT_NOT_EXISTS = 'FLIGHT_NOT_EXISTS';
    const FIRST_FLIGHT = 'FIRST_FLIGHT';

    const DYU_LBD_ROUNDTRIP = 'DYU_LBD_ROUNDTRIP';
    const DYU_LBD_SECTOR = 'DYU_LBD_SECTOR';
    const LBD_DYU_SECTOR = 'LBD_DYU_SECTOR';


    const OTHER_ROUNDTRIP = 'OTHER_ROUNDTRIP';
    const OTHER_SECTOR = 'OTHER_SECTOR';


    //-------------------------- ABSTRACT Functions List-----------------------------------

    abstract function Available_Crew();

    abstract function Pseudo_Crew_Create($pseudo_count, $settings);

    abstract function Pseudo_Crew_Check_Create($new_fdp, $settings);

    abstract public function AircraftMinCrewRequirement();

    //--------------------------------------------------------------------------------------



    /**
     * @param $keys_available
     * @param $new_fdp
     * @param $flight_date
     * @return array
     */
    public function Cmc_Check($keys_available, $new_fdp, $flight_date){
        $cmc_required_airports = ['FRA'];
        if (in_array($new_fdp[0]->flights[0]['arrival_airport'], $cmc_required_airports)){
            $valid_ids = [];
            foreach ($keys_available as $id){
                if (!is_null($this->crew[$id]->details['cmc_to']) && strtotime($this->crew[$id]->details['cmc_to']) > strtotime($flight_date))
                    $valid_ids[] = $id;
            }
            return $valid_ids;
        }
        else
            return $keys_available;

    }

    /**
     * Initial Setup of Designated Crew
     * @param $designated
     */
    public function Designated_Crew_Setup($designated){
        if (count($designated)) {
            foreach ($designated['flights'] as $i=>$flight) {
                if (isset($designated['crew'][$i])) {
                    foreach ($designated['crew'][$i] as $id) {
                        if (isset($this->crew[$id])) {
                            $this->crew[$id]->Set_Designated_Flights($flight);
                        }
                    }
                }
            }
        }
    }

    /**
     * @param $new_fdp
     * @return array
     */
    public function Designated_Crew($new_fdp){
        $designated_crew = [];
        foreach ($this->crew as $id => $each) {
            if (count($each->designated_flights)) {
                foreach($each->designated_flights as $flights) {
                    if ($flights[0] == $new_fdp[0]->flights[0]['id']) {
                        $designated_crew[] = $id;
                        break;
                    }
                }
            }
        }
        return $designated_crew;
    }


    /**
     * Get Last Location before Rostering
     * @param $userId
     * @param $from
     * @return mixed
     */
    public function GetCurrentLocation($userId, $from){
        $location = FlightCrew::select([
            "flights__crew.user_id",
            "arrAirport.iata",
            "flights.std",
            "flights.sta",
            "flights__numbers.flight_number",
            DB::raw("CONCAT(s_depAirport.iata, '-', s_arrAirport.iata) as sector")
        ])
            ->join("flights", "flights.id", "=", "flights__crew.flight_id")
            ->join("flights__numbers", "flights__numbers.id", "=", "flights.flight_number_id")
            ->join("airports AS depAirport", "depAirport.id", "=", "flights__numbers.departure_airport_id")
            ->join("airports AS arrAirport", "arrAirport.id", "=", "flights__numbers.arrival_airport_id")
            ->where("user_id", $userId)
            ->whereNotNull("flight_id")
            ->where(function($sql){
                $sql->whereNull("is_standby")
                    ->orWhere("is_standby", false);
            })
            ->where("flights.std", "<", $from." 00:00:00")
            ->orderBy("flights.std", "desc");

        return $location->first();
    }


    public function SetCrewDocumentationsConditions($id, $all_conditions, $from, $to){
        foreach ($all_conditions as $condition) {
            $eachCondition = $this->crew[$id]->details[$condition];
            if ($eachCondition && !is_null($eachCondition)) {
                if (strtotime($eachCondition) <= strtotime($to)) {
                    $period = [
                        'from' => strtotime($eachCondition) <= strtotime($from) ? $from : $eachCondition,
                        'to' => $to
                    ];

                    if (!in_array($period, $this->crew[$id]->unavailable_periods))
                    {
                        $this->crew[$id]->unavailable_periods[] = $period;
                    }
                }
            }
            /* Condition If Crew's Doc is Missing
            else {
                $this->crew[$id]->unavailable_periods[] = ['from' => $from, 'to' => $to];
            }
            */
        }
    }

    public function SetCrewHistoryConditions($user, $from, $to){
        $id = $user->id;
        if ($user->userHistory && count($user->userHistory)){
            foreach ($user->userHistory as $history){
                if ($history->condition_from <= $from && $history->condition_to >= $from){
                    $period = [
                        'from'  => strtotime($history->condition_from) <= strtotime($from) ? $from : $history->condition_from,
                        'to'    => strtotime($history->condition_to) > strtotime($to) ? $to : $history->condition_to,
                    ];

                    if (!in_array($period, $this->crew[$id]->unavailable_periods))
                    {
                        $this->crew[$id]->unavailable_periods[] = $period;
                    }
                }
                else if ($history->condition_from >= $from && $history->condition_from < $to){
                    $period = [
                        'from'  => $history->condition_from,
                        'to'    => strtotime($history->condition_to) > strtotime($to) ? $to : $history->condition_to
                    ];

                    if (!in_array($period, $this->crew[$id]->unavailable_periods))
                    {
                        $this->crew[$id]->unavailable_periods[] = $period;
                    }
                }
            }
        }
    }

    public function SetCrewRestrictions($user){
        $id = $user->id;
        if ($user->crewRestrictionRoutes && count($user->crewRestrictionRoutes)) {
            foreach ($user->crewRestrictionRoutes as $restriction) {
                if (!isset($this->crew[$id]->route_restrictions)){
                    $this->crew[$id]->route_restrictions = [];
                }
                $this->crew[$id]->route_restrictions[] = $restriction->flight_route_id;
            }
        }

        if ($user->crewRestrictionAircraftTypes && count($user->crewRestrictionAircraftTypes)) {
            foreach ($user->crewRestrictionAircraftTypes as $restriction) {
                if (!isset($this->crew[$id]->aircraft_types_restrictions)){
                    $this->crew[$id]->aircraft_types_restrictions = [];
                }
                $this->crew[$id]->aircraft_types_restrictions[] = $restriction->aircraft_type_id;
            }
        }

        if ($user->crewRestrictionPairings && count($user->crewRestrictionPairings)) {
            foreach ($user->crewRestrictionPairings as $restriction) {
                if (!isset($this->crew[$id]->pairing_restrictions)){
                    $this->crew[$id]->pairing_restrictions = [];
                }
                $this->crew[$id]->pairing_restrictions[] = $restriction->pair_user_id;
            }
        }
    }

    /**
     * If Flight Sector is DYU-LBD or LBD-DYU - std Date is saved.
     * @param $keys_final
     * @param $keys_dhc
     * @param $flight_date
     */
    public function Hub_Check($keys_final, $keys_dhc, $flight_date){
        if (in_array($this->flights_sector, [self::DYU_LBD_SECTOR, self::LBD_DYU_SECTOR])) {
            $keys_all = array_merge($keys_final, $keys_dhc);
            switch ($this->flights_sector) {
                case (self::DYU_LBD_SECTOR):
                    foreach ($keys_all as $id) {
                        if ($this->crew[$id]->base == $this->location[0]) {
                            $this->crew[$id]->different_hub_first_date = $flight_date;
                        }
                    }
                    break;
                case (self::LBD_DYU_SECTOR):
                    foreach ($keys_all as $id) {
                        if ($this->crew[$id]->base == $this->location[1]) {
                            $this->crew[$id]->different_hub_first_date = $flight_date;
                        }
                    }
                    break;
            }
        }
    }

    /**
     * If Crew is Not in Their Hub, Then Check if Stay Limitation Are NOT Crossed
     * @param $keys_available
     * @param $flight_date
     * @return array
     */
    public function Different_Hub_Stay_Limit($keys_available, $flight_date){
        if (in_array($this->flights_sector, [self::DYU_LBD_SECTOR, self::LBD_DYU_SECTOR])) {
            $priority1 = [];
            $priority2 = [];
            $hub_limit = $this->different_hub_stay_period;
            $date_limit = date('Y-m-d', strtotime("$flight_date - $hub_limit days"));
            switch ($this->flights_sector) {
                case (self::DYU_LBD_SECTOR):
                    foreach ($keys_available as $id) {
                        debug("DIF HUB Fir Date: ".$this->crew[$id]->different_hub_first_date."<= Date Limit".$date_limit);
                        if ($this->crew[$id]->base == $this->location[1] && $this->crew[$id]->current_location == $this->location[0] &&
                            strtotime($this->crew[$id]->different_hub_first_date) <= strtotime($date_limit)){
                            $priority1[] = $id;
                        }
                        else
                            $priority2[] = $id;
                    }
                    break;
                case (self::LBD_DYU_SECTOR):
                    foreach ($keys_available as $id) {
                        if ($this->crew[$id]->base == $this->location[0] && $this->crew[$id]->current_location == $this->location[1] &&
                            strtotime($this->crew[$id]->different_hub_first_date) <= strtotime($date_limit)) {
                            $priority1[] = $id;
                            //debug("ID:".$id." | First Date:". $this->crew[$id]->different_hub_first_date." DATE Limit:".$date_limit);

                        }
                        else
                            $priority2[] = $id;

                    }
                    //debug("BEFORE: ".implode(", ",$keys_available). " AFTER: ".implode(", ", array_merge($priority1, $priority2)));
                    break;

            }
            return array_merge($priority1, $priority2);
        }
        else
            return $keys_available;
    }

    /**
     * @param $standby_eligible
     * @return array
     */
    public function Standby_Crew_Check($standby_eligible){
        $keys['standby_passed'] = [];

        if (count($standby_eligible)) {
            /* foreach ($standby_eligible as $id) {
                 if (empty($this->crew[$id]->schedule) || end($this->crew[$id]->schedule)->is_standby == false)
                     $keys['standby_passed'][] = $id;
             }*/
            $keys['standby_passed'] = $standby_eligible;
            if (count($keys['standby_passed'])) {
                foreach ($keys['standby_passed'] as $id) {
                    $keys['with_standby_count'][$id] = $this->crew[$id]->standby_count;
                }
                // Sort by Standby Count
                asort($keys['with_standby_count']);
                $keys['standby_passed'] = array_keys($keys['with_standby_count']);
            }
        }

        return $keys['standby_passed'];
    }

    /**
     * @param $standby_eligible
     * @param $new_fdp
     * @return array
     */
    public function Standby_Schedule($standby_previously, $standby_eligible, $standby_pseudo, $new_fdp){
        // Priority 1 List Schedule
        $key['standby_final'] = [];

        if (count($standby_previously)){ // Is STANDBY for Another Flight
            $key['standby_final'][] = reset($standby_previously);
            foreach ($key['standby_final'] as $id) {
                foreach ($new_fdp as $fdp) {
                    $each_fdp = clone $fdp;
                    $each_fdp->is_standby = true;
                    $each_fdp->block_hours = 0;
                    $report_time = $each_fdp->flights[0]['std'];
                    $release_time = Add_Minutes_To_DateTime($each_fdp->flights[0]['std'], self::STANDBY_FLIGHT_WAIT_PERIOD);
                    $each_fdp->report_time_binded = end($this->crew[$id]->schedule)->report_time_binded;
                    $each_fdp->release_time = $release_time;
                    //$each_fdp->sectors++;
                    $each_fdp->fdp_current = $this->Calculate_Duration($report_time, $release_time);
                    $each_fdp->fdp_cumulative = $this->Calculate_Duration($each_fdp->report_time_binded, $release_time);
                    $each_fdp->required_rest = max($each_fdp->fdp_current, 12);
                    $each_fdp->location_departure = $each_fdp->flights[0]['departure_airport'];
                    $each_fdp->location_arrival = $each_fdp->flights[0]['departure_airport'];
                    $each_fdp->flights = [$each_fdp->flights[0]];
                    $this->crew[$id]->standby_count++;
                    $this->crew[$id]->schedule[] = $each_fdp;
                    break;
                }
            }
        }
        else {
            if (count($standby_eligible)) {
                $key['standby_final'][] = reset($standby_eligible); // one_standby Id

                foreach ($key['standby_final'] as $id) {
                    foreach ($new_fdp as $fdp) {
                        $each_fdp = clone $fdp;
                        $each_fdp->is_standby = true;
                        $each_fdp->block_hours = 0;
                        $report_time = Add_Minutes_To_DateTime($each_fdp->flights[0]['std'], self::STANDBY_CREW_REPORT_TIME, null, 'subtract');
                        $release_time = Add_Minutes_To_DateTime($each_fdp->flights[0]['std'], self::STANDBY_FLIGHT_WAIT_PERIOD);
                        $each_fdp->report_time = $report_time;
                        $each_fdp->sectors = 0;
                        $each_fdp->report_time_binded = $report_time;
                        $each_fdp->release_time = $release_time;
                        $each_fdp->fdp_current = $this->Calculate_Duration($report_time, $release_time);
                        $each_fdp->fdp_cumulative = $this->Calculate_Duration($report_time, $release_time);
                        $each_fdp->required_rest = max($each_fdp->fdp_current, 12);
                        $each_fdp->location_departure = $each_fdp->flights[0]['departure_airport'];
                        $each_fdp->location_arrival = $each_fdp->flights[0]['departure_airport'];
                        $each_fdp->flights = [$each_fdp->flights[0]];
                        $this->crew[$id]->standby_count++;
                        $this->crew[$id]->schedule[] = $each_fdp;
                        break;
                    }
                }
            }
            else {
                if (count($standby_pseudo)){
                    $key['standby_final'][] = reset($standby_pseudo); // one_standby Id

                    foreach ($key['standby_final'] as $id) {
                        foreach ($new_fdp as $fdp) {
                            $each_fdp = clone $fdp;
                            $each_fdp->is_standby = true;
                            $each_fdp->block_hours = 0;
                            $report_time = Add_Minutes_To_DateTime($each_fdp->flights[0]['std'], self::STANDBY_CREW_REPORT_TIME, null, 'subtract');
                            $release_time = Add_Minutes_To_DateTime($each_fdp->flights[0]['std'], self::STANDBY_FLIGHT_WAIT_PERIOD);
                            $each_fdp->report_time = $report_time;
                            $each_fdp->sectors = 0;
                            $each_fdp->report_time_binded = $report_time;
                            $each_fdp->release_time = $release_time;
                            $each_fdp->fdp_current = $this->Calculate_Duration($report_time, $release_time);
                            $each_fdp->fdp_cumulative = $this->Calculate_Duration($report_time, $release_time);
                            $each_fdp->required_rest = max($each_fdp->fdp_current, 12);
                            $each_fdp->location_departure = $each_fdp->flights[0]['departure_airport'];
                            $each_fdp->location_arrival = $each_fdp->flights[0]['departure_airport'];
                            $each_fdp->flights = [$each_fdp->flights[0]];
                            $this->crew[$id]->standby_count++;
                            $this->crew[$id]->schedule[] = $each_fdp;
                            break;
                        }
                    }
                }
            }
        }
        return $key['standby_final'];
    }

    /**
     * @param $keys_passed_limitations
     * @param $not_scheduled_yet
     * @param $new_fdp
     * @return array
     */
    public function Final_Key_Array($keys_passed_limitations, $not_scheduled_yet, $new_fdp){
        $keys['passed_limitations'] = $keys_passed_limitations;
        $keys['not_been_scheduled'] = $not_scheduled_yet;
        // Loop thru FCM who has previous fdp For Selection Priority 1 and 3

        $keys['passed_check_priority1'] = []; // Last FDP + New FDP within Last FDP Max
        $keys['passed_check_priority1_end'] = []; // Last FDP + New FDP within Last FDP Max
        $keys['passed_check_priority2'] = []; // Not Scheduled Ones
        $keys['passed_check_priority3'] = []; // Last FDP + Rest + New FDP
        $keys['passed_check_priority3_end'] = []; // Last FDP + Rest + New FDP

        $keys['priority1'] = [];
        $keys['priority2'] = [];
        $keys['priority3'] = [];
        $keys['standby'] = [];

        $new_fdp_sector =  end($new_fdp[0]->flights)['departure_airport'].'-'.end($new_fdp[0]->flights)['arrival_airport'];

//        debug($keys_passed_limitations);

        foreach ($keys['passed_limitations'] as $j){
            if (empty($this->crew[$j]->schedule))
                continue;

            $last_fdp = end($this->crew[$j]->schedule);

            $last_fdp_sector_outbound =  end($last_fdp->flights)['departure_airport'].'-'.end($last_fdp->flights)['arrival_airport'];
            $last_fdp_sector_inbound =  end($last_fdp->flights)['arrival_airport'].'-'.end($last_fdp->flights)['departure_airport'];
            $last_fdp_sectors_array = [$last_fdp_sector_outbound, $last_fdp_sector_inbound];

            if ($last_fdp->location_arrival == $new_fdp[0]->location_departure) {

                // Check if New FDP Report time is after Last FDP release times
                if (Compare_DateTimes($new_fdp[0]->report_time, $last_fdp->release_time)) {
                    // Get All FCM who can continue the next flight within FDP Extended Limit (Get Selection 1)

                    $period = $this->Calculate_Duration($last_fdp->report_time_binded, $new_fdp[0]->release_time);
                    $fdp_ext = $this->Max_FDP($last_fdp->report_time_binded, $last_fdp->sectors + $new_fdp[0]->sectors, $this->current_law ) + 1;
                    if ($period < $fdp_ext) {

                        if ($last_fdp->is_standby){
                            $period = $this->Calculate_Duration($last_fdp->report_time_binded, Add_Minutes_To_DateTime($new_fdp[0]->report_time, self::CREW_RELEASE_TIME_MIN));
                            $fdp_ext = self::STANDBY_MAX_FDP_PERIOD / 60; //$this->Max_FDP($last_fdp->report_time_binded, $last_fdp->sectors + $new_fdp[0]->sectors) + 1;
                            if ($period < $fdp_ext) {
                                $keys['standby'][] = $j;
                            }
                        }

                        if (in_array($new_fdp_sector, $last_fdp_sectors_array)){
                            $keys['passed_check_priority1_end'][] = $j;
                        }
                        else {
                            $keys['passed_check_priority1'][] = $j;
                        }
                    }
                    else {
                        // Check if New FDP Report Time > Last FDP Release Time + Required Rest Time
                        if (Compare_DateTimes($new_fdp[0]->report_time, Add_Minutes_To_DateTime($last_fdp->release_time, $last_fdp->required_rest, 'hours'))) {
                            if (in_array($new_fdp_sector, $last_fdp_sectors_array)){
                                $keys['passed_check_priority3_end'][] = $j;
                            }
                            else {
                                $keys['passed_check_priority3'][] = $j;
                            }
                        }
                    }
                }
            }
        }

        foreach ($keys['not_been_scheduled'] as $k){
            if ($this->crew[$k]->base == $new_fdp[0]->location_departure) {
                $keys['passed_check_priority2'][] = $k;
            }
        }

        if (count($keys['passed_check_priority1_end']))
            $keys['passed_check_priority1'] = array_merge($keys['passed_check_priority1'], $keys['passed_check_priority1_end']);

        if (count($keys['passed_check_priority3_end']))
            $keys['passed_check_priority3'] = array_merge($keys['passed_check_priority3'], $keys['passed_check_priority3_end']);

        if (count($keys['passed_check_priority1'])) {
            // Sector Done Least First
            $keys['priority1'] = $this->Crew_Flight_History_Sectors($keys['passed_check_priority1'], null, $new_fdp_sector);

            // Least Hours First
            $keys['priority1'] = $this->Get_Lesser_Hours_Crew($keys['priority1'], null, 'block_hours_28');

            // NON-Pseudo Crew First
            //$keys['priority1'] = $this->Priority_Selection($keys['priority1'], null, 'Non_Pseudo_First');
        }

        if (count($keys['passed_check_priority2'])) {
            // Sector Done Least First
            $keys['priority2'] = $this->Crew_Flight_History_Sectors($keys['passed_check_priority2'], null, $new_fdp_sector);

            // Least Hours First
            $keys['priority2'] = $this->Get_Lesser_Hours_Crew($keys['priority2'], null, 'block_hours_28');

            // NON-Pseudo Crew First
            //$keys['priority2'] = $this->Priority_Selection($keys['priority2'], null, 'Non_Pseudo_First');

        }

        if (count($keys['passed_check_priority3'])) {
            // Sector Done Least First
            $keys['priority3'] = $this->Crew_Flight_History_Sectors($keys['passed_check_priority3'], null, $new_fdp_sector);

            // Least Hours First
            $keys['priority3'] = $this->Get_Lesser_Hours_Crew($keys['priority3'], null, 'block_hours_28');

            // NON-Pseudo Crew First
            //$keys['priority3'] = $this->Priority_Selection($keys['priority3'], null, 'Non_Pseudo_First');

        }




        return [$keys['priority1'], $keys['priority2'], $keys['priority3'], $keys['standby']];

    }

    /**
     * Check FDP Regulation and Return NEW FDP Plans
     * @param $flights
     * @return array
     */
    public function New_FDP_Create($flights, $aircraft_type, $crew_min)
    {
        // Number of New FDP's
        $f = 0;
        $new_fdp = [];

        $sector_count = 0;
        switch ($this->flight_type) {
            case self::MANY_FLIGHTS :
                $max_fdp_allowed = $this->Max_FDP(date('Y-m-d H:i', strtotime($flights[0]['std'])), count($flights), $this->current_law);
                $max_fdp_extended = $max_fdp_allowed + 1;
                foreach ($flights as $j => $each_flight) {
                    $report_time[$j] = date('Y-m-d H:i', strtotime($each_flight['std']) - self::CREW_REPORT_TIME_SEC);
                    $release_time[$j] = Add_Minutes_To_DateTime($each_flight['sta'], self::CREW_RELEASE_TIME_MIN);

                    // FIRST Flight
                    if ($j == 0) {
                        $new_fdp[$f] = new stdClass();
                        $sector_count++;
                        $new_fdp[$f]->is_dhc = false;
                        $new_fdp[$f]->is_standby = false;
                        $new_fdp[$f]->report_time = $report_time[$j];
                        $new_fdp[$f]->report_time_binded = $new_fdp[$f]->report_time;
                        $new_fdp[$f]->fdp_max = $this->Max_FDP($new_fdp[$f]->report_time, $sector_count, $this->current_law);
                        $new_fdp[$f]->fdp_ext = $new_fdp[$f]->fdp_max + 1;
                        $new_fdp[$f]->sectors = $sector_count;
                        $new_fdp[$f]->release_time = $release_time[$j];
                        $new_fdp[$f]->fdp_current = $this->Calculate_Duration($new_fdp[$f]->report_time, $release_time[$j]);
                        $new_fdp[$f]->fdp_cumulative = $new_fdp[$f]->fdp_current;
                        $new_fdp[$f]->fdp_wocl_reduction = 0;
                        //Block Time
                        $new_fdp[$f]->block_hours = $this->Calculate_Duration($each_flight['std'], $each_flight['sta']);

                        $new_fdp[$f]->location_departure = $each_flight['departure_airport'];
                        $new_fdp[$f]->location_arrival = $each_flight['arrival_airport'];
                        $new_fdp[$f]->flights[] = ['id' => $each_flight['id'], 'parent_id' => $each_flight['parent_id'], 'flight_number' => $each_flight['flight_number'], 'departure_airport' => $each_flight['departure_airport'],
                            'arrival_airport' => $each_flight['arrival_airport'], 'aircraft_id' => $each_flight['aircraft_id'], 'std' => $each_flight['std'], 'sta' => $each_flight['sta']];

                        $this->Calculate_Max_Allowed_FDP($new_fdp, $f);

                        // If Previous Flight's Arrival Was In Base REST is 12 Hrs, otherwise it's 10 Hrs
                        $this->Rest_Period($new_fdp, $f);


                    } else { // SECOND AND SO ON FLIGHTS

                        // Determine Next Flight Type if Exists ($flight_many_type)
                        $crew_add = $this->Current_Flight_Type($new_fdp, $f, $each_flight['sta'], $report_time[$j - 1], $report_time[$j]);

                        switch ($this->flight_many_type) {
                            case self::FLIGHT_WITH_REST :
                                $f++;
                                $new_fdp[$f] = new stdClass();
                                $new_fdp[$f]->is_dhc = false;
                                $new_fdp[$f]->is_standby = false;
                                $new_fdp[$f]->report_time = $report_time[$j];
                                $new_fdp[$f]->report_time_binded = $new_fdp[$f]->report_time;
                                $new_fdp[$f]->fdp_max = $this->Max_FDP($new_fdp[$f]->report_time, $sector_count, $this->current_law);
                                $new_fdp[$f]->fdp_ext = $new_fdp[$f]->fdp_max + 1;
                                $new_fdp[$f]->sectors = $sector_count;
                                $new_fdp[$f]->release_time = $release_time[$j];
                                $new_fdp[$f]->fdp_current = $this->Calculate_Duration($new_fdp[$f]->report_time, $release_time[$j]);
                                $new_fdp[$f]->fdp_cumulative = $new_fdp[$f]->fdp_current;
                                $new_fdp[$f]->fdp_wocl_reduction = 0;

                                $this->Calculate_Max_Allowed_FDP($new_fdp, $f);

                                if ($new_fdp[$f]->fdp_cumulative > $new_fdp[$f]->fdp_max) {
                                    $this->Additional_Crew($new_fdp, $f, $crew_min, $sector_count);
                                }

                                // Error Handling
                                if ($this->Calculate_Duration($new_fdp[$f-1]->release_time, $new_fdp[$f]->report_time) < $new_fdp[$f-1]->required_rest)
                                    $new_fdp[$f]->error_message = 'Rest Requirements Are Not Met';

                                //Block Time
                                $new_fdp[$f]->block_hours = $this->Calculate_Duration($each_flight['std'], $each_flight['sta']);
                                // Maximum of : Previous Duty Time or 12 hours;

                                $new_fdp[$f]->required_rest = max($new_fdp[$f]->fdp_current, 12);
                                $new_fdp[$f]->location_departure = $each_flight['departure_airport'];
                                $new_fdp[$f]->location_arrival = $each_flight['arrival_airport'];
                                $new_fdp[$f]->flights[] = ['id' => $each_flight['id'], 'parent_id' => $each_flight['parent_id'], 'flight_number' => $each_flight['flight_number'], 'departure_airport' => $each_flight['departure_airport'],
                                    'arrival_airport' => $each_flight['arrival_airport'], 'aircraft_id' => $each_flight['aircraft_id'], 'std' => $each_flight['std'], 'sta' => $each_flight['sta']];
                                // If Previous Flight's Arrival Was In Base REST is 12 Hrs, otherwise it's 10 Hrs
                                $this->Rest_Period($new_fdp, $f);

                                break;
                            case self::FLIGHT_NO_REST :
                                $sector_count++;
                                $new_fdp[$f]->report_time = $report_time[$j - 1];
                                $new_fdp[$f]->report_time_binded = $new_fdp[$f]->report_time;
                                $new_fdp[$f]->fdp_max = $this->Max_FDP($new_fdp[$f]->report_time, $sector_count, $this->current_law);
                                $new_fdp[$f]->fdp_ext = $new_fdp[$f]->fdp_max + 1;
                                $new_fdp[$f]->sectors = $sector_count;
                                $new_fdp[$f]->release_time = $release_time[$j];
                                $new_fdp[$f]->fdp_current = $this->Calculate_Duration($new_fdp[$f]->report_time, $release_time[$j]);
                                $new_fdp[$f]->fdp_cumulative = $new_fdp[$f]->fdp_current;
                                $new_fdp[$f]->fdp_wocl_reduction = 0;

                                $this->Calculate_Max_Allowed_FDP($new_fdp, $f);

                                if ($new_fdp[$f]->fdp_cumulative > $new_fdp[$f]->fdp_max) {
                                    $this->Additional_Crew($new_fdp, $f, $crew_min, $sector_count);
                                }

                                //Block Time
                                $new_fdp[$f]->block_hours += $this->Calculate_Duration($each_flight['std'], $each_flight['sta']);
                                // Maximum of : Previous Duty Time or 12 hours;
                                $new_fdp[$f]->required_rest = max($new_fdp[$f]->fdp_current, 12);
                                // Departure Location is the first FDP departure airport
                                $new_fdp[$f]->location_arrival = $each_flight['arrival_airport'];
                                $new_fdp[$f]->flights[] = ['id' => $each_flight['id'], 'parent_id' => $each_flight['parent_id'], 'flight_number' => $each_flight['flight_number'], 'departure_airport' => $each_flight['departure_airport'],
                                    'arrival_airport' => $each_flight['arrival_airport'], 'aircraft_id' => $each_flight['aircraft_id'], 'std' => $each_flight['std'], 'sta' => $each_flight['sta']];

                                if (date("H:i", strtotime($new_fdp[$f]->report_time_binded)) == "13:50"){
                                   // debug($new_fdp[$f]->wocl);
                                }
                                break;
                        }
                    }
                }
                break;

            case self::MANY_FLIGHTS_WITH_DIVERSION:
                $max_fdp_allowed = $this->Max_FDP(date('Y-m-d H:i', strtotime($flights[0]['std'])), count($flights), $this->current_law);
                $max_fdp_extended = $max_fdp_allowed + 1;
                foreach ($flights as $j => $each_flight) {
                    $report_time[$j] = date('Y-m-d H:i', strtotime($each_flight['std']) - self::CREW_REPORT_TIME_SEC);
                    $release_time[$j] = Add_Minutes_To_DateTime($each_flight['sta'], self::CREW_RELEASE_TIME_MIN);

                    if ($j == 0) {
                        $new_fdp[$f] = new stdClass();
                        $sector_count++;

                        // If FIRST Flight is Diverted
                        if ($this->Flight_Diverted($each_flight)) {
                            $release_time[$j] = Add_Minutes_To_DateTime($each_flight['diversion']['sta'], self::CREW_RELEASE_TIME_MIN);
                            $new_fdp[$f]->is_dhc = false;
                            $new_fdp[$f]->is_standby = false;
                            $new_fdp[$f]->report_time = $each_flight['std'];
                            $new_fdp[$f]->report_time_binded = $new_fdp[$f]->report_time;
                            $new_fdp[$f]->fdp_max = $this->Max_FDP($new_fdp[$f]->report_time, $sector_count, $this->current_law);
                            $new_fdp[$f]->fdp_ext = $new_fdp[$f]->fdp_max + 1;
                            $new_fdp[$f]->sectors = $sector_count;
                            $new_fdp[$f]->release_time = $release_time[$j];
                            $new_fdp[$f]->fdp_current = $this->Calculate_Duration($new_fdp[$f]->report_time, $release_time[$j]);
                            $new_fdp[$f]->fdp_cumulative = $new_fdp[$f]->fdp_current;
                            //Block Time
                            $new_fdp[$f]->block_hours = $this->Calculate_Duration($each_flight['diversion']['std'], $each_flight['diversion']['sta']);
                            $new_fdp[$f]->location_departure = $each_flight['departure_airport'];
                            $new_fdp[$f]->location_arrival = $each_flight['arrival_airport'];
                            $new_fdp[$f]->flights[] = ['id' => $each_flight['id'], 'parent_id' => $each_flight['parent_id'], 'flight_number' => $each_flight['flight_number'], 'departure_airport' => $each_flight['departure_airport'],
                                'arrival_airport' => $each_flight['diversion']['airport'], 'aircraft_id' => $each_flight['aircraft_id'], 'std' => $each_flight['std'], 'sta' => $each_flight['diversion']['sta']];

                            // If Previous Flight's Arrival Was In Base REST is 12 Hrs, otherwise it's 10 Hrs
                            $this->Rest_Period($new_fdp, $f);

                            $crew_add = $this->Current_Flight_Type($new_fdp, $f, $each_flight['sta'], $each_flight['std'], $each_flight['diversion']['std']);

                            switch ($this->flight_many_type) {
                                case self::FLIGHT_WITH_REST :
                                    $f++;
                                    $new_fdp[$f] = new stdClass();
                                    $report_time[$j] = date('Y-m-d H:i', strtotime($each_flight['diversion']['std']) - self::CREW_REPORT_TIME_SEC);
                                    $release_time[$j] = Add_Minutes_To_DateTime($each_flight['sta'], self::CREW_RELEASE_TIME_MIN);
                                    $new_fdp[$f]->is_dhc = false;
                                    $new_fdp[$f]->is_standby = false;
                                    $new_fdp[$f]->report_time = $report_time[$j];
                                    $new_fdp[$f]->report_time_binded = $new_fdp[$f]->report_time;
                                    $new_fdp[$f]->fdp_max = $this->Max_FDP($new_fdp[$f]->report_time, $sector_count, $this->current_law);
                                    $new_fdp[$f]->fdp_ext = $new_fdp[$f]->fdp_max + 1;
                                    $new_fdp[$f]->sectors = $sector_count;
                                    $new_fdp[$f]->release_time = $release_time[$j];
                                    $new_fdp[$f]->fdp_current = $this->Calculate_Duration($new_fdp[$f]->report_time, $release_time[$j]);
                                    $new_fdp[$f]->fdp_cumulative = $new_fdp[$f]->fdp_current;
                                    //Block Time
                                    $new_fdp[$f]->block_hours = $this->Calculate_Duration($each_flight['std'], $each_flight['sta']);
                                    $new_fdp[$f]->location_departure = $each_flight['departure_airport'];
                                    $new_fdp[$f]->location_arrival = $each_flight['arrival_airport'];
                                    $new_fdp[$f]->flights[] = ['id' => $each_flight['id'], 'parent_id' => $each_flight['parent_id'], 'flight_number' => $each_flight['flight_number'], 'departure_airport' => $each_flight['diversion']['airport'],
                                        'arrival_airport' => $each_flight['arrival_airport'], 'aircraft_id' => $each_flight['aircraft_id'], 'std' => $each_flight['diversion']['std'], 'sta' => $each_flight['sta']];

                                    // If Previous Flight's Arrival Was In Base REST is 12 Hrs, otherwise it's 10 Hrs
                                    $this->Rest_Period($new_fdp, $f);
                                    break;
                                case self::FLIGHT_NO_REST :
                                    $sector_count++;
                                    $release_time[$j] = Add_Minutes_To_DateTime($each_flight['sta'], self::CREW_RELEASE_TIME_MIN);

                                    $new_fdp[$f]->report_time = $report_time[$j - 1];
                                    $new_fdp[$f]->report_time_binded = $new_fdp[$f]->report_time;
                                    $new_fdp[$f]->fdp_max = $this->Max_FDP($new_fdp[$f]->report_time, $sector_count, $this->current_law);
                                    $new_fdp[$f]->fdp_ext = $new_fdp[$f]->fdp_max + 1;
                                    $new_fdp[$f]->sectors = $sector_count;
                                    $new_fdp[$f]->release_time = $release_time[$j];
                                    $new_fdp[$f]->fdp_current = $this->Calculate_Duration($new_fdp[$f]->report_time, $release_time[$j]);
                                    $new_fdp[$f]->fdp_cumulative = $new_fdp[$f]->fdp_current;
                                    //Block Time
                                    $new_fdp[$f]->block_hours += $this->Calculate_Duration($each_flight['std'], $each_flight['sta']);
                                    // Departure Location is the first FDP departure airport
                                    $new_fdp[$f]->location_arrival = $each_flight['arrival_airport'];
                                    $new_fdp[$f]->flights[] = ['id' => $each_flight['id'], 'parent_id' => $each_flight['parent_id'], 'flight_number' => $each_flight['flight_number'], 'departure_airport' => $each_flight['diversion']['airport'],
                                        'arrival_airport' => $each_flight['arrival_airport'], 'aircraft_id' => $each_flight['aircraft_id'], 'std' => $each_flight['diversion']['std'], 'sta' => $each_flight['sta']];
                                    // If Previous Flight's Arrival Was In Base REST is 12 Hrs, otherwise it's 10 Hrs
                                    $this->Rest_Period($new_fdp, $f);

                                    break;
                            }

                        }


                    } else { // SECOND AND SO ON FLIGHTS

                        // Determine Flight Type (WITH REST OR NOT)
                        $crew_add = $this->Current_Flight_Type($new_fdp, $f, $each_flight['sta'], $report_time[$j - 1], $report_time[$j]);

                        switch ($this->flight_many_type) {
                            case self::FLIGHT_WITH_REST :
                                $f++;
                                $new_fdp[$f] = new stdClass();
                                $new_fdp[$f]->is_dhc = false;
                                $new_fdp[$f]->is_standby = false;
                                $new_fdp[$f]->report_time = $report_time[$j];
                                $new_fdp[$f]->report_time_binded = $new_fdp[$f]->report_time;
                                $new_fdp[$f]->fdp_max = $this->Max_FDP($new_fdp[$f]->report_time, $sector_count, $this->current_law);
                                $new_fdp[$f]->fdp_ext = $new_fdp[$f]->fdp_max + 1;
                                $new_fdp[$f]->sectors = $sector_count;
                                $new_fdp[$f]->release_time = $release_time[$j];
                                $new_fdp[$f]->fdp_current = $this->Calculate_Duration($new_fdp[$f]->report_time, $release_time[$j]);
                                $new_fdp[$f]->fdp_cumulative = $new_fdp[$f]->fdp_current;
                                //Block Time
                                $new_fdp[$f]->block_hours = $this->Calculate_Duration($each_flight['std'], $each_flight['sta']);
                                $new_fdp[$f]->location_departure = $each_flight['departure_airport'];
                                $new_fdp[$f]->location_arrival = $each_flight['arrival_airport'];
                                $new_fdp[$f]->flights[] = ['id' => $each_flight['id'], 'parent_id' => $each_flight['parent_id'], 'flight_number' => $each_flight['flight_number'], 'departure_airport' => $each_flight['departure_airport'],
                                    'arrival_airport' => $each_flight['arrival_airport'], 'aircraft_id' => $each_flight['aircraft_id'], 'std' => $each_flight['std'], 'sta' => $each_flight['sta']];
                                // If Previous Flight's Arrival Was In Base REST is 12 Hrs, otherwise it's 10 Hrs
                                $this->Rest_Period($new_fdp, $f);

                                break;
                            case self::FLIGHT_NO_REST :
                                $sector_count++;
                                $new_fdp[$f]->report_time = $report_time[$j - 1];
                                $new_fdp[$f]->report_time_binded = $new_fdp[$f]->report_time;
                                $new_fdp[$f]->fdp_max = $this->Max_FDP($new_fdp[$f]->report_time, $sector_count, $this->current_law);
                                $new_fdp[$f]->fdp_ext = $new_fdp[$f]->fdp_max + 1;
                                $new_fdp[$f]->sectors = $sector_count;
                                $new_fdp[$f]->release_time = $release_time[$j];
                                $new_fdp[$f]->fdp_current = $this->Calculate_Duration($new_fdp[$f]->report_time, $release_time[$j]);
                                $new_fdp[$f]->fdp_cumulative = $new_fdp[$f]->fdp_current;
                                //Block Time
                                $new_fdp[$f]->block_hours += $this->Calculate_Duration($each_flight['std'], $each_flight['sta']);
                                // Departure Location is the first FDP departure airport
                                $new_fdp[$f]->location_arrival = $each_flight['arrival_airport'];
                                $new_fdp[$f]->flights[] = ['id' => $each_flight['id'], 'parent_id' => $each_flight['parent_id'], 'flight_number' => $each_flight['flight_number'], 'departure_airport' => $each_flight['departure_airport'],
                                    'arrival_airport' => $each_flight['arrival_airport'], 'aircraft_id' => $each_flight['aircraft_id'], 'std' => $each_flight['std'], 'sta' => $each_flight['sta']];
                                // If Previous Flight's Arrival Was In Base REST is 12 Hrs, otherwise it's 10 Hrs
                                $this->Rest_Period($new_fdp, $f);

                                break;
                        }
                    }
                }
                break;

            case self::ONE_FLIGHT :
                $sector_count++;
                $report_time = date('Y-m-d H:i', strtotime($flights[0]['std']) - self::CREW_REPORT_TIME_SEC);
                $release_time = Add_Minutes_To_DateTime($flights[0]['sta'], self::CREW_RELEASE_TIME_MIN);

                $new_fdp[$f] = new stdClass();
                $new_fdp[$f]->is_dhc = false;
                $new_fdp[$f]->is_standby = false;
                $new_fdp[$f]->report_time = $report_time;
                $new_fdp[$f]->report_time_binded = $new_fdp[$f]->report_time;
                $new_fdp[$f]->fdp_max = $this->Max_FDP($new_fdp[$f]->report_time, $sector_count, $this->current_law);
                $new_fdp[$f]->fdp_ext = $new_fdp[$f]->fdp_max + 1;
                $new_fdp[$f]->sectors = $sector_count;
                $new_fdp[$f]->release_time = $release_time;
                $new_fdp[$f]->fdp_current = $this->Calculate_Duration($new_fdp[$f]->report_time, $release_time);
                $new_fdp[$f]->fdp_cumulative = $new_fdp[$f]->fdp_current;
                //Block Time
                $new_fdp[$f]->block_hours = $this->Calculate_Duration($flights[0]['std'], $flights[0]['sta']);
                $new_fdp[$f]->location_departure= $flights[0]['departure_airport'];
                $new_fdp[$f]->location_arrival= $flights[0]['arrival_airport'];
                $new_fdp[$f]->flights[] = ['id' => $flights[0]['id'], 'parent_id' => $flights[0]['parent_id'], 'flight_number' => $flights[0]['flight_number'], 'departure_airport' => $flights[0]['departure_airport'],
                    'arrival_airport' => $flights[0]['arrival_airport'], 'aircraft_id' => $flights[0]['aircraft_id'], 'std' => $flights[0]['std'], 'sta' => $flights[0]['sta']];
                // If Previous Flight's Arrival Was In Base REST is 12 Hrs, otherwise it's 10 Hrs
                $this->Rest_Period($new_fdp, $f);
                break;
        }

        return $new_fdp;
    }

    /**
     * Check Flight in Group, whether rest is required in between
     * @param $new_fdp
     * @param $f
     * @param $flight_sta
     * @param $report_time_previous_flight
     * @param $report_time_flight
     * @return bool
     */
    public function Current_Flight_Type($new_fdp, $f, $flight_sta, $report_time_previous_flight, $report_time_flight)
    {
        $crew_add = false;

        // This line should be deleted, is here for FRA sector problem
        $accuracy = 0.5; // Accuracy +/-30 Minutes

        if (isset($flight_sta)) {
            $release_time_flight = Add_Minutes_To_DateTime($flight_sta, self::CREW_RELEASE_TIME_MIN);
            if ($this->Calculate_Duration($report_time_previous_flight, $release_time_flight) > $new_fdp[$f]->fdp_max){// $max_fdp_extended) {
                // Flights are planned this way, Additional 2 crews are required
                if ($this->Calculate_Duration($new_fdp[$f]->release_time, $report_time_flight) + $accuracy < $new_fdp[$f]->required_rest) {
                    $this->flight_many_type = self::FLIGHT_NO_REST;
                    $crew_add = true;
                }
                else
                    $this->flight_many_type = self::FLIGHT_WITH_REST;
            }
            else
                $this->flight_many_type = self::FLIGHT_NO_REST;
        }
        else
            $this->flight_many_type = self::FLIGHT_NOT_EXISTS;

        return $crew_add;
    }




    /**
     * @param $new_fdp
     * @param $j
     */
    public function Rest_Period($new_fdp, $j){
        if (in_array($new_fdp[$j]->location_arrival, [$this->location[0], $this->location[1]]))
            $new_fdp[$j]->required_rest = max($new_fdp[$j]->fdp_current, 12);
        else
            $new_fdp[$j]->required_rest = max($new_fdp[$j]->fdp_current, 10);
    }

    /**
     * @param $array_keys
     * @param $departure_date
     * @param $limit_hours
     * @param $label_type
     */
    public function FTL_Limitation_Check($array_keys, $departure_date, $limit_hours, $label_type, $new_fdp_total_hours){
        foreach ($array_keys as $id){
            // Checks for Limitation And Sets Eligible Variable to TRUE or FALSE
            $this->crew[$id]->Count_Block_Hours($departure_date, $label_type, $limit_hours, $new_fdp_total_hours);
        }
    }

    public function CrewRestrictionsCheck($type, $flights, $availableKeys){
        switch ($type){
            case "aircraft_type":
                $currentACType = $flights[0]['aircraft_type'];
                $passedCheckKeys = [];
                if ($currentACType) {
                    foreach ($availableKeys as $key) {
                        if (isset($this->crew[$key]->aircraft_types_restrictions) && count($this->crew[$key]->aircraft_types_restrictions)){
                            // If Current AC Type not in users AC Types Restrictions List
                            if (!in_array($currentACType, $this->crew[$key]->aircraft_types_restrictions)){
                                $passedCheckKeys[] = $key;
                            }
                            continue;
                        }
                        // No Restrictions Found
                        $passedCheckKeys[] = $key;
                    }

                    return $passedCheckKeys;
                }
            break;

            case "route":
                $currentRoute = $flights[0]['flight_route_id'];
                $passedCheckKeys = [];
                if ($currentRoute) {
                    foreach ($availableKeys as $key) {
                        if (isset($this->crew[$key]->route_restrictions) && count($this->crew[$key]->route_restrictions)){
                            // If Current AC Type not in users AC Types Restrictions List
                            if (!in_array($currentRoute, $this->crew[$key]->route_restrictions)){
                                $passedCheckKeys[] = $key;
                            }
                            continue;
                        }
                        // No Restrictions Found
                        $passedCheckKeys[] = $key;
                    }

                    return $passedCheckKeys;
                }
            break;
        }

        return $availableKeys;
    }

    /**
     * Schedule Priority 1 and 2/3 CCM
     * @param $keys_priority1
     * @param $final_keys
     * @param $new_fdp
     */
    public function Schedule_Crew($keys_priority1, $final_keys, $new_fdp){
        // Priority 1 List Schedule
        if (count($final_keys)){
            foreach ($final_keys as $id){
                foreach($new_fdp as $fdp) {
                    if (in_array($id, $keys_priority1)) { // Id is Priority 1 Key
                        $each_fdp = clone $fdp;
                        $this->crew[$id]->current_location = $each_fdp->location_arrival;
                        $this->crew[$id]->planned_hours += $each_fdp->block_hours;

                        // Update New FDP Sector Count
                        $each_fdp->sectors += end($this->crew[$id]->schedule)->sectors;

                        // Update Last FDP's Max/Ext FDP Values based on sector increase
                        end($this->crew[$id]->schedule)->fdp_max = $this->Max_FDP(end($this->crew[$id]->schedule)->report_time, $each_fdp->sectors, $this->current_law);
                        end($this->crew[$id]->schedule)->fdp_ext = end($this->crew[$id]->schedule)->fdp_max + 1;

                        $each_fdp->report_time_binded = end($this->crew[$id]->schedule)->report_time_binded;
                        $each_fdp->fdp_cumulative = $this->Calculate_Duration($each_fdp->report_time_binded, $each_fdp->release_time);
                        $each_fdp->fdp_max = end($this->crew[$id]->schedule)->fdp_max;
                        $each_fdp->fdp_ext = end($this->crew[$id]->schedule)->fdp_ext;
                        $this->crew[$id]->schedule[] = $each_fdp;
                    }
                    else {
                        $each_fdp = clone $fdp;
                        $this->crew[$id]->current_location = $each_fdp->location_arrival;
                        $this->crew[$id]->schedule[] = $each_fdp;
                        $this->crew[$id]->planned_hours += $each_fdp->block_hours;
                    }

                    $this->crew[$id]->Add_Flight_Hours($each_fdp->block_hours, $each_fdp->flights[0]['std']);
                    $this->crew[$id]->Add_Sector_Count($each_fdp->flights);
                }
            }
        }
    }

    /**
     * DHC Crew Schedule out of given List and Return DHC List
     * @param $new_fdp
     * @param $keys_eligible
     * @param $keys_dhc_eligible
     * @return array
     */
    public function DHC_Crew_Schedule($new_fdp,$keys_eligible, $keys_dhc_eligible){
        $keys['dhc_final'] = [];
        if ($this->dhc_lbd && $this->lbd_crew_requirement > 0) {
            // Applies Only For DYU-LBD Flight
            if (in_array($this->flights_sector, [self::DYU_LBD_SECTOR, self::LBD_DYU_SECTOR, self::DYU_LBD_ROUNDTRIP])) {
                $current_date = date('Y-m-d', strtotime($new_fdp[0]->report_time));

                switch ($this->flights_sector) {
                    case self::DYU_LBD_SECTOR:
                        // LBD Total Crew Number Check
                        $keys['lbd_available'] = $this->Location_Available_Crew($keys_eligible, $this->location[1]);
                        $lbd_available_count = count($keys['lbd_available']);

                        if ($lbd_available_count < $this->lbd_crew_requirement) {
                            $required_number = $this->lbd_crew_requirement - $lbd_available_count;
                            if (count($keys_dhc_eligible) > $required_number)
                                $keys['dhc_final'] = array_slice($keys_dhc_eligible, 0, $required_number);
                            else
                                $keys['dhc_final'] = $keys_dhc_eligible;

                            if (count($keys['dhc_final'])) {
                                foreach ($keys['dhc_final'] as $id) {
                                    $this->crew[$id]->dhc_count++;
                                    $this->crew[$id]->current_location = $new_fdp[0]->location_arrival;
                                    $this->crew[$id]->schedule[] = clone $new_fdp[0];
                                    end($this->crew[$id]->schedule)->is_dhc = true;
                                    end($this->crew[$id]->schedule)->block_hours = 0;
                                }
                            }
                        }
                        break;
                    case self::LBD_DYU_SECTOR:
                        $keys['dch_date_limit'] = $this->Last_DHC_Date_Check($keys_dhc_eligible, $current_date);
                        if (count($keys['dch_date_limit'])) {
                            foreach ($keys['dch_date_limit'] as $id) {
                                $this->crew[$id]->current_location = $new_fdp[0]->location_arrival;
                                $this->crew[$id]->schedule[] = clone $new_fdp[0];
                                end($this->crew[$id]->schedule)->is_dhc = true;
                                end($this->crew[$id]->schedule)->block_hours = 0;
                            }
                        }

                        // LBD Total Crew Number Check
                        $keys['lbd_available'] = $this->Location_Available_Crew($keys_eligible, $this->location[1]);
                        $lbd_available_count = count($keys['lbd_available']);

                        if ($lbd_available_count > $this->lbd_crew_requirement) {
                            $not_required_number = $lbd_available_count - $this->lbd_crew_requirement;
                            if (count($keys_dhc_eligible) > $not_required_number)
                                $keys['dhc_final'] = array_slice($keys_dhc_eligible, 0, $not_required_number);
                            else
                                $keys['dhc_final'] = $keys_dhc_eligible;

                            if (count($keys['dhc_final'])) {
                                foreach ($keys['dhc_final'] as $id) {
                                    $this->crew[$id]->current_location = $new_fdp[0]->location_arrival;
                                    $this->crew[$id]->schedule[] = clone $new_fdp[0];
                                    end($this->crew[$id]->schedule)->is_dhc = true;
                                    end($this->crew[$id]->schedule)->block_hours = 0;
                                }
                            }
                        }
                        break;
                    case self::DYU_LBD_ROUNDTRIP:
                        break;
                }
            }
        }
        return $keys['dhc_final'];
    }

    /**
     * Check DHC Stay Period with different_hub_stay_period
     * @param $keys_check
     * @param $current_date
     * @return array
     */
    public function Last_DHC_Date_Check($keys_check, $current_date){
        $final_keys = [];
        $keys = [];
        foreach($keys_check as $id){
            foreach ($this->crew[$id]->schedule as $schedule){
                if ($schedule->is_dhc){
                    $date = date('Y-m-d', strtotime($schedule->report_time));
                    $keys[$id] = $date;
                }
            }
        }

        if (count($keys)) {
            foreach ($keys as $id => $date) {
                $date_limit = date('Y-m-d', strtotime("$date + $this->different_hub_stay_period days"));
                if (strtotime($current_date) >= strtotime($date_limit)) {
                    $final_keys[] = $id;
                }
            }
        }

        return $final_keys;
    }

    /**
     * Check DHC
     * @param $keys_dhc_eligible
     * @return array
     */
    public function DHC_Crew_Check($keys_dhc_eligible){
        $keys['dhc_passed'] = [];
        if ($this->dhc_lbd && count($keys_dhc_eligible) && in_array($this->flights_sector,[self::DYU_LBD_SECTOR, self::LBD_DYU_SECTOR, self::DYU_LBD_ROUNDTRIP])) {
            foreach ($keys_dhc_eligible as $id) {
                // Initial Location only DYU, FCM who lives in LBD will not be used as DHC
                if ($this->crew[$id]->base == $this->location[0]){
                    if (empty($this->crew[$id]->schedule) || (end($this->crew[$id]->schedule)->is_dhc == false && end($this->crew[$id]->schedule)->is_standby == false))
                        $keys['dhc_passed'][] = $id;
                }
            }

            if (count($keys['dhc_passed'])) {
                foreach ($keys['dhc_passed'] as $id) {
                    $keys['with_dhc_count'][$id] = $this->crew[$id]->dhc_count;
                }
                // Sort by DHC Count
                asort($keys['with_dhc_count']);
                $keys['dhc_passed'] = array_keys($keys['with_dhc_count']);
            }
        }

        return $keys['dhc_passed'];
    }

    /**
     * Determine sector/round trip of a given Flight
     * @param $flights
     */
    public function Flight_Sector($flights){
        $dep = $flights[0]['departure_airport'];
        $arr = $flights[0]['arrival_airport'];

        switch ($this->flight_type) {
            case self::ONE_FLIGHT :
                if ($dep == $this->location[0] && $arr == $this->location[1])
                    $this->flights_sector = self::DYU_LBD_SECTOR;
                else if ($dep == $this->location[1] && $arr == $this->location[0])
                    $this->flights_sector = self::LBD_DYU_SECTOR;
                else
                    $this->flights_sector = self::OTHER_SECTOR;
                break;

            case self::MANY_FLIGHTS :
                if ($dep == $this->location[0] && $arr == $this->location[1])
                    $this->flights_sector = self::DYU_LBD_ROUNDTRIP;
                else
                    $this->flights_sector = self::OTHER_ROUNDTRIP;
                break;
        }
    }

    /**
     * Get Array of Crew Ids(Keys) Available in Location Specified ( From Eligible List Array )
     * @param $keys_eligible
     * @param $location
     * @return array
     */
    public function Location_Available_Crew($keys_eligible, $location){
        // Find Eligible FCM Ids
        $keys['lbd_available'] = $this->Search_FCM_GetAllKeys($this->crew, 'location_available_crew',$location, $keys_eligible);
        return $keys['lbd_available'];
    }


    /**
     * Create Flights That Are Scheduled For Crew
     * @param $flights
     * @param $index
     * @param $keys_final
     * @param $keys_dhc
     * @param $keys_standby
     * @param $crew_policy
     * @param $crew_min
     * @param $crew_req
     */
    public function Create_Flights_Scheduled($flights, $index, $keys_final, $keys_dhc, $keys_standby, $crew_policy, $crew_min, $crew_req){
        $this->flights_scheduled[$index] = new stdClass();
        $this->flights_scheduled[$index]->crew_planned = count($keys_final);

        // Correct Position
        $keys_final_positioned = $this->Fix_Crew_Position($keys_final);
        $this->flights_scheduled[$index]->crew = $keys_final_positioned;

        $this->flights_scheduled[$index]->crew_policy = $crew_policy;
        $this->flights_scheduled[$index]->crew_required = $crew_req;
        $this->flights_scheduled[$index]->crew_min = $crew_min;

        $this->flights_scheduled[$index]->flights = $flights;
        if (count($keys_dhc)) {
            $this->flights_scheduled[$index]->flights[0]['crew_dhc'] = $keys_dhc;
        }
        if (count($keys_standby))
            $this->flights_scheduled[$index]->flights[0]['crew_standby'] = $keys_standby;

        //$this->flights_scheduled[$index]->crew_dhc = $keys_dhc;
        //$this->flights_scheduled[$index]->crew_standby = $keys_standby;

    }

    public function Fix_Crew_Position($keys){
        if (count($keys) == 0){
            return $keys;
        }

        $newKeys = [];
        $i = 0;
        if ($this->roster_type == CCM_CREW){
            // If First Member is already a PURSER
            if (in_array($this->crew[$keys[0]]->details['position'], [
                PURSER_POSITION,
                INSTRUCTOR_FLIGHT_ATTENDANT_POSITION,
                SENIOR_INSTRUCTOR_FLIGHT_ATTENDANT_POSITION,
            ])){
                return $keys;
            }

            foreach ($keys as $j => $id) {
               // Change index $i to 1
                $newKeys[++$i] = $id;
            }

            return $newKeys;
        }
        elseif ($this->roster_type == FCM_CREW) {
            foreach ($keys as $j => $id) {
                if ($j < 2) {
                    if (in_array($this->crew[$id]->details['position'], [
                        FLIGHT_OPERATIONS_DIRECTOR_POSITION,
                        CAPTAIN_POSITION,
                        CHIEF_PILOT_POSITION,
                        PILOT_INSTRUCTOR_POSITION,
                    ])) {
                        $newKeys[$i++] = $id;
                    } else {
                        if ($j == 0){
                            $i += 2;
                            $newKeys[$i++] = $id;
                        }
                        else if ($j == 1){
                            $i++;
                            $newKeys[$i++] = $id;
                        }
                        else {
                            $newKeys[$i++] = $id;
                        }
                    }
                    continue;
                }
                // Change index $i to 1
                $newKeys[$i++] = $id;
            }

            return $newKeys;
        }

        return $keys;
    }

    /**
     * Determine if Given List of FCM are available in a given date
     * @param $array
     * @param $datetime
     * @param $type
     * @param null $new_fdp
     */
    public function Unavailable_Periods_Check($array, $datetime, $type, $new_fdp = null){
        switch ($type){
            case "Unavailable_Periods":
                $date = date('Y-m-d', strtotime($datetime));
                foreach ($array as $id => $value){
                    $this->crew[$id]->eligible = TRUE;
                    if (count($this->crew[$id]->unavailable_periods)) {
                        foreach ($this->crew[$id]->unavailable_periods as $period) {
                            // If Date is between Unavailable FROM and TO then -> FCM is unavailable for the date
                            if (is_null($period['from'])) {
                                if (strtotime($date) <= strtotime($period['to'])) {
                                    $this->crew[$id]->eligible = FALSE;
                                    break;
                                }
                            } else {
                                if (!is_null($period['to'])) {
                                    if (strtotime($date) >= strtotime($period['from']) && strtotime($date) <= strtotime($period['to'])) {
                                        $this->crew[$id]->eligible = FALSE;
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
                break;
            case "Designated_Periods":
                foreach ($array as $id => $value){
                    if (count($this->crew[$id]->designated_rest_period)) {
                        foreach ($this->crew[$id]->designated_rest_period as $period) {
                            if ((strtotime($new_fdp[0]->report_time_binded) >= strtotime($period['from']) && strtotime($new_fdp[0]->report_time_binded) <= strtotime($period['to'])) ||
                                (strtotime($new_fdp[0]->release_time) >= strtotime($period['from']) && strtotime($new_fdp[0]->release_time) <= strtotime($period['to']))) {
                                $this->crew[$id]->eligible = FALSE;
                                break;
                            }
                        }
                    }
                }
                break;
        }
    }

    /**
     * Determines whether Crew Policy has to be applied or NOT
     * @param $flights
     * @param $crew_policy_flight_numbers
     * @return bool
     */
    public function Crew_Policy_For_Flights($flights, $crew_policy_flight_numbers){
        $flt_no = $flights['flight_number'];
        $policy = false;
        if (count($crew_policy_flight_numbers) && is_array($crew_policy_flight_numbers) && in_array($flt_no, $crew_policy_flight_numbers)){
            $policy = true;
        }
        return $policy;
    }

    /**
     * Determine New FDP Total Hours
     * @param $new_fdp
     * @return float
     */
    public function New_FDP_Total_Hours($new_fdp){
        $total_hours = 0;
        if (count($new_fdp) > 1){
            foreach($new_fdp as $each){
                $total_hours += $each->fdp_current;
            }
        }
        else if (count($new_fdp) == 1){
            $total_hours = $new_fdp[0]->fdp_current;
        }

        return $total_hours;
    }

    /**
     * Determines whether flight is round trip or one-way and Sets $this-flight_type variable
     * @param $flights
     */
    public function Flight_Count($flights){
        if (count($flights) > 1) {
            $this->flight_type = self::MANY_FLIGHTS;

            foreach($flights as $j => $each_flight) {
                if ($each_flight['is_diversion'] == 1){
                    $this->flight_type = self::MANY_FLIGHTS_WITH_DIVERSION;
                    break;
                }
            }

        }
        else {
            if ($flights[0]['is_diversion'] == 1)
                $this->flight_type = self::ONE_FLIGHT_WITH_DIVERSION;
            else
                $this->flight_type = self::ONE_FLIGHT;
        }
    }

    /**
     * Check if FDP Check
     * @param $each_flight
     * @param $previous_report_time
     * @param $sector_count
     * @return bool
     */
    public function Check_Flight_Binded_FDP($each_flight, $previous_report_time, $sector_count){
        // self::CREW_RELEASE_TIME_MIN mins added after the flight arrival. Subject to change
        $release_time = Add_Minutes_To_DateTime($each_flight['sta'], self::CREW_RELEASE_TIME_MIN);

        // Check Previous Report Time and Current Release Time
        $fdp_check = $this->Calculate_Duration($previous_report_time, $release_time);

        $report_time = date('Y-m-d H:i', strtotime($each_flight['std']));              // - self::CREW_REPORT_TIME_SEC);
        $fdp_max = $this->Max_FDP($report_time, $sector_count, $this->current_law);
        $fdp_ext = $fdp_max + 1;

        return $fdp_ext >= $fdp_check;
    }



    /**
     * Check if given flight is diverted or not
     * @param $flight
     * @return boolean
     */
    public function Flight_Diverted($flight){
        return $flight['is_diversion'] == 1 ? TRUE : FALSE;
    }

    /**
     * @param $type
     * @param $array_keys
     * @param $cc_min
     * @param $new_fdp
     * @return array
     */
    public function Priority_First($type, $array_keys, $cc_min,  $new_fdp){

        if (count($array_keys)) {
            $additional_crew = isset($new_fdp[0]->additional_crew) ? $new_fdp[0]->additional_crew : 0;
            $quantity = $cc_min + $additional_crew;

            $priority_first = [];
            $priority_last = [];
            $priority_end = [];

            $captainNumbers = 1;
            $instructorOnBoard = false;
            $allowJuniorFlight = false;

            switch ($type) {
                case 'captain' :
                    foreach ($array_keys as $id) {
                        $position = $this->crew[$id]->details['position'];
                        if ($this->crew[$id]->captain){
                            if (count($priority_first) == 0){
                                if ($position == PILOT_INSTRUCTOR_POSITION){
                                    $instructorOnBoard = true;
                                    $allowJuniorFlight = true;
                                }
                                $priority_first[] = $id;
                            }
                            else {
                                if ($additional_crew > 0 && count($priority_first) == 1){
                                    if ($position == PILOT_INSTRUCTOR_POSITION){
                                        $instructorOnBoard = true;
                                    }
                                    // If Augmented Crew Not Allow Juniors to fly
                                    $allowJuniorFlight = false;
                                    $priority_first[] = $id;
                                    $captainNumbers++;
                                }
                                else {
                                    $priority_last["$id"] = CAPT;
                                }
                            }
                        }
                        else {
                            if ($position == COPILOT_JUNIOR_POSITION){
                                $priority_last["$id"] = FO_JNR;
                            }
                            else {
                                $priority_last["$id"] = FO;
                            }
                        }
                    }

                    $was = $priority_last;

                    // Put FO-JNR In-Front
                    if (in_array(FO_JNR, $priority_last) && in_array(FO, $priority_last) && $instructorOnBoard && $allowJuniorFlight) {

                        $priority_last = Preferred_Value_First($priority_last, FO);

                        $priority_last = Preferred_Value_First_Once($priority_last, FO_JNR);

                        //debug("1) Before: ".implode(",", $was)." 2)After: ". implode(", ",$priority_last));

                        // Add 1 more FO if FO-JNR on board
                        $quantity++;
                    }
                    else {
                        if (count($priority_last)) {
                            // Remove FO-JNR
                            $priority_last = array_diff($priority_last, [FO_JNR]);

                            if (count($priority_last)){
                                $priority_last = Preferred_Value_First($priority_last, FO);
                            }
                        }
                    }

                    // Get Quantity - 1 Part
                    if ($quantity > $captainNumbers) {
                        $priority_last = array_slice($priority_last, 0, $quantity - $captainNumbers, true);
                    }

                    if (count($priority_last)){
                        $priority_last = Preferred_Value_First($priority_last, FO);
                        $priority_last = Preferred_Value_First($priority_last, CAPT);
                    }


                    // Put Keys to Values
                    $priority_last = array_keys($priority_last);

                    $final = array_merge($priority_first, $priority_last);

                    break;
                case 'purser' :
                    foreach ($array_keys as $id) {
                        if ($this->crew[$id]->purser && empty($priority_first)) {
                            $priority_first = [$id];
                        } else {
                            $priority_last["$id"] = $this->crew[$id]->purser ? 1 : 0;
                        }
                    }
                    // Put 0's at First
                    $priority_last = Preferred_Value_First($priority_last, 0);

                    $priority_last = array_keys($priority_last);

                    $final = array_merge($priority_first, $priority_last);
                    break;
                case 'male_last':
                    foreach ($array_keys as $i=>$id) {
                        if ($i == 0){
                            $priority_first = [$id];
                            continue;
                        }

                        if ($this->crew[$id]->gender == 'male' && empty($priority_end)) {
                            $priority_end = [$id];
                        } else {
                            $priority_last["$id"] = $this->crew[$id]->purser ? 1 : 0;
                        }
                    }

                    // Put 0's at First
                    $priority_last = Preferred_Value_First($priority_last, 0);

                    // Get Quantity - 1 Part
                    $priority_last = array_slice($priority_last, 0, $quantity - 2, true);

                    // Put Prioritised Group in Front
                    $priority_last = Preferred_Value_First($priority_last, 1);

                    // Put Keys to Values
                    $priority_last = array_keys($priority_last);

                    $final = array_merge($priority_first, $priority_last, $priority_end);
                    break;
            }

            return $final;
        }
        else
            return $array_keys;
    }

    /**
     * Check CCM Flight History And Order by Least Done Specified Flight Number
     * @param $array_keys
     * @param null $quantity
     * @param $sector
     * @return array
     */
    public function Crew_Flight_History_Sectors($array_keys, $quantity = null, $sector){
        if (count($array_keys)) {
            foreach ($array_keys as $id) {
                $flight_count[$id] = isset($this->crew[$id]->flight_history[$sector]) ? $this->crew[$id]->flight_history[$sector] : 0;
            }

            if (count($flight_count)) {
                asort($flight_count);
                $keys_sorted = array_keys($flight_count);
                return isset($quantity) ? array_slice($keys_sorted, 0, $quantity) : array_slice($keys_sorted, 0);
            }
        }
    }

    /**
     * Prioritising Based On Different Requirements
     * @param $array_keys
     * @param null $quantity
     * @param $object_variable
     * @return array
     */
    public function Priority_Selection($array_keys, $quantity = null, $object_variable){
        switch ($object_variable){
            case 'Non_Pseudo_First':
                foreach ($array_keys as $id) {
                    $crew[$id] = $this->crew[$id]->pseudo_crew ? 1 : 0;
                }
                // Put 0's First, Meaning Non Pseudo Crew First
                asort($crew);
                $keys_sorted = array_keys($crew);
                break;
        }
        return isset($quantity) ? array_slice($keys_sorted, 0, $quantity) : array_slice($keys_sorted,0);
    }

    /**
     * Goes Through FCM[certain keys] and Returns the ones that have lesser Hours
     * @param $array_keys
     * @param $quantity
     * @return array the resulting array.
     */
    public function Get_Lesser_Hours_Crew($array_keys, $quantity = null, $object_variable){

        foreach ($array_keys as $id) {
            $id_hours[$id] = $this->crew[$id]->$object_variable;
        }

        asort($id_hours);
        $keys_sorted = array_keys($id_hours);
        return isset($quantity) ? array_slice($keys_sorted, 0, $quantity) : array_slice($keys_sorted,0);
    }

    /**
     * Randomly selects from array and returns array of Quantity Length
     * @param $array_numbers
     * @param $quantity
     * @return array
     */
    public function Unique_Random_Numbers_Array($array_numbers, $quantity) {
        shuffle($array_numbers);
        return array_slice($array_numbers, 0, $quantity);
    }

    /**
     * Search array with different constrictions
     * @param $option_variable
     * @param value $search_value [optional]
     * @param value $search_keys [optional]
     * @return array final array after constriction applied
     */
    public function Search_FCM_GetAllKeys($option_variable, $search_value = null, $search_keys = null){
        $keys = [];
        switch ($option_variable) {
            case 'all_keys' :
                foreach ($this->crew as $i=>$each_object) {
                    $keys[$i] = $i;
                }
                break;
            case 'aircraft_type' :
                if (isset($search_keys)){
                    foreach ($search_keys as $id){
                        // NOTE ADD Aircraft TYPES
                        if (isset($this->crew[$id]->details['aircraft_types'])) {
                            $aircraft_types = $this->crew[$id]->details['aircraft_types'];
                            if ($aircraft_types && count($aircraft_types)) {
                                if (in_array($search_value, $aircraft_types)) {
                                    $keys[$id] = $id;
                                }
                                else {
                                }
                            } else { // Pseudo Crew Will Be Considered Here
                                $keys[$id] = $id;
                            }
                        }
                    }
                }
                else {
                    foreach ($this->crew as $id => $each_object) {
                        $aircraft_types = $each_object->details['aircraft_types'];
                        if ($aircraft_types && count($aircraft_types)) {
                            if (in_array($search_value, $aircraft_types)) {
                                $keys[$id] = $id;
                            }
                        }
                    }
                }
                break;
            case 'start_eligible' :
                foreach ($this->crew as $i=>$each_object) {
                    if ($each_object->start_eligible == TRUE) {
                        $keys[$i] = $i;
                    }
                }
                break;
            case 'eligible' :
                foreach ($this->crew as $i=>$each_object) {
                    if ($each_object->eligible == TRUE) {
                        $keys[$i] = $i;
                    }
                }
                break;
            case 'location_available_crew' :
                foreach ($this->crew as $i=>$each_object) {
                    if (isset($search_value)) {
                        if (isset($search_keys)) {
                            if (!in_array($i, $search_keys))
                                continue;
                        }

                        if ($each_object->current_location == $search_value) {
                            $keys[$i] = $i;
                        }
                    }
                }
                break;
            case 'not_scheduled_yet' :
                foreach ($this->crew as $i=>$each_object) {
                    if (isset($search_keys)){
                        if (!in_array($i, $search_keys))
                            continue;
                    }
                    if (empty($each_object->schedule)) {
                        $keys[$i] = $i;
                    }
                }
                break;
            case 'scheduled' :
                foreach ($this->crew as $i=>$each_object) {
                    if (count($each_object->schedule)) {
                        $keys[$i] = $i;
                    }
                }
                break;
        }

        return $keys;
    }


    public function Calculate_Max_Allowed_FDP($new_fdp, $f){
        $fdpMaxReduction = $this->WOCL_Period_Check($new_fdp, $f);
        if ($new_fdp[$f]->wocl && $fdpMaxReduction > 0){
            $new_fdp[$f]->fdp_max -= $fdpMaxReduction;
            $new_fdp[$f]->fdp_ext -= $fdpMaxReduction;
            $new_fdp[$f]->fdp_wocl_reduction = $fdpMaxReduction;
        }
    }


    /**
     * Check if WOCL Flight and Return Flight's WOCL Encroachment
     * @param $new_fdp
     * @param $f
     * @return float|int
     */
    public function WOCL_Period_Check($new_fdp, $f)
    {
        $encroachment = 0;
        $fdpMaxReduction = 0;
        $reportReleaseSameDate = false;
        if (date("Y-m-d", strtotime($new_fdp[$f]->report_time_binded)) == date("Y-m-d", strtotime($new_fdp[$f]->release_time))){
            $reportReleaseSameDate = true;
        }

        $report_time = strtotime(date('H:i', strtotime($new_fdp[$f]->report_time_binded) + $this::WOCL_LOCAL));
        $release_time = strtotime(date('H:i', strtotime($new_fdp[$f]->release_time) + $this::WOCL_LOCAL));
        $wocl_start = strtotime($this::WOCL_START);
        $wocl_end = strtotime($this::WOCL_END);

        $new_fdp[$f]->wocl = false;

        // FDP starts at WOCL
        if ($report_time >= $wocl_start && $report_time <= $wocl_end){
            $new_fdp[$f]->wocl = true;
            // Omitted for time being
//            $encroachment = round(($wocl_end - $report_time) / 3600, 1);
            //  When the FDP starts in the WOCL, the maximum stated in point 1.3 and point 1.4 will be reduced by 100 % of its encroachment up to a maximum of two hours
//            $fdpMaxReduction = $encroachment > 2 ? 2 : $encroachment;
        }
        // FDP Ends or Fully encompasses WOCL
        else {
            if ($release_time >= $wocl_end){
                if ($report_time >= $wocl_start && $report_time <= $wocl_end){
                    $new_fdp[$f]->wocl = true;
                    $encroachment = round(($wocl_end - $report_time) / 3600, 1);

                    if (date("H:i", strtotime($new_fdp[$f]->report_time_binded)) == "19:30"){
//                        debug("Here 1");
                    }
                }
                elseif ($report_time < $wocl_start || $report_time > $release_time){
                    $new_fdp[$f]->wocl = true;
                    $encroachment = round(($wocl_end - $wocl_start) / 3600, 1);
                }
            }
            // Release Time < WOCL End
            else {
                if ($release_time >= $wocl_start){
                    $new_fdp[$f]->wocl = true;
                    $encroachment = round(($release_time - $wocl_start) / 3600, 1);
                    if (date("H:i", strtotime($new_fdp[$f]->report_time_binded)) == "19:30"){
//                        debug("Here 2");
                    }
                }
            }

            if ($encroachment > 0){
                // When the FDP ends in or fully encompasses the WOCL, the maximum FDP stated in point 1.3 and point 1.4 will be reduced by 50 % of its encroachment
                $fdpMaxReduction = 0.5 * $encroachment;
            }
        }

        if (date("H:i", strtotime($new_fdp[$f]->report_time_binded)) == "19:30"){
//            debug($new_fdp[$f]->wocl);
        }

        return $fdpMaxReduction;
    }

    /***
     * @param $new_fdp
     * @param $f
     * @param $crew_min
     * @param $sector_count
     */
    public function Additional_Crew($new_fdp, $f, $crew_min, $sector_count){
        $minCrew = $this->aircraft_min_crew[$new_fdp[$f]->flights[0]['aircraft_id']];

        $requiredAdditionalCrew = intval(ceil($minCrew * 0.3));

        $hour_minute_format = strtotime(date('H:i', strtotime($new_fdp[$f]->report_time) + self::LOCAL));
        $sector_index = $sector_count - 1;
        $currentFdp = $new_fdp[$f]->fdp_cumulative;
        $addition = [];

        switch($this->current_law) {
            case TAJIKISTAN_CAA:
                $augmentedHours = [
                    0 => [14.00, 14.00, 14.00, 14.00, 14.00, 14.00],
                    1 => [15.00, 15.00, 15.00, 15.00, 15.00, 15.00],
                    2 => [16.50, 16.50, 16.50, 16.50, 16.50, 16.50],
                    3 => [15.00, 15.00, 15.00, 15.00, 15.00, 15.00],
                    4 => [14.00, 14.00, 14.00, 14.00, 14.00, 14.00],
                ];

                $doubleCrewHours = [
                    0 => [16.00, 16.00, 14.00, 14.00, 14.00, 14.00],
                    1 => [15.00, 15.00, 13.00, 13.00, 13.00, 13.00],
                ];

                if ($hour_minute_format >= strtotime('00:00') && $hour_minute_format <= strtotime('05:59')){
                    $fdpAugmentedCrew = $augmentedHours[0][$sector_index]; //index -> sectors
                    $fdpDoubleCrew = $doubleCrewHours[1][$sector_index];
                }
                elseif ($hour_minute_format >= strtotime('06:00') && $hour_minute_format <= strtotime('06:59')){
                    $fdpAugmentedCrew = $augmentedHours[1][$sector_index]; //index -> sectors
                    $fdpDoubleCrew = $doubleCrewHours[0][$sector_index];
                }
                elseif ($hour_minute_format >= strtotime('07:00') && $hour_minute_format <= strtotime('12:59')){
                    $fdpAugmentedCrew = $augmentedHours[2][$sector_index]; //index -> sectors
                    $fdpDoubleCrew = $doubleCrewHours[0][$sector_index];
                }
                elseif ($hour_minute_format >= strtotime('13:00') && $hour_minute_format <= strtotime('16:59')){
                    $fdpAugmentedCrew = $augmentedHours[3][$sector_index]; //index -> sectors
                    $fdpDoubleCrew = $doubleCrewHours[0][$sector_index];
                }
                elseif ($hour_minute_format >= strtotime('17:00') && $hour_minute_format <= strtotime('21:59')){
                    $fdpAugmentedCrew = $augmentedHours[4][$sector_index]; //index -> sectors
                    $fdpDoubleCrew = $doubleCrewHours[0][$sector_index];
                }
                else{
                    $fdpAugmentedCrew = $augmentedHours[4][$sector_index]; //index -> sectors
                    $fdpDoubleCrew = $doubleCrewHours[1][$sector_index];
                }

                // STATIC PARAMETERS, REMOVE UPON REQUEST
                // $fdpAugmentedCrew = $augmentedHours[0][$sector_index];
                // $fdpDoubleCrew = $doubleCrewHours[1][$sector_index];
                // NEEDS TO BE REMOVED LATER ON

                // Can't Double Crew if Larger Than 5, Max Crew Set to 8
                $requiredDoubleCrew = $minCrew * 2 > 8 ? (8 - $minCrew) : $minCrew;

                $addition = [
                    $requiredAdditionalCrew     => $fdpAugmentedCrew - $new_fdp[$f]->fdp_wocl_reduction,
                    $requiredDoubleCrew         => $fdpDoubleCrew  - $new_fdp[$f]->fdp_wocl_reduction,
                ];

                break;

            case PREV_TAJIKISTAN_CAA:
                switch ($crew_min) {
                    case 2:
                        $addition = [1 => 13, 2 => 16, 3 => 16];
                        break;
                    case 3:
                        $addition = [1 => 14, 2 => 15, 3 => 16];
                        break;
                    case 4:
                        $addition = [2 => 14, 3 => 15, 4 => 16];
                        break;
                    case 5:
                        $addition = [2 => 14, 3 => 15, 4 => 15.5, 5 => 16];
                        break;
                    default:
                        $addition = [];
                        break;
                }
                break;
        }

        if (count($addition)) {
            $found = false;
            foreach ($addition as $add => $max_hours) {
                $additional_crew = $add;
                $new_fdp_max = $max_hours;
                if ($max_hours >= $currentFdp){
                    $found = true;
                    break;
                }
            }

            if ($found || (isset($additional_crew) && isset($new_fdp_max))) {
                // Double Crew Provision
                if ($additional_crew == $requiredDoubleCrew){
                    $new_fdp[$f]->double_crew = true;
                    debug($new_fdp);
                }

                $new_fdp[$f]->additional_crew = $additional_crew;
                $new_fdp[$f]->fdp_max = $new_fdp_max;
                $new_fdp[$f]->fdp_ext = $new_fdp_max + 1;
            }
        }
    }

    /**
     * Find Max Daily FDP Based on Report time & Number of Sectors
     * @param $time
     * @param $sector
     * @return int
     */
    public function Max_FDP($time, $sector, $rule){
        $hour_minute_format = strtotime(date('H:i', strtotime($time) + self::LOCAL));
        $sector_index = $sector - 1;
        $max_fdp_selected = 13;
        switch($rule) {
            case TAJIKISTAN_CAA:
                $max_fdp = [
                    0 => [13.00, 13.00, 13.00, 13.00, 13.00, 13.00],
                    1 => [12.00, 12.00, 12.00, 12.00, 12.00, 12.00],
                ];

                if ($hour_minute_format >= strtotime('06:01') && $hour_minute_format <= strtotime('21:59')){
                    $max_fdp_selected = $max_fdp[0][$sector_index]; //index -> sectors
                }
                else {
                    $max_fdp_selected = $max_fdp[1][$sector_index]; //index -> sectors
                }

                // STATIC PARAMETER REMOVE UPON REQUEST
                // $max_fdp_selected = $max_fdp[0][$sector_index];
                //
                break;

            case PREV_TAJIKISTAN_CAA:
                $max_fdp = [
                    0 => [12.00, 12.00, 10.50, 10.50, 08.50, 08.50, 08.50, 08.50, 08.50, 08.50, 08.50],
                    1 => [11.00, 11.00, 10.00, 10.00, 06.50, 06.50, 06.50, 06.50, 06.50, 06.50, 06.50],
                ];

                if ($hour_minute_format >= strtotime('06:01') && $hour_minute_format <= strtotime('21:59')){
                    $max_fdp_selected = $max_fdp[0][$sector_index]; //index -> sectors
                }
                else {
                    $max_fdp_selected = $max_fdp[1][$sector_index]; //index -> sectors
                }

                break;
            case EASA:
                $max_fdp = [
                    0 => [13.00, 13.00, 12.50, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00],
                    1 => [12.75, 12.75, 12.25, 11.75, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00],
                    2 => [12.50, 12.50, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00],
                    3 => [12.25, 12.25, 11.75, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00, 09.00],
                    4 => [12.00, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00, 09.00],
                    5 => [11.75, 11.75, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00, 09.00, 09.00],
                    6 => [11.50, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00, 09.00, 09.00],
                    7 => [11.25, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00, 09.00, 09.00, 09.00],
                    8 => [11.00, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00, 09.00, 09.00, 09.00],
                    9 => [12.00, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00, 09.00],
                    10 => [12.25, 12.25, 11.75, 11.25, 10.75, 10.25, 09.75, 09.25, 09.00, 09.00, 09.00],
                    11 => [12.50, 12.50, 12.00, 11.50, 11.00, 10.50, 10.00, 09.50, 09.00, 09.00, 09.00],
                    12 => [12.75, 12.75, 12.25, 11.75, 11.25, 10.75, 10.25, 09.75, 09.00, 09.00, 09.00],
                ];

                if ($hour_minute_format >= strtotime('06:00') && $hour_minute_format <= strtotime('13:29'))
                    $max_fdp_selected = $max_fdp[0][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('13:30') && $hour_minute_format <= strtotime('13:59'))
                    $max_fdp_selected = $max_fdp[1][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('14:00') && $hour_minute_format <= strtotime('14:29'))
                    $max_fdp_selected = $max_fdp[2][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('14:30') && $hour_minute_format <= strtotime('14:59'))
                    $max_fdp_selected = $max_fdp[3][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('15:00') && $hour_minute_format <= strtotime('15:29'))
                    $max_fdp_selected = $max_fdp[4][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('15:30') && $hour_minute_format <= strtotime('15:59'))
                    $max_fdp_selected = $max_fdp[5][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('16:00') && $hour_minute_format <= strtotime('16:29'))
                    $max_fdp_selected = $max_fdp[6][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('16:30') && $hour_minute_format <= strtotime('16:59'))
                    $max_fdp_selected = $max_fdp[7][$sector_index]; //index -> sectors

                else if ($hour_minute_format >= strtotime('05:00') && $hour_minute_format <= strtotime('05:14'))
                    $max_fdp_selected = $max_fdp[9][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('05:15') && $hour_minute_format <= strtotime('05:29'))
                    $max_fdp_selected = $max_fdp[10][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('05:30') && $hour_minute_format <= strtotime('05:44'))
                    $max_fdp_selected = $max_fdp[11][$sector_index]; //index -> sectors
                else if ($hour_minute_format >= strtotime('05:45') && $hour_minute_format <= strtotime('05:59'))
                    $max_fdp_selected = $max_fdp[12][$sector_index]; //index -> sectors
                else if (($hour_minute_format >= strtotime('00:00') && $hour_minute_format <= strtotime('04:59')) ||
                         ($hour_minute_format >= strtotime('17:00') && $hour_minute_format <= strtotime('23:59')) ) {
                    $max_fdp_selected = $max_fdp[8][$sector_index]; //index -> sectors
                }
                else
                    $max_fdp_selected = 13; //index -> sectors

                break;
        }

        return floatval($max_fdp_selected);
    }

    /**
     * Apply FTL limitation to a given array
     * @param $array_keys
     * @param $from
     * @param $to
     * @param $max_allowed_hours
     * @param $new_fdp_total_hours [optional]
     */
    public function FTL_Limitation_Block($array_keys, $from, $to, $max_allowed_hours, $new_fdp_total_hours = null, $label_hours = null, $flight_history = null){

        $crew_flights = FlightCrew::join('flights', 'flights__crew.flight_id', '=', 'flights.id')
            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->whereRaw("DATE(std) BETWEEN '$from' AND '$to' ")
            ->where('is_standby',0)
            ->where('is_dhc',0)
            ->orderBy('user_id')
            ->get(['user_id','flight_number','std', 'atd', 'sta', 'ata']);

        $crew_list = [];

        if (!$crew_flights->isEmpty()){
            $iteration = 0;
            foreach($crew_flights as $flight){
                $iteration++;
                $user_id = $flight->user_id;

                if (in_array($user_id,$array_keys)){
                    // Get Flight List that FCM Has done in a given Period
                    if (isset($crew_list[$user_id]['flight_history'])){
                        if (isset($crew_list[$user_id]['flight_history'][$flight->flight_number])){
                            $crew_list[$user_id]['flight_history'][$flight->flight_number]++;
                        }
                        else
                            $crew_list[$user_id]['flight_history'][$flight->flight_number] = 1;
                    }else
                        $crew_list[$user_id]['flight_history'][$flight->flight_number] = 1;

                    // Block Hours Calculation
                    if ($flight->atd && $flight->ata){
                        $total_hours = $this->Calculate_Duration($flight->atd, $flight->ata);
                    }
                    else {
                        $total_hours = $this->Calculate_Duration($flight->std, $flight->sta);
                    }

                    if (isset($crew_list[$user_id]['hours_label'])) {
                        $crew_list[$user_id]['hours_label'] += $total_hours;
                    }
                    else
                        $crew_list[$user_id]['hours_label'] = $total_hours;

                }
            }
        }

        foreach ($array_keys as $id){

            $before_schedule_hours = isset($crew_list[$id]['hours_label']) ? $crew_list[$id]['hours_label'] : 0;
            $new_fdp_total_hours = isset($new_fdp_total_hours) ? $new_fdp_total_hours : 0;
            $planned_hours = isset($this->crew[$id]->planned_hours) ? $this->crew[$id]->planned_hours : 0;
            // Total Hours
            $total = $before_schedule_hours + $planned_hours + $new_fdp_total_hours;

            if ($total >= $max_allowed_hours) {
                $this->crew[$id]->eligible = FALSE;
            }
            else if ($total < $max_allowed_hours) {
                $this->crew[$id]->eligible = TRUE;
            }

            if (isset($label_hours))
                $this->crew[$id]->$label_hours = $total;

            if (isset($flight_history)){
                $history = isset($crew_list[$id]['flight_history']) ? $crew_list[$id]['flight_history'] : null;
                $this->Flight_History($id,$history);
                //$this->crew[$id]->$flight_history = $crew_list[$id]['flight_history'];
            }

        }
    }

    /**
     * @param $array_keys
     * @param $from
     * @param $to
     * @param null $flight_history
     */
    public function FTL_Limitation_Flight_Hours($array_keys, $from, $to, $flight_history = null){

        $max_month = date('Y-m-d', strtotime("$from - 28 days - 1 months"));
        $min_month = date('Y-m-d', strtotime("$max_month - 7 months"));
        $last_90days = date('Y-m-d', strtotime("$to - 90 days"));

        $crew_flights = FlightCrew::join('flights', 'flights__crew.flight_id', '=', 'flights.id')
            ->join('flights__numbers', 'flights.flight_number_id', '=', 'flights__numbers.id')
            ->join('airports as depAirport', 'depAirport.id', '=', 'flights__numbers.departure_airport_id')
            ->join('airports as arrAirport', 'arrAirport.id', '=', 'flights__numbers.arrival_airport_id')
            ->whereBetween(DB::raw("DATE(std)"),[$from, $to])
            ->whereNull("flights.cancelled_at")
            ->whereNull("flights.deleted_at")
            ->where('is_standby', false)
            ->where('is_dhc', false)
            ->orderBy('user_id')
            ->orderBy('std')
            ->get([
                'user_id',
                'flight_number',
                'depAirport.iata as departure_airport',
                'arrAirport.iata as arrival_airport',
                'std',
                'atd',
                'sta',
                'ata'
            ]);

        $crew_list = [];

        if (!$crew_flights->isEmpty()){
            $iteration = 0;
            foreach($crew_flights as $flight){
                $iteration++;
                $user_id = $flight->user_id;

                if (in_array($user_id,$array_keys)) {
                    // Block Hours Calculation
                    if (!is_null($flight->atd) && !is_null($flight->ata)) {
                        $total_hours = $this->Calculate_Duration($flight->atd, $flight->ata);
                        $departure_date = date('Y-m-d', strtotime($flight->atd));
                    } else {
                        $total_hours = $this->Calculate_Duration($flight->std, $flight->sta);
                        $departure_date = date('Y-m-d', strtotime($flight->std));
                    }

                    if (strtotime($departure_date) >= strtotime($last_90days)) {
                        // Get Flight List that FCM Has done in a given Period
                        $sector = "$flight->departure_airport-$flight->arrival_airport";
                        if (isset($crew_list[$user_id]['flight_history'])) {
                            if (isset($crew_list[$user_id]['flight_history'][$sector])) {
                                $crew_list[$user_id]['flight_history'][$sector]++;
                            } else
                                $crew_list[$user_id]['flight_history'][$sector] = 1;
                        } else
                            $crew_list[$user_id]['flight_history'][$sector] = 1;
                    }
                    ///////////////////////////////////////////////////////

                    $month_name = date('Y-m-01', strtotime($departure_date));

                    if (date('m', strtotime($departure_date)) >= date('m', strtotime($min_month)) &&
                        date('m', strtotime($departure_date)) <= date('m', strtotime($max_month))
                    ) {
                        if (isset($crew_list[$user_id]['hours_label'][$month_name]))
                            $crew_list[$user_id]['hours_label'][$month_name] += $total_hours;
                        else
                            $crew_list[$user_id]['hours_label'][$month_name] = $total_hours;
                    } else {
                        if (isset($crew_list[$user_id]['hours_label'][$departure_date]))
                            $crew_list[$user_id]['hours_label'][$departure_date] += $total_hours;
                        else
                            $crew_list[$user_id]['hours_label'][$departure_date] = $total_hours;

                    }
                }
            }
        }

        foreach ($array_keys as $id){
            if (isset($crew_list[$id]['hours_label'])){
                $this->crew[$id]->flight_hours_date = $crew_list[$id]['hours_label'];
            }

            if (isset($flight_history)){
                $history = isset($crew_list[$id]['flight_history']) ? $crew_list[$id]['flight_history'] : null;
                $this->Flight_History($id,$history);
            }
        }

    }

    /**
     * @param null $id_one
     * @param null $before_schedule_history
     */
    public function Flight_History($id_one = null, $before_schedule_history = null){
        $keys['scheduled'] = $this->Search_FCM_GetAllKeys($this->crew, 'scheduled');

        if (isset($id_one)) {
            $flight_history = isset($before_schedule_history) ? $before_schedule_history : [];
            foreach ($this->crew[$id_one]->schedule as $schedule) {
                foreach ($schedule->flights as $flight) {
                    $flt_no = $flight['flight_number'];
                    if (isset($flight_history[$flt_no]))
                        $flight_history[$flt_no]++;
                    else
                        $flight_history[$flt_no] = 1;
                }
            }
            $this->crew[$id_one]->flight_history = $flight_history;
        }
        else {
            if (isset($everyone)) {
                foreach ($keys['scheduled'] as $id) {
                    $flight_history = [];
                    foreach ($this->crew[$id]->schedule as $schedule) {
                        foreach ($schedule->flights as $flight) {
                            $flt_no = $flight['flight_number'];
                            if (isset($flight_history[$flt_no]))
                                $flight_history[$flt_no]++;
                            else
                                $flight_history[$flt_no] = 1;
                        }
                    }
                    $this->crew[$id]->flight_history = $flight_history;
                }
            }
        }
    }

    /**
     * @param $array_keys
     * @param $from
     * @param $to
     * @param $max_allowed_hours
     * @param null $new_fdp_total_hours
     * @param null $label_hours
     */
    public function FTL_Limitation_Duty($array_keys, $from, $to, $max_allowed_hours, $new_fdp_total_hours = null, $label_hours = null){

        foreach ($array_keys as $id){

            $before_schedule_hours = isset($crew_list[$id]['hours_label']) ? $crew_list[$id]['hours_label'] : 0;
            $new_fdp_total_hours = isset($new_fdp_total_hours) ? $new_fdp_total_hours : 0;
            $planned_hours = isset($this->crew[$id]->planned_hours) ? $this->crew[$id]->planned_hours : 0;
            // Total Hours
            $total = $before_schedule_hours + $planned_hours + $new_fdp_total_hours;

            if ($total >= $max_allowed_hours) {
                $this->crew[$id]->eligible = FALSE;
            }
            else if ($total < $max_allowed_hours) {
                $this->crew[$id]->eligible = TRUE;
            }

            if (isset($label_hours))
                $this->crew[$id]->$label_hours = $total;
        }
    }

    /**
     * Get all route flights (combined)
     * @return array grouped out/inbound flights array
     */
    public function Route_Flights(){
        $from = $this->from;
        $to = $this->to;

        $group_flights = [];
        $processedFlightIds = [];

        $outbound_flight = Flight::join('aircraft','flights.aircraft_id','=','aircraft.id')
            ->join('aircraft__types','aircraft.aircraft_type_id','=','aircraft__types.id')
            ->join('flights__numbers','flights.flight_number_id','=','flights__numbers.id')
            ->whereBetween(DB::raw("DATE(std)"),[$from, $to])
            ->whereRaw("(NOT (CHAR_LENGTH(flight_number) = 4 AND flight_number LIKE '9%'))")
            ->whereRaw("(flight_number % 2 <> 0 OR (flight_number % 2 = 0 AND departure_airport = '{$this->location[1]}' AND arrival_airport = '{$this->location[0]}'))")
            /*->where(function($sql){
                  $sql->where('parent_id',0)
                      ->orWhereNull('parent_id');
            })*/
            ->where('flight_number','!=', '001')
            ->whereNull('flights.cancelled_at')
            ->whereNull('flights.deleted_at')
            ->orderBy('ptd')
            ->orderBy('std')
            ->get([
                'flights.id as id',
                'parent_id',
                'is_diversion',
                'diversion',
                'flight_number',
                'flight_route_id',
                'aircraft_id',
                'departure_airport',
                'arrival_airport',
                'aircraft__types.id as aircraft_type',
                'std',
                'sta',
                'ptd',
                'pta',
                'aircraft__types.min_ccm',
                'policy_ccm',
                'aircraft__types.min_fcm',
                'policy_fcm'
            ])->toArray();

        foreach($outbound_flight as $i => $flight){

            if (in_array($flight['id'], $processedFlightIds)){
                continue;
            }

            if ($flight['is_diversion'] == 1){
                $diversion = unserialize($flight['diversion']);
                $diversion_requirement = [];
                if ($diversion['diversion_sta'] != '' && $diversion['diversion_sta'] != '0000-00-00 00:00')
                    $diversion_requirement['sta'] = date('Y-m-d H:i', strtotime($diversion['diversion_sta']) + self::GMT);
                if ($diversion['diversion_std'] != '' && $diversion['diversion_std'] != '0000-00-00 00:00')
                    $diversion_requirement['std'] = date('Y-m-d H:i', strtotime($diversion['diversion_std']) + self::GMT);
                if (is_numeric($diversion['diversion_airport'])) {
                    $city = Airport::join('cities', 'airports.city','=','cities.city')
                        ->where('cities.id',$diversion['diversion_airport'])
                        ->first(['airports.iata as city']);

                    $diversion_requirement['airport'] = $city->city;
                }

                $outbound_flight[$i]->diversion = $diversion_requirement;
            }


            if ($flight['ptd'] && $flight['pta']){
                $departure_index = "ptd";
                $arrival_index = "pta";
            }
            else {
                $departure_index = "std";
                $arrival_index = "sta";
            }

            $outbound_flight[$i]['std'] =  date('Y-m-d H:i', strtotime($flight[$departure_index]) + self::GMT);
            $outbound_flight[$i]['sta'] =  date('Y-m-d H:i', strtotime($flight[$arrival_index]) + self::GMT);
            $outbound_std = date('Y-m-d', strtotime($flight[$departure_index]));

            $group_flights[$i][] = $outbound_flight[$i];

            $inbound_dates = [
                date('Y-m-d', strtotime($outbound_std)),
            ];


            $outbound_flight_number = $outbound_flight[$i]['flight_number'];
            $inbound_flight_number = [ intval($outbound_flight_number) + 1 ];

            if( ($flight['departure_airport'] == $this->location[1] && $flight['arrival_airport'] == $this->location[0])
                || ($flight['departure_airport'] == $this->location[0] && $flight['arrival_airport'] == $this->location[1]) ){

                $inbound_flight_number[] = intval($outbound_flight_number) - 1;
            }
            else {
                $inbound_dates[] = date('Y-m-d', strtotime("$outbound_std + 1 days"));

            }




            $inbound_flight = Flight::join('aircraft','flights.aircraft_id','=','aircraft.id')
                ->join('aircraft__types','aircraft.aircraft_type_id','=','aircraft__types.id')
                ->join('flights__numbers','flights.flight_number_id','=','flights__numbers.id')
                ->whereNull('flights.cancelled_at')
                ->whereNull('flights.deleted_at')
                ->where(function($sql) use ($flight, $inbound_flight_number, $inbound_dates){
                    $sql->where('parent_id', $flight['id'])
                        ->orWhere(function($sql2) use ($inbound_flight_number, $inbound_dates){
                            $sql2->whereIn("flight_number", $inbound_flight_number)
                                  ->whereIn(DB::raw("DATE(std)"), $inbound_dates);
                        });
                })
                ->orderBy('flights.ptd')
                ->orderBy("flights.std")
                ->first([
                    'flights.id as id',
                    'parent_id',
                    'is_diversion',
                    'diversion',
                    'flight_number',
                    'flight_route_id',
                    'aircraft_id',
                    'departure_airport',
                    'arrival_airport',
                    'aircraft__types.id as aircraft_type',
                    'std',
                    'sta',
                    'ptd',
                    'pta',
                    'aircraft__types.min_ccm',
                    'policy_ccm',
                    'aircraft__types.min_fcm',
                    'policy_fcm'
                ]);

            $processedFlightIds[] = $flight['id'];

            if ($inbound_flight) {
                $inbound_flight = $inbound_flight->toArray();


                if ($inbound_flight['ptd'] && $inbound_flight['pta']){
                    $departure_index = "ptd";
                    $arrival_index = "pta";
                }
                else {
                    $departure_index = "std";
                    $arrival_index = "sta";
                }

                $inbound_flight['std'] =  date('Y-m-d H:i', strtotime($inbound_flight[$departure_index]) + self::GMT);
                $inbound_flight['sta'] =  date('Y-m-d H:i', strtotime($inbound_flight[$arrival_index]) + self::GMT);

                $group_flights[$i][] = $inbound_flight;

                $processedFlightIds[] = $inbound_flight['id'];
            }
        }

        $group_flights = array_values($group_flights);

        return $group_flights;
    }

    /**
     * Calculation Duration in Hours between 2 dates
     * @param $from_date
     * @param $to_date
     * @return int
     */
    public function Calculate_Duration($from_date, $to_date){
        return round((strtotime($to_date) - strtotime($from_date)) / 3600,1);
    }

}

