<?php
/**
 * Created by PhpStorm.
 * User: user
 * Date: 30.03.17

 */

namespace App\Classes\Parsing;

use App\Classes\Parsing\Helpers\ParseHelper;
use App\Classes\Parsing\Helpers\ParseValidation;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Country;

class ParseOperations
{
    public $types = [
        CPM,
        UCM,
        LOADSHEET,
        LOADSHEET_FINAL,
        LDM,
        PSM,
        PTM,
        LPM,
        TPM,
        FAG,
        ONLOAD,
        NOT,
        NOTOC,
        COMRO,
        COM,
        WAB,
        TXT_ING,
        PNL,
        PRL,
        ADL,
        SLS,
        HBG,
        CREW_LIST,

        LIR,
        CAL,
        PAL,
    ];

    public $sentDate;
    public $airportIDs;
    public $countryIDs;
    public $countryISOIDs;

    public $airlineIDs;
    public $airlineICAOIDs;

    public $airlines;
    public $airlinesICAO;

    public $messageLog = [];

    public $emailBody;
    public $messageType;
    public $mvtType;

    public $handlingAirline;

    public $skipDateRangeCheck = null;

    public $airlineModule;
    public $AIRLINE;

    public $localTimeMessages = [
        PRL,
        PNL,
        ADL,
        PTM,
        PSM,
        SLS,
        TPM,
        PSM,
    ];

    public $exceptionStationsList = [
        "LDB",
    ];

    public $handlingAirports = [];

    public $parseHelper;
    public $parseValidation;

    public $hubStations = null;
    public $hubEmails = null;
    public $stationEmails = null;

    public function __construct($sentDate = null, $airportIDs = null, $countryIDs = null, $countryISOIDs = null, $skipDateRangeCheck = null){

        $this->airlineModule = airlineModule();
        $this->AIRLINE = env(AIRLINE);

        if ($this->airlineModule){
            // set email hub and stations
            $hubEmailsAndStations = Parse::setHubStationsAndEmails();
            $this->hubStations = $hubEmailsAndStations[0];
            $this->hubEmails = $hubEmailsAndStations[1];

            // set station emails
            $this->stationEmails = Parse::setStationEmailRecipients();
        }
        else {
            $this->handlingAirports = Airport::listHandlingStations();
        }

        $this->sentDate = $sentDate ? $sentDate : date("Y-m-d");

        $this->airportIDs = $airportIDs ? $airportIDs : Airport::whereNotNull("iata")->pluck("id", "iata")->all();

        $this->countryIDs = $countryIDs ? $countryIDs : Country::whereNotNull("abbr")->pluck("id", "abbr")->all();
        $this->countryISOIDs = $countryISOIDs ? $countryISOIDs : Country::whereNotNull("iso_alpha_3")->pluck("id", "iso_alpha_3")->all();

        $this->airlineIDs = Airline::whereNotNull("iata")->pluck("id", "iata")->all();
        $this->airlineICAOIDs = Airline::whereNotNull("icao")->pluck("id", "icao")->all();

        $this->airlines = Airline::whereNotNull("iata")->get()->keyBy("iata")->all();
        $this->airlinesICAO = Airline::whereNotNull("icao")->get()->keyBy("icao")->all();

        $this->skipDateRangeCheck = $skipDateRangeCheck;

        // Helper
        $this->parseHelper = new ParseHelper($this);

        // Validation
        $this->parseValidation = new ParseValidation($this);

    }

    public function processExceptionStations(&$messageDetails){
        if (!$this->airlineModule){
            return;
        }

        if (isset($messageDetails['departureAirport']) && $messageDetails['departureAirport']){
           if (in_array($messageDetails['departureAirport'], $this->exceptionStationsList)){
               $messageDetails['departureAirport'] = null;
           }
        }

        if (isset($messageDetails['arrivalAirport']) && $messageDetails['arrivalAirport']){
            if (in_array($messageDetails['arrivalAirport'], $this->exceptionStationsList)){
                $messageDetails['arrivalAirport'] = null;
            }
        }
    }

    public function proceedWithAutoCreation($parseEmail, $airline = null){
        if (!$parseEmail && !$this->messageType){
            return true;
        }

        $msgType = $parseEmail && $parseEmail->messageType ? $parseEmail->messageType : $this->messageType;

        try {
            if ($airline) {
                // Iraqi Airlines
                if (in_array($airline->id, [468])){
                    return false;
                }

                // U6 Ural
                if (in_array($airline->id, [854]) && in_array($msgType, [PNL, PSM, PTM])) {
                    return false;
                }

                if (in_array($airline->id, [470]) && in_array($msgType, [PSM, PTM])) {
                    return false;
                }
            }
        }
        catch (\Exception $e){}

        // List of messages that should not create flights automatically
        return !in_array($msgType, [ADL, NOTOC, NOT, UCM, SLS]);
    }


    /**
     * Applies for handling
     * skips unnecessary messages for airlines' airports which aren't handled by company
     * @param $airline
     * @param $flightNumber
     * @return bool|void
     */
    public function skipAirportMessages( $airline, $flightNumber){
        if ($this->airlineModule || !$airline || !$airline->iata || !$flightNumber){
            return;
        }

        $handlingAirports = $depAP = $arrAP = null;

        if ($flightNumber->departureAirport){
            $depAP = $flightNumber->departureAirport->iata;
        }
        if ($flightNumber->arrivalAirport){
            $arrAP = $flightNumber->arrivalAirport->iata;
        }

        switch ($airline->iata){
            case "A3":
                $handlingAirports = ["HAM"];
                break;

            case "8Q":
            case "5Q":
                $handlingAirports = ["FRA"];
                break;

            case "KK":
                $handlingAirports = ["HAM", "MUC", "STR"];
                break;

            default:
                break;
        }

        // Only applies to specific airlines
        if ($handlingAirports && ($depAP || $arrAP)) {
            if ($depAP && $arrAP) {
                if (!in_array($depAP, $handlingAirports) && !in_array($arrAP, $handlingAirports)) {
                    return true;
                }
            } else if ($depAP) {
                if (!in_array($depAP, $handlingAirports)) {
                    return true;
                }
            } else if ($arrAP) {
                if (!in_array($arrAP, $handlingAirports)) {
                    return true;
                }
            }
        }

        return false;
    }


    public function isLocalTimeMessage(){
        return in_array($this->messageType, $this->localTimeMessages);
    }

    public function run($emailBody, ParseEmail $parseEmail = null){

        $this->emailBody = str_replace("=0D", "", $emailBody);

        $foundMatch = $result = false;

        if (preg_match('/^(?:\s*.\s*)?(DIV)/m', $this->emailBody)) {
            $this->messageType = MVT;

            if ($parseEmail){
                $parseEmail->messageType = $this->messageType;
            }

            if ($parseEmail && $parseEmail->messageExists()) {
                return [
                    "parsed"        => true,
                    "exists"        => true,
                ];
            }
            else {

                $this->mvtType = MVT_DIVERSION;

                try {
                    $result = new ParseDIV($this);
                    $result->run($this->emailBody, $parseEmail);
                } catch (\Exception $e) {
                    Parse::sendSimpleEmailNotification("Type: DIV\n\n" . "Message: \n" . $this->emailBody . "\n\n Error: \n" . $e->getMessage());
                }
            }
        }
        else if (preg_match('/^(?:\s*.\s*)?(MVT)/m', $this->emailBody)) {

            $this->messageType = MVT;

            if ($parseEmail){
                $parseEmail->messageType = $this->messageType;
            }

            if ($parseEmail && $parseEmail->messageExists()) {
                return [
                    "parsed"        => true,
                    "exists"        => true,
                ];
            }
            else{
                if ($this->airlineModule && $parseEmail) {
                    $headers = imap_header($parseEmail->getStream(), $parseEmail->getEmailID());
                    if (trim($headers->senderaddress) == "grsadm@sirena-travel.ru") {
                        return [
                            "parsed"        => null,
                            "exists"        => false,
                        ];
                    }
                }

                if (preg_match('/\b(?:AD|ED)[0-9]+/m', $this->emailBody)) {

                    $foundMatch = true;

                    $this->mvtType = MVT_DEPARTURE;

                    if ($parseEmail){
                        $parseEmail->mvtType = $this->mvtType;
                    }

                    //*** Updated ***
//                    try {

                        $result = new ParseMVTDep($this);
                        $result->run($this->emailBody, $parseEmail);
//                    }
//                    catch(\Exception $e){
//                        Parse::sendSimpleEmailNotification("Type: MVT Departure\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
//                    }

                }
                // Arrival message
                else if (preg_match('/\bAA[0-9]+/m', $this->emailBody)) {

                    $foundMatch = true;

                    $this->mvtType = MVT_ARRIVAL;

                    if ($parseEmail){
                        $parseEmail->mvtType = $this->mvtType;
                    }

                    //*** Updated ***
                    try {
                        $result = new ParseMVTArr($this);
                        $result->run($this->emailBody, $parseEmail);
                    }
                    catch(\Exception $e){
                        Parse::sendSimpleEmailNotification("Type: MVT Arrival\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                    }
                }

            }
        }
        // LDM|UCM|PSM|PTM|CPM|LPM|TPM|COMRO|COM|WAB|TXT_ING|PNL|ADL
        else {
            foreach ($this->types as $type)
            {
                // Special case for TXT_ING
                if (preg_match('/^(?:\s*.\s*)?('.
                    ($type == TXT_ING ? 'TXT\/ING' : ($type == LIR ? "LOADING INSTRUCTION\/REPORT" : $type))
                    .')/m', $this->emailBody)){

                    $foundMatch = true;

                    $this->messageType = $type;

                    debug("TYPE: {$type}");
                    if ($parseEmail){
                        $parseEmail->messageType = $this->messageType;
                    }

                    if ($parseEmail && $parseEmail->messageExists()) {
                        return [
                            "parsed"        => true,
                            "exists"        => true,
                        ];
                    }

                    switch($type){

                        //*** Updated ***
                        case LOADSHEET:
                            try {
                                $result = new ParseLoadsheet($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case LOADSHEET_FINAL:
                            try {
                                $result = new ParseLoadsheetFinal($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case CPM:
                            try {
                                $result = new ParseCPM($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case UCM:
//                            try {
                                $result = new ParseUCM($this);
                                $result->run($this->emailBody, $parseEmail);
//                            }
//                            catch(\Exception $e){
//                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
//                            }
                            break;

                        //*** Updated ***
                        case PTM:
                            try {
                                $result = new ParsePTM($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case PSM:
//                            try {
                                $result = new ParsePSM($this);
                                $result->run($this->emailBody, $parseEmail);
//                            }
//                            catch(\Exception $e){
//                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
//                            }
                            break;

                        //*** Updated ***
                        case LDM:
//                            try {
                                $result = new ParseLDMUpdated($this);
                                $result->run($this->emailBody, $parseEmail);
//                            }
//                            catch(\Exception $e){
//                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
//                            }
                            break;

                        //*** Updated ***
                        case PNL:
                            try {
                                $result = new ParsePNL($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case ADL:
                            try {
                                $result = new ParseADL($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case NOT:
                        case NOTOC:
                            try {
                                $result = new ParseNOT($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                        break;

                        //*** Updated ***
                        case PRL:
                            try {
                                $result = new ParsePRL($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case SLS:
                            try {
                                $result = new ParseSLS($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case HBG:
                            try {
                                $result = new ParseHBG($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        case LIR:
                            try {
                                $result = new ParseLIR($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        case CAL:
                        case PAL:
                            try {
                                $result = new ParseCAL($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case CREW_LIST:
                            try {
                                $result = new ParseCrewList($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        //*** Updated ***
                        case TPM:
                            try {
                                $result = new ParseTPM($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        case FAG:
                            try {
                                $result = new ParseFAG($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;

                        case ONLOAD:
                            try {
                                $result = new ParseONLOAD($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;


                        //*** Updated ***
                        default:
                            try {
                                $result = new ParseOthers($this);
                                $result->run($this->emailBody, $parseEmail);
                            }
                            catch(\Exception $e){
                                Parse::sendSimpleEmailNotification("Type: \n".$type."\n\n". "Message: \n".$this->emailBody."\n\n Error: \n".$e->getMessage());
                            }
                            break;
                    }

                    // Break if result, else continue
                    if ($result === null){
                         break;
                    }

                }
            }
        }

        return [
            "parsed"        => $result ? true : null,
            "foundMatch"    => $foundMatch
        ];

        /*
        if ($parseEmail) {

            if (!$foundMatch) {

                if (!$parseEmail->messageExists()) {

                    if (env(SAVE_EMAIL_LOG)){
                        $parseEmail->createMessage();
                    }
                }
            }
        }
        */

    }

}
